/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_xdraw.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 26th May 1993
;;; File: go_xdraw.p
;;; Author: Ben L.E. Rabau

compile_mode :pop11 +strict;

include XpwPixmap.ph;
uses go_load_require;
uses go_pane;

;;;------------------------------------------------------------------
;;; INTRO: This file is used by go_screenobject.p to add an X-specific
;;; INTRO: go_draw() and go_draw_bounding_box() method.
;;; INTRO:
;;; INTRO: The go_draw() method uses the go_fgdraw() and go_bgdraw() methods
;;; INTRO: which ought to be defined in the subclasses.
;;; INTRO:
;;; INTRO: It also contains some other useful X-specific draw definitions...
;;; INTRO:

vars isgo_fillable, go_filled;      ;;; defined in go_fillable.p

define :method go_draw( pane :go_pane, obj :go_screen_object );
lvars pane, obj;
;;; REF: go_draw( PANE, SCREEN_OBJECT );
;;; REF: Draws the given object in the given pane by drawing the part in
;;; REF: background colour first (see REF * GO_FILLABLE and also
;;; REF: REF * SCREEN_OBJECT/go_bgcolour), then the part of the object in
;;; REF: foreground colour (see REF * SCREEN_OBJECT/go_fgcolour).
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object (REF * GO_SCREEN_OBJECT).
lvars window, oldforeground, old_linewidth;

	if go_visible_now( obj ) == "forbidden" then return; endif;

	the_go_window(pane) -> window;
	;;; first go at colors:
	if ( obj.stored_go_fgcolour or obj.isgo_fillable ) then
		XptValue(window, XtN foreground) -> oldforeground;
	endif;
	;;; go_linewidth
	if ( obj.go_linewidth ) then
		XptValue(window, XtN lineWidth) -> old_linewidth;
		obj.go_linewidth -> XptValue(window, XtN lineWidth);
	endif;

	if (obj.isgo_fillable) and (obj.go_filled) then
		;;; first go at colors:
		if (obj.stored_go_bgcolour) then
			go_get_colour( obj.stored_go_bgcolour, pane )
			-> XptValue(window, XtN foreground);
		else
			XptValue(window, XtN background)
			-> XptValue(window, XtN foreground);
		endif;
		go_bgdraw( pane, obj );
	endif;

	;;; first go at colors:
	if ( obj.stored_go_fgcolour ) then
		go_get_colour( obj.stored_go_fgcolour, pane )
		-> XptValue(window, XtN foreground);
	else if (obj.isgo_fillable) and (obj.go_filled) then
			 oldforeground -> XptValue(window, XtN foreground);
		 endif;
	endif;

	go_fgdraw( pane, obj );

	;;; first go at colors:
	if ( obj.stored_go_fgcolour ) then
		oldforeground -> XptValue(window, XtN foreground);
	endif;
	if ( obj.go_linewidth ) then
		old_linewidth -> XptValue(window, XtN lineWidth);
	endif;
	true -> go_visible_now( obj );
enddefine;

;;;------------------------------------------------------
;;; DRAW LINES

define :method go_draw_line( pane :go_pane, x1, y1, x2, y2 );
lvars pane, x1, y1, x2, y2;
;;; REF: go_draw_line( PANE, X1, Y1, X2, Y2 );
;;; REF: Draws a line between the given screen coordinates in the pane using
;;; REF: the current line function, colours and thickness.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: X1, Y1, X2, Y2 are the coordinates of the start(1) and end(2) point
	XpwDrawLine( the_go_window(pane), x1.round, y1.round, x2.round, y2.round );
enddefine;

define :method go_draw_lines( pane :go_pane, coord_list, coord_mode );
lvars pane, coord_list, coord_mode;
;;; REF: go_draw_lines( PANE, COORD_LIST, COORD_MODE );
;;; REF: Draw the list of continueous lines in the pane given the mode of the
;;; REF: sequence of screen coordinates (either all based around the origin or
;;; REF: each coordinate as an offset of the previous coordinate).
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: COORD_LIST is a list/vector of integer coordinates [x1, y1, x2, ...]
;;; REF: COORD_MODE is one of CoordModeOrigin, CoordModePrevious
	maplist( coord_list, round ) -> coord_list;
	XpwDrawLines( the_go_window(pane), coord_list, coord_mode );
	sys_grbg_list( coord_list );
enddefine;

define :method go_draw_line_segments( pane :go_pane, coord_list );
lvars pane, coord_list;
;;; REF: go_draw_line_segments( PANE, COORD_LIST );
;;; REF: Draw the list of discontinueous segments in the pane given the mode of
;;; REF: the sequence of screen coordinate-pairs. A segment is between two
;;; REF: coordinates without connection with the next pair of coordinates.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: COORD_LIST is a list/vector of integer coordinates [x1, y1, x2, ...]
	maplist( coord_list, round ) -> coord_list;
	XpwDrawSegments( the_go_window(pane), coord_list );
	sys_grbg_list( coord_list );
enddefine;

;;;------------------------------------------------------------------
;;; DRAW/CLEAR RECTANGLES

define :method go_draw_rectangle( pane :go_pane, x, y, w, h );
lvars pane, x, y, w, h;
;;; REF: go_draw_line_segments( PANE, X, Y, W, H );
;;; REF: Draw the rectangle from the given screen position with given
;;; REF: width and height in the given pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: X and Y are integer screen coordinates of top-left-hand-corner
;;; REF: W and H are width and height in screen-coordinate units
	XpwDrawRectangle( the_go_window(pane), x.round, y.round, w.round, h.round );
enddefine;

define :method go_clear_rectangle( pane :go_pane, x, y, w, h );
lvars pane, x, y, w, h;
;;; REF: go_clear_rectangle( PANE, X, Y, W, H );
;;; REF: Clears the rectangular area from the given screen position with given
;;; REF: width and height in the given pane.
;;; REF: Due to definition of filled rectangles in X, we need to add 1 pixel
;;; REF: See also: Xlib Reference Manual O'Reilly Vol2 p153
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: X and Y are integer screen coordinates of top-left-hand-corner
;;; REF: W and H are width and height in screen-coordinate units
	XpwClearArea( the_go_window(pane), x.round, y.round, round(w+1), round(h+1) );
enddefine;


;;;------------------------------------------------------
;;; CLEAR WINDOW

define :method go_clear_pane_window( pane :go_pane );
lvars pane;
;;; REF: go_clear_pane_window( PANE );
;;; REF: This clears the whole XpwGraphic's window of the pane (regardless of
;;; REF: any objects inside!).
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
lvars window = the_go_window(pane);
	if ( window ) then XpwClearWindow( window ); endif;
enddefine;


;;;------------------------------------------------------
;;; DRAW CURVES

define :method go_draw_arcs( pane :go_pane, coord_list );
lvars pane, coord_list;
;;; REF: go_draw_arcs( PANE, COORD_LIST );
;;; REF: Draw the given arcs on the pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: COORD_LIST is a list of vectors described in REF * XpwDrawArcs
	maplist( coord_list, round ) -> coord_list;
	XpwDrawArcs( the_go_window(pane),  coord_list );
	sys_grbg_list( coord_list );
enddefine;

define :method go_draw_filled_arcs( pane :go_pane, coord_list );
lvars pane, coord_list;
;;; REF: go_draw_filled_arcs( PANE, COORD_LIST );
;;; REF: Draw the filling of the given arcs on the pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: COORD_LIST is a list/vector of coordinates [x1, y1, x2, ...]
	maplist( coord_list, round ) -> coord_list;
	XpwFillArcs( the_go_window(pane), coord_list );
	sys_grbg_list( coord_list );
enddefine;


;;;------------------------------------------------------
;;; DRAW POLYGONS

define :method go_draw_filled_polygon( pane :go_pane, coords, type, coord_mode );
lvars pane, coords, type, coord_mode;
;;; REF: go_draw_filled_polygon( PANE, COORD_LIST, TYPE, COORD_MODE );
;;; REF: Draws a filled polygon from the given screen coordinates on the pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: COORD_LIST is a list/vector of coordinates [x1, y1, x2, ...]
;;; REF: TYPE is one of Convex or Complex depending on the shape of the polygon.
;;; REF: COORD_MODE is one of CoordModeOrigin, CoordModePrevious
	maplist( coords, round ) -> coords;
	XpwFillPolygon( the_go_window(pane), coords, type, coord_mode );
	sys_grbg_list( coords );
enddefine;


;;;------------------------------------------------------
;;; DRAW BITMAP

define :method go_draw_bitmap( pane :go_pane, bitmap, x, y, w, h, to_x, to_y );
lvars pane, x, y, w, h, to_x, to_y;
;;; REF: go_draw_bitmap( PANE, BITMAP, X, Y, W, H, SCREENX, SCREENY );
;;; REF: Draws an X bitmaps given region onto the screen of the given pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: BITMAP is an XpwPixmap widget (REF * XpwPixmap).
;;; REF: X and Y are the integer coordinates of the bitmap region
;;; REF: W and H are the integer dimensions of the bitmap region
;;; REF: SCREENX and SCREENY are the integer screen coordinates
lvars window = the_go_window(pane);
	;;; Warning: There is a bug in XpwCopyTo when PixmapStatus = PixmapOnly
	XpwCopyFrom(window, bitmap, x, y, w, h, to_x, to_y);
enddefine;


;;;------------------------------------------------------
;;; DRAW TEXT

define :method go_draw_text( pane :go_pane, x, y, text, font );
lvars pane, x, y, text;
;;; REF: go_draw_text( PANE, X, Y, STRING, FONT );
;;; REF: Draws text on the pane at bottonm left postion(x,y). See also
;;; REF: REF * GO_XDEFAULTS/go_default_font.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: X and Y are the integer screen-coordinates of the box around the text
;;; REF: FONT is a XFontStruct with a valid font for the XpwGraphics widgets.
lvars window = the_go_window(pane), oldfont = false;
	if font then
		XptValue ( window, XtN font, "exptr" ) -> oldfont;
		font -> XptValue ( window, XtN font, "exptr" );
	endif;
	XpwDrawString( window,  x.round, round(y - XpwFontDescent(window)), text );
	if oldfont then
		oldfont -> XptValue ( window, XtN font, "exptr" );
	endif;
enddefine;

define :method go_text_width( pane :go_pane, text ) /* -> width */;
lvars pane, coord_list /* , width */;
;;; REF: go_text_width( PANE, STRING ) -> INTEGER;
;;; REF: Returns the width of the text in the given pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
;;; REF: text is a simple string
	XpwTextWidth( the_go_window(pane), text ) /* -> width */;
enddefine;

define :method go_text_height( pane :go_pane ) /* -> height */;
lvars pane, coord_list /* , height */;
;;; REF: go_text_height( pane ) -> INTEGER;
;;; REF: Returns the height of any text (because of the height of the font)
;;; REF: in the given pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
	XpwFontHeight( the_go_window(pane) ) /* -> height */;
enddefine;


;;;------------------------------------------------------
;;; DRAW BOUNDING BOX & SAFE REGION

define :method go_draw_bounding_box( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_draw_bounding_box( SCREEN_OBJECT );
;;; REF: Draws a dotted line around the bounding box of the given object
;;; REF: on all the panes in which the object is visible (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object (REF * GO_SCREEN_OBJECT).
	procedure( obj, pane );
	lvars obj, pane, window, x, y, w, h, old_lf, old_ls, newlist;
		go_bounding_box( scrObj ) -> (x, y, w, h);
		the_go_window(pane) -> window;
		XptValue(window, XtN function ) -> old_lf;   ;;; remember linefunction
		XptValue(window, XtN lineStyle ) -> old_ls;  ;;; and linestyle
		maplist( go_translistout( [%   x,   y,
									 x+w,   y,
									 x+w, y+h,
									   x, y+h,
									   x,   y  %], pane ),
				 round ) -> newlist;
		go_draw_lines( pane, newlist, CoordModeOrigin );
		GXxor -> XptValue(window, XtN function );
		LineOnOffDash -> XptValue(window, XtN lineStyle );
		go_draw_lines( pane, newlist, CoordModeOrigin );
		old_lf -> XptValue(window, XtN function );   ;;; reinstate originals
		old_ls -> XptValue(window, XtN lineStyle );
		sys_grbg_list( newlist );
	endprocedure;
	go_applist( /* procedure on stack */, scrObj );
enddefine;


define :method go_draw_safe_region( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_draw_safe_region( SCREEN_OBJECT );
;;; REF: Draws a dotted line around the safe region of the given object
;;; REF: on all the panes in which the object is visible (REF * GO_PANE).
;;; REF: The safe region of an object takes into account the linewidth and
;;; REF: possible line-joins which could fall on the border of the bounding box.
;;; REF: SCREEN_OBJECT is a go_screen_object (REF * GO_SCREEN_OBJECT).
	procedure( obj, pane );
	lvars obj, pane, window, x, y, w, h, old_lf, old_ls;
		go_safe_region( pane, scrObj ) -> (x, y, w, h);
		the_go_window(pane) -> window;
		XptValue(window, XtN function ) -> old_lf;    ;;; remember linefunction
		XptValue(window, XtN lineStyle ) -> old_ls;   ;;; and linestyle
		GXxor -> XptValue(window, XtN function );
		LineOnOffDash -> XptValue(window, XtN lineStyle );
		go_draw_lines( pane,
					   [%   x,   y,
						  x+w,   y,
						  x+w, y+h,
							x, y+h,
							x,   y  %],
					   CoordModeOrigin );
		old_lf -> XptValue(window, XtN function );    ;;; reinstate originals
		old_ls -> XptValue(window, XtN lineStyle );
	endprocedure;
	go_applist( /* procedure on stack */, scrObj );
enddefine;

;;;------------------------------------------------------------------
;;; CLIPPING

;;; uses xlib;
;;; uses XGraphicsContext;

lconstant macro None = 0;			;;; see X.h
lconstant macro YXSorted = 2;		;;; see X.h

XptLoadProcedures 'clip_masks' lvars
	XSetClipMask
	XSetClipRectangles;

define XSetClipMask(display, GC, args);
lvars display, GC, args;
	exacc (3) raw_XSetClipMask(display, GC, args);
enddefine;

define XSetClipRectangles(display, GC, int1, int2, xrect, int3, int4);
lvars display, GC, int1, int2, xrect, int3, int4;
	exacc (7) raw_XSetClipRectangles(display, GC, int1, int2,
									xrect, int3, int4);
enddefine;

vars clip_rectangle = writeable initshortvec(4);

define :method go_reset_clipmask( pane :go_pane );
lvars pane;
;;; REF: go_reset_clipmask( PANE );
;;; REF: Remove clipping in the given pane.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
lvars window = the_go_window( pane );
	XSetClipMask( XtDisplay(window),
				  fast_XptValue(window, XtN usersGC),
				  None);
enddefine;

define :method go_set_clipmask( x, y, w, h, pane :go_pane );
lvars x, y, w, h, pane;
;;; REF: go_set_clipmask( X, Y, W, H, PANE );
;;; REF: Set clipping to the rectangle with bottom left corner in (X,Y) and
;;; REF: given width and height in the given pane.
;;; REF: X and Y are the integer screen coordinates of the clip rectangle.
;;; REF: W and H are the integer screen dimensions of the clip rectangle.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
lvars window = the_go_window( pane );
	XSetClipRectangles(
				  XtDisplay(window),
				  fast_XptValue(window, XtN usersGC),
				  0, 0,
				  fill( x, y, w+1, h+1, clip_rectangle ), ;;; see Xlib RM p.153
				  1, YXSorted );
enddefine;

;;;------------------------------------------------------------------
;;; Convenience function for drawing thick rectangles nicely:
define :method go_add_cap_style( pane :go_pane );
lvars pane;
;;; REF: go_add_cap_style( PANE );
;;; REF: Add a projecting line style for line-joins in the given pane. This
;;; REF: will make sharp corners where two consequetive lines meet.
;;; REF: PANE is an instance of the class go_pane (REF * GO_PANE).
	CapProjecting                               ;;; for thick line joins (rects)
	-> XptValue( pane.stored_go_window, XtN capStyle );
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_xdraw = true;

/* --- Revision History --------------------------------------------
 * BR 17/05/93
 *     Added copy of the pane's background colour to the foreground colour
 *     in case of a fillable object which has a default bgcolour.
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 16/04/93
 *     Removed direct call to XpwDrawString() by go_draw_text()
 *     Removed direct call to XpwTextWidth() by go_text_width()
 *     Removed direct call to XpwFontHeight() by go_text_height()
 * BR 16/04/93
 *     Changed add_cap_style() into go_add_capstyle();
 *     Removed direct call to XpwClearWindow() by go_clear_pane_window()
 * BR 08/04/93
 *     Removed direct call to XpwDrawLine[s]() by go_draw_line[s]();
 *     Removed direct call to XpwDrawSegments() by go_draw_line_segments();
 *     Removed direct call to XpwDrawArcs() by go_draw_arcs();
 *     Removed direct call to XpwFillArcs() by go_draw_filled_arcs();
 *     Removed direct call to XpwDrawRectangle() by go_draw_rectangle();
 *     Removed direct call to XpwClearArea() by go_clear_rectangle()
 *     Removed direct call to XpwFillPolygon() by go_draw_filled_polygon()
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * BR 01/03/93
 *     Made use of extra_variables for safety on interrupts...
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 24/11/92
 *    Separated the go_draw routines out which are X-specific and general enough...
 */
;;; eof
