/* --- Copyright University of Sussex 1998. All rights reserved. ----------
 > File:			C.alpha/src/ass.p
 > Purpose:			Alpha run-time assembler
 > Author:			John Gibson, Aug 25 1994 (see revisions)
 */

;;; --- DECLARATIONS -------------------------------------------------------

#_INCLUDE 'declare.ph'
#_INCLUDE 'vmdefs.ph'
#_INCLUDE 'drop_code.ph'

	;;; Assembler routines
constant
	procedure (fast_subscrl),

	_popenter			;;; enter procedure with checks (amain.s)
	_popuenter			;;; enter updater procedure with check (amain.s)
	_popuncenter		;;; enter updater procedure without checks (amain.s)
	_checkall			;;; check interupts et cetera (in amain.s)
	_bfield				;;; fetch unsigned bitfield (in amove.s)
	_sbfield			;;; fetch signed bitfield (in amove.s)
	_ubfield			;;; update bitfield (in amove.s)
	_setstklen			;;; set user stacklength (in alisp.s)
	_setstklen_diff		;;; set user stacklength (in alisp.s)
	;

vars
	_trap,				;;; trap flag
	;

section $-Sys$-Vm;

vars
	_asm_drop_ptr,		;;; pointer to drop code at
	_asm_code_offset,	;;; offset into executable code (sysint bytes)
	_asm_pass,			;;; assembler pass counter -- false on last pass
	asm_clist,			;;; moving pointer into codelist: asm_instr is asm_clist(1)
	asm_instr,			;;; current {^I_POP ...} type instruction
	asm_struct_list,	;;; list of structures to drop in table
	_asm_nplit_size,	;;; size of nonpop literal table
	_Nlocals,			;;; sysint total number of dynamic locals
	_Nstkvars,			;;; sysint total number of stack locals
	_Npopstkvars,		;;; sysint number of pop stack locals
	_Nreg,				;;; sysint total number of regs used
	_Npopreg,			;;; sysint number of pop regs used
	_Nframewords,		;;; sysint length of stack frame in words
	;

constant procedure
		(Np_literal_offset, Trans_structure,
		Get_procedure, Is_register, Code_pass, Drop_I_code,
		I_GOTO, I_GOTO_opt)
	;

endsection;


;;; ----------------------------------------------------------------------

section $-Sys$-Vm;

lvars
	_us_offs,			;;; user stack offset from _:RGusp
	_header_offs,		;;; word offset size from pdr start to PD_EXECUTE
	_nplit_offs,		;;; word offset size from pdr start to nonpop lit table
	_use_lda_larith = true,
	;

lconstant macro (
	;;; operand addressing modes
	MODE_REG		= 1,
	MODE_LIT		= 2,
	MODE_LIT_DEFER	= 3,
	MODE_REG_DEFER	= 4,
	MODE_REG_DEFER2	= 5,
	MODE_STACK		= 6,

	;;; For passing to Drop_structure
	DEFER			= true,
	DIRECT			= false,

	_NOREG			= _-1,
	_UNUSED			= _0,
);

;;; --- MAPPING REGISTERS TO IDENTIFIERS -------------------------------------

	;;; Register identifiers
	;;; value is popint reg number shifted left 1, plus bit 0 if pop reg
protected register constant
	pop_reg_A		= RGpl0  << 1 || 1,
	pop_reg_B		= RGpl1  << 1 || 1,
	pop_reg_C		= RGpl2  << 1 || 1,
	pop_reg_D		= RGpl3  << 1 || 1,
	pop_reg_E		= RGpl4  << 1 || 1,
	pop_reg_F		= RGpl5  << 1 || 1,
	pop_reg_G		= RGpl6  << 1 || 1,
	pop_reg_H		= RGpl7  << 1 || 1,
	pop_reg_I		= RGpl8  << 1 || 1,
	pop_reg_J		= RGpl9  << 1 || 1,
	pop_reg_K		= RGpl10 << 1 || 1,

	nonpop_reg_A	= RGnpl0 << 1,
	nonpop_reg_B	= RGnpl1 << 1,
	nonpop_reg_C	= RGnpl2 << 1,
	nonpop_reg_D	= RGnpl3 << 1,
	nonpop_reg_E	= RGnpl4 << 1,

	arg_reg_0		= RGt0 << 1,
	arg_reg_1		= RGt1 << 1,
	arg_reg_2		= RGt2 << 1,

	chain_reg		= RGchain << 1,
	;

	;;; Lists of pop and nonpop registers for local vars, in order of
	;;; allocation. These lists start with a list of registers that should be
	;;; local to every procedure.
constant
	asm_pop_registers	= [%[], ident pop_reg_A, ident pop_reg_B,
								ident pop_reg_C, ident pop_reg_D,
								ident pop_reg_E, ident pop_reg_F,
								ident pop_reg_G, ident pop_reg_H,
								ident pop_reg_I, ident pop_reg_J,
								ident pop_reg_K
						  %],

	asm_nonpop_registers= [%[], ident nonpop_reg_A, ident nonpop_reg_B,
								ident nonpop_reg_C, ident nonpop_reg_D,
								ident nonpop_reg_E
						  %];

;;; --------------------------------------------------------------------------

define Is_address_reg() with_nargs 1;
	Is_register()			;;; every reg is an address reg
enddefine;

	;;; temp register queue variables
lvars _Treg1, _Treg2, _Treg3;

define :inline lconstant IS_TMP_REG(_r);
	(_r == _:RGt3 or _r == _:RGt4 or _r == _:RGt5)
enddefine;

define :inline lconstant IS_NOREG(_r);
	_neg(_r)
enddefine;

;;; initialise register queue
define lconstant init_reg();
	_:RGt3 -> _Treg1;
	_:RGt4 -> _Treg2;
	_:RGt5 -> _Treg3
enddefine;

;;; Get next free temporary register from queue
define lconstant get_reg() -> _reg;
	lvars _reg = _Treg1;
	if IS_NOREG(_reg) then
		;;; none left in queue
		mishap(0, 'SYSTEM ERROR 1 IN get_reg');
	else
		_Treg2 -> _Treg1;
		_Treg3 -> _Treg2;
		_NOREG -> _Treg3
	endif
enddefine;

;;; Free temporary register and place at back of queue
define lconstant kill_reg(_reg);
	lvars _reg;
	returnif(IS_NOREG(_reg));
	if IS_NOREG(_Treg1) then
		_reg -> _Treg1
	elseif IS_NOREG(_Treg2) then
		_reg -> _Treg2
	elseif IS_NOREG(_Treg3) then
		_reg -> _Treg3
	else
		mishap(0, 'SYSTEM ERROR 1 IN kill_reg');
	endif
enddefine;


;;; --- INSTRUCTION-DROPPING PROCEDURES ------------------------------------

lvars
	_lasti_ptr,
	_lasti_arith,
	_lasti_src1,
	_lasti_src2,
	_lasti_dst,
;

#_IF ##(w)[_1|d] = _1
define :inline lconstant BIG_LITERAL(_Lit);
	;;; true if exceeds 32-bit signed
	(lblock
		lstackmem -i _testint; _Lit -> _testint!(-i), _testint!(-i)
	 endlblock /== _Lit)
enddefine;
#_ENDIF

	;;; Put an instruction (ie int) in the procedure record (on last pass)
define lconstant Drop_instr(_i);
	lvars _i;
	;;; just increment except on last pass
	unless _asm_pass then
		_i -> _asm_drop_ptr!(code)++ -> _asm_drop_ptr
	endunless;
	@@(code){_asm_code_offset}++ -> _asm_code_offset
enddefine;

define lconstant Drop_MEMf(_Op, _Rval, _Disp, _Raddr);
	lvars	is_load = false, is_store = false, _Op, _Rval, _Disp, _Raddr,
			_Rtmp, _sgn, _hi, _prev_inst, _pulled_prev = false;

	define lconstant drop_memf(_Op, _Rval, _Disp, _Raddr);
		lvars _Op, _Rval, _Disp, _Raddr;
		chain(_MEMf_INST(_Op, _Rval, _Disp, _Raddr), Drop_instr)
	enddefine;

	;;; try to move a preceding arith instruction or store to after
	;;; a load (move a store only if load is off PB)

	if _Op == _OP_ldl or _Op == _OP_ldq then
		true -> is_load
	elseif _Op == _OP_stl or _Op == _OP_stq then
		true -> is_store
	endif;

	if is_load and not(_asm_pass) and _asm_drop_ptr == _lasti_ptr
	and (_lasti_arith or _Raddr == _:RGpb)
	and _lasti_dst /== _Raddr and _lasti_dst /== _Rval
	and _lasti_src1 /== _Rval and _lasti_src2 /== _Rval
	then
		;;; put the previous instruction after the load
		_asm_drop_ptr--!(code) -> (_prev_inst, _asm_drop_ptr);
		true -> _pulled_prev
	endif;

	_shift(_Disp, _-15) -> _sgn;
	if _sgn /== _0 and _sgn /== _-1 then
		;;; hi part is not simple sign-extension of lo part
		_shift(_sgn, _-1) -> _hi;
		if _Op == _OP_lda and not(_Disp _bitst _16:FFFF) then
			_hi -> _Disp;
			_OP_ldah -> _Op
		else
			if is_store then get_reg() else _Rval endif -> _Rtmp;
			if _sgn _bitst _1 then
				;;; lo will be sign-extended to negative, effectively
				;;; subtracting 1 from hi -- add 1 to compensate
				if _hi /== _16:7FFF then
					_hi _add _1 -> _hi
				else
					;;; adding 1 would overflow, so need a 3rd instruction
					drop_memf(_OP_ldah, _Rtmp, _1, _Raddr);
					_Rtmp -> _Raddr
				endif
			endif;
			drop_memf(_OP_ldah, _Rtmp, _hi, _Raddr);
			_Rtmp -> _Raddr;
			if is_store then kill_reg(_Rtmp) endif
		endif
	endif;

	if _asm_pass then chain(_0, Drop_instr) endif;

	drop_memf(_Op, _Rval, _Disp, _Raddr);		;;; masks _Disp

	if is_load then
		if _pulled_prev then
			;;; _asm_code_offset already incremented for this
			_prev_inst -> _asm_drop_ptr!(code)++ -> _asm_drop_ptr
		endif
	else
		_asm_drop_ptr -> _lasti_ptr;	;;; points AFTER last instruction
		not(is_store) -> _lasti_arith;
		_Raddr -> _lasti_src1;
		_-1 -> _lasti_src2;
		_Rval -> _lasti_dst
	endif
enddefine;

	;;; Binary operations with two register sources
define lconstant Drop_OPfR(_Op, _Func, _Rsrc, _Rsrc2, _Rdst);
	lvars _Op, _Func, _Rsrc, _Rsrc2, _Rdst;
	Drop_instr(_OPfR_INST(_Op, _Func, _Rsrc, _Rsrc2, _Rdst));
	_asm_drop_ptr -> _lasti_ptr;	;;; points AFTER last instruction
	true -> _lasti_arith;
	_Rsrc -> _lasti_src1;
	_Rsrc2 -> _lasti_src2;
	_Rdst -> _lasti_dst
enddefine;

lconstant procedure Drop_mvlit;

	;;; Binary operations with one immediate source (drops 1, 2 or 3
	;;; instructions)
define lconstant Drop_OPfL(_Op, _Func, _Rsrc, _Lit, _Rdst);
	lvars _Op, _Func, _Rsrc, _Lit, _Rdst, _Rtmp, _v;
	if _Lit _lteq _16:FF then
		Drop_instr(_OPfL_INST(_Op, _Func, _Rsrc, _Lit, _Rdst));
		_asm_drop_ptr -> _lasti_ptr;	;;; points AFTER last instruction
		true -> _lasti_arith;
		_Rsrc -> _lasti_src1;
		_-1 -> _lasti_src2;
		_Rdst -> _lasti_dst;
		return
	endif;

	if _Op == _OP_ARITH
	and ((_Func == _FUNC_addl or _Func == _FUNC_subl) and _use_lda_larith
		 or _Func == _FUNC_addq or _Func == _FUNC_subq)
	then
		_Lit -> _v;
		if _Func == _FUNC_subl or _Func == _FUNC_subq then
			_negate(_v) -> _v
		endif;
#_IF ##(w)[_1|d] = _1
		unless BIG_LITERAL(_v) then
			chain(_OP_lda, _Rdst, _v, _Rsrc, Drop_MEMf)
		endunless;
#_ELSE
		chain(_OP_lda, _Rdst, _v, _Rsrc, Drop_MEMf)
#_ENDIF
	endif;

	if _Rdst == _Rsrc then get_reg() else _Rdst endif -> _Rtmp;
	Drop_mvlit(_Lit, _Rtmp);
	Drop_OPfR(_Op, _Func, _Rsrc, _Rtmp, _Rdst);
	if _Rdst == _Rsrc then kill_reg(_Rtmp) endif
enddefine;

	;;; Move literal to register (1 - 3 instructions)
define lconstant Drop_mvlit(_Lit, _Rdst);
	lvars _Lit, _Rdst, _base = _:RGzero, _fdiff, _org_Lit;
	if _Lit _lteq _16:FF then
		chain(_OP_or, _base, _Lit, _Rdst, Drop_OPfL)
	endif;
	_Lit -> _org_Lit;
	if _Lit _sgr _16:7FFF and (_Lit _sub false ->> _fdiff) _sgreq _negate(_Lit)
	then
		_fdiff -> _Lit;
		_:RGfalse -> _base
	endif;
#_IF ##(w)[_1|d] = _1
	if BIG_LITERAL(_Lit) then
		;;; exceeds 32-bit signed -- use load from literal table
		chain(_OP_ldq, _Rdst, Np_literal_offset(_org_Lit) _add _nplit_offs,
										_:RGpb, Drop_MEMf)
	endif;
#_ENDIF
	chain(_OP_lda, _Rdst, _Lit, _base, Drop_MEMf)
enddefine;

	;;; Move register to register
define lconstant Drop_mvreg(_Rsrc, _Rdst);
	lvars _Rsrc, _Rdst;
	chain(_OP_or, _Rsrc, _Rsrc, _Rdst, Drop_OPfR)
enddefine;

define lconstant Drop_nop();
	chain(_:RGzero, _:RGzero, Drop_mvreg)
enddefine;

define lconstant Drop_OPf(/* _Op, _Func, _Rsrc, _RorLit, _Rdst, islit */);
	if (/*islit*/) then chain(Drop_OPfL) else chain(Drop_OPfR) endif
enddefine;

define lconstant Drop_jmp(_Hint, _Raddr);
	lvars _Op, _Hint, _Raddr;
	chain(_MEMf_JMP_INST(_Hint,
						if _Hint == _HINT_jsr then _:RGret else _:RGzero endif,
						_Raddr),
			Drop_instr)
enddefine;

	;;; Adjust user stack pointer from cached offset. Occurs in branch
	;;; delay slot so drop NOP if none to do
define lconstant Drop_us_adjust();
	if _nonzero(_us_offs) then
		;;; lda rusp, offs(rusp)
		Drop_MEMf(_OP_lda, _:RGusp, _us_offs, _:RGusp);
		_0 -> _us_offs
	endif
enddefine;

define lconstant Drop_callsub(_routine);
	lvars _routine, _tmpreg = get_reg();
	Drop_mvlit(_routine, _tmpreg);
	Drop_us_adjust();
	Drop_jmp(_HINT_jsr, _tmpreg);
	kill_reg(_tmpreg)
enddefine;


;;; --- branches -----------------------------------------------------------

	;;; Branch to byte offset from current position
define lconstant Drop_BRAf(_Op, _Rval, _Disp);
	lvars _Op, _Rval, _Disp;
	chain(_BRAf_INST(_Op, _Rval, --@@(code){_Disp} _sub _asm_code_offset),
										Drop_instr)
enddefine;

	;;; Produce template for forward branch
define lconstant Drop_fwd_BRAf();
	Drop_instr(0);
	_asm_drop_ptr
enddefine;

	;;; Fill in offset for forward branch
define lconstant Set_fwd_BRAf(_Op, _Rval, _save);
	lvars _Op, _Rval, _save;
	unless _asm_pass then
		_BRAf_INST(_Op, _Rval, @@(code){_asm_drop_ptr, _save})
								-> _save!(code)[_-1];
		_NULL -> _lasti_ptr
	endunless;
enddefine;


;;; --- STRUCTURE ACCESS -----------------------------------------------------

;;; Form literal operand or false register if value is address of false
define lconstant Get_literal(item);
	lvars item;
	if item then
		(MODE_LIT, item, _UNUSED)
	else
		(MODE_REG, _:RGfalse, _UNUSED)
	endif;
enddefine;


;;; Form operand to transfer a structure (or its valof) to a register (or the
;;; stack), relative to structure address at start of code if in heap, or
;;; absolute if not.
define lconstant Get_structure(_opspec, defer) /* -> (mode, _opnd, _offs) */;
	lvars _opspec, defer, argstruct, _offs, _reg;

	;;; replace structure with offset or reg ident on first pass (nonop
	;;; on subsequent passes)
	Trans_structure(asm_instr!INST_ARGS[_opspec])
							->> argstruct -> asm_instr!INST_ARGS[_opspec];

	if iscompound(argstruct) then
		;;; literal or idval of absolute pointer
		if defer then
			if (Is_register(argstruct) ->> _reg) then
				;;; register operand
				return(MODE_REG, _int(_reg), _UNUSED)
#_IF ##(w)[_1|d] = _1
			elseif BIG_LITERAL(argstruct) then
				;;; need to go via nonpop literal table
				_pint(Np_literal_offset(argstruct))
							->> argstruct -> asm_instr!INST_ARGS[_opspec];
				;;; drop thru for simple case
#_ENDIF
			else
				;;; idval of absolute operand
				return(MODE_LIT_DEFER, argstruct, _UNUSED)
			endif
		else
			;;; literal operand
			return(Get_literal(argstruct))
		endif
	endif;

	;;; simple
	_int(argstruct) -> _offs;		;;; offset as a sysint
	if _neg(_offs) then
		;;; negated offset for on-stack lvar, shifted left 1
		;;; if bit 0 is set, access is via a ref, i.e. another indirection
		;;; this is disabled by _____defer being false (only for I_POPQ)
		unless _offs _bitst _1 then false -> defer endunless;
		_negate(_shift(_offs, _-1)) -> _offs;
		_:RGsp -> _reg
	elseif _offs _bitst _NONPOP_LIT then
		;;; value in nonpop literal table
		_nplit_offs _add _offs -> _offs;	;;; includes - _NONPOP_LIT
		_:RGpb -> _reg
	else
		;;; else via argstruct address in table
		;;; calculate offset for word offset to place in table
		@@PD_TABLE{_offs} -> _offs;
		_:RGpb -> _reg
	endif;
	(if defer then MODE_REG_DEFER2 else MODE_REG_DEFER endif, _reg, _offs)
enddefine;

;;; ---------------------------------------------------------------------

lvars
	tos_Mode = false,
	_tos_Opnd,
	_tos_Offs,
	_tos_iscopy,
;

lconstant procedure (Load_structure);

define lconstant Increment_us();
	false -> tos_Mode;
	@@(w){_us_offs}++ -> _us_offs;
	if _us_offs _sgr _16:7FFF or _us_offs _slt _-16:8000 then
		mishap(0, 'SYSTEM ERROR 1 IN Increment_us');
	endif
enddefine;

define lconstant Decrement_us();
	false -> _tos_iscopy;
	--@@(w){_us_offs} -> _us_offs;
	if _us_offs _sgr _16:7FFF or _us_offs _slt _-16:8000 then
		mishap(0, 'SYSTEM ERROR 1 IN Decrement_us');
	endif
enddefine;

define lconstant Load_tos_reg(_reg);
	lvars _reg, _tmp;
	if tos_Mode == MODE_REG then
		returnif(IS_NOREG(_reg));
		Drop_mvreg(_tos_Opnd, _reg);
		_tos_Opnd -> _tmp;
		if IS_TMP_REG(_tmp) then kill_reg(_tmp) endif;
		_reg -> _tos_Opnd
	else
		if IS_NOREG(_reg) then get_reg() -> _reg endif;
		Load_structure(_reg, (tos_Mode, _tos_Opnd, _tos_Offs));
		MODE_REG -> tos_Mode; _reg -> _tos_Opnd; _UNUSED -> _tos_Offs
	endif
enddefine;

define lconstant Flush_tos(kill);
	lvars kill, _tmp;
	returnunless(tos_Mode);
	unless _tos_iscopy then		;;; must be a reg if copied
		Load_tos_reg(_NOREG);
		Drop_MEMf(_OP_stW, _tos_Opnd, _us_offs, _:RGusp);	;;; stW reg, offs(rusp)
		true -> _tos_iscopy
	endunless;
	if kill then
		_tos_Opnd -> _tmp;
		if IS_TMP_REG(_tmp) then kill_reg(_tmp) endif;
		false -> tos_Mode
	endif
enddefine;

	;;; Push onto user stack -- stack grows down, pre-decrement.
	;;; Stack pointer not adjusted until change of control flow
define lconstant Drop_push(_reg);
	lvars _reg;
	Flush_tos(true);
	MODE_REG -> tos_Mode; _reg -> _tos_Opnd; _UNUSED -> _tos_Offs;
	Decrement_us()
enddefine;

	;;; Pop from user stack -- stack grows down, post-increment.
	;;; Stack pointer not adjusted until change of control flow
define lconstant Drop_pop_tmp(_reg) -> (_reg, _tmpreg);
	lvars _reg, _tmpreg = _NOREG;
	if tos_Mode then
		if tos_Mode == MODE_REG then
			_tos_Opnd -> _reg;
			if IS_TMP_REG(_reg) then _reg -> _tmpreg endif
		else
			if IS_NOREG(_reg) then get_reg() ->> _reg -> _tmpreg endif;
			Load_structure(_reg, (tos_Mode, _tos_Opnd, _tos_Offs))
		endif
	else
		if IS_NOREG(_reg) then get_reg() ->> _reg -> _tmpreg endif;
		Drop_MEMf(_OP_ldW, _reg, _us_offs, _:RGusp);	;;; ldW reg, offs(rusp)
	endif;

	Increment_us()
enddefine;

	;;; Pop from user stack -- stack grows down, post-increment.
	;;; Stack pointer not adjusted until change of control flow
define lconstant Drop_pop(_reg);
	lvars _reg, _tmp;
	if tos_Mode then
		if tos_Mode == MODE_REG then
			unless _reg == _:RGzero then Drop_mvreg(_tos_Opnd, _reg) endunless;
			_tos_Opnd -> _tmp;
			if IS_TMP_REG(_tmp) then kill_reg(_tmp) endif
		elseunless _reg == _:RGzero then
			Load_structure(_reg, (tos_Mode, _tos_Opnd, _tos_Offs))
		endif
	else
		_reg -> _tmp;
		if _reg == _:RGzero then
			;;; _:RGzero causes problems with stack underflow in OSF
			get_reg() -> _reg
		endif;
		Drop_MEMf(_OP_ldW, _reg, _us_offs, _:RGusp);	;;; ldW reg, offs(rusp)
		if _tmp == _:RGzero then kill_reg(_reg) endif
	endif;

	Increment_us()
enddefine;

;;; Drop code to load (store) into (from) register from (into) operand.
define lconstant Load_structure(_reg, Mode, _Opnd, _Offs);
	lvars _reg, Mode, _Opnd, _Offs;
	go_on Mode to REG LIT LIT_DEFER REG_DEFER REG_DEFER2 STACK else ERROR;

	REG:
		Drop_mvreg(_Opnd, _reg);
		return;
	LIT:
		Drop_mvlit(_Opnd, _reg);
		return;
	LIT_DEFER:
		Drop_MEMf(_OP_ldW, _reg, _Opnd@ID_VALOF, _:RGzero);
		return;
	REG_DEFER:
		Drop_MEMf(_OP_ldW, _reg, _Offs, _Opnd);
		return;
	REG_DEFER2:
		Drop_MEMf(_OP_ldW, _reg, _Offs, _Opnd);
		Drop_MEMf(_OP_ldW, _reg, @@ID_VALOF, _reg);
		return;
	STACK:
		Drop_pop(_reg);
		return;
	ERROR:
		mishap(0, 'SYSTEM ERROR 1 IN Load_structure');
enddefine;

define lconstant Store_structure(_reg, Mode, _Opnd, _Offs);
	lvars _reg, Mode, _Opnd, _Offs, _tmpreg;
	go_on Mode to REG LIT LIT_DEFER REG_DEFER REG_DEFER2 STACK else ERROR;

	REG:
		Drop_mvreg(_reg, _Opnd);
		return;
	LIT:
		mishap(0, 'SYSTEM ERROR 1 IN Store_structure');
		return;
	LIT_DEFER:
		Drop_MEMf(_OP_stW, _reg, _Opnd@ID_VALOF, _:RGzero);
		return;
	REG_DEFER:
		Drop_MEMf(_OP_stW, _reg, _Offs, _Opnd);
		return;
	REG_DEFER2:
		Drop_MEMf(_OP_ldW, get_reg() ->> _tmpreg, _Offs, _Opnd);
		Drop_MEMf(_OP_stW, _reg, @@ID_VALOF, _tmpreg);
		kill_reg(_tmpreg);
		return;
	STACK:
		Drop_push(_reg);
		return;
	ERROR:
		mishap(0, 'SYSTEM ERROR 2 IN Store_structure');
enddefine;

	;;; POP reg (or bring from structure)
define lconstant Get_fsource(_arg, no_lit, flush_tos)
										-> (_Opnd, islit, _tmpreg);
	lvars	_arg, instr, Mode, _Opnd, _Offs, islit = false, no_lit,
			op, _tmpreg = _NOREG, _r, flush_tos;
	dlocal asm_instr;
	if (asm_instr!INST_ARGS[_arg] ->> instr) then
		;;; specified by move instruction
		instr!INST_OP -> op;
		if op == I_MOVENUM or op == I_MOVEADDR then
			Get_literal(instr!INST_ARGS[_0])
		else
			instr -> asm_instr;
			Get_structure(_0, op==I_MOVE)
		endif;
		if flush_tos then Flush_tos(true) endif
	elseif tos_Mode then
		if tos_Mode == MODE_REG and (_tos_Opnd -> _r, IS_TMP_REG(_r)) then
			_r -> _tmpreg
		endif;
		tos_Mode, _tos_Opnd, _tos_Offs;
		Increment_us()
	else
		;;; from stack
		MODE_STACK, _UNUSED, _UNUSED
	endif -> (Mode, _Opnd, _Offs);
	returnif(Mode == MODE_REG);
	if Mode /== MODE_LIT or no_lit then
		if isinteger(no_lit) then
			_int(no_lit)		;;; work reg to use
		else
			get_reg() ->> _tmpreg
		endif -> _r;
		Load_structure(_r, (Mode, _Opnd, _Offs));
		_r -> _Opnd
	else
		true -> islit
	endif
enddefine;

define lconstant Get_fdestin() /* -> (Mode, _Opnd, _Offs, push_res) */;
	lvars instr, _op;
	;;; see if can optimise a following pop or store
	fast_front(fast_back(asm_clist)) -> instr;
	if isvector(instr)
	and ((instr!INST_OP ->> _op) == I_POP or _op == I_STORE) then
		;;; move to valof field of I_POP's variable
		instr -> asm_instr;
		Get_structure(_0, DEFER);
		fast_back(asm_clist) -> asm_clist;			;;; erase the I_POP
		_op == I_STORE
	else
		MODE_STACK, _UNUSED, _UNUSED, false
	endif
enddefine;


;;; --- ASSEMBLER PROCEDURES FOR INDIVIDUAL INSTRUCTIONS -------------------

;;; --- move procedures ----------------------------------------------------
;;;
;;; {MOVEX obj}			push object to userstack
;;; {MOVEX src dest}	move from src to valof(dest)
;;;
;;; I_MOVE		move from valof
;;; I_MOVEQ		move object
;;; I_MOVENUM	move literal number
;;; I_MOVEADDR	move literal address (= I_MOVENUM)
;;;
;;; I_MOVES		duplicate top of stack (dummy arg)
;;; I_POP		pop userstack into valof
;;; I_STORE		copy tos into valof
;;; I_ERASE		pop userstack (dummy arg)

define lconstant Drop_move(defer);
	lvars defer, isnum, sMode, _sOpnd, _sOffs, dMode, _dOpnd, _dOffs, _tmpreg;
	if isinteger(defer) then
		;;; source is a number/ system address
		Get_literal(asm_instr!INST_ARGS[_0])
	else
		Get_structure(_0, defer)
	endif -> (sMode, _sOpnd, _sOffs);
	if asm_instr!V_LENGTH == _2 then
		Flush_tos(true);
		sMode -> tos_Mode; _sOpnd -> _tos_Opnd; _sOffs -> _tos_Offs;
		Decrement_us()
	else
		Get_structure(_1, DEFER) -> (dMode, _dOpnd, _dOffs);
		if tos_Mode == dMode and _tos_Opnd == _dOpnd and _tos_Offs == _dOffs
		then
			Load_tos_reg(get_reg())
		endif;
		if dMode == MODE_REG then
			Load_structure(_dOpnd, (sMode, _sOpnd, _sOffs))
		elseif sMode == MODE_REG then
			Store_structure(_sOpnd, (dMode, _dOpnd, _dOffs))
		else
			get_reg() -> _tmpreg;
			Load_structure(_tmpreg, (sMode, _sOpnd, _sOffs));
			Store_structure(_tmpreg, (dMode, _dOpnd, _dOffs));
			kill_reg(_tmpreg)
		endif
	endif
enddefine;

define I_MOVE	  = Drop_move(% DEFER %)	enddefine;
define I_MOVEQ	  = Drop_move(% DIRECT %)	enddefine;
define I_MOVENUM  = Drop_move(% 0 %)		enddefine;
define I_MOVEADDR = Drop_move(% 1 %)		enddefine;


define I_MOVE_CALLER_RETURN();
	fast_chain(asm_instr!INST_ARGS[_2])		;;; I_MOVE or I_MOVEADDR
enddefine;

	;;; push an unsigned sysint, represented as a pop (big)integer
define I_PUSH_UINT();
	Flush_tos(true);
	MODE_LIT -> tos_Mode;
	Pint_->_uint(asm_instr!INST_ARGS[_0], _-1) -> _tos_Opnd;
	_UNUSED -> _tos_Offs;
	Decrement_us()
enddefine;

define lconstant Do_pop(defer);
	lvars	defer, _reg, _tmpreg,
			(dMode, _dOpnd, _dOffs) = Get_structure(_0, defer);
	if dMode == MODE_REG then
		if defer == DIRECT then
			mishap(0, 'SYSTEM ERROR 1 IN I_POPQ');
		else
			Drop_pop(_dOpnd)
		endif
	else
		Drop_pop_tmp(_NOREG) -> (_reg, _tmpreg);
		Store_structure(_reg, (dMode, _dOpnd, _dOffs));
		kill_reg(_tmpreg)
	endif
enddefine;

	;;; pop top of user stack into valof
define I_POP  = Do_pop(% DEFER %)  enddefine;
	;;; pop into quoted variable
	;;; (only meaningful with a stack lvar containing a runtime ident)
define I_POPQ = Do_pop(% DIRECT %) enddefine;

	;;; copy top of user stack into valof
define I_STORE();
	lvars (dMode, _dOpnd, _dOffs) = Get_structure(_0, DEFER), _tmpreg;
	if tos_Mode then
		Load_tos_reg(if dMode == MODE_REG then _dOpnd else _NOREG endif);
		unless dMode == MODE_REG then
			Store_structure(_tos_Opnd, (dMode, _dOpnd, _dOffs))
		endunless
	else
		if dMode == MODE_REG then
			Drop_MEMf(_OP_ldW, _dOpnd, _us_offs, _:RGusp)
		else
			get_reg() -> _tmpreg;
			Drop_MEMf(_OP_ldW, _tmpreg, _us_offs, _:RGusp);
			Store_structure(_tmpreg, (dMode, _dOpnd, _dOffs));
			_tmpreg -> _dOpnd
		endif;
		MODE_REG -> tos_Mode; _dOpnd -> _tos_Opnd; _UNUSED -> _tos_Offs;
		true -> _tos_iscopy
	endif
enddefine;

	;;; duplicate top of stack
define I_MOVES();
	lvars _tmpreg;
	if tos_Mode then
		Flush_tos(false);
		Decrement_us()
	else
		Drop_pop_tmp(_NOREG) -> (_tmpreg, );
		Decrement_us();
		Drop_push(_tmpreg)							;;; push %t0, %us
	endif
enddefine;

	;;; erase top of user stack
define I_ERASE();
	lvars instr = asm_instr, _tmp;
	if datalength(instr) == 2 and instr!INST_ARGS[_0] then
		;;; fast -- no access necessary
		if tos_Mode == MODE_REG and (_tos_Opnd -> _tmp, IS_TMP_REG(_tmp)) then
			kill_reg(_tmp)
		endif;
		Increment_us()
	else
		Drop_pop(_:RGzero)							;;; pop rzero, (rusp)
	endif
enddefine;

	;;; swap two elements on user stack
define I_SWAP();
	lvars	instr = asm_instr,
			_i = _int(instr!INST_ARGS[_0]), _j = _int(instr!INST_ARGS[_1]),
			_reg0, _reg1, _tmp0, _tmp1, Mode, _Opnd, _Offs, push;

	returnif(_i == _j);
	if _zero(_j) then _i, _j -> (_j, _i) endif;

	if _zero(_i) then
		;;; swapping with tos
		Drop_pop_tmp(_NOREG) -> (_reg0, _tmp0);
		_j _sub _1 -> _j;
		Get_fdestin() -> (Mode, _Opnd, _Offs, push);
		if Mode == MODE_REG then
			if _Opnd == _reg0 then
				Drop_mvreg(_reg0, get_reg() ->> _reg0 ->> _tmp0)
			endif;
			_Opnd, _NOREG
		else
			dup(get_reg())
		endif -> (_reg1, _tmp1);

		if _zero(_j) then
			;;; other one is now tos
			Drop_pop(_reg1);
			Drop_push(_reg0)
		else
			@@(w)[_j] _add _us_offs -> _i;
			Drop_MEMf(_OP_ldW, _reg1, _i, _:RGusp);		;;; ldW reg1, i(rusp)
			Drop_MEMf(_OP_stW, _reg0, _i, _:RGusp);		;;; stW reg0, i(rusp)
			kill_reg(_tmp0)
		endif;

		unless Mode == MODE_REG then
			Store_structure(_tmp1, (Mode, _Opnd, _Offs));
			_tmp1 -> _Opnd
		endunless;
		if push then Drop_push(_Opnd) endif;
		unless _tos_Opnd == _tmp1 and tos_Mode == MODE_REG then
			kill_reg(_tmp1)
		endunless

	else
		@@(w)[_i] _add _us_offs -> _i;
		@@(w)[_j] _add _us_offs -> _j;
		get_reg() -> _tmp0;
		get_reg() -> _tmp1;
		Drop_MEMf(_OP_ldW, _tmp0, _i, _:RGusp);		;;; ldW rt0, i(rusp)
		Drop_MEMf(_OP_ldW, _tmp1, _j, _:RGusp);		;;; ldW rt1, j(rusp)
		Drop_MEMf(_OP_stW, _tmp0, _j, _:RGusp);		;;; stW rt0, j(rusp)
		Drop_MEMf(_OP_stW, _tmp1, _i, _:RGusp);		;;; stW rt1, i(rusp)
		kill_reg(_tmp0);
		kill_reg(_tmp1)
	endif
enddefine;


;;; --- FIELD ACCESS INSTRUCTIONS (used by conskey etc) ---------------------

define lconstant Drop_exptr_deref(exptr, _reg, _dstreg) -> _reg;
	lvars exptr, _reg, _dstreg;
	fast_repeat exptr times
		Drop_MEMf(_OP_ldW, _dstreg, _0, _reg);		;;; ldW _dstreg, (reg)
		_dstreg -> _reg
	endrepeat
enddefine;


;;; --- Do_bitfield ----------------------------------------------------------
;;; deal with bit fields from I_POP_FIELD and I_PUSH_FIELD
;;; uses *bfield* routines in amove.s:
;;; Routine			Input						Output
;;; _(s)bfield		%o0 = structure address		%o0 = field value
;;;					%o1 = bit offset
;;;					%o2 = field width
;;; _ubfield		%o0 = structure address
;;;					%o1 = bit offset
;;;					%o2 = field width
;;;					tos = field value
;;; --------------------------------------------------------------------------

define lconstant Do_bitfield(upd);
	lvars
		instr = asm_instr,
		type,		;;; field type specifier
		_size,		;;; size in bits of field (includes sign if needed)
		offset,		;;; popint bit offset or lex ident representing register
		upd,		;;; true if updating structure
		exptr,		;;; integer deref count if external ptr access
		_reg, _tmpreg, subs_opnd;
	;

	instr!INST_ARGS[_0] -> type;
	instr!INST_ARGS[_1] -> _size;
	instr!INST_ARGS[_4] -> offset;
	instr!INST_ARGS[_5] -> exptr;

	Flush_tos(true);
	_int(_size) -> _size;
	if offset then
		;;; fixed-offset access
		Drop_mvlit(_int(offset), _:RGt1)				;;; mov offs, rt1
	else
		;;; subscripted access
		if instr!INST_ARGS[_3] ->> subs_opnd then
			Get_fsource(_3, true, true) -> (_reg, , _tmpreg)
		else
			;;; subscript on user stack - move to rt1 (may be below rec and
			;;; need frigging)
			_NOREG -> _tmpreg;
			Drop_MEMf(_OP_ldW, _:RGt1 ->> _reg,
					if instr!INST_ARGS[_2] then _us_offs else @@(w){_us_offs}++ endif,
								_:RGusp)
		endif;
		Drop_OPfL(_OP_sra, _reg, _:WORD_SHIFT, _:RGt1); ;;; cvt to m/c int
		kill_reg(_tmpreg);
		Drop_OPfL(_OP_subq, _:RGt1, _1, _:RGt1);		;;; correct subscr base
		if _size /== _1 then
			Drop_OPfL(_OP_mulq, _:RGt1, _size, _:RGt1)	;;; scale it
		endif
	endif;
	;;; bit offset now in rt1

	;;; move structure address to rt0
	Get_fsource(_2, RGt0, true) -> (_reg, , _tmpreg);

	;;; adjust user stack pointer if offset came from stack
	unless offset or subs_opnd then @@(w){_us_offs}++ -> _us_offs endunless;

	if exptr then
		;;; deref exptr times
		Drop_exptr_deref(exptr, _reg, _:RGt0) -> _reg
	endif;

	unless _reg == _:RGt0 then Drop_mvreg(_reg, _:RGt0) endunless;
	kill_reg(_tmpreg);

	;;; signed/unsigned/updater routine
	get_reg() -> _tmpreg;
	Drop_mvlit(if upd then				 _ubfield
			   elseif type == t_BIT then _bfield
			   else						 _sbfield
			   endif, _tmpreg);

	Drop_mvlit(_size, _:RGt2);		;;; field width -> rt2
	Drop_us_adjust();
	Drop_jmp(_HINT_jsr, _tmpreg);
	kill_reg(_tmpreg)
enddefine;


;;; --- Do_field -------------------------------------------------------------
;;; deal with non-bit fields from I_POP_FIELD and I_PUSH_FIELD
;;; --------------------------------------------------------------------------

define lconstant Do_field(isaddr, upd);
	lvars
		instr = asm_instr,
		isaddr,		;;; true for address calculation, false otherwise
		type,		;;; code for field type
		_size,		;;; field size in units of type (=1 except for push addr)
		opnd,		;;; operand register or false from stack
		offset,		;;; integer bit offset to field, or register with popint vector index
		upd,		;;; true if updating structure
		exptr,		;;; integer deref count if external ptr access
		basetype,
		_reg, _offs, _tmp1, _tmp2, _dstreg, _n, _tmpreg1, _tmpreg2, subs_opnd;

	instr!INST_ARGS[_0] -> type;
	instr!INST_ARGS[_1] -> _size;
	instr!INST_ARGS[_4] -> offset;
	instr!INST_ARGS[_5] -> exptr;

	unless instr!INST_ARGS[_2] or not(isaddr) or exptr or offset /== 0 then
		;;; pushing address of pop struct at 0 offset -- nothing to do
		return
	endunless;

	_:RGt6 -> _tmp1;
	Get_fsource(_2, _pint(_tmp1), false) -> (_reg, , _tmpreg1);

	if exptr then
		;;; deref _____exptr times
		Drop_exptr_deref(exptr, _reg, _tmp1) -> _reg	;;; returns ______tmp1
	endif;

	unless offset then
		;;; subscripted access
		_:RGt2 -> _tmp2;
		unless instr!INST_ARGS[_3] ->> subs_opnd then
			;;; subscript on stack -- move it to _tmp2
			Drop_pop_tmp(_tmp2) -> (_offs, _tmpreg2)
		endunless
	endunless;

	_:RGt0 -> _dstreg;
	if upd then
		;;; new value from stack -- move to _:RGchain
		Drop_pop(_:RGchain ->> _dstreg)
	endif;
	Flush_tos(true);

	type fi_&& t_BASE_TYPE -> basetype;

	if offset then
		;;; fixed-offset access
		_shift(_int(offset), _-3) -> _offs;		;;; convert bits to bytes
		if isaddr then
			;;; pushing field addr -- stack result
			if _nonzero(_offs) then
				Drop_MEMf(_OP_lda, _:RGt0, _offs, _reg);
				_:RGt0 -> _reg
			endif;
			Drop_push(_reg);
			goto END
		endif

	else
		;;; subscripted access
		if subs_opnd then
			Get_fsource(_3, _pint(_tmp2), true) -> (_offs, , _tmpreg2)
		;;; else subscript was on stack, now in reg ______offs
		endif;

		lconstant
			field_ops_shift =		;;; shift to scale vec index for addresses
				list_assoc_val(%[
					^t_BYTE		0
					^t_SHORT	1
					^t_INT		2
					^t_DOUBLE	3
				]%);

		_int(field_ops_shift(basetype)) -> _n;

		if _size == 1 then
			;;; normal case -- scale popint subscript to vector offset
			_n _sub _:WORD_SHIFT -> _n;
			if _n _slt _0 then
				Drop_OPfL(_OP_sra, _offs, _negate(_n), _tmp2 ->> _offs)
			elseif _n _sgr _0 then
				Drop_OPfL(_OP_sll, _offs, _n, _tmp2 ->> _offs)
			endif;
			@@V_WORDS _sub _shift(1, _n)		;;; note: popint 1
		else
			;;; pushing addr of array of compound external fields
			;;; shifting no good, need to multiply
			_shift(_int(_size), _n) -> _size;
			;;; cvt to m/c int
			Drop_OPfL(_OP_sra, _offs, _:WORD_SHIFT, _tmp2 ->> _offs);
			Drop_OPfL(_OP_subq, _offs, _1, _offs);		;;; correct subscr base
			Drop_OPfL(_OP_mulq, _offs, _size, _offs);	;;; scale it
			_0
		endif -> _n;		;;; fixed adjustment

		Drop_OPfR(_OP_addq, _reg, _offs, _tmp1 ->> _reg); ;;; add base to subs
		kill_reg(_tmpreg2);
		if isaddr then
			;;; pushing field addr -- stack result
			if _nonzero(_n) then Drop_MEMf(_OP_lda, _reg, _n, _reg) endif;
			Drop_push(_reg);
			goto END
		endif;
		_n -> _offs
	endif;

	;;; do access/update
#_IF ##(w)[_1|d] = _1
	if basetype == t_DOUBLE then
		Drop_MEMf(if upd then _OP_stq else _OP_ldq endif, _dstreg, _offs, _reg);
		goto END
	elseif basetype == t_INT then
		Drop_MEMf(if upd then _OP_stl else _OP_ldl endif, _dstreg, _offs, _reg);
		unless upd or type &&/=_0 tv_SIGNED then
			Drop_OPfL(_OP_SHIFT, _FUNC_extll, _dstreg, _0, _dstreg)	;;; unsigned
		endunless;
		goto END
	endif;
#_ELSE
	if basetype == t_INT then
		Drop_MEMf(if upd then _OP_stl else _OP_ldl endif, _dstreg, _offs, _reg);
		goto END
	elseif basetype == t_DOUBLE then
		;;; assume int-range field in double
		Drop_MEMf(if upd then _OP_stq else _OP_ldl endif, _dstreg, _offs, _reg);
		goto END
	endif;
#_ENDIF

	;;; short or byte -- complicated!
	lvars signed = type &&/=_0 tv_SIGNED, short = basetype == t_SHORT;
	if upd then
		lvars (_uinsfn, _umskfn) =
		if short then _FUNC_inswl, _FUNC_mskwl else _FUNC_insbl, _FUNC_mskbl endif;
	elseif signed then
		lvars (_blen, _ashft) = if short then _2, _48 else _1, _56 endif;
	else
		lvars _extfn = if short then _FUNC_extwl else _FUNC_extbl endif;
	endif;

	if offset and not(exptr) then
		;;; base pointer is to a pop record and so _____reg is longword aligned
		;;; -- slightly shorter code
		_offs _bimask _2:11 -> _n;
		_offs _biclear _2:11 -> _offs;
		Drop_MEMf(_OP_ldl, _:RGt1, _offs, _reg);
		if upd then
			Drop_OPfL(_OP_SHIFT, _uinsfn, _dstreg, _n, _dstreg);
			Drop_OPfL(_OP_SHIFT, _umskfn, _:RGt1, _n, _:RGt1);
			Drop_OPfR(_OP_or, _:RGt1, _dstreg, _:RGt1);
			Drop_MEMf(_OP_stl, _:RGt1, _offs, _reg)
		elseif signed then
			_n _add _blen -> _n;
			if _n == _2:100 then
				;;; already signed extended in _:RGt1
				_ashft _sub _32 -> _ashft
			else
				;;; get at top
				Drop_OPfL(_OP_extqh, _:RGt1, _n, _:RGt1)
			endif;
			Drop_OPfL(_OP_sra, _:RGt1, _ashft, _dstreg)
		else
			Drop_OPfL(_OP_SHIFT, _extfn, _:RGt1, _n, _dstreg)
		endif

	else
		;;; general case
		Drop_MEMf(_OP_ldq_u, _:RGt1, _offs, _reg);
		if signed and not(upd) then _offs _add _blen -> _offs endif;
		if _nonzero(_offs) then
			Drop_MEMf(_OP_lda, _:RGt2, _offs, _reg);
			_:RGt2 -> _reg
		endif;
		if upd then
			Drop_OPfR(_OP_SHIFT, _uinsfn, _dstreg, _reg, _dstreg);
			Drop_OPfR(_OP_SHIFT, _umskfn, _:RGt1, _reg, _:RGt1);
			Drop_OPfR(_OP_or, _:RGt1, _dstreg, _:RGt1);
			Drop_MEMf(_OP_stq_u, _:RGt1, _0, _reg)
		elseif signed then
			Drop_OPfR(_OP_extqh, _:RGt1, _reg, _:RGt1);
			Drop_OPfL(_OP_sra, _:RGt1, _ashft, _dstreg)
		else
			Drop_OPfR(_OP_SHIFT, _extfn, _:RGt1, _reg, _dstreg)
		endif
	endif;

END:
	unless _tos_Opnd == _tmpreg1 and tos_Mode == MODE_REG then
		kill_reg(_tmpreg1)
	endunless
enddefine;

	/*	{I_PUSH_FIELD ____type ____size _______operand ____________subs_operand ______offset _____exptr _______cvtpint}
		For fixed-offset access, ______offset is an integer and ____________subs_operand
		is ignored; for subscripted access, ______offset is false and
		____________subs_operand is the subscript operand.
	*/
define I_PUSH_FIELD();
	lvars type = asm_instr!INST_ARGS[_0];
	if type fi_&& t_BASE_TYPE == t_BIT then
		;;; bitfield
		Do_bitfield(false)
	else
		;;; non-bitfield
		Do_field(false, false)
	endif;

	;;; result now in _:RGt0
	if asm_instr!INST_ARGS[_6] then
		;;; make into popint first
		Drop_OPfL(_OP_sWaddW, _:RGt0, 0, _:RGt0)
	endif;
	;;; stack result
	Drop_push(_:RGt0)
enddefine;

	/*	{I_POP_FIELD ____type ____size _______operand ____________subs_operand ______offset _____exptr}
	*/
define I_POP_FIELD();
	lvars type = asm_instr!INST_ARGS[_0];
	if type fi_&& t_BASE_TYPE == t_BIT then
		;;; bitfield -- new value taken off stack
		Do_bitfield(true)
	else
		;;; non-bitfield
		Do_field(false, true)
	endif
enddefine;

	/*	{I_PUSH_FIELD_ADDR ____type ____size _______operand ____________subs_operand ______offset _____exptr}
	*/
define I_PUSH_FIELD_ADDR();
	;;; move the field address to stack
	Do_field(true, false)
enddefine;


;;; --- fast_subscrv, fast_front et cetera ---------------------------------

;;; for fast_front, fast_back, fast_destpair, fast_cont et cetera
define I_FASTFIELD();
	lvars _offs, _reg, _tmp, _tmp2, Mode1, _Opnd1, _Offs1, push1,
			_oneres, Mode2, _Opnd2, _Offs2, push2;

	asm_instr!INST_ARGS[_0] ->> _offs -> _oneres;	;;; false if fast_destpair
	Get_fsource(_1, true, true) -> (_reg, , _tmp);
	Get_fdestin() -> (Mode1, _Opnd1, _Offs1, push1);

	unless _oneres then
		;;; push1 front, push1 (or move) back of pair
		if Mode1 == MODE_STACK or push1 then
			MODE_STACK, _UNUSED, _UNUSED, false
		else
			Get_fdestin()
		endif -> (Mode2, _Opnd2, _Offs2, push2);
		if Mode2 == MODE_REG and _Opnd2 /== _reg then
			_NOREG -> _tmp2;
			Drop_MEMf(_OP_ldW, _Opnd2, @@P_FRONT, _reg)
		else
			get_reg() -> _tmp2;
			Drop_MEMf(_OP_ldW, _tmp2, @@P_FRONT, _reg)
		endif;
		_pint(@@P_BACK) -> _offs
	endunless;

	if Mode1 == MODE_REG then
		kill_reg(_tmp), _NOREG -> _tmp;
		Drop_MEMf(_OP_ldW, _Opnd1, _int(_offs), _reg)
	else
		if IS_NOREG(_tmp) then get_reg() -> _tmp endif;
		Drop_MEMf(_OP_ldW, _tmp, _int(_offs), _reg)
	endif;

	unless _oneres then
		unless Mode2 == MODE_REG and _Opnd2 /== _reg then
			Store_structure(_tmp2, (Mode2, _Opnd2, _Offs2));
			unless Mode2 == MODE_REG then _tmp2 -> _Opnd2 endunless
		endunless;
		if push2 then Drop_push(_Opnd2) endif;
		unless _tos_Opnd == _tmp2 and tos_Mode == MODE_REG then
			kill_reg(_tmp2)
		endunless
	endunless;

	unless Mode1 == MODE_REG then
		Store_structure(_tmp, (Mode1, _Opnd1, _Offs1));
		_tmp -> _Opnd1
	endunless;
	if push1 then Drop_push(_Opnd1) endif;
	unless _tos_Opnd == _tmp and tos_Mode == MODE_REG then
		kill_reg(_tmp)
	endunless
enddefine;

;;; updater for fast_front, fast_back, fast_cont et cetera
define I_UFASTFIELD();
	lvars _offs, _reg, _reg1, _tmp0, _tmp1;
	asm_instr!INST_ARGS[_0] -> _offs;
	Get_fsource(_1, true, false) -> (_reg, , _tmp0);
	Drop_pop_tmp(_NOREG) -> (_reg1, _tmp1);
	Drop_MEMf(_OP_stW, _reg1, _int(_offs), _reg);
	kill_reg(_tmp0);
	kill_reg(_tmp1);
enddefine;

	;;; arg 0 is offset to first vector element (as popint)
	;;; -- subtract popint 1 to account for base 1 subscript and popint bits
define I_FASTSUBV();
	lvars 	_reg, _offs = _int(asm_instr!INST_ARGS[_0]) _sub 1, _tmp0, _tmp1,
			 Mode, _Opnd, _Offs, _reg1, push;
	Get_fsource(_1, true, false) -> (_reg, , _tmp0);
	Drop_pop_tmp(_NOREG) -> (_reg1, _tmp1);
	if IS_NOREG(_tmp1) then
		_tmp0 -> _tmp1, _NOREG -> _tmp0;
		if IS_NOREG(_tmp1) then get_reg() -> _tmp1 endif
	endif;
	Drop_OPfR(_OP_addq, _reg1, _reg, _tmp1);
	kill_reg(_tmp0);
	Get_fdestin() -> (Mode, _Opnd, _Offs, push);
	if Mode == MODE_REG then
		Drop_MEMf(_OP_ldW, _Opnd, _offs, _tmp1)
	else
		Drop_MEMf(_OP_ldW, _tmp1, _offs, _tmp1);
		Store_structure(_tmp1, (Mode, _Opnd, _Offs));
		_tmp1 -> _Opnd
	endif;
	if push then Drop_push(_Opnd) endif;
	unless _tos_Opnd == _tmp1 and tos_Mode == MODE_REG then
		kill_reg(_tmp1)
	endunless
enddefine;

define I_UFASTSUBV();
	lvars _reg, _reg1, _tmp0, _tmp1, _tmp2;
	Get_fsource(_1, true, false) -> (_reg, , _tmp0);
	Drop_pop_tmp(_NOREG) -> (_reg1, _tmp1);
	if IS_NOREG(_tmp1) then get_reg() -> _tmp1 endif;
	Drop_OPfR(_OP_addq, _reg1, _reg, _tmp1);
	kill_reg(_tmp0);
	Drop_pop_tmp(_NOREG) -> (_tmp2, );
	;;; arg 0 is offset to first vector element (as popint)
	;;; -- subtract popint 1 to account for base 1 subscript and popint bits
	Drop_MEMf(_OP_stW, _tmp2, _int(asm_instr!INST_ARGS[_0]) _sub 1, _tmp1);
	kill_reg(_tmp1);
	kill_reg(_tmp2);
enddefine;


;;; --- FAST INTEGER + AND - -------------------------------------------------

	;;; {I_FAST_+-_any ________plusflag ________operand1 ________operand2}
define I_FAST_+-_any();
	lvars	_op, _func, _reglit, _reg, islit, _tmp0, _tmp1,
			Mode, _Opnd, _Offs, push;
	if asm_instr!INST_ARGS[_0] then _OP_addW else _OP_subW endif
													-> (_op, _func);
	Get_fsource(_1, false, false) -> (_reglit, islit, _tmp0);
	if islit then
		_reglit _sub 0 -> _reglit
	else
		if IS_NOREG(_tmp0) then get_reg() -> _tmp0 endif;
		Drop_OPfL(_OP_subW, _reglit, 0, _tmp0 ->> _reglit)
	endif;
	Get_fsource(_2, true, true) -> (_reg, , _tmp1);
	Get_fdestin() -> (Mode, _Opnd, _Offs, push);
	if Mode == MODE_REG then
		dlocal _use_lda_larith = false;		;;; "lda" gives a quad result
		Drop_OPf(_op, _func, _reg, _reglit, _Opnd, islit);
		kill_reg(_tmp0)
	else
		if IS_NOREG(_tmp1) then
			_tmp0 -> _tmp1, _NOREG -> _tmp0;
			if IS_NOREG(_tmp1) then get_reg() -> _tmp1 endif
		endif;
		Drop_OPf(_op, _func, _reg, _reglit, _tmp1, islit);
		kill_reg(_tmp0);
		Store_structure(_tmp1, (Mode, _Opnd, _Offs));
		_tmp1 -> _Opnd
	endif;
	if push then Drop_push(_Opnd) endif;
	unless _tos_Opnd == _tmp1 and tos_Mode == MODE_REG then
		kill_reg(_tmp1)
	endunless
enddefine;


;;; --- CALLING PROCEDURES -------------------------------------------------

lconstant macro USER = 0;	;;; indicates userstack is arg

	;;; call pop procedure
define lconstant Drop_call(defer, _routine);
	lvars defer, _routine, _reg, _tmpreg;
	if _routine then
		get_reg() -> _tmpreg;
		Drop_mvlit(_routine, _tmpreg);
		if _zero(_us_offs) then _NULL -> _lasti_ptr endif;
		_:RGt0
	else
		_:RGpb
	endif -> _reg;
	if defer == USER then
		Drop_pop(_reg)
	else
		Flush_tos(true);
		Load_structure(_reg, Get_structure(_0, defer))
	endif;
	unless _routine then
		get_reg() -> _tmpreg;
		Drop_MEMf(_OP_ldW, _tmpreg, @@PD_EXECUTE, _:RGpb);
	endunless;
	Drop_us_adjust();
	Drop_jmp(_HINT_jsr, _tmpreg);
	kill_reg(_tmpreg)
enddefine;

define I_CALL  = Drop_call(% DEFER, _popenter %) enddefine;	;;; call valof with check
define I_CALLQ = Drop_call(% DIRECT,_popenter %) enddefine;	;;; call pdr with check
define I_CALLP = Drop_call(% DEFER, false %) enddefine;		;;; call valof without check
define I_CALLS = Drop_call(% USER,  _popenter %) enddefine;	;;; call pdr off stack with check
define I_CALLPS= Drop_call(% USER,  false %) enddefine;		;;; call pdr off stack without check

define I_UCALL	= Drop_call(% DEFER, _popuenter %) enddefine;	;;; call valof's updater with checks
define I_UCALLQ	= Drop_call(% DIRECT,_popuenter %) enddefine;	;;; call pdr's updater with checks
define I_UCALLP	= Drop_call(% DEFER, _popuncenter %) enddefine;	;;; call valof's updater without checks
define I_UCALLPQ= Drop_call(% DIRECT,_popuncenter %) enddefine;	;;; call pdr's updater without checks
define I_UCALLS	= Drop_call(% USER,  _popuenter %) enddefine;	;;; call updater from stack with checks
define I_UCALLPS= Drop_call(% USER,  _popuncenter %) enddefine;	;;; call updater from stack without checks


define lconstant call_xoffs();
	Flush_tos(true);
	lvars	_tmp0 = get_reg(), _xoffs = _int(asm_instr!INST_ARGS[_1]),
			(_r1, _r2) = if _neg(_xoffs) then _tmp0, _:RGpb
						 else _:RGpb, _tmp0
						 endif;
	Load_structure(_r1, Get_structure(_0, DIRECT));
	Drop_us_adjust();
	Drop_MEMf(_OP_lda, _r2, _xoffs, _r1);
	Drop_jmp(_HINT_jsr, _tmp0);
	kill_reg(_tmp0)
enddefine;

	;;; call constant procedure without check
define I_CALLPQ();
	lvars p, _xoffs;

	if _asm_pass == 0 then
		asm_instr!INST_ARGS[_0] -> p;
		@@(code){p!PD_EXECUTE, p} -> _xoffs;		;;; offset to code start
		if _xoffs _lteq _16:7FFF then
			;;; within 15-bit displacement -- add offset to pdr to get execute
			;;; (i.e. don't need to bother with PD_EXECUTE)
			Garbage_inst(asm_instr);
			Cons_inst(call_xoffs, p, _pint(_xoffs), 3)
								->> asm_instr -> fast_front(asm_clist);
			chain(call_xoffs)
		endif
	endif;

	chain(DIRECT, false, Drop_call)
enddefine;

	/*	Call system procedure at absolute address
	*/
define I_CALLABS();
	lvars p, instr, _x, _xoffs, _tmp0;

	if _asm_pass == 0 then
		asm_instr -> instr;
		instr!INST_ARGS[_0] -> p;
		p!PD_EXECUTE -> _x;
		@@(code){_x, p} -> _xoffs;			;;; offset to code start

#_IF ##(w)[_1|d] = _1
		define lconstant call_ldl();
			Flush_tos(true);
			lvars	_tmp0 = get_reg(),
					_offs = _int(asm_instr!INST_ARGS[_0]) _add _nplit_offs;
			Drop_MEMf(_OP_ldl, _tmp0, _offs, _:RGpb);			;;; exec addr
			Drop_MEMf(_OP_ldl, _:RGpb, @@(i){_offs}++, _:RGpb);	;;; pdr
			Drop_us_adjust();
			Drop_jmp(_HINT_jsr, _tmp0);
			kill_reg(_tmp0)
		enddefine;

		unless _xoffs _lteq _16:7FFF
		or BIG_LITERAL(_x) or BIG_LITERAL(p) then
			;;; can combine two 32-bit literals into 1 quadword
			_pint(Np_literal_offset(
					_shift(p, _32) _biset (_x _bimask _16:FFFFFFFF)))
							-> instr!INST_ARGS[_0];
			fast_chain(call_ldl ->> instr!INST_OP)
		endunless;
#_ELSE
		_pint(Np_literal_offset(_x)) -> _x;
#_ENDIF

		Garbage_inst(instr);
		Cons_inst(I_CALLABS, _x, p, 3) ->> instr ->> asm_instr
										-> fast_front(asm_clist);
		if _xoffs _lteq _16:7FFF then
			;;; within 15-bit displacement -- add offset to execute to get pdr
			_pint(_negate(_xoffs)) -> instr!INST_ARGS[_1];
			fast_chain(call_xoffs ->> instr!INST_OP)
		else
			_pint(Np_literal_offset(p)) -> instr!INST_ARGS[_1]
		endif
	endif;

	Flush_tos(true);
	get_reg() -> _tmp0;
	Load_structure(_tmp0, Get_structure(_0, DIRECT));	;;; exec addr
	Load_structure(_:RGpb, Get_structure(_1, DIRECT));	;;; pdr
	Drop_us_adjust();
	Drop_jmp(_HINT_jsr, _tmp0);
	kill_reg(_tmp0)
enddefine;


	;;; {I_CHAIN_REG _________reg-ident}
	;;; chain procedure in reg
define I_CHAIN_REG();
	lvars _reg = _int(Is_register(asm_instr!INST_ARGS[_0]));
	Flush_tos(true);
	Drop_MEMf(_OP_ldW, _:RGt0, @@PD_EXECUTE, _reg);		;;; ldW rt0, (reg)
	Drop_mvreg(_reg, _:RGpb);							;;; mov reg, rpb
	Drop_us_adjust();
	Drop_jmp(_HINT_jmp, _:RGt0)							;;; jmp (rt0)
enddefine;


;;; --- SUBROUTINE CALLS ----------------------------------------------------

define I_CALLSUB();
	Flush_tos(true);
	Drop_callsub(asm_instr!INST_ARGS[_0])
enddefine;

define I_CALLSUB_REG();
	lvars _addr, _reg;
	Flush_tos(true);
	fast_front(asm_instr!INST_ARGS[_0]) -> _addr;
	if Is_register(_addr) ->> _reg then
		Drop_us_adjust();
		Drop_jmp(_HINT_jsr, _int(_reg))		;;; jsr (reg)
	else
		Drop_callsub(_addr)					;;; jsr routine
	endif
enddefine;

	;;; chain subroutine
define I_CHAINSUB();
	Flush_tos(true);
	Drop_mvlit(asm_instr!INST_ARGS[_0], _:RGt0);
	if _zero(_us_offs) then _NULL -> _lasti_ptr endif;
	;;; restore caller's pb from SF_OWNER
	Drop_MEMf(_OP_ldW, _:RGpb, @@SF_OWNER, _:RGsp);
	Drop_us_adjust();
	Drop_jmp(_HINT_jmp, _:RGt0)
enddefine;


;;; --- CONDITIONALS -------------------------------------------------------

;;; {I_IF_opt _____label _____is_so I_BRCOND _______operand}
;;; {I_BOOL_opt _____label _____is_so I_BRCOND _______operand}
;;; _____is_so is true for for ifso & and, false for ifnot & or

define I_BRCOND(_ifso, _ifnot, _reg, is_so, _broffset);
	lvars _ifso, _ifnot, _reg, is_so, _broffset;
	Flush_tos(true);
	Drop_us_adjust();
	Drop_BRAf(if is_so then _ifso else _ifnot endif, _reg, _broffset)
enddefine;

	;;; call the I_BRCOND routine for an I_IF_opt, I_BOOL_opt or I_IF_CMP
	;;; apply the conditional branch routine in arg2
	;;; (branch routine is always I_BRCOND)
define lconstant Drop_if(/*_ifso, _ifnot, _reg,*/ instr);
	lvars instr;
	chain((), instr!INST_ARGS[_1], _int(fast_front(instr!INST_ARGS[_0])),
										I_BRCOND)
enddefine;

define I_IF_opt();
	lvars _reglit, islit, _tmpreg;
	;;; compare input operand or stack pop with false
	Get_fsource(_3, false, true) -> (_reglit, islit, _tmpreg);
	Drop_OPf(_OP_cmpeq, _:RGfalse, _reglit, _:RGt0, islit);
	Drop_if(_OP_blbc, _OP_blbs, _:RGt0, asm_instr);		;;; ifso, ifnot
	kill_reg(_tmpreg)
enddefine;

define I_BOOL_opt();
	lvars	_reglit, islit, _tmpreg,
			_was_tos = tos_Mode, _was_copy = _tos_iscopy;
	Get_fsource(_3, false, true) -> (_reglit, islit, _tmpreg);
	if asm_instr!INST_ARGS[_3] or (_was_tos and not(_was_copy)) then
		;;; push the explicit operand
		if islit then
			if IS_NOREG(_tmpreg) then get_reg() -> _tmpreg endif;
			Drop_mvlit(_reglit, _tmpreg ->> _reglit)
		endif;
		Drop_push(_reglit);
		Flush_tos(false);
		false -> tos_Mode
	else
		;;; undo the stack pop
		--@@(w){_us_offs} -> _us_offs
	endif;
	Drop_OPfR(_OP_cmpeq, _:RGfalse, _reglit, _:RGt0);	;;; compare with false
	kill_reg(_tmpreg);
	Drop_if(_OP_blbc, _OP_blbs, _:RGt0, asm_instr);		;;; or, and
	Increment_us()									;;; erase top of stack
enddefine;

	;;; {I_IF_CMP _________routine _________operand-1 _________operand-2 I_IF_opt______-instr}
define I_IF_CMP();
	lvars instr = asm_instr, _type, islit, _reg, _reglit, _tmp1, _tmp2;

	;;; branch conditions are used _________operand-2 op _________operand-1

	lconstant
		compare_ops =
			[%	nonop _eq,		_pint(_FUNC_cmpeq), true,
				nonop _neq,		_pint(_FUNC_cmpeq), false,
				nonop _sgr,		_pint(_FUNC_cmple), false,
				nonop _sgreq,	_pint(_FUNC_cmplt), false,
				nonop _slt,		_pint(_FUNC_cmplt), true,
				nonop _slteq,	_pint(_FUNC_cmple), true,
			%];

	Get_fsource(_1, false, false) -> (_reglit, islit, _tmp1);
	Get_fsource(_2, true, true) -> (_reg, , _tmp2);

	;;; get compare op for subroutine
	fast_back(fast_lmember(instr!INST_ARGS[_0], compare_ops)) -> _type;

	Drop_OPf(_OP_ARITH, _int(fast_front(_type)), _reg, _reglit, _:RGt0,
														islit);
	kill_reg(_tmp1);
	kill_reg(_tmp2);

	;;; do the I_IF_opt
	Drop_if(if fast_front(fast_back(_type)) then _OP_blbs, _OP_blbc
			else _OP_blbc, _OP_blbs
			endif,
			_:RGt0,
			instr!INST_ARGS[_3])
enddefine;

	/*	{I_IF_TAG _________routine _______operand ______________I_IF_opt-instr}
		where _________routine is _issimple or _isinteger
	*/
define I_IF_TAG();
	lvars instr = asm_instr, _reg, _tmpreg;
	Get_fsource(_1, true, true) -> (_reg, , _tmpreg);
	if instr!INST_ARGS[_0] == _issimple then
		_OP_blbs, _OP_blbc, _reg		;;; ifso, ifnot, reg
	else
		Drop_OPfL(_OP_and, _reg, _2:10, _:RGt0);
		_OP_bne, _OP_beq, _:RGt0		;;; ifso, ifnot, rt0
	endif;
	Drop_if((), instr!INST_ARGS[_2]);
	kill_reg(_tmpreg)
enddefine;


;;; --- JUMPS --------------------------------------------------------------

	;;; branch instruction of standard size -- called from I_GOTO_opt
define I_BR_std(_broffset, _argnum);
	lvars _broffset, _argnum;
	if _nonzero(_us_offs) then
		mishap(0, 'SYSTEM ERROR 1 IN I_BR_std');
	endif;
	Drop_BRAf(_OP_br, _:RGzero, _broffset)
enddefine;

	;;; general branch instruction -- called from I_GOTO_opt
define I_BR(_broffset, _argnum);
	lvars _broffset, _argnum;
	Flush_tos(true);
	Drop_us_adjust();
	Drop_BRAf(_OP_br, _:RGzero, _broffset)
enddefine;

	/*	{I_SWITCH_base __________label-list __________else-label _______operand ____________base-integer}
		Plant a computed jump.
		If there's no else label, leave arg on stack for error message
	*/
define I_SWITCH_base();
	lvars	lablist, elselab, base, _laboffset, _save, _reg, _tmp0,
			_saveoffs, _argreg;
	asm_instr!INST_ARGS[_0] -> lablist;
	asm_instr!INST_ARGS[_1] -> elselab;
	asm_instr!INST_ARGS[_3] -> base;
	listlength(lablist) -> _laboffset;

	Get_fsource(_2, true, true) -> (_reg, , _tmp0);
	Drop_us_adjust();
	_reg -> _argreg;
	unless 0 fi_<= base and base fi_<= 4 then
		if IS_NOREG(_tmp0) then
			get_reg() -> _tmp0
		elseunless elselab then
			Drop_mvreg(_reg, get_reg() ->> _argreg)
		endif;
		Drop_OPfL(_OP_subq, _reg, base _sub 0, _tmp0 ->> _reg);
		0 -> base
	endunless;

	Drop_OPfL(_OP_cmpule, _reg, _laboffset fi_+ base fi_- 1, _:RGt0);
												;;; cmpule reg, offs, rt0
	Drop_fwd_BRAf() -> _save;					;;; blbc rt0, ENDLAB

	;;; _____reg contains a popint subscript -- jump to
	;;; instr following last planted instr for subscript 0
	;;; The offset from pb depends on strsize (which is unknown on the
	;;; first pass), and we must always plant the same number of words of code,
	;;; so we make it upto 4/5 instrs with nops after the jmp
#_IF ##(w)[_1|d] = _1
	_asm_code_offset _add @@(code)[_5] -> _saveoffs;	;;; max 5 instrs
	Drop_OPfL(_OP_sra, _reg, _1, _:RGt0);				;;; scale down to int
	Drop_OPfR(_OP_addq, _:RGt0, _:RGpb, _:RGt0);
#_ELSE
	_asm_code_offset _add @@(code)[_4] -> _saveoffs;	;;; max 4 instrs
	Drop_OPfR(_OP_addq, _reg, _:RGpb, _:RGt0);
#_ENDIF
	Drop_OPfL(_OP_addq, _:RGt0, _header_offs _add _saveoffs, _:RGt0);
	Drop_jmp(_HINT_jmp, _:RGt0);				;;; n.b. ignores bits 0,1

	;;; then make it up to 4/5 instrs if necessary
	while _asm_code_offset _lt _saveoffs do Drop_nop() endwhile;

	;;; branches for values < ____base
	@@(code)[_int(_laboffset)] -> _laboffset;
	until base == 0 do
		Drop_BRAf(_OP_br, _:RGzero,
				_asm_code_offset _add @@(code)[_int(base)] _add _laboffset);
		base fi_- 1 -> base
	enduntil;
	;;; branches for values ____base to listlength(_______lablist)+____base-1
	until lablist == [] do
		fast_front(destpair(lablist) -> lablist) -> _laboffset;
		Drop_BRAf(_OP_br, _:RGzero, _int(_laboffset))
	enduntil;

	;;; ENDLAB:
	Set_fwd_BRAf(_OP_blbc, _:RGt0, _save);			;;; blbc rt0, endlab
	unless elselab then Drop_push(_argreg) endunless;
	unless _tos_Opnd == _tmp0 and tos_Mode == MODE_REG then
		kill_reg(_tmp0)
	endunless
enddefine;

	;;; {I_PLOG_IFNOT_ATOM __________fail-label I_BRCOND}
define I_PLOG_IFNOT_ATOM();
	;;; apply the conditional branch routine in arg1 (always I_BRCOND)
	I_BRCOND(	_OP_blbc, _OP_blbs,		;;; ifso, ifnot
				_:RGt1,
				true,					;;; select blbc (i.e. not equal)
				_int(fast_front(asm_instr!INST_ARGS[_0])))
enddefine;

	;;; {I_PLOG_TERM_SWITCH __________fail-label I_BRCOND _________var-label I_BRCOND}
	;;; after call to plog switch routine, switch afterwards to _________var_label,
	;;; __________fail_label or drop thru.
define I_PLOG_TERM_SWITCH();
	lvars instr = asm_instr;
	;;; branch if rt1 less than zero to _________var-label
	I_BRCOND(	_OP_blt, _OP_bge,		;;; ifso, ifnot
				_:RGt1,
				true,					;;; select blt
				_int(fast_front(instr!INST_ARGS[_2])));

	;;; branch if rt1 bit0 zero to __________fail-label
	I_BRCOND(	_OP_blbc, _OP_blbs,		;;; ifso, ifnot
				_:RGt1,
				true,					;;; select blbc
				_int(fast_front(instr!INST_ARGS[_0])))
	;;; else fall thru for pair/term case (push on arg_reg_0 follows)
enddefine;

define I_CHECK();
	lvars _save_ok, instr;
	Flush_tos(true);

	Drop_MEMf(_OP_ldW, _:RGt0, _SVB_OFFS(_userlim), _:RGsvb);	;;; ldW rt0, _userlim
	Drop_MEMf(_OP_ldW, _:RGt1, _SVB_OFFS(_trap), _:RGsvb);	;;; ldW rt1, _trap
	Drop_OPfR(_OP_cmpult, _:RGusp, _:RGt0, _:RGt0);	;;; cmpult rusp, rt0, rt0
	Drop_us_adjust();
	Drop_OPfR(_OP_or, _:RGt0, _:RGt1, _:RGt0);			;;; or rt0, rt1, rt0

	if isvector(fast_front(fast_back(asm_clist)) ->> instr)
	and (instr!INST_OP == I_GOTO or instr!INST_OP == I_GOTO_opt)
	and instr!INST_ARGS[_1] == I_BR
	then
		;;; followed by unconditional branch
		Drop_BRAf(_OP_blbc, _:RGt0, _int(fast_front(instr!INST_ARGS[_0])));
		Drop_callsub(_checkall)						;;; jsr _checkall
	else
		Drop_fwd_BRAf() -> _save_ok;				;;; blbc rt0, OKLAB
		Drop_callsub(_checkall);					;;; jsr _checkall
		Set_fwd_BRAf(_OP_blbc, _:RGt0, _save_ok)	;;; OKLAB:
	endif
enddefine;

	;;; return user stack length
define I_STACKLENGTH();
	lvars _tmp0, _tmp1, Mode, _Opnd, _Offs, push;
	Flush_tos(true);
	get_reg() -> _tmp0;
	Drop_MEMf(_OP_ldW, _tmp0, _SVB_OFFS(_userhi), _:RGsvb);	;;; ldW rt0, _userhi
	get_reg() -> _tmp1;
	Drop_OPfL(_OP_subq, _:RGusp, 0 _sub _us_offs, _tmp1);
	Get_fdestin() -> (Mode, _Opnd, _Offs, push);
	if Mode == MODE_REG then
		Drop_OPfR(_OP_subW, _tmp0, _tmp1, _Opnd)
	else
		Drop_OPfR(_OP_subW, _tmp0, _tmp1, _tmp1);
		Store_structure(_tmp1, (Mode, _Opnd, _Offs));
		_tmp1 -> _Opnd
	endif;
	kill_reg(_tmp0);
	if push then Drop_push(_Opnd) endif;
	unless _tos_Opnd == _tmp1 and tos_Mode == MODE_REG then
		kill_reg(_tmp1)
	endunless
enddefine;

	;;; set user stack length to saved stacklength + nresults
define I_SETSTACKLENGTH();
	lvars sl, nr, _save, _reg, _tmp0;
	asm_instr!INST_ARGS[_0] -> sl;
	asm_instr!INST_ARGS[_1] -> nr;
	if nr then
		Get_fsource(_0, true, true) -> (_reg, , _tmp0);
		Drop_MEMf(_OP_ldW, _:RGt0, _SVB_OFFS(_userhi), _:RGsvb);
		Drop_us_adjust();
		if IS_NOREG(_tmp0) then get_reg() -> _tmp0 endif;
		Drop_OPfL(_OP_addq, _reg, @@(w)[_int(nr)] _sub 0, _tmp0);
		Drop_OPfR(_OP_subq, _:RGt0, _tmp0, _:RGt0);
		Drop_OPfR(_OP_subq, _:RGt0, _:RGusp, _:RGt1);
		Drop_fwd_BRAf() -> _save;
		Drop_callsub(_setstklen_diff);		;;; stklen in rt0, diff in rt1
		Set_fwd_BRAf(_OP_beq, _:RGt1, _save);
		kill_reg(_tmp0)
	else
		Flush_tos(true);
		Drop_callsub(_setstklen)
	endif
enddefine;

;;; convert pop11 result to lisp result i.e. replace false with []
define I_LISP_TRUE();
	lvars	_save, _tmp, _reg, updating,
			(Mode, _Opnd, _Offs, push) = Get_fdestin();
	(Mode == MODE_STACK and (not(tos_Mode) or _tos_iscopy))
	or (Mode == tos_Mode and _Opnd == _tos_Opnd and _Offs == _tos_Offs)
					-> updating;

	if Mode == MODE_REG then
		_Opnd -> _reg, _NOREG -> _tmp;
		Drop_pop(_reg)
	else
		Drop_pop_tmp(_NOREG) -> (_reg, _tmp);
		if IS_NOREG(_tmp) then
			get_reg() -> _tmp;
			Drop_mvreg(_reg, _tmp ->> _reg)
		endif
	endif;

	Drop_OPfR(_OP_cmpeq, _reg, _:RGfalse, _:RGt0);	;;; cmpeq reg, rfalse, rt0
	Drop_fwd_BRAf() -> _save;						;;; blbc rt0, NOTFALSE
	Drop_mvlit([], _reg);							;;; mov c.nil, reg
	if updating and Mode /== MODE_REG then
		Store_structure(_reg, (Mode, _Opnd, _Offs));
		Flush_tos(false)
	endif;

	Set_fwd_BRAf(_OP_blbc, _:RGt0, _save);			;;; NOTFALSE:

	if not(updating) and Mode /== MODE_REG then
		Store_structure(_reg, (Mode, _Opnd, _Offs))
	endif;
	if push then Drop_push(_reg) endif;
	unless _tos_Opnd == _tmp and tos_Mode == MODE_REG then
		kill_reg(_tmp)
	endunless
enddefine;


;;; --- PROLOGUE, EPILOGUE -------------------------------------------------

define lconstant Drop_sr_regs(_opcode, _spoffs) -> _spoffs;
	lvars _spoffs;
	dlvars _opcode;

	define lconstant sr_regs(_n, rlist, _spoffs) -> _spoffs;
		lvars rlist = fast_back(rlist), _n = _pint(_n), _spoffs, _reg;
		while _n fi_> 0 do
			_int(Is_register(fast_subscrl(_n,rlist))) -> _reg;
			Drop_MEMf(_opcode, _reg, _spoffs, _:RGsp);
			@@(w){_spoffs}++ -> _spoffs;
			_n fi_- 1 -> _n
		endwhile
	enddefine;

	;;; pop regs
	sr_regs(_Npopreg, asm_pop_registers, _spoffs) -> _spoffs;
	;;; nonpop regs
	sr_regs(_Nreg _sub _Npopreg, asm_nonpop_registers, _spoffs) -> _spoffs
enddefine;

	;;; generate code to construct stack frame
define I_CREATE_SF();
	lvars _n, _spoffs = @@SF_LOCALS[_Nstkvars], _pboffs;

	init_reg();					;;; initialise temp registers
	_0 -> _us_offs;				;;; zero user stack offset
	_NULL -> _lasti_ptr;
	false -> tos_Mode;

	;;; decrement sp for stack frame
	Drop_MEMf(_OP_lda, _:RGsp, @@(w)-[_Nframewords], _:RGsp);

	;;; save my address
	Drop_MEMf(_OP_stW, _:RGpb, @@SF_OWNER, _:RGsp);

	;;; initialise pop on-stack lvars to popint 0
	unless _zero(_Npopstkvars ->> _n) then
		Drop_mvlit(0, _:RGt0);
		_spoffs _sub @@(w)[_n] -> _spoffs;
		until _zero(_n) do
			Drop_MEMf(_OP_stW, _:RGt0, _spoffs, _:RGsp);
			@@(w){_spoffs}++ -> _spoffs;
			_n _sub _1 -> _n
		enduntil
	endunless;

	;;; save dynamic local identifiers
	_Nlocals -> _n;
	@@PD_TABLE[_n] -> _pboffs;		;;; dlocal table limit offset
	while _n _gr _0 do
		--@@(w){_pboffs} -> _pboffs;
		Drop_MEMf(_OP_ldW, _:RGt0, _pboffs, _:RGpb);
		if _n == _1 then
			Drop_MEMf(_OP_ldW, _:RGt0, @@ID_VALOF, _:RGt0);
			Drop_MEMf(_OP_stW, _:RGt0, _spoffs, _:RGsp);
			_n _sub _1 -> _n
		else
			--@@(w){_pboffs} -> _pboffs;
			Drop_MEMf(_OP_ldW, _:RGt1, _pboffs, _:RGpb);
			Drop_MEMf(_OP_ldW, _:RGt0, @@ID_VALOF, _:RGt0);
			Drop_MEMf(_OP_ldW, _:RGt1, @@ID_VALOF, _:RGt1);
			Drop_MEMf(_OP_stW, _:RGt0, _spoffs, _:RGsp);
			@@(w){_spoffs}++ -> _spoffs;
			Drop_MEMf(_OP_stW, _:RGt1, _spoffs, _:RGsp);
			_n _sub _2 -> _n
		endif;
		@@(w){_spoffs}++ -> _spoffs
	endwhile;

	;;; save dlocal registers
	Drop_sr_regs(_OP_stW, _spoffs) -> _spoffs;

	;;; save caller's return address
	Drop_MEMf(_OP_stW, _:RGret, _spoffs, _:RGsp)
enddefine;

	;;; generate code to unwind stack frame
	;;;
	;;; This code MUST NOT use chain_reg (_:RGchain), because this can contain
	;;; the saved return address after an I_CHAINS
define I_UNWIND_SF();
	lvars _n, _spoffs = @@SF_LOCALS[_Nstkvars], _pboffs;
	Flush_tos(true);

	;;; restore dynamic local identifiers
	_Nlocals -> _n;
	@@PD_TABLE[_n] -> _pboffs;			;;; dlocal table limit offset
	while _n _gr _0 do
		--@@(w){_pboffs} -> _pboffs;
		Drop_MEMf(_OP_ldW, _:RGt0, _pboffs, _:RGpb);
		Drop_MEMf(_OP_ldW, _:RGt1, _spoffs, _:RGsp);
		@@(w){_spoffs}++ -> _spoffs;
		if _n == _1 then
			Drop_MEMf(_OP_stW, _:RGt1, @@ID_VALOF, _:RGt0);
			_n _sub _1 -> _n
		else
			--@@(w){_pboffs} -> _pboffs;
			Drop_MEMf(_OP_ldW, _:RGt2, _pboffs, _:RGpb);
			Drop_MEMf(_OP_ldW, _:RGt3, _spoffs, _:RGsp);
			@@(w){_spoffs}++ -> _spoffs;
			Drop_MEMf(_OP_stW, _:RGt1, @@ID_VALOF, _:RGt0);
			Drop_MEMf(_OP_stW, _:RGt3, @@ID_VALOF, _:RGt2);
			_n _sub _2 -> _n
		endif
	endwhile;

	;;; restore dlocal registers
	Drop_sr_regs(_OP_ldW, _spoffs) -> _spoffs;

	;;; restore caller's return address
	Drop_MEMf(_OP_ldW, _:RGret, _spoffs, _:RGsp);

	;;; finally, increment sp to erase stack frame
	Drop_MEMf(_OP_lda, _:RGsp, @@(w)[_Nframewords], _:RGsp)
enddefine;

	;;; return to caller
define I_RETURN();
	;;; restore caller's pb from SF_OWNER
	Flush_tos(true);
	Drop_MEMf(_OP_ldW, _:RGpb, @@SF_OWNER, _:RGsp);
	Drop_us_adjust();
	;;; return to _:RGret
	Drop_jmp(_HINT_ret, _:RGret)
enddefine;

define I_LABEL();
	Flush_tos(true);
	Drop_us_adjust();
	_NULL -> _lasti_ptr;
	;;; set front of label pair to popint offset from code start PD_EXECUTE
	_pint(_asm_code_offset) -> fast_front(asm_clist)
enddefine;


;;; --- THE ASSEMBLER ------------------------------------------------------


;;; ASSEMBLER TO PRODUCE MACHINE CODE IN PROCEDURE RECORD FROM
;;; POP ASSEMBLER INPUT

;;; args: list, list
define Do_consprocedure(codelist, reg_locals) -> pdr;
	dlocal
		_asm_drop_ptr,
		_asm_pass,
		_header_offs,
		_us_offs,
		_nplit_offs,
	;

	lvars
		pdr,			;;; result
		reg_locals,		;;; argument list of local register identifiers
		codelist,		;;; static list of instructions
		_code_offset,	;;; size of code
		_offs,
		_size,			;;; size of procedure record
		_reg_spec,		;;; spec to go in PD_REGMASK
	;

	;;; first pass - calculate instruction offsets
	_0 -> _header_offs;
	_negate(_NONPOP_LIT) -> _nplit_offs;	;;; adjust for _NONPOP_LIT set
	Code_pass(0, codelist) -> _code_offset;

	;;; get offset to start of nonpop literal table
	@@PD_TABLE[_int(listlength(asm_struct_list))] -> _offs;
	_offs _sub _NONPOP_LIT -> _nplit_offs;	;;; adjust for _NONPOP_LIT set

	;;; add size of nonpop literal table to get total header offset
	_offs _add _asm_nplit_size -> _header_offs;

	if _header_offs _gr _16:7FFF then
		;;; PB offsets may overflow -- do another pass
		Code_pass(1, codelist) -> _code_offset
	endif;

	;;; can now calculate total size of procedure and allocate store for it
	_header_offs _add @@(w){_code_offset | code.r} _sub @@POPBASE -> _size;

	;;; register spec is switch offset to within TNP+1 blocks of cascaded
	;;; instructions which store/load the pop registers and then branch to
	;;; store/load the corresponding nonpop registers. Each block is
	;;; TP*2+1 instructions. (See aprocess.s)
	lconstant
		_TNP	 = _int(listlength(asm_nonpop_registers)) _sub _1,
		_TP		 = _int(listlength(asm_pop_registers)) _sub _1,
		_PBLKLEN = _TP _mult _2 _add _1;

	@@(code)[ (_TNP _sub (_Nreg _sub _Npopreg)) _mult _PBLKLEN
					_add
			  (_TP _sub _Npopreg) _mult _2
					_add _3		;;; 3 instructions from pc label to regcode
			] -> _reg_spec;

	;;; get the procedure record with header filled in and struct table
	;;; generated (sets up _asm_drop_ptr)
	Get_procedure(_size, _reg_spec) -> pdr;

	;;; final pass -- generate code
	Code_pass(false, codelist) ->
enddefine;

endsection;		/* $-Sys$-Vm */



/* --- Revision History ---------------------------------------------------
--- John Gibson, Jul  6 1998
		Fixed Do_field to use RGchain instead of RGt0 to hold update value off
		stack (since input pointer can be in RGt0).
--- John Gibson, Nov  4 1996
		# Converted I_PUSH/POP_FIELD(_ADDR) instructions to new format.
		# Improved I_SWAP.
		# Allowed I_ERASE to take optional extra boolean arg to say
		  stack access not necessary (i.e. just increment stack pointer).
--- John Gibson, Oct 24 1996
		Numerous improvements, including caching of top-of-stack operand.
		Now defines I_FAST_+-_any and I_SWITCH_base.
--- John Gibson, Oct 11 1996
		Changed I_ERASE to use RGt0 instead of RGzero (see comment).
--- John Gibson, Sep 27 1996
		Added I_IF_TAG
 */
