/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_arc.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 5th July 1993
;;; File: go_arc.p
;;; Author: J L Cunningham

compile_mode :pop11 +strict;

include XpwPixmap.ph;
uses go_screen_object;
uses go_fillable;
;;; uses go_rotatable;

;;;------------------------------------------------------

;;; INTRO: The go_arc CLASS offers arcs and pies (filled arcs).
;;; INTRO: For consistency with polygons the enclosing bounding box uses the
;;; INTRO: centre of the arc as the centre of the bounding box.
;;; INTRO: The bounding box for a pie-form arc is the same as for a full circle.
;;; INTRO: Therefore the centre of the circle and arc correspond to the
;;; INTRO: (go_xcentre, go_ycentre) location (see also LIB * GO_LOCATED).
;;; INTRO: For more complex shapes see REF * GO_POLYGON and its go_rounding.
;;; INTRO: Note that the go_arc class does not inherit from the go_rotatable
;;; INTRO: class because of a limitation in XLIB where scaling can only
;;; INTRO: happen in the screens X and Y direction, therefore a scaled arc
;;; INTRO: which becomes an ellips cannot be rotated to 45 degrees.


define :class go_arc;
	isa go_fillable /* go_rotatable */ go_screen_object;
	slot stored_go_radius      == 40;
;;; REF: the stored value of the go_radius of the go_arc
	slot stored_go_start_angle ==  0;
;;; REF: the stored value of the go_arc in degrees (right angle)
	slot stored_go_arc         == 90;
;;; REF: the stored value of the go_arc in degrees (right angle)

	on new do procedure() -> arc;
			  lvars arc = apply();
				  ;;; Initialise the size of the bounding box:
				  arc.stored_go_radius -> arc.go_radius;
			  endprocedure;
enddefine;


;;;------------------------------------------------------
;;; SHAPE:

define :method go_start_angle( arc :go_arc );
lvars arc;
;;; REF: go_start_angle( ARC ) -> INTEGER;
	arc.stored_go_start_angle;
enddefine;

define :method updaterof go_start_angle( ar, arc :go_arc );
lvars ar, arc;
;;; REF: INTEGER -> go_start_angle( ARC );
;;; REF: The length of the arc expressed in degrees from the start angle (see
;;; REF: go_start_angle) to the end of the arc. The direction of the arc depends
;;; REF: on the orientation of the axes and is counter-clockwise for a
;;; REF: positive ax combination.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
	go_clear( arc );
	round( ar )  -> arc.stored_go_start_angle;
	go_update_and_draw( arc );
enddefine;

define :method go_arc( arc :go_arc );
lvars arc;
;;; REF: go_arc( ARC ) -> INTEGER;
	arc.stored_go_arc;
enddefine;

define :method updaterof go_arc( ar, arc :go_arc );
lvars ar, arc;
;;; REF: INTEGER -> go_arc( ARC );
;;; REF: The length of the arc expressed in degrees from the start angle (see
;;; REF: go_start_angle) to the end of the arc. The direction of the arc depends
;;; REF: on the orientation of the axes and is counter-clockwise for a
;;; REF: positive ax combination.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
	go_clear( arc );
	round( ar )  -> arc.stored_go_arc;
	go_update_and_draw( arc );
enddefine;

;;;------------------------------------------------------
;;; DIMENSIONS:

define :method go_radius( arc :go_arc );
lvars arc;
;;; REF: go_radius( ARC ) -> INTEGER;
	arc.stored_go_radius;
enddefine;

define :method updaterof go_radius( r, arc :go_arc );
lvars r, arc;
;;; REF: INTEGER -> go_radius( ARC );
;;; REF: The radius in world coordinates of the arc object.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
	;;; relative stored_go_points change
	go_clear( arc );
	round( r ) ->> arc.stored_go_radius;
	* 2        ->> arc.stored_go_bounding_width
			   ->  arc.stored_go_bounding_height;
	go_update_and_draw( arc );
enddefine;


;;;------------------------------------------------------
;;; COORDINATES

define :method go_local_coords( arc :go_arc );
lvars arc;
;;; REF: go_local_coords( ARC ) -> LOCALCOORDLIST;
;;; REF: Returns the arc's starting point, its centre point and its end point.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
;;; REF: LOCALCOORDLIST is the list of local coordinates of the arc.
lvars rad = arc.go_radius, ang1 = arc.go_start_angle, ang2 = arc.go_arc + ang1;
	[% rad*cos(ang1), rad*sin(ang1), 0, 0, rad*cos(ang2), rad*sin(ang2) %]
	->> stored_go_local_coords(arc);
enddefine;


define :method go_screen_coords( pane :go_pane, arc :go_arc );
lvars pane, arc;
	if ( cached_go_coord_list(pane)(arc) ) then
		;;; valid screen coordinates saved
		cached_go_coord_list(pane)(arc);
	else
	lvars scoords  = go_translistout(arc.go_world_coords, pane),
		  cxc      = scoords(3),                      ;;; screen coord centre
		  cyc      = scoords(4),
		  rad      = arc.go_radius,
		  cxscale  = arc.go_xscale * pane.go_xscale,
		  cyscale  = arc.go_yscale * pane.go_yscale,
		  cbwidth  = abs(rad * cxscale),              ;;; screen bounding width
		  cbheight = abs(rad * cyscale),              ;;; screen bounding height
		  ang1     = round( go_start_angle(arc) ),
		  arc1     = round( go_arc(arc) );
		{%
		   [%  round( cxc - cbwidth  ) /* -> x */;
			   round( cyc - cbheight ) /* -> y */;
			   round( cbwidth  * 2 )   /* -> w */;
			   round( cbheight * 2 )   /* -> h */;
			   ;;; Normalise between -360 and 360
			   unless (ang1 == 0) then
				   ang1 fi_mod (sign(ang1) * 360) -> ang1;
			   endunless;
			   unless arc1 == 0 or ( arc1 >= -360 and arc1 <= 360) then
				   arc1 fi_mod (sign(arc1) * 360) -> arc1;
			   endunless;
			   ;;; Flip    none   :     ang1 -> ang1;  arc1 -> arc1;
			   ;;; Flip   up/down :    -ang1 -> ang1; -arc1 -> arc1; == XLIB
			   ;;; Flip left/right: 180-ang1 -> ang1; -arc1 -> arc1;
			   ;;; Flip    both   : 180+ang1 -> ang1;  arc1 -> arc1;
			   if     ( cxscale > 0 ) then
				   if ( cyscale < 0) then
					   ang1 << 6 ;
					   arc1 << 6 ;
				   elseif ( cyscale > 0) then
					   (ang1 * -1) << 6 ;
					   (arc1 * -1) << 6 ;
				   endif;
			   elseif ( cxscale < 0 ) then
				   if ( cyscale < 0) then
					   (180 - ang1) << 6 ;
					   (arc1 * -1) << 6 ;
				   elseif ( cyscale > 0) then
					   (180 + ang1) << 6 ;
					   arc1 << 6 ;
				   endif;
			   endif;
			%];
		   if go_filled(arc) then scoords;
							 else [];
		   endif;
		 %} ->> cached_go_coord_list(pane)(arc);
	endif;
enddefine;

;;;------------------------------------------------------
;;; DRAWING

define :method go_fgdraw( pane :go_pane, arc :go_arc );
lvars pane, arc;
;;; REF: go_fgdraw( PANE, ARC );
;;; REF: Draw the lines of the arc in the foreground colour (see also
;;; REF: REF * GO_COLOURABLE).
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars coordsvector;
	go_screen_coords( pane, arc ) -> coordsvector;

	go_draw_arcs( pane,  coordsvector(1) );
	if go_filled(arc) and not(go_arc(arc) == 360) then
		go_draw_lines( pane,  coordsvector(2), CoordModeOrigin );
	endif;
enddefine;

define :method go_bgdraw( pane :go_pane, arc :go_arc );
lvars pane, arc;
;;; REF: go_bgdraw( PANE, ARC );
;;; REF: Draw the filling of the arc in the background colour (see also
;;; REF: REF * GO_COLOURABLE and REF * GO_FILLABLE).
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars coordsvector;
	go_screen_coords( pane, arc ) -> coordsvector;

	go_draw_filled_arcs( pane, coordsvector(1) );
enddefine;

;;;------------------------------------------------------
;;; CONTAINS

define :method go_contains(mx, my, arc :go_arc );
lvars mx, my, arc;
;;; REF: go_contains( X, Y, ARC ) -> FALSE_OR_CIRCLE;
;;; REF: Returns false if the world-coordinates (X,Y) do not fall on the arc
;;; REF: or in the pie if its filled else it returns the arc object itself.
;;; REF: X and Y are the world coordinates of the test point.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars ang1 = arc.go_start_angle mod 360, arc1 = arc.go_arc, ang2 = ang1 + arc1;
lvars rad = arc.go_radius, sq_dist, mang;
	go_transxyin(mx, my, arc) -> (mx, my);            ;;; local coordinates
	mx*mx + my*my -> sq_dist;                         ;;; distance ** 2
	round(arctan2( mx, my )) -> mang;                 ;;; -180 < arctan2 < 180
	mang mod 360 -> mang;                             ;;;    0 <  mang   < 360
;;; [% mang, ang1, arc1, sqrt(sq_dist), rad %] =>
	if (
			if (arc.go_filled) then sq_dist <= rad * rad ;
			else abs(sq_dist - (rad * rad)) < 64;
			endif;
	   ) and (
		   if (ang2 > 360) then                       ;;; more than horizontal
			   (mang >= ang1) or (mang <= (ang2 -360))
		   elseif (ang2 < 0) then                     ;;; less than horizontal
			   (mang <= ang1) or (mang >= (ang2 +360))
		   else                                       ;;; inside 0-->360 circle
			   (mang >= min(ang1, ang2)) and (mang <= max(ang1, ang2))
		   endif;
	   )
	then arc else false endif;
enddefine;

;;;----------------------------------------------------------------
;;; CONVENIENCE FUNCTION FOR EDITING FACILITIES

define :method go_position_of_start_radial( arc :go_arc ) -> (x, y);
lvars x, y, arc;
;;; REF: go_position_of_start_radial( ARC ) -> (X, Y);
lvars radius = arc.go_radius, ang = arc.go_start_angle;
	go_transxyout( radius*cos( ang ), radius*sin( ang ), arc) -> (x, y);
enddefine;

define :method updaterof go_position_of_start_radial( x, y, arc :go_arc );
lvars x, y, arc;
;;; REF: (X, Y) -> go_position_of_start_radial( ARC );
;;; REF: The world coordinates of the starting point of the arc. This is a
;;; REF: convenience function to convert the position into an angle.
;;; REF: X and Y are the world coordinates of the starting point of the arc.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars ang;
	go_transxyin( x, y, arc) -> (x, y);   ;;; Local coords
	round( sqrt( x**2 + y**2 ) ) -> arc.go_radius;
	arctan2( x, y ) mod 360 -> ang;
	ang -> arc.go_start_angle;
enddefine;


define :method go_position_of_end_radial( arc :go_arc ) -> (x, y);
lvars x, y, arc;
;;; REF: go_position_of_end_radial( ARC ) -> (X, Y);
lvars radius = arc.go_radius, ang = (arc.go_start_angle + arc.go_arc);
	go_transxyout( radius*cos( ang ), radius*sin( ang ), arc ) -> (x, y);
enddefine;

define :method updaterof go_position_of_end_radial( x, y, arc :go_arc );
lvars x, y, arc;
;;; REF: (X, Y) -> go_position_of_end_radial( ARC );
;;; REF: The world coordinates of the end point of the arc. This is a
;;; REF: convenience function to convert the position into an arc.
;;; REF: X and Y are the world coordinates of the end point of the arc.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars ang, old_xcentre, old_ycentre;
	go_transxyin( x, y, arc ) -> (x, y) ;
	round( sqrt( x**2 + y**2 ) ) -> arc.go_radius;
	(arctan2( x, y ) - arc.go_start_angle) mod 360 -> ang;
	ang -> arc.go_arc;
enddefine;


define :method go_make_uneditable( arc :go_arc );
lvars arc;
;;; REF: go_make_uneditable( ARC );
;;; REF: Remove editors if any from the arc object. See also REF * GO_EDITOR.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars pnt, redraw_flag = false;
	for pnt in arc.go_editors do
		go_remove_from( pnt, go_default_pane );
		false -> go_edited_object( pnt );
		true -> redraw_flag;
	endfor;
	if (redraw_flag) then [] -> arc.go_editors; go_redraw( arc ); endif;
enddefine;

vars isgo_circle;   ;;; defined by class go_circle in go_circle.p

define :method go_make_editable( arc :go_arc );
lvars arc, lin;
;;; REF: go_make_editable( ARC );
;;; REF: Add editors to the start, the end and the centre of the arc object.
;;; REF: See also REF * GO_EDITOR.
;;; REF: ARC is an instance of the go_arc class (see REF * GO_ARC).
lvars centre_pnt, pnt1, pnt2;
	;;; remove any pending editors:
	go_make_uneditable( arc );
	;;; create new editors: one in centre and one in the vertex
	[% newgo_edit_point() ->> centre_pnt;         ;;; puts the editor on stack
	   arc -> go_edited_object( centre_pnt );
	   go_add_to( centre_pnt, go_default_pane );
	   newgo_edit_point() ->> pnt1;           ;;; puts the editor on stack
	   arc -> go_edited_object( pnt1 );
	   go_position_of_start_radial -> go_move_absolute( pnt1 );
	   go_add_to( pnt1, go_default_pane );
	   unless ( isgo_circle.isprocedure and arc.isgo_circle ) then
		   newgo_edit_point() ->> pnt2;           ;;; puts the editor on stack
		   arc -> go_edited_object( pnt2 );
		   go_position_of_end_radial -> go_move_absolute( pnt2 );
		   go_add_to( pnt2, go_default_pane );
	   endunless;
	  %] -> arc.go_editors;
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses" not needed because go_arc already defined.


/* --- Revision History --------------------------------------------
 * BR 05/07/93
 *     Added a slot go_editors to avoid the overuse of go_live_object to
 *     also keep the editors within go_make_editable().
 * BR 14/06/93
 *     Changed comments in go_[x/y]radius to allow REALs rather than INTs
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 07/05/93
 *     Split the file go_circle.p into go_arc.p and go_circle.p
 *     Adapted the xscale/yscale for new default value = 1;
 *     Corrected go_contains() for different go_[x/y]scale.
 *     Cleaned up variable declarations.
 * BR 28/04/93
 *     Added a few missing lvars pane; declarations.
 *     Optimised method go_screen_coords() avoiding some multiple calls.
 * BR 22/04/93
 *     Abandoned support for both -CoordModePrevious- & -CoordModeOrigin- and
 *     therefore replaced the GO variable ScreenCoordMode by CoordModeOrigin.
 * BR 08/04/93
 *     Removed direct call to XpwDrawLine[s]() by go_draw_line[s]();
 *     Removed direct call to XpwDrawArcs() by go_draw_arcs();
 *     Removed direct call to XpwFillArcs() by go_draw_filled_arcs();
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * BR 15/02/93
 *     Removed link to OLD.p (edit_stroke)
 * BR 21/12/92
 *     Changed the origin to be the x/go_ycentre rather than x/go_yloc.
 *     Changed the go_make_editable() method to reflect the go_polygon method...
 *     KNOWN BUG !!!!  go_contains() and go_overlaps() don't work correctly
 *                     when go_xscale/go_yscale are not equal.
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 23/11/92
 *     Added in new class of arcs as a more general case of circles. The
 *     go_overlaps() method has not yet been refined..
 * BR 03/11/92
 *     Changed go_contains to return the object rather than true or false...
 * BR 16/10/92
 *     Added argument to go_draw() to indicate pane (removed rc_window)...
 * BR 13/10/92
 *     Seriously changed go_draw() for colors and filling;
 *     Adapted for all scales rc_xscale and rc_yscale;
 *     Made go_radius() active method if visible on screen.
 * BR 05/10/92
 *     Removed isa go_located because of new definition of go_screen_object.
 *     Added argument to make_(in)visible() to indicate pane.
 *     Moved go_overlaps() code from layout_findspace.p here...
 *     Moved go_bounding_width(), go_bounding_height(), go_xcentre() and go_ycentre() to beginning of file.
 *     Removed go_clear() method.
 * BR 03/08/92
 *     first attempt at go_make_editable (make edit_line visible)...
 * BR 20/07/92
 *     made include file from basic object: go_circle, previously defined
 *     in sketch_basics.p
 */
;;; eof
