/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_pane.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 24th Aug 1993

;;; File: go_pane.p
;;; Author: B Rabau (based on original work of J L Cunningham)

compile_mode :pop11 +strict;

uses go_located;
uses go_composite;

/* Extra dependencies further in this file:
uses go_editor;
uses go_xdrag;
 */

vars go_panes = [];   ;;; REF: Global var: all current panes (REF * GO_PANE)
vars go_default_pane; ;;; REF: Global var: last go_pane to be actively used;
;;; REF: Global var: either on creation or on mouse actions.

;;; INTRO: MIXIN go_pane : basic structure for an XpwGraphic related pane
;;; INTRO: This mixin is inherited by all the pane related objects whether they
;;; INTRO: are pure screen objects or connected to X-specific shells.
;;; INTRO: See also TEACH * GO_PANE

;;;----------------------------------------------------------------
vars procedure go_initialise_pane, procedure go_redraw;

define :mixin go_pane;
	isa go_composite go_located go_short_print;
	slot stored_go_window     == false;
;;; REF: The XpwGraphics window (REF * XpwGraphic).
	slot stored_go_edit_mode  == true;
;;; REF: Boolean: whether the pane is in edit or live mode (REF * GO_LIVE).
	slot go_editors           == [];
;;; REF: The editor objects visible in this pane (REF * GO_EDITOR).
	slot go_selected_object   == false;
;;; REF: The selected object (or group) visible in this pane (REF * ???).
	slot stored_go_xscale     == false;
;;; REF: The horizontal compression factor for all objects. This will is on
;;; REF: top of any compression of the objects (REF * GO_LOCATED/go_xscale).
	slot stored_go_yscale     == false;
;;; REF: The vertical compression factor for all objects. This will is on
;;; REF: top of any compression of the objects (REF * GO_LOCATED/go_yscale).
	slot stored_go_xorigin    == 0;
;;; REF: The horizontal origin in screen coordinates from the top-left.
	slot stored_go_yorigin    == 0;
;;; REF: The vertical origin in screen coordinates from the top-left.
	slot go_double_buffer     == false;
;;; REF: Whether or not drawing in this pane is double buffered.
	slot the_go_double_buffer == false;
;;; REF: Whether or not drawing in this pane is double buffered.

	;;; CACHING SCREEN INFO PER OBJECT
	slot cached_go_coord_list  == false;
;;; REF: cached go_screen_coords (per object) on the pane's SCREEN.
	slot cached_go_coord_reg   == false;
;;; REF: cached regions (per object) on the pane's SCREEN.
	slot cached_go_colourtable  == false;
;;; REF: Property to cache colours on this window (REF * GO_COLOURABLE).

	;;; INITIALISATION: OC 5.01
	on cons do procedure(); go_initialise_pane(); endprocedure;
enddefine;

define:method go_initialise_window_pane( pane :go_pane);
lvars pane;
;;; REF: go_initialise_window_pane( PANE );
;;; REF: Execute user initialisations. The default adds a colour caching table
;;; REF: (see REF * GO_COLOURABLE) and adds the new pane to the global vars
;;; REF: "go_panes" (REF * GO_VARS/go_panes). This is executed each time a new
;;; REF: window pane is created. Users can add extra initialisations which will
;;; REF: be executed each time a new pane is created.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	newproperty([],  256, false, "tmparg") -> cached_go_colourtable( pane );
	pane :: go_panes -> go_panes;
enddefine;

define go_initialise_pane( constructor ) -> pane ;
lvars constructor, pane;
;;; REF: go_initialise_pane( [CONS ARGS|PANE] ) -> PANE;
;;; REF: Do internal initialisations.
;;; REF: This routine should not be used by the users of GO. This is not a
;;; REF: method in the current implementation. The user should redefine the
;;; REF: go_initialise_window_pane() method.
;;; REF: The argument is either a go_pane or its constructor and all its
;;; REF: arguments.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	if (constructor.isgo_pane) then constructor -> pane;
	else constructor() -> pane;
	endif;
	newproperty([], 1024, false, "tmparg") -> cached_go_coord_list( pane );
	newproperty([], 1024, false, "tmparg") -> cached_go_coord_reg ( pane );
	[] -> the_go_components( pane );
	[] -> go_editors( pane );

	;;; MAKE IT THE DEFAULT ACTIVE WINDOW (see go_add_to)
	pane -> go_default_pane;

	go_initialise_window_pane(pane);
enddefine;

;;;----------------------------------------------------------------
;;; WINDOW CONNECTION

define :method the_go_window( pane :go_pane ) -> window;
lvars pane , window;
;;; REF: the_go_window( PANE ) -> XPWGRAPHIC_OR_FALSE;
	pane.stored_go_window -> window;
enddefine;

define :method updaterof the_go_window( win_or_false, pane :go_pane );
lvars win_or_false, pane;
;;; REF: XPWGRAPHIC_OR_FALSE -> the_go_window( PANE );
;;; REF: Attached XpwGraphic widget. If none is attached the boolean "false"
;;; REF: used (i.e. updating with false detaches the current XpwGraphic widget
;;; REF: from the given pane).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	win_or_false -> pane.stored_go_window;
enddefine;


;;;----------------------------------------------------------------
;;; CONVENIENCE FUNCTIONS

define :method the_go_pane( pane :go_pane ) -> pane;
lvars pane;
;;; REF: the_go_pane( PANE ) -> PANE;
;;; REF: Return pane in which pane is displayed. The default pane for
;;; REF: displaying a pane is itself (see go_panearea.p).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
enddefine;

define :method go_update( pane :go_pane ); ;;; forget screen coords
lvars pane;
;;; REF: go_update( PANE );
;;; REF: Reinitialise pane. This function assures that all stored screen
;;; REF: coordinates are removed.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars obj;
	clearproperty( cached_go_coord_list(pane) );
enddefine;


define :method go_update_and_draw( pane :go_pane );
lvars pane;
;;; REF: go_update_and_draw( PANE );
;;; REF: Reinitialise and refresh pane. This function assures that all screen
;;; REF: coordinates are recalculated. Uses REF * GO_PANE/go_update
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars obj;
	go_update( pane );
	go_redraw( pane );
enddefine;

;;;----------------------------------------------------------------
;;; MODE

define :method go_edit_mode( pane :go_pane ) -> boolean;
lvars pane , boolean = pane.stored_go_edit_mode;
;;; REF: go_live_mode( PANE ) -> BOOLEAN;
enddefine;

define :method updaterof go_edit_mode( val, pane :go_pane );
lvars val, pane;
;;; REF: BOOLEAN -> go_live_mode( PANE );
;;; REF: Returns or changes the state of the pane (opposite of go_edit_mode).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	val -> pane.stored_go_edit_mode;
	go_redraw( pane );
enddefine;

define :method go_live_mode( pane :go_pane ) -> boolean;
lvars pane , boolean = not( pane.go_edit_mode );
;;; REF: go_live_mode( PANE ) -> BOOLEAN;
enddefine;

define :method updaterof go_live_mode( val, pane :go_pane );
lvars val, pane;
;;; REF: BOOLEAN -> go_live_mode( PANE );
;;; REF: Returns or changes the state of the pane (opposite of go_edit_mode).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	not( val ) -> pane.go_edit_mode;
enddefine;

;;;----------------------------------------------------------------
;;; ORIGIN OF AXES  (see also go_located)

define :method go_xorigin( pane :go_pane ) -> screen_xcoord;
lvars pane , screen_xcoord;
;;; REF: go_xorigin( PANE ) -> INTEGER;
	pane.stored_go_xorigin -> screen_xcoord;
enddefine;

define :method updaterof go_xorigin( screen_xcoord, pane :go_pane );
lvars screen_xcoord, pane;
;;; REF: INTEGER -> go_xorigin( PANE );
;;; REF: Changes horizontal origin of objects in pane
;;; REF: Integer screen coord of pane relative to top-left corner of shell
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	screen_xcoord -> pane.stored_go_xorigin;
	go_update_and_draw( pane );
enddefine;


define :method go_yorigin( pane :go_pane ) -> screen_ycoord;
lvars pane , screen_ycoord;
;;; REF: go_yorigin( PANE ) -> INTEGER;
	pane.stored_go_yorigin -> screen_ycoord;
enddefine;

define :method updaterof go_yorigin( screen_ycoord, pane :go_pane );
lvars screen_ycoord, pane;
;;; REF: INTEGER -> go_yorigin( PANE );
;;; REF: Changes vertical origin of objects in pane.
;;; REF: Integer screen coord of pane relative to top-left corner of shell
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	screen_ycoord -> pane.stored_go_yorigin;
	go_update_and_draw( pane );
enddefine;

;;;----------------------------------------------------------------
;;; SCALING

define :method updaterof go_xscale( scale, pane :go_pane );
lvars scale, pane;
;;; REF: go_xscale( LOCATED ) -> REAL;
;;; REF: Sets the horizontal scaling factor of the given pane. This will
;;; REF: preserve the position of the centre of the pane. It influences
;;; REF: the screen coordinate calculations (See REF * GO_PANE/go_transxyout).
;;; REF: For more on scaling see REF * GO_LOCATED.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	call_next_method( scale, pane );
	go_update_and_draw( pane );
enddefine;

define :method updaterof go_yscale( scale, pane :go_pane );
lvars scale, pane;
;;; REF: go_yscale( LOCATED ) -> REAL;
;;; REF: Sets the vertical scaling factor of the given pane. This will
;;; REF: preserve the position of the centre of the pane. It influences
;;; REF: the screen coordinate calculations (See REF * GO_PANE/go_transxyout).
;;; REF: For more on scaling see REF * GO_LOCATED.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	call_next_method( scale, pane );
	go_update_and_draw( pane );
enddefine;


;;;----------------------------------------------------------------
;;; EVENTS

define :method go_accepts_events( type, raw_event, pane :go_pane ) -> accepts;
lvars type, raw_event, pane, accepts = false;
;;; REF: go_accepts_events( TYPE, RAW_EVENT_DATA, PANE ) -> BOOLEAN;
;;; REF: Returns whether or not this object accepts the event. By default
;;; REF: this is FALSE for a pane.
;;; REF: RAW_EVENT_DATA is a vector (see REF * GO_XACTION/go_expand_event_data)
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
enddefine;

;;;----------------------------------------------------------------
;;; MOUSE ACTIONS (EDIT MODE)

define :method go_edit_drag_action( event_data, pane :go_pane );
lvars event_data, obj;
;;; REF: go_edit_drag_action( event_data, PANE );
;;; REF: When the mouse is dragged on the pane in edit mode, then scroll the
;;; REF: pane if mouse is dragged outside the pane (when it is held down over
;;; REF: an empty spot of the pane).
;;; REF: This will influence the go_xorigin and go_yorigin slots of the pane.
;;; REF: event_data is a vector with mouse event information (REF * GO_XACTION).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars new_x = event_data(1), new_y = event_data(2),
	  w, h, min_x, max_x, min_y, max_y,
	  dir_x = sign(go_xscale(pane)), dir_y = sign(go_yscale(pane));
	go_bounding_box( pane ) -> (min_x, min_y, w, h);
	min_x + w -> max_x;
	min_y + h -> max_y;
	;;; Take negative scaling into account...
	if ( dir_x < 0 ) then (min_x, max_x) -> (max_x, min_x); endif;
	if ( dir_y < 0 ) then (min_y, max_y) -> (max_y, min_y); endif;
	;;; Shift if outside window
	if ( new_x < min_x ) then
		go_xorigin(pane) + (min_x - new_x)*dir_x -> go_xorigin(pane);
	elseif ( new_x > max_x ) then
		go_xorigin(pane) + (max_x - new_x)*dir_x -> go_xorigin(pane);
	endif;
	if ( new_y < min_y ) then
		go_yorigin(pane) + (min_y - new_y)*dir_y -> go_yorigin(pane);
	elseif ( new_y > max_y ) then
		go_yorigin(pane) + (max_y - new_y)*dir_y -> go_yorigin(pane);
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; COLOUR
;;;     caching colours: (see also go_colourable.p)

define :method go_get_colour( colourkey, pane :go_pane ) -> colournumber;
lvars colourkey, pane;
;;; REF: go_get_colour( COLOUR, PANE ) -> INTEGER;
;;; REF: Get the cached index in the colour table of the pane's XpwGraphics
;;; REF: widget or if it doesn't yet exist retrieve the index and cache it.
;;; REF: See also REF * GO_COLOURABLE.
;;; REF: COLOUR is a word or a string describing the colour or the index itself.
;;; REF: Such an index is not screen independent and is unsafe.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars window;
	;;; Get the colourkey from the hash table if it exists, otherwise try to
	;;; get it from the setting the foreground and retrieving the colourkey
	if isnumber(colourkey) then
		fi_check(colourkey, 0, false) -> ;
		colourkey -> colournumber;
		return;
	endif;
	the_go_window( pane ) -> window;
	unless ( cached_go_colourtable( pane )( colourkey ) ->> colournumber ) then
		XptValue(window, XtN foreground);     ;;; remember original colour
		'' sys_>< colourkey -> window("foreground");
		XptValue(window, XtN foreground) -> colournumber;
		-> XptValue(window, XtN foreground);  ;;; reinstate original colour
		colournumber -> cached_go_colourtable( pane )( colourkey );
	endunless;
enddefine;


define lconstant check_the_colour( colour )  -> colourcode;
lvars colour, colourcode;
	;;; Local convenience routine to check the validity of a colour:
	;;; Either a string (converted to a word) or a word or an integer.
	if isstring(colour) then consword(colour);
	elseif isword(colour) then colour;
	else fi_check(colour, 0, false);
	endif  -> colourcode;
enddefine;

;;;----------------------------------------------------------------

define :method go_fgcolour( pane :go_pane ); ;;; default foreground colr
lvars pane;
;;; REF: go_fgcolour( PANE ) -> INTEGER;
;;; REF: Get the foreground colour of the pane. The colour is the index in the
;;; REF: colour table of the pane's XpwGraphics widget.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	XptValue( the_go_window(pane), XtN foreground, "int" );
enddefine;

define :method updaterof go_fgcolour( colour, pane :go_pane);
lvars colour, pane;
;;; REF: COLOUR -> go_fgcolour( PANE );
;;; REF: Set the foreground colour of the pane. The colour is a word or a
;;; REF: string describing the colour or the index in the colour table of the
;;; REF: pane's XpwGraphics widget.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	check_the_colour( colour ) -> pane.stored_go_fgcolour;
	go_get_colour( pane.stored_go_fgcolour, pane )
	-> XptValue( the_go_window(pane), XtN foreground );
	go_redraw( pane );
enddefine;


define :method go_bgcolour( pane :go_pane ); ;;; default background colr
lvars pane;
;;; REF: go_bgcolour( PANE ) -> INTEGER;
;;; REF: Get the background colour of the pane. The colour is the index in the
;;; REF: colour table of the pane's XpwGraphics widget.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	XptValue( the_go_window(pane), XtN background, "int" );
enddefine;

define :method updaterof go_bgcolour( colour,  pane :go_pane);
lvars colour, pane;
;;; REF: COLOUR -> go_bgcolour( PANE );
;;; REF: Set the background colour of the pane. The colour is a word or a
;;; REF: string describing the colour or the index in the colour table of the
;;; REF: pane's XpwGraphics widget.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	check_the_colour( colour ) -> pane.stored_go_bgcolour;
	go_get_colour( pane.stored_go_bgcolour, pane )
	-> XptValue( the_go_window(pane), XtN background );
	go_redraw( pane );
enddefine;

;;;----------------------------------------------------------------
;;; METHODS USING GO_SCREEN_OBJECT:
;;; uses go_screen_object;
vars procedure go_draw;                ;;; DEFINED IN GO_SCREEN_OBJECT
vars procedure go_clear;               ;;; DEFINED IN GO_SCREEN_OBJECT
vars procedure go_visible_now;         ;;; DEFINED IN GO_SCREEN_OBJECT
vars procedure go_live_object;         ;;; DEFINED IN GO_SCREEN_OBJECT
;;;----------------------------------------------------------------

;;;----------------------------------------------------------------
;;; DRAWING

define :method go_redraw( pane :go_pane );
lvars pane;
;;; REF: go_redraw( PANE );
;;; REF: Clean pane and redraw all the visible objects in it. This method
;;; REF: mirrors behaviour of the screen objects (see REF * GO_SCREEN_OBJECT).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars obj;
	go_clear( pane );
	for obj in rev(pane.the_go_components) do
		false -> go_visible_now( obj );
		go_draw( pane, obj );
	endfor;
	for obj in rev(pane.go_editors) do
		false -> go_visible_now( obj );
		go_draw( pane, obj );
	endfor;
enddefine;

;;;----------------------------------------------------------------
;;; FIND ACTION GRABBING OBJECT UNDER THE MOUSE:

define :method go_find_ui_object(x, y, type, raw_event_data, pane :go_pane ) -> found_flag;
lvars x, y, type, raw_event_data, pane, found_flag = true;
;;; REF: go_find_ui_object( X, Y, TYPE, RAW_EVENT_DATA, PANE ) -> BOOLEAN;
;;; REF: Finds the object(s) amongst those visible in the given pane which
;;; REF: accepts interaction at the given world coordinate position. An object
;;; REF: accepts interaction if it is in edit mode but can only accept inter-
;;; REF: action in live mode when it is sensitive (see REF * GO_MOUSE_SENSITIVE
;;; REF: and REF * GO_KEY_SENSITIVE). The latter can be further subdivided if
;;; REF: an object was a group: the first element of the group which is
;;; REF: sensitive will accept the interaction if the group is in live mode,
;;; REF: whereas in edit mode the whole group accepts interaction.
;;; REF: See also REF * GO_SCREEN_OBJECT/go_accepts_events.
;;; REF: The objects if any are added to the global variable go_forward_event_to
;;; REF: (See the global variable REF * GO_PANE/go_forward_event_to and also
;;; REF: the global variable REF * GO_PANE/go_transparent_to_events).
;;; REF: X and Y are integer world coordinates of the test position.
;;; REF: TYPE is one of "mouse" or "keyboard"
;;; REF: RAW_EVENT_DATA is a vector (see REF * GO_XACTION/go_expand_event_data)
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars obj, found_obj, edit_pane = pane.go_edit_mode;

	if ( edit_pane ) then
		;;; In edit mode the editors have priority!
		for obj in pane.go_editors do
			if( go_contains(x, y, obj) ->> found_obj ) then
				go_forward_event_to <> [^found_obj] -> go_forward_event_to;
				returnif( not(go_transparent_to_events) );
				false -> go_transparent_to_events;
			endif;
		endfor;
	endif;
	for obj in pane.the_go_components do
		if obj.isgo_composite and ( obj.go_live_object or not(edit_pane) ) then
			if go_find_ui_object(x, y, type, raw_event_data, obj) then
				returnif( not(go_transparent_to_events) );
				false -> go_transparent_to_events;
			endif;
		elseif ( go_contains(x, y, obj) ->> found_obj ) then
			if ( edit_pane and not( obj.go_live_object ) )
			or ( go_accepts_events( type, raw_event_data, found_obj ) )
			then
				go_forward_event_to <> [^obj] -> go_forward_event_to;
				returnif( not(go_transparent_to_events) );
				false -> go_transparent_to_events;
			endif;
		endif;
	endfor;
	;;; Default:
	go_forward_event_to <> [^pane] -> go_forward_event_to;
enddefine;


;;;----------------------------------------------------------------
;;; MEMBERSHIP FUNCTIONS (CHILDREN)

define :method go_remove_from( obj :go_screen_object, pane :go_pane );
lvars obj, pane;
;;; REF: go_remove_from( LOCATED, PANE );
;;; REF: Removes the screen object from the list of objects visible in the given
;;; REF: pane.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	call_next_method( obj, pane );
	false -> cached_go_coord_list( pane )( obj );
enddefine;


;;;----------------------------------------------------------------
;;; COPY

define :method go_copy_object( pane :go_pane ) -> new_pane;
lvars pane, new_pane;
;;; REF: go_copy_object( PANE ) -> NEW_PANE;
;;; REF: Creates a new pane as an exact copy of the given pane. This also copies
;;; REF: all objects visible in the pane but not the editors.
;;; REF: PANE and NEW_PANE are instances of the go_pane class (REF * GO_PANE).
lvars obj;
	copy( pane ) -> new_pane;
	;;; avoid copying stuff which is not sharable
	go_initialise_pane( new_pane ) ->;
	for obj in rev( pane.the_go_components ) do
		go_add_to( obj, new_pane );
	endfor;
	go_redraw( new_pane );
enddefine;

;;;------------------------------------------------------------------
;;; DESTROY: tentative definition for destroying a pane

define :method go_empty_pane( pane :go_pane );
lvars pane;
;;; REF: go_empty_pane( PANE );
;;; REF: Empties the pane object by removing all objects from it. These
;;; REF: objects themselves are not destroyed since they could be visible in
;;; REF: other panes (see also REF * GO_SCREEN_OBJECT/go_destroy_object).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
lvars scrObj, edsObj;
	go_clear( pane );
	for scrObj in pane.the_go_components do
		go_remove_from( scrObj, pane );
	endfor;
	for edsObj in pane.go_editors do
		go_remove_from( edsObj, pane );
	endfor;
enddefine;

define :method go_destroy_object( pane :go_pane );
lvars pane;
;;; REF: go_destroy_object( PANE );
;;; REF: Destroys the pane object after removing all objects from it. These
;;; REF: objects themselves are not destroyed since they could be visible in
;;; REF: other panes (see also REF * GO_SCREEN_OBJECT/go_destroy_object).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	go_empty_pane( pane );
	delete( pane, go_panes, nonop ==) -> go_panes;
	if (pane == go_default_pane) then
		if go_panes == [] then    false    -> go_default_pane;
						  else go_panes(1) -> go_default_pane;
		endif
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; MULTI-METHODS USING GO_EDITOR:
uses go_editor;
;;;----------------------------------------------------------------

define :method go_is_in( obj :go_editor, pane :go_pane );
lvars obj, pane;
;;; REF: go_is_in( EDITOR, PANE) -> BOOLEAN;
;;; REF: Checks whether or not the object is one of the editors visible in
;;; REF: the given pane. It might or might not be visible by the user at this
;;; REF: point since it could be hidden below other objects or be outside the
;;; REF: visible part of the pane on the screen or even be temporarily cleared.
;;; REF: See also REF * GO_EDITOR.
;;; REF: EDITOR is an instance of the go_editor class (REF * GO_EDITOR).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	if ( lmember(obj, pane.go_editors) ) then true else false endif;
enddefine;

define :method go_remove_from( obj :go_editor, pane :go_pane );
lvars obj, pane;
;;; REF: go_remove_from( EDITOR, PANE );
;;; REF: Removes the screen object from the list of objects visible in the given
;;; REF: pane. It does not attempt to clear it from the screen.
;;; REF: EDITOR is an instance of the go_editor class (REF * GO_EDITOR).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	go_clear( obj );
	delete(pane, obj.go_visible_in, nonop ==) -> obj.go_visible_in;
	delete(obj, pane.go_editors, nonop ==) -> pane.go_editors;
enddefine;

;;; Add to the beginning of the list:
define :method go_add_to( obj :go_editor, pane :go_pane );
lvars obj, pane;
;;; REF: go_add_to( EDITOR, PANE );
;;; REF: Makes the screen object part of the objects visible in the given pane.
;;; REF: It does not attempt to draw it but the object will appear on top of
;;; REF: every other object and editor in the pane when drawn.
;;; REF: EDITOR is an instance of the go_editor class (REF * GO_EDITOR).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
	unless ( go_is_in(obj, pane) ) then
		pane :: (obj.go_visible_in) -> obj.go_visible_in;
	endunless;
	if ( lmember(obj, pane.go_editors) ) then
		pr('Warning: this editor is already visible!\n');
	else
		obj :: (pane.go_editors) -> pane.go_editors;
		go_redraw( obj );
	endif;
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_pane = true;

/* --- Revision History --------------------------------------------
 * BR 24/08/93
 *     Renamed go_add_to_pane() and go_annex_to_pane() to go_add_to() and
 *     go_annex_to(). Also renamed go_remove_from_pane() to go_remove_from().
 * BR 23/06/93
 *     Major redesign of go_find_ui_object() to include event type.
 * BR 21/06/93
 *     Added go_empty_pane() and adapted go_destroy_object() accordingly.
 * AS 13/06/93
 *     Slightly optimised :method go_transxyin(x, y, pane :go_pane)
 * BR 02/06/93
 *     Reversed stacking order in go_copy_object().
 * BR 09/05/93
 *     Replaced calls to go_refresh_pane() with go_redraw().
 * BR 20/04/93
 *     Added default go_edit_drag_action() which allows scrolling if the
 *     mouse is moved outside the pane when it is held down over pane.
 * BR,AR 14/04/93
 *     Removed go_prut() and replaced by a new MIXIN go_short_print
 *     which overwrites the standard print_object() method of OC to
 *     avoid printing out objects which refer to themselves. This is
 *     only necessarry until OC has a objectclass_print_limit vars.
 * BR 06/04/93
 *     Introduced the CLASS go_composite with a "go_is_editable" slot.
 *     This has new go_make_editable()/go_make_uneditable() methods which
 *     are based on a test in: go_find_ui_object() in go_go_pane.p.
 * IR 09/03/93
 *     Added go_initialise_window_pane() method
 * IR 02/02/93
 *     Added number support in go_get_colour() method.
 * BR 01/02/93
 *     Removed all links to "hilitable" mixin.
 *     edit_key() renamed as go_edit_key_action();
 *     go_mouse_event() now obsolete (needs a go_mouse_sensitive go_pane class)
 * IR 26/01/93
 *     corrected header of go_make_invisible() to use pane2 variable.
 * BR 21/01/93
 *     Obsoleted find_occluded() method !? see go_find_occluded_list() in
 *     go_screen_object.p
 * BR 19/01/93
 *     Changed go_internal_redraw() into go_draw(pane,) for go_refresh_pane() method !!
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 09/11/92
 *     Adapted for OC 5.01 : uses initialise;
 * BR 05/11/92
 *     Updated go_find_ui_object() to take advantage of new go_contains method.
 * BR 04/11/92
 *     Removed inheritance from UI_object (now obsolete).
 *     Added new method go_destroy_object()...
 * BR 03/11/92
 *     Changed go_contains to return the object rather than true or false...
 * BR 16/10/92
 *     Added argument to go_safe_region() to indicate pane (removed rc_window)...
 * BR 13/10/92
 *     Added go_editors slot.
 *     Added membership functions go_is_in_pane(), go_add_to_pane(), go_remove_from_pane().
 * BR 05/10/92
 *     Resolved conflicts between places where go_visible_objects was used and defined
 *     for the file go_screen_object.p.
 *     Removed children() methods... (see SKETCH MajorChanges)
 * BR 04/09/92
 *     added go_safe_region() method for go_internal_redraw function. Changed go_draw() into
 *     go_internal_redraw() for the go_refresh_pane() method
 * BR 25/08/92
 *     Changed to go_screen_object for (re-)go_draw method instead because changes in
 *     definitions in go_object.p
 * BR 24/07/92
 *     Made include file: go_pane was originally defined in sketch_basics.p
 */
;;; eof
