/* --- Copyright University of Sussex 1998. All rights reserved. ----------
 > File:            C.hppa/src/syscomp/asmout.p
 > Purpose:         Output code procedures for HP 9000/700 assembler (PA-RISC)
 > Author:          Robert Duncan, Dec  4 1992 (based on HP 9000/300 version) (see revisions)
 */


#_INCLUDE 'common.ph'

section $-Popas;

define global popint(n);
	lvars n;
	(n<<2) || 3;
enddefine;

define global mcint(n);
	lvars n;
	n >> 2;
enddefine;

global constant macro (
	$- ASM_TEXT_STR = '\t.code',
	$- ASM_DATA_STR = '\t.data',
	$- ASM_BYTE_STR = '\t.byte\t\t',
	$- ASM_SHORT_STR= '\t.half\t\t',
	$- ASM_INT_STR  = '\t.word\t\t',
	$- ASM_WORD_STR = ASM_INT_STR,
);

	/*  Labels
	*/

vars
	nextlabel,
;

;;; nextlab:
;;;     generates a sequence of local labels (NB: the very-local HP
;;;     'L$n' labels won't do, because they can't be used for addressing
;;;     local data)

define nextlab();
	dlocal pop_pr_radix = 36;
	'L' >< nextlabel;
	nextlabel fi_+ 1 -> nextlabel;
enddefine;

define lconstant asm_label(word, prefix_char);
	lvars n, len, string, q, r, prefix_char, word;
	lconstant SEP_CHAR = `$`;
	if isstring(word) then word else fast_word_string(word) endif -> string;
	datalength(string) -> len;
	consstring(#|
		prefix_char, SEP_CHAR;
		fast_for n to len do
			go_on dup(fast_subscrs(n, string)) to
			;;; 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6
				c c c c c c c c c c c c c c c c     ;;; Ctrl-A -- Ctrl-P
				c c c c c c c c c c c c c c c c     ;;; Ctrl-Q -- SPACE
				s s s $ s s s s s s s s s s s d     ;;;   !    --   0
				d d d d d d d d d s s s s s s s     ;;;   1    --   @
				A A A A A A A A A A A A A A A A     ;;;   A    --   P
				A A A A A A A A A A s s s s u s     ;;;   Q    --   `
				a a a a a a a a a a a a a a a a     ;;;   a    --   p
				a a a a a a a a a a s s s s c c     ;;;   q    -- \(128)
			else c;

			A: a: d: u :        ;;; alpha, digit, underscore
				nextloop;

			$ :
				if n /== len and fast_subscrs(n fi_+ 1, string) == `-`
				and (n /== 1 or len fi_> 2) then
					;;; replace section separator $- with SEP_CHAR
					-> ;            ;;; erase $
					unless n == 1 then SEP_CHAR endunless;
					n fi_+ 1 -> n;  ;;; skip -
					nextloop
				;;; else drop thru to "s"
				endif;

			s: c:               ;;; other sign, control etc
				;;; represent as 3 decimal digits
				() fi_// 100 -> q -> r;
				q fi_+ `0`;
				r fi_// 10 -> q -> r;
				q fi_+ `0`, r fi_+ `0`;

		endfast_for;
	|#);
enddefine;
;;;
define asm_symlabel     = asm_label(% `c` %) enddefine;
define asm_identlabel   = asm_label(% `i` %) enddefine;
define asm_wordlabel    = asm_label(% `w` %) enddefine;
define asm_wordidlabel  = asm_label(% `z` %) enddefine;
define asm_pdpropslabel = asm_label(% `p` %) enddefine;
define asm_testdeflabel = asm_label(% `t` %) enddefine;

;;; is_global_label:
;;;     <true> if -item- is a label (string) not starting with 'L' or
;;;     'xL'and hence assumed to be non-local. Returns the label if so,
;;;     with any trailing offset adjustment stripped off.

define lconstant is_global_label(item) -> lab;
	lvars item, lab = false, c;
	if isstring(item) and datalength(item) fi_> 1 then
		fast_subscrs(1, item) -> c;
		if isalphacode(c) or c == `_` then
			unless c == `L`
			or c == `x` and datalength(item) fi_> 2
						and fast_subscrs(2, item) == `L`
			then
				item -> lab;
				if (locchar(`-`, 2, lab) ->> c)
				or (locchar(`+`, 2, lab) ->> c)
				then
					substring(1, c fi_- 1, lab) -> lab;
				endif;
			endunless;
		endif;
	endif;
enddefine;

	/*  Register names
	 */
#_IF DEF M_DEBUG

;;; Use Poplog register names for readability
global constant
	asm_reg_name = {

		/* General registers */

		\0          ;;;  0  Zero
		\%r1        ;;;  1  ADDIL/CHAIN_REG
		\%r2        ;;;  2  Scratch
		\%npop4     ;;;  3  Non-pop register 4
		\%npop3     ;;;  4  Non-pop register 3
		\%npop2     ;;;  5  Non-pop register 2
		\%npop1     ;;;  6  Non-pop register 1
		\%npop0     ;;;  7  Non-pop register 0
		\%pop5      ;;;  8  Pop register 5
		\%pop4      ;;;  9  Pop register 4
		\%pop3      ;;; 10  Pop register 3
		\%pop2      ;;; 11  Pop register 2
		\%pop1      ;;; 12  Pop register 1
		\%pop0      ;;; 13  Pop register 0
		\%pzero     ;;; 14  Pop zero (3)
		\%false     ;;; 15  Pop false
		\%svb       ;;; 16  Pointer to special var block
		\%pb        ;;; 17  Procedure base
		\%usp       ;;; 18  User stack pointer
		\%r19       ;;; 19  Scratch
		\%t3        ;;; 20  WK_ADDR_REG_2
		\%t2        ;;; 21  WK_ADDR_REG_1
		\%t1        ;;; 22  WK_REG
		\%arg3      ;;; 23  Argument register 3
		\%arg2      ;;; 24  Argument register 2
		\%arg1      ;;; 25  Argument register 1
		\%arg0      ;;; 26  Argument register 0
		\%dp        ;;; 27  Global data pointer
		\%r28       ;;; 28  Scratch
		\%r29       ;;; 29  Scratch
		\%sp        ;;; 30  Stack pointer
		\%r31       ;;; 31  Return address (external)

		/* Space registers */

		\%sr0       ;;;  0  Scratch
		\%sr4       ;;;  4  Code space ID
		\%sr5       ;;;  5  Data space ID

		/* Floating point registers */

		\%fr8       ;;;  8  Scratch
		\%fr8R      ;;;  8  Right half of %fr8
		\%fr8L      ;;;  8  Left half of %fr8
	},
;

;;; These are Poplog-specific, so have to be defined in each file
lconstant POPLOG_REGISTERS =
	[ 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 ],
;

#_ELSE

;;; Use basic register names
global constant
	asm_reg_name = {

		/* General registers */
		\0  \%r1  \%r2  \%r3  \%r4  \%r5  \%r6  \%r7  \%r8  \%r9  \%r10
		\%r11 \%r12 \%r13 \%r14 \%r15 \%r16 \%r17 \%r18 \%r19 \%r20 \%r21
		\%r22 \%r23 \%r24 \%r25 \%r26 \%r27 \%r28 \%r29 \%r30 \%r31

		/* Space registers */
		\%sr0 \%sr4 \%sr5

		/* Floating point registers */
		\%fr8 \%fr8R \%fr8L

	},
;

#_ENDIF

	/*  External symbols (C library functions etc.)
	*/

;;; extern_symbol_type:
;;;     returns the type -- "code" or "data" -- of an external symbol.
;;;     Given that there's no way of working this out automatically, we
;;;     simply assume that everything's a code symbol unless it's
;;;     entered here explicitly as data.

define lconstant extern_symbol_type =
	newmapping([], 64, false, true);
enddefine;

;;; extern_name_translate:
;;;     translate an external symbol, using the same conventions as the
;;;     C compiler and linker (e.g. add a leading '_', truncate to N
;;;     characters, etc.) Also records all external symbols in the
;;;     extern_symbol_type table.

define global extern_name_translate(lang, symbol, type) -> symbol;
	lvars lang, symbol, type;
	if lang = 'FORTRAN' then uppertolower(symbol) <> '_' -> symbol endif;
	unless extern_symbol_type(symbol) then
		if type = 'data' then "data" else "code" endif
									-> extern_symbol_type(symbol);
	endunless;
enddefine;


	/*  Recording imports and exports from files
	 */

define lconstant new_file_property(item, prop) -> p;
	lvars item, prop, p;
	newanyproperty([], 64, 1, 56, syshash, nonop =, "perm", false, false) -> p;
	p -> prop(item);
enddefine;

;;; file_current_space:
;;;     returns the current space ("code" or "data") in which code is
;;;     being generated

define lconstant file_current_space =
	newproperty([], 8, false, "tmparg");
enddefine;

;;; file_uses:
;;;     returns a property containing global symbols referenced within a
;;;     file

define lconstant file_uses =
	newanyproperty([], 8, false, false, false, false, "tmparg", false,
		new_file_property);
enddefine;

;;; file_defines:
;;;     returns a property containing symbols defined within a file

define lconstant file_defines =
	newanyproperty([], 8, false, false, false, false, "tmparg", false,
		new_file_property);
enddefine;

;;; asm_startcode, asm_startdata:
;;;     switch into the code or data space

define lconstant startspace(name);
	lvars name;
	asmf_printf(name, '\n\t.%p\n');
	name -> file_current_space(asmf_charout);
enddefine;
;;;
define asm_startcode = startspace(% "code" %) enddefine;
define asm_startdata = startspace(% "data" %) enddefine;

;;; asm_startfile, asm_endfile:
;;;     start and end an assembly code file

define asm_startfile(name);
	lvars name;
	asm_startcode();
#_IF DEF POPLOG_REGISTERS
	;;; define Poplog register names
	lvars i;
	for i in POPLOG_REGISTERS do
		asmf_printf(i, asm_reg_name(i+1), '%p\t.reg\t\t%%r%p\n');
	endfor;
#_ENDIF
enddefine;

define asm_endfile();
	asmf_charout(`\n`);
	;;; declare all imported symbols
	lvars procedure defines = file_defines(asmf_charout);
	appproperty(
		file_uses(asmf_charout),
		procedure(lab, flag);
			lvars lab, flag;
			unless defines(lab) then
				asmf_printf(
					extern_symbol_type(lab) or "data",
					lab, '\t.import\t\t%p,%p\n');
			endunless;
		endprocedure);
enddefine;

;;; asm_uselab:
;;;     generates an appropriate high/low pair for a reference to a
;;;     symbol. Also records the usage in the -file_uses- property in
;;;     case it has to be imported.

define global asm_uselab(lab) -> (l_lab, r_lab);
	lvars lab, l_lab, r_lab;
	if extern_symbol_type(lab) == "code" then
		;;; external function -- take its PLABEL
		'LP\'' <> lab -> l_lab;
		'RP\'' <> lab -> r_lab;
	elseif isstartstring('t$', lab) then
		;;; testdef label must have the value 0 or 1, so its high half is
		;;; always zero
		0 -> l_lab;
		'R\'' <> lab -> r_lab;
	else
		'L\'' <> lab -> l_lab;
		'R\'' <> lab -> r_lab;
	endif;
	;;; if the label appears to be non-local, it's a potential import
	if is_global_label(lab) ->> lab then
		true -> file_uses(asmf_charout)(lab);
	endif;
enddefine;

;;; asm_labtype:
;;;     returns the type of a lable ("code" or "data") if known

define global asm_labtype(lab);
	lvars lab;
	file_defines(asmf_charout)(lab);
enddefine;

;;; asm_outlab, asm_outglab:
;;;     plant a (global) label

define lconstant outlab(lab, export);
	lvars lab, export;
	if export then
		;;; all pop symbols are exported as "data" regardless of the
		;;; current space
		asmf_printf(lab, '\t.export\t\t%p,data\n');
	endif;
	asmf_printf(lab, '%p\n');
	;;; record the definition
	file_current_space(asmf_charout) -> file_defines(asmf_charout)(lab);
enddefine;
;;;
define asm_outlab  = outlab(% false %) enddefine;
define asm_outglab = outlab(% true  %) enddefine;

;;; asm_outlabset, asm_outglabset:
;;;     define an absolute (global) label

define lconstant outlabset(lab, val, export);
	lvars lab, val, export;
	asmf_printf(val, lab, '%p\t.equ\t\t%p\n');
	if export then
		;;; export must come after the equate and type is "absolute"
		asmf_printf(lab, '\t.export\t\t%p,absolute\n');
		;;; record the definition
		"absolute" -> file_defines(asmf_charout)(lab);
	endif;
enddefine;
;;;
define asm_outlabset  = outlabset(% false %) enddefine;
define asm_outglabset = outlabset(% true  %) enddefine;

;;; asm_outbyte, asm_outshort, asm_outword:
;;;     output -n- data items. Each item is represented as a word or
;;;     string, but may stand for a numeric value.

define lconstant outdatum(n, string);
	lvars i, n, string, datum, symbol;
	returnif(n == 0);
	asmf_charout(string);
	fast_for i from n by -1 to 1 do
		subscr_stack(i) -> datum;
		if isword(datum) then fast_word_string(datum) -> datum endif;
		if is_global_label(datum) ->> symbol then
			;;; record as possible file import
			true -> file_uses(asmf_charout)(symbol);
			;;; external function should be denoted by its PLABEL
			if extern_symbol_type(symbol) == "code" then
				'P\'' <> datum -> datum;
			endif;
		endif;
		asmf_pr(datum);
		unless i == 1 then asmf_charout(`,`) endunless;
	endfast_for;
	erasenum(n);
	asmf_charout(`\n`);
enddefine;
;;;
define global asm_outbyte  = outdatum(% ASM_BYTE_STR  %) enddefine;
define global asm_outshort = outdatum(% ASM_SHORT_STR %) enddefine;
define global asm_outint   = outdatum(% ASM_INT_STR  %) enddefine;
define global asm_outword  = outdatum(% ASM_WORD_STR  %) enddefine;

;;; asm_out_dfloat(hi_part, lo_part):
;;;     output a double float

define asm_out_dfloat =
	asm_outword(% 2 %);
enddefine;

;;; asm_align_double:
;;;     output alignment directive for double word operand (e.g. double
;;;     float)

define asm_align_double =
	outcode(% '.align\t\t8\n' %);
enddefine;

;;; asm_align_file:
;;;     output alignment directive for end of file (don't define if not
;;;     needed)
/*
define asm_align_file =
	identfn(%%);
enddefine;
*/

;;; asm_struct_wrap:
;;;		wrap an arbitrary (word-aligned) code sequence in a string-type
;;;		structure (e.g. a rawstruct)

define asm_struct_wrap(keylab, code_p_list);
	lvars keylab, code_p_list, code_p;
	lvars startlab = nextlab(), endlab = nextlab();
	asm_outlab(startlab);
	asm_outword(endlab <> '-' <> startlab, keylab, 2);
	fast_for code_p in code_p_list do code_p() endfor;
	asm_outlab(endlab);
enddefine;


	/*  -asm_addbits- is used to accumulate a sequence of bitfields into
		a (big)integer which -asm_outbits- is then called to output when the
		total number of bits accumulated occupies an exact number of bytes.
			-val- is the currently accumulated (big)integer, containing
		-nbits- bits so far; -new_val- is the field to be added, containing
		-new_nbits- bits.
			The value returned will be the -val- input for the next field
		(with -nbits- incremented by -new_nbits-), and so on. Other than
		initialising -val- to 0 at the start, nothing is assumed about how
		the fields are added to it, so it is the responsibility of this
		procedure and -asm_outbits- to output the bitfields correctly.
	*/

	;;; 'big endian' version adds the fields at the bottom and outputs
	;;; the bytes from the top down
define asm_addbits(new_val, new_nbits, val, nbits);
	lvars new_val, new_nbits, val, nbits;
	(val << new_nbits) || new_val   ;;; add the new field at the bottom
enddefine;

	/*  In this, -nbits- is a multiple of BYTE_BITS, and -val- has been
		produced by repeated calls of -asm_addbits-
	*/
define asm_outbits(val, nbits);
	lvars val, nbits, count = 0;
	lconstant BYTE_MASK = (1<<BYTE_BITS)-1;
	until nbits == 0 do
		nbits-BYTE_BITS -> nbits;
		(val >> nbits) && BYTE_MASK;    ;;; output bytes from the top down
		;;; this just saves putting each byte on a separate line
		if (count+1 ->> count) == 12 then asm_outbyte(count), 0 -> count endif;
	enduntil;
	if count /== 0 then asm_outbyte(count) endif;
enddefine;


	;;; for closures generated by poplink for undef procedures
define asm_gen_poplink_code(outlabs, nfroz, jmplab);
	lvars outlabs, nfroz, jmplab;
	;;; register numbers (cf. "genproc.p")
	lconstant t1 = asm_reg_name(23), t2 = asm_reg_name(22),
			  usp = asm_reg_name(19);
	;;; plant exec labels
	outlabs();
	;;; load frozval pointer to %t1
	asmf_printf(t1, '\tbl\t\t.+8,%p\n');
	asmf_printf(t1, t1, nfroz*4+8+3, '\tldo\t\t-%p(%p),%p\n');
	;;; push frozvals
	fast_repeat nfroz times
		asmf_printf(t2, t1, '\tldwm\t\t4(%p),%p\n');
		asmf_printf(usp, t2, '\tstwm\t\t%p,-4(%p)\n');
	endfast_repeat;
	;;; chain -jmplab-
	lvars (l_lab, r_lab) = asm_uselab(jmplab);
	asmf_printf(t1, l_lab, '\tldil\t\t%p, %p\n');
	asmf_printf(t1, r_lab, '\tbe\t\t%p(%%sr4, %p)\n');
	asmf_printf('\tnop\n');
	;;; return number of words of code planted
	nfroz*2 + 5;
enddefine;


/*
 *	Exfunc closure code -- jump to subroutine _exfunc_clos_action (aextern.s)
 *	with exfunc_closure record address in a reg, etc. This procedure is
 *	passed the label of _exfunc_clos_action. The code generated must be
 *	padded to exactly 4 words.
 */

define asm_gen_exfunc_clos_code(action_lab);
	lvars action_lab;
	;;; register numbers (cf. "genproc.p")
	lconstant t1 = asm_reg_name(23), t2 = asm_reg_name(22);
	;;; save return address in %t2
	asmf_printf(t2, '\tcopy\t\t%%r31, %p\n');
	;;; external branch to __________action_lab;
	;;; %r31 is left pointing to the (exfunc_clos address) + 16
	lvars (l_lab, r_lab) = asm_uselab(action_lab);
	asmf_printf(t1, l_lab, '\tldil\t\t%p, %p\n');
	asmf_printf(t1, r_lab, '\tble\t\t%p(%%sr4, %p)\n');
	asmf_printf('\tnop\n');
enddefine;


	/*	Arguments to the assembler, specifying the default architecture
	*/
constant unix_as_options = ['+DA1.1'];


	/*  String for first line of Unix "ld" command -- used in os_comms.p
		(Image name is in the environment variable "IM")
	*/
constant unix_ld_command_header =
		'/bin/ld -E -x -o $IM -e \'$START$\' -X 9999 +s \\\n';


	/*	Extra libraries to be added to the link command before -lc.

		This is to support external load of shared libraries wholly
		or partly written in C++ where the application needs additional
		run-time support. Only possible if the linking machine has
		the required static components, i.e. if the C++ compiler is
		installed.

		This has been added primarily to support loading of Intersolv
		ODBC drivers but could be relevant for other things.

		At present it works only for objects built with the older
		"cfront" compiler CC, not aCC.
	*/
define active unix_ld_extra_libraries;
	lconstant libcxx = '/opt/CC/lib/libcxx.a';
	if sys_file_exists(libcxx) then
		'-lC ' <> libcxx
	else
		false
	endif;
enddefine;


endsection;     /* $-Popas */


/* --- Revision History ---------------------------------------------------
--- Robert Duncan, Jul  2 1998
		Added unix_ld_extra_libraries.
--- Robert Duncan, May 12 1997
		Added +DA1.1 assembler option required for HP-UX 10.20
--- John Gibson, Aug 16 1996
		New version of extern_name_translate. All symbols for which
		an explicit extern_symbol_type of "data" was necessary are now marked
		data in the _extern expressions referring to them.
--- Integral Solutions Ltd, Aug 31 1995 (Julian Clinton)
		Added -E option to unix_ld_command_header.
		Added __pop_malloc_exhausted, __pop_malloc_min_alloc and
		errno to list of data segment symbols.

--- John Gibson, Apr  3 1995
		Added asm_outint
--- Robert John Duncan, Mar 22 1994
		Changed asm_gen_poplink_code to plant the execute labels
--- Robert John Duncan, Jun 30 1993
		Added '+s' option to ld command line so that the pop external
		library can be located at run-time through SHLIB_PATH and removed
		the '-E', since all symbols which might be accessed from external
		loads should now be in the pop library.
--- Robert John Duncan, May 25 1993
		Changed outdatum to accept words as well as strings (passed by
		poplink).
--- Robert John Duncan, May 24 1993
		Added asm_gen_exfunc_clos_code to generate template code previously
		in aextern.s
--- John Gibson, May 19 1993
		Changed extern_name_translate to always return a string
--- Robert Duncan, May 12 1993
		Added asm_struct_wrap (called from "genstruct.p")
 */
