/* --- Copyright University of Sussex 1998. All rights reserved. ----------
 > File:			C.sparc/src/ass.p
 > Purpose:			Sun-4 run-time assembler
 > Author:          Robert Smith, Aug 19 1988 (see revisions)
 */

;;; --- DECLARATIONS -------------------------------------------------------

#_INCLUDE 'declare.ph'
#_INCLUDE 'vmdefs.ph'
#_INCLUDE 'drop_code.ph'

	;;; Assembler routines
global constant
	_popenter			;;; enter procedure with checks (amain.s)
	_popuenter			;;; enter updater procedure with check (amain.s)
	_popuncenter		;;; enter updater procedure without checks (amain.s)
	_checkall			;;; check interupts et cetera (in amain.s)
	_bfield				;;; fetch unsigned bitfield (in amove.s)
	_sbfield			;;; fetch signed bitfield (in amove.s)
	_ubfield			;;; update bitfield (in amove.s)
	_vecsub_mult		;;; mult vector subscript by size (in amove.s)
	_setstklen			;;; set user stacklength (in alisp.s)
	_setstklen_diff		;;; set user stacklength (in alisp.s)
	;

vars
	_trap,				;;; trap flag
	;

section $-Sys$-Vm;

vars
	_asm_drop_ptr,		;;; pointer to drop code at
	_asm_code_offset,	;;; offset into executable code (sysint bytes)
	_asm_pass,			;;; assembler pass counter -- false on last pass
	asm_clist,			;;; moving pointer into codelist: asm_instr is asm_clist(1)
	asm_instr,			;;; current {^I_POP ...} type instruction
	asm_struct_list,	;;; list of structures to drop in table
	_Nlocals,			;;; sysint total number of dynamic locals
	_Nstkvars,			;;; sysint total number of stack locals
	_Npopstkvars,		;;; sysint number of pop stack locals
	_Nreg,				;;; sysint total number of regs used
	_Npopreg,			;;; sysint number of pop regs used
	asm_cllr_return_id,	;;; return address register id
	;

constant procedure
		(Drop_l, Drop_hi, Drop_lo, Structab_offset, Trans_structure,
		Get_procedure, Is_register, Code_pass, Drop_I_code,
		I_GOTO, I_GOTO_opt)
	;

endsection;


;;; ----------------------------------------------------------------------

section $-Sys$-Vm;

lvars
	_us_off,			;;; user stack offset from %rusp
	_strsize,			;;; sysint byte size of structure table
	_big_procedure,		;;; flag set if big procedure -- should be false
	;;; _condbr_nop,	;;; code offset of last squashable nop
	;

lconstant

	;;; operand addressing modes
	REGISTER	= 1,
	IMMED		= 2,
	ABS_IDVAL	= 3,
	RELATIVE	= 4,
	REL_IDVAL	= 5,
	AUTO		= 6,

	;;; For passing to Drop_structure
	DEFER		= true,
	DIRECT		= false,

	;;; Instruction opcodes
	;;; the 16:100 set on these indicates an arith instruction
	_OP_ADD		= _OP_add   _biset _16:100,
	_OP_OR		= _OP_or    _biset _16:100,
	_OP_SUB		= _OP_sub   _biset _16:100,
	_OP_ANDcc	= _OP_andcc _biset _16:100,
	_OP_SUBcc	= _OP_subcc _biset _16:100,
	_OP_SLL		= _OP_sll   _biset _16:100,
	_OP_SRA		= _OP_sra   _biset _16:100,

	_UNUSED	= _0,
	_NOREG	= _-1,
	;

;;; --- MAPPING REGISTERS TO IDENTIFIERS -------------------------------------

	;;; Register identifiers
	;;; value is popint reg number shifted left 1, plus bit 0 if pop reg
protected register constant
	pop_reg_A		= Rl0 << 1 || 1,
	pop_reg_B		= Rl1 << 1 || 1,
	pop_reg_C		= Rl2 << 1 || 1,
	pop_reg_D		= Rl3 << 1 || 1,
	pop_reg_E		= Rl4 << 1 || 1,
	pop_reg_F		= Rl5 << 1 || 1,
	pop_reg_G		= Rl6 << 1 || 1,
	pop_reg_H		= Rl7 << 1 || 1,
	nonpop_reg_A	= Ri0 << 1,
	nonpop_reg_B	= Ri1 << 1,
	nonpop_reg_C	= Ri2 << 1,
	nonpop_reg_D	= Ri3 << 1,
	nonpop_reg_E	= Ri4 << 1,

	arg_reg_0		= Ro0 << 1,
	arg_reg_1		= Ro1 << 1,
	arg_reg_2		= Ro2 << 1,

	chain_reg		= Rg1 << 1,
	;

	;;; Lists of pop and nonpop registers for local vars, in order of
	;;; allocation. These lists start with a list of registers that should be
	;;; local to every procedure.
constant
	asm_pop_registers	= [%[], ident pop_reg_A, ident pop_reg_B,
								ident pop_reg_C, ident pop_reg_D,
								ident pop_reg_E, ident pop_reg_F,
								ident pop_reg_G, ident pop_reg_H%],
	asm_nonpop_registers= [%[], ident nonpop_reg_A, ident nonpop_reg_B,
								ident nonpop_reg_C, ident nonpop_reg_D,
								ident nonpop_reg_E%];

;;; --------------------------------------------------------------------------

define Is_address_reg() with_nargs 1;
	Is_register()			;;; every reg is an address reg
enddefine;

	;;; temp register queue variables
lvars _Treg1, _Treg2, _Treg3;

define :inline lconstant IS_TMP_REG(_r);
	(_r == _:Ro3 or _r == _:Ro4 or _r == _:Ro7)
enddefine;

define :inline lconstant IS_NOREG(_r);
	_neg(_r)
enddefine;

;;; initialise register queue
define lconstant init_reg();
	_:Ro3 -> _Treg1;
	_:Ro4 -> _Treg2;
	_:Ro7 -> _Treg3
enddefine;

;;; Get next free temporary register from queue
define lconstant get_reg() -> _reg;
	lvars _reg = _Treg1;
	if IS_NOREG(_reg) then
		;;; none left in queue
		mishap(0, 'SYSTEM ERROR 1 IN get_reg');
	else
		_Treg2 -> _Treg1;
		_Treg3 -> _Treg2;
		_NOREG -> _Treg3
	endif
enddefine;

;;; Free temporary register and place at back of queue
define lconstant kill_reg(_reg);
	lvars _reg;
	returnif(IS_NOREG(_reg));
	if IS_NOREG(_Treg1) then
		_reg -> _Treg1
	elseif IS_NOREG(_Treg2) then
		_reg -> _Treg2
	elseif IS_NOREG(_Treg3) then
		_reg -> _Treg3
	else
		mishap(0, 'SYSTEM ERROR 1 IN kill_reg');
	endif
enddefine;


;;; --- INSTRUCTION-DROPPING PROCEDURES ------------------------------------

lvars
	_lasti_ptr,
	_lasti_arith,
	_lasti_src1,
	_lasti_src2,
	_lasti_dst,
;

	;;; Set high 22 bits of register from immediate (drops 1 instruction)
define lconstant Drop_sethi(_imm, _dst);
	lvars _imm, _dst;
	unless _asm_pass then
		_asm_drop_ptr -> _lasti_ptr;
		true -> _lasti_arith;
		_-1 ->> _lasti_src1 -> _lasti_src2;
		_dst -> _lasti_dst;

		_F2_INST(_OP_sethi, _imm, _dst)
	else
		_0
	endunless;
	chain((), Drop_l)
enddefine;

lconstant procedure Drop_ibinop;

	;;; Set register from immediate (drops 1 or 2 instructions)
define lconstant Drop_set(_imm, _dst);
	lvars _imm, _dst, _hi = Drop_hi(_imm), _lo, _fdiff;
	if _zero(_hi) then
		Drop_ibinop(_OP_OR, _:Rg0, _imm, _dst)
	elseif _nonzero(Drop_lo(_imm) ->> _lo)
	and (_imm _sub false ->> _fdiff) _slteq _16:FFF and _fdiff _sgreq _-16:1000
	then
		Drop_ibinop(_OP_ADD, _:Rfalse, _fdiff, _dst)
	else
		Drop_sethi(_hi, _dst);
		unless _zero(_lo) then
			Drop_ibinop(_OP_OR, _dst, _lo, _dst)
		endunless
	endif
enddefine;

	;;; Binary operations with two register sources (drops 1 instruction)
define lconstant Drop_rbinop(_op3, _src1, _src2, _dst);
	lvars _op3, _src1, _src2, _dst;
	if _asm_pass then chain(_0, Drop_l) endif;

	if _op3 _bitst _16:100 then
		_asm_drop_ptr -> _lasti_ptr;
		true -> _lasti_arith;
		_src1 -> _lasti_src1;
		_src2 -> _lasti_src2;
		_dst -> _lasti_dst
	endif;

	chain( _F3rA_INST(_op3, _src1, _src2, _dst), Drop_l)
enddefine;

	;;; Binary operations with one immediate source (drops 1, 2 or 3
	;;; instructions)
define lconstant Drop_ibinop(_op3, _src, _imm, _dst);
	lvars _op3, _src, _imm, _dst, _tmp4;
	if _imm _sgr _16:FFF or _imm _slt _-16:1000 then
		get_reg() -> _tmp4;
		Drop_set(_imm, _tmp4);
		Drop_rbinop(_op3, _src, _tmp4, _dst);
		kill_reg(_tmp4);
	elseif _asm_pass then
		chain(_0, Drop_l)
	else
		if _op3 _bitst _16:100 then
			_asm_drop_ptr -> _lasti_ptr;
			true -> _lasti_arith;
			_src -> _lasti_src1;
			_-1 -> _lasti_src2;
			_dst -> _lasti_dst
		endif;
		chain( _F3iA_INST(_op3, _src, _imm, _dst), Drop_l)
	endif;
enddefine;

	;;; Load/store integer with register + register address (drops 1
	;;; instruction)
define lconstant Drop_index(_op3, _base, _index, _reg);
	lvars _op3, _base, _index, _reg;
	chain( _F3rM_INST(_op3, _base, _index, _reg), Drop_l)
enddefine;

	;;; Load/store integer with register + displacement address (drops 1, 2
	;;; or 3 instructions)
define lconstant Drop_disp(_op3, _base, _imm, _reg);
	lvars _op3, _base, _imm, _reg, _tmp4, _r, _inst, _drop_ptr;
	if _imm _sgr _16:FFF or _imm _slt _-16:1000 then
		if _op3 == _OP_ld then
			_reg -> _r, _NOREG -> _tmp4
		else
			get_reg() ->> _r -> _tmp4
		endif;
		Drop_set(_imm, _r);
		Drop_index(_op3, _base, _r, _reg);
		kill_reg(_tmp4)
	elseif _asm_pass then
		chain(_0, Drop_l)
	else
		_asm_drop_ptr -> _drop_ptr;
		if _op3 == _OP_st then
			_drop_ptr -> _lasti_ptr;
			false -> _lasti_arith;
			_base -> _lasti_src1;
			_-1 -> _lasti_src2;
			_reg -> _lasti_dst
		endif;

		_F3iM_INST(_op3, _base, _imm, _reg) -> _inst;

		;;; try to move a preceding arith instruction or store to after
		;;; a load (move a store only if load is off PB)

		if _op3 == _OP_ld and _drop_ptr@(w)[_-1] == _lasti_ptr
		and (_lasti_arith or _base == _:Rpb)
		and _lasti_dst /== _base and _lasti_dst /== _reg
		and _lasti_src1 /== _reg and _lasti_src2 /== _reg
		then
			;;; swap the load with the previous instruction
			_drop_ptr!(w)[_-1], _inst -> (_inst, _drop_ptr!(w)[_-1])
		endif;

		chain(_inst, Drop_l)
	endif;
enddefine;

	;;; Load/store integer with 32-bit absolute address (drops 2 instructions)
define lconstant Drop_abs(_op3, _imm, _reg);
	lvars _op3, _imm, _reg, _r = _reg, _tmp4 = _NOREG;
	unless _op3 == _OP_ld then get_reg() ->> _r -> _tmp4 endunless;
	Drop_sethi(Drop_hi(_imm), _r);					;;; sethi %hi(i), %t
	Drop_disp(_op3, _r, Drop_lo(_imm), _reg);		;;; ld/st %r, [%t+%lo(i)]
	kill_reg(_tmp4);
enddefine;

	;;; Arithmetic/logical with 32-bit immediate (drops 2 instructions)
define lconstant Drop_imm(_op3, _imm, _dst);
	lvars _op3, _imm, _dst, _tmp4;
	get_reg() -> _tmp4;
	Drop_sethi(Drop_hi(_imm), _tmp4);				;;; sethi %hi(i), %t
	Drop_ibinop(_op3, _tmp4, Drop_lo(_imm), _dst);	;;; jmpl %d, [%t+%lo(i)]
	kill_reg(_tmp4);
enddefine;

define lconstant Drop_binop(/* _opcode, _src1, _src2, _dst, */ imm);
	lvars imm;
	if imm then
		chain(Drop_ibinop)
	else
		chain(Drop_rbinop)
	endif
enddefine;

define lconstant Drop_mvreg(_sreg, _dreg);
	lvars _sreg, _dreg;
	chain(_OP_OR, _sreg, _:Rg0, _dreg, Drop_rbinop)		;;; mov %sreg, %dreg
enddefine;

	;;; No operation -- sethi 0, %g0
define lconstant Drop_nop = Drop_l(% _NOP_INST %) enddefine;

	;;; Adjust user stack pointer from cached offset. Occurs in branch
	;;; delay slot so drop NOP if none to do
define lconstant Drop_us_adjust(delay);
	lvars delay;
	if _nonzero(_us_off) then
		Drop_ibinop(_OP_ADD, _:Rusp, _us_off, _:Rusp);	;;; add %us, i, %us
		if delay then _NULL -> _lasti_ptr endif;
		_0 -> _us_off;
	elseif delay then
		Drop_nop();
	endif;
enddefine;


;;; --- branches -----------------------------------------------------------

	;;; Branch to byte offset from current position
define lconstant Drop_branch(_cond, _offset, annul);
	lvars _cond, _offset, annul;
	if annul then _cond _biset _16:10 -> _cond endif;
	Drop_l( _F2_INST(_OP_branch, _shift(_offset _sub _asm_code_offset, _-2),
											_cond) )
enddefine;

	;;; Produce template for forward branch
define lconstant branch_src();
	_asm_drop_ptr;
	Drop_l(_16:00000000);
enddefine;

	;;; Fill in offset for forward branch
define lconstant branch_dst(_cond, annul, _save);
	lvars _cond, annul, _save, _drop_ptr = _asm_drop_ptr;
	dlocal _asm_code_offset = 0, _asm_drop_ptr = _save;
	unless _asm_pass then
		Drop_branch(_cond, ##(b){_drop_ptr, _save} _add _4, annul);
		_NULL -> _lasti_ptr
	endunless;
enddefine;


;;; --- STRUCTURE ACCESS -----------------------------------------------------

;;; Form immediate operand or false register if value is address of false
define lconstant Get_literal(item) /* -> (Mode, _Opnd, _Offs) */;
	lvars item;
	if item then
		(IMMED, item, _UNUSED)
	else
		(REGISTER, _:Rfalse, _UNUSED)
	endif
enddefine;


;;; Form operand to transfer a structure (or its valof) to a register (or the
;;; stack), relative to structure address at start of code if in heap, or
;;; absolute if not.
define lconstant Get_structure(_opspec, defer) /* -> (Mode, _Opnd, _Offs) */;
	lvars _opspec, defer, argstruct, _Opnd, _Offs;

	;;; replace structure with offset or reg ident on first pass (nonop
	;;; on subsequent passes)
	Trans_structure(asm_instr!INST_ARGS[_opspec])
							->> argstruct -> asm_instr!INST_ARGS[_opspec];

	if issimple(argstruct) then
		_int(argstruct) -> argstruct;		;;; offset as a sysint
		if _neg(argstruct) then
			;;; negated offset for on-stack var, shifted left 1
			;;; if bit 0 is set, access is via a ref, i.e. another indirection
			;;; this is disabled by defer being false (only for I_POPQ)
			unless argstruct _bitst _1 then false -> defer endunless;
			_:Rsp -> _Opnd;
			_negate(_shift(argstruct, _-1)) -> _Offs
		else
			;;; else via argstruct address in table
			;;; calculate offset for word offset to place in table
			_:Rpb -> _Opnd;
			@@PD_TABLE{argstruct} -> _Offs
		endif;
		(if defer then REL_IDVAL else RELATIVE endif, _Opnd, _Offs)
	else
		;;; literal or idval of absolute pointer
		if defer then
			if (Is_register(argstruct) ->> _opspec) then
				;;; register operand
				(REGISTER, _int(_opspec), _UNUSED)
			else
				;;; idval of absolute operand
				(ABS_IDVAL, argstruct, _UNUSED)
			endif
		else
			;;; immediate operand
			Get_literal(argstruct)
		endif
	endif
enddefine;


;;; ------------------------------------------------------------------

lvars
	tos_Mode = false,
	_tos_Opnd,
	_tos_Offs,
	_tos_iscopy,
;

lconstant procedure (Load_structure);

define lconstant Increment_us();
	false -> tos_Mode;
	_us_off _add _4 -> _us_off;
	if _us_off _sgreq _16:1000 or _us_off _slteq _-16:1000 then
		mishap(0, 'SYSTEM ERROR 1 IN Increment_us');
	endif
enddefine;

define lconstant Decrement_us();
	false -> _tos_iscopy;
	_us_off _sub _4 -> _us_off;
	if _us_off _sgreq _16:1000 or _us_off _slteq _-16:1000 then
		mishap(0, 'SYSTEM ERROR 1 IN Decrement_us');
	endif;
enddefine;

define lconstant Load_tos_reg(_reg);
	lvars _reg, _tmp;
	if tos_Mode == REGISTER then
		returnif(IS_NOREG(_reg));
		Drop_mvreg(_tos_Opnd, _reg);
		_tos_Opnd -> _tmp;
		if IS_TMP_REG(_tmp) then kill_reg(_tmp) endif;
		_reg -> _tos_Opnd
	else
		if IS_NOREG(_reg) then get_reg() -> _reg endif;
		Load_structure((tos_Mode, _tos_Opnd, _tos_Offs), _reg);
		REGISTER -> tos_Mode; _reg -> _tos_Opnd; _UNUSED -> _tos_Offs
	endif
enddefine;

define lconstant Flush_tos(kill);
	lvars kill, _tmp;
	returnunless(tos_Mode);
	unless _tos_iscopy then		;;; must be a reg if copied
		Load_tos_reg(_NOREG);
		Drop_disp(_OP_st, _:Rusp, _us_off, _tos_Opnd);		;;; st %reg, [%us]
		true -> _tos_iscopy
	endunless;
	if kill then
		_tos_Opnd -> _tmp;
		if IS_TMP_REG(_tmp) then kill_reg(_tmp) endif;
		false -> tos_Mode
	endif
enddefine;

	;;; Push onto user stack -- stack grows down, pre-decrement.
	;;; Stack pointer not adjusted until change of control flow
define lconstant Drop_push(_reg);
	lvars _reg;
	Flush_tos(true);
	REGISTER -> tos_Mode; _reg -> _tos_Opnd; _UNUSED -> _tos_Offs;
	Decrement_us()
enddefine;

	;;; Pop from user stack -- stack grows down, post-increment.
	;;; Stack pointer not adjusted until change of control flow
define lconstant Drop_pop_tmp(_reg) -> (_reg, _tmpreg);
	lvars _reg, _tmpreg = _NOREG;
	if tos_Mode then
		if tos_Mode == REGISTER then
			_tos_Opnd -> _reg;
			if IS_TMP_REG(_reg) then _reg -> _tmpreg endif
		else
			if IS_NOREG(_reg) then get_reg() ->> _reg -> _tmpreg endif;
			Load_structure((tos_Mode, _tos_Opnd, _tos_Offs), _reg)
		endif
	else
		if IS_NOREG(_reg) then get_reg() ->> _reg -> _tmpreg endif;
		Drop_disp(_OP_ld, _:Rusp, _us_off, _reg)			;;; ld [%us], %reg
	endif;

	Increment_us()
enddefine;

define lconstant Drop_pop(_reg);
	lvars _reg, _tmp;
	if tos_Mode then
		if tos_Mode == REGISTER then
			unless _reg == _:Rg0 then Drop_mvreg(_tos_Opnd, _reg) endunless;
			_tos_Opnd -> _tmp;
			if IS_TMP_REG(_tmp) then kill_reg(_tmp) endif
		elseunless _reg == _:Rg0 then
			Load_structure((tos_Mode, _tos_Opnd, _tos_Offs), _reg)
		endif
	else
		Drop_disp(_OP_ld, _:Rusp, _us_off, _reg)			;;; ld [%us], %reg
	endif;

	Increment_us()
enddefine;

;;; Drop code to load (store) into (from) register from (into) operand
;;; represented by (Mode, _Opnd, _Offs)
define lconstant Load_structure(Mode, _Opnd, _Offs, _reg);
	lvars Mode, _Opnd, _Offs, _reg;
	go_on Mode to register immed abs_idval relative rel_idval auto
	else error;
register:
		Drop_mvreg(_Opnd, _reg);
		return;
immed:
		Drop_set(_Opnd, _reg);
		return;
abs_idval:
		Drop_abs(_OP_ld, _Opnd@ID_VALOF, _reg);
		return;
relative:
		Drop_disp(_OP_ld, _Opnd, _Offs, _reg);
		return;
rel_idval:
		Drop_disp(_OP_ld, _Opnd, _Offs, _reg);
		Drop_disp(_OP_ld, _reg, @@ID_VALOF, _reg);
		return;
auto:
		Drop_pop(_reg);
		return;
error:
		mishap(0, 'SYSTEM ERROR 1 IN Load_structure');
enddefine;

define lconstant Store_structure(_reg, Mode, _Opnd, _Offs);
	lvars _reg, Mode, _Opnd, _Offs, _tmp;
	go_on Mode to register immed abs_idval relative rel_idval auto
	else error;
register:
		Drop_mvreg(_reg, _Opnd);
		return;
immed:
		mishap(0, 'SYSTEM ERROR 1 IN Store_structure');
		return;
abs_idval:
		Drop_abs(_OP_st, _Opnd@ID_VALOF, _reg);
		return;
relative:
		Drop_disp(_OP_st, _Opnd, _Offs, _reg);
		return;
rel_idval:
		get_reg() -> _tmp;
		Drop_disp(_OP_ld, _Opnd, _Offs, _tmp);
		Drop_disp(_OP_st, _tmp, @@ID_VALOF, _reg);
		kill_reg(_tmp);
		return;
auto:
		Drop_push(_reg);
		return;
error:
		mishap(0, 'SYSTEM ERROR 2 IN Store_structure');
enddefine;

;;; POP reg (or bring from structure)
define lconstant Get_fsource(_arg, no_imm, flush_tos)
										-> (_Opnd, isimm, _tmpreg);
	lvars	_arg, instr, Mode, _Opnd, _Offs, isimm = false, no_imm,
			op, _tmpreg = _NOREG, _r, flush_tos, use_tmp = false;
	dlocal asm_instr;
	if (asm_instr!INST_ARGS[_arg] ->> instr) then
		;;; specified by move instruction
		instr!INST_OP -> op;
		if op == I_MOVENUM or op == I_MOVEADDR then
			Get_literal(instr!INST_ARGS[_0])
		else
			instr -> asm_instr;
			Get_structure(_0, op==I_MOVE)
		endif;
		if flush_tos then Flush_tos(true) endif
	elseif tos_Mode then
		if tos_Mode == REGISTER and (_tos_Opnd -> _r, IS_TMP_REG(_r)) then
			_r -> _tmpreg
		endif;
		tos_Mode, _tos_Opnd, _tos_Offs;
		Increment_us()
	else
		;;; from stack
		AUTO, _UNUSED, _UNUSED
	endif -> (Mode, _Opnd, _Offs);
	returnif(Mode == REGISTER);
	if Mode /== IMMED or no_imm then
		if isinteger(no_imm) then
			_int(no_imm)		;;; work reg to use
		else
			get_reg() ->> _tmpreg
		endif -> _r;
		Load_structure((Mode, _Opnd, _Offs), _r);
		_r -> _Opnd
	else
		true -> isimm
	endif
enddefine;

define lconstant Get_fdestin() /* -> (Mode, _Opnd, _Offs, push_res) */;
	lvars instr, _op;
	;;; see if can optimise a following pop or store
	fast_front(fast_back(asm_clist)) -> instr;
	if isvector(instr)
	and ((instr!INST_OP ->> _op) == I_POP or _op == I_STORE) then
		;;; move to valof field of I_POP's variable
		instr -> asm_instr;
		Get_structure(_0, DEFER);
		fast_back(asm_clist) -> asm_clist;			;;; erase the I_POP
		_op == I_STORE
	else
		AUTO, _UNUSED, _UNUSED, false
	endif
enddefine;



;;; --- ASSEMBLER PROCEDURES FOR INDIVIDUAL INSTRUCTIONS -------------------

;;; --- move procedures ----------------------------------------------------
;;;
;;; {MOVEX obj}			push object to userstack
;;; {MOVEX src dest}	move from src to valof(dest)
;;;
;;; I_MOVE		move from valof
;;; I_MOVEQ		move object
;;; I_MOVENUM	move literal number
;;; I_MOVEADDR	move literal address (= I_MOVENUM)
;;;
;;; I_MOVES		duplicate top of stack (dummy arg)
;;; I_POP		pop userstack into valof
;;; I_STORE		copy tos into valof
;;; I_ERASE		pop userstack (dummy arg)

define lconstant Drop_move(defer);	/* boolean or 0 */
	lvars	defer, _tmp0, sMode, _sOpnd, _sOffs, dMode, _dOpnd, _dOffs,
			instr = asm_instr;
	if defer == 0 then
		;;; number
		Get_literal(instr!INST_ARGS[_0])
	else
		Get_structure(_0, defer)
	endif -> (sMode, _sOpnd, _sOffs);
	if instr!V_LENGTH == _2 then
		Flush_tos(true);
		sMode -> tos_Mode; _sOpnd -> _tos_Opnd; _sOffs -> _tos_Offs;
		Decrement_us()
	else
		Get_structure(_1, DEFER) -> (dMode, _dOpnd, _dOffs);
		if tos_Mode == dMode and _tos_Opnd == _dOpnd and _tos_Offs == _dOffs
		then
			Load_tos_reg(get_reg())
		endif;
		if dMode == REGISTER then
			Load_structure((sMode, _sOpnd, _sOffs), _dOpnd)
		elseif sMode == REGISTER then
			Store_structure(_sOpnd, (dMode, _dOpnd, _dOffs))
		else
			get_reg() -> _tmp0;
			Load_structure((sMode, _sOpnd, _sOffs), _tmp0);
			Store_structure(_tmp0, (dMode, _dOpnd, _dOffs));
			kill_reg(_tmp0)
		endif
	endif
enddefine;

define I_MOVE		= Drop_move(% DEFER %)	enddefine;
define I_MOVEQ		= Drop_move(% DIRECT %) enddefine;
define I_MOVENUM	= Drop_move(% 0 %)		enddefine;
define I_MOVEADDR	= Drop_move(% 0 %)		enddefine;


define I_MOVE_CALLER_RETURN();
	lvars instr = asm_instr, _src, _tmp0, Mode, _Opnd, _Offs;
	if (instr!INST_ARGS[_0] ->> _src) == asm_cllr_return_id then
		Get_structure(_1, DEFER) -> (Mode, _Opnd, _Offs);
		if Mode == REGISTER then
			Drop_ibinop(_OP_ADD, _:Ri7, _8, _Opnd)
		else
			get_reg() -> _tmp0;
			Drop_ibinop(_OP_ADD, _:Ri7, _8, _tmp0);
			Store_structure(_tmp0, (Mode, _Opnd, _Offs));
			kill_reg(_tmp0)
		endif
	elseif instr!INST_ARGS[_1] == asm_cllr_return_id then
		if instr!INST_ARGS[_2] == I_MOVE then
			Get_structure(_0, DEFER) -> (Mode, _Opnd, _Offs);
			if Mode == REGISTER then
				Drop_ibinop(_OP_SUB, _Opnd, _8, _:Ri7)
			else
				get_reg() -> _tmp0;
				Load_structure((Mode, _Opnd, _Offs), _tmp0);
				Drop_ibinop(_OP_SUB, _tmp0, _8, _:Ri7);
				kill_reg(_tmp0)
			endif
		else
			Drop_set(_src _sub _8, _:Ri7);
		endif
	else
		fast_chain(instr!INST_ARGS[_2]);
	endif;
enddefine;

	;;; push an unsigned sysint, represented as a pop (big)integer
define I_PUSH_UINT();
	Flush_tos(true);
	IMMED -> tos_Mode;
	Pint_->_uint(asm_instr!INST_ARGS[_0], _-1) -> _tos_Opnd;
	_UNUSED -> _tos_Offs;
	Decrement_us()
enddefine;

	;;; pop top of user stack into valof
define I_POP();
	lvars _tmp, _reg, (Mode, _Opnd, _Offs) = Get_structure(_0, DEFER);
	if Mode == REGISTER then
		Drop_pop(_Opnd)
	else
		Drop_pop_tmp(_NOREG) -> (_reg, _tmp);
		Store_structure(_reg, (Mode, _Opnd, _Offs));
		kill_reg(_tmp)
	endif
enddefine;

	;;; pop into quoted variable
	;;; only meaningful with an lvar containing a ref
define I_POPQ();
	lvars _tmp, _reg, (Mode, _Opnd, _Offs) = Get_structure(_0, DIRECT);
	if Mode == REGISTER then
		mishap(0, 'SYSTEM ERROR 1 IN I_POPQ');
	endif;
	Drop_pop_tmp(_NOREG) -> (_reg, _tmp);
	Store_structure(_reg, (Mode, _Opnd, _Offs));
	kill_reg(_tmp)
enddefine;

	;;; copy top of user stack into valof
define I_STORE();
	lvars _tmp, (Mode, _Opnd, _Offs) = Get_structure(_0, DEFER);
	if tos_Mode then
		Load_tos_reg(if Mode == REGISTER then _Opnd else _NOREG endif);
		unless Mode == REGISTER then
			Store_structure(_tos_Opnd, (Mode, _Opnd, _Offs))
		endunless
	else
		if Mode == REGISTER then
			Drop_disp(_OP_ld, _:Rusp, _us_off, _Opnd)
		else
			get_reg() -> _tmp;
			Drop_disp(_OP_ld, _:Rusp, _us_off, _tmp);
			Store_structure(_tmp, (Mode, _Opnd, _Offs));
			_tmp -> _Opnd
		endif;
		REGISTER -> tos_Mode; _Opnd -> _tos_Opnd; _UNUSED -> _tos_Offs;
		true -> _tos_iscopy
	endif
enddefine;

	;;; duplicate top of stack
define I_MOVES();
	lvars _tmp0;
	if tos_Mode then
		Flush_tos(false);
		Decrement_us()
	else
		Drop_pop_tmp(_NOREG) -> (_tmp0, );
		Decrement_us();
		Drop_push(_tmp0)							;;; push %t0, %us
	endif
enddefine;

	;;; erase top of user stack
define I_ERASE();
	lvars instr = asm_instr, _tmp;
	if datalength(instr) == 2 and instr!INST_ARGS[_0] then
		;;; fast -- no access necessary
		if tos_Mode == REGISTER and (_tos_Opnd -> _tmp, IS_TMP_REG(_tmp)) then
			kill_reg(_tmp)
		endif;
		Increment_us()
	else
		Drop_pop(_:Rg0)								;;; pop %us, %g0
	endif
enddefine;

	;;; swap two elements on user stack (if overall index larger than 13
	;;; bits then this will produce correct but awful code)
define I_SWAP();
	lvars	instr = asm_instr,
			_i = _int(instr!INST_ARGS[_0]), _j = _int(instr!INST_ARGS[_1]),
			_reg0, _reg1, _tmp0, _tmp1, Mode, _Opnd, _Offs, push;

	returnif(_i == _j);
	if _zero(_j) then _i, _j -> (_j, _i) endif;

	if _zero(_i) then
		;;; swapping with tos
		Drop_pop_tmp(_NOREG) -> (_reg0, _tmp0);
		_j _sub _1 -> _j;
		Get_fdestin() -> (Mode, _Opnd, _Offs, push);
		if Mode == REGISTER then
			if _Opnd == _reg0 then
				Drop_mvreg(_reg0, get_reg() ->> _reg0 ->> _tmp0)
			endif;
			_Opnd, _NOREG
		else
			dup(get_reg())
		endif -> (_reg1, _tmp1);

		if _zero(_j) then
			;;; other one is now tos
			Drop_pop(_reg1);
			Drop_push(_reg0)
		else
			@@(w)[_j] _add _us_off -> _i;
			Drop_disp(_OP_ld, _:Rusp, _i, _reg1);		;;; ld [%us+i], %reg1
			Drop_disp(_OP_st, _:Rusp, _i, _reg0);		;;; st %reg0, [%us+i]
			kill_reg(_tmp0)
		endif;

		unless Mode == REGISTER then
			Store_structure(_tmp1, (Mode, _Opnd, _Offs));
			_tmp1 -> _Opnd
		endunless;
		if push then Drop_push(_Opnd) endif;
		unless _tos_Opnd == _tmp1 and tos_Mode == REGISTER then
			kill_reg(_tmp1)
		endunless

	else
		@@(w)[_i] _add _us_off -> _i;
		@@(w)[_j] _add _us_off -> _j;
		get_reg() -> _tmp0;
		get_reg() -> _tmp1;
		Drop_disp(_OP_ld, _:Rusp, _i, _tmp0);		;;; ld [%us+i], %t0
		Drop_disp(_OP_ld, _:Rusp, _j, _tmp1);		;;; ld [%us+j], %t1
		Drop_disp(_OP_st, _:Rusp, _j, _tmp0);		;;; st %t0, [%us+j]
		Drop_disp(_OP_st, _:Rusp, _i, _tmp1);		;;; st %t1, [%us+1]
		kill_reg(_tmp0);
		kill_reg(_tmp1);
	endif
enddefine;


;;; --- FIELD ACCESS INSTRUCTIONS (used by "conskey") ------------------------

	;;; converting field types to opcodes
lconstant

	field_get_ops =			;;; memory -> reg
		list_assoc_val(%[%
			t_BYTE,				_pint(_OP_ldub),
			t_SHORT,			_pint(_OP_lduh),
			t_WORD,				_pint(_OP_ld),
			t_BYTE||t_SIGNED,	_pint(_OP_ldsb),
			t_SHORT||t_SIGNED,	_pint(_OP_ldsh),
			t_WORD||t_SIGNED,	_pint(_OP_ld),
		%]%),

	field_put_ops =			;;; reg -> memory
		list_assoc_val(%[%
			t_BYTE,				_pint(_OP_stb),
			t_SHORT,			_pint(_OP_sth),
			t_WORD,				_pint(_OP_st),
			t_BYTE||t_SIGNED,	_pint(_OP_stb),
			t_SHORT||t_SIGNED,	_pint(_OP_sth),
			t_WORD||t_SIGNED,	_pint(_OP_st),
		%]%),

	field_ops_offs =		;;; offset to first unit
		list_assoc_val(%[
			^t_BYTE		-1
			^t_SHORT	-3
			^t_WORD		-7
			^t_DOUBLE	-14
		]%),

	field_ops_shift =		;;; shift to scale vector index for addresses
		list_assoc_val(%[
			^t_BYTE		-2
			^t_SHORT	-1
			^t_WORD		0
			^t_DOUBLE	1
		]%),
	;

define lconstant Drop_exptr_deref(exptr, _reg, _dstreg) -> _reg;
	lvars exptr, _reg, _dstreg;
	fast_repeat exptr times
		;;; ld [%_reg], %_dstreg
		Drop_disp(_OP_ld, _reg, _0, _dstreg ->> _reg)
	endrepeat
enddefine;

define lconstant Call_vecsub_mult(_size);
	lvars _size;
	Drop_sethi(Drop_hi(_size), _:Ro1);
	Drop_imm(_OP_jmpl, _vecsub_mult, _:Ro7);			;;; jmpl bf, %o7
	Drop_ibinop(_OP_OR, _:Ro1, Drop_lo(_size), _:Ro1);
	_NULL -> _lasti_ptr;
enddefine;


;;; --- Do_bitfield ----------------------------------------------------------
;;; deal with bit fields from I_POP_FIELD and I_PUSH_FIELD
;;; uses *bfield* routines in amove.s:
;;; Routine			Input						Output
;;; _vecsub_mult	%o0 = popint subscript		%o1 = offset
;;;					%o1 = field size
;;; _(s)bfield		%o0 = structure address		%o0 = field value
;;;					%o1 = bit offset
;;;					%o2 = field shift
;;;					%o3 = field mask
;;; _ubfield		%o0 = structure address
;;;					%o1 = bit offset
;;;					%o2 = field shift
;;;					%o3 = field mask
;;;					tos = field value
;;; --------------------------------------------------------------------------

define lconstant Do_bitfield(upd);
	lvars
		instr = asm_instr,
		type,		;;; field type specifier
		_size,		;;; size in bits of record (includes sign if needed)
		offset,		;;; popint bit offset or lex ident representing register
		upd,		;;; true if updating structure
		exptr,		;;; integer deref count if external ptr access
		_reg, _tmpreg, subs_opnd;
	;

	instr!INST_ARGS[_0] -> type;
	instr!INST_ARGS[_1] -> _size;
	instr!INST_ARGS[_4] -> offset;
	instr!INST_ARGS[_5] -> exptr;

	Flush_tos(true);
	_int(_size) -> _size;
	if offset then
		;;; fixed-offset access
		Drop_set(_int(offset), _:Ro1)				;;; mov offset, %o1
	else
		;;; subscripted access
		if instr!INST_ARGS[_3] ->> subs_opnd then
			;;; move subscript to %o0
			Get_fsource(_3, Ro0, true) -> (_reg, , _tmpreg);
			unless _reg == _:Ro0 then Drop_mvreg(_reg, _:Ro0) endunless;
			kill_reg(_tmpreg)
		else
			;;; subscript on user stack - move to %o0 (may be below rec and
			;;; need frigging)
			Drop_disp(_OP_ld, _:Rusp,
				(if instr!INST_ARGS[_2] then _0 else _4 endif) _add _us_off,
									_:Ro0)
		endif;
		Call_vecsub_mult(_size)
	endif;
	;;; bit offset now in %o1

	;;; move structure address to %o0
	Get_fsource(_2, Ro0, true) -> (_reg, , _tmpreg);

	;;; adjust user stack pointer if subscript came from stack
	unless offset or subs_opnd then _us_off _add _4 -> _us_off endunless;

	if exptr then
		;;; deref exptr times
		Drop_exptr_deref(exptr, _reg, _:Ro0) -> _reg
	endif;
	unless _reg == _:Ro0 then Drop_mvreg(_reg, _:Ro0) endunless;
	kill_reg(_tmpreg);

	Drop_us_adjust(false);

	;;; signed/unsigned/updater routine
	Drop_imm(_OP_jmpl, if upd then _ubfield
					elseif type == t_BIT then _bfield
					else _sbfield
					endif, _:Ro7);
	Drop_l(_16:94102000 _add (_size _bimask _16:1FFF));	;;; set _size, %o2
enddefine;


;;; --- Do_field -------------------------------------------------------------
;;; deal with non-bit fields from I_POP_FIELD and I_PUSH_FIELD
;;; --------------------------------------------------------------------------

define lconstant Do_field(_opcode, upd);
	lvars
		instr = asm_instr,
		_opcode,	;;; load, store, or false for address calculation
		type,		;;; code for field type
		_size,		;;; field size in units of type (=1 except for push addr)
		offset,		;;; integer bit offset to field, or register with popint vector index
		upd,		;;; true if updating structure
		exptr,		;;; integer deref count if external ptr access
		_reg, _offs, _tmp1, _tmp2, _dstreg, _n, _tmpreg1, _tmpreg2, subs_opnd;

	instr!INST_ARGS[_0] -> type;
	instr!INST_ARGS[_1] -> _size;
	instr!INST_ARGS[_4] -> offset;
	instr!INST_ARGS[_5] -> exptr;

	unless instr!INST_ARGS[_2] or _opcode or exptr or offset /== 0 then
		;;; pushing address of pop struct at 0 offset -- nothing to do
		return
	endunless;

	_:Rg2 -> _tmp1;	   	;;; unchanged by _vecsub_mult
	Get_fsource(_2, _pint(_tmp1), false) -> (_reg, , _tmpreg1);

	if exptr then
		;;; deref exptr times
		Drop_exptr_deref(exptr, _reg, _tmp1) -> _reg	;;; returns _tmp1
	endif;

	unless offset then
		;;; subscripted access
		if _size /== 1 then _:Ro0 else _:Ro2 endif -> _tmp2;
		unless instr!INST_ARGS[_3] ->> subs_opnd then
			;;; subscript on stack -- move it to _tmp2
			Drop_pop_tmp(_tmp2) -> (_offs, _tmpreg2)
		endunless
	endunless;

	_:Ro0 -> _dstreg;
	if upd then
		;;; new value from stack
		Drop_pop(_:Rg3 ->> _dstreg)
	endif;
	Flush_tos(true);

	if offset then
		;;; fixed-offset access
		_shift(_int(offset), _-3) -> _offs;		;;; convert bits to bytes
		if _opcode then
			Drop_disp(_int(_opcode), _reg, _offs, _dstreg)
		else
			;;; pushing field addr -- stack result
			if _nonzero(_offs) then
				Drop_ibinop(_OP_ADD, _reg, _offs, _:Ro0);
				_:Ro0 -> _reg
			endif;
			Drop_push(_reg)
		endif
	else
		;;; subscripted access
		if subs_opnd then
			Get_fsource(_3, _pint(_tmp2), true) -> (_offs, , _tmpreg2)
		;;; else subscript was on stack, now in reg ______offs
		endif;
		type fi_&& t_BASE_TYPE -> type;
		field_ops_shift(type) -> _n;

		if _size == 1 then
			;;; normal case
			;;; scale popint subscript to vector offset
			_int(_n) -> _n;
			if _n _slt _0 then
				Drop_ibinop(_OP_SRA, _offs, _negate(_n), _tmp2 ->> _offs)
			elseif _n _sgr _0 then
				Drop_ibinop(_OP_SLL, _offs, _n, _tmp2 ->> _offs)
			endif;
			;;; adjust offset
			@@V_WORDS _add _int(field_ops_offs(type)) -> _n;
			Drop_ibinop(_OP_ADD, _offs, _n, _tmp2 ->> _offs)
		else
			;;; pushing addr of array of compound external fields
			;;; shifting no good, need to multiply
			if _reg == _tmpreg1 then Drop_mvreg(_reg, _tmp1 ->> _reg) endif;
			if _offs /== _:Ro0 then Drop_mvreg(_offs, _:Ro0) endif;
			Call_vecsub_mult(_int(_size fi_<< (_n fi_+ 2)));
			_:Ro1 -> _offs
		endif;

		if _opcode then
			Drop_index(_int(_opcode), _reg, _offs, _dstreg)
		else
			;;; pushing field addr -- stack result
			Drop_rbinop(_OP_ADD, _reg, _offs, _:Ro0);
			Drop_push(_:Ro0)
		endif;
		kill_reg(_tmpreg2)
	endif;

	unless _tos_Opnd == _tmpreg1 and tos_Mode == REGISTER then
		kill_reg(_tmpreg1)
	endunless
enddefine;

	/*	{I_PUSH_FIELD ____type ____size _______operand ____________subs_operand ______offset _____exptr _______cvtpint}
		For fixed-offset access, ______offset is an integer and ____________subs_operand
		is ignored; for subscripted access, ______offset is false and
		____________subs_operand is the subscript operand.
	*/
define I_PUSH_FIELD();
	lvars type = asm_instr!INST_ARGS[_0];
	if type fi_&& t_BASE_TYPE == t_BIT then
		;;; bitfield
		Do_bitfield(false)
	else
		;;; non-bitfield
		Do_field(field_get_ops(type), false)
	endif;

	;;; result now in %o0
	if asm_instr!INST_ARGS[_6] then
		;;; make into popint first
		Drop_ibinop(_OP_SLL, _:Ro0, _2, _:Ro0);
		Drop_ibinop(_OP_ADD, _:Ro0, _3, _:Ro0);
	endif;
	;;; stack result
	Drop_push(_:Ro0)
enddefine;

	/*	{I_POP_FIELD ____type ____size _______operand ____________subs_operand ______offset _____exptr}
	*/
define I_POP_FIELD();
	lvars type = asm_instr!INST_ARGS[_0];
	if type fi_&& t_BASE_TYPE == t_BIT then
		;;; bitfield -- new value taken off stack
		Do_bitfield(true)
	else
		;;; non-bitfield
		Do_field(field_put_ops(type), true)
	endif
enddefine;

	/*	{I_PUSH_FIELD_ADDR ____type ____size _______operand ____________subs_operand ______offset _____exptr}
		Push the field address.
	*/
define I_PUSH_FIELD_ADDR();
	Do_field(false, false)
enddefine;


;;; --- FAST_SUBSCRV, FAST_FRONT et cetera ---------------------------------

;;; for fast_front, fast_back, fast_destpair, fast_cont et cetera
define I_FASTFIELD();
	lvars _offs, _reg, _tmp, _tmp2, Mode1, _Opnd1, _Offs1, push1,
			_oneres, Mode2, _Opnd2, _Offs2, push2;

	asm_instr!INST_ARGS[_0] ->> _offs -> _oneres;	;;; false if fast_destpair
	Get_fsource(_1, true, true) -> (_reg, , _tmp);
	Get_fdestin() -> (Mode1, _Opnd1, _Offs1, push1);

	unless _oneres then
		;;; push1 front, push1 (or move) back of pair
		if Mode1 == AUTO or push1 then
			AUTO, _UNUSED, _UNUSED, false
		else
			Get_fdestin()
		endif -> (Mode2, _Opnd2, _Offs2, push2);
		if Mode2 == REGISTER and _Opnd2 /== _reg then
			_NOREG -> _tmp2;
			Drop_disp(_OP_ld, _reg, @@P_FRONT, _Opnd2)
		else
			get_reg() -> _tmp2;
			Drop_disp(_OP_ld, _reg, @@P_FRONT, _tmp2)
		endif;
		_pint(@@P_BACK) -> _offs
	endunless;

	if Mode1 == REGISTER then
		Drop_disp(_OP_ld, _reg, _int(_offs), _Opnd1);
		kill_reg(_tmp);
		_NOREG -> _tmp
	else
		if IS_NOREG(_tmp) then get_reg() -> _tmp endif;
		Drop_disp(_OP_ld, _reg, _int(_offs), _tmp)
	endif;

	unless _oneres then
		unless Mode2 == REGISTER and _Opnd2 /== _reg then
			Store_structure(_tmp2, (Mode2, _Opnd2, _Offs2));
			unless Mode2 == REGISTER then _tmp2 -> _Opnd2 endunless
		endunless;
		if push2 then Drop_push(_Opnd2) endif;
		unless _tos_Opnd == _tmp2 and tos_Mode == REGISTER then
			kill_reg(_tmp2)
		endunless
	endunless;

	unless Mode1 == REGISTER then
		Store_structure(_tmp, (Mode1, _Opnd1, _Offs1));
		_tmp -> _Opnd1
	endunless;
	if push1 then Drop_push(_Opnd1) endif;
	unless _tos_Opnd == _tmp and tos_Mode == REGISTER then
		kill_reg(_tmp)
	endunless
enddefine;

;;; updater for fast_front, fast_back, fast_cont et cetera
define I_UFASTFIELD();
	lvars _offs, _reg, _reg1, _tmp0, _tmp1;
	asm_instr!INST_ARGS[_0] -> _offs;
	Get_fsource(_1, true, false) -> (_reg, , _tmp0);
	Drop_pop_tmp(_NOREG) -> (_reg1, _tmp1);
	Drop_disp(_OP_st, _reg, _int(_offs), _reg1);
	kill_reg(_tmp0);
	kill_reg(_tmp1);
enddefine;

define I_FASTSUBV();
	;;; arg 0 is offset to first vector element (as popint)
	;;; -- subtract popint 1 to account for base 1 subscript and popint bits
	lvars 	_reg, _offs = _int(asm_instr!INST_ARGS[_0]) _sub 1, _tmp0, _tmp1,
			 Mode, _Opnd, _Offs, _reg1, push;
	Get_fsource(_1, true, false) -> (_reg, , _tmp0);
	Drop_pop_tmp(_NOREG) -> (_reg1, _tmp1);
	if IS_NOREG(_tmp1) then
		_tmp0 -> _tmp1, _NOREG -> _tmp0;
		if IS_NOREG(_tmp1) then get_reg() -> _tmp1 endif
	endif;
	Drop_rbinop(_OP_ADD, _reg1, _reg, _tmp1);
	kill_reg(_tmp0);
	Get_fdestin() -> (Mode, _Opnd, _Offs, push);
	if Mode == REGISTER then
		Drop_disp(_OP_ld, _tmp1, _offs, _Opnd)
	else
		Drop_disp(_OP_ld, _tmp1, _offs, _tmp1);
		Store_structure(_tmp1, (Mode, _Opnd, _Offs));
		_tmp1 -> _Opnd
	endif;
	if push then Drop_push(_Opnd) endif;
	unless _tos_Opnd == _tmp1 and tos_Mode == REGISTER then
		kill_reg(_tmp1)
	endunless
enddefine;

define I_UFASTSUBV();
	lvars _reg, _reg1, _tmp0, _tmp1, _tmp2;
	Get_fsource(_1, true, false) -> (_reg, , _tmp0);
	Drop_pop_tmp(_NOREG) -> (_reg1, _tmp1);
	if IS_NOREG(_tmp1) then get_reg() -> _tmp1 endif;
	Drop_rbinop(_OP_ADD, _reg1, _reg, _tmp1);
	kill_reg(_tmp0);
	Drop_pop_tmp(_NOREG) -> (_tmp2, );
	;;; arg 0 is offset to first vector element (as popint)
	;;; -- subtract popint 1 to account for base 1 subscript and popint bits
	Drop_disp(_OP_st, _tmp1, _int(asm_instr!INST_ARGS[_0]) _sub 1, _tmp2);
	kill_reg(_tmp1);
	kill_reg(_tmp2);
enddefine;


;;; --- FAST INTEGER + AND - -------------------------------------------------

	;;; {I_FAST_+-_any <plusflag> <operand 1> <operand 2>}
define I_FAST_+-_any();
	lvars _opcode, _regnum, _reg, imm, _tmp0, _tmp1, Mode, _Opnd, _Offs, push;
	if asm_instr!INST_ARGS[_0] then _OP_ADD else _OP_SUB endif -> _opcode;
	Get_fsource(_1, false, false) -> (_regnum, imm, _tmp0);
	if imm then
		_regnum _sub 0 -> _regnum
	else
		if IS_NOREG(_tmp0) then get_reg() -> _tmp0 endif;
		Drop_ibinop(_OP_SUB, _regnum, 0, _tmp0 ->> _regnum)
	endif;
	Get_fsource(_2, true, true) -> (_reg, , _tmp1);
	Get_fdestin() -> (Mode, _Opnd, _Offs, push);
	if Mode == REGISTER then
		Drop_binop(_opcode, _reg, _regnum, _Opnd, imm);
		kill_reg(_tmp0)
	else
		if IS_NOREG(_tmp1) then
			_tmp0 -> _tmp1, _NOREG -> _tmp0;
			if IS_NOREG(_tmp1) then get_reg() -> _tmp1 endif
		endif;
		Drop_binop(_opcode, _reg, _regnum, _tmp1, imm);
		kill_reg(_tmp0);
		Store_structure(_tmp1, (Mode, _Opnd, _Offs));
		_tmp1 -> _Opnd
	endif;
	if push then Drop_push(_Opnd) endif;
	unless _tos_Opnd == _tmp1 and tos_Mode == REGISTER then
		kill_reg(_tmp1)
	endunless
enddefine;


;;; --- CALLING PROCEDURES -------------------------------------------------

lconstant macro USER = 0;	;;; indicates userstack is arg

	;;; call pop procedure
define lconstant Drop_call(defer, _routine);
	lvars defer, _routine, _reg, Mode, _Opnd, _Offs;
	if _routine then _:Ro0 else _:Ro5 endif -> _reg;
	if defer == USER then
		Drop_pop(_reg)
	else
		Flush_tos(true);
		Get_structure(_0, defer) -> (Mode, _Opnd, _Offs);
		Load_structure((Mode, _Opnd, _Offs), _reg)
	endif;
	if _routine then
		Drop_imm(_OP_jmpl, _routine, _:Ro7)
	else
		lvars _tmp0 = get_reg();
		Drop_disp(_OP_ld, _:Ro5, _0, _tmp0);
		Drop_ibinop(_OP_jmpl, _tmp0, _8, _:Ro7);
		kill_reg(_tmp0)
	endif;
	Drop_us_adjust(true)
enddefine;

define I_CALL  = Drop_call(% DEFER, _popenter %) enddefine;	;;; call valof with check
define I_CALLQ = Drop_call(% DIRECT,_popenter %) enddefine;	;;; call pdr with check
define I_CALLP = Drop_call(% DEFER, false %) enddefine;		;;; call valof without check
define I_CALLS = Drop_call(% USER,  _popenter %) enddefine;	;;; call pdr off stack with check
define I_CALLPS= Drop_call(% USER,  false %) enddefine;		;;; call pdr off stack without check

	;;; call procedure without check
define I_CALLPQ();
	lvars p, _xoffs;

	define lconstant cpq_xoffs();
		lvars	(Mode, _Opnd, _Offs) = Get_structure(_0, DIRECT),
				p = asm_instr!INST_ARGS[_1], _iptr = p!PD_EXECUTE@(code)[_2],
				_xoffs = @@(code){_iptr, p};
		Flush_tos(true);
		Load_structure((Mode, _Opnd, _Offs), _:Ro5);
		if _zero(_us_off) then @@(code){_xoffs}++ -> _xoffs endif;
		Drop_ibinop(_OP_jmpl, _:Ro5, _xoffs, _:Ro7);
		if _zero(_us_off) then
			;;; use the 3rd instruction (the save) as the delay
			Drop_l(_iptr!(code));
			_NULL -> _lasti_ptr
		else
			Drop_us_adjust(true)
		endif
	enddefine;

	if _asm_pass == 0 then
		asm_instr!INST_ARGS[_0] -> p;
		;;; offset to 4th instruction (first after the save)
		if @@(code){p!PD_EXECUTE@(code)[_3], p} _slteq _16:FFF then
			;;; within 12-bit displacement -- call as jmpl ro5+_xoffs
			;;; with ro5 containing the procedure directly (i.e. don't
			;;; need to bother with PD_EXECUTE)
			Garbage_inst(asm_instr);
			Cons_inst(cpq_xoffs, p, p, 3) ->> asm_instr
											-> fast_front(asm_clist);
			chain(cpq_xoffs)
		endif
	endif;

	chain(DIRECT, false, Drop_call)
enddefine;

	;;; call system procedure at absolute address.
define I_CALLABS();
	lvars	p = asm_instr!INST_ARGS[_0], _hi = Drop_hi(p),
			_lo = Drop_lo(p), _iptr, _xoffs;
	Flush_tos(true);
	Drop_sethi(_hi, _:Ro5);
	p!PD_EXECUTE@(code)[_2] -> _iptr;
	@@(code){_iptr, _shift(_hi, _10)} -> _xoffs;
	if _xoffs _slteq _16:FFF then
		;;; within 12-bit displacement of hi part
		if _zero(_us_off) then
			Drop_ibinop(_OP_jmpl, _:Ro5, _xoffs, _:Ro7);
			Drop_ibinop(_OP_OR, _:Ro5, _lo, _:Ro5);
			_NULL -> _lasti_ptr;
			return
		elseif _iptr!(code)[-1] /== _NOP_INST then	;;; i.e. not "nop"
			;;; POPC-compiled procedure
			Drop_ibinop(_OP_jmpl, _:Ro5, --@@(code){_xoffs}, _:Ro7);
			Drop_us_adjust(true);
			return
		endif
	endif;

	unless _zero(_lo) then
		Drop_ibinop(_OP_OR, _:Ro5, _lo, _:Ro5)
	endunless;
	Drop_ibinop(_OP_jmpl, _:Ro5, @@(code){_iptr, p}, _:Ro7);
	Drop_us_adjust(true)
enddefine;

	;;; {I_CHAIN_REG <reg ident>}
	;;; chain procedure in reg
define I_CHAIN_REG();
	Flush_tos(true);
	Drop_mvreg(_int(Is_register(asm_instr!INST_ARGS[_0])),
												_:Ro5);	;;; mov %rn, %o5
	Drop_disp(_OP_ld, _:Ro5, _0, _:Ro0);						;;; ld [%o5], %o0
	Drop_ibinop(_OP_jmpl, _:Ro0, _8, _:Rg0);				;;; jmp [%o0+8]
	Drop_us_adjust(true);
enddefine;

define lconstant Drop_subr_call(_addr_or_reg, _addr, _retreg);
	lvars _addr_or_reg, _addr, _retreg, _reg, _xoffs = _0;
	Flush_tos(true);
	if _zero(_us_off) then @@(code)[_1] -> _xoffs endif;
	if Is_register(_addr_or_reg) ->> _reg then
		Drop_ibinop(_OP_jmpl, _int(_reg), _xoffs, _retreg)
	else
		Drop_imm(_OP_jmpl, _addr@(code){_xoffs}, _retreg)
	endif;
	if _zero(_us_off) then
		Drop_l(_addr!(code)[_0]);		;;; use 1st instr as delay
		_NULL -> _lasti_ptr
	else
		Drop_us_adjust(true)
	endif
enddefine;

define I_CALLSUB();
	Drop_subr_call(dup(asm_instr!INST_ARGS[_0]), _:Ro7)
enddefine;

	;;; chain subroutine
define I_CHAINSUB();
	Drop_subr_call(dup(asm_instr!INST_ARGS[_0]), _:Rg0)
enddefine;

define I_CALLSUB_REG();
	Drop_subr_call(fast_destpair(asm_instr!INST_ARGS[_0]), _:Ro7)
enddefine;


define I_UCALL	= Drop_call(% DEFER, _popuenter %) enddefine;	;;; call valof's updater with checks
define I_UCALLQ	= Drop_call(% DIRECT,_popuenter %) enddefine;	;;; call pdr's updater with checks
define I_UCALLP	= Drop_call(% DEFER, _popuncenter %) enddefine;	;;; call valof's updater without checks
define I_UCALLPQ= Drop_call(% DIRECT,_popuncenter %) enddefine;	;;; call pdr's updater without checks
define I_UCALLS	= Drop_call(% USER,  _popuenter %) enddefine;	;;; call updater from stack with checks
define I_UCALLPS= Drop_call(% USER,  _popuncenter %) enddefine;	;;; call updater from stack without checks


;;; --- CONDITIONALS -------------------------------------------------------

;;; {I_IF_opt label flag I_BRCOND}
;;; {I_BOOL_opt label flag I_BRCOND}
;;; flag is 0 for for ifso & and, 1 for ifnot & or
;;;
define I_BRCOND(_ifso, _ifnot, is_so, _broffset, _argnum);
	lvars is_so, _argnum, _broffset, _ifso, _ifnot;

	Flush_tos(true);
	Drop_branch(
		if is_so then
			_ifso		;;; cond marker for ifnot, or etc
		else
			_ifnot		;;; cond marker for ifso, and etc
		endif, _broffset, false);
	Drop_us_adjust(true)
enddefine;

	;;; call the I_BRCOND routine for an I_IF_opt, I_BOOL_opt or I_IF_CMP
	;;; apply the conditional branch routine in arg2
	;;; (initial branch routine is I_BRCOND)
define lconstant Drop_if(instr) with_nargs 3;		;;; ifso/ifnot codes on stack
	lvars instr;
	dlocal asm_instr = instr;
	fast_apply(/* ifso, ifnot */
				instr!INST_ARGS[_1],
				_int(fast_front(instr!INST_ARGS[_0])),
				_2,
				instr!INST_ARGS[_2]);
enddefine;

	/*	{I_IF_opt _____label ____ifso I_BRCOND _______operand}
	*/
define I_IF_opt();
	lvars _reg, imm, _tmp0;
	;;; drop cpw input operand or stack pop with false
	Get_fsource(_3, false, true) -> (_reg, imm, _tmp0);
	Drop_binop(_OP_SUBcc, _:Rfalse, _reg, _:Rg0, imm);
	kill_reg(_tmp0);
	Drop_if(_cond_NE, _cond_EQ, asm_instr);			;;; ifso, ifnot
enddefine;

	/*	{I_BOOL_opt _____label ____ifso I_BRCOND _______operand}
	*/
define I_BOOL_opt();
	lvars _reg, imm, _tmp0, _was_tos = tos_Mode, _was_copy = _tos_iscopy;
	Get_fsource(_3, false, true) -> (_reg, imm, _tmp0);
	if asm_instr!INST_ARGS[_3] or (_was_tos and not(_was_copy)) then
		;;; push the explicit operand
		if imm then
			if IS_NOREG(_tmp0) then get_reg() -> _tmp0 endif;
			Drop_set(_reg, _tmp0 ->> _reg)
		endif;
		Drop_push(_reg);
		Flush_tos(false)
	else
		;;; undo the stack pop
		_us_off _sub _4 -> _us_off
	endif;
	Drop_rbinop(_OP_SUBcc, _reg, _:Rfalse, _:Rg0);	;;; compare with false
	kill_reg(_tmp0);
	false -> tos_Mode;
	Drop_if(_cond_NE, _cond_EQ, asm_instr);			;;; or, and
	Increment_us()									;;; erase top of stack
enddefine;

	/*	{I_IF_CMP _________routine ________operand1 ________operand2 ______________I_IF_opt-instr}
	*/
define I_IF_CMP();
	lvars instr = asm_instr, _type, imm, _reg, _regnum, _tmp0, _tmp1;

	;;; branch conditions are used ARG1 op ARG2
	lconstant
		compare_ops =
			[%	nonop _eq,		_pint(_cond_EQ), _pint(_cond_NE),
				nonop _neq,		_pint(_cond_NE), _pint(_cond_EQ),
				nonop _sgr,		_pint(_cond_GT), _pint(_cond_LE),
				nonop _sgreq,	_pint(_cond_GE), _pint(_cond_LT),
				nonop _slt,		_pint(_cond_LT), _pint(_cond_GE),
				nonop _slteq,	_pint(_cond_LE), _pint(_cond_GT),
			%];

	Get_fsource(_1, false, false) -> (_regnum, imm, _tmp0);
	Get_fsource(_2, true, true) -> (_reg, , _tmp1);
	Drop_binop(_OP_SUBcc, _reg, _regnum, _:Rg0, imm);
	kill_reg(_tmp0);
	kill_reg(_tmp1);

	;;; get ifso/ifnot branch conditions for compare subroutine
	fast_back(lmember(instr!INST_ARGS[_0], compare_ops)) -> _type;

	;;; do the I_IF_opt
	Drop_if(_int(fast_front(_type)),			;;; opcode for SO
			_int(fast_front(fast_back(_type))), ;;; opcode for NOT
			instr!INST_ARGS[_3])
enddefine;

	/*	{I_IF_TAG _________routine _______operand ______________I_IF_opt-instr}
		where _________routine is _issimple or _isinteger
	*/
define I_IF_TAG();
	lvars instr = asm_instr, _regnum, _tmp0;
	Get_fsource(_1, true, true) -> (_regnum, , _tmp0);
	Drop_ibinop(_OP_ANDcc, _regnum,
				if instr!INST_ARGS[_0] == _issimple then _2:01
				else _2:10
				endif,
				_:Rg0);
	kill_reg(_tmp0);
	Drop_if(_cond_NE, _cond_EQ, instr!INST_ARGS[_2])	;;; ifso, ifnot
enddefine;


;;; --- JUMPS --------------------------------------------------------------

	;;; branch instruction of standard size
define I_BR_std(_broffset, _argnum);
	lvars _broffset, _argnum;
	if _nonzero(_us_off) then
		mishap(0, 'SYSTEM ERROR 1 IN I_BR_std');
	endif;
	Drop_branch(_cond_ALL, _broffset, true);
enddefine;

define I_BR(_broffset, _argnum);
	lvars _broffset, _argnum;
	Flush_tos(true);
	if _zero(_us_off) then
		Drop_branch(_cond_ALL, _broffset, true);
	else
		Drop_branch(_cond_ALL, _broffset, false);
		Drop_us_adjust(true);
	endif;
enddefine;

	/*	{I_SWITCH_base __________label-list __________else-label _______operand ____________base-integer}
		Plant a computed jump.
		If there's no else label, leave arg on stack for error message
	*/
define I_SWITCH_base();
	lvars	instr = asm_instr, lablist, elselab, base, _laboffset,
			_save, _reg, _tmp0, _argreg, _aftersw;
	instr!INST_ARGS[_0] -> lablist;
	instr!INST_ARGS[_1] -> elselab;
	instr!INST_ARGS[_3] -> base;
	listlength(lablist) -> _laboffset;

	Get_fsource(_2, true, true) -> (_reg, , _tmp0);
	_reg -> _argreg;
	unless 0 fi_<= base and base fi_<= 4 then
		if IS_NOREG(_tmp0) then
			get_reg() -> _tmp0
		elseunless elselab then
			Drop_mvreg(_reg, get_reg() ->> _argreg)
		endif;
		Drop_ibinop(_OP_SUB, _reg, base _sub 0, _tmp0 ->> _reg);
		0 -> base
	endunless;

	Drop_ibinop(_OP_SUBcc, _reg, _laboffset fi_+ base fi_- 1, _:Rg0);	;;; cmp off, %t0
	branch_src() -> _save;							;;; bgu END
	;;; branch delay slot safe if there is an else label
	Drop_us_adjust(not(elselab));

	if IS_NOREG(_tmp0) then
		get_reg() -> _tmp0
	elseunless elselab then
		Drop_mvreg(_reg, get_reg() ->> _argreg)
	endif;

	/*	_____reg contains a popint subscript -- jump to
		instr following last planted instr for subscript 0
		The offset from pb depends on _strsize (which is unknown on the
		first pass), and we must always plant the same number of words of code,
		so we make it upto 5 instrs with nops after the jmp delay
	*/
	_asm_code_offset _add @@(w)[_5] -> _aftersw;	;;; max 5 instrs

	Drop_ibinop(_OP_ADD, _reg, @@PD_TABLE{_strsize _add _aftersw _sub 0},
								_tmp0);				;;; add %tmp, base, %tmp
	Drop_rbinop(_OP_jmpl, _:Rpb, _tmp0, _:Rg0);			;;; jmp %pb+%tmp
	Drop_us_adjust(true);

	;;; then make it up to 5 instrs if necessary
	while _asm_code_offset _lt _aftersw do Drop_nop() endwhile;
	kill_reg(_tmp0);

	@@(code)[_int(_laboffset)] -> _laboffset;
	until base == 0 do
		Drop_branch(_cond_ALL, _asm_code_offset _add @@(code)[_int(base)]
								_add _laboffset, true);
		base fi_- 1 -> base
	enduntil;
	until lablist == [] do
		fast_front(destpair(lablist) -> lablist) -> _laboffset;
		Drop_branch(_cond_ALL, _int(_laboffset), true)
	enduntil;
	branch_dst(_cond_GTU, false, _save);            ;;; END:
	unless elselab then Drop_push(_argreg) endunless
enddefine;

	;;; {I_PLOG_IFNOT_ATOM <fail_label} I_BRCOND}
define I_PLOG_IFNOT_ATOM();
	lvars instr = asm_instr;
	lconstant _FLAB = _0, _FBR = _1;
	;;; apply the conditional branch routine in arg1
	;;; (initial branch routine is I_BRCOND)
	fast_apply(_cond_NE, _cond_EQ,		;;; ne, eq
				true,						;;; select ne
				_int(fast_front(instr!INST_ARGS[_FLAB])),
				_FBR,
				instr!INST_ARGS[_FBR]);
enddefine;

	;;; {I_PLOG_TERM_SWITCH <fail_label} I_BRCOND <var_label> I_BRCOND}
	;;; after call to plog switch routine, switch afterwards to var_label,
	;;; fail_label or drop thru.
define I_PLOG_TERM_SWITCH();
	lvars instr = asm_instr;
	lconstant _FLAB = _0, _FBR = _1, _VLAB = _2, _VBR = _3;
	;;; branch if greater than set to var_label
	fast_apply(_cond_GT, _cond_LE,					;;; SO/gt, NOT/le
				true,
				_int(fast_front(instr!INST_ARGS[_VLAB])),
				_VBR,
				instr!INST_ARGS[_VBR]);
	;;; branch if less than set to fail_label
	fast_apply(_cond_LT, _cond_GE,					;;; SO/lt, NOT/ge
				true,
				_int(fast_front(instr!INST_ARGS[_FLAB])),
				_FBR,
				instr!INST_ARGS[_FBR]);
	;;; else fall thru for pair/term case (push on arg_reg_0 follows)
enddefine;

define I_CHECK();
	lvars _save_ch, _save_ok, _tmp0, _tmp1, instr, optim = false;
	Flush_tos(true);
	get_reg() -> _tmp0;
	Drop_disp(_OP_ld, _:Rsvb, _SVB_OFFS(_trap), _tmp0);	;;; ld _trap, %t0
	get_reg() -> _tmp1;
	Drop_disp(_OP_ld, _:Rsvb, _SVB_OFFS(_userlim), _tmp1);	;;; ld _userlim, %t1
	Drop_us_adjust(false);
	Drop_ibinop(_OP_ANDcc, _tmp0, _1, _:Rg0);			;;; andcc %t0, 1, %t0
	kill_reg(_tmp0);
	branch_src() -> _save_ch;						;;; bne ch
	Drop_rbinop(_OP_SUBcc, _:Rusp, _tmp1, _:Rg0);			;;; subcc %us, %t1, %t1
	kill_reg(_tmp1);
	fast_front(fast_back(asm_clist)) -> instr;
	if isvector(instr)
	and (instr!INST_OP == I_GOTO or instr!INST_OP == I_GOTO_opt)
	and instr!INST_ARGS[_1] == I_BR then
		Drop_branch(_cond_GTU, _int(fast_front(instr!INST_ARGS[_0])), false);
		true -> optim;
	else
		branch_src() -> _save_ok;						;;; bgu ok
		;;; branch delay slot safe
	endif;
	branch_dst(_cond_NE, false, _save_ch);			;;; ch:
	Drop_imm(_OP_jmpl, _checkall, _:Ro7);				;;; jmpl _checkall, %o7
	Drop_us_adjust(true);
	unless optim then
		branch_dst(_cond_GTU, false, _save_ok);			;;; ok:
	endunless;
enddefine;

	;;; return user stack length
define I_STACKLENGTH();
	lvars _tmp0, _tmp1, Mode, _Opnd, _Offs, push;
	Flush_tos(true);
	get_reg() -> _tmp0;
	Drop_disp(_OP_ld, _:Rsvb, _SVB_OFFS(_userhi), _tmp0);	;;; ld _userhi, %t0
	get_reg() -> _tmp1;
	Drop_ibinop(_OP_SUB, _:Rusp, _3 _sub _us_off, _tmp1);
	Get_fdestin() -> (Mode, _Opnd, _Offs, push);
	if Mode == REGISTER then
		Drop_rbinop(_OP_SUB, _tmp0, _tmp1, _Opnd)
	else
		Drop_rbinop(_OP_SUB, _tmp0, _tmp1, _tmp1);
		Store_structure(_tmp1, (Mode, _Opnd, _Offs));
		_tmp1 -> _Opnd
	endif;
	kill_reg(_tmp0);
	if push then Drop_push(_Opnd) endif;
	unless _tos_Opnd == _tmp1 and tos_Mode == REGISTER then
		kill_reg(_tmp1)
	endunless
enddefine;

	;;; set user stack length to saved stacklength + nresults
define I_SETSTACKLENGTH();
	lvars sl, nr, _save, _reg, _tmp0;
	asm_instr!INST_ARGS[_0] -> sl;
	asm_instr!INST_ARGS[_1] -> nr;
	if nr then
		Get_fsource(_0, true, true) -> (_reg, , _tmp0);
		Drop_disp(_OP_ld, _:Rsvb, _SVB_OFFS(_userhi), _:Ro0);
		Drop_us_adjust(false);
		if IS_NOREG(_tmp0) then get_reg() -> _tmp0 endif;
		Drop_ibinop(_OP_ADD, _reg, _int((nr fi_* 4) fi_- 3), _tmp0);
		Drop_rbinop(_OP_SUB, _:Ro0, _tmp0, _:Ro0);
		kill_reg(_tmp0);
		Drop_rbinop(_OP_SUBcc, _:Ro0, _:Rusp, _:Rg0);
		branch_src() -> _save;
		;;; branch delay slot safe
		Drop_imm(_OP_jmpl, _setstklen_diff, _:Ro7);
		Drop_us_adjust(true);
		branch_dst(_cond_EQ, false, _save);
	else
		Flush_tos(true);
		Drop_imm(_OP_jmpl, _setstklen, _:Ro7);
		Drop_us_adjust(true);
	endif;
enddefine;

;;; convert pop11 result to lisp result i.e. replace false with nil
define I_LISP_TRUE();
	lvars	_save, _tmp, _reg, updating,
			(Mode, _Opnd, _Offs, push) = Get_fdestin();
	(Mode == AUTO and (not(tos_Mode) or _tos_iscopy))
	or (Mode == tos_Mode and _Opnd == _tos_Opnd and _Offs == _tos_Offs)
					-> updating;

	if Mode == REGISTER then
		_Opnd -> _reg, _NOREG -> _tmp;
		Drop_pop(_reg)
	else
		Drop_pop_tmp(_NOREG) -> (_reg, _tmp);
		if IS_NOREG(_tmp) then
			get_reg() -> _tmp;
			Drop_mvreg(_reg, _tmp ->> _reg)
		endif
	endif;

	Drop_rbinop(_OP_SUBcc, _reg, _:Rfalse, _:Rg0);	;;; cmp %reg, %g6
	branch_src() -> _save;							;;; bne NOTFALSE
	Drop_nop();
	Drop_set([], _reg);								;;; set c.nil, %reg
	if updating and Mode /== REGISTER then
		Store_structure(_reg, (Mode, _Opnd, _Offs));
		Flush_tos(false)
	endif;

	branch_dst(_cond_NE, false, _save);				;;; NOTFALSE:

	if not(updating) and Mode /== REGISTER then
		Store_structure(_reg, (Mode, _Opnd, _Offs))
	endif;
	if push then Drop_push(_reg) endif;
	unless _tos_Opnd == _tmp and tos_Mode == REGISTER then
		kill_reg(_tmp)
	endunless
enddefine;


;;; --- PROLOGUE, EPILOGUE -------------------------------------------------

	;;; generate code to construct stack frame
define I_CREATE_SF();
	lvars _n, _dloc_offs, _offs, _stoff, _opcode, _popreg, _maxpopreg;
	;;; _-1 -> _condbr_nop;		;;; reset squashable nop pointer
	init_reg();					;;; initialise temp registers
	_0 -> _us_off;				;;; zero user stack offset
	_NULL -> _lasti_ptr;
	false -> tos_Mode;

	Drop_nop();
	Drop_nop();

	;;; save callers window
	@@SF_LOCALS _add @@(w)[_Nstkvars] -> _dloc_offs;
	Drop_ibinop(_OP_save, _:Rsp, _negate(_dloc_offs _add @@(w)[_Nlocals]), _:Rsp);

	;;; initialise pop registers to popint zero -- set up counter
	_Npopreg _add (_:Rl0 _sub _1) ->> _popreg -> _maxpopreg;

	;;; save dynamic locals
	_0 -> _offs;
	@@PD_TABLE -> _stoff;                        ;;; struct table offset
	_Nlocals -> _n;
	while _n _gr _0 do
		if _n == _1 then
			_offs _sub _4 -> _offs;
			Drop_disp(_OP_ld, _:Rpb, _stoff, _:Ro0);
			Drop_disp(_OP_ld, _:Ro0, @@ID_VALOF, _:Ro0);
			_OP_st -> _opcode;
			_stoff _add _4 -> _stoff;
			_n _sub _1 -> _n;
		else
			_offs _sub _8 -> _offs;
			Drop_disp(_OP_ld, _:Rpb, _stoff, _:Ro1);
			Drop_disp(_OP_ld, _:Rpb, _stoff _add _4, _:Ro0);
			Drop_disp(_OP_ld, _:Ro1, @@ID_VALOF, _:Ro1);
			Drop_disp(_OP_ld, _:Ro0, @@ID_VALOF, _:Ro0);
			_OP_std -> _opcode;
			_stoff _add _8 -> _stoff;
			_n _sub _2 -> _n;
		endif;
		if _popreg _greq _:Rl0 then
			Drop_ibinop(_OP_OR, _:Rg0, _3, _popreg);
			_popreg _sub _1 -> _popreg;
		endif;
		Drop_disp(_opcode, _:Ri6, _offs, _:Ro0);
	endwhile;

	;;; initialise remainder of pop registers
	until _popreg _lt _:Rl0 do
		Drop_ibinop(_OP_OR, _:Rg0, _3, _popreg);
		_popreg _sub _1 -> _popreg;
	enduntil;

	;;; create on-stack vars -- initialised to popint zero
	_Npopstkvars -> _n;
	_dloc_offs -> _offs;
	while _n _gr _0 do
		if _maxpopreg _lt _:Rl0 then
			Drop_ibinop(_OP_OR, _:Rg0, _3, _:Rl0 ->> _maxpopreg)
		endif;
		if _n == _1 or _nonzero(_offs _bimask _4) then
			_offs _sub _4 -> _offs;
			Drop_disp(_OP_st, _:Rsp, _offs, _:Rl0);
			_n _sub _1 -> _n;
		else
			if _maxpopreg _lt _:Rl1 then
				Drop_ibinop(_OP_OR, _:Rg0, _3, _:Rl1 ->> _maxpopreg)
			endif;
			_offs _sub _8 -> _offs;
			Drop_disp(_OP_std, _:Rsp, _offs, _:Rl0);
			_n _sub _2 -> _n;
		endif;
	endwhile;
enddefine;

	;;; return
define I_RETURN();
	Flush_tos(true);
	Drop_ibinop(_OP_jmpl, _:Ro7, _8, _:Rg0);					;;; ret
	Drop_us_adjust(true);
enddefine;

	;;; generate code to unwind stack frame
	;;;
	;;; This code MUST NOT use CHAIN_REG (%g1), because this can contain
	;;; the saved return address after an I_CHAINS
define I_UNWIND_SF();
	lvars _fpoff, _stoff, _n, instr;
	Flush_tos(true);

	;;; restore dynamic locals
	_0 -> _fpoff;
	@@PD_TABLE -> _stoff;
	_Nlocals -> _n;
	while _n _gr _0 do
		if _n == _1 then
			_fpoff _sub _4 -> _fpoff;
			Drop_disp(_OP_ld, _:Ri6, _fpoff, _:Ro0);
			Drop_disp(_OP_ld, _:Rpb, _stoff, _:Ro1);
			Drop_disp(_OP_st, _:Ro1, @@ID_VALOF, _:Ro0);
			_stoff _add _4 -> _stoff;
			_n _sub _1 -> _n;
		else
			_fpoff _sub _8 -> _fpoff;
			Drop_disp(_OP_ldd, _:Ri6, _fpoff, _:Ro0);
			Drop_disp(_OP_ld, _:Rpb, _stoff, _:Ro5);
			Drop_disp(_OP_ld, _:Rpb, _stoff _add _4, _:Ro7);
			Drop_disp(_OP_st, _:Ro5, @@ID_VALOF, _:Ro1);
			Drop_disp(_OP_st, _:Ro7, @@ID_VALOF, _:Ro0);
			_stoff _add _8 -> _stoff;
			_n _sub _2 -> _n;
		endif;
	endwhile;

	;;; restore caller's window and adjust user stack
	fast_front(fast_back(asm_clist)) -> instr;
	if isvector(instr) and instr!INST_OP == I_RETURN then
		Drop_ibinop(_OP_jmpl, _:Ri7, _8, _:Rg0);
		fast_back(asm_clist) -> asm_clist;
	endif;
	Drop_ibinop(_OP_restore, _:Rusp, _us_off, _:Rusp);
	_0 -> _us_off;
enddefine;

define I_LABEL();
	;;; _-1 -> _condbr_nop;			;;; preceding nop cannot be squashed
	Flush_tos(true);
	Drop_us_adjust(false);
	_NULL -> _lasti_ptr;
	;;; set front of label pair to popint offset from code start PD_EXECUTE
	_pint(_asm_code_offset) -> fast_front(asm_clist)
enddefine;


;;; --- THE ASSEMBLER ------------------------------------------------------


;;; ASSEMBLER TO PRODUCE MACHINE CODE IN PROCEDURE RECORD FROM
;;; POP ASSEMBLER INPUT

;;; args: list, list
define Do_consprocedure(codelist, reg_locals) -> pdr;
	dlocal
		_asm_drop_ptr,		;;; sysint pointer to drop code at
		_asm_pass,			;;; pass number or false if last
		_strsize,			;;; sysint byte size of structure table
		_us_off,			;;; user stack offset from %rusp
	;

	lvars
		pdr,			;;; result
		reg,			;;; looper
		reg_locals,		;;; argument list of local register identifiers
		codelist,		;;; static list of instructions
		_code_offset,	;;; size of code
		_size,			;;; size of procedure record
		_reg_spec = _0, ;;; spec to go in PD_REGMASK
	;

	;;; first pass - calculate instruction offsets
	_0 -> _strsize;
	false -> _big_procedure;
	Code_pass(0, codelist) -> _code_offset;

	@@(w)[_int(listlength(asm_struct_list))] -> _strsize;
	_strsize _add _code_offset -> _size;

	if _size _greq _16:800000 then
		;;; You want me to handle an 8MB procedure?
		mishap(0, 'SYSTEM ERROR: PROCEDURE TOO LARGE');
		;;; true -> _big_procedure;
		;;; Code_pass(1, codelist) -> _code_offset;
	endif;

	;;; third pass for branch size optimisations in bigprocs (not yet implemented)
	;;; unless pop_debugging then
	;;;		Code_pass(2, codelist) -> _code_offset;
	;;; endunless;

	;;; can now calculate total size of procedure and allocate store for it
	@@PD_TABLE{_strsize _add _code_offset | b.r} _sub @@POPBASE -> _size;

	;;; get the procedure record with header filled in and struct table
	;;; generated (sets up _asm_drop_ptr)
	Get_procedure(_size, _reg_spec) -> pdr;

	;;; final pass - generate code
	Code_pass(false, codelist) ->;		;;; generate code
enddefine;

endsection;		/* $-Sys$-Vm */



/* --- Revision History ---------------------------------------------------
--- John Gibson, Jul  6 1998
		Fixed Do_field to use Rg3 instead of Ro0 to hold update value off
		stack (since input pointer can be in Ro0).
--- John Gibson, Oct  1 1997
		Now includes drop_code.ph.
--- John Gibson, Nov  4 1996
		# Converted I_PUSH/POP_FIELD(_ADDR) instructions to new format.
		# Improved I_SWAP.
		# Allowed I_ERASE to take optional extra boolean arg to say
		  stack access not necessary (i.e. just increment stack pointer).
--- John Gibson, Oct 17 1996
		Numerous improvements, including caching of top-of-stack operand.
		Now defines I_FAST_+-_any and I_SWITCH_base.
--- John Gibson, Sep 27 1996
		Added I_IF_TAG
--- John Gibson, Sep 16 1994
		Rewrote register allocation routines and added optimisation of trying
		to move an arith instruction or store to after a following load
--- John Gibson, Sep 12 1994
		Improved I_BOOL_opt
--- John Gibson, Sep  5 1994
		Made Drop_set use _:Rfalse + offset where possible for a literal
--- John Gibson, Mar 20 1992
		Fixed nasty bug in I_SWITCH. It was overlooking that the fixed
		offset from PB for the indexed jump depends on -strsize-, which
		hasn't been computed on the first pass (therefore can't tell
		whether it fits in 12 bits, so must assume the worst case to ensure
		that the same code is dropped on the second pass).
--- John Gibson, Nov 26 1990
		Optimised I_PUSH_FIELD_ADDR to save code with 0 offset.
--- John Gibson, Aug 18 1990
		-size- argument to field instructions now in units of type size;
		changed -Do_field- to detect size/==1 case for subscripting
		external array of structures/arrays etc.
--- John Gibson, Mar 21 1990
		Changed field access procedures so that when -exptr- argument
		is true, it's an integer specifying deref first that many times.
--- John Gibson, Dec 11 1989
		Changes for new pop pointers: (1) use ID_VALOF offset
		for deferred accesses in -P*ut_structure-, and for
		saving/restoring dynamic locals in I_CREATE/UNWIND_SF;
		(2) I_(U)FASTSUBV changed to take offset to vector elements
		start in structure (rather than offset from V_WORDS).
--- John Gibson, Sep  7 1989
		Fixed bug in I_CREATE_SF which failed to initialise pop stack lvars
		correctly when procedure had 0 or 1 reg lvars (can happen when
		stack lvars are run-time idents, which don't get allocated to
		registers).
--- John Gibson, Jun 30 1989
		Added I_CALLPS, I_UCALLPS (for -fast_apply-)
--- John Gibson, Apr 30 1989
		Put into section $-Sys$-Vm
--- Roger Evans, Oct 10 1988 (actually Robert Smith)
	Modified Do_field to accept -offset- <false> to
	mean vector subscript on stack, and have a new arg -ext- meaning
	plant code for external field access (ie indirect on E_PTR field
	and do 3rd longword correction on offset). I_PUSH_FIELD,
	I_PUSH_FIELD_ADDR and I_POP_FIELD take this -ext- arg an pass it on
--- Robert Smith, Sep 10 1988
	Changed temporary variable %o2 to %o7 as former defined as arg_reg_2.
--- Robert Smith, Sep  8 1988
	Assign temporary registers from queue. Access special variable block
	variables using base register. Improved I_SWITCH, I_CHAIN_REG, Do_field,
	Drop_call. Many other minor mods.
--- Robert Smith, Sep  5 1988
	Fixed bug in I_SETSTACKLENGTH. Improved I_CHECK and I_BR.
 */
