/* --- Copyright University of Sussex 1994. All rights reserved. ----------
 > File:            C.vms/src/sysio.p
 > Purpose:			Basic system I/O procedures (Vax/Vms)
 > Author:          John Gibson (see revisions)
 > Documentation:	REF *SYSIO
 */


#_INCLUDE 'declare.ph'
#_INCLUDE 'io.ph'
#_INCLUDE 'vmsdefs.ph'

global constant
		procedure (Sys$-Endstring_to_num)
	;

section $-Sys$-Io;

constant
		procedure (Opencreate, Cons_device, Kill_device,
		Rms_open, Rms_create, Rms_erase)
	;

vars
		_over_diskquota
	;

endsection;


;;; ----------------------------------------------------------------------

section $-Sys$-Io => pop_file_versions, sysopen, readable, syscreate,
					 sysflush, sysclose, sysdelete;

constant
	work_ctrl_blk	= writeable inits(RAB_ICB_LEN),
	;

vars
	pop_file_versions	= false,
	;


;;; --- CHANGING STANDARD DEVICES ------------------------------------------

define New_std_dev(new_dev, std_dev_id, check_flags);
	lvars new_dev, std_dev_id, check_flags;
	Check_device(new_dev, check_flags);
	new_dev -> fast_idval(std_dev_id)
enddefine;


;;;-------------------------------------------------------------------------

define sysdelete(file);
	lvars file;
	sysfileok(file) -> file;
	Rms_erase(file, work_ctrl_blk)		;;; return result of this
enddefine;

define sysopen(file, mode, arg3);
	lvars file, ctrl_blk, mode, arg3, fullname, _errc = `F`;

	if isinteger(arg3) then
		;;; optional character arg specifying error action
		((), file, mode, arg3) -> (file, mode, arg3, _errc);
		unless _errc==`N` or _errc==`F` or _errc==`D` or _errc==`A` then
			mishap(_errc, 1, 'sysopen: INVALID ERROR CHARACTER CODE')
		endunless
	endif;

	Check_string(file);
	Check_integer(mode, 0);
	inits(RAB_ICB_LEN) -> ctrl_blk;		;;; control block
	returnif( Opencreate(file, mode, arg3, ctrl_blk, Rms_open) ->> fullname )
					( Cons_device(file, fullname, mode, arg3, ctrl_blk) );

	;;; can't open file
	unless _syserror == _:'RMS$_ACC' or _syserror == _:'RMS$_DME' then
		;;; error to do with file/dir not existing or invalid name etc
		;;; (as opposed to a resource error, e.g. too many files open).
		returnif(
			_errc == `A`
			or _errc == `D` and (_syserror == _:'RMS$_DNF'
								or _syserror == _:'RMS$_DEV')
			or (_errc == `D` or _errc == `F`) and _syserror == _:'RMS$_FNF'
			) (false)
	endunless;
	Syserr_mishap(file, 1, 'CAN\'T OPEN FILE')
enddefine;

define readable = sysopen(% 0, false, `A` %) enddefine;;


define Delete_old_version(open_name, full_name);
	lvars open_name, full_name, _n, _len, _vers;
	returnunless(isinteger(pop_file_versions) and pop_file_versions fi_> 0);
	if locchar(`;`, 1, open_name) ->> _n then
		;;; don't mess with old versions if file opened with explicit
		;;; version (other than 0)
		datalength(open_name) -> _len;
		returnif(_n fi_< _len and
				(_n fi_+ 1 /== _len or fast_subscrs(_len,open_name) /== `0`))
	endif;
	;;; get version number
	datalength(full_name) -> _len;
	if (locchar_back(`;`, _len, full_name) ->> _n) and _n /== _len
	and (Endstring_to_num(_n fi_+ 1, full_name) ->> _vers)
	and _pint(_vers) fi_> pop_file_versions then
		;;; get full_name of deletable file and delete
		sysdelete(substring(1, _n, full_name)
							sys_>< (_pint(_vers) fi_- pop_file_versions)) -> ;
	endif
enddefine;

define syscreate(file, mode, arg3) -> dev;
	lvars file, ctrl_blk, mode, arg3, dev, fullname, _devflags, _errc = `N`;

	if isinteger(arg3) then
		;;; optional character arg specifying error action
		((), file, mode, arg3) -> (file, mode, arg3, _errc);
		unless _errc == `N` or _errc == `F` then
			mishap(_errc, 1, 'syscreate: INVALID ERROR CHARACTER CODE')
		endunless
	endif;

	Check_string(file);
	Check_integer(mode, 0);

	mode fi_&& 2:11 -> mode;
	if _errc == `F` then
		;;; exclusive create, i.e. return false if exists
		mode fi_|| 2:100 -> mode
	endif;

	inits(RAB_ICB_LEN) -> ctrl_blk;		;;; control block
	if Opencreate(file, mode, arg3, ctrl_blk, Rms_create) ->> fullname then
		if (ctrl_blk!ICB_DEVCHAR ->> _devflags) _bitst _:'DEV$M_FOD'
		and _devflags _bitst _:'DEV$M_RND' then
			;;; (file orientated/random = disk file)
			;;; can't be over quota if disk file created
			false -> _over_diskquota
		endif;
		Cons_device(file, fullname, mode, arg3, ctrl_blk) -> dev;
		Delete_old_version(dev!D_OPEN_NAME, dev!D_FULL_NAME)
	elseif _errc == `F` and _syserror == _:'RMS$_FEX' then
		;;; exclusive create failed because file exists
		false -> dev
	else
		Syserr_mishap(file, 1, 'CAN\'T CREATE FILE')
	endif
enddefine;

define sysclose(dev);
	lvars dev;
	Check_device(dev, false);
	fast_apply(dev, dev!D_CLOSE);
	Kill_device(dev)
enddefine;

define sysflush(dev);
	lvars dev;
	if isboolean(dev) then () -> dev endif;		;;; for Unix compatibility
	Check_device(dev, true);
	fast_apply(dev, dev!D_FLUSH)
enddefine;

endsection;		/* $-Sys$-Io */



/* --- Revision History ---------------------------------------------------
--- John Gibson, Feb 22 1994
		Added 'exclusive' create mode to syscreate
--- John Gibson, Sep 28 1993
		Changed sysflush to ignore optional 2nd bool arg for Unix
		compatibility
--- Robert John Duncan, Aug  7 1992
		Endstring_to_num now returns a system integer.
--- John Gibson, Apr  1 1991
		-Delete_old_version- changed to use -Endstring_to_num-.
--- John Gibson, May 27 1990
		Added optional character arg specifying error action to -sysopen-
--- John Gibson, Aug  1 1989
		Extracted -Delete_old_version-, stopped it ignoring
		pop_file_versions == 1, made it do nothing when explicit version
		given.
--- John Gibson, Feb 19 1989
		Included io.ph
--- John Williams, Oct  6 1988
		Fixed for VED devices
--- John Gibson, Apr 26 1988
		-current_directory- to vms_dir.p, -sys_file_match- to sys_file_match.p
--- John Gibson, Mar 15 1988
		Moved -sysread- and -syswrite- to separate file sysreadwrite.p
--- John Gibson, Feb 25 1988
		Rewrote -sys_file_match- to work without processes.
--- John Gibson, Feb 22 1988
		Check_string into section Sys
--- John Gibson, Feb 11 1988
		Check_integer in section Sys
--- John Gibson, Jan 12 1988
		Added lconstants RAB_ICB_LEN and NAM_ICB_LEN to tie initialisation
		of control blocks with -inits- to size of IO_CONTROL_BLOCK struct
		defined in vmsdefs.ph.
 */
