/* --- Copyright Integral Solutions Ltd 1998. All rights reserved. --------
 > File:			C.win32/lib/lib/windows_sockets.p
 > Purpose:			Partial implementation of Unix-style Sockets for Windows
 > Author:			Robert Duncan, Apr  3 1997 (see revisions)
 */
compile_mode :pop11 +strict;

include sysdefs;

#_IF not(DEF WIN32)
	#_TERMIN_IF DEF POPC_COMPILING
	mishap(0, 'SOCKETS NOT SUPPORTED IN THIS SYSTEM');
#_ENDIF

section;
exload_batch;

uses popwinlib;

include win_types;

lconstant macro (

	INVALID_SOCKET		= 16:FFFFFFFF,
	SOCKET_ERROR		= -1,
	INADDR_ANY			= 0,
	INADDR_NONE			= 16:FFFFFFFF,

	SOCK_STREAM			= 1,
	SOCK_DGRAM			= 2,

	AF_UNIX				= 1,
	AF_INET				= 2,

	FD_READ_BIT			= 0,
	FD_READ				= (1 << FD_READ_BIT),
	FD_WRITE_BIT		= 1,
	FD_WRITE			= (1 << FD_WRITE_BIT),
	FD_OOB_BIT			= 2,
	FD_OOB				= (1 << FD_OOB_BIT),
	FD_ACCEPT_BIT		= 3,
	FD_ACCEPT			= (1 << FD_ACCEPT_BIT),
	FD_CONNECT_BIT		= 4,
	FD_CONNECT			= (1 << FD_CONNECT_BIT),
	FD_CLOSE_BIT		= 5,
	FD_CLOSE			= (1 << FD_CLOSE_BIT),
	FD_QOS_BIT			= 6,
	FD_QOS				= (1 << FD_QOS_BIT),
	FD_GROUP_QOS_BIT	= 7,
	FD_GROUP_QOS		= (1 << FD_GROUP_QOS_BIT),
	FD_MAX_EVENTS		= 8,
	FD_ALL_EVENTS		= ((1 << FD_MAX_EVENTS) - 1),

	FD_READ_CLOSE		= FD_READ||FD_CLOSE,

	WSADESCRIPTION_LEN	= 256+1,
	WSASYS_STATUS_LEN	= 128+1,

	WSABASEERR			= 10000,
	WSAEINVAL			= (WSABASEERR+22),
	WSAEMFILE			= (WSABASEERR+24),
	WSAEWOULDBLOCK		= (WSABASEERR+35),
	WSAEINPROGRESS		= (WSABASEERR+36),
	WSAENETDOWN			= (WSABASEERR+50),
	WSAENETUNREACH		= (WSABASEERR+51),
	WSAECONNABORTED		= (WSABASEERR+53),
	WSAECONNRESET		= (WSABASEERR+54),
	WSAENOBUFS			= (WSABASEERR+55),
	WSAEISCONN			= (WSABASEERR+56),
	WSAENOTCONN			= (WSABASEERR+57),
	WSAETIMEDOUT		= (WSABASEERR+60),
	WSAECONNREFUSED		= (WSABASEERR+61),
	WSAEHOSTUNREACH		= (WSABASEERR+65),
	WSANOTINITIALISED	= (WSABASEERR+93),

	IOCPARM_MASK		= 16:7F,
	IOC_VOID			= 16:20000000,
	IOC_OUT				= 16:40000000,
	IOC_IN				= 16:80000000,
	IOC_INOUT			= (IOC_IN||IOC_OUT),

);

define :inline _IO(x,y);
	(IOC_VOID||((x)<<8)||(y))
enddefine;

define :inline _IOR(x,y,t);
	(IOC_OUT||((SIZEOFTYPE(:t)&&IOCPARM_MASK)<<16)||((x)<<8)||(y))
enddefine;

define :inline _IOW(x,y,t);
	(IOC_IN||((SIZEOFTYPE(:t)&&IOCPARM_MASK)<<16)||((x)<<8)||(y))
enddefine;

lconstant macro (
	FIONREAD			= _IOR(`f`, 127, ulong),
	FIONBIO				= _IOW(`f`, 126, ulong),
);

	;;; common error messages (not available anywhere else)
define lconstant Error_message =
	newproperty([
		[^WSAECONNABORTED	'Software caused connection abort']
		[^WSAECONNREFUSED	'Connection refused']
		[^WSAECONNRESET		'Connection reset by peer']
		[^WSAEHOSTUNREACH	'No route to host']
		[^WSAEISCONN		'Socket is already connected']
		[^WSAEMFILE			'Too many open files']
		[^WSAENETDOWN		'Network is down']
		[^WSAENETUNREACH	'Network is unreachable']
		[^WSAENOBUFS		'No buffer space available']
		[^WSAETIMEDOUT		'Connection timed out']
	], 16, false, "perm");
enddefine;

l_typespec

	SOCKET :uint,

	WSADATA {
		wVersion		:ushort,
		wHighVersion	:ushort,
		szDescription	:byte[WSADESCRIPTION_LEN],
		szSystemStatus	:byte[WSASYS_STATUS_LEN],
		iMaxSockets		:ushort,
		iMaxUdpDg		:ushort,
		lpVendorInfo	:exptr,
	},

	WSANETWORKEVENTS {
		lNetworkEvents	:long,
		iErrorCodes		:int[FD_MAX_EVENTS],
	},

	sockaddr_un {
		sun_family	:short,
		sun_path	:byte[].exacc_ntstring
	},

	sockaddr_in {
		sin_family	:short,
		sin_port	:ushort,
		sin_addr_b	:byte[0],	;;; dummy for getting byte addr of sin_addr
		sin_addr	:uint,
		sin_zero	:byte[8]
	},

	hostent {
		h_name		!exptr,
		h_aliases	!exptr,
		h_addrtype	!short,
		h_length	!short,
		h_addr		!exptr.:exptr
	},

	servent {
		s_name		!exptr,
		s_aliases	!exptr,
		s_port		!short,
		s_proto		!exptr.exacc_ntstring.consword
	},

	protoent {
		p_name		!exptr,
		p_aliases	!exptr,
		p_proto		!short,
	},

;

exload windows_sockets [ws2_32]
lconstant

	W_accept(s,name,namelenp) :int					<- accept,
	W_bind(s,name,namelen) :int						<- bind,
	W_closesocket(fd) :int							<- closesocket,
	W_gethostbyname(name) :exptr.:hostent			<- gethostbyname,
	W_getpeername(s,name,namelenp) :int				<- getpeername,
	W_getprotobyname(name) :exptr.:protoent			<- getprotobyname,
	W_getservbyname(name,proto) :exptr.:servent		<- getservbyname,
	W_getsockname(s,name,namelenp) :int				<- getsockname,
	W_getsockopt(s,level,option,valp,lenp) :int		<- getsockopt,
	W_htonl(val) :uint								<- ntohl,
	W_htons(val) :ushort							<- htons,
	W_inet_addr(cp) :uint							<- inet_addr,
	W_inet_ntoa(in) :exptr.exacc_ntstring			<- inet_ntoa,
	W_listen(s,qlen) :int							<- listen,
	W_ntohl(val) :uint								<- ntohl,
	W_ntohs(val) :ushort							<- ntohs,
	W_recv(s,buf,len,flags) :int					<- recv,
	W_recvfrom(s,buf,len,flags,from,fromlenp) :int	<- recvfrom,
	W_send(s,buf,len,flags) :int					<- send,
	W_sendto(s,buf,len,flags,to,tolen) :int			<- sendto,
	W_setsockopt(s,level,option,valp,len) :int		<- setsockopt,
	W_shutdown(s,how) :int							<- shutdown,
	W_socket(af,type,proto) :SOCKET					<- socket,

	WSAAccept(5) :SOCKET,
	WSACleanup(0) :int,
	WSAConnect(7) :int,
	WSACreateEvent(0) :exptr,
	WSAEnumNetworkEvents(3) :int,
	WSAEventSelect(3) :int,
	WSAGetLastError(0) :int,
	WSAIoctl(9) :int,
	WSAStartup(2) :int,

endexload;

lconstant macro (
	NAMEBUF_SIZE = 256,
);



;;; -- INITIALISATION -----------------------------------------------------

lvars
	init_done = false,
;

define lconstant Cleanup();
	returnunless(init_done);
	false -> init_done;
	exacc WSACleanup() -> ;
enddefine;

define lconstant Initialise();
	returnif(init_done);
	lvars WSADATA = EXPTRINITSTR(:WSADATA);
	lvars res = exacc WSAStartup(2, WSADATA);
	unless res == 0 and exacc WSADATA.wVersion == 2 then
		mishap(0, 'FAILED TO INITIALISE WINDOWS SOCKETS 2.0');
	endunless;
	true -> init_done;
	;;; arrange for cleanup on exit
	popexit <> Cleanup -> popexit;
	sys_grbg_fixed(WSADATA);
enddefine;

define lconstant Socket_error(nargs, msg, code);
	lvars string;
	consstring(#|
		explode(msg), `\s`, `(`;
		if Error_message(code) ->> string then
			explode(string)
		else
			explode('error:\s'), dest_characters(code)
		endif;
		`)`;
	|#) -> msg;
	mishap(nargs, msg);
enddefine;

define lconstant Check_status(err, retry) -> retry;
	if err == WSANOTINITIALISED and not(init_done) then
		Initialise();
		true -> retry;
	elseif err == WSAEINPROGRESS then
		;;; allow time to complete
		syssleep(1);
		true -> retry;
	elseif (err == WSAENOBUFS or err == WSAEMFILE) and retry == true then
		;;; resource problem: try one GC
		sysgarbage();
		"tried_gc" -> retry;
	else
		false -> retry;
	endif;
enddefine;


;;; -- SOCKET DEVICE DATA AND METHODS -------------------------------------

/*
	Sockets have to be user devices for now, because:
	1) System-level I/O can't be made to work with asynchronous sockets
	2) ReadFile/WriteFile aren't guaranteed to work with sockets anyway

	Disadvantages:
	1) Sockets can't be used as system devices, e.g. as stdin/stdout
	2) Don't get benefit of I/O translations -- text mode, encoding, etc.
	3) Slow

	There should be an alternative, low-level mapping direct to the
	Windows Sockets routines.
*/

defclass lconstant SocketBuffer [external_ptr] {
	sb_data,
	sb_ptr :exptr,
	sb_size,
	sb_bsub,
	sb_nbytes,
};
;;;
procedure(sb);
	printf('<SocketBuffer %p %p %p>', [% sb.sb_size, sb.sb_bsub, sb.sb_nbytes %]);
endprocedure -> class_print(SocketBuffer_key);

defclass lconstant SocketData [external_deref] {
	sd_svec,
	sd_socket :exval,
	sd_ibuff,
	sd_obuff,
	sd_closed,
	sd_line_mode,
	sd_text_mode,
	sd_event,
	sd_event_mask,
	sd_event_record,
	sd_event_asts,
};

lconstant macro
	DISABLE_ASTS = [
		lvars asts_enabled;
		dlocal 0 ^("%")
			(pop_asts_enabled, false) -> (asts_enabled, pop_asts_enabled),
			asts_enabled -> pop_asts_enabled
		^("%");
	],
;

lvars event_filter = 0;

define lconstant newSocketBuffer();
	lconstant INIT_BUFFER_SIZE = 0;
	consSocketBuffer(
		init_fixed(INIT_BUFFER_SIZE, string_key),
		fill_external_ptr(dup(), consexternal_ptr()),
		INIT_BUFFER_SIZE,
		1,
		0,
	);
enddefine;

define lconstant newSocketData(sock, svec, org);
	consSocketData(
		svec,	;;; svec = {^af ^type ^proto}
		sock,
		newSocketBuffer(),
		newSocketBuffer(),
		false,
		org == "line" or org == "record",
		org == false or org == "line",
		false,
		0,
		false,
		false,
	);
enddefine;

define lconstant SetBufferSize(sb, nbytes);
	lvars data = init_fixed(nbytes, string_key);
	min(nbytes, sb.sb_nbytes) -> sb.sb_nbytes;
	move_bytes(sb.sb_bsub, sb.sb_data, 1, data, sb.sb_nbytes);
	data -> sb.sb_data;
	fill_external_ptr(data, sb.sb_ptr) -> sb.sb_ptr;
	nbytes -> sb.sb_size;
	1 -> sb.sb_bsub;
enddefine;

define lconstant ExpandBuffer(sb, nbytes);
	lconstant MIN_BUFFER_SIZE = 512;
	lconstant MAX_BUFFER_SIZE = 16384;
	if nbytes > MAX_BUFFER_SIZE then MAX_BUFFER_SIZE -> nbytes endif;
	if sb.sb_size < nbytes then
		SetBufferSize(sb, max(1 << integer_length(nbytes-1), MIN_BUFFER_SIZE));
	endif;
enddefine;

define lconstant TestInput(sd);
	;;; test for input not yet read
	lconstant obuff = EXPTRINITSTR(:ulong), nret = EXPTRINITSTR(:DWORD);
	if exacc WSAIoctl(sd, FIONREAD, 0, 0, obuff, SIZEOFTYPE(:ulong), nret, 0,
						0) == SOCKET_ERROR
	then
		false
	else
		exacc :ulong obuff
	endif;
enddefine;

define lconstant InitEvents(sd);
	unless sd.sd_event then
		lvars event = exacc WSACreateEvent();
		if is_null_external_ptr(event) then
			Socket_error(0, 'FAILED TO CREATE EVENT', exacc WSAGetLastError());
		endif;
		sys_cons_handle(event, "EVENT", sd, false) ->> event -> sd.sd_event;
		EXPTRINITSTR(:WSANETWORKEVENTS) -> sd.sd_event_record;
		{^false ^false ^false} -> sd.sd_event_asts;
	endunless;
enddefine;

define lconstant EventSelect(sd, event_mask);
	InitEvents(sd);
	lvars event = sd.sd_event;
	if exacc WSAEventSelect(sd, event, event_mask) == SOCKET_ERROR then
		lvars err = exacc WSAGetLastError();
		;;; likely to return spurious error when cancelling
		unless event_mask == 0 and err == WSAEINVAL then
			Socket_error(0, 'EVENT SELECT FAILED ON SOCKET', err);
		endunless;
	endif;
enddefine;

define lconstant ProcessEvents(sd);
	InitEvents(sd);
	lvars events = sd.sd_event_record;
	l_typespec events :WSANETWORKEVENTS;
	lvars res = exacc WSAEnumNetworkEvents(sd, sd.sd_event, events);
	if res == SOCKET_ERROR then
		Socket_error(0, 'ERROR ENUMERATING NETWORK EVENTS',
			exacc WSAGetLastError());
	endif;
	lvars event_mask = sd.sd_event_mask &&~~ event_filter;
	lvars event_bits = exacc events.lNetworkEvents && event_mask;
	unless event_bits == 0 then
		lvars event_asts = sd.sd_event_asts;
		if event_bits &&/=_0 FD_CLOSE and not(sd.sd_closed) then
			;;; set closed flag if no more bytes to read
			if TestInput(sd) == 0 then
				"recv" -> sd.sd_closed;
			endif;
			sys_raise_ast(event_asts(1));
		elseif event_bits &&/=_0 FD_READ then
			sys_raise_ast(event_asts(1));
		endif;
		if event_bits &&/=_0 FD_WRITE then
			sys_raise_ast(event_asts(2));
		endif;
		if event_bits &&/=_0 FD_OOB then
			sys_raise_ast(event_asts(3));
		endif;
	endunless;
enddefine;

define lconstant WaitForEvent(sd, event_mask, asts_enabled);
	dlocal pop_asts_enabled = asts_enabled;
	EventSelect(sd, sd.sd_event_mask || event_mask);
	sys_wait_for_handle(sd.sd_event) -> ;
	ProcessEvents(sd);
enddefine;

define lconstant AsyncTrap(event, cancelled);
[AsyncTrap ^event ^cancelled]=>
	;;; event_filter /== 0 ==> inside read/write, so ProcessEvents will be
	;;; called anyway
	if sys_handle_data(event) and event_filter == 0 and not(cancelled) then
		ProcessEvents(sys_handle_data(event));
		AsyncTrap -> sys_async_handle(event);
	endif;
enddefine;

define lconstant AsyncEnable(sd);
	EventSelect(sd, sd.sd_event_mask);
	if sd.sd_event_mask == 0 then
		false -> sys_async_handle(sd.sd_event);
	else
		AsyncTrap -> sys_async_handle(sd.sd_event);
	endif;
enddefine;

define lconstant SocketRead(sdev, bsub, buffer, nbytes);
	DISABLE_ASTS;
	dlocal event_filter = event_filter || FD_READ_CLOSE;

	define FindNL(bsub, n, string);
		lvars lim = bsub fi_+ n fi_- 1;
		fast_for n from bsub to lim do
			returnif(fast_subscrs(n, string) == `\n`)(n fi_- bsub fi_+ 1);
		endfor;
		false;
	enddefine;

	lvars sd = device_user_data(sdev);

	lvars sb = sd.sd_ibuff;
	ExpandBuffer(sb, nbytes);

	lvars navail = sb.sb_nbytes;
	repeat

		;;; adjust the number of bytes to read based on the device mode
		if sd.sd_line_mode then
			lvars nread = FindNL(sb.sb_bsub, navail, sb.sb_data);
			if nread and nread < nbytes then
				nread -> nbytes;
			endif;
		elseif sd.sd_text_mode then
			;;; text_mode and not(line_mode) => org was <false>
			if navail > 0 and navail < nbytes then
				navail -> nbytes;
			endif;
		endif;

		;;; test whether we need to read anything
		if navail > nbytes then
			quitloop;
		elseif sd.sd_closed then
			navail -> nbytes;
			quitloop;
		elseif navail == nbytes
		and not(sd.sd_text_mode and (sb.sb_data)(sb.sb_bsub+navail-1) == `\r`)
		then
			quitloop;
		endif;

		;;; try to restock the buffer from the socket
		repeat
			if navail > 0 and sb.sb_bsub > 1 then
				move_bytes(sb.sb_bsub, sb.sb_data, 1, sb.sb_data, navail);
			endif;
			1 -> sb.sb_bsub;
			lvars nread = exacc W_recv(sd, exacc[@] :byte[] sb[navail+1],
										sb.sb_size - navail, 0);
			lvars err = nread == SOCKET_ERROR and exacc WSAGetLastError();
			quitunless(err == WSAEWOULDBLOCK);
			;;; set up interruptable wait for input
			WaitForEvent(sd, FD_READ_CLOSE, asts_enabled);
			sb.sb_nbytes -> navail;
		endrepeat;

		if err then
			if err == WSAECONNABORTED or err == WSAECONNRESET then
				;;; socket no longer valid
				exacc W_closesocket(sd) -> ;
				true -> sd.sd_closed;
			endif;
			Socket_error(sdev, 1, 'ERROR READING SOCKET', err);
		elseif nread == 0 then
			;;; EOF -- graceful shutdown by peer
			"recv" -> sd.sd_closed;
			if navail < nbytes then navail -> nbytes endif;
			quitloop;
		elseif sd.sd_text_mode then
			;;;  map CR/NL --> NL
			lvars i = fi_max(navail, 1), j, data = sb.sb_data;
			fast_for j from i to navail fi_+ nread fi_- 1 do
				unless fast_subscrs(j, data) == `\r`
				and fast_subscrs(j fi_+ 1, data) == `\n`
				then
					fast_subscrs(j, data) -> fast_subscrs(i, data);
					i fi_+ 1 -> i;
				endunless;
			endfor;
			fast_subscrs(j, data) -> fast_subscrs(i, data);
			i ->> navail -> sb.sb_nbytes;
		else
			navail + nread ->> navail -> sb.sb_nbytes;
		endif;
	endrepeat;

	move_bytes(sb.sb_bsub, sb.sb_data, bsub, buffer, nbytes);
	navail - nbytes -> sb.sb_nbytes;
	sb.sb_bsub + nbytes -> sb.sb_bsub;
	nbytes -> nread;

	;;; leave result
	nread;
	;;; re-enable async I/O outside dlocal context
	chain(sd, AsyncEnable);
enddefine;

define lconstant SocketTestInput(sdev) -> navail;
	DISABLE_ASTS;
	lvars sd = device_user_data(sdev);
	sd.sd_ibuff.sb_nbytes -> navail;
	;;; test for data not yet read
	lvars nwaiting = TestInput(sd);
	if nwaiting then
		navail + nwaiting -> navail;
	endif;
	if navail == 0 and not(sd.sd_closed) then false -> navail endif;
enddefine;

define lconstant SocketClearInput(sdev);
	DISABLE_ASTS;
	lvars sb = device_user_data(sdev).sd_ibuff;
	0 -> sb.sb_nbytes;
	1 -> sb.sb_bsub;
enddefine;

define lconstant Write(sd, sb, nbytes) -> res;
	lvars res = 0;
	until nbytes == 0 do
		exacc W_send(sd, exacc[@] :byte[] sb[sb.sb_bsub], nbytes, 0) -> res;
		quitif(res == SOCKET_ERROR);
		sb.sb_bsub + res -> sb.sb_bsub;
		sb.sb_nbytes - res -> sb.sb_nbytes;
		nbytes - res -> nbytes;
	enduntil;
enddefine;

define lconstant WriteError(sdev, asts_enabled, err);
	lvars sd = device_user_data(sdev);
	if err == WSAEWOULDBLOCK then
		;;; write blocked -- set up interruptable wait for output
		WaitForEvent(sd, FD_WRITE, asts_enabled);
	else
		dlocal pop_asts_enabled = asts_enabled;
		if err == WSAECONNABORTED or err == WSAECONNRESET then
			;;; socket no longer valid
			exacc W_closesocket(sd) -> ;
			true -> sd.sd_closed;
			;;; signal fatal error
			pop_file_write_error(sdev);
		endif;
		Socket_error(0, 'ERROR WRITING SOCKET', err);
	endif;
enddefine;

define lconstant SocketWrite(sdev, bsub, buffer, nbytes);
	DISABLE_ASTS;
	dlocal event_filter = event_filter || FD_WRITE;

	define FindNL(bsub, n, string);
		lvars lim = bsub fi_+ n fi_- 1;
		fast_for n from lim by -1 to bsub do
			returnif(fast_subscrs(n, string) == `\n`)(n fi_- bsub fi_+ 1);
		endfor;
		false;
	enddefine;

	lvars sd = device_user_data(sdev);
	lvars sb = sd.sd_obuff;
	ExpandBuffer(sb, nbytes);

	until nbytes == 0 do
		if sb.sb_bsub > 1 and sb.sb_nbytes > 0 then
			move_bytes(sb.sb_bsub, sb.sb_data, 1, sb.sb_data, sb.sb_nbytes);
		endif;
		1 -> sb.sb_bsub;
		if sd.sd_text_mode then
			;;; expand NL --> CR/NL
			lvars i = sb.sb_nbytes + 1, data = sb.sb_data;
			while i fi_<= sb.sb_size do
				lvars c = fast_subscrs(bsub, buffer);
				if c == `\n` then
					quitif(i == sb.sb_size);
					`\r` -> fast_subscrs(i, data);
					i fi_+ 1 -> i;
				endif;
				c -> fast_subscrs(i, data);
				i fi_+ 1 -> i;
				nbytes fi_- 1 -> nbytes;
				quitif(nbytes == 0);
				bsub fi_+ 1 -> bsub;
			endwhile;
			i - 1 -> sb.sb_nbytes;
		else
			lvars nwrite = sb.sb_size - sb.sb_nbytes;
			if nwrite > nbytes then nbytes -> nwrite endif;
			move_bytes(bsub, buffer, sb.sb_nbytes+1, sb.sb_data, nwrite);
			nbytes - nwrite -> nbytes;
			bsub + nwrite -> bsub;
			sb.sb_nbytes + nwrite -> sb.sb_nbytes;
		endif;
		repeat
			lvars nwrite = false;
			if sd.sd_line_mode then
				FindNL(sb.sb_bsub, sb.sb_nbytes, sb.sb_data) -> nwrite;
			endif;
			if not(nwrite) and sb.sb_nbytes == sb.sb_size then
				sb.sb_nbytes -> nwrite;
			endif;
			quitunless(nwrite);
			if Write(sd, sb, nwrite) == SOCKET_ERROR then
				WriteError(sdev, asts_enabled, exacc WSAGetLastError());
			endif;
		endrepeat;
	enduntil;

	;;; re-enable async I/O outside dlocal context
	chain(sd, AsyncEnable);
enddefine;

define lconstant SocketFlush(sdev);
	DISABLE_ASTS;
	dlocal event_filter = event_filter || FD_WRITE;
	lvars sd = device_user_data(sdev);
	lvars sb = sd.sd_obuff;
	lvars nwrite = sb.sb_nbytes;
	until nwrite == 0 do
		if Write(sd, sb, nwrite) == SOCKET_ERROR then
			WriteError(sdev, asts_enabled, exacc WSAGetLastError());
		endif;
		sb.sb_nbytes -> nwrite;
	enduntil;
	;;; re-enable async I/O outside dlocal context
	chain(sd, AsyncEnable);
enddefine;

define lconstant SocketClose(sdev);
	lvars sd = device_user_data(sdev);
	;;; turn off async processing
	0 -> sd.sd_event_mask;
	unless sd.sd_closed == true then
		;;; send buffered data (could block)
		SocketFlush(sdev);
		;;; close the socket
		true -> sd.sd_closed;
		if exacc W_closesocket(sd) == SOCKET_ERROR then
			Socket_error(sdev, 'ERROR CLOSING SOCKET', exacc WSAGetLastError());
		endif;
	endunless;
	;;; free resources
	if sd.sd_event then
		sys_close_handle(sd.sd_event);
		false ->> sys_handle_data(sd.sd_event) -> sd.sd_event;
		false ->> sd.sd_event_record -> sd.sd_event_asts;
	endif;
	;;; reclaim buffer space
	SetBufferSize(sd.sd_ibuff, 0);
	SetBufferSize(sd.sd_obuff, 0);
enddefine;


;;; --- NAME <-> SOCKADDR MAPPING -----------------------------------------

;;; AF_UNIX

define lconstant name_to_sa_UNIX(name) -> (sockaddr_un, namelen);
	lvars name, sockaddr_un, namelen;
	sysfileok(name) -> name;
	SIZEOFTYPE(:short) + datalength(name) -> namelen;
	initexptr_mem(namelen+1) -> sockaddr_un;
	AF_UNIX -> exacc sockaddr_un.sun_family;
	name	-> exacc sockaddr_un.sun_path;
enddefine;
;;;
define updaterof name_to_sa_UNIX(sockaddr_un, namelen) /* -> name */;
	lvars sockaddr_un, namelen;
	exacc sockaddr_un.sun_path;
enddefine;

;;; AF_INET

	;;; NB: these host <-> network byte-order converters are inet-specific
	;;;
define :inline lconstant HTONS(val);
	(exacc W_htons(val) fi_&& 16:FFFF)
enddefine;
;;;
define :inline lconstant HTONL(val);
	exacc W_htonl(val)
enddefine;
;;;
define :inline lconstant NTOHS(val);
	(exacc W_ntohs(val) fi_&& 16:FFFF)
enddefine;
;;;
define :inline lconstant NTOHL(val);
	exacc W_ntohl(val)
enddefine;

define lconstant inet_name_to_sa(name) -> (sockaddr_in, namelen, proto);
	lvars	name, sockaddr_in, namelen, addr, hostent, servent,
			netent, lna, port = 0, org_name = name, proto = false;

	define lconstant inv_name(ms);
		lvars ms;
		mishap(org_name, 1, 'INVALID INTERNET SOCKET NAME (' <> ms <> ')')
	enddefine;

	SIZEOFTYPE(:sockaddr_in) -> namelen;
	initexptr_mem(namelen) -> sockaddr_in;

	0 -> port;
	if islist(name) then
		if listlength(name) == 2 then
			dl(name) -> (name, port);
			unless isinteger(port) then
				if isvector(port) then
					if datalength(port) == 2 then
						explode(port) -> (port, proto)
					else
						inv_name('invalid service vector')
					endif
				endif;
				if isword(proto) then fast_word_string(proto) -> proto endif;
				if isstring(port) and (not(proto) or isstring(proto)) then
					;;; service spec
					exacc W_getservbyname(port, proto) -> servent;
					unless is_valid_external_ptr(servent) then
						inv_name('unknown server')
					endunless;
					exacc servent.s_proto -> proto;
					NTOHS(exacc servent.s_port) -> port
				elseif isstring(proto) then
					consword(proto) -> proto
				endif
			endunless;
			unless isinteger(port) and port >= 0 then
				inv_name('invalid port')
			endunless
		else
			inv_name('invalid host/port list')
		endif
	endif;

	if name == "*" then INADDR_ANY -> name endif;

	if isvector(name) then
		;;; net spec
		mishap(name, 1, 'NETWORK SPECIFICATION NOT SUPPORTED');
	elseif isstring(name) then
		;;; try numeric notation first
		if datalength(name) /== 0 and isnumbercode(name(1))
		and (exacc W_inet_addr(name) ->> addr) /= INADDR_NONE then
			addr -> exacc sockaddr_in.sin_addr
		else
			;;; try as hostname
			exacc W_gethostbyname(name) -> hostent;
			unless is_valid_external_ptr(hostent)
			and exacc hostent.h_addrtype == AF_INET then
				inv_name('unknown or invalid hostname')
			endunless;
			move_bytes( 1, exacc[nc] hostent.h_addr,
						1, exacc[nc] sockaddr_in.sin_addr_b,
						SIZEOFTYPE(:uint))
		endif
	elseif isintegral(name) and name >= 0 then
		HTONL(name) -> exacc sockaddr_in.sin_addr
	else
		inv_name('address not a string or (big)integer >= 0')
	endif;

	HTONS(port) -> exacc sockaddr_in.sin_port;
	set_bytes(0, 1, exacc[nc] sockaddr_in.sin_zero, 8);
	AF_INET -> exacc sockaddr_in.sin_family
enddefine;

define lconstant name_to_sa_INET = inet_name_to_sa <> erase enddefine;
;;;
define updaterof name_to_sa_INET(sockaddr_in, namelen) /* -> name */;
	lvars sockaddr_in, namelen, inaddr, port;
	NTOHS(exacc sockaddr_in.sin_port) -> port;
	returnif(port == 0) (false);
	NTOHL(exacc sockaddr_in.sin_addr) -> inaddr;
	if inaddr == INADDR_ANY then
		"*"
	else
		;;; this relies on knowledge of Intel calling convention for structs
		exacc W_inet_ntoa(exacc sockaddr_in.sin_addr)
	endif -> inaddr;
	if port == 0 then
		inaddr
	else
		[^inaddr ^port]
	endif
enddefine;

define sys_socket_name_trans =
	newassoc([
		[^AF_UNIX	^name_to_sa_UNIX]
		[^AF_INET	^name_to_sa_INET]
	]);
enddefine;

lconstant
	noname_ms = 'NO NAME TRANSLATION AVAILABLE FOR SOCKET ADDRESS FAMILY';

define lconstant name_to_sockaddr(name, af) /* -> (namebuf, namelen) */;
	lvars name, af, trans_p;
	if sys_socket_name_trans(af) ->> trans_p then
		chain(name, trans_p)
	else
		mishap(af, 1, noname_ms)
	endif
enddefine;

define lconstant sockaddr_to_name(namebuf, namelen) /* -> name */;
	lvars namebuf, namelen, trans_p, af = exacc :short namebuf;
	if sys_socket_name_trans(af) ->> trans_p then
		chain(namebuf, namelen, updater(trans_p))
	else
		mishap(af, 1, noname_ms)
	endif
enddefine;


;;; --- SOCKET CREATION -------------------------------------------------

define is_socket(dev) /* -> svec */;
	lvars sd;
	if isdevice(dev) and isSocketData(device_user_data(dev) ->> sd) then
		sd.sd_svec;
	else
		false;
	endif;
enddefine;

define lconstant check_sock(sock) /* -> af */;
	lvars svec;
	if is_socket(sock) ->> svec then
		svec(1)			;;; af
	else
		mishap(sock, 1, 'SOCKET NEEDED')
	endif
enddefine;

define lconstant open_sock(af, type, protocol) -> res;
	lvars retry = true, err;
	repeat
		lvars res = exacc W_socket(af, type, protocol);
		returnunless(res = INVALID_SOCKET);
		exacc WSAGetLastError() -> err;
		quitunless(Check_status(err, retry) ->> retry);
	endrepeat;
	Socket_error(af, type, protocol, 3, 'CAN\'T CREATE SOCKET', err);
enddefine;

define lconstant proto_number(proto) -> protonum;
	define cache =
		newassoc([]);
	enddefine;
	returnif(cache(proto) ->> protonum);
	lvars protoent = exacc W_getprotobyname(fast_word_string(proto));
	if is_valid_external_ptr(protoent) then
		exacc protoent.p_proto -> protonum;
	elseunless (strnumber(proto) ->> protonum) and isinteger(protonum)
	and protonum >= 0 then
		mishap(proto, 1, 'UNKNOWN OR INVALID PROTOCOL NAME');
	endif;
	protonum -> cache(proto);
enddefine;

define lconstant create_sock(af, type) /* -> (res,svec) */;
	lvars protocol = 0;
	if isword(type) then
		;;; optional protocol specified
		((), af, type) -> (af, type, protocol);
		proto_number(protocol) -> protocol;
	endif;
	checkinteger(af, 0, false);
	if af == `u` then
		AF_UNIX -> af
	elseif af == `i` then
		AF_INET -> af
	endif;
	checkinteger(type, 0, false);
	if type == `S` then
		SOCK_STREAM -> type
	elseif type == `D` then
		SOCK_DGRAM -> type
	endif;
	open_sock(af, type, protocol),
	consvector(af, type, protocol, 3);
enddefine;

define lconstant make_sock_dev(fd, svec, org) -> sdev;
	;;; set to non-blocking mode
	lconstant ibuff = EXPTRINITSTR(:ulong), nret = EXPTRINITSTR(:DWORD);
	1 -> exacc :ulong ibuff;
	if exacc WSAIoctl(fd, FIONBIO, ibuff, SIZEOFTYPE(:ulong), 0, 0, nret, 0, 0)
						== SOCKET_ERROR
	then
		Socket_error(0, 'ERROR CREATING SOCKET', exacc WSAGetLastError());
	endif;
	;;; create user device
	lvars sd = newSocketData(fd, svec, org);
	consdevice('socket', false, sd, 0, {
		{^SocketRead ^SocketTestInput ^SocketClearInput}
		{^SocketWrite ^SocketFlush}
		^false
		^SocketClose
	}) -> sdev;
	unless sd.sd_text_mode then
		false -> device_encoding(sdev);
	endunless;
	;;; close on destroy
	sysclose -> sys_destroy_action(sdev);
enddefine;

define sys_socket(/*af, type,*/ org) with_nargs 4;
	make_sock_dev(create_sock(()), org);
enddefine;


;;; --- SOCKET NAMES ---------------------------------------------------

define lconstant get_sock_name(sock, routine) -> name;
	check_sock(sock) -> ;
	lvars namebuf = EXPTRINITSTR(:byte[NAMEBUF_SIZE]), namelen = initintvec(1);
	lvars res, err, retry = true;
	repeat
		NAMEBUF_SIZE -> namelen(1);
		exacc (3):int routine(device_user_data(sock), namebuf, namelen) -> res;
		quitunless(res == SOCKET_ERROR);
		exacc WSAGetLastError() -> err;
		quitunless(Check_status(err, retry) ->> retry);
	endrepeat;
	if res /== SOCKET_ERROR then
		sockaddr_to_name(namebuf, namelen(1))
	elseif err == WSAENOTCONN or err == WSAEINVAL then
		false
	else
		Socket_error(sock, 1, 'CAN\'T GET SOCKET NAME', err)
	endif -> name;
	sys_grbg_fixed(namebuf);
enddefine;

define sys_socket_name(sock) /* -> name */;
	get_sock_name(sock, W_getsockname)
enddefine;
;;;
define updaterof sys_socket_name(name, sock);
	lvars qlen = false, fd, namebuf, namelen;

	if isinteger(sock) then
		;;; queue length specified -- start listening after bind
		((), name, sock) -> (name, sock, qlen);
		checkinteger(qlen, 1, false)
	endif;

	lvars (namebuf, namelen) = name_to_sockaddr(name, check_sock(sock));
	lvars fd = device_user_data(sock);

	lvars res, err, retry = true;
	repeat
		exacc W_bind(fd, namebuf, namelen) -> res;
		quitunless(res == SOCKET_ERROR);
		exacc WSAGetLastError() -> err;
		quitunless(Check_status(err, retry) ->> retry);
	endrepeat;

	sys_grbg_fixed(namebuf);
	if res == SOCKET_ERROR then
		Socket_error(name, sock, 2, 'CAN\'T ASSIGN SOCKET NAME', err)
	endif;

	if qlen then
		lvars res, err, retry = true;
		repeat
			exacc W_listen(fd, qlen) -> res;
			quitunless(res == SOCKET_ERROR);
			exacc WSAGetLastError() -> err;
			quitunless(Check_status(err, retry) ->> retry);
		endrepeat;
		if res == SOCKET_ERROR then
			Socket_error(name, sock, qlen, 3, 'LISTEN FAILED ON SOCKET', err);
		endif;
	endif;
enddefine;

define lconstant do_connect(peername, sock, namebuf, namelen, changes_p,
														conn_retries);
	lvars fd = device_user_data(sock);
	lvars res, err, ntries;
	for ntries to conn_retries do
		changes_p(sock);
		lvars retry = true;
		repeat
			exacc WSAConnect(fd, namebuf, namelen, 0, 0, 0, 0) -> res;
			quitunless(res == SOCKET_ERROR);
			exacc WSAGetLastError() -> err;
			quitunless(Check_status(err, retry) ->> retry);
		endrepeat;
		if ntries > 1 then
			;;; this is curious: if the first attempt fails, and we go
			;;; too quickly into the second EventSelect, then subsequent
			;;; connection attempts will fail with WSAEALREADY
			syssleep(50);
		endif;
		while res == SOCKET_ERROR and err == WSAEWOULDBLOCK do
			WaitForEvent(fd, FD_CONNECT, pop_asts_enabled);
			lvars events = fd.sd_event_record;
			l_typespec events :WSANETWORKEVENTS;
			if exacc events.lNetworkEvents &&/=_0 FD_CONNECT then
				exacc (exacc events.iErrorCodes)[FD_CONNECT_BIT+1] -> err;
				if err == 0 then 0 -> res endif;
			endif;
			AsyncEnable(fd);
		endwhile;
		quitunless(res == SOCKET_ERROR and err == WSAECONNREFUSED);
		;;; wait half a second and try again
		syssleep(50);
	endfor;
	returnunless(namebuf);
	sys_grbg_fixed(namebuf);
	if res == SOCKET_ERROR then
		Socket_error(peername, sock, 2, 'CAN\'T ASSIGN SOCKET PEERNAME', err)
	endif;
enddefine;

define sys_socket_peername(sock) /* -> peername */;
	get_sock_name(sock, W_getpeername);
enddefine;
;;;
define updaterof sys_socket_peername(peername, sock);
	lvars conn_retries = 5, changes_p = erase;

	if isinteger(sock) then
		;;; optional connect retry count
		((), peername, sock) -> (peername, sock, conn_retries);
		checkinteger(conn_retries, 1, false);
	endif;
	if isprocedure(sock) then
		;;; optional procedure to apply to sock before connect attempt
		((), peername, sock) -> (peername, sock, changes_p);
	endif;

	lvars af = check_sock(sock), namebuf, namelen;
	if peername or is_socket(sock)(2) /== SOCK_DGRAM then
		name_to_sockaddr(peername, af)
	else
		;;; false for SOCK_DGRAM
		(false, 0)
	endif -> (namebuf, namelen);

	do_connect(peername, sock, namebuf, namelen, changes_p, conn_retries);
enddefine;


;;; --- OTHER ROUTINES ---------------------------------------------------

define sys_socket_to_service(peername, org) -> sock;
	lvars type, (namebuf, namelen, proto) = inet_name_to_sa(peername);
	unless proto then
		mishap(peername, 1, 'INTERNET SERVICE NAME NEEDED')
	endunless;
	if proto = "tcp" then
		SOCK_STREAM
	elseif proto = "udp" then
		SOCK_DGRAM
	else
		mishap(peername, 1, 'SERVICE PROTOCOL NOT tcp OR udp')
	endif -> type;
	sys_socket(AF_INET, type, org) -> sock;
	do_connect(peername, sock, namebuf, namelen, erase, 5)
enddefine;

define sys_socket_accept(sock, org) -> conn_sock;
	check_sock(sock) -> ;
	lvars fd = device_user_data(sock);
	lvars res, err;
	repeat
		;;; disable event selection on the socket to stop it being
		;;; inherited by the accepted socket
		EventSelect(fd, 0);
		lvars retry = true;
		repeat
			exacc WSAAccept(fd, 0, 0, 0, 0) -> res;
			quitunless(res = INVALID_SOCKET);
			exacc WSAGetLastError() -> err;
			quitunless(Check_status(err, retry) ->> retry);
		endrepeat;
		quitunless(res = INVALID_SOCKET);
		while err == WSAEWOULDBLOCK do
			WaitForEvent(fd, FD_ACCEPT, pop_asts_enabled);
			lvars events = fd.sd_event_record;
			l_typespec events :WSANETWORKEVENTS;
			if exacc events.lNetworkEvents &&/=_0 FD_ACCEPT then
				exacc (exacc events.iErrorCodes)[FD_ACCEPT_BIT+1] -> err;
			endif;
		endwhile;
		quitunless(err == 0);
	endrepeat;
	;;; re-enable event selection as required
	AsyncEnable(fd);
	if res = INVALID_SOCKET then
		Socket_error(sock, 1, 'ACCEPT FAILED ON SOCKET', err);
	endif;
	make_sock_dev(res, is_socket(sock), org) -> conn_sock;
enddefine;

define sys_socket_shutdown(sock, how);
	check_sock(sock) -> ;
	checkinteger(how, 0, 2);
	if exacc W_shutdown(device_user_data(sock), how) == SOCKET_ERROR then
		Socket_error(sock, how, 2, 'SHUTDOWN FAILED ON SOCKET',
			exacc WSAGetLastError());
	endif
enddefine;


;;; -- ASYNCHRONOUS I/O ---------------------------------------------------

/*	This is for sockets only -- uses WSAEventSelect
*/

define sys_async_socket(sock, condition) -> ast_p;
	checkinteger(condition, 0, 2);
	check_sock(sock) -> ;
	lvars event_asts = device_user_data(sock).sd_event_asts;
	event_asts and event_asts(condition+1) -> ast_p;
enddefine;
;;;
define updaterof sys_async_socket(ast_p, sock, condition);
	checkinteger(condition, 0, 2);
	check_sock(sock) -> ;
	lvars sd = device_user_data(sock);
	if ast_p then
		;;; crude check on AST_P
		unless isprocedure(ast_p) or ispair(ast_p) and isprocedure(front(ast_p))
		then
			mishap(ast_p, 1, 'INVALID AST PROCEDURE ARGUMENT');
		endunless;
	else
		returnunless(sd.sd_event);
	endif;
	InitEvents(sd);
	lconstant EVENT_MASK = {^FD_READ_CLOSE ^FD_WRITE ^FD_OOB};
	if ast_p ->> (sd.sd_event_asts)(1) then
		sd.sd_event_mask || EVENT_MASK(condition+1) -> sd.sd_event_mask;
	else
		sd.sd_event_mask &&~~ EVENT_MASK(condition+1) -> sd.sd_event_mask;
	endif;
	AsyncEnable(sd);
enddefine;


;;; -----------------------------------------------------------------------

constant windows_sockets = true;

endexload_batch;
endsection;


/* --- Revision History ---------------------------------------------------
--- Robert Duncan, Dec  9 1998
		Disabled device encoding for sockets opened in binary mode (same
		behaviour as for system devices)
--- Robert Duncan, Jan 30 1998
		Moved some calls to WSAGetLastError so that the error number can't
		be changed by an intervening interrupt
 */
