/* --- Copyright University of Sussex 1997. All rights reserved. ----------
 > File:			C.win32/src/devio.p
 > Purpose:			Device I/O (Win32)
 > Author:			Robert John Duncan, Apr 14 1994 (see revisions)
 > Related Files:	C.{unix,vms}/src/devio.p
 */

#_INCLUDE 'declare.ph'
#_INCLUDE 'io.ph'
#_INCLUDE 'win32defs.ph'


vars procedure (pop_timeout, pop_file_write_error)
	pop_timeout_secs;

weak constant procedure (sys_input_waiting);

section $-Sys;

constant procedure (Add_file_tab_entry, App_open_devs, Ensure_writeable),
	no_device_encoding,
;

endsection;		/* $-Sys */

section $-Sys$-Io;

constant procedure (Cons_iobuffer, Init_device, Kill_device, No_write, Prompt);

weak constant procedure (Test_input);

endsection;		/* $-Sys$-Io */

;;; ------------------------------------------------------------------------


section $-Sys$-Io => Sys_cons_device;

	/*	Keep applying a file routine until it comes back uninterrupted
	*/
define lconstant Pf_apply(_pf, _routine);
	lvars _res;
	while _zero(_extern[INDIR] _routine(_pf) ->> _res)
	and _pf!PF_FLAGS _bitst _:POP_FILE_INTERRUPT
	do
		_pf!PF_FLAGS _biclear _:POP_FILE_INTERRUPT -> _pf!PF_FLAGS;
		_CHECKINTERRUPT;
	endwhile;
	_nonzero(_res);
enddefine;


;;; --- READING DEVICES ----------------------------------------------------

lconstant string16_ms = 'STRING16 NEEDED FOR ENCODED CHARACTER INPUT';

define lconstant Timeout();
	if isinteger(pop_timeout_secs) and pop_timeout_secs fi_> 0 then
		_int(pop_timeout_secs) _mult _1000
	else
		_-1
	endif;
enddefine;

define lconstant Read_error(dev);
	lvars dev;
	dev!D_CTRL_BLK!DCB_FILE!PF_ERROR -> _syserror;
	Syserr_mishap(dev, 1, 'ERROR READING DEVICE')
enddefine;

define lconstant Do_read(_pf, _bsub, userbuf, _nchars, _);
	_extern pop_file_read(_pf, userbuf@V_BYTES[_bsub], _nchars);
enddefine;

define lconstant Do_special_read(_pf, _bsub, userbuf, _nchars, dev);
	_extern pop_file_special_read(_pf, userbuf@V_BYTES[_bsub], _nchars,
		dev!D_CTRL_BLK!DCB_MODE, _-1);
enddefine;

define lconstant Do_interactive_read(_pf, _bsub, userbuf, _nchars, dev);
	_extern pop_file_special_read(_pf, userbuf@V_BYTES[_bsub], _nchars,
		dev!D_CTRL_BLK!DCB_MODE, Timeout());
enddefine;

define lconstant Do_console_read(_pf, _bsub, userbuf, _nchars, dev);
	lvars t_prompt = Tchars_in(Prompt(dev), wkstring1);
	_extern pop_file_console_read(_pf, userbuf@V_BYTES[_bsub], _nchars,
		 dev!D_CTRL_BLK!DCB_MODE, Timeout(), t_prompt@V_TCHARS);
enddefine;

define lconstant Read_file(dev, _bsub, userbuf, _nchars, procedure read_p);
	lvars _count = _int(_nchars), _nread, _pf = dev!D_CTRL_BLK!DCB_FILE;
	_int(_bsub) _sub _1 -> _bsub;
	if _nonzero(fast_idval(dev!D_ENCODING_ID)!XP_PTR ->> _pf!PF_ENCODING) then
		unless userbuf!KEY!K_FLAGS _bitst _:M_K_STRING16 then
			mishap(dev, 1, string16_ms);
		endunless;
		@@(b)[_bsub|s] -> _bsub;
	endif;
	until (read_p(_pf, _bsub, userbuf, _count, dev) ->> _nread) == _count do
		lvars _flags = _pf!PF_FLAGS;
		if _flags _bitst _:POP_FILE_ERROR then
			Read_error(dev);
		elseif _flags _bitst _:POP_FILE_INTERRUPT then
			_flags _biclear _:POP_FILE_INTERRUPT -> _pf!PF_FLAGS;
			_CHECKINTERRUPT;
		elseif _flags _bitst _:POP_FILE_TIMEOUT then
			_flags _biclear _:POP_FILE_TIMEOUT -> _pf!PF_FLAGS;
			pop_timeout();
		else
			;;; partial read
			_nchars fi_- _pint(_count _sub _nread) -> _nchars;
			quitloop;
		endif;
		;;; try reading more
		if _zero(_pf!PF_ENCODING) then
			_bsub _add _nread -> _bsub;
		else
			_bsub _add @@(b)[_nread|s] -> _bsub;
		endif;
		_count _sub _nread -> _count;
	enduntil;
	_nchars;
enddefine;

define File_read = Read_file(% Do_read %);
enddefine;
;;;
define Special_read = Read_file(% Do_special_read %);
enddefine;
;;;
define Interactive_read = Read_file(% Do_interactive_read %);
enddefine;
;;;
define Console_read() -> _nread;
	lvars _nread = Read_file((), Do_console_read);
	if _zero(_nread) then
		;;; probably EOF, but it could be an interrupted console read!
		;;; Until we can identify that case properly, better check for
		;;; interrupt before returning, just in case; sleep 1/100th of a
		;;; second first to give the ctrl handler thread -- if it exists
		;;; -- a chance to run
		_extern pop_sleep(_10) -> ;
		_CHECKINTERRUPT;
	endif;
enddefine;


;;; --- WRITING DEVICES -----------------------------------------------------

define lconstant Write_error(dev);

	;;; just return if ignoring write errors on this device's unit
	lvars unit_n = dev!D_UNIT_N;
	returnif(unit_n!UNT_FLAGS _bitst _:M_UNT_IGNORE_WRITE_ERR);

	lvars _pf = dev!D_CTRL_BLK!DCB_FILE;
	if (dev == dev_out or dev == dev_err)
	;;; if it's still recognisable as a console, assume it's OK
	and _extern pop_get_file_type(_pf!PF_HANDLE) /== _:POP_FILE_TYPE_CONSOLE
	then
		;;; Assume unrecoverable error on a standard output/error
		;;; device, and prevent any further mishaps when trying to write
		;;; it (or any other with the same unit)
		unit_n!UNT_FLAGS _biset _:M_UNT_IGNORE_WRITE_ERR -> unit_n!UNT_FLAGS;
	endif;

	if dev!D_FLAGS _bitst _M_D_INTERACTIVE then
		_CHECKINTERRUPT;
	elseif unit_n!UNT_TYPE == _:POP_FILE_TYPE_DISK then
		;;; for disk file, close without flushing
		unless unit_n!UNT_FLAGS _bitst _:M_UNT_IGNORE_WRITE_ERR then
			_extern pop_file_close(_pf) -> ;
			Kill_device(dev);
		endunless;
		;;; try user error procedure
		_pf!PF_ERROR -> _syserror;
		pop_file_write_error(dev);
	endif;

	_pf!PF_ERROR -> _syserror;
	Syserr_mishap(dev, 1, 'ERROR WRITING DEVICE');
enddefine;

define lconstant Do_write(_pf, _buff, _nchars, _mode);
	_extern pop_file_write(_pf, _buff, _nchars, _mode);
enddefine;

define lconstant Do_special_write(_pf, _buff, _nchars, _mode);
	_extern pop_file_special_write(_pf, _buff, _nchars, _mode, _-1);
enddefine;

define lconstant Do_console_write(_pf, _buff, _nchars, _mode);
	_extern pop_file_console_write(_pf, _buff, _nchars, _mode, _-1);
enddefine;

define lconstant Write_file(dev, _bsub, userbuf, _nchars, procedure write_p);
	lvars _count = _int(_nchars), _nwrtn, _ntries = _5;
	lvars _pf = dev!D_CTRL_BLK!DCB_FILE, _mode = dev!D_CTRL_BLK!DCB_MODE;
	_int(_bsub) _sub _1 -> _bsub;
	if _nonzero(fast_idval(dev!D_ENCODING_ID)!XP_PTR ->> _pf!PF_ENCODING) then
		if userbuf!KEY!K_FLAGS _bitst _:M_K_STRING16 then
			_mode _biset _:POP_FILE_MODE_16BIT -> _mode;
			@@(b)[_bsub|s] -> _bsub;
		else
			_mode _biclear _:POP_FILE_MODE_16BIT -> _mode;
		endif;
	endif;
	until (write_p(_pf, userbuf@V_BYTES[_bsub], _count, _mode) ->> _nwrtn)
				== _count
	do
		lvars _flags = _pf!PF_FLAGS;
		if _flags _bitst _:POP_FILE_ERROR then
			Write_error(dev);
		elseif _flags _bitst _:POP_FILE_INTERRUPT then
			_flags _biclear _:POP_FILE_INTERRUPT -> _pf!PF_FLAGS;
			_CHECKINTERRUPT;
		elseif _zero(_nwrtn) then
			;;; Hmmm. Something non-blocking? Don't want to be stuck in
			;;; this loop for ever
			if _zero(_ntries _sub _1 ->> _ntries) then
				Write_error(dev);
			endif;
		endif;
		;;; try writing more
		if _mode _bitst _:POP_FILE_MODE_16BIT then
			_bsub _add @@(b)[_nwrtn|s] -> _bsub;
		else
			_bsub _add _nwrtn -> _bsub;
		endif;
		_count _sub _nwrtn -> _count;
	enduntil;
enddefine;

define lconstant File_write = Write_file(% Do_write %);
enddefine;
;;;
define lconstant Special_write = Write_file(% Do_special_write %);
enddefine;
;;;
define lconstant Console_write = Write_file(% Do_console_write %);
enddefine;


;;; --- FLUSHING ------------------------------------------------------------

define lconstant File_flush(dev);
	unless Pf_apply(dev!D_CTRL_BLK!DCB_FILE, _extern pop_file_flush) then
		Write_error(dev);
	endunless;
enddefine;

define lconstant File_clear_input(dev);
	_extern pop_file_clear_input(dev!D_CTRL_BLK!DCB_FILE) -> ;
enddefine;


;;; --- SEEKING  ------------------------------------------------------------

define lconstant File_seek(dev, _pos, _mode);
	_extern pop_file_seek(dev!D_CTRL_BLK!DCB_FILE, _int(_pos), _int(_mode))
		-> _pos;
	if (_pos == _16:FFFFFFFF) then
		dev!D_CTRL_BLK!DCB_FILE!PF_ERROR -> _syserror;
		Syserr_mishap(dev, 1, 'ERROR SEEKING DEVICE');
	endif;
	_pint(_pos);
enddefine;


;;; --- CLOSING -------------------------------------------------------------

define lconstant File_close(dev);
	lvars dev;
	unless dev!D_WRITE == No_write then
		fast_apply(dev, dev!D_FLUSH)
	endunless;
	Pf_apply(dev!D_CTRL_BLK!DCB_FILE, _extern pop_file_close) -> ;
	;;; pop_file_close may have failed, but there's not much we can do...
	lvars unit_n = dev!D_UNIT_N;
	unit_n!UNT_NDEVS _sub _1 -> unit_n!UNT_NDEVS;
	dev!D_FLAGS _biset _M_D_CLOSED -> dev!D_FLAGS;
enddefine;


;;; --- OPENING -------------------------------------------------------------

define lconstant Set_dev_unit(dev, arg3, _force_console);

	define lconstant Get_dev_unit(_handle, _force_console) -> (unit_n, unit_p);
		dlvars _volume = _0, _index_hi = _0, _index_lo = _0;
		;;; get the device type
		lvars _type =
			if _force_console then
				;;; console to be opened later
				_:POP_FILE_TYPE_CONSOLE
			else
				_extern pop_get_file_type(_handle);
			endif;
		;;; search for an existing device referring to the same file
		if _type == _:POP_FILE_TYPE_DISK then
			define lvars Test_same_unit(dev);
				lvars unit_n = dev!D_UNIT_N;
				if unit_n!UNT_TYPE == _:POP_FILE_TYPE_DISK
					and unit_n!UNT_VOLUME_NUMBER == _volume
					and unit_n!UNT_INDEX_HI == _index_hi
					and unit_n!UNT_INDEX_LO == _index_lo
				then
					;;; same device
					exitto((unit_n, dev!D_UNIT_P), Set_dev_unit);
				endif;
			enddefine;
			_extern pop_get_file_id(_handle, ident _volume, ident _index_hi,
				ident _index_lo) -> ;
			;;; this call won't return if a matching device is found
			App_open_devs(Test_same_unit);
		elseif _type == _:POP_FILE_TYPE_CONSOLE then
			define lvars Test_same_unit(dev);
				if dev!D_UNIT_N!UNT_TYPE == _:POP_FILE_TYPE_CONSOLE then
					;;; only ever one console
					exitto((dev!D_UNIT_N, dev!D_UNIT_P), Set_dev_unit);
				endif;
			enddefine;
			unless _force_console then
				;;; true console device: set event handlers, etc.
				_extern pop_init_console() -> ;
			endunless;
			;;; this call won't return if a matching device is found
			App_open_devs(Test_same_unit);
		endif;
		;;; create new unit structures for this handle
		Ensure_writeable(inits(UNIT_N_LENGTH)) -> unit_n;
		Ensure_writeable(initv(UNIT_P_LENGTH)) -> unit_p;
		_0			-> unit_n!UNT_NDEVS;
		_type		-> unit_n!UNT_TYPE;
		_volume		-> unit_n!UNT_VOLUME_NUMBER;
		_index_hi	-> unit_n!UNT_INDEX_HI;
		_index_lo	-> unit_n!UNT_INDEX_LO;
		_0			-> unit_n!UNT_FLAGS;
		false		-> unit_p!UNT_INPUT_TRAP;
	enddefine;

	lvars (unit_n, unit_p) = Get_dev_unit(dev!D_CTRL_BLK!DCB_HANDLE,
										  _force_console);
	(unit_n, unit_p) -> (dev!D_UNIT_N, dev!D_UNIT_P);
	unless _force_console then
		;;; increase ref count for unit
		unit_n!UNT_NDEVS _add _1 -> unit_n!UNT_NDEVS;
	endunless;
enddefine;

define lconstant Set_dev_methods(dev, access, arg3, _overlapped);
	lvars ctrl_blk = dev!D_CTRL_BLK;

	;;; allocate file structure
	lvars _pf = _extern pop_file_create(ctrl_blk!DCB_HANDLE,
										_overlapped and _1 or _0);
	if _zero(_pf) then
		GET_LAST_ERROR;
		Kill_device(dev);
		Syserr_mishap(dev, 1, 'ERROR ALLOCATING DEVICE');
	endif;
	_pf -> ctrl_blk!DCB_FILE;

	;;; set the device unit structures
	Set_dev_unit(dev, arg3, false);

	;;; determine the device mode and flags
	lvars _type = dev!D_UNIT_N!UNT_TYPE, _mode = _0, _flags = _0;
	if arg3 == false or arg3 == "line" then
		;;; text mode
		_mode _biset _:POP_FILE_MODE_TEXT -> _mode;
	else
		;;; binary mode -- set D_ENCODING_ID to constant id
		;;; containing false
		ident no_device_encoding -> dev!D_ENCODING_ID;
	endif;
	if arg3 == "line" or arg3 == "record" then
		_mode _biset _:POP_FILE_MODE_LINE -> _mode;
	endif;
	if _type == _:POP_FILE_TYPE_CONSOLE then
		_flags _biset _M_D_TERMINAL _biset _M_D_LOGICAL_TERM -> _flags;
		if arg3 then
			_mode _biset _:POP_FILE_MODE_RAW -> _mode;
		else
			_flags _biset _M_D_TERM_PROMPT -> _flags;
		endif;
	endif;
	unless _type == _:POP_FILE_TYPE_DISK then
		_flags _biset _M_D_INTERACTIVE -> _flags;
	endunless;
	_mode -> ctrl_blk!DCB_MODE;
	_flags -> dev!D_FLAGS;

	;;; use access + mode to select device methods
	unless access == 1 then
		;;; readable
		if _type == _:POP_FILE_TYPE_CONSOLE then
			Console_read
		elseif _flags _bitst _M_D_INTERACTIVE then
			Interactive_read
		elseif _nonzero(_mode) then
			Special_read
		else
			File_read
		endif -> dev!D_READ;
		File_clear_input -> dev!D_CLEAR_INPUT;
		weakref[sys_input_waiting] Test_input -> dev!D_TEST_INPUT
	endunless;
	unless access == 0 then
		;;; writeable
		if _type == _:POP_FILE_TYPE_CONSOLE then
			Console_write
		elseif _nonzero(_mode) then
			Special_write
		else
			File_write
		endif -> dev!D_WRITE;
		File_flush -> dev!D_FLUSH;
	endunless;
	if _type == _:POP_FILE_TYPE_DISK then
		File_seek -> dev!D_SEEK;
	endif;
	File_close -> dev!D_CLOSE;

	Add_file_tab_entry(dev);
enddefine;

define lconstant New_device(file, fullname, _handle) -> dev;
	Init_device() -> dev;
	file -> dev!D_OPEN_NAME;
	fullname -> dev!D_FULL_NAME;
	;;; create device control block
	lvars ctrl_blk = Ensure_writeable(inits(CTRL_BLK_LENGTH));
	ctrl_blk -> dev!D_CTRL_BLK;
	_handle -> ctrl_blk!DCB_HANDLE;
enddefine;

define Cons_device(file, fullname, access, arg3, _handle, _overlapped) -> dev;
	New_device(file, fullname, _handle) -> dev;
	Set_dev_methods(dev, access, arg3, _overlapped);
enddefine;

define Cons_stddev(name, fd, arg3) -> dev;
	define Get_handle(fd, arg3, create);
		if arg3 then
			;;; raw i/o always goes to the console
			_extern pop_get_console_handle(_int(fd))
		else
			_extern pop_get_std_handle(_int(fd), create and _1 or _0)
		endif;
	enddefine;
	lvars _handle = Get_handle(fd, arg3, false);
	unless _handle == _:WIN32_INVALID_HANDLE_VALUE then
		chain(name, false, fd == 0 and 0 or 1, arg3, _handle, false,
			Cons_device);
	endunless;
	;;; special case for when std i/o not set up (e.g. from GUI app)
	New_device(name, false, _handle) -> dev;
	;;; allocate unit structures now so that they come from the
	;;; no-restore segment
	Set_dev_unit(dev, arg3, true);
	;;; set device methods to open the console on first use
	if fd == 0 then
		;;; stdin: allow reads as if from a null device
		procedure(dev, _bsub, userbuf, _nbytes);
			lvars _handle = Get_handle(fd, arg3, false);
			returnif(_handle == _:WIN32_INVALID_HANDLE_VALUE)(0);
			_handle -> dev!D_CTRL_BLK!DCB_HANDLE;
			Set_dev_methods(dev, 0, arg3, false);
			chain(dev, _bsub, userbuf, _nbytes, dev!D_READ);
		endprocedure -> dev!D_READ;
		procedure(dev);
			0;
		endprocedure -> dev!D_TEST_INPUT;
		erase -> dev!D_CLEAR_INPUT;
	else
		;;; stdout/err: attach to console on first write
		procedure(dev, _bsub, userbuf, _nbytes);
			returnunless(_nbytes fi_> 0);
			lvars _handle = Get_handle(fd, arg3, true);
			returnif(_handle == _:WIN32_INVALID_HANDLE_VALUE);
			_handle -> dev!D_CTRL_BLK!DCB_HANDLE;
			Set_dev_methods(dev, 1, arg3, false);
			chain(dev, _bsub, userbuf, _nbytes, dev!D_WRITE);
		endprocedure -> dev!D_WRITE;
		erase -> dev!D_FLUSH;
	endif;
enddefine;

	;;; Public interface
define Sys_cons_device(file, fullname, access, arg3, handle, _overlapped);
	Cons_device(file, fullname, access, arg3, Pint_->_uint(handle, _-1),
		_overlapped);
enddefine;


;;; --- OTHER ---------------------------------------------------------------

define Get_devio_trap(_data);
	lvars _data;
	false;
enddefine;


endsection;		/* $-Sys$-Io */


/* --- Revision History ---------------------------------------------------
--- Robert Duncan, Jun 16 1997
		Fixed _bsub in Read_file and Write_file to take account of
		character size.
--- Robert Duncan, May 22 1997
		Made Write_error set M_UNT_IGNORE_WRITE_ERR flag when called on a
		standard output/error device to stop further mishaps on the device
--- Robert Duncan, May 16 1997
		Added check in Read_file for a buffer of the right type
--- Robert Duncan, Apr  9 1997
		Replaced D_ENCODING with D_ENCODING_ID
--- Robert Duncan, Mar 18 1997
		Exported Sys_cons_device
--- Robert Duncan, Mar 12 1997
		New representation for device encoding
--- Robert Duncan, Feb 27 1997
		Changes to support device encoding
--- Robert Duncan, Jan 29 1997
		Modifications for UNICODE compilation
--- Robert Duncan, Nov 18 1996
		Fixed Cons_stddev so that it creates the unit structures at the
		same time as the device so that they're properly allocated from the
		no-restore segment.
--- Robert Duncan, Apr 19 1996
		Added Cons_stddev which arranges for a console window to be created
		as needed when writing to stdout/stderr.
--- Robert John Duncan, Jan  8 1996
		Complete rewrite for new I/O interface
 */
