/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_rc_window.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993 (see revisions)
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 2nd June 1993
;;; File go_rc_window.p
;;; Author Ben Rabau (see Revision History)
;;; Info: see go.p

compile_mode :pop11 +strict;

;;; This file contains two classes:
;;;     1. CLASS go_rc_window_pane
;;;     2. CLASS go_rc_window

uses rc_graphic;
uses go_window_pane;

;;;------------------------------------------------------
;;; MIXIN GO_RC_WINDOW_PANE

;;; INTRO: CLASS go_rc_window_pane makes a go_window_pane inside which can be
;;; INTRO: synchronized with the basic RC_GRAPHICS variables. In principle
;;; INTRO: this is only safe when used in conjunction with a physical window
;;; INTRO: which was created with rc_start(); this is due to the limited
;;; INTRO: number of variables which are synchronised: rc_[x/y]scale,
;;; INTRO: rc_[x/y]origin, rc_linewidth, rc_linestyle and rc_linefunction.
;;; INTRO: Others may need to be added for some usages...

define :class go_rc_window_pane;
	isa go_window_pane;

enddefine;

;;;------------------------------------------------------

define :method go_copy_from_rc( rc_pane :go_rc_window_pane);
lvars rc_pane;
;;; REF: go_copy_from_rc( GO_RC_WINDOW_PANE );
;;; REF: Synchronise the pane with the RC_WINDOW environment (REF * RC_GRAPHICS).
;;; REF: A limited number of variables are synchronised: rc_[x/y]scale,
;;; REF: rc_[x/y]origin, rc_linewidth, rc_linestyle and rc_linefunction.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
;;; The following vars are needed because of mutual influencing of scale/origin
lvars xorigin = rc_xorigin, yorigin = rc_yorigin;
	rc_xscale       -> go_xscale( rc_pane );
	rc_yscale       -> go_yscale( rc_pane );
	rc_linewidth    -> go_linewidth( rc_pane );
	rc_linestyle    -> go_linestyle( rc_pane );
	rc_linefunction -> go_linefunction( rc_pane );
	;;; This needs to be done last because things like scaling influence it!
	xorigin -> go_xorigin( rc_pane );
	yorigin -> go_yorigin( rc_pane );
enddefine;

;;;------------------------------------------------------

define :method go_copy_to_rc( rc_pane :go_rc_window_pane);
lvars rc_pane;
;;; REF: go_copy_to_rc( GO_RC_WINDOW_PANE );
;;; REF: Synchronise the RC_WINDOW environment with the pane (REF * RC_GRAPHICS).
;;; REF: A limited number of variables are synchronised: go_[x/y]scale,
;;; REF: go_[x/y]origin, go_linewidth, go_linestyle and go_linefunction.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	go_xscale( rc_pane )       -> rc_xscale;
	go_yscale( rc_pane )       -> rc_yscale;
	go_xorigin( rc_pane )      -> rc_xorigin;
	go_yorigin( rc_pane )      -> rc_yorigin;
	go_linewidth( rc_pane )    -> rc_linewidth;
	go_linestyle( rc_pane )    -> rc_linestyle;
	go_linefunction( rc_pane ) -> rc_linefunction;
enddefine;

;;;------------------------------------------------------
;;; SCALING

define :method updaterof go_xscale( scale, rc_pane :go_rc_window_pane );
lvars scale, rc_pane;
;;; REF: go_xscale( RC_WINDOW_PANE );
;;; REF: Synchronise rc_xscale from rc_graphics with the pane's go_xscale.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( scale, rc_pane );
	rc_pane.stored_go_xscale -> rc_xscale;
enddefine;

define :method updaterof go_yscale( scale, rc_pane :go_rc_window_pane );
lvars scale, rc_pane;
;;; REF: go_yscale( RC_WINDOW_PANE );
;;; REF: Synchronise rc_yscale from rc_graphics with the pane's go_yscale.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( scale, rc_pane );
	rc_pane.stored_go_yscale -> rc_yscale;
enddefine;

;;;------------------------------------------------------
;;; ORIGIN

define :method updaterof go_xorigin( origin, rc_pane :go_rc_window_pane );
lvars origin, rc_pane;
;;; REF: go_xorigin( RC_WINDOW_PANE );
;;; REF: Synchronise the rc_xorigin from rc_graphics with the pane's go_xorigin.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( origin, rc_pane );
	rc_pane.stored_go_xorigin -> rc_xorigin;
enddefine;

define :method updaterof go_yorigin( origin, rc_pane :go_rc_window_pane );
lvars origin, rc_pane;
;;; REF: go_yorigin( RC_WINDOW_PANE );
;;; REF: Synchronise the rc_yorigin from rc_graphics with the pane's go_yorigin.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( origin, rc_pane );
	rc_pane.stored_go_yorigin -> rc_yorigin;
enddefine;

;;;------------------------------------------------------
;;; LINES

define :method updaterof go_linewidth( lwidth, rc_pane :go_rc_window_pane );
lvars lwidth, rc_pane;
;;; REF: go_linewidth( RC_WINDOW_PANE );
;;; REF: Synchronise rc_linewidth from rc_graphics with the pane's
;;; REF: go_linewidth.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( lwidth, rc_pane );
	lwidth -> rc_linewidth;
enddefine;

define :method updaterof go_linestyle( lstyle, rc_pane :go_rc_window_pane );
lvars lstyle, rc_pane;
;;; REF: go_linestyle( RC_WINDOW_PANE );
;;; REF: Synchronise the rc_linestyle from rc_graphics with the pane's
;;; REF: go_linestyle.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( lstyle, rc_pane );
	lstyle -> rc_linestyle;
enddefine;

define :method updaterof go_linefunction( lfunction, rc_pane :go_rc_window_pane );
lvars lfunction, rc_pane;
;;; REF: go_linefunction( RC_WINDOW_PANE );
;;; REF: Synchronise rc_linefunction from rc_graphics with the pane's
;;; REF: go_linefunction.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
	call_next_method( lfunction, rc_pane );
	lfunction -> rc_linefunction;
enddefine;

;;;------------------------------------------------------
;;; SAFE REGION

define :method go_safe_region( dummy, pane :go_rc_window_pane ) -> (x, y, w, h);
lvars dummy, pane, x = 0, y = 0, w = rc_window_xsize, h = rc_window_ysize;
;;; REF: go_safe_region( dummy, RC_WINDOW_PANE ); returns the
;;; REF: location and size of the rc_window in screen coords.
;;; REF: The dummy argument (normally "false") indicates that the
;;; REF: shell has no parent in which it is visualized.
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
enddefine;


;;;------------------------------------------------------
;;; CLASS GO_RC_WINDOW

;;; INTRO: CLASS go_rc_window makes a RC_WINDOW with a go_window_pane inside.
;;; INTRO: This is an alternative to the go_shell class (see REF * GO_SHELL).

vars procedure go_create_new_rc;               ;;; defined below;
vars procedure go_copy_from_rc;                ;;; defined below;

define :class go_rc_window;
	isa go_rc_window_pane;

	on cons do procedure() -> rc_pane;
		lvars rc_pane = apply();
		go_create_new_rc( rc_pane );           ;;; create new RC window
		go_copy_from_rc( rc_pane );            ;;; copy attributes from RC
	endprocedure;
enddefine;

;;;------------------------------------------------------

define :method go_create_new_rc( rc_pane :go_rc_window);
lvars rc_pane;
;;; REF: go_create_new_rc( GO_RC_WINDOW );
;;; REF: Creates a new RC_WINDOW and attaches it to the go_rc_window object
;;; REF: (see also REF * RC_GRAPHIC).
;;; REF: This method overwrites any existing contents of the global
;;; REF: variable rc_window, and uses rc_start() to create a new rc_window.
;;; REF: GO_RC_WINDOW is a go_rc_window instance (REF * GO_RC_WINDOW).
		false -> rc_window;                    ;;; forget any old rc_window(s)
		rc_start();                            ;;; create a new go_rc_window
		rc_window -> the_go_window( rc_pane ); ;;; store the access for GO
enddefine;

;;;------------------------------------------------------
;;; THE SHELL
define :method the_go_shell( rc_pane :go_rc_window_pane) -> shell;
lvars rc_pane, shell = false;
;;; REF: the_go_shell( RC_WINDOW_PANE );
;;; REF: Returns the pane's physical shell (see also REF * GO_SHELL).
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
lvars window = the_go_window( rc_pane );
	if ( window ) then
		XtParent( window ) -> shell;
	endif;
enddefine;


;;;------------------------------------------------------
;;; VISIBILITY:

define :method go_show( rc_pane :go_rc_window );
lvars pane;
;;; REF: go_show( GO_RC_WINDOW );
;;; REF: Make the window. with a pane using rc_graphics, appear on the screen.
;;; REF: Note that the shell has no parent in which it is visualized.
;;; REF: GO_RC_WINDOW is a go_rc_window instance (REF * GO_RC_WINDOW).
lvars shell = the_go_shell( rc_pane );
	;;; To assure it comes on top we make it dissapear first.
	if ( shell ) then XtUnmapWidget( shell ); XtMapWidget( shell ); endif;
enddefine;

define :method go_hide( rc_pane :go_rc_window );
lvars pane;
;;; REF: go_hide( GO_RC_WINDOW );
;;; REF: Make the window, with a pane using rc_graphics, disappear from the
;;; REF: screen. This affects the shell in which the pnae is visualized.
;;; REF: GO_RC_WINDOW is a go_rc_window instance (REF * GO_RC_WINDOW).
lvars shell = the_go_shell( rc_pane );
	if ( shell ) then XtUnmapWidget( shell ); endif;
enddefine;


;;;------------------------------------------------------
;;; DIMENSIONS:


define :method updaterof go_window_height( h, pane :go_rc_window );
lvars h, pane;
;;; REF: INTEGER -> go_window_height( GO_RC_WINDOW);
;;; REF: The vertical screen height of the window pane
;;; REF: relative to its parent (see REF * GO_SHELL). This can have an
;;; REF: effect on other windows from the same parent as the window manager
;;; REF: might decide to move them around for a better fit.
;;; REF: Must also update rc_window_ysize,
	abs(h) -> rc_window_ysize;
	call_next_method(h, pane);
enddefine;

define :method updaterof go_window_width( w, pane :go_rc_window );
lvars w, pane;
;;; REF: INTEGER -> go_window_width( GO_RC_WINDOW);
;;; REF: The horizontal screen width of the window pane
;;; REF: relative to its parent (see REF * GO_SHELL). This can have an
;;; REF: effect on other windows from the same parent as the window manager
;;; REF: might decide to move them around for a better fit.
;;; REF: Must also update rc_window_xsize,
	abs(w) -> rc_window_xsize;
	call_next_method(w, pane);
enddefine;


;;;------------------------------------------------------
;;; DESTROY: tentative definition for destroying a rc_pane

define :method go_destroy_object( rc_pane :go_rc_window_pane );
lvars rc_pane;
;;; REF: go_destroy_object( RC_WINDOW_PANE );
;;; REF: Make the RC_WINDOW disappear from the screen and overwrite the
;;; REF: global variable rc_window with false (see also REF * RC_GRAPHIC).
;;; REF: Note: currently a safe approach is used to avoid problems with
;;; REF: active variables from RC_GRAPHICS: the widget is not destroyed
;;; REF: but only un-realized (see also REF * XtUnrealizeWidget).
;;; REF: RC_WINDOW_PANE is a go_rc_window_pane instance (REF * GO_RC_WINDOW_PANE)
lvars shell = the_go_shell( rc_pane ); ;;; before overwritten
	if (rc_pane == go_default_pane) then
		false -> rc_window;
	endif;
	call_next_method( rc_pane );
	if ( shell ) then
		;;; Destroy the widget itself. Note that Unrealize only makes
		;;; it dissapear from the screen (waiting for the next Realize).
		;;; This is done for safety but needn't be: could use XtDestroyWidget.
		XtUnrealizeWidget( shell );
	endif;
enddefine;

;;;----------------------------------------------------------------
;;; COPY

define :method go_copy_object( rc_pane :go_rc_window ) -> new_rc_pane;
lvars rc_pane, new_rc_pane;
lvars obj;
;;; REF: go_copy_object( GO_RC_WINDOW ) -> GO_RC_WINDOW;
;;; REF: Make exact copy of the shell with its rc_graphics pane inside.
;;; REF: This copy will contain its own Graphics panes (go_window_pane) and
;;; REF: the objects in the panes of the original shell are  SHARED!: they
;;; REF: will move in both panes if dragged... This is not the case for
;;; REF: the go_shell class (see REF * GO_SHELL).
;;; REF: This can for example be used to provide a zoom-facility.
;;; REF: GO_RC_WINDOW is a go_rc_window instance (REF * GO_RC_WINDOW).
	copy( rc_pane ) -> new_rc_pane;
	go_create_new_rc( new_rc_pane );
	;;; avoid copying stuff which is not sharable
	go_initialise_pane( new_rc_pane ) ->;
	go_fgcolour( rc_pane ) -> go_fgcolour( new_rc_pane ) ;
	go_bgcolour( rc_pane ) -> go_bgcolour( new_rc_pane ) ;
	for obj in rev( rc_pane.the_go_components ) do
		go_add_to( obj, new_rc_pane );
	endfor;
	go_show( new_rc_pane );
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_rc_window = true;

/* --- Revision History ---------------------------------------------------
--- Simon Nichols, Dec  1 1993
		Changed go_rcwindow to go_rc_window.
--- Integral Solutions Ltd (Julian Clinton), Nov 24 1993
		Added updater methods for go_window_width/height from Aaron
		Sloman and Ben Rabau.
 * BR 02/06/93
 *     Reversed stacking order in go_copy_object().
 * BR 09/05/93
 *     Replaced calls to go_refresh_pane() with go_redraw().
 * BR 28/04/93
 *     Created new CLASS go_rc_window_pane.
 *     Obsoleted the go_top(), go_right(), go_bottom() & go_left() methods
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 16/11/92
 *     Created this file to act as front end to RC_GRAPHIC windows.
 */
