/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_bitmap.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993 (see revisions)
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 20th Sept 1993
;;; File go_bitmap.p
;;; Author Ben Rabau (see Revision History)
;;; Info: HELP * GO, REF * GO

compile_mode :pop11 +strict;

;;; INTRO: The CLASS go_bitmap adds iconic objects to the go library.
;;; INTRO: Bitmaps can only be represented horizontally on a bit per bit
;;; INTRO: basis between the file and the screen. Scaling will only affect
;;; INTRO: the portion of the bitmap visible on the screen. It will not
;;; INTRO: zoom in or out on the original bits in the picture.
;;; INTRO: This uses the fast the fast go_drag method defined in go_xdrag.p.

uses go_screen_object;
uses go_xdrag.p

include XpwPixmap;

;;; ---------------------------------------------------------
;;; TWO LEVEL EXTENSION: go_load_bitmap_file()
;;;    Creates a Pixmap widget which will hold the go_bitmap
;;;    Then loads it from file with XpwLoadPixmap (we can't
;;;    use XpwGetPixmap if we want:
;;;         1. to destroy the unnecessary ones, except by
;;;            keeping track of the # of repetitions
;;;         2. to change the pixmaps, except if not shared
;;; ---------------------------------------------------------

;;; INTRO: The bitmaps which are read in can have different size, unfortunately
;;; INTRO: it is rather tricky to find out which size. It is therefore up to
;;; INTRO: the user to deal with the size of the bitmap created. This is done
;;; INTRO: through the go_bounding_width() and go_bounding_height() methods.

;;; ---------------------------------------------------------

define :class go_bitmap;
	isa go_screen_object;

	slot stored_go_bounding_width    == 64;
;;; REF: Stored value of width in world coordinates (size = go_bitmap 64*64)
	slot stored_go_bounding_height   == 64;
;;; REF: Stored value of height in world coordinates (size = go_bitmap 64*64)
	slot stored_go_iconname == '$usepop/pop/x/ved/bitmaps/tmp_64.xbm';
;;; REF: Stored value of the bitmap file name:

	slot cached_go_icons = newproperty([],  16, false, "tmparg");
;;; REF: property with cached bitmap pixmaps indexed per visible_in_pane
enddefine;

;;; ---------------------------------------------------------
;;; FILENAME OF ICON/BITMAP

define :method go_iconname( slf :go_bitmap );
lvars slf;
;;; REF: go_iconname( BITMAP ) -> STRING;
;;; REF: Gets the string representing the bitmap's file-name.
;;; REF: BITMAP is an instance of the go_bitmap class (REF * GO_BITMAP)
	slf.stored_go_iconname;
enddefine;


define :method updaterof go_iconname( name_string, slf :go_bitmap );
lvars name_string, slf;
;;; REF: STRING -> go_iconname( BITMAP );
;;; REF: Sets the string representing the bitmap's file-name and loads it.
;;; REF: BITMAP is an instance of the go_bitmap class (REF * GO_BITMAP)
	sysfileok(name_string) -> slf.stored_go_iconname;
	go_update( slf );
	go_internal_redraw( slf );
enddefine;

;;; ---------------------------------------------------------

define :method go_load_bitmap_file( pane, slf :go_bitmap ) -> photo_widget;
lvars pane, slf;
;;; REF: go_load_bitmap_file( PANE, BITMAP );
;;; REF: Load the bitmap from file, and safe as a XpwPixmap (REF * XpwPixmap).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE)
;;; REF: BITMAP is an instance of the go_bitmap class (REF * GO_BITMAP)
lvars filename, photo_pixmap;
lvars widget, fg_pixel, bg_pixel, depth_pixel = 0 /*whatever the widget has*/;
	the_go_window( pane ) -> widget;
	go_iconname( slf ) -> filename;

	;;; get the go_bitmap in the same colors as the sketch tool
	if ( go_fgcolour(slf) ) then
		go_get_colour( go_fgcolour(slf), pane )
	else
		go_fgcolour(pane)
	endif -> fg_pixel;
	if ( go_bgcolour(slf) ) then
		go_get_colour( go_bgcolour(slf), pane )
	else
		go_bgcolour(pane)
	endif -> bg_pixel;
	XtCreateWidget('go_bitmap image', XpwPixmap, widget, XptArgList([
					{^XtN width   ^(abs(slf.go_bounding_width))  }
					{^XtN height  ^(abs(slf.go_bounding_height)) }
				   ])) -> photo_widget;

	XpwLoadPixmap( photo_widget, filename, fg_pixel, bg_pixel, depth_pixel)
	-> photo_pixmap ;

	unless ( photo_pixmap ) then
		printf('Filename %p contains invalid bitmap data\n',[^filename]);
	else
		photo_pixmap -> XptValue( photo_widget, XtN pixmap);
	endunless;

enddefine;

;;; ---------------------------------------------------------

define :method go_reload_bitmap( slf :go_bitmap, pane :go_pane );
lvars slf, pane;
;;; REF: go_reload_bitmap( BITMAP );
;;; REF: Replace bitmap with new one if that is necessary and if the new
;;; REF: go_iconname refers to a valid bitmap.
;;; REF: BITMAP is an instance of the go_bitmap class (REF * GO_BITMAP)
lvars old_icon, new_icon;
							;;; load new pixmap
	go_load_bitmap_file(pane, slf) -> new_icon;
							;;; check whether pixmap is loaded
	if (new_icon) and (cached_go_icons(slf)(pane) ->> old_icon) then
							;;; don't destroy before new is installed
		XtDestroyWidget( old_icon );
	elseif (cached_go_icons(slf)(pane) ->> old_icon) then
							;;; remember old pixmap
		old_icon -> new_icon;
	endif;
	new_icon -> cached_go_icons( slf )( pane );
enddefine;

;;; ---------------------------------------------------------

define :method go_update( slf :go_bitmap );
lvars slf;
;;; REF: go_update( BITMAP );
;;; REF: Reload the bitmap for all known panes in which it is visible.
;;; REF: BITMAP is an instance of the go_bitmap class (REF * GO_BITMAP)
	go_applist( go_reload_bitmap, slf ); ;;; load new pixmap for each pane
enddefine;

;;; ---------------------------------------------------------
;;; DRAWING:

define :method go_draw( pane :go_pane, slf :go_bitmap );
lvars pane, slf;
;;; REF: go_draw( PANE, BITMAP );
;;; REF: Draws the bitmap object in the pane. The size will be determined
;;; REF: by scaling but the bitmap itself is not altered (i.e. if scaled
;;; REF: part of the picture might be blank or missing). The scaling will
;;; REF: be done around the centre.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE)
;;; REF: BITMAP is an instance of the go_bitmap class (REF * GO_BITMAP)
lvars icon_x, icon_y, icon_w2, icon_h2, x, y, w, h, w2, h2;
	if go_visible_now( slf ) == "forbidden" then return; endif;

							;;; showing go_bitmap after scaling position
	;;; The following misplaces the go_bitmap into the safety border...
	;;; go_safe_region(pane, slf) -> (icon_x, icon_y, icon_w, icon_h) ;
	go_transxyout( go_xcentre(slf), go_ycentre(slf), pane ) -> (icon_x, icon_y);

	unless ( cached_go_icons(slf)(pane) ) then
		go_reload_bitmap( slf, pane );
	endunless;

	;;; Find original size but halved (from centre away):
	slf.stored_go_bounding_width  div 2 -> w2;
	slf.stored_go_bounding_height div 2 -> h2;
	;;; Find the screen width and height but halved (from centre away):
	min(w2, abs((slf.go_bounding_width  * pane.go_xscale) div 2) ) -> icon_w2;
	min(h2, abs((slf.go_bounding_height * pane.go_yscale) div 2) ) -> icon_h2;

	icon_x - icon_w2 -> icon_x;
	icon_y - icon_h2 -> icon_y;

	max( 0, (w2 - icon_w2)) -> x;
	icon_w2*2 -> w;
	max( 0, (h2 - icon_h2)) -> y;
	icon_h2*2 -> h;
	go_draw_bitmap(pane, cached_go_icons(slf)(pane), x, y, w, h, icon_x, icon_y);
	true -> go_visible_now( slf );
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_bitmap = true;

/*  --- Revision History ---------------------------------------------------
--- Integral Solutions Ltd (Julian Clinton), Nov 24 1993
		go_iconname updater now applies -sysfileok- to name_string
 * BR 20/09/93
 *     Removed go_internal_redraw() from the go_update() method and added
 *     it to the go_iconname() updater method.
 * BR 27/08/93
 *     Renamed go_renew_bitmap() go_update() in line with other classes.
 *     Made scaling to go around the centre.
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 *     Changed name of default bitmap for OpenWindows 3.x
 *     Adapted (simplified) because new default of go_[x/y]scale() = 1.
 *     Added comments
 * BR 30/04/93
 *     Removed usage of the_go_icon() replaced directly by cached_go_icons().
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * BR 25/01/93
 *     Changed XpwCopyFrom() to start from (0, 0) rather than (1, 1)
 * BR 06/01/93
 *     There is a bug in XpwCopyTo when PixmapStatus = PixmapOnly; it draws
 *     to the screen instead of the pixmap => dissappears when PixmapOn:
 *     replaced by XpwCopyFrom();
 * BR 04/01/93
 *     Bug repair: property slot should never be shared ("==" -> "=")
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 12/11/92
 *    Adapted go_bounding_height to be a positive ax-system.
 * BR 09/11/92
 *    Adapted for OC 5.01 (no more initialise)
 * BR 16/10/92
 *    Added argument to go_draw() to indicate pane (removed rc_window)...
 * BR 08/10/92
 *    Changed to a go library (rather than sketch).
 *    go_make_invisible() now has pane as an extra argument (see MajorChanges)
 *  BR, Sep 28 1992
 *    Changed self into slf to avoid clashes with flavours
 *  BR, Jul 30 1992
 *    Added the_pop_up_menu = false;
 *  BR, Jul 24 1992
 *    Changed from container to go_group; slot fixed in go_group's definition
 *    Adapted to objectclass
 *  BR, May 22 1992
 *    Adapted go_bounding_height * -1 when dragged;
 */
;;; eof
