/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_circle.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 5th May 1993
;;; File: go_circle.p
;;; Author: J L Cunningham

compile_mode :pop11 +strict;

uses go_arc;

;;; INTRO: The go_circle CLASS offers circles and discs (filled circles).
;;; INTRO: For consistency with rectangles the x,y position won't be at the
;;; INTRO: centre, but at the top left corner of the enclosing bounding box.
;;; INTRO: This is a special case of the go_arc class see REF * GO_ARC.


;;;------------------------------------------------------
;;; CLASS CIRCLE (360 degrees go_arc)

define :class go_circle;
	isa go_arc;
	slot stored_go_arc == 360;
;;; REF: Full 360 circle is a go_arc
enddefine;

;;;------------------------------------------------------
;;; DIMENSIONS:

define :method updaterof go_arc( ar, circ :go_circle );
lvars ar, circ;
;;; REF: INTEGER -> ( CIRCLE );
;;; REF: Refuse to update the go_arc slot to a non-closed arc.
;;; REF: CIRCLE is an instance of the go_circle class (see REF * GO_CIRCLE).
	 unless ( ar == 360 ) then
		 mishap('Cannot update the go_arc of a go_circle', [^ar]);
	 endunless;
enddefine;

;;;------------------------------------------------------
;;; CONTAINS

define :method go_contains(mx, my, circ :go_circle );
lvars mx, my, circ;
;;; REF: go_contains( X, Y, CIRCLE ) -> FALSE_OR_CIRCLE;
;;; REF: Returns false if the world-coordinates (X,Y) do not fall in the circle
;;; REF: else it returns the circle object itself.
;;; REF: X and Y are the world coordinates of the test point.
;;; REF: CIRCLE is an instance of the go_circle class (see REF * GO_CIRCLE).
lvars sq_dist, the_radius = circ.go_radius;
lvars lw = circ.go_linewidth, cx = circ.go_xcentre, cy = circ.go_ycentre;
lvars the_xscale = circ.go_xscale, the_yscale = circ.go_yscale;
	if lw then (the_radius + (lw div 2)) -> the_radius; endif;
	((cx - mx)/the_xscale) * dup() + ((cy - my)/the_yscale) * dup() -> sq_dist;
	if  ( sq_dist <= the_radius * dup() )
	then circ else false endif;
enddefine;


;;;------------------------------------------------------------------
;;; OVERLAPS:

define :method go_overlaps( circ:go_circle, rec:go_located );
lvars circ, rec;
;;; REF: go_overlaps( CIRCLE, LOCATED ) -> BOOLEAN;
;;; REF: Checks whether the circle object overlaps the bounding box of a
;;; REF: located object.
;;; REF: CIRCLE is an instance of the go_circle class (see REF * GO_CIRCLE).
;;; REF: LOCATED is an go_located instance (see REF * GO_LOCATED).
;;; REF: The method is based on the following algorithm:
;;; REF:
;;; REF: To test whether the go_circle falls completely in the rec: (1)
;;; REF: if rec go_contains() the centre
;;; REF: To test whether the rec cuts at least one go_angle inside: (2)
;;; REF: if the nearest go_point from the centre to the line (perpendicular)
;;; REF: is inside the go_circle (distance < go_radius)
;;; REF: To test whether the go_circle takes a bit out of the side of a rec: (3)
;;; REF: if the North, South, East or West-most point of the circle falls
;;; REF: inside the go_rectangle (because rects are horizontal or vertical).
;;; REF:                   _-_
;;; REF:  E.g.            /   \  This is the smallest slice a rec can take.
;;; REF:                __\___/___
;;; REF:               |    x     |
;;; REF:               |__________|
;;; REF: Note that the "biggest possible slice" is covered by (1)
;;; REF: WARNING: this does not work properly when the scaling factors are not
;;; REF: equal (see go_xscale and go_yscale).
lvars cx = circ.go_xcentre, cy = circ.go_ycentre;
lvars left, right, top, bott, x, y;

	if go_contains(cx, cy, rec) then
		true;				;;; (1)
	else
		rec.go_xloc -> left;
		left + rec.go_bounding_width  -> right;
		rec.go_yloc -> top;
		top  + rec.go_bounding_height -> bott;

		;;; closest (x,y) from rect to go_circle
		if ( abs(left-cx) < abs(right-cx) ) then left else right endif -> x ;
		if ( abs(top-cy)  < abs(bott-cy)  ) then top  else bott  endif -> y ;

	if ( go_contains(x, y, circ) ) then
			true;			;;; (2)
	else
			;;; KNOWN BUG !!!!  go_overlaps doesn't work correctly
			;;;                 when go_xscale/go_yscale are not equal.
			lvars lw=circ.go_linewidth, rad = circ.go_radius, xrad, yrad,
				  xscale=circ.go_xscale, yscale = circ.go_yscale;
			rad; unless xscale == 1 then * xscale;   endunless;
				 if        lw       then + lw div 2; endif -> xrad;
			;;; line cuts in two go_points
			;;; check the closest TWO of North, South, East or West
			sign(x-cx)*xrad + cx -> x ;
			if ( go_contains(x, cy, rec) ) then
				true;			;;; (3a)
		else
				rad; unless yscale == 1 then * yscale;   endunless;
					 if        lw       then + lw div 2; endif -> yrad;
				sign(y-cy)*yrad + cy -> y ;
				if ( go_contains(cx, y, rec) ) then
					true;		;;; (3b)
			else
					false;
			endif;
		endif;
		endif;
	endif
enddefine;

define :method go_overlaps( rec:go_located, circ:go_circle );
lvars rec, circ;
;;; REF: go_overlaps( LOCATED, CIRCLE ) -> BOOLEAN;
;;; REF: Checks whether the circle object overlaps the bounding box of a
;;; REF: located object. See go_overlaps( CIRCLE, LOCATED ).
;;; REF: CIRCLE is an instance of the go_circle class (see REF * GO_CIRCLE).
;;; REF: LOCATED is an go_located instance (see REF * GO_LOCATED).
	go_overlaps( circ, rec );
enddefine;

define :method go_overlaps( circ1:go_circle, circ2:go_circle );
lvars circ1, circ2;
;;; REF: go_overlaps( CIRCLE1, CIRCLE2 ) -> BOOLEAN;
;;; REF: Checks whether the two circle objects overlaps each other.
;;; REF: CIRCLE1 is an instance of the go_circle class (see REF * GO_CIRCLE).
;;; REF: CIRCLE2 is an instance of the go_circle class (see REF * GO_CIRCLE).
	;;; two circles overlap if the distance between their centres
	;;; is smaller than the sum of their radii
	(circ1.go_xcentre - circ2.go_xcentre)*dup() +
	(circ1.go_ycentre - circ2.go_ycentre)*dup() <
	(circ1.go_radius  + circ2.go_radius )*dup() ;
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_circle = true;

/* --- Revision History --------------------------------------------
 * BR 07/05/93
 *     Corrected go_contains() & go_overlaps() for scaling of object to ellips.
 *     Split the file go_circle.p into go_arc.p and go_circle.p
 *     See go_arc.p for more information on original revision history.
 */
;;; eof
