/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_screen_object.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 24th Aug 1993
;;; File: go_screen_object.p
;;; Author: Ben L.E. Rabau

compile_mode :pop11 +strict;

uses go_located;
uses go_colourable;

/* Extra dependencies further in this file:
uses go_xdraw;
uses go_xdrag;
uses go_pane;
 */

;;; INTRO: The CLASS go_screen_object is one of the base classes in the
;;; INTRO: Graphical Objects library and introduces visualisation. The screen
;;; INTRO: objects will be visualised in one or more panes with a number of
;;; INTRO: characteristics such as the line width and colours used
;;; INTRO: (see REF * GO_PANE, REF * GO_COLOURABLE and REF * GO_LOCATED).
;;;----------------------------------------------------------------

define :class go_screen_object;
	isa go_colourable go_located;
	slot go_visible_in       == [];
;;; REF: List of panes in which the object is visible (REF * GO_PANE).
	slot go_visible_now      == false;
;;; REF: True if drawn now, false if cleared, "forbidden" if in batch mode.
	slot go_fixed_image      == false;
;;; REF: Boolean for optimised dragging, true = use pixmap (REF * GO_XDRAG).
	slot stored_go_linewidth == false;
;;; REF: Stored width of lines, if false use pane default.
	slot go_live_object      == false;
;;; REF: If false then take pane's mode, if true then always "live".
	slot go_editors          == [];
;;; REF: This contains a list of the objects' editors if any.
enddefine;


vars procedure go_clear,               ;;; DEFINED BELOW
	 procedure go_internal_redraw;     ;;; DEFINED BELOW

;;;----------------------------------------------------------------
;;; SENSITIVITY
vars go_transparent_to_events;    ;;; GLOB VAR
;;; REF: Global variable indicating whether an object is transparent to events,
;;; REF: i.e. it gets the event but the event is also passed to underlying
;;; REF: object(s). The default is not to be transparent, but this can be
;;; REF: overruled upon return of the method go_accepts_events.

vars go_forward_event_to = [];   ;;; GLOB VAR
;;; REF: Global variable indicating the objects to which an event is passed on.
;;; REF: By default an object does not pass on events, so only one element is
;;; REF: expected: that of the first object which accepts the event. If the
;;; REF: object broadcasts the event to other objects then these objects can
;;; REF: be added to this global variable (in the method go_accepts_events),
;;; REF: or the global variable go_transparent_to_events can be set to true upon
;;; REF: return of the method go_accepts_events in which case the search for
;;; REF: objects which accept the event will continue as before.


define :method go_accepts_events( type, raw_event_data, scrObj :go_screen_object ) -> accepts;
lvars type, raw_event_data, scrObj, accepts = false;
;;; REF: go_accepts_events( TYPE, RAW_EVENT_DATA, MOUSE_SENSITIVE ) -> BOOLEAN;
;;; REF: Returns whether or not this object accepts the event. By default
;;; REF: this is FALSE.
;;; REF: TYPE is one of "mouse" or "keyboard"
;;; REF: RAW_EVENT_DATA is a vector (see REF * GO_XACTION/go_expand_event_data).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
enddefine;


;;;----------------------------------------------------------------
;;; BATCHING UPDATES

define :method go_batch_mode_on( scrObj :go_screen_object );
lvars scrObj;
	go_clear( scrObj );
	"forbidden" -> go_visible_now( scrObj );
enddefine;

define :method go_batch_mode_off( scrObj :go_screen_object );
lvars scrObj;
	false -> go_visible_now( scrObj );
	go_internal_redraw( scrObj );
enddefine;

define :method go_batch_mode( scrObj :go_screen_object );
lvars scrObj;
	go_visible_now( scrObj ) == "forbidden";
enddefine;

define :method updaterof go_batch_mode( val, scrObj :go_screen_object );
lvars val, scrObj;
lvars old_batch = go_batch_mode( scrObj);
	if val and not(old_batch) then
		go_batch_mode_on( scrObj );
	elseif not(val) and old_batch then
		go_batch_mode_off( scrObj );
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; DRAW METHODS

vars procedure go_redraw_area;    ;;; DEFINED BELOW...

define lconstant go_do_the_redraw( show_scrObj, draw_below, scrObj );
lvars show_scrObj, draw_below, scrObj;
lvars pane;
	if ( scrObj.go_batch_mode ) then return; endif;
	for pane in scrObj.go_visible_in do
		go_redraw_area( show_scrObj, draw_below, pane, scrObj );
	endfor;
enddefine;


define :method go_redraw( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_redraw( SCREEN_OBJECT );
;;; REF: Redraws the screen object and all the objects above it in the area
;;; REF: occupied by the screen object.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_do_the_redraw( true, true, scrObj );
enddefine;

define :method go_internal_redraw( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_internal_redraw( SCREEN_OBJECT );
;;; REF: Redraws the screen object and all the objects above it in the area
;;; REF: occupied by the screen object. Since it does not redraw the objects
;;; REF: below the screen object this is not safe to use unless the object
;;; REF: was cleared just before (see REF * GO_SCREEN_OBJECT/go_clear).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_do_the_redraw( true, false, scrObj );
enddefine;

define :method go_clear( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_clear( SCREEN_OBJECT );
;;; REF: Redraws all the objects below and above it in the area occupied by the
;;; REF: screen object but not the screen object itself. This clears the object
;;; REF: from the screen but does not physically remove if from any pane object.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars pane;
	if ( scrObj.go_batch_mode ) then return; endif;
	false -> scrObj.go_visible_now;
	go_do_the_redraw( false, true, scrObj );
enddefine;


define :method go_clear_bounding_box( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_clear_bounding_box( SCREEN_OBJECT );
;;; REF: Convenience function to clear a bounding box drawn with the method
;;; REF: go_draw_bounding_box (see REF * GO_XDRAW).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_redraw( scrObj );
enddefine;


;;;----------------------------------------------------------------------------
;;; APPEARANCE

define :method go_linewidth( scrObj :go_screen_object );
;;; REF: go_linewidth( SCREEN_OBJECT ) -> INTEGER_OR_FALSE;
;;; REF: Gets the line width in which the screen object is drawn on a pane.
;;; REF: If false then the default from the pane itself is used (see also
;;; REF: REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars scrObj;
	scrObj.stored_go_linewidth;
enddefine;

define :method updaterof go_linewidth( lw, scrObj :go_screen_object );
lvars lw, scrObj;
;;; REF: INTEGER_OR_FALSE -> go_linewidth( SCREEN_OBJECT );
;;; REF: Sets the line width in which the screen object is drawn on a pane.
;;; REF: If false then the default from the pane itself is used (see also
;;; REF: REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_clear( scrObj );
	lw -> stored_go_linewidth( scrObj );
	go_internal_redraw( scrObj );
enddefine;


;;;----------------------------------------------------------------
;;; UPDATES:

vars procedure isgo_pane;                  ;;; DEFINED IN GO_PANE
vars procedure cached_go_coord_list;       ;;; DEFINED IN GO_PANE

define :method go_applist( proc, obj :go_screen_object );
lvars proc, obj;
;;; REF: go_applist( PROCEDURE, SCREEN_OBJECT );
;;; REF: Apply the procedure to the screen object for each of the panes
;;; REF: it is visible in (See also REF * APPLIST).
;;; REF: The procedure needs to take the format:
;;; REF:        <procedure name>( <SCREEN_OBJECT>, <PANE> );
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: PROCEDURE is a pop11 procedure with two arguments (object and pane).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	define lconstant applist_to_panes( child );
	lvars child, parent;
		for parent in child.go_visible_in do
			if parent.isgo_pane then proc( obj, parent );
								else applist_to_panes( parent );
			endif;
		endfor;
	enddefine;

	applist_to_panes( obj );
enddefine;


define :method go_update( obj :go_screen_object );
lvars obj;
;;; REF: go_update( SCREEN_OBJECT );
;;; REF: Remove all cached screen coordinates.
;;; REF: See also REF * GO_PANE/cached_go_coord_list
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_applist( procedure( scrobj, pane ); lvars scrobj, pane;
					false -> cached_go_coord_list(pane)(scrobj);
				endprocedure, obj );
enddefine;

define :method go_update_and_draw( obj :go_screen_object );
lvars obj;
;;; REF: go_update_and_draw( SCREEN_OBJECT );
;;; REF: Remove all cached screen coordinates then redraw (which recalculates).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars pane;
	go_update( obj );
	go_internal_redraw( obj );
enddefine;


;;;------------------------------------------------------------------
;;; LOCAL COORDINATES

define :method updaterof go_local_coords( offset_list, obj :go_screen_object );
lvars offset_list, obj;
;;; REF: LIST -> go_localcoords( SCREEN_OBJECT );
;;; REF: LIST is the list of horizontal and vertical integer coordinates of
;;; REF: points in the object's local coordinate system.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	procedure;
	dlocal 1 % go_batch_mode( obj ) % = true;
		call_next_method( offset_list, obj );
		go_update( obj );
	endprocedure();
enddefine;

;;;------------------------------------------------------------------
;;; ORIGIN

define :method updaterof go_xorigin( x, scrObj :go_screen_object );
lvars x, scrObj;
;;; REF: INTEGER -> go_xorigin( SCREEN_OBJECT );
;;; REF: The horizontal origin of the local coordinates of the object.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	call_next_method( x, scrObj );
	go_update( scrObj );
enddefine;

define :method updaterof go_yorigin( y, scrObj :go_screen_object );
lvars y, scrObj;
;;; REF: INTEGER -> go_yorigin( SCREEN_OBJECT );
;;; REF: The vertical origin of the local coordinates of the object.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	call_next_method( y, scrObj );
	go_update( scrObj );
enddefine;


;;;------------------------------------------------------------------
;;; SCALING

define :method updaterof go_xscale( scale, scrObj :go_screen_object );
lvars scale, scrObj;
;;; REF: INTEGER -> go_xscale( SCREEN_OBJECT );
;;; REF: Sets the horizontal scaling of the screen object which is reflected
;;; REF: in the panes in which it is visible (see REF * GO_PANE, GO_LOCATED).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars pane;
	go_clear( scrObj );
	call_next_method( scale, scrObj );
	go_update_and_draw( scrObj );
enddefine;


define :method updaterof go_yscale( scale, scrObj :go_screen_object );
lvars scale, scrObj;
;;; REF: INTEGER -> go_xscale( SCREEN_OBJECT );
;;; REF: Sets the vertical scaling of the screen object which is reflected
;;; REF: in the panes in which it is visible (see REF * GO_PANE, GO_LOCATED).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars pane;
	go_clear( scrObj );
	call_next_method( scale, scrObj );
	go_update_and_draw( scrObj );
enddefine;

;;;----------------------------------------------------------------
;;; PROGRAMMATICAL MOTION INTERFACE

define :method go_origin_to( x, y, scrObj :go_screen_object );
lvars x, y, scrObj;
;;; REF: go_origin_to( X, Y, SCREEN_OBJECT );
;;; REF: Sets both horizontal and vertical origin of a located object.
;;; REF: This is a convenience method using go_xorigin and go_yorigin.
;;; REF: X and Y are world coordinates of the reference point.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	if (x and x /= go_xorigin(scrObj)) or (y and y /= go_yorigin(scrObj)) then
		go_clear(scrObj);
		call_next_method( x, y, scrObj );
		go_update_and_draw( scrObj );
	endif;
enddefine;

define :method go_centre_to( x, y, scrObj :go_screen_object );
lvars x, y, scrObj;
;;; REF: go_centre_to( X, Y, SCREEN_OBJECT );
;;; REF: Move the centre point of the given screen object to the new location
;;; REF: (in world coordinates).
;;; REF: X and Y are world coordinates of the centre point.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	if (x and x /= go_xcentre(scrObj)) or (y and y /= go_ycentre(scrObj)) then
		go_clear(scrObj);
		call_next_method( x, y, scrObj );
		go_update_and_draw( scrObj );
	endif;
enddefine;

define :method go_move_to( x, y, scrObj :go_screen_object );
lvars x, y, scrObj;
;;; REF: go_move_to( X, Y, SCREEN_OBJECT );
;;; REF: Move the corner point of the given screen object to the new location
;;; REF: (in world coordinates).
;;; REF: X and Y are world coordinates of the corner point.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	if (x and x /= go_xloc(scrObj)) or (y and y /= go_yloc(scrObj)) then
		go_clear(scrObj);
		call_next_method( x, y, scrObj );
		go_update_and_draw( scrObj );
	endif;
enddefine;

;;;----------------------------------------------------------------
;;; THIS IS THE END OF THE SINGLE-METHODS
;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_screen_object = true;


;;;----------------------------------------------------------------
;;; THIS IS THE START OF THE MULTI-METHODS
;;;----------------------------------------------------------------

;;;----------------------------------------------------------------
;;; METHODS USING DRAWING: DEFAULT X-WINDOWS:
;;; uses go_xdraw;
vars procedure go_draw;                ;;; DEFINED IN GO_XDRAW
vars procedure go_draw_bounding_box;   ;;; DEFINED IN GO_XDRAW
vars procedure go_clear_rectangle;     ;;; DEFINED IN GO_XDRAW
vars procedure go_set_clipmask;        ;;; DEFINED IN GO_XDRAW
vars procedure go_reset_clipmask;      ;;; DEFINED IN GO_XDRAW
;;;----------------------------------------------------------------

;;;----------------------------------------------------------------
;;; MULTI-METHODS USING GO_PANE
uses go_pane;
;;;----------------------------------------------------------------


define :method go_fgdraw( pane :go_pane, scrObj :go_screen_object );
lvars pane, scrObj;
;;; REF: go_fgdraw( PANE, SCREEN_OBJECT );
;;; REF: Draws the part of the screen object in the pane which is to be drawn
;;; REF: in the foreground colour (see REF * GO_COLOURABLE/go_fgcolour). This
;;; REF: typically represents the outline of an object. The default picture
;;; REF: for a screen object is its bounding box although this will generate
;;; REF: a warning at present (see also REF * GO_XDRAW/go_draw_bounding_box).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; NOTE: could hide go_draw() methods from e.g. menus; if deleted it does
;;; however mean that you need a go_draw for "pane"s
	;;; do nothing, shadowed by the objects
	go_draw_bounding_box( scrObj );
enddefine;

define :method go_bgdraw( pane :go_pane, scrObj :go_screen_object );
lvars pane, scrObj;
;;; REF: go_fgdraw( PANE, SCREEN_OBJECT );
;;; REF: Draws the part of the screen object in the pane which is to be drawn
;;; REF: in the background colour (see REF * GO_COLOURABLE/go_bgcolour). This
;;; REF: typically represents the filling of an object (see REF * GO_FILLABLE).
;;; REF: If no definition is given nothing is drawn although this will generate
;;; REF: a warning at present.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	;;; do nothing, shadowed by the objects
enddefine;

;;;----------------------------------------------------------------

/* The ideas:
 *   1. When redrawing a rectangular go_area, the objects need to be drawn in
 * 	correct order to preserve the depth appearance.
 *   2. The objects can be clipped to a go_rectangle before drawing: clipped
 * 	roughly by sketch, finer by X
 *   3. If an object is found which fills the whole go_area then it is
 * 	unnecessary to print the rest.
 *   4. Dragging:
 *          1. CLEAR AREA
 *          2. AFFECTED OBJECTS to be REDRAWN
 *          3. TURN into TEMPORARY DRAWING mode
 *          4. DRAG, DRAW OBJECT
 *          5. REDRAW AFFECTED HIGHER OBJECTS
 *          6. DROP and turn TEMPORARY into PERMANENT
 *          7. REDRAW ALL AFFECTED AREAS
 * 	(6) will depend on whether we will used double buffering or not
 *
 * The depth is needed == the order in which they are stored in the list; this
 * will slowly become a lot more complex to allow for objects to be on the same
 * level even when they are in a go_group.
 */

vars go_edited_object;	  ;;; defined in MIXIN go_editor,    file go_editor.p

define :method go_find_occluded_list( find_below, pane, search_area,
									  scrObj :go_screen_object
									 ) -> (below, on_top, eds);
lvars find_below, pane, search_area, scrObj, below, on_top, eds;
;;; REF: go_find_occluded_list( BELOW, PANE, AREA, SCREEN_OBJECT ) -> OBJECTS;
;;; REF: This finds three lists of objects which are either below the screen
;;; REF: object, on top of the screen object or editors on top of it. In some
;;; REF: cases the objects below the screen object are not important which is
;;; REF: indicated with the "BELOW" flag. If this is the case then the area
;;; REF: searched will correspond to the screen object, otherwise the area is
;;; REF: extended to its bounding box.
;;; REF: BELOW is a boolean indicating if objects underneath are requested.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: AREA is a go_area or a go_screen_object instance (REF * GO_AREA)
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: OBJECTS is a tuple of three lists of located objects representing the
;;; REF: objects below and those above and the editors above the screen object.
lvars obj, edit_flag, topflag;
	;;; collect a list of items below or on top of the object
	define lconstant add_occluded( anobj );
	lvars anobj, grpobj;
		if  anobj.isgo_composite and (anobj.go_visible_now == true)
		and (edit_flag or anobj.go_live_object ) then
			for grpobj in rev(anobj.the_go_components) do
				add_occluded( grpobj ) ->;
			endfor;
		else
			if (topflag) and ( anobj == scrObj ) then
				if (find_below ) then false -> topflag;
				else return( false );
				endif;
			elseif ( anobj == pane ) then
			elseif ( go_overlaps(anobj, search_area) ) then
				if (topflag) then anobj::on_top -> on_top;
				else anobj::below  -> below;
				endif;
			endif;
		endif;
		true;
	enddefine;

	[] ->> below ->> on_top -> eds;
	not(go_edit_mode(pane)) -> edit_flag;
	true -> topflag;
	for obj in pane.the_go_components do
		unless( add_occluded( obj ) ) then quitloop; endunless;
	endfor;
	for obj in pane.go_editors do
		if  go_visible_now(go_edited_object(obj)) == true
		/* and go_overlaps(obj, search_area) ;;; ALWAYS DRAW EDITORS  */
		then
			obj::eds -> eds;
		endif;
	endfor;
enddefine;



define :method go_safe_region( pane :go_pane, scrObj :go_screen_object )
														-> (x, y, w, h);
lvars pane, scrObj;
;;; REF: go_safe_region( PANE, SCREEN_OBJECT ) -> (X, Y, W, H);
;;; REF: Returns a tuple with the screen region occupied by the screen object
;;; REF: in the given pane. It includes a safety border which includes line
;;; REF: width and space for editors if any (see REF * GO_EDITOR).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: X and Y are screen coordinates of the bottom left corner of the region.
;;; REF: W and H are the width and height in screen coordinates of the region.
;;; !!!! see also definition of go_safe_size in GO_DRAG !!!!
lvars extra_space, x, y, w, h, x0, y0, x1, y1, xp, yp, wp, hp;
lvars scrObj_lw = (scrObj.go_linewidth or 1), pane_lw = pane.go_linewidth;
	;;; The visible area:
	go_safe_region( the_go_pane(pane), pane ) -> (xp, yp, wp, hp);
	;;; The object's size (world coords ---> screen coords)
	go_bounding_box( scrObj ) -> (x, y, w, h);
	applist(go_translistout( [% x, y, x+w, y+h %], pane ), round)
	-> (x0, y0, x1, y1);

	;;; Make sure CapProjecting lines under an angle of 45degrees are
	;;; covered. Formula: extra_space = lw/2 * sqrt(2)
	max( 1, (max(scrObj_lw, pane_lw) * 1.5) div 2 ) -> extra_space;
	;;; Make sure the editors have some extra space:
	if (scrObj.go_editors) then extra_space + 2 -> extra_space; endif;

	;;; WARNING: Normalize go_rectangle: Make sure the corner is top left
	;;;          Might need adaptation for thicker lines
	if (x0 < x1) then max(x0 - extra_space, xp     )     -> x;
					  min(x1 + extra_space, xp + wp) - x -> w;
				 else max(x1 - extra_space, xp     )     -> x;
					  min(x0 + extra_space, xp + wp) - x -> w;
	endif;
	if (y0 < y1) then max(y0 - extra_space, yp     )     -> y;
					  min(y1 + extra_space, yp + hp) - y -> h;
				 else max(y1 - extra_space, yp     )     -> y;
					  min(y0 + extra_space, yp + hp) - y -> h;
	endif;

	;;; Minimum bounding box is size: one*one (if box outside pane)
	if ( w < 0 ) then xp+wp -> x; 1 -> w endif;
	if ( h < 0 ) then yp+hp -> y; 1 -> h endif;
enddefine;


;;; ----------------------------------------------------------------
;;; MULTI-METHODS USING GO_AREA:
uses go_area;
;;; ----------------------------------------------------------------

;;; ----------------------------------------------------------------
;;; RETURN AREA OCCUPIED BY A LOCATED OBJECT:

lvars bounding_rectangle = false;  ;;; Local var: one unique area

define :method go_get_bounding_area(x, y, w, h, pane, scrObj :go_screen_object);
lvars x, y, w, h, pane, scrObj;
;;; REF: go_get_bounding_area( X, Y, W, H, PANE, SCREEN_OBJECT ) -> AREA;
;;; REF: Gets the area object which corresponds to the given size in screen
;;; REF: coordinates. This is used to simplify calculations of complex
;;; REF: objects by using the simplest object (an area) instead. The area
;;; REF: returned is always the same and only its dimensions change. Care
;;; REF: should be taken never to use this method recursively!
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: AREA is a unique instance of the go_area class (REF * GO_AREA)
lvars x0, y0, x1, y1;
	unless (bounding_rectangle) then
		instance go_area; go_visible_now = "forbidden"; endinstance;
		-> bounding_rectangle;
	endunless;
	;;; Reverse transformation of safe_area to the world coordinates
	explode(go_translistin( [% x, y, x+w, y+h %], pane )) -> (x0, y0, x1, y1);
	[] -> bounding_rectangle.go_visible_in;
	x1 - x0 -> w;
	y1 - y0 -> h;
	x0 + (w div 2) -> bounding_rectangle.stored_go_xorigin;
	y0 + (h div 2) -> bounding_rectangle.stored_go_yorigin;
	w -> bounding_rectangle.stored_go_bounding_width;
	h -> bounding_rectangle.stored_go_bounding_height;
	scrObj.go_linewidth -> bounding_rectangle.stored_go_linewidth;
	scrObj.go_visible_in -> bounding_rectangle.go_visible_in;
	bounding_rectangle;
enddefine;


define :method go_redraw_area(show_scrObj, draw_below, pane :go_pane, scrObj:go_screen_object);
lvars scrObj, show_scrObj, draw_below, pane;
;;; REF: go_redraw_area( DRAW, BELOW, PANE, SCREEN_OBJECT );
;;; REF: Redraws the area occupied by the screen object if requested by
;;; REF: redrawing all the objects underneath the screen object, then if
;;; REF: requested the screen object itself, then all the objects and editors
;;; REF: on top of it. This is used by the go_redraw and go_clear methods.
;;; REF: DRAW is a boolean indicating if the object itself needs to be drawn.
;;; REF: BELOW is a boolean indicating if objects underneath are requested.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars search_area, below, on_top, eds;
lvars obj, x, y, w, h, xp, yp, wp, hp;
dlocal sys_signals_enabled = false;

	returnif( pane.the_go_window == false );

	go_safe_region( pane, scrObj ) -> (x, y, w, h);
	go_get_bounding_area( x, y, w, h, pane, scrObj ) -> search_area;

	if go_batch_mode( pane ) then
		"forbidden" -> scrObj.go_visible_now;
		false -> show_scrObj;
	endif;

	;;; If object will be shown also show editors.
	if  show_scrObj and not(scrObj.go_batch_mode)  then
		true -> scrObj.go_visible_now;
	endif;

	go_find_occluded_list( draw_below, pane, search_area, scrObj )
	-> (below, on_top, eds);

	go_set_clipmask( x, y, w, h, pane );

	if draw_below then
		go_clear_rectangle( pane, x, y, w, h );
		for obj in below do go_draw( pane, obj );
		endfor;
	endif;
	if    show_scrObj  then go_draw(pane,scrObj);
	endif;
	for obj  in  on_top  do go_draw( pane, obj );
	endfor;
	if pane.go_edit_mode then
		for obj in  eds  do go_draw( pane, obj );
		endfor;
	endif;
	go_reset_clipmask( pane );
enddefine;

;;;----------------------------------------------------------------
;;; METHODS USING DRAGGING: DEFAULT X-WINDOWS:
;;; uses go_xdrag;
vars procedure go_drag;                ;;; DEFINED IN GO_XDRAG
vars procedure inDrag;                 ;;; DEFINED IN GO_XDRAG
vars procedure lastCoords;             ;;; DEFINED IN GO_XDRAG
vars procedure startXOffset;           ;;; DEFINED IN GO_XDRAG
vars procedure startYOffset;           ;;; DEFINED IN GO_XDRAG
;;;----------------------------------------------------------------


define :method go_check_drag_limits(x, y, drag_obj :go_screen_object) -> (x, y);
lvars x, y, drag_obj;
;;; REF: go_check_drag_limits( X, Y, SCREEN_OBJECT ) -> (X, Y);
;;; REF: Checks whether the new position of the centre point of the given
;;; REF: screen object would keep it completely visible in the default pane.
;;; REF: If the position would move the object (partially) outside the pane,
;;; REF: then the nearest position inside the pane is returned. The default pane
;;; REF: is the last created or activated pane (REF * GO_VARS/go_default_pane).
;;; REF: This method can be redefined to increase or restrict the dragging
;;; REF: movements allowed by a certain type of screen objects.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars xp, yp, wp, hp, xmin, ymin, xmax, ymax;
lvars width=go_bounding_width(drag_obj), height=go_bounding_height(drag_obj);
lvars xloc = x - (width div 2), yloc = y - (height div 2);

	go_bounding_box( go_default_pane ) -> (xp, yp, wp, hp);
	if wp > 0 then xp   ->> xmin;
				   + wp ->  xmax;
			  else xp   ->> xmax;
				   + wp ->  xmin;
	endif;
	if hp > 0 then yp   ->> ymin;
				   + hp ->  ymax;
			  else yp   ->> ymax;
				   + hp ->  ymin;
	endif;
	if ( xloc < xmin ) then xmin + (width div 2) -> x;
	elseif ( xloc + width > xmax )
		then xmax - (width div 2) -> x;
	endif;
	if ( yloc < ymin ) then ymin + (height div 2) -> y;
	elseif ( yloc + height > ymax )
		then ymax - (height div 2) -> y;
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; ARRANGE = DRAG AND DROP INTERFACE
;;;      uses the fast go_drag mechanism from LIB * GO_XDRAG
;;;      needs go_motionhint to be true
;;;
;;; Warning: is redefined in go_filling.p (go_filled field)

vars go_drag_outline = false;    ;;; REF: Global var: to drag just the box
;;; REF: Global var: The outline exists of the bounding box, the object
;;; REF: Global var: itself is not shown.


define :method go_draw_drag(client, drag_obj :go_screen_object);
lvars client, drag_obj;
;;; REF: go_draw_drag( CLIENT_DATA, SCREEN_OBJECT );
;;; REF: Execute the drawing part when dragging a screen object. This will
;;; REF: be executed whenever the object has changed. See also the slot
;;; REF: go_fixed_image which optimised the drag operation by not calling
;;; REF: this function when the object simply moves (see TEACH * GO_DRAG).
;;; REF: The allowed positions of the object are controlled by the method:
;;; REF: go_check_drag_limits (see REF * GO_SCREEN_OBJECT/go_check_drag_limits).
;;; REF: The global variable go_drag_outline offers further optimisation:
;;; REF:     1. If go_drag_filled is false then objects will be dragged over the
;;; REF:        screen without being filled and on top of every other object.
;;; REF:     2. If go_drag_filled is true, a filled object will be filled and
;;; REF:        shown in its right screen depth even during dragging.
;;; REF: When the drop occurs the "inDrag" flag in the client data will be set
;;; REF: to false (see also REF * GO_XDRAG).
;;; REF: This method is called from the go_drag method and should never be
;;; REF: called directly.
;;; REF: CLIENT_DATA is a structure containing information for dragging.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars x, y;
	explode(client.lastCoords) -> (x, y);
	go_check_drag_limits(x, y, drag_obj) -> (x, y);
	x -> drag_obj.go_xcentre;
	y -> drag_obj.go_ycentre;
	if (client.inDrag) then
		unless (go_drag_outline) then
			go_internal_redraw( drag_obj );
		endunless;
		go_draw_bounding_box(drag_obj);
	else
		go_internal_redraw( drag_obj );
	endif;
	{% x, y %} -> client.lastCoords;
enddefine;

define :method go_clear_drag(client, drag_obj :go_screen_object);
lvars client, drag_obj;
;;; REF: go_draw_drag( CLIENT_DATA, SCREEN_OBJECT );
;;; REF: Execute the cleaning part when dragging a screen object. This will
;;; REF: be executed when the object is first picked up (see REF * GO_XDRAG).
;;; REF: CLIENT_DATA is a structure containing information for dragging.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	drag_obj.go_clear;
enddefine;

define :method go_drag_and_drop( drag_obj :go_screen_object );
lvars drag_obj;
;;; REF: go_drag_and_drop( SCREEN_OBJECT );
;;; REF: Pick up a screen object and move it around the screen. The object
;;; REF: can be picked up in any of the panes it is visible in. See also the
;;; REF: the go_clear_drag and go_draw_drag methods.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_drag(go_clear_drag(% drag_obj %), go_draw_drag(% drag_obj %),
			drag_obj.go_fixed_image, drag_obj);
enddefine;


define :method go_draw_drag_in_pane( client, drag_obj :go_screen_object );
lvars drag_obj;
;;; REF: go_draw_drag_in_pane( CLIENT_DATA, SCREEN_OBJECT );
;;; REF: Identical to the go_draw_drag method but always moving the reference
;;; REF: point of the bounding box of the screen object. This is needed if
;;; REF: the object hasn't been picked up with the mouse but has been
;;; REF: "picked" up programmatically with the go_drag_in_pane method.
;;; REF: CLIENT_DATA is a structure containing information for dragging.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	0 -> client.startXOffset;
	0 -> client.startYOffset;
	go_draw_drag( client, drag_obj );
enddefine;

define :method go_drag_in_pane( drag_obj :go_screen_object );
lvars drag_obj;
;;; REF: go_drag_in_pane( SCREEN_OBJECT );
;;; REF: "picked" up the screen object programmatically and move it with the
;;; REF: mouse motions until it is dropped . This is an alternative to the
;;; REF: go_drag_and_drop method.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
	go_drag(go_clear_drag(% drag_obj %), go_draw_drag_in_pane(% drag_obj %),
			drag_obj.go_fixed_image, drag_obj);
enddefine;

;;;------------------------------------------------------------------
;;; COPY: tentative definition for copying an object

define :method go_copy_object( scrObj :go_screen_object ) -> new_scrObj;
lvars scrObj;
;;; REF: go_copy_object( SCREEN_OBJECT ) -> NEW_SCREEN_OBJECT;
;;; REF: Creates a new screen object as an exact copy of the given one. This
;;; REF: will result in an object which is visible in the same panes on the
;;; REF: same place but on top of all other objects. This will always "hide"
;;; REF: the original object which will be underneath the new one.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars new_scrObj, parent;
	;;; copies the object and leaves new one on the stack

	;;; Prevent descending into infinite loop (pane go_contains object)...
	procedure;
	dlocal 1 % scrObj.go_visible_in % = [];
		copydata( scrObj ) -> new_scrObj;    ;;; This does not update parents
	endprocedure();

	for parent in go_visible_in( scrObj ) do
		go_add_to( new_scrObj, parent ); ;;; Update the parents
	endfor;
enddefine;

;;;------------------------------------------------------------------
;;; DESTROY: tentative definition for destroying an object

define :method go_destroy_object( scrObj :go_screen_object );
lvars scrObj;
;;; REF: go_destroy_object( SCREEN_OBJECT );
;;; REF: Destroys the object by removing it from all panes it was in. It also
;;; REF: removes any editors involved. Currently it does not make the structure
;;; REF: of the instance completely free.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars eds, pane;
	go_applist( go_remove_from, scrObj );
	applist( scrObj.go_editors, go_destroy_object );
	go_update( scrObj );		;;; ready for garbage collection
enddefine;


/* --- Revision History --------------------------------------------
 * BR 20/09/93
 *     Added the go_move_to() method to ensure redrawing if needed.
 * JJC 08/09/93
 *     Changed object..endobject to instance...endinstance.
 * BR 24/08/93
 *     Renamed go_visible_in_panes() just go_visible_in() because other
 *     objects can be containers. Added method go_applist().
 * BR 13/07/93
 *     Added the go_batch_mode() method and its updater.
 * BR 05/07/93
 *     Added a slot go_editors to avoid the overuse of go_live_object to
 *     also keep the editors within go_make_editable().
 * BR 21/06/93
 *     Added the go_accepts_events() method.
 * BR 26/05/93
 *     Added optimisation for batch mode : a pane in batch mode will
 *     now put each changed object in batch mode too. Therefore at the
 *     end of the panes batch mode the object needs to be redrawn.
 * BR 19/05/93
 *     (See also go_group.p) Added a choice "forbidden" to the
 *     go_visible_now slot of the go_screen_object class which is taken
 *     into account in the methods: go_draw(), go_clear(), go_redraw()
 *     or go_internal_redraw().
 * BR 08/05/93
 *     Moved go_area class into separate file
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 08/04/93
 *     Removed direct call to XpwDrawLine[s]() by go_draw_line[s]();
 *     Removed direct call to XpwDrawArcs() by go_draw_arcs();
 *     Removed direct call to XpwFillArcs() by go_draw_filled_arcs();
 *     Removed direct call to XpwDrawRectangle() by go_draw_rectangle();
 *     Removed direct call to XpwClearArea() by go_clear_rectangle()
 * BR 06/04/93
 *     Renamed the mixin go_editable to go_editor
 *     Removed dummy go_make_editable() and go_make_uneditable() methods
 *     from screen_object CLASS.
 *     Updated go_get_bounding_area() to set go_visible_in_panes slot
 * BR 07/03/93
 *     Added a go_show() method with the pane as second argument.
 * BR 01/02/93
 *     Removed all links to "hilitable" mixin.
 * BR 18/01/93
 *     Optimized go_move_to() and go_centre_to() not to go_clear/go_internal_redraw if position
 *     hasn't changed...
 * BR 10/12/92
 *     Changed check_pane_limits into go_check_drag_limits.
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 24/11/92
 *     Separated out the go_draw() method (see also go_xdraw.p)
 *     and added in the corresponding go_fgdraw() & go_bgdraw()
 * BR 11/11/92
 *     Moved colour definitions in separate mixin: go_colourable go_(colour.p).
 *     Moved go_centre_to methods here (from go_polygon.p)
 * BR 04/11/92
 *     Removed inheritance from UI_object (now obsolete).
 *     Moved UI_object slot go_live_object and method go_copy_object() here with
 *     new additions to make it visible on screen!
 *     Added new method go_destroy_object()...
 * BR 16/10/92
 *     Added argument to go_draw() and go_safe_region() to indicate pane
 *     Removed rc_window ...
 *     Added scaling !!!
 * BR 05/10/92
 *     Split the file into go_object.p and go_screen_object.p because a
 *     screen_object needs to be go_located ...
 *     Resolved conflicts between places where go_draw was used and defined
 *     for the file go_located.p.
 *     Resolved conflicts between places where the_go_components was used and defined
 *     for the file go_screen_object.p.
 *     go_make_visible() and go_make_invisible() now take a pane as first argument.
 *     go_show() needs to be adapted further!!!
 * BR 04/09/92
 *     new definitions of go_internal_redraw() and go_clear() which use a new function
 *     called go_redraw_area() which finds out what to go_internal_redraw below/on go_top.
 *     go_redraw() has been kept for compatibility reasons...
 * BR 28/08/92
 *     moved go_draw in go_screen_object a bit higher (before go_show)
 * BR 24/08/92
 *     moved go_draw from UI_object to go_screen_object- redrawn only at go_screen_object
 * BR 19/08/92
 *     added go_copy_object method (based on copydata) to UI_object
 *     added fixedImage slot to go_screen_object for draggin purposes
 * BR 23/07/92
 *     added go_clear
 * BR 20/07/92
 *     made include file from basic objects: UI_object and go_screen_object,
 *     previously defined in sketch_basics.p
 */
;;; eof
