/* --- Copyright University of Sussex 1996. All rights reserved. ----------
 > File:			C.mips/src/syscomp/asmout.p
 > Purpose:         Output code procedures for MIPS assembler
 > Author:			Robert Duncan, 29 Nov 1989 (based on Sun version) (see revisions)
 */


#_INCLUDE 'common.ph'

#_IF DEFV SYSTEM_V >= 4.0

;;; SVR4 ABI for MIPS requires position-independent code
constant macro PIC = true;

#_ENDIF

;;; -----------------------------------------------------------------------

section $-Popas;

weak global constant procedure (
	identlabel,
);

vars
	nextlabel,
;

define global popint(n);
	lvars n;
	(n<<2) || 3;
enddefine;

define global mcint(n);
	lvars n;
	n >> 2;
enddefine;

define nextlab();
	dlocal pop_pr_radix = 36;
	'L' >< nextlabel;
	nextlabel fi_+ 1 -> nextlabel;
enddefine;

define lconstant asm_label(word, prefix_char);
	lconstant SEP_CHAR = `_`;
	lvars n, len, string, q, r, prefix_char, word;
	if isstring(word) then word else fast_word_string(word) endif -> string;
	datalength(string) -> len;
	cons_with consstring {%
		prefix_char, SEP_CHAR;
		fast_for n to len do
			go_on dup(fast_subscrs(n, string)) to
			;;; 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6
				c c c c c c c c c c c c c c c c		;;; Ctrl-A -- Ctrl-P
				c c c c c c c c c c c c c c c c		;;; Ctrl-Q -- SPACE
				s s s $ s s s s s s s s s s s d		;;;   !    --   0
				d d d d d d d d d s s s s s s s		;;;   1    --   @
				A A A A A A A A A A A A A A A A		;;;   A    --   P
				A A A A A A A A A A s s s s u s		;;;   Q    --   `
				a a a a a a a a a a a a a a a a		;;;   a    --   p
				a a a a a a a a a a s s s s c c		;;;   q    -- \(128)
			else c;

			A: a: d: u :		;;; alpha, digit, underscore
				nextloop;

			$ :
				if n /== len and fast_subscrs(n fi_+ 1, string) == `-`
				and (n /== 1 or len fi_> 2) then
					;;; replace section separator $- with SEP_CHAR
					-> ;			;;; erase $
					unless n == 1 then SEP_CHAR endunless;
					n fi_+ 1 -> n;	;;; skip -
					nextloop
				;;; else drop thru to "s"
				endif;

			s: c:				;;; other sign, control etc
				;;; represent as 3 decimal digits
				() fi_// 100 -> q -> r;
				q fi_+ `0`;
				r fi_// 10 -> q -> r;
				q fi_+ `0`, r fi_+ `0`;

		endfast_for;
	%};
enddefine;

constant procedure (
	asm_symlabel	= asm_label(%`c`%),
	asm_identlabel	= asm_label(%`i`%),
	asm_wordlabel	= asm_label(%`w`%),
	asm_wordidlabel	= asm_label(%`z`%),
	asm_pdpropslabel= asm_label(%`p`%),
	asm_testdeflabel= asm_label(%`t`%),
	);


	;;; list of pop symbol prefixes for use by -Extern_make_base-
	;;; see unixextern.p
global constant macro $- POP_SYMBOL_PREFIXES = [
	['c_' 'xc_' 'uc_' 'xuc_' 'i_' 'w_' 'z_' 'p_' 't_' 'end\^@' '_gp\^@']
];


global constant macro (
	$- ASM_TEXT_STR		= '\t.text',
	$- ASM_DATA_STR		= '\t.data',
	$- ASM_RDATA_STR	= '\t.rdata',
	$- ASM_BYTE_STR		= '\t.byte\t',
	$- ASM_SHORT_STR	= '\t.half\t',
	$- ASM_INT_STR		= '\t.word\t',
	$- ASM_WORD_STR		= ASM_INT_STR,
);

#_IF DEF PIC

;;; asm_alternate_entry:
;;;     alternative entry points for procedures defined within the
;;;     current file (may bypass some entry code)

define asm_alternate_entry =
	newproperty([], 64, false, "tmparg");
enddefine;

#_ENDIF

define asm_startfile(name);
	lvars name;
#_IF DEF PIC
	clearproperty(asm_alternate_entry);
#_ENDIF
#_IF DEF M_DEBUG
	;;; Include definitions of software register names
	asmf_printf(sysfileok('$popsrc/pop_regdef.h'), '\n#include "%p"\n');
#_ENDIF
enddefine;

constant procedure asm_endfile = identfn;

vars asm_current_section = "text";

define asm_startcode();
	;;; "code" is used by popc/poplink to mean "non-writeable"
#_IF DEF PIC
	;;; by default, non-writeable data must go in the read-only data
	;;; segment, with the text segment reserved for instructions only
	outcode('\n'); outcode(ASM_RDATA_STR);
	"rdata" -> asm_current_section;
#_ELSE
	;;; use the text segment for all non-writeable stuff
	outcode('\n'); outcode(ASM_TEXT_STR);
	"text" -> asm_current_section;
#_ENDIF
enddefine;

define asm_startdata();
	outcode('\n'); outcode(ASM_DATA_STR);
	"data" -> asm_current_section;
enddefine;

define asm_outlab(lab);
	lvars lab;
	asmf_charout(lab), asmf_charout(`:`), asmf_charout(`\n`)
enddefine;

define asm_outglab(lab);
	lvars lab;
	asmf_charout('\t.globl\s');
	asmf_charout(lab), asmf_charout(`\n`), asm_outlab(lab);
enddefine;

define asm_outlabset(lab, val);
	lvars lab, val;
	if isstring(val) and locchar(`-`, 1, val) then
		;;; Horrible hack to cope with MIPS assembler's inability to
		;;; handle differences between relocatable labels:
		;;; we have to output a patch record which gets fixed later on
		lvars l = pdtolist(incharitem(stringin(val)));
		if lmember(">>", l) then
			;;; Procedure length definition (?):
			;;; lab = ((l1 - l2) >> 2) + c
			asmf_printf(l(11),l(5),l(3),lab,lab,'%p = 0 ##PATCH## %p %p %p %p\n');
		else
			;;; Simple difference (?)
			;;; lab = l1 - l2
			asmf_printf(l(3),l(1),lab,lab,'%p = 0 ##PATCH## %p %p %p\n');
		endif;
		return;
	endif;
	asmf_printf(val, lab, '%p = %p\n')
enddefine;

define asm_outglabset(lab, val);
	lvars lab, val;
	asmf_printf(val, lab, lab, '\t.globl %p\n%p = %p\n')
enddefine;

;;; outdatum:
;;;		output -n- data values from the stack.
;;;		The MIPS assembler can't cope with sequences of symbols: every
;;;		symbol must have its own directive

define lconstant outdatum(n, string);
	lvars i, m, n, string, first = true, tmp;
	fast_for m from n by -1 to 1 do
		subscr_stack(m) -> i;
		if isstring(i) and (strnumber(i) ->> tmp) then tmp -> i endif;
		if isintegral(i) then
			asmf_charout(if first then string else `,` endif);
			if isinteger(i) or i < 0 then
				asmf_pr(i)
			else
				;;; +ve bigint -- put out as hex to stop "as" complaining
				procedure;
					dlocal pop_pr_radix = 16;
					asmf_printf((), '0x%p')
				endprocedure(i);
			endif;
			false -> first;
		else
			if not(first) then asmf_charout(`\n`) endif;
			asmf_charout(string), asmf_pr(i), asmf_charout(`\n`);
			true -> first;
		endif;
	endfast_for;
	if not(first) then asmf_charout(`\n`) endif;
	erasenum(n);
enddefine;

global constant procedure (
	asm_outbyte		= outdatum(% ASM_BYTE_STR %),
	asm_outshort	= outdatum(% ASM_SHORT_STR %),
	asm_outint		= outdatum(% ASM_INT_STR %),
	asm_outword		= outdatum(% ASM_WORD_STR %),
	);

	/*	-asm_addbits- is used to accumulate a sequence of bitfields into
		a (big)integer which -asm_outbits- is then called to output when the
		total number of bits accumulated occupies an exact number of bytes.
			-val- is the currently accumulated (big)integer, containing
		-nbits- bits so far; -new_val- is the field to be added, containing
		-new_nbits- bits.
			The value returned will be the -val- input for the next field
		(with -nbits- incremented by -new_nbits-), and so on. Other than
		initialising -val- to 0 at the start, nothing is assumed about how
		the fields are added to it, so it is the responsibility of this
		procedure and -asm_outbits- to output the bitfields correctly.
	*/

#_IF DEF BIG_ENDIAN

;;; Add the fields at the bottom and output the bytes from the top down

define asm_addbits(new_val, new_nbits, val, nbits);
	lvars new_val, new_nbits, val, nbits;
	(val << new_nbits) || new_val	;;; add the new field at the bottom
enddefine;

define asm_outbits(val, nbits);
	lvars val, nbits, count = 0;
	lconstant BYTE_MASK = (1<<BYTE_BITS)-1;
	until nbits == 0 do
		nbits-BYTE_BITS -> nbits;
		(val >> nbits) && BYTE_MASK;	;;; output bytes from the top down
		;;; this just saves putting each byte on a separate line
		if (count+1 ->> count) == 12 then asm_outbyte(count), 0 -> count endif;
	enduntil;
	if count /== 0 then asm_outbyte(count) endif
enddefine;

#_ELSE

;;;	Add the fields at the top and output the bytes from the bottom up

define asm_addbits(new_val, new_nbits, val, nbits);
	lvars new_val, new_nbits, val, nbits;
	val || (new_val << nbits) 	;;; add the new field at the top
enddefine;

define asm_outbits(val, nbits);
	lvars val, nbits, count = 0;
	lconstant BYTE_MASK = (1<<BYTE_BITS)-1;
	until nbits == 0 do
		val && BYTE_MASK;			;;; output bytes from the bottom up
		val >> BYTE_BITS -> val;
		nbits-BYTE_BITS -> nbits;
		;;; this just saves putting each byte on a separate line
		if (count+1 ->> count) == 12 then asm_outbyte(count), 0 -> count endif;
	enduntil;
	if count /== 0 then asm_outbyte(count) endif;
enddefine;

#_ENDIF

global constant procedure (
	;;; asm_align_double() -- output alignment for double (e.g. double float)
	;;; the space directive is a dummy to prevent any preceding label being
	;;; aligned as well
	asm_align_double = outcode(%'\t.space\t0\n\t.align\t3'%),
);

;;; asm_out_dfloat:
;;;		output double float

define asm_out_dfloat(hipart, lopart);
	lvars hipart, lopart;
#_IF DEF BIG_ENDIAN
	asm_outword(hipart, lopart, 2);
#_ELSE
	asm_outword(lopart, hipart, 2);
#_ENDIF
enddefine;

	;;; optional procedure used (if present) by poplink
	;;; to do more compact generation of strings using ".ascii"
#_IF not(DEF IRIX)
define asm_quick_genstring(string);
	lvars c, n, l = datalength(string), string;
	returnif(l == 0)(false);	;;; assembler disallows zero length strings
	fast_for n to l do
		f_subs(n, string) -> c;
		if c fi_< `\s` or c fi_> `~` or c == `\\` or c == `"` then
			return(false)
		endif
	endfast_for;
	asmf_charout('\t.ascii "');
	asmf_charout(string);
	l && (WORD_BYTES-1) -> l;
	unless l == 0 then
		repeat WORD_BYTES-l times
			asmf_charout(`\\`), asmf_charout(`0`)
		endrepeat;
	endunless;
	asmf_charout(`"`), asmf_charout(`\n`);
	true
enddefine;
#_ENDIF

	;;; for closures generated by poplink for undef procedures, etc
define asm_gen_poplink_code(outlabs, nfroz, jmplab) -> nwords;
	lvars outlabs, nfroz, jmplab, nwords;
	lconstant a0 = "\$4", t0 = "\$8", t9 = "\$25", usp = "\$30";
	;;; label end of frozvals
	lvars lab = nextlab();
	asm_outlab(lab);
#_IF DEF PIC
	;;; switch to text section for instructions
	asmf_printf('\t.text\n');
	;;; plant exec labels
	outlabs();
	;;; set context pointer to get label
	asmf_printf('\t.set\tnoreorder\n');
	asmf_printf(t9, '\t.cpload\t%p\n');
	asmf_printf('\t.set\treorder\n');
#_ELSE
	;;; plant exec labels
	outlabs();
	asmf_printf('\t.set\tnoreorder\n');
#_ENDIF
	;;; load pointer to first frozval
	asmf_printf(nfroz*WORD_OFFS, lab, a0, '\tla\t%p, %p-%p\n');
	lvars offs = 0;
	repeat nfroz times
		asmf_printf(a0, offs, t0, '\tlw\t%p, %p(%p)\n');
		asmf_printf(WORD_OFFS, usp, '\tsubu\t%p, %p\n');
		asmf_printf(usp, t0, '\tsw\t%p, (%p)\n');
		offs + WORD_OFFS -> offs;
	endrepeat;
	;;; chain pdpart off t9
	asmf_printf(jmplab, t9, '\tla\t%p, %p\n');
	asmf_printf(t9, '\tj\t%p\n');
#_IF DEF PIC
	;;; switch back to previous section
	asmf_printf(asm_current_section, '\t.%p\n');
	;;; no code planted in this section
	0 -> nwords;
#_ELSE
	asmf_printf('\tnop\n');
	asmf_printf('\t.set\treorder\n');
	6 + nfroz*3 -> nwords;
#_ENDIF
enddefine;

/*
 *	Exfunc closure code -- jump to subroutine _exfunc_clos_action (aextern.s)
 *	with exfunc_closure record address in a reg, etc. This procedure is
 *	passed the label of _exfunc_clos_action. The code generated must be
 *	padded to exactly 5 words (see "external.ph").
 */

define asm_gen_exfunc_clos_code(action_lab);
	lvars action_lab;
#_IF DEF PIC
	;;; Referencing _exfunc_clos_action as an absolute address is only
	;;; possible in a data section, but since the MIPS assembler doesn't
	;;; allow code in data sections, we have to resort to machine code!
	;;; We can assume that t9 points to the first instruction (i.e., the
	;;; closure address).
	asmf_printf('\t.word\t0x8f2f0010\t#\tlw\tcr, 16(t9)\n');
	asmf_printf('\t.word\t0\t\t#\tnop\n');
	asmf_printf('\t.word\t0x01e00008\t#\tjr\tcr\n');
	asmf_printf('\t.word\t0\t\t#\tnop\n');
	asmf_printf(action_lab, '\t.word\t%p\n');
#_ELSE
	asmf_printf('\t.set\tnoreorder\n');
	asmf_printf(action_lab, '\tla\t$15, %p\n');
	asmf_printf('\tjal\t$25, $15\n');		;;; save return address in t9
	asmf_printf('\tsubu\t$25, 16\n');		;;; offset back to closure start
	asmf_printf('\tnop\n');
	asmf_printf('\t.set\treorder\n');
#_ENDIF
enddefine;


/*
 *	extern_name_translate:
 *		translate an external symbol, using the same conventions as the
 *		C compiler and linker (e.g. add a leading '_', truncate to N
 *		characters, etc.)
 */

define global extern_name_translate(lang, symbol, type) -> symbol;
	lvars lang, symbol, type;
	if lang = 'FORTRAN' then uppertolower(symbol) <> '_' -> symbol endif
enddefine;


	/*	String for first line of Unix "ld" command -- used in os_comms.p
		(Image name is in the environment variable "IM")
	*/
constant
	unix_ld_command_header =
		;;; Links the data segment start at 0x800000. This brings it close
		;;; enough to the text segment (0x400000) to allow jumps from one
		;;; to the other; external load won't work otherwise. It also
		;;; improves the efficiency of absolute calls planted in run-time
		;;; code.
#_IF DEF ULTRIX
		'/bin/ld -x -o $IM -e __start -D 800000 \\\n'
#_ELSEIF DEFV IRIX >= 6.0
		'/usr/bin/ld -x -32 -o $IM -e __start -rdata_shared \\\n'
#_ELSEIF DEFV IRIX >= 5.0
		/*
		'/usr/bin/ld -x -o $IM -e __start -rdata_shared \\\n'

		TEMPORARY hack to place C functions from libpop at the start of the
		image -- necessary to get the symbol types right
		*/
		'/usr/bin/ld -x -o $IM -e __start -rdata_shared -all $popexternlib/libpop.a -none \\\n'
#_ELSEIF DEFV IRIX < 5.0
		'/usr/bin/ld -x -o $IM -e __start -D 800000 \\\n'
#_ELSE_ERROR
#_ENDIF
	;

	/*	Location of crt*.o files used by "ld"
	*/
constant
	unix_ld_crt_objects =
#_IF DEF ULTRIX
		'/usr/lib/cmplrs/cc/crt0.o'
#_ELSEIF DEF IRIX
		'/usr/lib/crt[1n].o'
#_ELSE_ERROR
#_ENDIF
	;

	/*	Assembler command line arguments
	*/
constant unix_as_options =
	#_IF DEFV IRIX >= 6.0
		['-32' '-KPIC' '-G' '0']
	#_ELSEIF DEF PIC
		['-KPIC' '-G' '0']
	#_ELSE
		['-G' '0']
	#_ENDIF
;

	/*	This causes poplink to use command file

			$popsrc/poplink___________xlink-type_xlibs

		to expand X libraries as object files (but only when given
		-x_complete as argument from pglink).
	*/
#_IF not(DEF PIC)
constant macro X_COMPLETE_ENABLED = true;
#_ENDIF

endsection;		/* $-Popas */


/* --- Revision History ---------------------------------------------------
--- John Gibson, Aug 15 1996
		New version of extern_name_translate.
--- Julian Clinton, 21 Jun 1996
		Added '-32' options for IRIX >= 6.0.
--- Robert John Duncan, May 23 1995
		Installed temporary(?) patch to unix_ld_command_header for IRIX 5.2.
		A bug in the linker means that forward references to external
		routines used as signal handlers, etc., are unsafe; solved by
		loading all of $popexternlib/libpop.a first, at the cost of being
		unable to link an executable without X (because XtPoplog.o is always
		included).
--- John Gibson, Apr  3 1995
		Added asm_outint
--- Robert John Duncan, Mar  3 1995
		Removed cases for RIS*COS
--- Robert John Duncan, Nov  2 1994
		Added -G 0 option to the assembler for all systems (i.e. so as never
		to use the global pointer) and so removed all reference to the small
		data section
--- Robert John Duncan, Mar 29 1994
		Added _____________-rdata_shared option to link command for IRIX 5 (it doesn't
		work, but is there in case it ever does)
--- Robert John Duncan, Mar 22 1994
		Changed asm_gen_poplink_code to plant the execute labels and (once
		again) not to assume its own address passed in a0.
		Added unix_as_options for generating position-independent code.
--- Robert John Duncan, Mar 16 1994
		Changed ld command header for IRIX 5+: can cope without the data
		segment being relocated. No need for a complete X link either.
--- Robert John Duncan, Mar 15 1994
		Changes for position-independent code (made the default for SVR4).
--- Robert John Duncan, Mar 10 1994
		Changes to support use of register t8 as special var block pointer:
		this removes the dependency on the global pointer so that extern
		declarations for special vars aren't needed any more, and the
		external closure code template has to use chain reg rather than t8.
--- Robert John Duncan, Mar  7 1994
		Changed external closure code template to use register t8 rather
		than t9.
--- Robert John Duncan, Jun 11 1993
		Added unix_ld_crt_objects (used by "os_comms.p")
--- John Gibson, May 19 1993
		Added asm_gen_exfunc_clos_code to generate template code previously
		in aextern.s
--- John Gibson, May 14 1993
		Added X_COMPLETE_ENABLED
--- Robert John Duncan, May  5 1993
		Disabled asm_quick_genstring for Irix because the .ascii directive
		is broken in the 3.12 release of the development tools
--- John Gibson, Oct 15 1992
		Changed outdatum to (a) convert string integers to integers, and
		(b) put out +ve bigints in hex.
--- Robert John Duncan, Jul 27 1992
		Added -extern_name_translate-
--- Robert John Duncan, Jun 24 1991
		Added definitions for SG IRIX
--- Robert John Duncan, Apr 17 1991
		Changed to use hardware register names except when M_DEBUG is <true>
		(reduces dependency on "pop_regdef.h")
--- Robert John Duncan, Nov 13 1990
		Changed "note" to "weak".
--- Robert John Duncan, Jul  5 1990
		Changed -asm_startfile- to output "extern" declarations for
		everything in the special var block
--- Rob Duncan, Jun 18 1990
		Added -asm_starts*data- for switching to the "small" data section.
 */
