/* --- Copyright University of Sussex 1994. All rights reserved. ----------
 > File:            C.mips/src/ass.p
 > Purpose:         MIPS run-time code generator
 > Author:          Rob Duncan and Simon Nichols, Dec  4 1989 (see revisions)
 */


#_INCLUDE 'declare.ph'
#_INCLUDE 'vmdefs.ph'
#_INCLUDE 'external.ph'


global constant

	$-Sys$- _data_seg_start,

	;;; Assembly code subroutines referenced by user procedures

	_popenter,
	_popuenter,
	_popuncenter,
	_checkall,
	_bfield,
	_sbfield,
	_ubfield,
	_setstklen,
	_setstklen_diff,

;

global vars
	pop_debugging,		;;; <false> if optimisation pass wanted
	_trap,				;;; interrupt flag
;

section $-Sys$-Vm;

constant procedure (

	;;; VM interface

	Code_pass,
	Drop_I_code,
	Get_procedure,
	Is_register,
	Trans_structure,
);

vars
	asm_clist,			;;; list of I-code to be assembled
	asm_instr,			;;; the current I-code instruction
	asm_struct_list,	;;; list of items to go in the structure table
	_asm_pass,			;;; assembly pass counter - false when dropping code
	_asm_drop_ptr,		;;; pointer to drop code at
	_asm_code_offset,	;;; offset into executable code (in bytes)
	_Nreg,				;;; number of registers
	_Nlocals,			;;; number of dynamic locals
	_Npopstkvars,		;;; number of pop on-stack lvars
	_Nstkvars,			;;; total number of on-stack lvars
;

endsection;

lvars
	_fill_delay,		;;; automatic filling of branch delay slots
	_delay_slot,		;;; integer: register loaded in last instruction
						;;; <true>: last instruction is in branch delay slot
	_last_nop,			;;; pointer to last NOP
	_usp_offs,			;;; user stack adjustment
	_framesize,			;;; stack frame size in bytes
	_regmask,			;;; mask for register save/restore
	_strsize,			;;; structure table size in bytes
;


;;; -----------------------------------------------------------------------

section $-Sys$-Vm;

;;; R2000 Instruction Opcode Bit Encoding

define lconstant macro OPCODE op;
	lvars op;
	[_int(^op << 26)].dl;
enddefine;

define lconstant macro SPECIAL op;
	lvars op;
	[_int(^op)].dl;
enddefine;

define lconstant macro BCOND op;
	lvars op;
	[_int((8:01 << 26) || (^op << 16))].dl;
enddefine;

lconstant

	_J		= OPCODE 8:02,
	_JAL	= OPCODE 8:03,
	_BEQ	= OPCODE 8:04,
	_BNE	= OPCODE 8:05,
	_BLEZ	= OPCODE 8:06,
	_BGTZ	= OPCODE 8:07,
	_ADDIU	= OPCODE 8:11,
	_SLTI	= OPCODE 8:12,
	_SLTIU	= OPCODE 8:13,
	_LUI	= OPCODE 8:17,
	_LB		= OPCODE 8:40,
	_LH		= OPCODE 8:41,
	_LW		= OPCODE 8:43,
	_LBU	= OPCODE 8:44,
	_LHU	= OPCODE 8:45,
	_SB		= OPCODE 8:50,
	_SH		= OPCODE 8:51,
	_SW		= OPCODE 8:53,

	_SLL	= SPECIAL 8:00,
	_SRA	= SPECIAL 8:03,
	_JR		= SPECIAL 8:10,
	_JALR	= SPECIAL 8:11,
	_MFLO	= SPECIAL 8:22,
	_MULT	= SPECIAL 8:30,
	_ADDU	= SPECIAL 8:41,
	_SUBU	= SPECIAL 8:43,
	_OR		= SPECIAL 8:45,
	_SLT	= SPECIAL 8:52,

	_BLTZ	= BCOND 8:00,
	_BGEZ	= BCOND 8:01,

	_NOP	= _0,	;;; sll zero, zero, 0
;

;;; Registers

lconstant

	_ZERO			= _00,
	_AT				= _01,
	_RESULT_REG_0	= _02,
	_ARG_REG_0		= _04,
	_ARG_REG_1		= _05,
	_ARG_REG_2		= _06,
	_ARG_REG_3		= _07,
	_TMP_REG_0		= _08,
	_TMP_REG_1		= _09,
	_TMP_REG_2		= _10,
	_TMP_REG_3		= _11,
	_TMP_SRC_ADDR	= _13,
	_TMP_DST_ADDR	= _14,
	_CHAIN_REG		= _15,
	_POP_REG_0		= _16,
	_POP_REG_1		= _17,
	_NPOP_REG_0		= _18,
	_NPOP_REG_1		= _19,
	_NPOP_REG_2		= _20,
	_NPOP_REG_3		= _21,
	_FALSE			= _22,
	_PB				= _23,
	_SVB_REG		= _24,
	_CALL_REG		= _25,
	_GP				= _28,
	_SP				= _29,
	_USP			= _30,
	_RA				= _31,
;

;;; Register identifiers used by the VM

protected register constant

	arg_reg_0		= _pint(_ARG_REG_0)  << 1,
	arg_reg_1		= _pint(_ARG_REG_1)  << 1,
	arg_reg_2		= _pint(_ARG_REG_2)  << 1,

	pop_reg_0		= _pint(_POP_REG_0)  << 1 || 1,
	pop_reg_1		= _pint(_POP_REG_1)  << 1 || 1,
	nonpop_reg_0	= _pint(_NPOP_REG_0) << 1,
	nonpop_reg_1	= _pint(_NPOP_REG_1) << 1,
	nonpop_reg_2	= _pint(_NPOP_REG_2) << 1,
	nonpop_reg_3	= _pint(_NPOP_REG_3) << 1,

	chain_reg		= _pint(_CHAIN_REG)  << 1,

;

#_IF DEF PIC

;;; Every procedure should save the context pointer (_GP)
protected register constant context_ptr = _pint(_GP) << 1;
lconstant nonpop_always_save = [% ident context_ptr %];

#_ELSE

lconstant nonpop_always_save = [];

#_ENDIF

;;; Register lvars

constant
	asm_pop_registers	= [%
		[],
		ident pop_reg_0,
		ident pop_reg_1,
	%],
	asm_nonpop_registers= [%
		nonpop_always_save,
		ident nonpop_reg_0,
		ident nonpop_reg_1,
		ident nonpop_reg_2,
		ident nonpop_reg_3,
	%],
;

;;; Is_address_reg:
;;;		all registers are address registers

define Is_address_reg() with_nargs 1;
	Is_register();
enddefine;


;;; -- Instruction Planting -----------------------------------------------

lconstant procedure (	;;; forward
	Drop_uadjust,
	Drop_loadi,
);


;;; I_type, R_type, J_type:
;;;		instruction types

define lconstant I_type(_op, _rs, _rt, _imm);
	lvars _op, _rs, _rt, _imm;
	_op _biset _shift(_rs, _21) _biset _shift(_rt, _16)
		_biset (_imm _bimask _16:FFFF);
enddefine;

define lconstant R_type(_op, _rs, _rt, _rd);
	lvars _op, _rs, _rt, _rd;
	_op _biset _shift(_rs, _21) _biset _shift(_rt, _16)
		_biset _shift(_rd, _11);
enddefine;

define lconstant J_type(_op, _target);
	lvars _op, _target;
	_op _biset _shift(_target _bimask _16:FFFFFFF, _-2);
enddefine;


;;; Drop:
;;; 	plant a single word in the instruction stream

define lconstant Drop(_word);
	lvars _word;
	unless _asm_pass then
		;;; only output data on the last pass
		_word -> _asm_drop_ptr!(w)++ -> _asm_drop_ptr;
	endunless;
	@@(w){_asm_code_offset}++ -> _asm_code_offset;
enddefine;

constant procedure Drop_l = Drop;		;;; for external references

;;;	unDrop:
;;;		removes the last word planted in the instruction stream

define lconstant unDrop() -> _word;
	lvars _word = _0;
	--@@(w){_asm_code_offset} -> _asm_code_offset;
	unless _asm_pass then
		_asm_drop_ptr--!(w) -> _asm_drop_ptr -> _word;
	endunless;
enddefine;

;;; Drop_nop:
;;;		plant a NOP instruction

define lconstant Drop_nop();
	_asm_drop_ptr -> _last_nop;
	Drop(_NOP);
	false -> _delay_slot;
enddefine;

;;; Patch_nop:
;;;		overwrite the last NOP planted with -_word-.

define lconstant Patch_nop(_word);
	lvars _word;
	unless _asm_pass then
		_word -> _last_nop!(w);
	endunless;
	false -> _last_nop;
enddefine;

;;; Drop_lui:
;;;		plant a LUI instruction

define lconstant Drop_lui(_reg, _imm, _patch) -> _reg;
	lvars _reg, _imm, _patch;
	if _patch and _last_nop then
		;;; overwrite last NOP
		Patch_nop(I_type(_LUI, _ZERO, _AT ->> _reg, _imm));
	else
		if _reg == _delay_slot then Drop_nop() endif;
		Drop(I_type(_LUI, _ZERO, _reg, _imm));
		false -> _delay_slot;
	endif;
enddefine;

;;; Drop_load:
;;;		plant a load instruction

define lconstant Drop_load(_op, _dst, _src, _offs);
	lvars _op, _dst, _src, _offs;
	if _src == _delay_slot or _dst == _delay_slot then Drop_nop() endif;
	Drop(I_type(_op, _src, _dst, _offs));
	_dst -> _delay_slot;
enddefine;

;;; Drop_store:
;;;		plant a store instruction

define lconstant Drop_store(_op, _src, _dst, _offs);
	lvars _op, _src, _dst, _offs;
	if _src == _delay_slot or _dst == _delay_slot then Drop_nop() endif;
	Drop(I_type(_op, _dst, _src, _offs));
	false -> _delay_slot;
enddefine;

;;; Drop_arith:
;;;		plant an arithmetic instruction

define lconstant Drop_arith(_op, _dst, _src1, _src2);
	lvars _op, _dst, _src1, _src2;
	if _src1 == _delay_slot or _src2 == _delay_slot or _dst == _delay_slot
	then
		Drop_nop();
	endif;
	Drop(R_type(_op, _src1, _src2, _dst));
	false -> _delay_slot;
enddefine;

;;; Drop_arithi:
;;;		plant arithmetic on an immediate value

define lconstant Drop_arithi(_op, _dst, _src, _imm);
	lvars _op, _dst, _src, _imm;
	if _src == _delay_slot or _dst == _delay_slot then Drop_nop() endif;
	Drop(I_type(_op, _src, _dst, _imm));
	false -> _delay_slot;
enddefine;

;;; Drop_shifti:
;;;		plant a shift by an immediate quantity

define lconstant Drop_shifti(_op, _dst, _src, _shamt);
	lvars _op, _dst, _src, _shamt;
	if _src == _delay_slot or _dst == _delay_slot then Drop_nop() endif;
	Drop(R_type(_op, _ZERO, _src, _dst) _biset _shift(_shamt, _6));
	false -> _delay_slot;
enddefine;

;;; Drop_cond_branch:
;;;		plants a conditional relative branch.
;;;		Returns a pointer to the offset part of the instruction to allow
;;;		patching later.

define lconstant Drop_cond_branch(_op, _src1, _src2, _offs) -> _offs_ptr;
	lvars _op, _src1, _src2, _offs, _offs_ptr;
	if _src1 == _delay_slot or _src2 == _delay_slot then
		Drop_nop();
	endif;
#_IF DEF BIG_ENDIAN
	_asm_drop_ptr@(s)++ -> _offs_ptr;
#_ELSE
	_asm_drop_ptr -> _offs_ptr;
#_ENDIF
	Drop(I_type(_op, _src1, _src2,
		_shift(_offs _sub _asm_code_offset _sub _4, _-2)));
	if _fill_delay then
		if _zero(_usp_offs) then Drop_nop() else Drop_uadjust() endif;
	endif;
	true -> _delay_slot;
enddefine;

;;; Drop_branch:
;;;		plants an unconditional relative branch.

define lconstant Drop_branch(_offs);
	lvars _offs, _instr = false;
	if _nonzero(_usp_offs) then
		Drop_uadjust();
	endif;
	if _fill_delay and not(_delay_slot) then
		;;; safe to move last instruction
		unDrop() -> _instr;
	endif;
	Drop(I_type(_BEQ, _ZERO, _ZERO,
		_shift(_offs _sub _asm_code_offset _sub _4, _-2)));
	if _fill_delay then
		if _instr then Drop(_instr) else Drop_nop() endif;
	endif;
	false -> _last_nop;
	true -> _delay_slot;
enddefine;

;;; Patch_branch:
;;;		backpatch a conditional branch offset

define lconstant Patch_branch(_save_ptr);
	lvars _save_ptr;
	unless _asm_pass then
#_IF DEF BIG_ENDIAN
		##(w){_asm_drop_ptr@(s)++, _save_ptr} _sub _1 -> _save_ptr!(s);
#_ELSE
		##(w){_asm_drop_ptr, _save_ptr} _sub _1 -> _save_ptr!(s);
#_ENDIF
	endunless;
enddefine;

;;; Drop_jr:
;;;		plant a jump (op == JR) or call (op == JALR) to an address in a
;;;		register

define lconstant Drop_jr(_op, _reg);
	lvars _op, _reg;
	if _reg == _delay_slot then Drop_nop() endif;
	Drop(R_type(_op, _reg, _ZERO, if _op == _JR then _ZERO else _RA endif));
	if _fill_delay then
		if _zero(_usp_offs) then Drop_nop() else Drop_uadjust() endif;
	endif;
	false -> _last_nop;
	true -> _delay_slot;
enddefine;

;;;	Drop_j:
;;;		plant a jump (op == J) or call (op == JAL) to an arbitrary address.
;;; 	This must set CALL_REG ($t9) with the target address to satisfy
;;;		calling conventions. In the general case, the jump goes off the
;;; 	register, but if the target is within the data segment, this can
;;; 	be replaced by a straight jump.

define lconstant Drop_j(_op, _target);
	lvars _op, _target;

		;;; test whether ______addr is within the data segment: i.e. has the
		;;; same 4 top bits as the data seg start address
	define lconstant Is_data_address(_addr);
		lvars _addr;
		_shift(_addr, _-28) == _shift(_data_seg_start, _-28);
	enddefine;

	Drop_loadi(_CALL_REG, _target);
	if Is_data_address(_target) then
		lvars _instr = false;
		Drop_uadjust();
		if _fill_delay then
			unDrop() -> _instr;
		endif;
		Drop(J_type(_op, _target));
		if _fill_delay then
			if _instr then Drop(_instr) else Drop_nop() endif;
		endif;
		false -> _last_nop;
		true -> _delay_slot;
	else
		Drop_jr(if _op == _J then _JR else _JALR endif, _CALL_REG);
	endif;
enddefine;

;;; Drop_uadjust:
;;;		adjust the user stack pointer by usp_offs

define lconstant Drop_uadjust();
	if _nonzero(_usp_offs) then
		Drop_arithi(_ADDIU, _USP, _USP, _usp_offs);
		_0 -> _usp_offs;
	endif;
enddefine;


;;; -- General Operands ---------------------------------------------------

;;; isimm16, hi16, lo16:
;;;		separate immediate values into 16-bit quantities

define isimm16(_imm);
	lvars _imm;
	_-16:8000 _slteq _imm and _imm _slt _16:8000;
enddefine;

define hi16(_imm);
	lvars _imm;
	_shift(_imm, _-16);
	;;; adjust for sign-extension of lower half
	if _imm _bitst _16:8000 then () _add _1 endif;
enddefine;

define lo16(_imm);
	lvars _imm;
	_imm _bimask _16:FFFF;
enddefine;


;;;	An "operand" is a pair of values with the following interpretations:
;;;		(reg, false)	register:	reg
;;;		(false, imm)	immediate:	imm
;;;		(reg, offs)		based: 		offs(reg)

define lconstant Get_upush();
	--@@(w){_usp_offs} -> _usp_offs;
	(_USP, _usp_offs);
enddefine;

define lconstant Get_upop();
	(_USP, _usp_offs);
	@@(w){_usp_offs}++ -> _usp_offs;
enddefine;

define lconstant Get_utop();
	(_USP, _usp_offs);
enddefine;

define lconstant Get_usub(_i);
	lvars _i;
	(_USP, _usp_offs _add @@(w)[_i]);
enddefine;

define lconstant Get_imm(_imm);
	lvars _imm;
	if _zero(_imm) then
		(_ZERO, false);
	elseif _imm == false then
		(_FALSE, false);
	else
		(false, _imm);
	endif;
enddefine;

define lconstant Get_abs(_addr, _tmp_reg);
	lvars _addr, _tmp_reg;
	if isimm16(_addr _sub _special_var_block) then
		;;; SVB-relative
		(_SVB_REG, _addr _sub _special_var_block);
	else
		(Drop_lui(_tmp_reg, hi16(_addr), false), lo16(_addr));
	endif;
enddefine;

define lconstant Get_based(_reg, _offs, _tmp_reg);
	lvars _reg, _offs, _tmp_reg;
	if isimm16(_offs) then
		(_reg, _offs);
	else
		Drop_arith(_ADDU,
			_tmp_reg,
			Drop_lui(_tmp_reg, hi16(_offs), true),
			_reg);
		(_tmp_reg, lo16(_offs));
	endif;
enddefine;

define lconstant Get_deferred(_reg, _offs, _tmp_reg);
	lvars _reg, _offs, _tmp_reg;
	Drop_load(_LW, _tmp_reg, _reg, _offs);
	(_tmp_reg, @@ID_VALOF);
enddefine;

define lconstant Get_opnd(_arg, defer, _tmp_reg);
	lvars opnd, defer, _arg, _reg, _tmp_reg;
	asm_instr!INST_ARGS[_arg] -> opnd;
	if iscompound(opnd) and opnd >=@(w) _system_end then
		;;; replace operand with offset or register ident on pass 1
		Trans_structure(opnd) ->> opnd -> asm_instr!INST_ARGS[_arg];
	endif;
	if issimple(opnd) then
		_int(opnd) -> opnd;
		if _neg(opnd) then
			;;; On-stack lvar:
			;;; -opnd- is the offset, shifted left by 1 and negated.
			;;; The bottom bit indicates whether access is via a ref,
			;;; requiring a double indirection if -defer- is <true>
			unless opnd _bitst _1 then false -> defer endunless;
			;;; Get the positive, unshifted offset
			_negate(_shift(opnd, _-1)) -> opnd;
			Get_based(_SP, opnd, _tmp_reg);
		else
			;;; Entry in literal table:
			;;; -opnd- is the offset from PB
			Get_based(_PB, @@PD_TABLE{opnd}, _tmp_reg);
		endif;
		;;; Extra indirection needed if -defer- is <true>
		if defer then Get_deferred((), _tmp_reg) endif;
	elseif Is_register(opnd) ->> _reg then
		unless defer then
			mishap(0, 'SYSTEM ERROR IN Get_opnd (register used as immediate operand)');
		endunless;
		(_int(_reg), false);
	elseif defer then
		;;; Absolute
		Get_abs(opnd, _tmp_reg);
	else
		;;; Immediate
		Get_imm(opnd);
	endif;
enddefine;


;;; -- Move Instructions --------------------------------------------------

lconstant
	DEFER	= true,
	DIRECT	= false,
;

;;; Drop_loadi:
;;;		load an immediate value to a register

define lconstant Drop_loadi(_reg, _imm);
	lvars _reg, _imm;
	Drop_arithi(_ADDIU,
		_reg,
		if isimm16(_imm) then
			_ZERO, lo16(_imm)
		elseif isimm16(_imm _sub false) then
			;;; use the FALSE register
			_FALSE, _imm _sub false
		else
			Drop_lui(_reg, hi16(_imm), true), lo16(_imm)
		endif);
enddefine;

;;; Drop_move:
;;;		move any -src- to any -dst-

define lconstant Drop_move(_src_reg, _src_offs, _dst_reg, _dst_offs);
	lvars _src_reg, _src_offs, _dst_reg, _dst_offs;
	if _dst_offs then
		;;; to memory
		if _src_reg and _src_offs then
			;;; from memory
			Drop_load(_LW, _TMP_REG_0, _src_reg, _src_offs);
			Drop_store(_SW, _TMP_REG_0, _dst_reg, _dst_offs);
		else
			unless _src_reg then
				;;; immediate
				Drop_loadi(_TMP_REG_0 ->> _src_reg, _src_offs);
			endunless;
			;;; from register
			Drop_store(_SW, _src_reg, _dst_reg, _dst_offs);
		endif;
	else
		;;; to register
		if _src_reg then
			if _src_offs then
				;;; from memory
				Drop_load(_LW, _dst_reg, _src_reg, _src_offs);
			elseif _src_reg /== _dst_reg then
				;;; from register
				Drop_arith(_OR, _dst_reg, _src_reg, _ZERO);
			endif;
		else
			;;; immediate
			Drop_loadi(_dst_reg, _src_offs);
		endif;
	endif;
enddefine;

;;; Get_src, Get_dst:
;;;		get source/destination operand of a MOVE instruction

define lconstant Get_src(defer);
	lvars defer;
	Get_opnd(_0, defer, _TMP_SRC_ADDR);
enddefine;

define lconstant Get_dst();
	if asm_instr!V_LENGTH == _2 then
		;;; destination is a stack push
		Get_upush();
	else
		Get_opnd(_1, DEFER, _TMP_DST_ADDR);
	endif;
enddefine;

;;; MOVE instructions:

define I_POP();
	Drop_move(Get_upop(), Get_opnd(_0, DEFER, _TMP_DST_ADDR));
enddefine;

define I_POPQ();
	Drop_move(Get_upop(), Get_opnd(_0, DIRECT, _TMP_DST_ADDR));
enddefine;

define I_STORE();
	Drop_move(Get_utop(), Get_opnd(_0, DEFER, _TMP_DST_ADDR));
enddefine;

define I_MOVE();
	Drop_move(Get_src(DEFER), Get_dst());
enddefine;

define I_MOVEQ();
	Drop_move(Get_src(DIRECT), Get_dst());
enddefine;

define I_MOVES();
	Drop_move(Get_utop(), Get_upush());
enddefine;

define I_MOVENUM();
	Drop_move(Get_imm(asm_instr!INST_ARGS[_0]), Get_dst());
enddefine;

define I_MOVEADDR();
	Drop_move(Get_imm(asm_instr!INST_ARGS[_0]), Get_dst());
enddefine;

;;; Move to/from return address slot in stack frame
;;; (same as I_MOVE or I_MOVEADDR)
define I_MOVE_CALLER_RETURN();
	fast_chain(asm_instr!INST_ARGS[_2]);		;;; I_MOVE or I_MOVEADDR
enddefine;

define I_PUSH_UINT();
	Drop_move(Get_imm(Pint_->_uint(asm_instr!INST_ARGS[_0],_-1)),Get_upush());
enddefine;

define I_ERASE();
	lvars _reg, _offs;
	Get_upop() -> (_reg, _offs);
	if _nonneg(_offs) then
		;;; may be an underflow: drop a dummy load to reg 0
		Drop_load(_LW, _ZERO, _reg, _offs);
		;;; don't want a delay slot
		false -> _delay_slot;
	endif;
enddefine;

define I_SWAP();
	lvars _reg1, _offs1, _reg2, _offs2;
	Get_based(Get_usub(_int(asm_instr!INST_ARGS[_0])),_TMP_SRC_ADDR) -> _offs1 -> _reg1;
	Get_based(Get_usub(_int(asm_instr!INST_ARGS[_1])),_TMP_DST_ADDR) -> _offs2 -> _reg2;
	Drop_load(_LW, _TMP_REG_0, _reg1, _offs1);
	Drop_load(_LW, _TMP_REG_1, _reg2, _offs2);
	Drop_store(_SW, _TMP_REG_0, _reg2, _offs2);
	Drop_store(_SW, _TMP_REG_1, _reg1, _offs1);
enddefine;


;;; -- Field Access Instructions ------------------------------------------

lconstant procedure (

	field_push_op =
		list_assoc_val(%[%
			t_BYTE,				_LBU,
			t_SHORT,			_LHU,
			t_WORD,				_LW,
			t_BYTE ||t_SIGNED,	_LB,
			t_SHORT||t_SIGNED,	_LH,
			t_WORD ||t_SIGNED,	_LW,
		%]%),

	field_pop_op =
		list_assoc_val(%[%
			t_BYTE,				_SB,
			t_SHORT,			_SH,
			t_WORD,				_SW,
			t_BYTE ||t_SIGNED,	_SB,
			t_SHORT||t_SIGNED,	_SH,
			t_WORD ||t_SIGNED,	_SW,
		%]%),

	field_shamt =
		list_assoc_val(%[%
			t_BYTE,				-2,
			t_SHORT,			-1,
			t_WORD,				 0,
			t_DOUBLE,			 1,
		%]%),

	field_offs =
		list_assoc_val(%[%
			t_BYTE,				-1,
			t_SHORT,			-3,
			t_WORD,				-7,
			t_DOUBLE,		   -14,
		%]%),
);

;;; Drop_mulq:
;;;		multiply register _src by integer _n > 1; result in _dst /== _src.
;;;		Uses shift, add and subtract rather than multiply. There must be
;;;		some limit to the usefulness of this, but it's never used here
;;;		for _n > 32.

define lconstant Drop_mulq(_dst, _src, _n);
	lvars _s = _0, _dst, _src, _n;
	until _n _bitst _1 do
		_s _add _1 -> _s;
		_shift(_n, _-1) -> _n;
	enduntil;
	if _n _bitst _2 then
		Drop_mulq(_dst, _src, _n _add _1);
		Drop_arith(_SUBU, _dst, _dst, _src);
	elseif _n /== _1 then
		Drop_mulq(_dst, _src, _n _sub _1);
		Drop_arith(_ADDU, _dst, _dst, _src);
	endif;
	if _nonzero(_s) then
		Drop_shifti(_SLL, _dst, if _n == _1 then _src else _dst endif, _s);
	endif;
enddefine;

;;;	Drop_vecsub_mult:
;;;		convert a vector index to an offset.

define lconstant Drop_vecsub_mult(_dst_reg, _src_reg, _size, _tmp_reg);
	lvars _dst_reg, _src_reg, _size, _tmp_reg;
	;;; convert popint to sysint and multiply by field size
	if _size == _1 then
		Drop_shifti(_SRA, _dst_reg, _src_reg, _2);
	elseif _size _lteq _32 then
		Drop_shifti(_SRA, _tmp_reg, _src_reg, _2);
		Drop_mulq(_dst_reg, _tmp_reg, _size);
	else
		Drop_loadi(_tmp_reg, _size);
		Drop_shifti(_SRA, _dst_reg, _src_reg, _2);
		Drop_arith(_MULT, _ZERO, _dst_reg, _tmp_reg);
		Drop_arith(_MFLO, _dst_reg, _ZERO, _ZERO);
	endif;
enddefine;

;;;	Drop_exptr_deref:
;;;		deref an external pointer -n- times

define lconstant Drop_exptr_deref(n, _dst_reg, _src_reg) -> _src_reg;
	lvars n, _dst_reg, _src_reg;
	fast_repeat n times
		Drop_load(_LW, _dst_reg, _src_reg, _0);
		_dst_reg -> _src_reg;
	endrepeat;
enddefine;

define lconstant Do_bitfield(base, offs, _size, exptr, upd);
	lvars base, offs, exptr, upd, _size, _reg, _ireg = false;
	_int(_size) -> _size;
	;;; Get structure base address in ARG_REG_0
	if not(base) then
		;;; structure is on top of stack
		Drop_load(_LW, _ARG_REG_0, Get_upop());
	elseif Is_register(Trans_structure(base)) ->> _reg then
		;;; structure is already in a register lvar
		Drop_arith(_OR, _ARG_REG_0, _int(_reg), _ZERO);
	else
		mishap(0, 'SYSTEM ERROR IN Do_bitfield (bad structure)');
	endif;
	;;; Interpret offset:
	if isinteger(offs) then
		;;; Constant offset for record-type structure
		Drop_loadi(_ARG_REG_1, _int(offs));
	elseif not(offs) then
		;;; Vector index on stack
		Drop_load(_LW, _ARG_REG_1 ->> _ireg, Get_upop());
	elseif Is_register(Trans_structure(offs)) ->> _ireg then
		;;; Vector index in a register lvar
		_int(_ireg) -> _ireg;
	else
		mishap(0, 'SYSTEM ERROR IN Do_bitfield (bad offset)');
	endif;
	;;; Get field size in ARG_REG_2, new value in ARG_REG_3 (if updating)
	Drop_arithi(_ADDIU, _ARG_REG_2, _ZERO, _size);
	if upd then Drop_load(_LW, _ARG_REG_3, Get_upop()) endif;
	if _ireg then
		Drop_vecsub_mult(_ARG_REG_1, _ireg, _size, _TMP_REG_1);
		Drop_arithi(_ADDIU, _ARG_REG_1, _ARG_REG_1,
			##(1){@@V_BYTES|b} _sub _size);
	endif;
	;;; If it's an external structure, get the real address
	if exptr then
		Drop_exptr_deref(exptr, _ARG_REG_0, _ARG_REG_0) -> ;
	endif;
enddefine;

define lconstant Do_field(base, offs, _size, type, exptr, upd);
	lvars base, offs, type, exptr, upd, _size, _reg, _ireg = false, _n;
	;;; Get structure base address in a register
	if not(base) then
		;;; Structure is on top of stack
		Drop_load(_LW, _TMP_REG_1 ->> _reg, Get_upop());
	elseif Is_register(Trans_structure(base)) ->> _reg then
		;;; Structure is already in a register lvar
		_int(_reg) -> _reg;
	else
		mishap(0, 'SYSTEM ERROR IN Do_field (bad structure)');
	endif;
	;;; If it's an external structure, get the real address
	if exptr then
		;;; deref exptr times
		Drop_exptr_deref(exptr, _TMP_REG_1, _reg) -> _reg;
	endif;
	;;; Interpret offset:
	if isinteger(offs) then
		;;; Constant offset for record-type structure:
		;;; convert bit offset to bytes
		##(b){_int(offs)|1} -> _n;
	elseif not(offs) then
		;;; Vector index on stack:
		;;; pop to TMP_REG_2
		Drop_load(_LW, _TMP_REG_2 ->> _ireg, Get_upop());
	elseif Is_register(Trans_structure(offs)) ->> _ireg then
		;;; Vector index in register lvar:
		_int(_ireg) -> _ireg;
	else
		mishap(0, 'SYSTEM ERROR IN Do_field (bad offset)');
	endif;
	;;; Pop new value from stack if updating
	if upd then Drop_load(_LW, _TMP_REG_0, Get_upop()) endif;
	if _ireg then
		;;; Convert vector index to sysint and scale for field size
		type fi_&& t_BASE_TYPE -> type;
		field_shamt(type) -> _n;
		if _size == 1 then
			_int(_n) -> _n;
			if _n _slt _0 then
				Drop_shifti(_SRA, _TMP_REG_2, _ireg, _negate(_n));
				_TMP_REG_2 -> _ireg;
			elseif _n _sgr _0 then
				Drop_shifti(_SLL, _TMP_REG_2, _ireg, _n);
				_TMP_REG_2 -> _ireg;
			endif;
			;;; Compute offset for V_BYTES, base-1 indexing and popint bits
			@@V_BYTES _add _int(field_offs(type)) -> _n;
		else
			_int(_size fi_<< (_n fi_+ 2)) -> _n;
			Drop_vecsub_mult(_TMP_REG_2, _ireg, _n, _TMP_REG_3);
			@@V_BYTES-{_n} -> _n;
			_TMP_REG_2 -> _ireg;
		endif;
		;;; Add offset to base
		Drop_arith(_ADDU, _TMP_REG_1, _reg, _ireg);
		_TMP_REG_1 -> _reg;
	endif;
	;;; Return reg+offs for the field access
	Get_based(_reg, _n, _TMP_REG_1);
enddefine;

define I_PUSH_FIELD();
	lvars type, size, base, offs, cvtpint, exptr, _reg;
	explode(asm_instr) -> exptr -> cvtpint -> offs -> base -> size -> type -> ;
	if type fi_&& t_BASE_TYPE == t_BIT then
		;;; bitfield:
		;;; set up argument registers
		Do_bitfield(base, offs, size, exptr, false);
		;;; and call bitfield access routine
		Drop_j(_JAL, if type == t_BIT then _bfield else _sbfield endif);
		;;; Result returned in RESULT_REG_0
		_RESULT_REG_0 -> _reg;
	else
		;;; non-bitfield:
		Drop_load(
			field_push_op(type),
			_TMP_REG_0 ->> _reg,
			Do_field(base, offs, 1, type, exptr, false));
	endif;
	;;; Result now in _reg
	if cvtpint then
		;;; Convert to popint
		Drop_shifti(_SLL, _reg, _reg, _2);
		Drop_arithi(_ADDIU, _reg, _reg, _3);
	endif;
	;;; Stack the result
	Drop_store(_SW, _reg, Get_upush());
enddefine;

define I_POP_FIELD();
	lvars type, size, base, offs, exptr, _op;
	explode(asm_instr) -> exptr -> offs -> base -> size -> type -> ;
	if type fi_&& t_BASE_TYPE == t_BIT then
		;;; bitfield:
		;;; set up argument registers
		Do_bitfield(base, offs, size, exptr, true);
		;;; and call bitfield update routine
		Drop_j(_JAL, _ubfield);
	else
		;;; non-bitfield:
		Drop_store(
			field_pop_op(type),
			_TMP_REG_0,
			Do_field(base, offs, 1, type, exptr, true));
	endif;
enddefine;

define I_PUSH_FIELD_ADDR();
	lvars type, size, base, offs, exptr;
	explode(asm_instr) -> exptr -> offs -> base -> size -> type -> ;
	;;; Field address to TMP_REG_0
	Drop_arithi(_ADDIU, _TMP_REG_0,
		Do_field(base, offs, size, type, exptr, false));
	;;; Stack the result
	Drop_store(_SW, _TMP_REG_0, Get_upush());
enddefine;


;;; -- Fast Field Access --------------------------------------------------

;;; Get_fsrc:
;;;		get the effective address of an operand of a "fast_" procedure or
;;;		optimised instruction. The operand is argument -_arg- of the current
;;;		instruction. It could be a MOVE instruction (representing a push
;;;		not yet done) or <false> if the item is already on the stack.

define lconstant Get_fsrc(_arg, _tmp_reg);
	lvars	op, _arg, _tmp_reg, _imm;
	dlocal	asm_instr;
	if asm_instr!INST_ARGS[_arg] ->> asm_instr then
		;;; source specified by a MOVE instruction
		if (asm_instr!INST_OP ->> op) == I_MOVENUM or op == I_MOVEADDR then
			;;; immediate data
			Get_imm(asm_instr!INST_ARGS[_0]);
		else
			;;; general move:
			;;; op == I_MOVE implies DEFER, op == I_MOVEQ implies DIRECT
			Get_opnd(_0, op == I_MOVE, _tmp_reg);
		endif;
	else
		;;; on stack
		Get_upop();
	endif;
enddefine;

;;; Get_fdst:
;;;		returns the destination operand of a "fast_" procedure.
;;;		This should be a push on the user stack, but it can be optimised
;;;		away if there's an immediately-following POP instruction.

define lconstant Get_fdst();
	lvars	instr;
	dlocal	asm_instr;
	;;; Look out for a following POP instruction
	fast_front(fast_back(asm_clist)) -> instr;
	if isvector(instr) and instr!INST_OP == I_POP then
		;;; Erase the instruction
		fast_back(asm_clist) -> asm_clist;
		;;; Return the destination of the POP
		instr -> asm_instr;
		Get_opnd(_0, DEFER, _TMP_DST_ADDR);
	else
		;;; Return a stack push
		Get_upush();
	endif;
enddefine;

;;; Drop_fsrc:
;;;		move an operand of a "fast_" procedure to a register.

define lconstant Drop_fsrc(_arg, _tmp_reg) -> _reg;
	lvars _arg, _tmp_reg, _reg, _offs;
	Get_fsrc(_arg, _tmp_reg) -> _offs -> _reg;
	if _offs then
		;;; operand is immediate or memory: load it to _tmp_reg
		Drop_move(_reg, _offs, _tmp_reg ->> _reg, false);
	endif;
enddefine;

;;;	Do_fastsubv:
;;;		compute the operand of a FASTSUBV instruction.

define lconstant Do_fastsubv();
	lvars _reg;
	;;; Structure in _reg
	Drop_fsrc(_1, _TMP_REG_1) -> _reg;
	;;; Popint index from stack to _TMP_REG_2
	Drop_load(_LW, _TMP_REG_2, Get_upop());
	;;; Add base to index
	Drop_arith(_ADDU, _TMP_REG_1, _reg, _TMP_REG_2);
	;;; Arg 0 is offset to first vector element (as popint)
	;;; -- subtract popint 1 to account for base 1 subscript and popint bits
	(_TMP_REG_1, _int(asm_instr!INST_ARGS[_0]) _sub 1);
enddefine;


;;; I-code fast field access instructions:

define I_FASTFIELD();
	lvars _reg, _offs;
	Drop_fsrc(_1, _TMP_REG_1) -> _reg;
	if asm_instr!INST_ARGS[_0] ->> _offs then
		_int(_offs) -> _offs;
	else
		;;; <false> means -fast_destpair-
		Drop_move(Get_based(_reg, @@P_FRONT, _TMP_REG_2), Get_upush());
		@@P_BACK -> _offs;
	endif;
	Drop_move(Get_based(_reg, _offs, _TMP_REG_2), Get_fdst());
enddefine;

define I_UFASTFIELD();
	lvars _reg;
	Drop_fsrc(_1, _TMP_REG_1) -> _reg;
	Drop_move(Get_upop(),
		Get_based(_reg, _int(asm_instr!INST_ARGS[_0]), _TMP_REG_2));
enddefine;

define I_FASTSUBV();
	Drop_move(Do_fastsubv(), Get_fdst());
enddefine;

define I_UFASTSUBV();
	lvars _reg, _offs;
	Do_fastsubv() -> _offs -> _reg;
	Drop_move(Get_upop(), _reg, _offs);
enddefine;


;;; -- Fast Integer +/- ---------------------------------------------------

;;; Drop_pfsrc:
;;;		gets a pop integer argument for a fi_+/- op.
;;;		Result is either a register or a short immediate.
;;;		Because there's no immediate subtraction on the MIPS,
;;;		we negate an immediate argument when the op is not +

define lconstant Drop_pfsrc(_arg, _tmp_reg, plus) -> _offs -> _reg;
	lvars plus, _arg, _tmp_reg, _offs, _reg;
	Get_fsrc(_arg, _tmp_reg) -> _offs -> _reg;
	if _reg then
		if _offs then
			;;; memory operand
			Drop_move(_reg, _offs, _tmp_reg ->> _reg, false ->> _offs);
		endif;
		Drop_arithi(_ADDIU, _tmp_reg, _reg, _-3);
		_tmp_reg -> _reg;
	elseif plus then
		unless isimm16(_offs _sub _3 ->> _offs) then
			Drop_loadi(_tmp_reg, _offs);
			_tmp_reg -> _reg; false -> _offs;
		endunless;
	elseif isimm16(_3 _sub _offs) then
		;;; (_3 _sub _n) negates popint _n, leaving the popint bits clear
		_3 _sub _offs -> _offs;
	else
		Drop_loadi(_tmp_reg, _offs _sub _3);
		_tmp_reg -> _reg; false -> _offs;
	endif;
enddefine;

define I_FAST_+-_2();
	lvars	plus, _src, _dst, _offs, _reg, _imm;
	dlocal	asm_instr;
	asm_instr!INST_ARGS[_0] -> plus;
	;;; Get source operand as register or short immediate
	;;;	and with popint bits cleared
	Drop_pfsrc(_1, _TMP_REG_1, plus) -> _imm -> _src;
	;;; Get target operand
	if asm_instr!INST_ARGS[_2] ->> asm_instr then
		Get_opnd(_0, DEFER, _TMP_REG_0)
	else
		Get_utop()
	endif -> _offs ->> _dst -> _reg;
	;;; Use a temporary if target is in memory
	if _offs then
		Drop_load(_LW, _TMP_REG_2 ->> _reg, _dst, _offs);
	endif;
	;;; Do the addition
	if _imm then
		Drop_arithi(_ADDIU, _reg, _reg, lo16(_imm));
	else
		Drop_arith(if plus then _ADDU else _SUBU endif, _reg, _reg, _src);
	endif;
	;;; Store result back to target if in memory
	if _offs then
		Drop_store(_SW, _reg, _dst, _offs);
	endif;
enddefine;

define I_FAST_+-_3();
	lvars plus, _src1, _src2, _dst, _offs, _reg, _imm;
	asm_instr!INST_ARGS[_0] -> plus;
	;;; Get second source operand as register or short immediate
	;;; and with popint bits cleared
	Drop_pfsrc(_1, _TMP_REG_1, plus) -> _imm -> _src2;
	;;; Get first source operand in register
	Drop_fsrc(_2, _TMP_REG_0) -> _src1;
	;;; Get destination operand
	if asm_instr!INST_ARGS[_3] then
		Get_opnd(_3, DEFER, _TMP_REG_2)
	else
		Get_upush()
	endif -> _offs ->> _dst -> _reg;
	;;; Use a temporary if destination is memory
	if _offs then _TMP_REG_0 -> _reg endif;
	;;; Do the addition
	if _imm then
		Drop_arithi(_ADDIU, _reg, _src1, lo16(_imm));
	else
		Drop_arith(if plus then _ADDU else _SUBU endif, _reg, _src1, _src2);
	endif;
	;;; Store result at destination if in memory
	if _offs then
		Drop_move(_reg, false, _dst, _offs);
	endif;
enddefine;


;;; -- Branches and Tests -------------------------------------------------

;;; is_long_branch:
;;;		true if -target- is not within range of a single branch instruction.
;;;		A branch allows for an 18-bit signed offset, but for forward branches
;;;		we have to allow for possible expansion of the intervening code so
;;;		use a 17-bit limit; for backward branches the target is fixed, but
;;;		the branch-dropping procedure may add a single extra instruction.

define lconstant is_long_branch(_target);
	lvars _target, _offs;
	_target _sub _asm_code_offset _sub _4 -> _offs;
	_offs _slteq _-16:20000 or _offs _sgreq _16:10000;
enddefine;

define lconstant Drop_long_br(_broffset, _arg);
	lvars _broffset, _arg;
	Drop_loadi(_TMP_REG_0, @@PD_TABLE{_strsize _add _broffset});
	Drop_arith(_ADDU, _TMP_REG_0, _TMP_REG_0, _PB);
	Drop_jr(_JR, _TMP_REG_0);
enddefine;

define lconstant Drop_long_brcond(_ifso, _ifnot, _is_so, _rs, _rt, _broffset,
																		_arg);
	lvars _ifso, _ifnot, _is_so, _rs, _rt, _broffset, _arg, _save;
	;;; Plant conditional branch with dummy offset
	Drop_cond_branch(if _is_so then _ifnot else _ifso endif, _rs, _rt, _0)
		-> _save;
	Drop_long_br(_broffset, _arg);
	Patch_branch(_save);
enddefine;

;;; I_LABEL:
;;;		called from -Code_pass- when a label is encountered in -asm_clist-

define I_LABEL();
	false -> _last_nop;
	Drop_uadjust();
	;;; set front of the label pair to the popint offset from the code start
	_pint(_asm_code_offset) -> fast_front(asm_clist);
	unless _delay_slot then true -> _delay_slot endunless;
enddefine;

;;; I_BR_std:
;;;		plant a relative branch instruction of a known size (8 bytes).
;;;		NB: assumes that the target is within reach of a 16-bit offset.

define I_BR_std(_broffset, _arg);
	lvars _broffset, _arg;
	if _asm_pass == 1 and is_long_branch(_broffset) then
		mishap(0, 'SYSTEM ERROR IN I_BR_std (target out of range)');
	endif;
	Drop_branch(_broffset);
enddefine;

;;; I_BR:
;;;		plant an unconditional branch

define I_BR(_broffset, _arg);
	lvars _broffset, _arg;
	if _asm_pass == 1 and is_long_branch(_broffset) then
		Drop_long_br -> asm_instr!INST_ARGS[_arg];
		chain(_broffset, _arg, Drop_long_br);
	endif;
	Drop_branch(_broffset);
enddefine;

;;; I_BRCOND:
;;;		standard conditional branch, used as argument to I_IF_opt etc.

define I_BRCOND(_ifso, _ifnot, _rs, _rt, _is_so, _broffset, _arg);
	lvars _ifso, _ifnot, _rs, _rt, _is_so, _broffset, _arg;
	if _asm_pass == 1 and is_long_branch(_broffset) then
		Drop_long_brcond -> asm_instr!INST_ARGS[_arg];
		chain(_ifso, _ifnot, _rs, _rt, _is_so, _broffset, _arg,
			Drop_long_brcond);
	endif;
	Drop_cond_branch(if _is_so then _ifso else _ifnot endif, _rs, _rt,
		_broffset) -> ;
enddefine;

;;; Drop_if:
;;;		call the branch-dropping routine from an argument of a test
;;;		instruction (I_IF_opt or I_BOOL_opt). Currently, this will
;;;		always be I_BRCOND or a Drop_long_brcond.

define lconstant Drop_if(/* _ifso, _ifnot, _rs, _rt, */ instr) with_nargs 5;
	lvars	instr;
	dlocal	asm_instr = instr;
	fast_apply(/* _ifso, _ifnot, _rs, _rt, */
		instr!INST_ARGS[_1],					;;; -is_so- flag
		_int(fast_front(instr!INST_ARGS[_0])),	;;; branch offset
		_2,
		instr!INST_ARGS[_2]);					;;; the I_BRCOND or similar
enddefine;

;;; {I_IF_opt ^target ^is_so ^opd}
;;;		the standard test instruction: compares the operand with <false>
;;;		and jumps to -target- if the test succeeds (or fails, depending on
;;;		the flag -is_so-).

define I_IF_opt();
	Drop_if(_BNE, _BEQ, Drop_fsrc(_3, _TMP_REG_0), _FALSE, asm_instr);
enddefine;

;;; {I_BOOL_opt ^target ^is_so ^opd}
;;;		like I_IF_opt, but used for sysAND and sysOR: if the branch is
;;;		taken, the operand should be left on the stack.

define I_BOOL_opt();
	lvars opd;
	if asm_instr!INST_ARGS[_3] ->> opd then
		;;; explicit operand -- we have to push it
		Drop_I_code(opd);
	endif;
	Drop_load(_LW, _TMP_REG_0, Get_utop());
	Drop_if(_BNE, _BEQ, _TMP_REG_0, _FALSE, asm_instr);
	;;; If the branch wasn't taken, remove the item from the stack again
	Get_upop() -> -> ;
enddefine;

;;;	{I_IF_CMP ^cmp_routine ^opd1 ^opd2 ^I_IF_opt}
;;;		optimised comparison:
;;;		"cmp_routine" is a comparison subroutine - _eq, _slt etc. - meant
;;;		to be applied to "opd1" and "opd2", but optimised to an inline test.
;;;		The two operands are "fsrc"-type operands, i.e. either MOVE
;;;		instructions or <false> for top-of-stack. The last argument is an
;;;		I_IF_opt instruction for testing the result of the comparison.
;;;		The sense of the test is: cmp_routine(opd2, opd1)

lconstant
	cmp_inverse = list_assoc_val(%[%
		nonop _neq,		nonop _eq,
		nonop _slteq,	nonop _sgr,
		nonop _sgreq,	nonop _slt,
	%]%),
;

define I_IF_CMP();
	lvars instr, _cmp, _inv, _reg1, _reg2, _imm;
	asm_instr!INST_ARGS[_0] -> _cmp;
	asm_instr!INST_ARGS[_3] -> instr;
	if cmp_inverse(_cmp) ->> _inv then _inv -> _cmp endif;
	if _cmp == nonop _eq then
		Get_fsrc(_1, _TMP_REG_1) -> _imm -> _reg1;
		if _imm and _reg1 then
			;;; memory operand -- load that first
			Drop_move(_reg1, _imm, _TMP_REG_1 ->> _reg1, false ->> _imm);
		endif;
		Drop_fsrc(_2, _TMP_REG_2) -> _reg2;
		if _imm then
			;;; load immediate operand now -- may fill a delay slot
			Drop_move(_reg1, _imm, _TMP_REG_1 ->> _reg1, false ->> _imm);
		endif;
		Drop_if(if _inv then _BNE,_BEQ else _BEQ,_BNE endif,
			_reg1, _reg2, instr);
	else
		if _cmp == nonop _slt then
			Get_fsrc(_1, _TMP_REG_1) -> _imm -> _reg1;
			Drop_fsrc(_2, _TMP_REG_2) -> _reg2;
		else
			;;; _SGR: reverse the arguments
			Drop_fsrc(_1, _TMP_REG_2) -> _reg2;
			Get_fsrc(_2, _TMP_REG_1) -> _imm -> _reg1;
		endif;
		if _imm and (_reg1 or not(isimm16(_imm))) then
			Drop_move(_reg1, _imm, _TMP_REG_1 ->> _reg1, false ->> _imm);
		endif;
		if _imm then
			;;; Operand 1 is 16-bit immediate:
			;;; slti t0, _reg2, _imm
			;;; bne  t0, zero
			Drop_arithi(_SLTI, _TMP_REG_0, _reg2, _imm);
			Drop_if(if _inv then _BEQ,_BNE else _BNE,_BEQ endif,
				_TMP_REG_0, _ZERO, instr);
		elseif _reg1 == _ZERO then
			;;; bltz _reg2
			Drop_if(if _inv then _BGEZ,_BLTZ else _BLTZ,_BGEZ endif,
				_reg2, _ZERO, instr);
		elseif _reg2 == _ZERO then
			;;; bgtz _reg1
			Drop_if(if _inv then _BLEZ,_BGTZ else _BGTZ,_BLEZ endif,
				_reg1, _ZERO, instr);
		else
			;;; slt t0, _reg2, _reg1
			;;; bne t0, zero
			Drop_arith(_SLT, _TMP_REG_0, _reg2, _reg1);
			Drop_if(if _inv then _BEQ,_BNE else _BNE,_BEQ endif,
				_TMP_REG_0, _ZERO, instr);
		endif;
	endif;
enddefine;


;;; {I_SWITCH ^lablist ^elselab ^opd}
;;;		computed goto.

define I_SWITCH();
	lvars lablist, elselab, _ncases, _offs, _reg, _save;

	asm_instr!INST_ARGS[_0] -> lablist;
	asm_instr!INST_ARGS[_1] -> elselab;
	listlength(lablist) + 1 -> _ncases;

	;;; Load the operand to a register
	Drop_fsrc(_2, _TMP_REG_0) -> _reg;

	;;; Compare the argument with the number of cases (both popints)
	unless isimm16(_ncases) then
		mishap(0, 'SYSTEM ERROR IN I_SWITCH');
	endunless;
	Drop_arithi(_SLTIU, _TMP_REG_1, _reg, _ncases);

	;;; If the argument was out of range, jump around the table
	Drop_cond_branch(_BEQ, _TMP_REG_1, _ZERO, _0) -> _save;

	;;; Use TMP_REG_0 as index into the jump offset table
	;;;	and load corresponding value to TMP_REG_1
	;;; The _24 is the length of the six instructions up to the table start
	;;;	(including the branch delay slot)
	;;;	and the _3 accounts for the popint bits in the index
	@@PD_TABLE{_strsize _add _asm_code_offset _add _24 _sub _3} -> _offs;
	Drop_arith(_ADDU, _TMP_REG_1, _PB, _reg);
	Drop_load(_LW, _TMP_REG_1, _TMP_REG_1, _offs);

	;;; Add PB to TMP_REG_1 to get absolute address and jump to it
	Drop_arith(_ADDU, _TMP_REG_1, _TMP_REG_1, _PB);
	Drop_jr(_JR, _TMP_REG_1);

	;;; Now plant the offset table:
	;;; 0 case first (an error, so jumps to after the table) ...
	Drop(@@PD_TABLE{_strsize _add _asm_code_offset _add _int(_ncases * 4)});
	;;; ... then all the given labels
	until lablist == [] do
		fast_front(fast_destpair(lablist) -> lablist) -> _offs;
		Drop(@@PD_TABLE{_strsize _add _int(_offs)});
	enduntil;

	;;; Patch the error jump
	Patch_branch(_save);

	;;; After the table: if there was no explicit "else" case, push the
	;;; argument back on the stack for a following error
	unless elselab then
		Drop_store(_SW, _reg, Get_upush());
	endunless;
	;;; Then fall through:
	;;; no reordering please!
	true -> _delay_slot;
enddefine;

;;; {I_PLOG_IFNOT_ATOM ^fail_label ^I_BRCOND}
;;;		planted after a call to -prolog_unify_atom- from "aprolog.s".
;;;		That will have set RESULT_REG_0 to 0 if unification succeeded,
;;;		non-zero otherwise.
;;;		If it failed, use the I_BRCOND instruction to jump to -fail_label-.

define I_PLOG_IFNOT_ATOM();
	fast_apply(
		_BNE, _BEQ,									;;; fail/succeed opcodes
		_RESULT_REG_0, _ZERO,
		true,										;;; select failure code
		_int(fast_front(asm_instr!INST_ARGS[_0])),	;;; the jump offset
		_1, asm_instr!INST_ARGS[_1]);				;;; the BR instruction
enddefine;

;;; {I_PLOG_TERM_SWITCH ^fail_label I_BRCOND ^var_label I_BRCOND}
;;;		planted after a call to -prolog_pair(term)_switch- from "aprolog.s".
;;;		That will have set RESULT_REG_0 as follows:
;;;			0	matching term
;;;		  > 0	variable
;;;		  < 0	failure

define I_PLOG_TERM_SWITCH();
	;;; Branch to -var_label- if item was a variable
	fast_apply(
		_BGTZ, _BLEZ,								;;; var/nonvar opcodes
		_RESULT_REG_0, _ZERO,
		true,										;;; select var code
		_int(fast_front(asm_instr!INST_ARGS[_2])),	;;; var label offset
		_3, asm_instr!INST_ARGS[_3]);				;;; the BR instruction
	;;; Branch to -fail_label- if item didn't match
	fast_apply(
		_BLTZ, _BGEZ,								;;; fail/succeed opcodes
		_RESULT_REG_0, _ZERO,
		true,										;;; select failure code
		_int(fast_front(asm_instr!INST_ARGS[_0])),	;;; fail label offset
		_1, asm_instr!INST_ARGS[_1]);				;;; the BR instruction
	;;; Fall through if item matched
enddefine;


;;; -- Procedure Call and Return ------------------------------------------

define lconstant Drop_call(on_stack, defer, _routine);
	lvars on_stack, defer, _routine, _reg, _offs;
	if on_stack then
		Get_upop();
	else
		Get_opnd(_0, defer, _TMP_REG_0);
	endif -> _offs -> _reg;
	if _routine then
		;;; Call via checking routine
		Drop_move(_reg, _offs, _ARG_REG_0, false);
		Drop_j(_JAL, _routine);
	else
		;;; Call direct: get execute address
		if _offs then
			Drop_move(_reg, _offs, _TMP_REG_0 ->> _reg, false);
		endif;
		Drop_load(_LW, _CALL_REG, _reg, @@PD_EXECUTE);
		Drop_jr(_JALR, _CALL_REG);
	endif;
enddefine;

define I_CALL	 = Drop_call(% false, DEFER,  _popenter    %) enddefine;
define I_CALLQ	 = Drop_call(% false, DIRECT, _popenter    %) enddefine;
define I_CALLP	 = Drop_call(% false, DEFER,   false       %) enddefine;
define I_CALLPQ	 = Drop_call(% false, DIRECT,  false       %) enddefine;
define I_CALLS	 = Drop_call(% true,  DIRECT, _popenter    %) enddefine;
define I_CALLPS	 = Drop_call(% true,  DIRECT,  false       %) enddefine;
define I_UCALL	 = Drop_call(% false, DEFER,  _popuenter   %) enddefine;
define I_UCALLQ	 = Drop_call(% false, DIRECT, _popuenter   %) enddefine;
define I_UCALLP	 = Drop_call(% false, DEFER,  _popuncenter %) enddefine;
define I_UCALLPQ = Drop_call(% false, DIRECT, _popuncenter %) enddefine;
define I_UCALLS	 = Drop_call(% true,  DIRECT, _popuenter   %) enddefine;
define I_UCALLPS = Drop_call(% true,  DIRECT, _popuncenter %) enddefine;

define I_CALLABS();
	Drop_j(_JAL, asm_instr!INST_ARGS[_0]!PD_EXECUTE);
enddefine;

define I_CHAIN_REG();
	;;; Chain a procedure from a register
	lvars _reg = _int(Is_register(asm_instr!INST_ARGS[_0]));
	Drop_load(_LW, _CALL_REG, _reg, @@PD_EXECUTE);
	Drop_jr(_JR, _CALL_REG);
enddefine;

define I_CALLSUB();
	Drop_j(_JAL, asm_instr!INST_ARGS[_0]);
enddefine;

define I_CHAINSUB();
	Drop_j(_J, asm_instr!INST_ARGS[_0]);
enddefine;

define I_CALLSUB_REG();
	lvars _target, _reg;
	fast_front(asm_instr!INST_ARGS[_0]) -> _target;
	if Is_register(_target) ->> _reg then
		_int(_reg) -> _reg;
		if _reg == _CALL_REG then
			Drop_jr(_JALR, _reg);
		else
			dlocal _fill_delay = false;
			Drop_uadjust();
			Drop_jr(_JALR, _reg);
			Drop_move(_reg, false, _CALL_REG, false);
			true -> _delay_slot;
		endif;
	else
		Drop_j(_JAL, _target);
	endif;
enddefine;

define I_RETURN();
	Drop_jr(_JR, _RA);
enddefine;


;;; -- Miscellaneous ------------------------------------------------------

;;; {I_STACKLENGTH}
;;;		push the length of the user stack

define I_STACKLENGTH();
	Drop_load(_LW, _TMP_REG_0, Get_abs(ident _userhi, _TMP_REG_0));
	Drop_arithi(_ADDIU, _TMP_REG_1, _USP, _usp_offs _sub _3);
	Drop_arith(_SUBU, _TMP_REG_0, _TMP_REG_0, _TMP_REG_1);
	Drop_store(_SW, _TMP_REG_0, Get_upush());
enddefine;

;;; {I_SETSTACKLENGTH ^saved_stacklength ^nresults}
;;;		check that the stacklength is equal to the saved stacklength plus
;;;		an expected number of results. If not, call -setstklen_diff- to fix.
;;;		This does the same job as -setstklen- from "alisp.s", but avoids the
;;;		subroutine call when the number of results is known in advance.

define I_SETSTACKLENGTH();
	lvars _nresults, _save, _reg;
	if asm_instr!INST_ARGS[_1] ->> _nresults then
		;;; Load stack base (_userhi) to ARG_REG_0
		Drop_load(_LW, _ARG_REG_0, Get_abs(ident _userhi, _TMP_REG_0));
		;;; Load saved stacklength to a register
		Drop_fsrc(_0, _TMP_REG_0) -> _reg;
		;;; Adjust stack base for number of results and popint bits
		;;; (in _nresults and saved stacklength)
		Drop_arithi(_ADDIU, _ARG_REG_0, _ARG_REG_0, _6 _sub _nresults);
		;;; Subtract saved stacklength to get desired stack pointer
		Drop_arith(_SUBU, _ARG_REG_0, _ARG_REG_0, _reg);
		;;; Compare desired stack pointer with actual: if not the same,
		;;; call _setstklen_diff to fix
		Drop_uadjust();
		Drop_cond_branch(_BEQ, _ARG_REG_0, _USP, _0) -> _save;
		Drop_j(_JAL, _setstklen_diff);
		Patch_branch(_save);
		true -> _delay_slot;
	else
		;;; both stacklength and nresults are on the stack:
		;;; call c_setstklen
		Drop_j(_JAL, _setstklen);
	endif;
enddefine;

;;; {I_LISP_TRUE}
;;;		replace <false> result on top of stack by nil.

define I_LISP_TRUE();
	lvars	_save;
	dlocal	_fill_delay = false;
	Drop_load(_LW, _TMP_REG_0, Get_utop());
	Drop_lui(_TMP_REG_1, hi16(nil), false) -> ;
	Drop_cond_branch(_BNE, _TMP_REG_0, _FALSE, _0) -> _save;
	Drop_arithi(_ADDIU, _TMP_REG_1, _TMP_REG_1, lo16(nil));
	Drop_store(_SW, _TMP_REG_1, Get_utop());
	Patch_branch(_save);
	true -> _delay_slot;
enddefine;

;;; {I_CHECK}
;;;		plant checks on backward jumps.

define I_CHECK();
	lvars	_save1, _save2;
	dlocal	_fill_delay;
	Drop_load(_LW, _TMP_REG_0, Get_abs(ident _trap, _TMP_REG_0));
	Drop_load(_LW, _TMP_REG_1, Get_abs(ident _userlim, _TMP_REG_1));
	;;; If interrupt flag set, jump to call checkall
	false -> _fill_delay;
	Drop_cond_branch(_BNE, _TMP_REG_0, _ZERO, _0) -> _save1;
	;;; Check for userstack overflow:
	Drop_uadjust();
	Drop_arith(_SLT, _TMP_REG_1, _USP, _TMP_REG_1);
	Drop_cond_branch(_BEQ, _TMP_REG_1, _ZERO, _0) -> _save2;
	Patch_branch(_save1);
	Drop_nop();
	true -> _delay_slot;
	true -> _fill_delay;
	Drop_j(_JAL, _checkall);
	Patch_branch(_save2);
enddefine;


;;; -- Procedure Entry and Exit -------------------------------------------

define I_CREATE_SF();
	lvars _i, _offs;

	;;; Set the procedure base register: calling convention requires
	;;; that the execute address (i.e. pointer to this first
	;;; instruction) is in _CALL_REG so we can recover the procedure
	;;; address from that by subtracting the header size.
	_negate(@@PD_TABLE{_strsize}) -> _offs;
	if isimm16(_offs)then
		Drop_arithi(_ADDIU, _PB, _CALL_REG, lo16(_offs));
	else
		;;; we wouldn't epect this while the maximum size of an entire
		;;; procedure is limited to 64K words, but just in case ...
		Drop_loadi(_TMP_REG_0, _negate(_offs));
		Drop_arith(_SUBU, _PB, _CALL_REG, _TMP_REG_0);
	endif;

	;;; Adjust the stack pointer
	_framesize -> _offs;
	Drop_arithi(_ADDIU, _SP, _SP, lo16(_negate(_offs)));

	;;; Save the return address
	Drop_store(_SW, _RA, _SP, _offs--@(w) ->> _offs);

#_IF DEF PIC
	;;; Leave space for the context pointer: we don't actually need to
	;;; save it because it's never used, but the space has to be left to
	;;; keep the stack frame format consistent
	_offs--@(w) -> _offs;
#_ENDIF

	;;; Save register lvars
	if _nonzero(_regmask) then
		_NPOP_REG_3 -> _i;
		repeat
			if _regmask _bitst _shift(_1, _i) then
				Drop_store(_SW, _i, _SP, _offs--@(w) ->> _offs);
			endif;
		quitif(_i == _POP_REG_0);
			_i _sub _1 -> _i;
		endrepeat;
	endif;

	;;; Save dynamic locals:
	;;; done pairwise to fill load delay slots
	if _Nlocals _bitst _1 then
		;;; Odd number of locals: do the first one specially, saving the
		;;; procedure base register in the delay slot
		Drop_load(_LW, _TMP_REG_0, _PB, @@PD_TABLE);
		Drop_load(_LW, _TMP_REG_0, _TMP_REG_0, @@ID_VALOF);
		Drop_store(_SW, _PB, _SP, _0);
		Drop_store(_SW, _TMP_REG_0, _SP, _offs--@(w) ->> _offs);
		_1 -> _i;
	else
		;;; Just save the procedure base register
		Drop_store(_SW, _PB, _SP, _0);
		_0 -> _i;
	endif;
	until _i == _Nlocals do
		Drop_load(_LW, _TMP_REG_0, _PB, @@PD_TABLE[_i]);
		Drop_load(_LW, _TMP_REG_1, _PB, @@PD_TABLE[_i _add _1]);
		Drop_load(_LW, _TMP_REG_0, _TMP_REG_0, @@ID_VALOF);
		Drop_load(_LW, _TMP_REG_1, _TMP_REG_1, @@ID_VALOF);
		Drop_store(_SW, _TMP_REG_0, _SP, _offs--@(w) ->> _offs);
		Drop_store(_SW, _TMP_REG_1, _SP, _offs--@(w) ->> _offs);
		_i _add _2 -> _i;
	enduntil;

	;;; Initialise POP on-stack lvars to zero
	unless _zero(_Npopstkvars) then
		Drop_arithi(_ADDIU, _TMP_REG_0, _ZERO, _3);
		fast_repeat _pint(_Npopstkvars) times
			Drop_store(_SW, _TMP_REG_0, _SP, _offs--@(w) ->> _offs);
		endrepeat;
	endunless;
enddefine;

define I_UNWIND_SF();
	lvars _i, _offs = _framesize;

	;;; Restore the return address
	Drop_load(_LW, _RA, _SP, _offs--@(w) ->> _offs);

#_IF DEF PIC
	;;; Skip space left for the context pointer
	_offs--@(w) -> _offs;
#_ENDIF

	;;; Restore register lvars
	if _nonzero(_regmask) then
		_NPOP_REG_3 -> _i;
		repeat
			if _regmask _bitst _shift(_1, _i) then
				Drop_load(_LW, _i, _SP, _offs--@(w) ->> _offs);
			endif;
		quitif(_i == _POP_REG_0);
			_i _sub _1 -> _i;
		endrepeat;
	endif;

	;;; Restore dynamic locals:
	;;; done pairwise to fill load delay slot
	_1 -> _i;
	while _i fi_< _Nlocals do
		Drop_load(_LW, _TMP_REG_0, _PB, @@PD_TABLE[_i _sub _1]);
		Drop_load(_LW, _TMP_REG_1, _SP, _offs--@(w) ->> _offs);
		Drop_load(_LW, _TMP_REG_2, _PB, @@PD_TABLE[_i]);
		Drop_load(_LW, _TMP_REG_3, _SP, _offs--@(w) ->> _offs);
		Drop_store(_SW, _TMP_REG_1, _TMP_REG_0, @@ID_VALOF);
		Drop_store(_SW, _TMP_REG_3, _TMP_REG_2, @@ID_VALOF);
		_i _add _2 -> _i;
	endwhile;
	if _i == _Nlocals then
		;;; one more to go: interleave it with resetting the stack pointer
		Drop_load(_LW, _TMP_REG_0, _PB, @@PD_TABLE[_i _sub _1]);
		Drop_load(_LW, _TMP_REG_1, _SP, _offs--@(w) ->> _offs);
		Drop_arithi(_ADDIU, _SP, _SP, _framesize);
		Drop_store(_SW, _TMP_REG_1, _TMP_REG_0, @@ID_VALOF);
	else
		;;; reset the stack pointer
		Drop_arithi(_ADDIU, _SP, _SP, _framesize);
	endif;

	;;; Restore the procedure base register
	Drop_load(_LW, _PB, _SP, _0);
enddefine;


;;; -- The Assembler ------------------------------------------------------

#_IF DEF I_DEBUG

section;
global constant procedure (initintvec, isclosed);
endsection;

lvars
	logdev	= false,
	logbuff	= writeable initintvec(1),
;

define lconstant Dump_pdr(pdr, codesize);
	lvars pdr, codesize, n;
	;;; Open I-code log file
	if not(logdev) or isclosed(logdev) then
		unless syscreate('ICODE.LOG', 1, true) ->> logdev then
			mishap(0, 'CANNOT OPEN I-CODE LOG FILE' <> sysiomessage());
		endunless;
	endif;
	;;; Dump procedure code
	if codesize fi_> datalength(logbuff) then
		initintvec(codesize) -> logbuff;
	endif;
	4*codesize ->> codesize -> logbuff(1);
	syswrite(logdev, logbuff, 4);
	_moveq(_int(codesize), pdr@PD_TABLE{_strsize}, logbuff@V_WORDS) -> ;
	syswrite(logdev, logbuff, codesize);
	sysflush(logdev);
enddefine;

#_ENDIF

;;; Do_consprocedure:
;;;		outputs machine code into a procedure record from pop assembler
;;;		input

define Do_consprocedure(codelist, reg_locals) -> pdr;
	lvars	codelist, reg_locals, pdr, reg, _code_offset, _size;
	dlocal	_asm_drop_ptr, _asm_pass, _framesize, _regmask, _strsize = _0;

	define lconstant Do_code_pass(/* _asm_pass, codelist */) with_nargs 2;
		dlocal
			_fill_delay = true,
			_delay_slot = true,
			_last_nop	= false,
			_usp_offs	= _0,
		;
		Code_pass(/* _asm_pass, codelist */);
	enddefine;

	;;; Compute stack frame size (the 2 is for return address + owner)
	@@(w)[_Nreg _add _Nlocals _add _Nstkvars _add _2] -> _framesize;

	;;; Compute register mask
	_0 -> _regmask;
	fast_for reg in reg_locals do
		lvars _reg = _int(Is_register(reg));
#_IF DEF PIC
		;;; ignore context pointer -- always saved
		nextif(_reg == _GP);
#_ENDIF
		_regmask _biset _shift(_1, _reg) -> _regmask;
	endfor;

	;;; Pass 1 -- calculate instruction offsets
	Do_code_pass(0, codelist) -> _code_offset;
	@@(w)[_int(listlength(asm_struct_list))] -> _strsize;

	;;; Pass 2 -- adjust branch offsets for big procedures
	if ##(w){_code_offset} _gr _16:7FFF then
		Do_code_pass(1, codelist) -> _code_offset;
	endif;

	;;; Pass 3 -- optimisations
	;;; Not made use of (yet)
	;;; unless pop_debugging then
	;;;		Do_code_pass(2, codelist) -> _code_offset;
	;;;	endunless;

	;;; Now calculate total size of procedure and allocate store for it.
	;;; The procedure record will be returned with the header and structure
	;;;	table already filled in, and with _asm_drop_ptr pointing to the
	;;;	start of the executable code section.
	@@PD_TABLE{_strsize _add _code_offset | b.r} _sub @@POPBASE -> _size;
	Get_procedure(_size, _shift(_regmask, _-16)) -> pdr;

	;;; Final pass -- plants the code
	Do_code_pass(false, codelist) -> ;
#_IF DEF I_DEBUG
	;;; write to log file
	Dump_pdr(pdr, _pint(##(w){_code_offset | b.r}));
#_ENDIF
enddefine;

endsection;		/* $-Sys$-Vm */


/* --- Revision History ---------------------------------------------------
--- Robert John Duncan, Dec 15 1994
		Changed Drop_vecsub_mult not to return an offset, since the value
		may differ depending on the context of use.
--- Robert John Duncan, Jun  8 1994
		Fix to I_CHECK
--- Robert John Duncan, Mar 29 1994
		Changed stack frames to allow space for the context pointer in
		position-independent code.
--- Robert John Duncan, Mar 22 1994
		Undid the last change.
--- Robert John Duncan, Mar 15 1994
		Restored the convention that pop calls should pass the procedure
		address in ARG_REG_0, but that __in ________addition __to the execute address
		in CALL_REG, and now for system procedures as well as user ones.
		(Generates lots more code!)
--- Robert John Duncan, Mar 11 1994
		Minor improvement to I_IF_CMP which will remove the odd no-op.
--- Robert John Duncan, Mar  9 1994
		Replaced use of the global pointer with the special var block
		register
--- Robert John Duncan, Mar  8 1994
		All jumps now place the target address in CALL_REG. This makes the
		old distinction of small vs. large memory models unnecessary.
--- Robert John Duncan, Mar  7 1994
		Changed for new pop calling convention: instead of the procedure
		address being passed in ARG_REG_0, the execute address is passed
		in CALL_REG ($t9).
--- Robert John Duncan, Apr 14 1993
		Fixes to _asm_pass 1 (dropping long branches).
--- Robert John Duncan, May 26 1991
		Added check for stack underflow to ERASE
--- Simon Nichols, Aug 28 1990
		-size- argument to field instructions now in units of type size;
		changed -Do_field- to detect size/==1 case for subscripting
		external array of structures/arrays etc.
--- Robert John Duncan, Jul 26 1990
		Fixed I_SWITCH to use new-style branch-dropping procedures
--- Robert John Duncan, Jul 10 1990
		Added GP-relative addressing for system identifiers.
		Changed branch-dropping instructions to take the target rather
		than an offset.
		Tidied up handling of immediate values.
		Made most procedures lconstant.
--- Simon Nichols, Jun 11 1990
		Improved filling of branch delay slots (i.e. fewer NOPs).
--- Simon Nichols, Jun  8 1990
		Immediate values now accessed as offsets from FALSE register if
		they are within range.
--- Simon Nichols, Jun  7 1990
		Improved code planted for I_CALLABS and also (if LARGE_MEMORY is
		not defined) all jumps and subroutine calls. These are planted as
		single J/JAL instructions with 28 bit offsets.
 */
