/* --- Copyright University of Sussex 1994. All rights reserved. ----------
 > File:		C.mips/src/syscomp/genproc.p
 > Purpose:     Compiles M-Code to MIPS R2000 assembler
 > Author:      Robert Duncan, Nov  1 1989 (see revisions)
 */


#_INCLUDE 'common.ph'

section $-Popas$-M_trans;

global constant procedure (
	negate_test, perm_const_opnd,
);

global vars
	current_pdr_label,
	current_pdr_exec_label,
	false_immediate,
;


	/****************************************************************
	* This file is contained entirely in section Genproc, and must  *
	*   define all the exports to section M_trans listed below.     *
	****************************************************************/

section Genproc =>

		/*	M-opcode Procedures */

		M_ADD
		M_ASH
		M_BIC
		M_BIM
		M_BIS
		M_BIT
		M_BRANCH
		M_BRANCH_std
		M_BRANCH_ON
		M_BRANCH_ON_INT
		M_CALL
		M_CALL_WITH_RETURN
		M_CALLSUB
		M_CHAIN
		M_CHAINSUB
		M_CLOSURE
		M_CMP
		M_CMPKEY
		M_CREATE_SF
		M_END
		M_ERASE
		M_LABEL
		M_LOGCOM
		M_MOVE
		M_MOVEb
		M_MOVEbit
		M_MOVEs
		M_MOVEsb
		M_MOVEsbit
		M_MOVEss
		M_MULT
		M_NEG
		M_PADD
		M_PADD_TEST
		M_PCMP
		M_PLOG_IFNOT_ATOM
		M_PLOG_TERM_SWITCH
		M_PSUB
		M_PSUB_TEST
		M_PTR_ADD_OFFS
		M_PTR_CMP
		M_PTR_SUB
		M_PTR_SUB_OFFS
		M_RETURN
		M_SETSTKLEN
		M_SUB
		M_TEST
		M_UNWIND_SF
		M_UPDb
		M_UPDbit
		M_UPDs

		/*	Registers */

		SP
		USP
		USP_+
		-_USP
		i_USP
;;;		i_USP_+			;;; these two are optional
;;;		ii_USP
		WK_ADDR_REG_1
		WK_ADDR_REG_2
		WK_REG
		CHAIN_REG
		SPECIAL_VAR_BLOCK_REG	;;; optional

		/*	Register Lists */

		nonpop_registers
		pop_registers

		/*	Register Procedures */

		reglabel
		regnumber
		autoidreg

		/*	M-code Mapping of Subroutines */

		mc_inline_conditions_list
		mc_inline_procs_list

		/*	Procedures Needed by -m_optimise- */

		cvt_pop_subscript
		can_defer_opnd
		pdr_index_opnd

		/*	Procedure to Generate Code */

		mc_code_generator
;


;;; -- REGISTER ALLOCATION ------------------------------------------------

;;; Register names (software names are defined in "pop_regdef.h")
lconstant registers = [
/*
				 H/W	 S/W
		 No.	Name	Name			Usage (* = reserved)

*/
	{	   0	 \$0	zero	}	;;; Hard-wired zero
	{	   1	 \$1	  at	}	;;; Assembler temporary (*)
	{	   2	 \$2	  v0	}	;;; RESULT_REG_0
	{	   3	 \$3	  v1	}	;;; RESULT_REG_1
	{	   4	 \$4	  a0	}	;;; ARG_REG_0
	{	   5	 \$5	  a1	}	;;; ARG_REG_1
	{	   6	 \$6	  a2	}	;;; ARG_REG_2
	{	   7	 \$7	  a3	}	;;; ARG_REG_3
	{	   8	 \$8	  t0	}	;;; WK_REG
	{	   9	 \$9	  t1	}	;;; TMP_REG_1	)
	{	  10	\$10	  t2	}	;;; TMP_REG_2   )   scratch registers
	{	  11	\$11	  t3	}	;;; TMP_REG_3   )
	{	  12	\$12	  t4	}	;;; TMP_REG_4   )
	{	  13	\$13	  t5	}	;;; WK_ADDR_REG_1
	{	  14	\$14	  t6	}	;;; WK_ADDR_REG_2
	{	  15	\$15	  cr	}	;;; CHAIN_REG
	{	  16	\$16	  p0	}	;;; POP_REG_0
	{	  17	\$17	  p1	}	;;; POP_REG_1
	{	  18	\$18	 np0	}	;;; NPOP_REG_0
	{	  19	\$19	 np1	}	;;; NPOP_REG_1
	{	  20	\$20	 np2	}	;;; NPOP_REG_2
	{	  21	\$21	 np3	}	;;; NPOP_REG_3
	{	  22	\$22   false	}	;;; FALSE
	{	  23	\$23	  pb	}	;;; PB
	{	  24	\$24	 svb	}	;;; SPECIAL_VAR_BLOCK_REG
	{	  25	\$25	  t9	}	;;; Call register
	{	  26	\$26	  k0	}	;;; Kernel register 1 (*)
	{	  27	\$27	  k1	}	;;; Kernel register 2 (*)
	{	  28	\$28	  gp	}	;;; Global pointer (*)
	{	  29	\$29	  sp	}	;;; SP
	{	  30	\$30	 usp	}	;;; USP
	{	  31	\$31	  ra	}	;;; Return address
];

constant
	WK_REG					= "t0",
	WK_ADDR_REG_1			= "t5",
	WK_ADDR_REG_2			= "t6",
	SP						= "sp",
	USP						= "usp",
	CHAIN_REG				= "cr",
	SPECIAL_VAR_BLOCK_REG	= "svb",
;

lconstant
	ZERO			= "zero",
	RESULT_REG_0	= "v0",
	RESULT_REG_1	= "v1",
	ARG_REG_0		= "a0",
	ARG_REG_1		= "a1",
	ARG_REG_2		= "a2",
	ARG_REG_3		= "a3",
	TMP_REG_1		= "t1",
	TMP_REG_2		= "t2",
	TMP_REG_3		= "t3",
	TMP_REG_4		= "t4",
	CALL_REG		= "t9",
	POP_REG_0		= "p0",
	POP_REG_1		= "p1",
	NPOP_REG_0		= "np0",
	NPOP_REG_1		= "np1",
	NPOP_REG_2		= "np2",
	NPOP_REG_3		= "np3",
	FALSE			= "false",
	PB				= "pb",
	GP				= "gp",
	RA				= "ra",
;

;;; reglabel:
;;;		maps register numbers to software names

define reglabel = newproperty(
	maplist(registers,
		procedure(v);
			lvars v;
			[% v(1), v(3) %];
		endprocedure),
	32, false, "perm");
enddefine;

;;; regnumber:
;;;		maps software names to register numbers

define regnumber = newproperty(
	maplist(registers,
		procedure(v);
			lvars v;
			[% v(3), v(1) %];
		endprocedure),
	32, false, "perm");
enddefine;

;;; asm_regname:
;;;		maps software names to assembler names

#_IF DEF M_DEBUG
;;; Keep software names for readability
constant procedure asm_regname = identfn;
#_ELSE
;;; Use hardware names
define constant procedure asm_regname = newproperty(
	maplist(registers,
		procedure(v);
			lvars v;
			[% v(3), v(2) %];
		endprocedure),
	32, false, "perm");
enddefine;
#_ENDIF

;;; autoidreg:
;;;		pretend that all the registers support auto-indirection
identof("regnumber") -> identof("autoidreg");

;;; Temporary registers:

lconstant TMP_REGS = [^TMP_REG_1 ^TMP_REG_2 ^TMP_REG_3 ^TMP_REG_4];

lvars tmp_regs = [], n_tmp_regs_used = 0;

define lconstant tmp_reg() -> reg;
	lvars reg;
	if tmp_regs == [] then
		TMP_REGS -> tmp_regs;
	endif;
	fast_destpair(tmp_regs) -> (reg, tmp_regs);
	n_tmp_regs_used fi_+ 1 -> n_tmp_regs_used;
	if n_tmp_regs_used fi_> #_< length(TMP_REGS) >_# then
		;;; all registers used up within one instruction!
		mishap(0, 'OUT OF TEMPORARY REGISTERS');
	endif;
enddefine;

define lconstant use_reg(reg) -> reg;
	lvars reg;
	;;; if no register given, use a temporary
	unless reg then
		tmp_reg() -> reg;
	endunless;
enddefine;

define lconstant reuse_reg(reg) -> reg;
	lvars reg;
	;;; if register is temporary, reuse it, otherwise get a new one
	unless fast_lmember(reg, TMP_REGS) then
		tmp_reg() -> reg;
	endunless;
enddefine;


;;; Register lvars:

#_IF DEF PIC
;;; context pointer is always saved as part of the entry code
lconstant nonpop_always_save = [% regnumber(GP) %];
#_ELSE
lconstant nonpop_always_save = [];
#_ENDIF

constant
	pop_registers = [] ::
		maplist([^POP_REG_0 ^POP_REG_1], regnumber),
	nonpop_registers = [^^nonpop_always_save] ::
		maplist([^NPOP_REG_0 ^NPOP_REG_1 ^NPOP_REG_2 ^NPOP_REG_3], regnumber),
;

;;; Special operands:

constant
	USP_+			= {^USP ^true},
	-_USP			= {^USP ^false},
	i_USP			= {^USP 0},
;


;;; -- OPERAND TYPES ------------------------------------------------------

;;; wof:
;;;		convert word offset to bytes

define lconstant wof =
	nonop fi_*(% WORD_OFFS %);
enddefine;

;;; isimm, isreg, isregind, isabs, ismem:
;;;		recognisers for operand types

lconstant procedure (
	isreg		= isword,
	isregind	= isvector,
	isabs		= isstring,
);

define lconstant isimm(opnd);
	lvars opnd;
	isintegral(opnd) or isref(opnd);
enddefine;

define lconstant ismem(opnd);
	lvars opnd;
	isabs(opnd) or isregind(opnd);
enddefine;

;;; can_defer_opnd:
;;;		constructs a field-access operand from an address -opnd- and an
;;;		offset -offs-. Returns <false> if no suitable operand exists.

define can_defer_opnd(opnd, offs, upd);
	lvars opnd, offs, upd;
	if isreg(opnd) then
		{^opnd ^offs};
	elseif isref(opnd) then
		;;; immediate label
		if offs == 0 then
			fast_cont(opnd);
		elseif offs > 0 then
			fast_cont(opnd) <> '+' >< offs;
		else
			fast_cont(opnd) >< offs;
		endif;
	else
		false;
	endif;
enddefine;

;;; pdr_index_opnd:
;;;		constructs an operand for accessing values from the current procedure
;;;		header (not used in closures, so PB will hold the procedure address)

define pdr_index_opnd(fld_index);
	lvars fld_index;
	{% PB, fld_index.wof %}
enddefine;


;;; -- THE CODE STREAM ----------------------------------------------------

lconstant

	;;; Special opcodes (directives, etc.)

	COMMENT		= "#",
	CPADD		= "'.cpadd'",
	CPLOAD		= "'.cpload'",
	CPRESTORE	= "'.cprestore'",
	END			= "'.end'",
	ENT			= "'.ent'",
	FRAME		= "'.frame'",
	GPWORD		= "'.gpword'",
	LABEL		= ":",
	MASK		= "'.mask'",
	NOP			= "nop",
	RDATA		= "'.rdata'",
	SET			= "'.set'",
	TEXT		= "'.text'",
	WORD		= "'.word'",
;

lvars
	codelist,
	codelist_end,
	m_instr,
;

;;; plant:
;;;		adds an instruction to the end of the code list

define lconstant plant();
	conspair(consvector(),[]) ->> f_tl(codelist_end) -> codelist_end;
enddefine;

define lconstant plant0 = plant(% 1 %) enddefine;
define lconstant plant1 = plant(% 2 %) enddefine;
define lconstant plant2 = plant(% 3 %) enddefine;
define lconstant plant3 = plant(% 4 %) enddefine;


;;; -- ADJUSTING THE USER STACK -------------------------------------------

lvars
	USP_offs,
;

;;; USP_adjust:
;;;		plants an instruction to adjust USP by the amount of USP_offs

define USP_adjust();
	unless USP_offs == 0 then
		if USP_offs > 0 then
			plant3("addu", USP, USP, USP_offs);
		else
			plant3("subu", USP, USP, -USP_offs);
		endif;
		0 -> USP_offs;
	endunless;
enddefine;

;;; USP_opnd:
;;;		converts the user-stack operand -opnd- to remove pushes and pops
;;;		and to take account of USP_offs

define USP_opnd(opnd) -> opnd;
	lvars opnd, offs;
	f_subv(2, opnd) -> offs;
	if offs == true then
		;;; pop
		{% USP, USP_offs %} -> opnd;
		USP_offs + 4 -> USP_offs;
	elseif offs == false then
		;;; push
		USP_offs - 4 -> USP_offs;
		{% USP, USP_offs %} -> opnd;
	elseif USP_offs /== 0 then
		{% USP, offs + USP_offs %} -> opnd;
	endif;
enddefine;


;;; -- TRANSLATING M-CODE INSTRUCTIONS ------------------------------------

/*
 *	Data Movement Instructions
 */


lvars
	loadop	= "lw",
	storeop	= "sw",
;

define lconstant opndsize =
	newassoc([
		[lb		1]
		[lbu	1]
		[lh		2]
		[lhu	2]
		[lw		4]
		[sb		1]
		[sh		2]
		[sw		4]
	]);
enddefine;

;;; cvt_imm:
;;;		converts an immediate operand to a register or integer immediate.
;;;		___reg is an optional register to use for loading addresses.

define lconstant cvt_imm(imm, reg) -> imm;
	lvars imm, reg;
	if imm == 0 then
		ZERO -> imm;
	elseif isref(imm) then
		;;; symbol:
		if imm = false_immediate then
			FALSE -> imm;
		else
			;;; load as an address
			plant2("la", use_reg(reg) ->> reg, imm);
			reg -> imm;
		endif;
	endif;
enddefine;

;;; cvt_src:
;;;		converts an arbitrary source operand to a register, doing a load
;;;		where necessary: ___reg is an optional register to use in this case.

define lconstant cvt_src(src, reg) -> reg;
	lvars src, reg, src_reg, adjust;
	if isimm(src) then cvt_imm(src, reg) -> src endif;
	if isreg(src) then
		if src == USP then USP_adjust() endif;
		src -> reg;
	elseif isintegral(src) then
		plant2("li", use_reg(reg) ->> reg, src);
	else
		/* ismem(src) */
		false -> adjust;
		if isregind(src) then
			f_subv(1, src) -> src_reg;
			if src_reg == USP then
				USP_opnd(src) -> src;
			elseif f_subv(2, src) == true then
				;;; post increment
				if src_reg /== reg then opndsize(loadop) -> adjust endif;
				{% src_reg, 0 %} -> src;
			elseif f_subv(2, src) == false then
				;;; pre decrement
				plant3("subu", src_reg, src_reg, opndsize(loadop));
				{% src_reg, 0 %} -> src;
			endif;
		endif;
		plant2(loadop, use_reg(reg) ->> reg, src);
		if adjust then plant3("addu", src_reg, src_reg, adjust) endif;
	endif;
enddefine;

;;; do_load:
;;;		loads -src- to -reg-

define lconstant do_load(src, reg);
	lvars src, reg;
	returnif(src == reg);
	cvt_src(src, reg) -> src;
	if src /== reg then plant2("move", reg, src) endif;
	if reg == USP then 0 -> USP_offs endif;
enddefine;

;;; do_store:
;;;		stores -reg- to -dst-

define lconstant do_store(reg, dst);
	lvars reg, dst, dst_reg, adjust;
	if reg == USP then USP_adjust() endif;
	if isreg(dst) then
		unless reg == dst then
			plant2("move", dst, reg);
			if dst == USP then 0 -> USP_offs endif;
		endunless;
	elseif isimm(dst) then
		mishap(dst, 1, 'BAD DESTINATION OPERAND');
	else
		/* ismem(dst) */
		false -> adjust;
		if isregind(dst) then
			f_subv(1, dst) -> dst_reg;
			if dst_reg == USP then
				USP_opnd(dst) -> dst;
			elseif f_subv(2, dst) == true then
				;;; post increment
				opndsize(storeop) -> adjust;
				{% dst_reg, 0 %} -> dst;
			elseif f_subv(2, dst) == false then
				;;; pre decrement
				if dst_reg == reg then
					-opndsize(storeop) -> adjust;
					{% dst_reg, adjust %} -> dst;
				else
					plant3("subu", dst_reg, dst_reg, opndsize(storeop));
					{% dst_reg, 0 %} -> dst;
				endif;
			endif;
		endif;
		plant2(storeop, reg, dst);
		if adjust then plant3("addu", dst_reg, dst_reg, adjust) endif;
	endif;
enddefine;

define lconstant move_instr(loadop, storeop);
	lvars	src, dst;
	dlocal	loadop, storeop;
	explode(m_instr) -> dst -> src -> ;
	unless src = dst or src == USP_+ and dst == -_USP then
		if isreg(dst) then
			do_load(src, dst);
		else
			do_store(cvt_src(src, false), dst);
		endif;
	endunless;
enddefine;

define M_MOVE	= move_instr(% "lw",  "sw" %) enddefine;
define M_MOVEs	= move_instr(% "lhu", "sw" %) enddefine;
define M_MOVEss	= move_instr(% "lh",  "sw" %) enddefine;
define M_MOVEb	= move_instr(% "lbu", "sw" %) enddefine;
define M_MOVEsb = move_instr(% "lb",  "sw" %) enddefine;
define M_UPDs	= move_instr(% "lw",  "sh" %) enddefine;
define M_UPDb	= move_instr(% "lw",  "sb" %) enddefine;

define lconstant movebit_instr(routine);
	lvars size, offs, srcp, dst, routine;
	explode(m_instr) -> dst -> srcp -> offs -> size -> ;
	do_load(size, ARG_REG_2);
	do_load(offs, ARG_REG_1);
	do_load(srcp, ARG_REG_0);
	USP_adjust();
	plant1("jal", symlabel(routine));
	do_store(RESULT_REG_0, dst);
enddefine;

define M_MOVEbit	= movebit_instr(% "\^_bfield" %) enddefine;
define M_MOVEsbit	= movebit_instr(% "\^_sbfield" %) enddefine;

define M_UPDbit();
	lvars size, offs, dstp, src;
	explode(m_instr) -> src -> dstp -> offs -> size -> ;
	do_load(size, ARG_REG_2);
	do_load(offs, ARG_REG_1);
	do_load(dstp, ARG_REG_0);
	do_load(src,  ARG_REG_3);
	USP_adjust();
	plant1("jal", symlabel("\^_ubfield"));
enddefine;


/*
 *	Arithmetic and Logical Instructions
 */

;;; do_arith3:
;;;		3-operand arithmetic:
;;;			dst := src2 op src1

define lconstant do_arith3(src1, src2, dst, op);
	lvars src1, src2, dst, op, tmp;
	if isimm(src1) then
		cvt_imm(src1, false) -> src1;
	else
		cvt_src(src1, false) -> src1;
	endif;
	cvt_src(src2, false) -> src2;
	if isreg(dst) then dst -> tmp else tmp_reg() -> tmp endif;
	if isprocedure(op) then
		fast_apply(tmp, src2, src1, op);
	else
		plant3(op, tmp, src2, src1);
	endif;
	if tmp /== dst then do_store(tmp, dst) endif;
enddefine;

;;; do_parith3:
;;;		3-operand arithmetic on pop integers

define lconstant do_parith3(src1, src2, dst, op);
	lvars n, src2, src1, dst, op;
	if isintegral(src1) then
		src1 - 3 -> src1;
	elseif isintegral(src2) then
		if op == "addu" then
			src1, src2 - 3 -> src1 -> src2;
		elseif op == "subu"
		and (src2 <= 0 or integer_length(src2 + 3) fi_<= 31)
		then
			;;; adding 3 won't overflow
			src2 + 3 -> src2;
		else
			lvars tmp = cvt_src(src1, false);
			plant3("subu", reuse_reg(tmp) ->> src1, tmp, 3);
		endif;
	else
		lvars tmp = cvt_src(src1, false);
		plant3("subu", reuse_reg(tmp) ->> src1, tmp, 3);
	endif;
	do_arith3(src1, src2, dst, op);
enddefine;

;;; do_parith_test:
;;;		perform -op- on pop integers -src1- & -src2- and push the result.
;;;		Also test the result and jump to -lab- on -cond-.
;;;		-op- is assumed to be "addu" or "subu" only.

define lconstant do_parith_test(src1, src2, cond, lab, op);
	lvars src1, src2, cond, lab, op;

	define lconstant test(dst, src2, src1);
		lvars dst, src2, src1;
		plant3(op, dst, src2, src1);
		do_store(dst, -_USP);
		USP_adjust();
		if cond == "POS" then
			plant2("bgez", dst, lab);
		elseif cond == "NEG" then
			plant2("bltz", dst, lab);
		elseif cond == "OVF" then
			lvars lab1 = genlab();
			lvars tmp = tmp_reg();
			plant3("xor", tmp, src2, src1);
			plant2(if op == "addu" then "bltz" else "bgez" endif, tmp, lab1);
			plant3("xor", tmp, dst, src2);
			plant2("bltz", tmp, lab);
			plant1(LABEL, lab1);
		elseif cond == "NOVF" then
			lvars tmp = tmp_reg();
			plant3("xor", tmp, src2, src1);
			plant2(if op == "addu" then "bltz" else "bgez" endif, tmp, lab);
			plant3("xor", tmp, dst, src2);
			plant2("bgez", tmp, lab);
		else
			mishap(cond, 1, 'BAD CONDITION CODE');
		endif;
	enddefine;

	do_parith3(src1, src2, tmp_reg(), test);
enddefine;

define lconstant arith3_instr(op);
	lvars op;
	do_arith3(explode(m_instr), op) -> /*m_opcode*/;
enddefine;

define lconstant ptr_arith_instr(op);
	lvars op;
	do_arith3(explode(m_instr), op) -> /*type*/ -> /*m_opcode*/;
enddefine;

define lconstant parith3_instr(op);
	lvars op;
	do_parith3(explode(m_instr), op) -> /*m_opcode*/;
enddefine;

define lconstant parith_test_instr(op);
	lvars op;
	do_parith_test(explode(m_instr), op) -> /*m_opcode*/;
enddefine;

define M_ADD	= arith3_instr(% "addu" %) enddefine;
define M_SUB	= arith3_instr(% "subu" %) enddefine;
define M_MULT	= arith3_instr(% "mul"  %) enddefine;

define M_NEG();
	do_arith3(f_subv(2,m_instr), ZERO, f_subv(3,m_instr), "subu");
enddefine;

define M_BIS	= arith3_instr(% "or"  %) enddefine;
define M_BIM	= arith3_instr(% "and" %) enddefine;

define M_BIC();
	lvars src1, src2, dst;
	explode(m_instr) -> dst -> src2 -> src1 -> ;
	if isintegral(src1) then
		~~ src1 -> src1;
	else
		lvars tmp = cvt_src(src1, false);
		plant2("not", reuse_reg(tmp) ->> src1, tmp);
	endif;
	do_arith3(src1, src2, dst, "and");
enddefine;

define M_LOGCOM();
	do_arith3(f_subv(2,m_instr), ZERO, f_subv(3,m_instr), "nor");
enddefine;

define M_ASH();
	lvars shift, src, dst;

	define lconstant varshift(dst, src, shift);
		lvars shift, src, dst, lab1 = genlab(), lab2 = genlab();
		plant2("bgez", shift, lab1);
		lvars tmp = reuse_reg(shift);
		plant2("negu", tmp, shift);
		plant3("sra", dst, src, tmp);
		plant1("b", lab2);
		plant1(LABEL, lab1);
		plant3("sll", dst, src, shift);
		plant1(LABEL, lab2);
	enddefine;

	explode(m_instr) -> dst -> src -> shift ->;
	if not(isinteger(shift)) then
		do_arith3(shift, src, dst, varshift);
	elseif shift fi_< 0 then
		do_arith3(-shift, src, dst, "sra");
	else
		do_arith3(shift, src, dst, "sll");
	endif;
enddefine;

define M_PADD			= parith3_instr(% "addu" %) enddefine;
define M_PSUB			= parith3_instr(% "subu" %) enddefine;

define M_PADD_TEST		= parith_test_instr(% "addu" %) enddefine;
define M_PSUB_TEST		= parith_test_instr(% "subu" %) enddefine;

define M_PTR_ADD_OFFS	= ptr_arith_instr(% "addu" %) enddefine;
define M_PTR_SUB_OFFS 	= ptr_arith_instr(% "subu" %) enddefine;
define M_PTR_SUB		= ptr_arith_instr(% "subu" %) enddefine;


/*
 *	Compare and Test Instructions
 */

;;; do_cmp:
;;;		compare -src1- and -src2- and jump to -lab- on -cond-.

define lconstant do_cmp(src1, src2, cond, lab);
	lvars src1, src2, cond, lab, op;

	;;; Branch instructions for M-code conditions
	;;; NB: order of operands is reversed, so sense of test is reversed too
	define lconstant testop =
		newassoc([
			[EQ		beq]
			[NEQ	bne]
			[LT		bgt]
			[LEQ	bge]
			[GT		blt]
			[GEQ	ble]
			[ULT	bgtu]
			[ULEQ	bgeu]
			[UGT	bltu]
			[UGEQ	bleu]
			[POS	ble]
			[NEG	bgt]
		]);
	enddefine;

	define lconstant test(dst, src2, src1);
		lvars dst, src2, src1;
		USP_adjust();
		plant3(op, src2, src1, lab);
	enddefine;

	unless testop(cond) ->> op then
		mishap(cond, 1, 'BAD CONDITION CODE');
	endunless;
	do_arith3(src1, src2, ZERO, test);
enddefine;

;;; do_bit:
;;;		test for any bits of -mask- set in -src-

define lconstant do_bit(mask, src, cond, lab);
	lvars mask, src, cond, lab;
	lvars tmp = tmp_reg();
	do_arith3(mask, src, tmp, "and");
	do_cmp(tmp, ZERO, cond, lab);
enddefine;

define M_CMP();
	do_cmp(explode(m_instr)) -> /*m_opcode*/;
enddefine;

define M_PCMP();
	do_cmp(explode(m_instr)) -> /*m_opcode*/;
enddefine;

define M_PTR_CMP();
	do_cmp(explode(m_instr)) -> /*type*/ -> /*m_opcode*/;
enddefine;

define M_TEST();
	lvars src, cond, lab;
	explode(m_instr) -> lab -> cond -> src -> ;
	do_cmp(src, ZERO, cond, lab);
enddefine;

define M_BIT();
	do_bit(explode(m_instr)) -> /*m_opcode*/;
enddefine;

define M_CMPKEY();
	lvars src, cond, lab, lab1, key;
	explode(m_instr) -> lab -> cond -> src -> key -> ;
	if cond == "EQ" then genlab() else lab endif -> lab1;
	cvt_src(src, false) -> src;
	lvars tmp = tmp_reg();
	plant3("and", tmp, src, 1);
	USP_adjust();
	plant2("bnez", tmp, lab1);
	cvt_src({% src, field_##("KEY").wof %}, tmp) -> src;
	if isintegral(key) then
		;;; testing flag(s) nonzero in K_FLAGS field
		do_bit(key, {% src, field_##("K_FLAGS").wof %}, negate_test(cond), lab);
	else
		;;; test for specific key
		do_cmp(key, src, cond, lab);
	endif;
	if cond == "EQ" then plant1(LABEL, lab1) endif;
enddefine;


/*
 *	Branch Instructions
 */

define lconstant do_switch(src, labs, else_case, sysint);
	lvars src, labs, else_case, sysint, table_start, table_end, nlabs;
	genlab() -> table_start;
	genlab() -> table_end;
	listlength(labs) -> nlabs;
	cvt_src(src, false) -> src;
	USP_adjust();
	lvars tmp = tmp_reg();
	if sysint then
		plant3("sll", tmp, src, 2);
		plant3("bgtu", src, nlabs, table_end);
		plant2("lw", tmp, {% tmp, table_start %});
	else
		plant3("bgtu", src, popint(nlabs), table_end);
		plant2("lw", tmp, {% src, table_start<>'-3' %});
	endif;
	plant1("j", tmp);
#_IF DEF PIC
	;;; the jump table must be in the data segment
	plant0(RDATA);
#_ENDIF
	plant1(LABEL, table_start);
	plant1(WORD, table_end);
	lvars lab;
	fast_for lab in labs do
		plant1(WORD, lab);
	endfor;
#_IF DEF PIC
	;;; return to text segment for following code
	plant0(TEXT);
#_ENDIF
	plant1(LABEL, table_end);
	unless else_case then do_store(src, -_USP) endunless;
enddefine;


define M_BRANCH();
	USP_adjust();
	plant1("b", f_subv(2, m_instr));
enddefine;

define M_BRANCH_std();
	USP_adjust();
	plant1(SET, 'noreorder');
	plant1("b", f_subv(2, m_instr));
	plant0(NOP);
	plant1(SET, 'reorder');
enddefine;

define M_BRANCH_ON();
	do_switch(explode(m_instr), false) -> /*m_opcode*/;
enddefine;

define M_BRANCH_ON_INT();
	do_switch(explode(m_instr), true, true) -> /*m_opcode*/;
enddefine;


/*
 *	Procedure Call and Return
 */

define lconstant do_pop_call(op);
	lconstant PD_EXECUTE = field_##("PD_EXECUTE").wof;
	lvars op, target = f_subv(2, m_instr);
	if isimm(target) then
		;;; system procedure
		execlabof(cont(target), true) -> target;
#_IF DEF PIC
		lvars entry;
		if asm_alternate_entry(target) ->> entry then
			;;; defined within this file: the alternate entry point avoids
			;;; the overhead of resetting the context pointer
			entry -> target;
			;;; and a local branch avoids the overhead of setting the
			;;; call register
			if op == "j" then "b" else "bal" endif -> op;
		elseif op == "j" then
			;;; must go off the call register
			plant2("la", CALL_REG, target);
			CALL_REG -> target;
		endif;
#_ENDIF
	else
		unless isreg(target) then
			do_load(target, tmp_reg() ->> target);
		endunless;
		do_load({% target, PD_EXECUTE %}, CALL_REG ->> target);
	endif;
	USP_adjust();
	plant1(op, target);
enddefine;

define lconstant do_nonpop_call(op);
	lvars op, target = f_subv(2, m_instr);
	if ismem(target) then
		do_load(target, CALL_REG ->> target);
#_IF DEF PIC
	elseunless isimm(target) and op == "jal" then
		;;; must go off the call register
		do_load(target, CALL_REG);
		if isimm(target) then CALL_REG -> target endif;
#_ENDIF
	endif;
	USP_adjust();
	plant1(op, target);
enddefine;

define M_CALL	= do_pop_call(% "jal" %) enddefine;
define M_CHAIN	= do_pop_call(% "j"   %) enddefine;

define M_CALL_WITH_RETURN();
#_IF DEF PIC
	;;; This instruction is called at the end of _chainfrom_caller,
	;;; after the current and caller stack frames have been unwound. So
	;;; we can't rely on the value of the context pointer, and have to
	;;; reset it from scratch.
	lvars lab = genlab();
	plant1(SET, 'noreorder');
	plant1("bal", lab);
	plant0(NOP);
	plant1(LABEL, lab);
	plant1(CPLOAD, RA);
	plant1(SET, 'reorder');
#_ENDIF
	do_load(f_subv(3, m_instr), RA);
	do_pop_call("j");
enddefine;

define M_CALLSUB();
	lvars l = datalength(m_instr);
	if l  ==   5 then do_load(m_instr(3),   ARG_REG_2) endif;
	if l fi_>= 4 then do_load(m_instr(l-1), ARG_REG_1) endif;
	if l fi_>= 3 then do_load(m_instr(l),   ARG_REG_0) endif;
	do_nonpop_call("jal");
enddefine;

define M_CHAINSUB = do_nonpop_call(% "j" %) enddefine;

define M_RETURN();
	USP_adjust();
	plant1("j", RA);
enddefine;


/*
 *	Prolog Instructions
 */

define M_PLOG_IFNOT_ATOM();
	plant2("bnez", RESULT_REG_0, f_subv(2, m_instr));
enddefine;

define M_PLOG_TERM_SWITCH();
	plant2("bgtz", RESULT_REG_0, f_subv(3, m_instr));
	plant2("bltz", RESULT_REG_0, f_subv(2, m_instr));
	;;; dereferenced item left in ARG_REG_0
	do_store(ARG_REG_0, f_subv(4, m_instr));
enddefine;


/*
 *	Special Instructions
 */

define lconstant do_cpload(entry);
	lvars entry;
#_IF DEF PIC
	;;; set the context pointer on entry to a procedure
	plant1(SET, 'noreorder');
	plant1(CPLOAD, CALL_REG);
	plant1(SET, 'reorder');
	;;; ... and create an alternate entry point which avoids it
	lvars alt_entry = genlab();
	plant1(LABEL, alt_entry);
	alt_entry -> asm_alternate_entry(entry);
#_ENDIF
enddefine;

define M_CLOSURE();
	lvars (, frozvals, pdpart_opnd) = explode(m_instr);
	lvars nfroz = listlength(frozvals);
	do_cpload(current_pdr_exec_label);
	unless nfroz == 0 and pdpart_opnd then
		;;; get own address
		lvars self = tmp_reg();
		plant2("la", self, current_pdr_label);
		if nfroz fi_<= 16 then
			;;; push the frozvals and chain the pdpart
			lvars offs = field_##("PD_CLOS_FROZVALS").wof;
			lvars t1 = tmp_reg(), t2 = tmp_reg();
			fast_repeat nfroz times
				do_load({^self ^offs}, t1);
				do_store(t1, -_USP);
				offs fi_+ WORD_OFFS -> offs;
				(t1, t2) -> (t2, t1);
			endrepeat;
			unless pdpart_opnd then
				{^self ^(field_##("PD_CLOS_PDPART").wof)} -> pdpart_opnd;
			endunless;
		else
			;;; pass closure to Exec_closure
			do_store(self, -_USP);
			perm_const_opnd([Sys Exec_closure]) -> pdpart_opnd;
		endif;
	endunless;
	dlocal m_instr = {% M_CHAIN, pdpart_opnd %};
	M_CHAIN();
enddefine;

define M_SETSTKLEN();
	lvars sl, offs, lab = genlab();
	explode(m_instr) -> sl -> offs -> ;
	;;; compute desired stack pointer:
	;;; subu a0, userhi, sl
	do_arith3(sl, identlabel("\^_userhi"), ARG_REG_0, "subu");
	;;; subu a0, a0, offs-3
	;;; (-3 accounts for the popint bits in sl)
	do_arith3(offs-3, ARG_REG_0, ARG_REG_0, "subu");
	;;; compare desired and actual stack pointers:
	;;; if equal, jump to end, otherwise call "setstklen_diff" to fix
	USP_adjust();
	plant3("beq", ARG_REG_0, USP, lab);
	plant1("jal", symlabel("\^_setstklen_diff"));
	plant1(LABEL, lab);
enddefine;


/*
 *	Miscellaneous Instructions
 */

define M_LABEL();
	USP_adjust();
	plant1(LABEL, f_subv(2, m_instr));
enddefine;

define M_ERASE();
	lvars opnd = f_subv(2, m_instr);
	if isregind(opnd) then
		if f_subv(1, opnd) == USP then
			USP_opnd(opnd) -> ;
		else
			do_load(opnd, ZERO);
		endif;
	endif;
enddefine;

define M_END();
enddefine;


/*
 *	Stack Frame Instructions
 */

lblock;

lvars
	frame_length,
	reg_spec_id,
	reg_locals,
	Npopregs,
	dlocal_labs,
	Nstkvars,
	Npopstkvars,
;

define M_CREATE_SF();
	lvars i, offs, regmask;

	explode(m_instr)
		-> reg_spec_id
		-> dlocal_labs
		-> Npopstkvars
		-> Nstkvars
		-> Npopregs
		-> reg_locals
		-> ;

	;;; Set the context pointer
	do_cpload(current_pdr_exec_label);

	;;; Compute the size of the stack frame:
	(listlength(reg_locals) + listlength(dlocal_labs) + Nstkvars + 2).wof
		-> frame_length;

	;;; Adjust the stack pointer appropriately
	plant3("subu", SP, SP, frame_length);

	;;; Plant debugging information:
	true -> testbit(0, regnumber(RA)) -> regmask;
	fast_for i in reg_locals do
		true -> testbit(regmask, i) -> regmask;
	endfor;
	plant3(FRAME, SP, frame_length, RA);
	plant2(MASK, regmask, -4);

	;;; The procedure register spec is that part of the debugger's register
	;;; mask relating to pop and non-pop register lvars
	;;; (6 registers, first is number 16)
	(regmask >> 16) fi_&& 2:111111 -> idval(reg_spec_id);

	;;; Starting from the top of the stack frame ...
	frame_length -> offs;

	;;; ... save the return address
	plant2("sw", RA, {% SP, offs - WORD_OFFS ->> offs %});

	;;; ... save registers
	syssort(reg_locals, nonop fi_>) -> reg_locals;
#_IF DEF PIC
	;;; first should be the context pointer (always saved)
	unless f_hd(reg_locals) == #_< regnumber(GP) >_# then
		mishap(0, 'SYSTEM ERROR IN M_CREATE_SF (no context pointer)');
	endunless;
	plant1(CPRESTORE, offs - WORD_OFFS ->> offs);
	f_tl(reg_locals) -> reg_locals;
#_ENDIF
	fast_for i in reg_locals do
		plant2("sw", reglabel(i), {% SP, offs - WORD_OFFS ->> offs %});
	endfor;

	;;; ... save dynamic locals, alternating registers to increase
	;;; opportunities for interleaving
	lvars t1 = tmp_reg(), t2 = tmp_reg();
	fast_for i in dlocal_labs do
		plant2("lw", t1, i);
		plant2("sw", t1, {% SP, offs - WORD_OFFS ->> offs %});
		(t1, t2) -> (t2, t1);
	endfor;

	;;; ... and initialise pop on-stack lvars to popint 0
	unless Npopstkvars == 0 then
		plant2("li", tmp_reg() ->> t1, 3);
		fast_repeat Npopstkvars times
			plant2("sw", t1, {% SP, offs - WORD_OFFS ->> offs %});
		endrepeat;
	endunless;

	;;; Set and save the procedure base register
	plant2("la", PB, current_pdr_label);
	plant2("sw", PB, {% SP, 0 %});
enddefine;

define M_UNWIND_SF();
	lvars offs, i;

	;;; Starting from the top of the stack frame ...
	frame_length -> offs;

	;;; ... restore the return address
	plant2("lw", RA, {% SP, offs - WORD_OFFS ->> offs %});

	;;; ... restore other registers
#_IF DEF PIC
	;;; (ignoring GP)
	offs fi_- WORD_OFFS -> offs;
#_ENDIF
	fast_for i in reg_locals do
		plant2("lw", reglabel(i), {% SP, offs - WORD_OFFS ->> offs %});
	endfor;

	;;; ... and restore dynamic locals
	lvars t1 = tmp_reg(), t2 = tmp_reg();
	fast_for i in dlocal_labs do
		plant2("lw", t1, {% SP, offs - WORD_OFFS ->> offs %});
		plant2("sw", t1, i);
		(t1, t2) -> (t2, t1);
	endfor;

	;;; Reset the stack pointer
	plant3("addu", SP, SP, frame_length);

	;;; Restore the procedure base register from the previous frame
	plant2("lw", PB, {% SP, 0 %});
enddefine;

endlblock;


;;; -- GENERATING OUTPUT CODE ---------------------------------------------

define lconstant codetrans(m_codelist) -> codelist;
	lvars	opcode, m_codelist;
	dlocal	m_instr, codelist, codelist_end, USP_offs = 0,
			tmp_regs = [], n_tmp_regs_used;
	;;; initialise output code list
	[{% ENT, current_pdr_exec_label %}{% LABEL, current_pdr_exec_label %}]
		-> codelist;
	f_tl(codelist) -> codelist_end;
	;;; translate each m-code instruction
	for m_instr in m_codelist do
		0 -> n_tmp_regs_used;
		f_subv(1, m_instr) -> opcode;
		if isprocedure(opcode) then
#_IF DEF M_DEBUG
			;;; include M-code instruction as a comment
			plant(COMMENT, destvector(m_instr) fi_+ 1);
#_ENDIF
			fast_apply(opcode);
		else
			mishap(opcode, 1, 'UNKNOWN M-OPCODE');
		endif;
	endfor;
	;;; end output code list
	[{% END, current_pdr_exec_label %}] -> f_tl(codelist_end);
enddefine;

;;; outopnd:
;;;		write out an operand of an instruction

define lconstant outopnd(opnd);
	lvars opnd, dis;

	define lconstant extend(n);
		lconstant mask = -1 << 32;
		lvars n;
		if testbit(n, 31) then n || mask else n &&~~ mask endif;
	enddefine;

	if isref(opnd) then
		fast_cont(opnd), '%p';
	elseif isreg(opnd) then
		asm_regname(opnd), '%p';
	elseif isregind(opnd) then
		f_subv(2, opnd) -> dis;
		asm_regname(f_subv(1, opnd)) -> opnd;
		if dis == 0 then
			opnd, '(%p)';
		else
			opnd, dis, '%p(%p)';
		endif;
	elseif isbiginteger(opnd) then
		extend(opnd), '%p';
	else
		opnd, '%p';
	endif,
	asmf_printf();
enddefine;

;;; outinst:
;;;		write out an assembler instruction

define lconstant outinst(instr);
	lvars i, instr, l, opcode;
	f_subv(1, instr) -> opcode;
	if opcode == LABEL then
		outlab(f_subv(2, instr));
	elseif opcode == COMMENT then
		asmf_printf(pdprops(f_subv(2, instr)), opcode, '\s%p\t%p');
		fast_for i from 3 to datalength(instr) do
			asmf_printf(f_subv(i, instr), ' %p');
		endfor;
		asmf_charout(`\n`);
	else
		asmf_printf(opcode, '\t%p\t');
		datalength(instr) -> l;
		unless l == 1 then
			fast_for i from 2 to l fi_- 1 do
				outopnd(f_subv(i, instr));
				asmf_charout(`,`);
			endfor;
			outopnd(f_subv(l, instr));
		endunless;
		asmf_charout(`\n`);
	endif;
enddefine;

define mc_code_generator(codelist, hdr_len) -> (gencode, pdr_len);
	lvars codelist, hdr_len, pdr_len;
	lconstant procedure gencode;

	;;; translate to assembler
	codetrans(codelist) -> codelist;

#_IF DEF PIC
	;;; procedure length is header-length only --- code is elsewhere
	hdr_len -> pdr_len;
#_ELSE
	;;; procedure length is header-length + code-length, computed as the
	;;; difference between the code end and the code start
	lvars startlab, endlab;
	genlab() -> startlab;
	genlab() -> endlab;
	genlab() -> pdr_len;
#_ENDIF

	;;; produce code generator
	define lconstant gencode();
#_IF DEF PIC
		;;; switch to text section
		asmf_printf('\t.text\n');
		;;; output code
		applist(codelist, outinst);
		;;; return to previous section
		asmf_printf(asm_current_section, '\t.%p\n');
#_ELSE
		;;; plant start label
		outlab(startlab);
		;;; output code
		applist(codelist, outinst);
		;;; plant end label and value for pdr_len
		outlab(endlab);
		outlabset(pdr_len, '(('<>endlab<>'-'<>startlab<>')>>2)+'sys_>< hdr_len);
#_ENDIF
	enddefine;
enddefine;


;;; -- OTHER DEFINITIONS NEEDED BY "m_trans.p" ----------------------------


/*
 *	M-code tables for machine-dependent in-line subroutines.
 *	These are added to the corresponding properties in m_trans.p
 */

constant

	mc_inline_procs_list = [
		[ \^_ptr_to_offs	[{^M_ERASE ^USP_+}]]
		[ \^_offs_to_ptr	[{^M_ERASE ^USP_+}]]
		[ \^_int			[{^M_ASH -2 ^USP_+ ^ -_USP}]]
		[ \^_pint			[{^M_ASH  2 ^USP_+ ^ -_USP}
							 {^M_ADD 3 ^USP_+ ^ -_USP}]]
		[ \^_por			[{^M_BIS ^USP_+ ^USP_+ ^ -_USP}]]
		[ \^_pand			[{^M_BIM ^USP_+ ^USP_+ ^ -_USP}]]
		[ \^_mksimple		[{^M_ADD 1 ^USP_+ ^ -_USP}]]
		[ \^_mkcompound		[{^M_SUB 1 ^USP_+ ^ -_USP}]]
		[ \^_mksimple2		[{^M_ADD 3 ^USP_+ ^ -_USP}]]
		[ \^_mkcompound2	[{^M_SUB 3 ^USP_+ ^ -_USP}]]
	],

	mc_inline_conditions_list = [
		[ \^_iscompound		{^M_BIT  2:01   ^USP_+ EQ  ?}]
		[ \^_issimple		{^M_BIT  2:01   ^USP_+ NEQ ?}]
		[ \^_issimple2		{^M_BIT	 2:10	^USP_+ NEQ ?}]
		[ \^_isinteger		{^M_BIT  2:10   ^USP_+ NEQ ?}]
		[ \^_isaddress		{^M_BIT  2:11   ^USP_+ EQ  ?}]
	],
;


	/*	Procedure to convert a pop integer subscript to an appropriate offset
		for the data type being accessed, used by OP_SUBV in m_trans.p to
		compile code for fast_subscrv, vectorclass field accesses, etc.
			-scale- is the scale for the data type involved; the results are
		the M-code instructions (if any) necessary to convert the subscript
		on top of the stack to an offset, plus a constant correction to be
		added.
	*/
define cvt_pop_subscript(scale);
	lvars pow, scale;
	if is_power2(scale) ->> pow then
		;;; pow-2 accounts for popint being shifted left 2
		unless (pow-2 ->> pow) == 0 then
			{^M_ASH ^pow ^USP_+ ^ -_USP}
		endunless,
		-(popint(0) << pow)		;;; additive correction to remove popint bits
	else
		;;; just convert to sysint and multiply
		{^M_ASH -2 ^USP_+ ^ -_USP},		;;; _int()
		{^M_MULT ^scale ^USP_+ ^ -_USP},
		0 						;;; no correction necessary
	endif
enddefine;


endsection;		/* Genproc */

endsection;		/* $-Popas$-M_trans */


/* --- Revision History ---------------------------------------------------
--- John Gibson, Aug 31 1994
		Removed M_AR*RAY_SUB (no longer necessary)
--- Robert John Duncan, Mar 22 1994
		Changed the pop calling convention yet again: the procedure address
		does not have to be passed in ARG_REG_0. Also, system code is no
		longer copyable, so doesn't have to be relocatable; specifically,
		closure code can use absolute addresses to obtain frozvals, etc.
		(which is why the procedure address is no longer needed).
			Modified the strategy for allocation of temporary registers, to
		cycle through the register set: should offer the assembler more
		opportunities for code reordering.
			Changed switch code to place the jump table in the data segment
		because it doesn't work otherwise when generating position-
		independent code.
--- Robert John Duncan, Mar 15 1994
		Changes for position-independent code. Also returned to the
		previous convention that pop calls should pass the procedure address
		in ARG_REG_0, but now in addition to the execute address in CALL_REG
		and for system procedures as well as user ones.
--- Robert John Duncan, Mar  9 1994
		Reassigned register t8 to be the SPECIAL_VAR_BLOCK_REG, removing
		any dependancy on the global pointer. This is a caller-save register,
		so will need special handling in the assembly code routines.
--- Robert John Duncan, Mar  7 1994
		Changed for new pop calling convention: instead of the procedure
		address being passed in ARG_REG_0, the execute address is passed
		in new CALL_REG (t9). Also, all indirect calls now go through
		CALL_REG. Required some reorganisation of register usage, with
		t8/t9 previously used as work registers, now reassigned to t5/t6.
--- Robert John Duncan, Feb 11 1993
		Fixed the code generated by M_CLOSURE for copyable closures which
		can't assume that the closure address will be in ARG_REG_0.
--- Robert John Duncan, Oct 15 1992
		Changed M_CLOSURE to call Exec_closure for more than 16 frozvals
--- Robert John Duncan, Jan  9 1992
		Changed M_CMPKEY to allow key arg to be an integer specifying flag(s)
		to be tested in K_FLAGS field.
--- John Gibson, Jan  8 1992
		Added _por and _pand to mc_inline_procs_list
--- Robert John Duncan, Apr 17 1991
		Changed to use hardware register names except when M_DEBUG is <true>
		(reduces dependency on "pop_regdef.h")
 */
