/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_shell.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 9th May 1993
;;; File: go_shell.p
;;; Author: B Rabau (based on original work of I. Rogers)

compile_mode :pop11 +strict;


;;; INTRO: The OBJECTCLASS go_shell represents a physical window in which
;;; INTRO: a number of graphical panes can be placed. Multiple GO pane objects
;;; INTRO: can be added to the shell which will manage their visibility and
;;; INTRO: their relative location (see also go_add_to_shell).
;;; INTRO: The global variable "go_shells" holds all currently active instances
;;; INTRO: of the go_shell class in GO.

uses go_located;
include xt_constants;
include xpt_constants;

;;;------------------------------------------------------
vars procedure go_initialise_shell;   ;;; defined below
vars procedure go_add_to_shell;       ;;; defined below...


vars go_shells = [];   ;;; Global var: all currently used shell objects in GO

define :class go_shell;
	isa go_located;
	slot the_go_panes = [];
;;; REF: Handles to all go_panes in this shell (REF * GO_PANE).
	slot the_go_shell = false;
;;; REF: Handle to the shell's Widget.
	slot the_go_board = false;
;;; REF: Handle to the shell's Bulletin Board Widget.
	slot stored_go_bounding_width  = 500;
;;; REF: Stored value of horizontal size (screen coord = world coord)
	slot stored_go_bounding_height = 500;
;;; REF: Stored value of vertical size (screen coord = world coord)

	on new do procedure() -> shell;
			  lvars pane, shell = apply();
				  go_initialise_shell( shell );

				  newgo_window_pane() -> pane;
				  go_add_to_shell(
					 0, 0,
					 go_bounding_width(shell), go_bounding_height(shell),
					 pane, shell );

			  endprocedure;

enddefine;

exload_batch;
	lconstant
			go_app    = XptWidgetSet("Toolkit")("ApplicationShellWidget"),
			go_gfx    = XptWidgetSet("Poplog")("GraphicWidget"),
			go_snm    = 'GO Shell',
			go_pnm    = 'GO Pane',
			go_gnm    = 'GO XpwGraphics';
#_IF XLINK_TYPE = 'motif'
	lconstant
			go_bul    = XptWidgetSet("Motif")("BulletinBoardWidget");
#_ELSEIF XLINK_TYPE = 'openlook'
	lconstant
			go_bul    = XptWidgetSet("OpenLook")("BulletinBoardWidget");
#_ELSE_ERROR
#_ENDIF
endexload_batch;

define :method go_initialise_shell( shell: go_shell );
lvars shell;
lvars the_shell, the_board;
;;; REF: go_initialise_shell( SHELL );
;;; REF: Create the physical window of a new shell object.
;;; REF: This will call upon the Xt toolkit to create the shell (see also
;;; REF: HELP * XtVaAppCreateShell) and a bulletin board inside it (see also
;;; REF: HELP * BulletinBoardWidget). The latter will provide a composite
;;; REF: widget in which the panes (see slot "the_go_panes" can be placed on
;;; REF: specific locations like a board on which you can pin papers. It also
;;; REF: provides a simple geometry management which handles locations when
;;; REF: a pane is removed from the shell. The exact behaviour depends on the
;;; REF: window manager which is used (currently OPENLOOK and MOTIF have been
;;; REF: foreseen).
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	XtVaAppCreateShell(
		go_snm, 'GO', go_app, XptDefaultDisplay,
		XptVaArgList([
			% consXptArgPtr('title',go_snm), consXptArgPtr('iconName',go_snm) %
			{allowShellResize  ^true}
			{mappedWhenManaged ^false}
		])
	) ->> the_go_shell(shell) -> the_shell;

	XtVaCreateManagedWidget(go_pnm, go_bul, the_shell, XptVaArgList([])
	) ->>  the_go_board(shell) -> the_board;

#_IF DEF MOTIF
	;;; The following is only for MOTIF: margins are 10 by default: not needed
	0 -> XptValue( the_board, XtN marginWidth,  TYPESPEC(:XptShort) );
	0 -> XptValue( the_board, XtN marginHeight, TYPESPEC(:XptShort) );
#_ENDIF

	XtRealizeWidget(the_shell);
	true -> XptBusyCursorFeedback(the_shell);
	true -> XptGarbageCursorFeedback(the_shell);

	;;; Forget any old references; replace by new object
	delete( shell, go_shells, nonop ==) -> go_shells;
	shell :: go_shells -> go_shells;
enddefine;

;;;------------------------------------------------------
;;; MEMBERSHIP FUNCTIONS (CHILDREN = GO_WINDOW_PANES)

define :method go_add_to_shell(x, y, w, h, pane :go_pane, shell :go_shell);
lvars x, y, w, h, pane, shell;
;;; REF: go_add_to_shell( X, Y, W, H, PANE, SHELL );
;;; REF: Add the given pane as a new pane with given dimensions to the shell.
;;; REF: This creates a new XpwGraphic widget and attaches it to the pane which
;;; REF: is then added to the shell (see also REF * GO_PANE).
;;; REF: X/Y integer screen coord of pane relative to top-left corner of shell
;;; REF: W/H integer width and height in screen coord of new pane.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
lvars window_pane;
	XtVaCreateManagedWidget(
		go_gnm, go_gfx, the_go_board( shell),
		XptVaArgList([
			{x         ^x}
			{y         ^y}
			{width     ^w}
			{height    ^h}
		])
	) -> window_pane;
	window_pane -> the_go_window( pane );
	the_go_panes( shell ) <> [^pane] -> the_go_panes(shell);
	true -> go_motionhint( pane );
enddefine;

define :method go_add_to( pane :go_pane, shell :go_shell );
lvars pane, shell;
;;; REF: go_add_to( PANE, SHELL );
;;; REF: Make the given pane visible in the given shell. This can only be
;;; REF: done if the pane is an element of the shell (see also go_add_to_shell).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	if member( pane, the_go_panes(shell) ) then
		 go_redraw( pane );
	else
		 mishap('Use go_add_to_shell to make the pane belong to the shell',
				[% pane, shell %]);
	endif;
enddefine;

define :method go_annex_to( pane :go_pane, shell :go_shell );
lvars pane, shell;
;;; REF: go_add_to( PANE, SHELL );
;;; REF: This is exactly the same as REF * GO_SHELL/go_add_to.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	go_add_to( pane, shell );
enddefine;

define :method go_remove_from( pane :go_window_pane, shell :go_shell );
lvars pane, shell;
;;; REF: go_remove_from( PANE, SHELL );
;;; REF: Removes the pane from the shell, and also detroys the object since
;;; REF: the pane cannot exist outside its parent shell.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	delete( pane, the_go_panes( shell ), nonop ==) -> the_go_panes( shell );
	go_destroy_object( pane );
enddefine;

;;;------------------------------------------------------
;;; VISIBILITY:

define :method go_show( shell :go_shell );
lvars shell;
;;; REF: go_show( SHELL );
;;; REF: Make the given shell visible on the screen.
;;; REF: Note that the shell has no parent in which it is visualized.
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
lvars the_shell = the_go_shell( shell );
	if ( the_shell ) then
		 ;;; should really do XRaiseWindow on the_go_shell( shell )
		 XtUnmapWidget( the_shell );
		 XtMapWidget( the_shell );
	endif;
enddefine;

define :method go_hide( shell :go_shell );
lvars shell;
;;; REF: go_hide( SHELL );
;;; REF: Make the given shell invisible on the screen.
;;; REF: Note that the shell has no parent in which it is visualized.
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
lvars the_shell = the_go_shell( shell );
	if ( the_shell  ) then XtUnmapWidget( the_shell ); endif;
enddefine;


;;;------------------------------------------------------
;;; DIMENSIONS

define :method go_window_xloc( shell :go_shell );
lvars shell, window=the_go_shell(shell);
;;; REF: go_window_xloc( SHELL ) -> INT;
	if ( window ) then
		XptValue( window, XtN x, TYPESPEC(:XptShort));
	else
		0
	endif;
enddefine;

define :method updaterof go_window_xloc( x, shell :go_shell );
lvars x, shell, window=the_go_shell(shell);
;;; REF: INT -> go_window_xloc( SHELL );
;;; REF: Horizontal screen coordinate (on the display) from top-left hand
;;; REF: corner of the screen.
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	if ( window ) then
		abs(x) -> XptValue( window, XtN x, TYPESPEC(:XptShort));
	endif;
enddefine;

define :method go_window_yloc( shell :go_shell );
lvars shell, window=the_go_shell(shell);
;;; REF: go_window_yloc( SHELL ) -> INT;
	if ( window ) then
		XptValue( window, XtN y, TYPESPEC(:XptShort));
	else
		0
	endif;
enddefine;

define :method updaterof go_window_yloc( y, shell :go_shell );
lvars y, shell, window=the_go_shell(shell);
;;; REF: INT -> go_window_xloc( SHELL );
;;; REF: Vertical screen coordinate (on the display) of the shell calculated
;;; REF: from top-left hand corner.
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	if ( window )  then
		abs(y) -> XptValue( window, XtN y, TYPESPEC(:XptShort));
	endif;
enddefine;


define :method go_window_width( shell :go_shell );
lvars shell, window=the_go_shell(shell);
;;; REF: go_window_width( SHELL ) -> INT;
	if ( window ) then
		XptValue( window, XtN width, TYPESPEC(:XptShort));
	else
		0
	endif;
enddefine;

define :method updaterof go_window_width( w, shell :go_shell );
lvars w, shell, window=the_go_shell(shell);
;;; REF: INT -> go_window_width( SHELL );
;;; REF: The physical screen width (on the display) of the shell.
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	if ( window ) then
		abs(w) -> XptValue( window, XtN width, TYPESPEC(:XptShort));
	endif;
enddefine;

define :method go_window_height( shell :go_shell );
lvars shell, window=the_go_shell(shell);
;;; REF: go_window_height( SHELL ) -> INT;
	if ( window ) then
		XptValue( window, XtN height, TYPESPEC(:XptShort));
	else
		0
	endif;
enddefine;

define :method updaterof go_window_height( h, shell :go_shell );
lvars h, shell, window=the_go_shell(shell);
;;; REF: INT -> go_window_height( SHELL );
;;; REF: The physical screen height (on the display) of the shell.
;;; REF: SHELL is an instance of the go_shell class (REF * GO_SHELL).
	if ( window )  then
		abs(h) -> XptValue( window, XtN height, TYPESPEC(:XptShort));
	endif;
enddefine;

;;;----------------------------------------------------------------
;;; COPY

define :method go_copy_object( shell :go_shell ) -> new_shell;
lvars shell, new_shell;
;;; REF: go_copy_object( SHELL ) -> SHELL;
;;; REF: Make a new shell object which is a complete copy of given shell.
;;; REF: This copy will contain its own Graphics panes (go_window_pane) and
;;; REF: the objects in the panes of the original shell are also copied
;;; REF: They are not shared: they will move independently if dragged...
;;; REF: The algorithm used will first create a copy in the original pane,
;;; REF: then change its visibility from the old to the new pane. This is
;;; REF: safer if the_go_components is method rather than a slot.
lvars the_panes = shell.the_go_panes, the_shell = shell.the_go_shell;
lvars pane, obj, new_pane, new_obj, old_pane, old_panes;
	;;; Prevent copying uncopyable stuff
	false -> the_go_shell( shell );
	[] -> the_go_panes( shell );
	copy( shell ) -> new_shell;
	;;; Reinstate the original values:
	the_panes -> the_go_panes( shell );
	the_shell -> the_go_shell( shell );
	;;; Making the new shell:
	go_initialise_shell( new_shell );
	for pane in the_panes do
		class_new( datakey(pane) )() -> new_pane;
		go_add_to_shell( pane.go_window_xloc,  pane.go_window_yloc,
						 pane.go_window_width, pane.go_window_height,
						 new_pane, new_shell );
		for obj in pane.the_go_components do
			;;; avoid getting the pane which is a default visible element ??
			unless ( obj == pane ) then
				go_visible_in(obj) -> old_panes;
				go_copy_object( obj ) -> new_obj;
				for old_pane in old_panes do
					go_remove_from( new_obj, old_pane );
				endfor;
				go_add_to( new_obj, new_pane );
			endunless;
		endfor;
		go_redraw( new_pane );
	endfor;
enddefine;

;;;------------------------------------------------------
;;; DESTROY: tentative definition for destroying a shell

define :method go_destroy_object( shell :go_shell );
lvars shell;
;;; REF: go_destroy_object( SHELL );
;;; REF: Makes the shell invisible, and destroys the shell and all its panes.
;;; REF: First all panes are destroyed with go_destroy_object( PANE ), then
;;; REF: the physical widget representing the shell (see slot "the_go_shell")
;;; REF: is destroyed (see XtDestroyWidget).
lvars all_panes = the_go_panes( shell ), widget = the_go_shell(shell), pane;
	go_hide( shell );
	for pane in all_panes do
		go_destroy_object( pane );
	endfor;
	delete( shell, go_shells, nonop ==) -> go_shells;
	false -> the_go_shell( shell );
	false -> the_go_board( shell );
	if ( widget ) then
		;;; Destroy the widget itself. Note that Unrealize would only make
		;;; it dissapear from the screen (waiting for the next Realize).
		XtDestroyWidget( widget );
	endif;
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_shell = true;

/* --- Revision History --------------------------------------------
 * JJC 06/10/93
 *      Removed redundant code.
 * BR 09/05/93
 *     Replaced calls to go_refresh_pane() with go_redraw().
 * BR 28/04/93
 *     Changed usage of go_add_to_shell() and added comments in file.
 */
;;; eof
