/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_window_pane.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 24th Aug 1993
;;; File: go_window_pane.p
;;; Author: B Rabau (based on original work of J L Cunningham)

compile_mode :pop11 +strict;

uses go_pane;
uses go_xdefaults;

include xpt_coretypes;
include XpwPixmap;

;;; INTRO: The CLASS go_window_pane allows connection of GO to a XpwGraphic
;;; INTRO: widget. See HELP * XpwGraphic for more details on this graphical
;;; INTRO: X-windows widget. Graphical Objects can be created, visualised and
;;; INTRO: manipulated in these panes.


;;;------------------------------------------------------
define :class go_window_pane;
	isa go_pane;
	slot go_live_object       == false;
;;; REF: false if in edit mode, true if live or list of editor objects.
	slot stored_go_xscale     ==  1;
;;; REF: The window pane's horizontal compression for all objects.
	slot stored_go_yscale     == -1;
;;; REF: The window pane's vertical compression for all objects.
	slot stored_go_fgcolour   == false;
;;; REF: The default foreground colour, uses LIB * GO_XDEFAULTS
	slot stored_go_bgcolour   == false;
;;; REF: The default background colour, uses LIB * GO_XDEFAULTS
	slot go_auto_resize       == false;
;;; REF: Automatically resize the window if the pane is scaled
	slot stored_go_batch_mode == false;
;;; REF: Boolean: whether in batch mode drawing or not.

enddefine;


;;;------------------------------------------------------
;;; WINDOW CONNECTION

vars procedure go_window_width;         ;;; DECLARED BELOW...
vars procedure go_window_height;        ;;; DECLARED BELOW...
vars procedure go_add_notice_button;    ;;; as defined in go_xaction.p
vars procedure go_add_notice_key;       ;;; as defined in go_xaction.p
vars procedure go_remove_notice_button; ;;; as defined in go_xaction.p
vars procedure go_remove_notice_key;    ;;; as defined in go_xaction.p
vars procedure go_add_cap_style;        ;;; as defined in go_xdraw.p

define :method updaterof the_go_window( win, pane :go_window_pane );
lvars win, pane;
;;; REF: XPWGRAPHIC_OR_FALSE -> the_go_window( WINDOW_PANE );
;;; REF: If not false then attach XpwGraphic widget and callback routines to
;;; REF: the given pane, else detach the XpwGraphic widget from pane and
;;; REF: detach all remaining callback from the widget.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	;;; CALLBACK EVENTS (BUTTON & MOTION)
	if win then
		win -> pane.stored_go_window;
		go_fgcolour( pane ) -> stored_go_fgcolour( pane ) ;
		go_bgcolour( pane ) -> stored_go_bgcolour( pane ) ;
		go_add_notice_button( pane );    ;;; attach mouse actions
		go_add_notice_key( pane );       ;;; attach mouse actions
		go_add_cap_style( pane );        ;;; for thick line joins (rects)
	else
		go_remove_notice_button( pane ); ;;; detach mouse actions
		go_remove_notice_key( pane );    ;;; detach mouse actions
	false -> pane.stored_go_window;
	endif;
enddefine;


;;;------------------------------------------------------
;;; LINE APPEARANCE

define :method go_linestyle( pane :go_window_pane );    ;;; the default style
lvars pane;
;;; REF: go_linestyle( WINDOW_PANE ) -> INTEGER;
		XptValue( the_go_window(pane), XtN lineStyle, "int" );
enddefine;

define :method updaterof go_linestyle( int, pane :go_window_pane);
lvars pane;
;;; REF: go_linestyle( WINDOW_PANE ) -> INTEGER;
;;; REF: The current linestyle used by the_go_window in this pane and
;;; REF: redraws the objects in the pane with this new linestyle (see also
;;; REF: REF * XtNlineStyle) if the value is changed.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
		;;; change current go_linestyle
		fi_check( int, 0, false)
		-> XptValue( the_go_window(pane), XtN lineStyle );
		go_redraw( pane );
enddefine;


define :method go_linefunction( pane :go_window_pane );    ;;; the default function
lvars pane;
;;; REF: go_linefunction( WINDOW_PANE ) -> INTEGER;
	XptValue( the_go_window(pane), XtN function, "int" );
enddefine;

define :method updaterof go_linefunction( int, pane :go_window_pane);
lvars pane;
;;; REF: INTEGER -> go_linefunction( WINDOW_PANE );
;;; REF: The current line function used by the_go_window in this pane and
;;; REF: redraws the objects in the pane with this new line function (see also
;;; REF: REF * XtNfunction) if this is changed.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	;;; change current go_linefunction
	fi_check( int, 0, false)
	-> XptValue( the_go_window(pane), XtN function );
	go_redraw( pane );
enddefine;


define :method go_linewidth( pane :go_window_pane );  ;;; default line width
lvars pane;
;;; REF: go_linewidth( WINDOW_PANE ) -> INTEGER;
	XptValue( the_go_window(pane), XtN lineWidth, "int" );
enddefine;

define :method updaterof go_linewidth( int, pane :go_window_pane);
lvars pane;
;;; REF: INTEGER -> go_linewidth( WINDOW_PANE );
;;; REF: The (always positive) default linewidth of the pane.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	;;; change current go_linewidth
	fi_check( int, 0, false)
	-> XptValue( the_go_window(pane), XtN lineWidth );
	go_redraw( pane );
enddefine;


define :method go_font( pane :go_window_pane ) -> the_font;  ;;; default font
lvars pane, the_font = XptValue( the_go_window(pane), XtN font, "exptr" );
;;; REF: go_linewidth( WINDOW_PANE ) -> XFONT;
enddefine;

define :method updaterof go_font( the_font, pane :go_window_pane);
lvars the_font, pane;
;;; REF: XFONT -> go_font( WINDOW_PANE );
;;; REF: The default font of the pane.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	;;; change current font
	the_font -> XptValue( the_go_window(pane), XtN lineWidth, "exptr" );
	go_redraw( pane );
enddefine;

;;;------------------------------------------------------
;;; SCREEN DRAWING ROUTINES:
;;; The following definitions mirror go_screen_object methods ...

define :method go_draw( pane :go_window_pane, pane :go_window_pane );
lvars pane;
;;; REF: go_draw( WINDOW_PANE, WINDOW_PANE );
;;; REF: Does nothing. This mirrors the method of the go_screen_object
;;; REF: class (REF * GO_SCREEN_OBJECT).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	;;; dummy routine!!!
enddefine;

define :method go_internal_redraw( pane :go_window_pane );
;;; REF: go_internal_redraw( WINDOW_PANE );
;;; REF: Does nothing. This mirrors the method of the go_screen_object
;;; REF: class (REF * GO_SCREEN_OBJECT).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars pane;
	;;; dummy routine!!!
enddefine;

;;;------------------------------------------------------
;;; VISIBILITY:
;;; The following definitions mirror go_screen_object methods ...

define :method go_show( pane :go_window_pane );
lvars pane;
;;; REF: go_show( WINDOW_PANE );
;;; REF: Make the pane appear on the screen if the pane is the window pane.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars window = the_go_window( pane );
	if ( window ) then XtMapWidget( window ); endif;
enddefine;

define :method go_hide( pane :go_window_pane );
lvars pane;
;;; REF: go_hide( WINDOW_PANE );
;;; REF: Make the window, with a pane using rc_graphics, disappear from the
;;; REF: screen.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars window = the_go_window( pane );
	if ( window ) then XtUnmapWidget( window ); endif;
enddefine;

;;;------------------------------------------------------
;;; DRAWING:

vars procedure go_clear_pane_window;   ;;; DEFINED IN GO_XDRAW

define :method go_clear( pane :go_window_pane );
lvars pane;
;;; REF: go_clear( WINDOW_PANE );
;;; REF: Clear the whole visible part of the pane; this does not call any
;;; REF: methods on the objects inside this pane. It also makes the pane
;;; REF: disappear from the screen. This mirrors the method of the
;;; REF: go_screen_object class (REF * GO_SCREEN_OBJECT).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars obj;
	go_clear_pane_window( pane );
	for obj in pane.the_go_components do
		"forbidden" -> go_visible_now( obj );
	endfor;
enddefine;

define :method go_drag_and_drop( pane :go_window_pane );
lvars pane;
;;; REF: go_drag_and_drop( WINDOW_PANE );
;;; REF: Does nothing. This mirrors the method of the go_screen_object
;;; REF: class (REF * GO_SCREEN_OBJECT).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	;;; dummy routine!!! [cannot move background]
enddefine;

;;;------------------------------------------------------
;;; BATCH MODE

define :method go_batch_mode( pane :go_window_pane ) -> the_status;
lvars pane, the_status;
;;; REF: go_batch_mode( WINDOW_PANE ) -> BOOLEAN;
	stored_go_batch_mode( pane ) -> the_status;
enddefine;

vars procedure XpwRedrawArea;  ;;; defined in go_xdrag.p
vars procedure go_safe_region; ;;; defined below

define :method updaterof go_batch_mode( the_status, pane :go_window_pane );
lvars the_status, pane;
;;; REF: BOOLEAN -> go_batch_mode( WINDOW_PANE );
;;; REF: The current pixmap status of the pane, this is false when the
;;; REF: batch mode is off and true when it is on.
;;; REF: If the batch mode is off all drawings appear directly on the screen
;;; REF: otherwise they are buffered until the batch mode is off again.
;;; REF: The other values of pixmap status: PixmapOff, PixmapHasNone are not
;;; REF: used in this library (see also HELP * XtNpixmapStatus).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars obj;
	if the_status  then
		true -> stored_go_batch_mode( pane );
		PixmapOnly -> XptValue( the_go_window( pane ) , XtN pixmapStatus );
	else
		false -> stored_go_batch_mode( pane );
		for obj in rev(pane.the_go_components) do
			unless (obj.go_visible_now == true) then
				go_batch_mode_off( obj );
			endunless;
		endfor;
		PixmapOn -> XptValue( the_go_window( pane ) , XtN pixmapStatus );
		;;; The following is needed because of a bad policy in pixmapStatus
		;;; which generates an Exposure event and therefore does not always
		;;; gets updated correctly if a succession of on/off batches happen.
		XpwRedrawArea( the_go_window( pane ), go_safe_region( pane, pane ) );
	endif;
enddefine;


define :method go_batch_mode_on( pane :go_window_pane );
lvars pane;
;;; REF: go_batch_mode_on( WINDOW_PANE );
;;; REF: Switches the current pixmap status of the pane on. All drawings are
;;; REF: buffered until the batch mode is off again.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	true -> go_batch_mode( pane );
enddefine;

define :method go_batch_mode_off( pane :go_window_pane );
lvars pane;
;;; REF: go_batch_mode_off( WINDOW_PANE );
;;; REF: Switches the current pixmap status of the pane off. The screen is
;;; REF: updated by emptying the buffer of drawings and all drawings will
;;; REF: appear directly on the screen. The objects which where themselves
;;; REF: in batch mode or not yet drawn will get redrawn as well.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	false -> stored_go_batch_mode( pane );
	false -> go_batch_mode( pane );
enddefine;

;;;------------------------------------------------------
;;; CONVENIENCE FUNCTIONS

define :method go_update_and_draw( pane :go_window_pane );
lvars pane;
;;; REF: go_update_and_draw( WINDOW_PANE );
;;; REF: Reinitialise and refresh pane. This function assures that all screen
;;; REF: coordinates are recalculated and that the foreground and background
;;; REF: colours are reset.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars obj, old_batch_mode;
	go_batch_mode( pane ) -> old_batch_mode;
	go_batch_mode_on( pane );
	if ( cached_go_colourtable(pane) and the_go_window( pane )) then
		stored_go_fgcolour( pane ) -> go_fgcolour( pane ) ;
		stored_go_bgcolour( pane ) -> go_bgcolour( pane ) ;
	endif;
	call_next_method( pane );
	old_batch_mode -> go_batch_mode( pane );
enddefine;

;;;------------------------------------------------------
;;; SCALING

define :method updaterof go_xscale( scale, pane :go_window_pane );
lvars scale, pane;
;;; REF: REAL -> go_xscale( WINDOW_PANE ) -> REAL;
;;; REF: Sets the horizontal scaling factor of a located object. This will
;;; REF: preserve the position of the centre of the pane and redraw all
;;; REF: objects in the given pane with the new horizontal scale.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars the_xscale = go_xscale( pane );
	if go_auto_resize( pane ) then
		abs( (go_bounding_width(pane) * scale) div the_xscale )
		-> go_bounding_width(pane);
		abs( (go_xorigin(pane)  * scale) div the_xscale )
		-> go_xorigin(pane);
	endif;
	call_next_method( scale, pane );
enddefine;

define :method updaterof go_yscale( scale, pane :go_window_pane );
lvars scale, pane;
;;; REF: REAL -> go_yscale( WINDOW_PANE );
;;; REF: Sets the vertical scaling factor of a located object. This will
;;; REF: preserve the position of the centre of the pane and redraw all
;;; REF: objects in the given pane with the new vertical scale.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars the_yscale = go_yscale( pane );
	if go_auto_resize( pane ) then
		abs( (go_bounding_height(pane)  * scale) div the_yscale )
		-> go_bounding_height(pane);
		abs( (go_yorigin(pane)  * scale) div the_yscale )
		-> go_yorigin(pane);
	endif;
	call_next_method( scale, pane );
enddefine;

define :method updaterof go_scale( scale, pane :go_window_pane );  ;;; both x&y
lvars scale, pane;
;;; REF: REAL -> go_scale( LOCATED );
;;; REF: Sets both horizontal and vertical scaling factor of a pane and redraw
;;; REF: all objects in the given pane with the new scale.
;;; REF: This is a convenience method using go_xscale and go_yscale.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars old_batch_mode_status;
	go_batch_mode( pane ) -> old_batch_mode_status;
	go_batch_mode_on( pane );
	call_next_method( scale, pane );
	old_batch_mode_status -> go_batch_mode( pane );
enddefine;


;;;------------------------------------------------------
;;; DIMENSIONS: IN SCREEN COORDS

define :method go_window_xloc( pane :go_window_pane ) -> x;
lvars pane, window=the_go_window(pane), x = 0;
;;; REF: go_window_xloc( WINDOW_PANE ) -> INTEGER;
	if ( window ) then
		XptValue( window, XtN x, TYPESPEC(:XptShort)) -> x;
	endif;
enddefine;

define :method updaterof go_window_xloc( x, pane :go_window_pane );
lvars x, pane, window=the_go_window(pane);
;;; REF: INTEGER -> go_window_xloc( WINDOW_PANE );
;;; REF: The horizontal screen position of the top left corner of the
;;; REF: window pane relative to its parent (see REF * GO_SHELL). This can
;;; REF: have an effect on other windows from the same parent as the window
;;; REF: manager might decide to move them around for a better fit.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	if ( window ) then
		abs(x) -> XptValue( window, XtN x, TYPESPEC(:XptShort));
	endif;
enddefine;

define :method go_window_yloc( pane :go_window_pane ) -> y;
lvars pane, window=the_go_window(pane), y = 0;
;;; REF: go_window_yloc( WINDOW_PANE ) -> INTEGER;
	if ( window ) then
		XptValue( window, XtN y, TYPESPEC(:XptShort)) -> y;
	endif;
enddefine;

define :method updaterof go_window_yloc( y, pane :go_window_pane );
lvars y, pane, window=the_go_window(pane);
;;; REF: INTEGER -> go_window_yloc( WINDOW_PANE );
;;; REF: The vertical screen position of the top left corner of the
;;; REF: window pane relative to its parent (see REF * GO_SHELL). This can
;;; REF: have an effect on other windows from the same parent as the window
;;; REF: manager might decide to move them around for a better fit.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	if ( window )  then
		abs(y) -> XptValue( window, XtN y, TYPESPEC(:XptShort));
	endif;
enddefine;


define :method go_window_width( pane :go_window_pane ) -> w;
lvars pane, window=the_go_window(pane), w = 0;
;;; REF: go_window_width( WINDOW_PANE ) -> INTEGER;
	if ( window ) then
		XptValue( window, XtN width, TYPESPEC(:XptShort)) -> w;
	endif;
enddefine;

define :method updaterof go_window_width( w, pane :go_window_pane );
lvars w, pane, window=the_go_window(pane);
;;; REF: INTEGER -> go_window_width( WINDOW_PANE );
;;; REF: The horizontal screen width of the window pane relative window
;;; REF: pane relative to its parent (see REF * GO_SHELL). This can have an
;;; REF: effect on other windows from the same parent as the window manager
;;; REF: might decide to move them around for a better fit.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	abs(w) -> w;
	if ( window and (w /= go_window_width(pane)) ) then
		w -> XptValue( window, XtN width, TYPESPEC(:XptShort));
		go_redraw( pane );
	endif;
enddefine;

define :method go_window_height( pane :go_window_pane ) -> h;
lvars pane, window=the_go_window(pane), h = 0;
;;; REF: go_window_height( WINDOW_PANE ) -> INTEGER;
	if ( window ) then
		XptValue( window, XtN height, TYPESPEC(:XptShort)) -> h;
	endif;
enddefine;

define :method updaterof go_window_height( h, pane :go_window_pane );
lvars h, pane, window=the_go_window(pane);
;;; REF: INTEGER -> go_window_height( WINDOW_PANE );
;;; REF: The vertical screen height of the window pane relative window
;;; REF: pane relative to its parent (see REF * GO_SHELL). This can have an
;;; REF: effect on other windows from the same parent as the window manager
;;; REF: might decide to move them around for a better fit.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	abs(h) -> h;
	if ( window and (h /= go_window_height(pane)) ) then
		h -> XptValue( window, XtN height, TYPESPEC(:XptShort));
		go_redraw( pane );
	endif;
enddefine;


;;;------------------------------------------------------
;;; DIMENSIONS: IN WORLD COORDINATES

define :method go_xloc( pane :go_window_pane ) -> x;
lvars pane;
;;; REF: go_xloc( WINDOW_PANE ) -> INTEGER;
;;; REF: Get the horizontal world coordinate of the top left corner of the
;;; REF: window pane relative to its origin (see REF * GO_SHELL).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars x, y;
	go_transxyin( 0, 0, pane ) -> (x, y);
	round(x) -> x;
enddefine;

define :method go_yloc( pane :go_window_pane ) -> y;
lvars pane;
;;; REF: go_yloc( WINDOW_PANE ) -> INTEGER;
;;; REF: Get the vertical world coordinate of the top left corner of the
;;; REF: window pane relative to its origin (see REF * GO_SHELL).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars x, y;
	go_transxyin( 0, 0, pane ) -> (x, y);
	round(y) -> y;
enddefine;

define :method go_bounding_width( pane :go_window_pane ) -> w;
lvars pane, w = pane.go_window_width;
;;; REF: go_bounding_width( WINDOW_PANE ) -> INTEGER;
	if ( pane.go_xscale ) then round(w div (pane.go_xscale)) -> w; endif;
enddefine;

define :method updaterof go_bounding_width( w, pane :go_window_pane );
lvars w, pane;
;;; REF: INTEGER -> go_bounding_width( WINDOW_PANE );
;;; REF: The horizontal width in world coordinates of the window pane
;;; REF: relative to its origin (see REF * GO_SHELL). This influences the
;;; REF: physical window width (see REF * GO_WINDOW_PANE/go_window_width)
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	if ( pane.go_xscale ) then round(w * (pane.go_xscale)) -> w; endif;
	w ->> go_window_width( pane ) -> pane.stored_go_bounding_width;
enddefine;

define :method go_bounding_height( pane :go_window_pane ) -> h;
lvars pane, h = pane.go_window_height;
;;; REF: go_bounding_height( WINDOW_PANE ) -> INTEGER;
	if ( pane.go_yscale ) then round(h div (pane.go_yscale)) -> h; endif;
enddefine;

define :method updaterof go_bounding_height( h, pane :go_window_pane );
lvars h, pane;
;;; REF: INTEGER -> go_bounding_height( WINDOW_PANE );
;;; REF: The vertical height in world coordinates of the window pane
;;; REF: relative to its origin (see REF * GO_SHELL). This influences the
;;; REF: physical window height (see REF * GO_WINDOW_PANE/go_window_height)
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
	if ( pane.go_yscale ) then round(h * (pane.go_yscale)) -> h; endif;
	h ->> go_window_height( pane ) -> pane.stored_go_bounding_height;
enddefine;

define :method go_bounding_box( pane :go_window_pane ) -> (x, y, w, h);
lvars pane, x = pane.go_xloc, y = pane.go_yloc,
			w = pane.go_bounding_width, h = pane.go_bounding_height;
;;; REF: go_bounding_box( WINDOW_PANE ) -> (X, Y, W, H);
;;; REF: Gets the dimensions in world coordinates of the bounding box of a
;;; REF: window pane. This doesn't use the go_xcentre, go_ycentre but uses the
;;; REF: go_xloc, go_yloc, go_bounding_width and go_bounding_height methods.
;;; REF: X and Y are the world coordinates of the object's reference point.
;;; REF: W and H are sizes in world coordinates of the object's bounding box.
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
enddefine;

;;;------------------------------------------------------
;;; DIMENSIONS: IN SCREEN COORDS
;;; needed for new go_internal_redraw function

define :method go_safe_region( pane :go_window_pane, win_pane :go_window_pane );
lvars pane, win_pane;
;;; REF: go_safe_region( PANE, WINDOW_PANE ) -> (X, Y, W, H);
;;; REF: Returns a tuple with the screen region occupied by the pane in the
;;; REF: given window pane.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
;;; REF: X and Y are screen coordinates of the bottom left corner of the region.
;;; REF: W and H are the width and height in screen coordinates of the region.
	if (pane == win_pane) then
		0; 0; pane.go_window_width; pane.go_window_height;
	else
		0; 0; 0; 0;
	endif;
enddefine;

;;;------------------------------------------------------------------
;;; DESTROY: tentative definition for destroying a pane

define :method go_destroy_object( pane :go_window_pane );
lvars pane;
;;; REF: go_destroy_object( WINDOW_PANE );
;;; REF: Destroys the pane object after removing all objects from it. These
;;; REF: objects themselves are not destroyed since they could be visible in
;;; REF: other panes (see also REF * GO_SCREEN_OBJECT/go_destroy_object).
;;; REF: WINDOW_PANE is a go_window_pane instance (REF * GO_WINDOW_PANE).
lvars window = the_go_window( pane );
	call_next_method( pane );
	false -> the_go_window( pane); ;;; ready for garbage collection
								   ;;; sysgarbage();
	XtUnrealizeWidget( window );
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_window_pane = true;

/* --- Revision History --------------------------------------------
 * BR 24/08/93
 *     Changed go_reinit_pane() method  into go_update_and_draw()
 * BR 13/07/93
 *     Added initialisation for go_[x/y]centre() methods.
 * BR 26/05/93
 *     Added slot for stored value of batch mode...
 *     Added optimisation for batch mode : a pane in batch mode will
 *     now put each changed object in batch mode too. Therefore at the
 *     end of the panes batch mode the objects need to be redrawn.
 * BR 25/05/93
 *     Added call to XpwRedrawArea() defined in go_xdrag.p to circumvent
 *     bug in PixmapOn PixmapStatus.
 * BR 09/05/93
 *     Replaced calls to go_refresh_pane() with go_redraw().
 *     Added comments.
 * BR 09/05/93
 *     Replaced calls to go_refresh_pane() with go_redraw().
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 20/04/93
 *     Removed fast_XptValue() replaced by XptValue...
 * BR 19/04/93
 *     Added methods for go_window_{xloc,yloc,width,height}() some of which
 *     replace the stored_go_bounding_(width,height}() methods.
 * BR 16/04/93
 *     Changed add_cap_style() into go_add_capstyle();
 *     Removed direct call to XpwClearWindow() by go_clear_pane_window()
 * BR 30/03/93
 *     Based on the new go_pane MIXIN and the old window_pane CLASS
 */
;;; eof
