/*  --- Copyright University of Sussex 1988. All rights reserved. ----------
 |  File:			C.all/lib/turtle/convolve.p
 |  Purpose:		convolve grey level picture with a mask
 |  Author:			Dave Hogg, Jan 21 1983 (see revisions)
 |  Documentation:	TEACH * CONVOLVE
 |  Related Files:
 */

;;; Pictures are stored on disk as numbers between 0 and 9 corresponding
;;; to grey levels from dark to bright respectively.

;;; CONVOLVE takes a picture represented by numbers and convolves with a mask
;;; represented as a list of rows containing the weights to be used. Each row
;;; must contain the same number of weights.
;;; FINDPEAKS also takes a mask to define neighbours to be considered when
;;; looking for local maxima generated by convolution. This procedure removes
;;; 'multiple responses' to intensity discontinuities by a process reminisent
;;; of lateral inhibition.
;;; Finally, weak responses are removed by THRESHOLD to generate an 'asterisk'
;;; picture.
;;; PROCESSPICTURE performs all three operations on a picture in sequence.
;;; ADDEDGESTOPICTURE adds edges represented in one picture to those already
;;; in a second picture.
;;; EXAMPLE applies simple vertical and horizontal edge masks to a picture
;;; and generates a combined 'asterisk' picture

;;;A variable to control tracing

vars chatty;
unless isboolean then false -> chatty; endunless;

uses turtle;

define constant zerofn (x); 0 enddefine; /* Trivial function to avoid n*0 */

define constant greypic (oldpic) -> picture;

	;;; Converts picture from numbers 1 - 8 to characters of proportional
	;;; brightness

	vars smax xmax ymax x y grey greychar;
	boundslist(oldpic) --> [= ?xmax = ?ymax];
	newpicture(xmax,ymax);
	' .:;|%OQM#' -> greychar;
	{% for grey from 0 to 9 do
			consword(greychar(grey+1),1)
		endfor %} -> greychar;

	for y from 1 to ymax do
		for x from 1 to xmax do
			greychar(oldpic(x,y)+1) -> picture(x,y)
		endfor
	endfor
enddefine;

define constant grey_display (picture);
	sysdisplay(greypic(picture));
enddefine;

define constant setpicture (picture,val);

	;;; Writes 'val' into all locations of picture

	vars xmax ymax x y;
	boundslist(picture) --> [= ?xmax = ?ymax];

	for y from 1 to ymax do
		for x from 1 to xmax do
			val -> picture(x,y)
		endfor;
	endfor;
enddefine;


define constant arrayofmask (mask) -> amask;

	;;; Converts a list of lists corresponding to rows of a mask into
	;;; an array with functional elements corresponding to weights

	vars xoffset xresidue yoffset yresidue xmin xmax ymin ymax x y row val;

	(length(mask(1))-1) // 2 -> xoffset -> xresidue;
	(length(mask)-1) // 2 -> yoffset -> yresidue;

	-xoffset -> xmin;
	xoffset + xresidue -> xmax;
	-yoffset -> ymin;
	yoffset + yresidue -> ymax;

	newarray([^xmin ^xmax ^ymin ^ymax]) -> amask;

	for y from ymin to ymax do
		mask --> [?row ??mask];
		for x from xmin to xmax do
			row --> [?val ??row];
			if      val == 0
			then    zerofn
			elseif  val == 1
			then    identfn
			else    nonop * (%val%)
			endif    -> amask(x,y);
		endfor;
	endfor;
enddefine;


define constant normalise (oldpic) -> picture;

	;;; Normalises values so that they are between 0 and 9

	vars xmax ymax minval maxval x y val scale;

	boundslist(oldpic) --> [= ?xmax = ?ymax];
	newpicture(xmax,ymax);

	;;; find maximum and minimum values

	1000000 -> minval;  -1000000 -> maxval;

	for y from 1 to ymax do
		for x from 1 to xmax do
			oldpic(x,y) -> val;
			if val < minval then val -> minval endif;
			if val > maxval then val -> maxval endif;
		endfor;
	endfor;

	;;; scale values

	10 / (maxval - minval + 1) -> scale;

	for y from 1 to ymax do
		for x from 1 to xmax do
			intof((oldpic(x,y) - minval) * scale) -> picture(x,y)
		endfor
	endfor;
enddefine;


define constant convolve(ipicture,mask) -> picture;

	;;; Convolves 'ipicture' with 'mask'

	vars amask mxmin mxmax mymin mymax xmax ymax x y dx dy total;
	vars procedure pixcelfn ;
	;;; make a new picture of the same size to receive convolved picture

	boundslist(ipicture) --> [= ?xmax = ?ymax];
	newpicture(xmax,ymax); setpicture(picture,0);

	;;; convert list representation for mask into an array
	if chatty then
		pr('\nCONVOLVING: input picture');
		grey_display(ipicture);
		pr('\nUsing mask: ');
		pr(mask); nl(1);
	endif;

	arrayofmask(mask) -> amask;

	;;; get upper and lower bounds for the mask array

	boundslist(amask) --> [?mxmin ?mxmax ?mymin ?mymax];

	for y from 1-mymin to ymax-mymax do
		0 -> picture(1,y);
		for x from 1-mxmin to xmax-mxmax do

			0 -> total;
			for dy from mymin to mymax do
				for dx from mxmin to mxmax do
					unless  (amask(dx,dy) ->> pixcelfn) == zerofn
					then    pixcelfn(ipicture(x+dx,y+dy)) + total -> total
					endunless;
				endfor;
			endfor;
			abs(total) -> picture(x,y);

		endfor;
	endfor;

	normalise(picture) -> picture;
	if chatty then
		pr('RESULT OF CONVOLUTION:');
		grey_display(picture);
	endif;
enddefine;





define constant findpeaks(ipicture,mask) -> picture;

	;;; Finds local maxima in  'ipicture' using 'mask' to select points
	;;; to consider in local neighbourhood. Those which are zero will
	;;; be ignored

	vars amask mxmin mxmax mymin mymax xmax ymax x y dx dy total val;

	;;; make a new picture of the same size to receive convolved picture

	if chatty then
		pr('\nFINDING PEAKS IN PICTURE USING MASK: '); pr(mask); nl(1);
	endif;
	boundslist(ipicture) --> [= ?xmax = ?ymax];
	newpicture(xmax,ymax); setpicture(picture,0);

	;;; convert list representation for mask into an array

	arrayofmask(mask) -> amask;

	;;; get upper and lower bounds for the mask array

	boundslist(amask) --> [?mxmin ?mxmax ?mymin ?mymax];

	for y from 1-mymin to ymax-mymax do
		for x from 1-mxmin to xmax-mxmax do

			ipicture(x,y) -> val;
			for dy from mymin to mymax do
				for dx from mxmin to mxmax do
					if      amask(dx,dy) /== zerofn and ipicture(x+dx,y+dy) > val
					then    0
					else    val
					endif    -> picture(x,y);
				endfor;
			endfor;

		endfor;
	endfor;
	if chatty then grey_display(picture) endif;
enddefine;


define constant threshold(ipicture,val) -> picture;

	;;; Generates a picture in which asterisks mark those positions above
	;;; the given threshold

	vars xmax ymax x y;
	if chatty then
		pr('\nTHRESHOLDING picture with value: ' >< val);
	endif;
	boundslist(ipicture) --> [1 ?xmax 1 ?ymax];
	newpicture(xmax,ymax);

	for y from 1 to ymax do
		for x from 1 to xmax do
			if ipicture(x,y) >= val then
				"*" -> picture(x,y)
			endif
		endfor
	endfor;
	if chatty then
		pr('\n\nTHRESHOLDING done');
		sysdisplay(picture)
	endif;
enddefine;


define constant processpicture(ipicture,mask,peakmask) -> picture;
	convolve(ipicture,mask) -> picture;
	findpeaks(picture,peakmask) -> picture;
	threshold(picture,2) -> picture;
enddefine;


define constant addedgestopicture(pic1,pic2,paint);

	vars xmax ymax x y;
	boundslist(pic1) --> [1 ?xmax 1 ?ymax];
	for y from 1 to ymax do
		for x from 1 to xmax do
			if pic1(x,y) = "*" then
				if      pic2(x,y) == space
				then    paint
				else    "*"
				endif   -> pic2(x,y);
			endif;
		endfor
	endfor
enddefine;


define constant example (ex_picture) ;

	vars vrtmask vrtpeakmask hrzmask hrzpeakmask vrtpic hrzpic smax ymax;
	vars chatty;
	true -> chatty;

	[[1 0 -1]] -> vrtmask;
	[[1] [0] [-1]] -> hrzmask;

	[[1 0 1]] -> vrtpeakmask;
	[[1] [0] [1]] -> hrzpeakmask;

	['processing picture looking for vertical edges with mask' ^vrtmask
		and peakmask ^vrtpeakmask] ==>
	processpicture(ex_picture,vrtmask,vrtpeakmask) -> vrtpic;
	['processing picture looking for horizontal edges with mask' ^hrzmask
		and peakmask ^hrzpeakmask] ==>
	processpicture(ex_picture,hrzmask,hrzpeakmask) -> hrzpic;

	boundslist(ex_picture) --> [1 ?xmax 1 ?ymax];
	newpicture(xmax,ymax);
	[adding vertical edges to picture] =>
	addedgestopicture(vrtpic,picture,"|");
	display();
	[adding horizontal edges to picture]=>
	addedgestopicture(hrzpic,picture,"-");
	display();

enddefine;

/*  --- Revision History ---------------------------------------------------
--- Aaron Sloman, Aug 26 1988 tabified
--- Allan Ramsay, Nov  7 1985 - minor cosmetic changes and speed-up
 */
