/* --- Copyright University of Sussex 1997. All rights reserved. ----------
 > File:			C.win32/src/sys_create_process.p
 > Purpose:			Create a new process
 > Author:			Robert John Duncan, May 17 1994 (see revisions)
 > Documentation:
 > Related Files:
 */

#_INCLUDE 'declare.ph'
#_INCLUDE 'io.ph'
#_INCLUDE 'win32defs.ph'

constant procedure ( sys_fname_nam, sys_raise_ast, );

section $-Sys;

constant procedure ( Check_astp_arg, Check_vector, );

endsection;

;;; -----------------------------------------------------------------------


section $-Sys =>
		sys_create_process,
		sys_get_process_id,
		sys_get_process_status,
		sys_resume_process,
		sys_suspend_process,
		sys_terminate_process,
	;

	/*	Get a Process ID from a process handle. Currently, this works only
		for processes created by sys_create_process which stashes the PID
		in the back of the handle's props field
	*/
define sys_get_process_id(hnd);
	sys_check_handle(hnd, "PROCESS", true);
	lvars props = hnd!HND_PROPS;
	ispair(props) and isintegral(fast_back(props) ->> props) and props;
enddefine;

	/*	Get the exit code of a process, or return <false> if it's still
		executing
	*/
define sys_get_process_status(hnd);
	dlvars _status;
	sys_check_handle(hnd, "PROCESS", true);
	if _zero(_extern pop_get_exit_code_process(hnd!XP_PTR, ident _status)) then
		GET_LAST_ERROR;
		mishap(hnd, 1, 'FAILED TO GET PROCESS STATUS');
	elseif _status == _:WIN32_STILL_ACTIVE then
		false;
	else
		Uint_->_pint(_status);
	endif;
enddefine;

	/*	Create a new Windows process and return a handle for it
	*/
define sys_create_process(path, command_line, stdio, will_do_wait) -> proc;
	lvars astp = false;

	unless isboolean(will_do_wait) then
		;;; optional ____astp arg given
		((), path, command_line, stdio, will_do_wait) ->
			(path, command_line, stdio, will_do_wait, astp);
		Check_astp_arg(astp);
		unless isboolean(will_do_wait) then
			mishap(will_do_wait, 1, 'BOOLEAN NEEDED');
		endunless;
	endunless;

	if path then
		sysfileok(path, false) -> path;
	endif;
	if ispair(command_line) then
		lvars string;
		consstring(#|
			explode(fast_front(command_line));
			for string in fast_back(command_line) do
				`\s`, explode(string);
			endfor;
		|#) -> command_line;
	else
		if command_line == false or command_line == [] then
			;;; command line should contain at least the executable name
			sys_fname_nam(path) -> command_line;
		endif;
		Check_string(command_line);
	endif;

	;;; determine std I/O handles for child
	lconstant proc_info = writeable inits(PROCINFO_LENGTH);
	lvars _flags = _0, _usestdhandles = _0;
	if stdio then
		if stdio == true then
			;;; run in new console
			_:WIN32_CREATE_NEW_CONSOLE -> _flags;
		elseif stdio == "undef" then
			;;; run detached
			_:WIN32_DETACHED_PROCESS -> _flags;
		else
			;;; explicit I/O redirection
			_:WIN32_DETACHED_PROCESS -> _flags;
			_1 -> _usestdhandles;
			;;; stdio should be a vector containing at least 3 devices
			;;; supplying the handles for stdin, stdout & stderr
			Check_vector(stdio);
			define Get_handle(dev, _checkopen) -> _handle;
				Check_device(dev, _checkopen);
				lvars _handle = dev!D_CTRL_BLK!DCB_HANDLE;
				if dev!D_UNIT_N!UNT_TYPE == _:POP_FILE_TYPE_CONSOLE
				and _handle /== _:WIN32_INVALID_HANDLE_VALUE
				then
					;;; active console handle: allow it to be inherited
					;;; as a default
					_0 -> _flags;
					_NULL -> _handle;	;;; _:WIN32_INVALID_HANDLE_VALUE?
				endif;
			enddefine;
			Get_handle(stdio(1), 2:0011) -> proc_info!PI_HSTDIN;
			Get_handle(stdio(2), 2:0101) -> proc_info!PI_HSTDOUT;
			Get_handle(stdio(3), 2:0101) -> proc_info!PI_HSTDERR;
		endif;
	endif;

	;;; create process
	lvars t_path = path and Tchars_in(path, wkstring1);
	lvars t_command_line = Tchars_in(command_line, wkstring2);
	lvars _status = _extern pop_create_process(
		t_path and t_path@V_TCHARS or _NULL, t_command_line@V_TCHARS,
		_flags, _usestdhandles, proc_info@V_BYTES
	);
	if _zero(_status) then
		GET_LAST_ERROR;
		Syserr_mishap(path or command_line, 1, 'FAILED TO CREATE PROCESS');
	endif;

	;;; create process handle, with thread handle as datum
	lvars thread = sys_cons_handle(Cons_extern_ptr(proc_info!PI_HTHREAD),
						"THREAD", false, true);
	lvars pid = Uint_->_pint(proc_info!PI_PID);
	lvars proc = sys_cons_handle(Cons_extern_ptr(proc_info!PI_HPROCESS),
						conspair("PROCESS",pid), thread, true);

	;;; add async trap to process handle if required
	if astp then
		define Process_terminated(proc, err, astp);
			returnif(err);
			if ispair(astp) then fast_front(astp) else astp endif,
			consclosure((), proc, sys_get_process_status(proc), 2),
			if ispair(astp) then conspair((), fast_back(astp)) endif,
			chain(sys_raise_ast);
		enddefine;
		Process_terminated(%astp%) -> sys_async_handle(proc);
	endif;
enddefine;

define sys_terminate_process(hnd);
	;;; optional method
	lvars method = true;
	if isboolean(hnd) or isinteger(hnd) then
		((), hnd) -> (hnd, method);
	endif;
	sys_check_handle(hnd, "PROCESS", true);
	if method == true then
		;;; fast kill
		if _nonzero(_extern pop_terminate_process(hnd!XP_PTR)) then
			sys_wait_for_handle(hnd) -> ;
		endif;
	else
		;;; method is false or integer: send WM_CLOSE message
		lvars pid = sys_get_process_id(hnd);
		if pid and _nonzero(_extern pop_close_process_windows(hnd!XP_PTR,
												Pint_->_uint(pid, _-1)))
		then
			;;; use method as timeout for wait
			sys_wait_for_handle(hnd, method) -> ;
		endif;
	endif;
	;;; return exit code, or <false> if still active
	sys_get_process_status(hnd);
enddefine;

define lconstant Control_thread(hnd, susp);
	unless sys_test_handle(hnd, "THREAD", true) then
		sys_check_handle(hnd, "PROCESS", true);
		sys_check_handle(sys_handle_data(hnd) ->> hnd, "THREAD", true);
	endunless;
	lvars _res =
		if susp then
			_extern pop_suspend_thread(hnd!XP_PTR)
		else
			_extern pop_resume_thread(hnd!XP_PTR)
		endif;
	_res /== _16:FFFFFFFF and Uint_->_pint(_res);
enddefine;

define sys_suspend_process(hnd);
	Control_thread(hnd, true);
enddefine;

define sys_resume_process(hnd);
	Control_thread(hnd, false);
enddefine;


	/*	Dummy definition referenced elsewhere but never used
	*/
define Waitpid(_pid, _status_id);
	termin;
enddefine;

endsection;		/* $-Sys */


/* --- Revision History ---------------------------------------------------
--- Robert Duncan, Jan 29 1997
		Modifications for UNICODE compilation
--- Robert Duncan, Nov 18 1996
		Added a check for whether the console has been opened.
--- Robert Duncan, May 24 1996
		Reimplementation based on process handles.
--- Robert John Duncan, Jan  8 1996
		Change to how child's standard I/O is set up: _____stdio == false means
		use the Windows default (i.e. _____flags == 0) whereas if specific I/O
		devices are supplied, the child must be run detached (_____flags ==
		DETACHED_PROCESS).
 */
