/* --- Copyright University of Sussex 1997. All rights reserved. ----------
 > File:			C.win32/src/sysio.p
 > Purpose:			Basic system I/O procedures (Win32)
 > Author:			Robert John Duncan, May  6 1994 (see revisions)
 > Related Files:	C.{unix,vms}/src/sysio.p
 */

#_INCLUDE 'declare.ph'
#_INCLUDE 'io.ph'
#_INCLUDE 'win32defs.ph'

section $-Sys;

constant procedure (Get_full_pathname, Io$-Cons_device, Io$-Kill_device);

endsection;		/* $-Sys */

;;; -----------------------------------------------------------------------


section $-Sys$-Io =>
	pop_file_versions
	readable
	sys_create_file
	sysclose
	syscreate
	sysdelete
	sysflush
	sysopen
;

vars
	pop_file_attributes = WIN32_FILE_ATTRIBUTE_NORMAL,
	pop_file_versions	= 2,
;

constant procedure (Backup_file);	;;; forward

define New_std_dev(new_dev, std_dev_id, check_flags);
	lvars new_dev, std_dev_id, check_flags, _which;
	Check_device(new_dev, check_flags);
	if std_dev_id == ident dev_in then
		_0
	elseif std_dev_id == ident dev_out then
		_1
	elseif std_dev_id == ident dev_err then
		_2
	else
		mishap(std_dev_id, 1, 'NOT A STANDARD I/O DEVICE');
	endif -> _which;
	if _zero(_extern pop_set_std_handle(_which, new_dev!D_CTRL_BLK!DCB_HANDLE))
	then
		mishap(std_dev_id, 1, 'FAILED TO CHANGE STANDARD I/O DEVICE');
	endif;
	new_dev -> fast_idval(std_dev_id);
enddefine;

define Create_file(file, _access, _share, _security, _create, _attrs) -> _handle;
	if _attrs _bitst _:WIN32_FILE_FLAG_NO_BUFFERING then
		;;; has requirements we don't meet
		mishap(file, 1, 'FILE_FLAG_NO_BUFFERING NOT SUPPORTED');
	endif;
	if _create == _:WIN32_TRUNCATE_EXISTING then
		;;; needs write access
		_access _biset _:WIN32_GENERIC_WRITE -> _access;
	endif;
	lvars t_file = Tchars_in(file, wkstring1);
	_extern pop_create_file(t_file@V_TCHARS, _access, _share, _security,
		_create, _attrs, _NULL) -> _handle;
	if _handle == _:WIN32_INVALID_HANDLE_VALUE then
		GET_LAST_ERROR;
		if _syserror == _:WIN32_ERROR_SHARING_VIOLATION
		or _syserror == _:WIN32_ERROR_TOO_MANY_OPEN_FILES
		or _syserror == _:WIN32_ERROR_NOT_ENOUGH_QUOTA
		then
			;;; problem may be caused by one of our own files, still open but
			;;; not accessible anywhere; do GC to clean up and try again
			Sysgarbage(true, 'fopn');
			;;; static buffer could be overwritten during GC
			Tchars_in(file, wkstring1) -> t_file;
			_extern pop_create_file(t_file@V_TCHARS, _access, _share,
				_security, _create, _attrs, _NULL) -> _handle;
		endif;
	endif;
enddefine;

define Move_file(old, new);
	lvars _replace = _0, _tried_gc = false;
	lvars t_old = Tchars_in(old, wkstring1);
	lvars t_new = Tchars_in(new, wkstring2);
	while _zero(_extern pop_move_file(t_old@V_TCHARS, t_new@V_TCHARS, _replace))
	do
		;;; move failed
		GET_LAST_ERROR;
		if (_syserror == _:WIN32_ERROR_ALREADY_EXISTS or
			_syserror == _:WIN32_ERROR_FILE_EXISTS)
		and _replace == _0
		then
			;;; create a backup version, then try moving again but with
			;;; REPLACE_EXISTING set
			Backup_file(new);
			_1 -> _replace;
		elseif _syserror == _:WIN32_ERROR_SHARING_VIOLATION and not(_tried_gc)
		then
			;;; can't move an open file -- do GC in case it's held by a
			;;; garbage device
			Sysgarbage(true, 'fmov');
			true -> _tried_gc;
		else
			Syserr_mishap(old, new, 2, 'CAN\'T MOVE FILE');
		endif;
		;;; Backup_file or GC may have overwritten static buffers
		Tchars_in(old, wkstring1) -> t_old;
		Tchars_in(new, wkstring2) -> t_new;
	endwhile;
enddefine;

define Backup_file(file);
	lvars file;
	dlocal pop_file_versions;
	returnunless(pop_file_versions and pop_file_versions > 1);
	pop_file_versions - 1 -> pop_file_versions;
	Move_file(file, sysfileok(file <> '-'));
enddefine;

	/*	Convert Poplog file modes to Win32
	*/
define lconstant Check_mode(mode) -> mode;
	if isword(mode) then
		mode!W_STRING -> mode;
	else
		Check_string(mode);
	endif;
enddefine;

define lconstant File_access_mode(access);
	Check_integer(access, false);
	if access == 0 then
		_:WIN32_GENERIC_READ;
	elseif access == 1 then
		_:WIN32_GENERIC_WRITE;
	elseif access == 2 then
		_:WIN32_GENERIC_READ_WRITE;
	else
		mishap(access, 1, 'INVALID FILE ACCESS MODE');
	endif;
enddefine;

define lconstant File_share_mode(share) -> _share;
	returnif(isinteger(share))(_int(share) -> _share);
	(Check_mode(share), _0) -> (share, _share);
	lvars _i, _len = datalength(share);
	fast_for _i to _len do
		lvars _c = fast_subscrs(_i, share);
		if _c == `D` then
			_share _biset _:WIN32_FILE_SHARE_DELETE -> _share;
		elseif _c == `R` then
			_share _biset _:WIN32_FILE_SHARE_READ -> _share;
		elseif _c == `W` then
			_share _biset _:WIN32_FILE_SHARE_WRITE -> _share;
		else
			mishap(share, 1, 'ILLEGAL FILE SHARE MODE');
		endif;
	endfor;
enddefine;

define lconstant File_security(security);
	;;; currently this is limited to inherited/not-inherited
	unless isboolean(security) then
		mishap(security, 1, 'BOOLEAN NEEDED');
	endunless;
	if security then _1 else _0 endif;
enddefine;

define lconstant File_create_mode(create);
	if create == true then
		_:WIN32_CREATE_NEW
	elseif create == false then
		_:WIN32_OPEN_EXISTING
	else
		Check_integer(create, false);
		_int(create)
	endif;
enddefine;

define lconstant File_flags(flags);
	Check_integer(flags, false);
	_int(flags);
enddefine;

define lconstant File_attrs(attrs) -> _attrs;
	if attrs == false then pop_file_attributes -> attrs endif;
	returnif(isinteger(attrs))(_int(attrs) -> _attrs);
	(Check_mode(attrs), _0) -> (attrs, _attrs);
	lvars _i, _len = datalength(attrs);
	fast_for _i to _len do
		lvars _c = fast_subscrs(_i, attrs);
		if _c == `A` then
			_attrs _biset _:WIN32_FILE_ATTRIBUTE_ARCHIVE -> _attrs;
		elseif _c == `C` then
			_attrs _biset _:WIN32_FILE_ATTRIBUTE_COMPRESSED -> _attrs;
		elseif _c == `H` then
			_attrs _biset _:WIN32_FILE_ATTRIBUTE_HIDDEN -> _attrs;
		elseif _c == `R` then
			_attrs _biset _:WIN32_FILE_ATTRIBUTE_READONLY -> _attrs;
		elseif _c == `S` then
			_attrs _biset _:WIN32_FILE_ATTRIBUTE_SYSTEM -> _attrs;
		elseif _c == `T` then
			_attrs _biset _:WIN32_FILE_ATTRIBUTE_TEMPORARY -> _attrs;
		else
			mishap(attrs, 1, 'ILLEGAL FILE ATTRIBUTES');
		endif;
	endfor;
enddefine;


;;; -- Public Procedures --------------------------------------------------

define sys_create_file(file, access, share, inherit, create, flags, attrs,
						org, errc);
	sysfileok(file, false) -> file;
	unless errc == `A` or errc == `D` or errc == `F` or errc == `N` then
		mishap(errc, 1, 'INVALID ERROR CHARACTER CODE');
	endunless;
	lvars _flags_&_attrs = File_flags(flags) _biset File_attrs(attrs);
	lvars _handle = Create_file(
						file,
						File_access_mode(access),
						File_share_mode(share),
						File_security(inherit),
						File_create_mode(create),
						_flags_&_attrs);
	if _handle == _:WIN32_INVALID_HANDLE_VALUE then
		GET_LAST_ERROR;
		if _syserror == _:WIN32_ERROR_FILE_EXISTS then
			;;; must be CREATE_NEW
			returnif(errc == `F`)(false);
		elseif _syserror == _:WIN32_ERROR_FILE_NOT_FOUND then
			returnif(errc == `A` or errc == `D` or errc == `F`)(false);
		elseif _syserror == _:WIN32_ERROR_PATH_NOT_FOUND then
			returnif(errc == `A` or errc == `D`)(false);
		elseif _syserror == _:WIN32_ERROR_ACCESS_DENIED then
			returnif(errc == `A`)(false);
		endif;
		Syserr_mishap(file, 1,
			if create == true then
				'CAN\'T CREATE FILE'
			elseif create == false then
				'CAN\'T OPEN FILE'
			else
				'CAN\'T OPEN/CREATE FILE'
			endif);
	endif;
	Cons_device(file, Get_full_pathname(file), access, org, _handle,
		;;; overlapped?
		_flags_&_attrs _bitst _:WIN32_FILE_FLAG_OVERLAPPED);
enddefine;

define syscreate(file, access, org) -> dev;
	lvars file, access, org, _errc = `N`;
	if isinteger(org) then
		((), file, access, org) -> (file, access, org, _errc);
		unless _errc == `F` or _errc == `N` then
			mishap(_errc, 1, 'INVALID ERROR CHARACTER CODE');
		endunless;
	endif;
	lvars share = if access == 0 then WIN32_FILE_SHARE_READ else 0 endif;
	;;; try to create a new file
	sys_create_file(file, access, share, false, true, 0, false, org, `F`)
		-> dev;
	returnif(dev or _errc == `F`);
	;;; file exists: check that we have permission to replace the
	;;; current version (NB: we're actually checking only for write
	;;; access to the file -- that may not be the same thing)
	lvars okFile = sysfileok(file);
	lvars _access = File_access_mode(access) _biset _:WIN32_GENERIC_WRITE;
	lvars _handle = Create_file(okFile, _access, _0, _0,
								_:WIN32_OPEN_EXISTING, _0);
	if _handle == _:WIN32_INVALID_HANDLE_VALUE then
		GET_LAST_ERROR;
		Syserr_mishap(file, 1, 'CAN\'T CREATE FILE');
	else
		_extern pop_close_handle(_handle) -> ;
	endif;
	;;; keep a backup
	Backup_file(okFile);
	;;; replace existing
	sys_create_file(file, access, share, false, WIN32_CREATE_ALWAYS, 0, false,
		org, `N`) -> dev;
enddefine;

define sysopen(file, access, org);
	lvars file, access, org, _errc = `F`;
	if isinteger(org) then
		((), file, access, org) -> (file, access, org, _errc);
	endif;
	lvars share = if access == 0 then WIN32_FILE_SHARE_READ else 0 endif;
	sys_create_file(file, access, share, false, false, 0, false, org, _errc);
enddefine;

define readable =
	sysopen(% 0, false, `A` %);
enddefine;

define sysdelete(file);
	lvars _tried_gc = false;
	sysfileok(file, false) -> file;
	lvars t_file = Tchars_in(file, wkstring1);
	while _zero(_extern pop_delete_file(t_file@V_TCHARS)) do
		GET_LAST_ERROR;
		if _syserror == _:WIN32_ERROR_FILE_NOT_FOUND then
			return(false);
		elseif _syserror == _:WIN32_ERROR_SHARING_VIOLATION and not(_tried_gc)
		then
			;;; can't delete an open file -- do GC in case it's held by
			;;; a garbage device
			Sysgarbage(true, 'fdel');
			true -> _tried_gc;
		else
			Syserr_mishap(file, 1, 'CAN\'T DELETE FILE');
		endif;
		;;; GC may have overwritten static buffer
		Tchars_in(file, wkstring1) -> t_file;
	endwhile;
	returnunless(pop_file_versions);
	;;; restore back versions
	dlocal pop_file_versions;
	lvars backup, t_backup;
	while pop_file_versions > 1 do
		sysfileok(file <> '-') -> backup;
		Tchars_in(backup, wkstring2) -> t_backup;
		Tchars_in(file, wkstring1) -> t_file;
		if _nonzero(_extern pop_move_file(t_backup@V_TCHARS, t_file@V_TCHARS,
											_0))
		then
			;;; moved OK
			backup -> file;
			pop_file_versions - 1 -> pop_file_versions;
		else
			;;; move failed
			GET_LAST_ERROR;
			if _syserror == _:WIN32_ERROR_FILE_NOT_FOUND then
				;;; no backup -- done
				quitloop;
			elseif _syserror == _:WIN32_ERROR_SHARING_VIOLATION
			and not(_tried_gc)
			then
				;;; can't move an open file -- do GC in case it's held by a
				;;; garbage device
				Sysgarbage(true, 'fmov');
				true -> _tried_gc;
			else
				Syserr_mishap(backup, file, 2, 'CAN\'T MOVE FILE');
			endif;
		endif;
	endwhile;
	true;
enddefine;

define sysclose(dev);
	lvars dev;
	Check_device(dev, false);
	fast_apply(dev, dev!D_CLOSE);
	Kill_device(dev);
enddefine;

define sysflush(dev);
	lvars dev;
	;;; ignore optional _______do_sync parameter
	if isboolean(dev) then () -> dev endif;
	Check_device(dev, true);
	fast_apply(dev, dev!D_FLUSH);
enddefine;

endsection;		/* $-Sys$-Io */


/* --- Revision History ---------------------------------------------------
--- Robert Duncan, Jun  6 1997
		Fixed syscreate to check for write access before replacing an
		existing file.
		Improvements to sys_create_file.
--- Robert Duncan, Jan 29 1997
		Modifications for UNICODE compilation
--- Robert Duncan, Apr 19 1996
		Standard I/O handles now identified by conventional 0,1,2
--- Robert Duncan, Mar 21 1996
		Added test for extra error code in Move_file which can arise when
		moving between volumes (i.e. when copying)
--- Robert John Duncan, Jan  8 1996
		Added sys_create_file to allow creation of devices using overlapped
		I/O. Changed syscreate and sysopen to use it.
 */
