/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_polyarrow.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 4th June 1993
;;; File: go_polyarrow.p
;;; Author: Ben L.E. Rabau

compile_mode :pop11 +strict;

include XpwPixmap.ph;
uses go_polyline;

;;;------------------------------------------------------
;;; DOUBLE ARROW CLASS:


;;; INTRO: The go_polyarrow CLASS represents polylines with arrow heads.
;;; INTRO: This is just a special case of the go_polyline class (see also
;;; INTRO: REF * GO_POLYLINE). The lines can have an arrow head at the
;;; INTRO: and at the end, if none is visible then it behaves like a
;;; INTRO: normal go_polyline but with a bigger sensitive area (see slot
;;; INTRO: go_border).

define :class go_polyarrow;
	isa go_polyline;
	slot go_from_arrow    ==  true;
;;; REF: Boolean: whether or not there is an arrow on the start of the line.
	slot go_to_arrow      ==  true;
;;; REF: Boolean: whether or not there is an arrow on the end of the line.
	slot go_border        ==     8;
;;; REF: Sensitive area in pixels around the arrow for the mouse.
enddefine;

;;;------------------------------------------------------
;;; ARROW SIZE
define :method go_arrow_size( lin :go_polyarrow ) /* -> size */;
lvars lin /* , size */;
;;; REF: go_arrow_size( ARROW ) -> INTEGER; Returns size of arrow head.
;;; REF: ARROW is an instance of the GO_POLYARROW (see REF * GO_POLYARROW).
;;; REF: The return value is in screen coordinates and based on go_border.
	lin.go_border /* -> size */;
enddefine;

define :method updaterof go_arrow_size( size, lin :go_polyarrow );
lvars size, lin;
;;; REF: INTEGER -> go_arrow_size( ARROW ); Sets size of arrow head.
;;; REF: ARROW is an instance of the GO_POLYARROW (see REF * GO_POLYARROW).
;;; REF: The update value is in screen coordinates and used as go_border.
	go_clear( lin );
	size -> lin.go_border;
	go_internal_redraw( lin );
enddefine;

;;;------------------------------------------------------
;;; DRAW ARROW

define :method go_arrow_screen_coords( pane :go_pane, lin :go_polyarrow );
lvars pane, lin;
;;; REF: go_arrow_screen_coords( PANE, POLYARROW );
;;; REF: This method will leave the lists with screen coordinates of the
;;; REF: arrows in the given pane on the stack if any (there can be 0,1 or
;;; REF: 2 lists on the stack).
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: POLYARROW is an instance of go_polyarrow (see REF * GO_POLYARROW).
;;; REF: The lists have the form: [ x1 y1 x2 y2 x3 ... ]
lvars n, x1, x2, y1, y2, alpha, cosalpha, sinalpha;
lvars sinarrow_up, sinarrow_down, cosarrow_up, cosarrow_down, factor;
lvars pointlist = stored_go_local_coords(lin);

	define lconstant calc_arrow( x, y, cos_up, cos_down, sin_up, sin_down );
	;;; Uses "pane" and "lin" from method go_arrow_screen_coords().
	lvars x, y, cos_up, cos_down, sin_up, sin_down, pointlist;
		go_translistout( [% x,          y,
							x+cos_up,   y+sin_up,
							x+cos_down, y+sin_down,
							x,          y %],        lin) -> pointlist;
		go_translistout( pointlist, pane );
		sys_grbg_list( pointlist );
	enddefine;

	;;; size of the arrow:
	(sin(45) * lin.go_border) -> factor;    ;;; 4 pixel size

	;;; inbetween the line
	pointlist( 1 ) -> x1;
	pointlist( 2 ) -> y1;
	pointlist( 3 ) -> x2;
	pointlist( 4 ) -> y2;
	arctan2( (x2 - x1 + 1e-15), (y2 - y1) ) -> alpha;

	;;; Since sin(45) = cos(45) we can simplify arrows to:
	;;; cos(alpha + 45) = (cos(alpha) - sin(alpha)) * sin(45)
	;;; cos(alpha - 45) = (cos(alpha) + sin(alpha)) * sin(45)
	;;; sin(alpha + 45) = (cos(alpha) + sin(alpha)) * sin(45) =  cos(alpha-45)
	;;; sin(alpha - 45) = (sin(alpha) - cos(alpha)) * sin(45) = -cos(alpha+45)

	factor * sin(alpha) -> sinalpha;
	factor * cos(alpha) -> cosalpha;
	round( cosalpha - sinalpha ) -> cosarrow_up;
	round( cosalpha + sinalpha ) -> cosarrow_down;
	cosarrow_down -> sinarrow_up;
	-cosarrow_up  -> sinarrow_down;

	if ( lin.go_from_arrow ) then
		calc_arrow( x1,          y1,
					cosarrow_up, cosarrow_down,
					sinarrow_up, sinarrow_down  );
	endif;
	if ( lin.go_to_arrow ) then
		if ( go_npoints( lin ) > 2 ) then
			;;; if this is a polyline rather than a simple line then
			;;; the second arrow has a different go_orientation:
			go_npoints( lin ) * 2 -> n;
			pointlist( n - 3 ) -> x1;
			pointlist( n - 2 ) -> y1;
			pointlist( n - 1 ) -> x2;
			pointlist(   n   ) -> y2;
			arctan2( (x2 - x1 + 1e-15), (y2 - y1) ) -> alpha;

			factor * sin(alpha) -> sinalpha;
			factor * cos(alpha) -> cosalpha;
			round( cosalpha - sinalpha ) -> cosarrow_up;
			round( cosalpha + sinalpha ) -> cosarrow_down;
			cosarrow_down -> sinarrow_up;
			-cosarrow_up  -> sinarrow_down;
		endif;
		calc_arrow( x2,           y2,
					-cosarrow_up, -cosarrow_down,
					-sinarrow_up, -sinarrow_down  );
	endif;
enddefine;


define :method go_fgdraw( pane :go_pane, lin :go_polyarrow );
lvars pane, lin;
;;; REF: go_fgdraw( PANE, POLYARROW ); draws the polyarrow in the pane
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: POLYARROW is an instance of go_polyarrow (see REF * GO_POLYARROW).
lvars pointlist;
	call_next_method( pane, lin );

	;;; go_draw arrows inside the line
	if lin.go_to_arrow or lin.go_from_arrow then
		go_arrow_screen_coords( pane, lin );
		if ( lin.go_to_arrow ) then
			-> pointlist;
			go_draw_lines( pane, pointlist, CoordModeOrigin );
		endif;
		if ( lin.go_from_arrow ) then
			-> pointlist;
			go_draw_lines( pane, pointlist, CoordModeOrigin );
		endif;
	endif;
enddefine;

;;;----------------------------------------------------------------

define :method go_set_bounding_box( lin :go_polyarrow );
lvars lin;
;;; REF: go_set_bounding_box( POLYARROW ); checks dimensions of the polyarrow
;;; REF: This method ensures that the dimensions of the bounding box are
;;; REF: set correctly taking the go_border into account (see REF * GO_LOCATED).
;;; REF: POLYARROW is an instance of go_polyarrow (see REF * GO_POLYARROW).
lvars factor, dw, dh;
	call_next_method( lin );
	;;; go_draw arrows inside the line
	if ( lin.go_bounding_width  < 0 ) then -2 else 2 endif -> dw;
	if ( lin.go_bounding_height < 0 ) then -2 else 2 endif -> dh;
	lin.go_border -> factor;
	lin.go_bounding_width  + factor*dw -> lin.stored_go_bounding_width;
	lin.go_bounding_height + factor*dh -> lin.stored_go_bounding_height;
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_polyarrow = true;


/* --- Revision History --------------------------------------------
 * BR 04/06/93
 *     Removed all references to go_stored_[x/y]loc (go_set_bounding_box).
 * BR 25/05/93
 *     Removed last argument from go_lines_coords()...
 * BR 28/04/93
 *     Moved go_polyarrow class into separate file: go_polyarrow.p
 *     Changed calc_arrow to be a local definition of go_arrow_screen_coords().
 *     Added comments.
 * BR 08/04/93
 *     Removed direct call to XpwDrawLine[s]() by go_draw_line[s]();
 *     Removed direct call to XpwDrawArcs() by go_draw_arcs();
 *     Removed direct call to XpwFillArcs() by go_draw_filled_arcs();
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * BR 19/01/93
 *     Optimised go_arrow_screen_coords to avoid multiple calls to stored_go_local_coords
 *     which are calculated in the case of connectors!!
 * BR 04/12/92
 *     Created to replace go_stroke.p (class stroke and class arrow).
 *     These new classes go_polyline and go_polyarrow work also for polylines!
 */
;;; eof
