/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_roundlines.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 7th May 1993
;;; File: go_roundlines.p
;;; Author: Ben L.E. Rabau

compile_mode :pop11 +strict;

/* Intention:

;;; INTRO: Program to automate the rounding of line intersections. The lines
;;; INTRO: are only defined by their endpoints.
;;; INTRO: There are two methods which work for go_polyline & go_polygon:
;;; INTRO:         1. go_get_round_lines()
;;; INTRO:                Round two given lines given their endpoints.
;;; INTRO:         2. go_get_round_polylines()
;;; INTRO:                Round the lines in the polyline (or polygon).


;;; REF:    /\                    +------------------Key------------------+
;;; REF:    ||          /         | a/2 : calculated half-go_angle between|
;;; REF:   Y-ax        /          |       lines                           |
;;; REF:    || (x2,y2)/           | r   : given rounding radius           |
;;; REF:    ||       +     c      | R   : calculated distance from endpnt |
;;; REF:    ||      /             | (01): new endpoint on line to (x1,y1) |
;;; REF:    || (02)/     .'       | (02): new endpoint on line to (x2,y2) |
;;; REF:    ||    X. r .'         +---------------------------------------+
;;; REF:    ||   /  `.'  a/2 _ +
;;; REF:    || R/  .' \  _ -    (x1,y1)
;;; REF:    || / .'  _ X     b
;;; REF:    ||/.'_ -  (01)
;;; REF:  ==##=====================> X-ax
;;; REF:    ||(x0,y0)
;;; REF:
;;; REF:
;;; REF:
;;; REF: Calculation of a:
;;; REF:       - The go_angle a/2 can be calculated from the angles b and c:
;;; REF:             a/2 = (c - b)/2
;;; REF:
;;; REF:       - The go_angle b between the Y-ax and (x1,y1) is:
;;; REF:             tan( b ) = (y1 - y0) / (x1 - x0 + 1e-15)
;;; REF:             b = arctan2( (x1 - x0 + 1e-15), (y1 - y0) )
;;; REF:
;;; REF:       - The go_angle c between the Y-ax and (x2,y2) is:
;;; REF:             tan( c ) = (y2 - y0) / (x2 - x0 + 1e-15)
;;; REF:             c = arctan2( (x2 - x0 + 1e-15), (y2 - y0) )
;;; REF:
;;; REF: Calculation of R:
;;; REF:       - R = abs( r / tan(a/2) )        ;;; is a distance = positive
;;; REF:
;;; REF: Calculation of new endpoints (01) and (02):
;;; REF:       - (x01,y01):
;;; REF:              x01 = x0 + R*cos(b)
;;; REF:              y01 = y0 + R*sin(b)
;;; REF:       - (x02,y02):
;;; REF:              x02 = x0 + R*cos(c)
;;; REF:              y02 = y0 + R*sin(c)
;;; REF:
;;; REF: Calculation of Arc: i.e. centre of go_circle, arc1/arc2 and
;;; REF:                     go_bounding_width/go_bounding_height:
;;; REF:             sin line  ^
;;; REF:                    +1 |      .'
;;; REF:                    __-+-__ .' a/2
;;; REF:                   /   |  .\             _ -
;;; REF:               -1 /    |.'  \ +1     _ -    b
;;; REF:                -{-----0-----}----_----------------->
;;; REF:                  \    |\   / _ -                  cos line
;;; REF:                   \_  | \_/-    arc1 = -(90-b)
;;; REF:                    -1-+-'\ (01)
;;; REF:                       |    perpendicular on go_angle b
;;; REF:       - (xc,yc):
;;; REF:              xc = x01 - r*cos(arc1)
;;; REF:                 = x01 - r*cos(b-90)     or  = x01 - r*cos(b+90)
;;; REF:                 = x01 - r*sin( b )          = x01 + r*sin( b )
;;; REF:              yc = y01 - r*sin(arc1)
;;; REF:                 = y01 - r*sin(b-90)     or  = y01 - r*sin(b+90)
;;; REF:                 = y01 + r*cos( b )          = y01 - r*cos( b )
;;; REF:       - arc1/arc2 (in 64ths of a degree):
;;; REF:              if (b < c) then
;;; REF:                  arc1 = round( (b-90)*64 )
;;; REF:                  arc2 = round( ((c-b) - 180)*64 )
;;; REF:              elseif (b >= c) then
;;; REF:                  arc1 = round( (b+90)*64 )
;;; REF:                  arc2 = round( ((c-b) + 180)*64 )
;;; REF:              endif
;;; REF:       - go_bounding_width/go_bounding_height:
;;; REF:              go_bounding_width = go_bounding_height = 2*r
;;; REF:
;;; REF: Note: Reuse of tan() calculus avoids calling both sin() and cos()
;;; REF:              sin(a/2) = tan(a/2) * cos(a/2)
;;; REF:              sin(b) = tan(b) * cos(b)
;;; REF:              sin(c) = tan(c) * cos(c)
;;; REF:
 */

define :method go_get_round_lines( x1, y1, x0, y0, x2, y2, r, pane :go_pane, poly :go_polygon ) /* -> (Arcs, Segments) */;
lvars x1, y1, x0, y0, x2, y2, r, pane, poly;
;;; REF: go_get_round_lines( x1, y1, x0, y0, x2, y2, r, PANE, POLYGON ) -> (Arcs,Segments);
;;; REF: This finds the arcs and line segments needed to draw two lines
;;; REF: from the polygon rounded with radius "r" in the pane.
;;; REF: x1,y1 are the world coordinates of the startpoint of line one.
;;; REF: x0,y0 are the world coordinates of the common endpoint of the lines.
;;; REF: x2,y2 are the world coordinates of the startpoint of line two.
;;; REF: r is the integer radius (represented in world-coordinates.
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
;;; REF: Arcs is a arc-vector in screen coordinates.
;;; REF: Segments is a segment-vector in screen coordinates.
lvars tana2, a2, tanb, b, tanc, c, R;
lvars cosc, cosb, sinb, sinc;
lvars x01, y01, x02, y02, xc, yc, arc1, arc2, w, h;
lvars distance01, distance02, R2, rotsign, temp_x, temp_y;

	(x1 - x0) -> temp_x; (y1 - y0) -> temp_y;
	temp_x**2 + temp_y**2 -> distance01;

	arctan2( temp_x + 1e-10, temp_y ) -> b;
	if ( abs(temp_x) < 1e-10) then
		0 -> cosb; sin( b ) -> sinb;
	else
		cos( b )  -> cosb;
		temp_y / temp_x   /* -> tanb; tanb */  * cosb  -> sinb;
	endif;

	(x2 - x0) -> temp_x; (y2 - y0) -> temp_y;
	temp_x**2 + temp_y**2 -> distance02;
	arctan2( temp_x + 1e-10, temp_y ) -> c;
	if ( abs(temp_x) < 1e-10) then
		0 -> cosc; sin( c ) -> sinc;
	else
		cos( c )  -> cosc;
		temp_y / temp_x   /* -> tanc; tanc */  * cosc  -> sinc;
	endif;

	abs(c - b)/2 -> a2;
	if ( a2 < 1e-10) then
		go_transxyout( x01, y01, pane ) -> (x01, y01) ;
		go_transxyout( x02, y02, pane ) -> (x02, y02) ;
		[];                           ;;; NO ARC
		[% x01, y01, x02, y02 %]      ;;; START AND END POINT OF ARC
	else
		tan( a2 ) -> tana2;
		abs( r / tana2 ) -> R;
		(2 * R)**2 -> R2;
		if (R2 > distance01) or (R2 > distance02) then
			;;; 'Rounding too big' =>
			sqrt( min(distance01, distance02) ) / 2 -> R;
			abs(R * tana2) -> r;
		endif;

		x0  + R*cosb -> x01;
		y0  + R*sinb -> y01;
		x0  + R*cosc -> x02;
		y0  + R*sinc -> y02;

		if (a2 > 90) then
			if (b > c) then
				b - 90 -> arc1;
				((c - b) + 180) -> arc2;
				x01 - r*sinb -> xc;
				y01 + r*cosb -> yc;
			else
				b + 90 -> arc1;
				((c - b) - 180) -> arc2;
				x01 + r*sinb -> xc;
				y01 - r*cosb -> yc;
			endif
		elseif ( b < c) then
			b - 90 -> arc1;
			((c - b) - 180) -> arc2;
			x01 - r*sinb -> xc;
			y01 + r*cosb -> yc;
		else
			b + 90 -> arc1;
			((c - b) + 180) -> arc2;
			x01 + r*sinb -> xc;
			y01 - r*cosb -> yc;
		endif;

		;;; flip if needed: (positive ax-system if pane.go_yscale == -1)
		if ( pane.go_xscale < 0 ) then 180 - arc1 -> arc1; -arc2 -> arc2; endif;
		if ( pane.go_yscale > 0 ) then -arc1 -> arc1; -arc2 -> arc2; endif;
		;;; calculate in 64ths
		round( arc1 * 64 ) -> arc1;
		round( arc2 * 64 ) -> arc2;

		round( 2 * r * abs(pane.go_xscale) ) -> w;
		round( 2 * r * abs(pane.go_yscale) ) -> h;

		;;; WHERE IS TOP OF BOUNDING ?? CALC IN SCREEN COORD !!!
		go_transxyout( xc, yc, pane ) -> (temp_x, temp_y) ;
		round( temp_x - w/2 ) -> temp_x;
		round( temp_y - h/2 ) -> temp_y;

		go_transxyout( x01, y01, pane ) -> (x01, y01) ;
		go_transxyout( x02, y02, pane ) -> (x02, y02) ;

		[% temp_x, temp_y, w, h, arc1, arc2 %];  ;;; ARC
		[% x01, y01, x02, y02 %];                ;;; START & END POINT OF ARC
	endif;


enddefine;



;;; Routine that calculates a set of multiple rounded lines
define :method go_get_round_polylines( pane, poly :go_polygon )
											 /* -> {% ArcList, SegList %} */;
lvars pane, poly ;
lvars PtsList, n, r ;
;;; REF: go_get_round_polylines( PANE, POLYGON ) -> (ArcList,SegmentList);
;;; REF: This finds the arcs and line segments needed to draw all the lines
;;; REF: from the polygon rounded with radius "r" in the pane. This uses
;;; REF: the go_get_round_lines() method.
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
;;; REF: ArcList is a list of arc-vectors in screen coordinates.
;;; REF: SegmentList is a list of segment-vectors in screen coordinates.
lvars m, i, start_index, end_index, top_coord;
lvars Arc, Segment, ArcList, SegList;

	poly.go_world_coords -> PtsList;
	poly.go_npoints -> n;
	poly.go_rounding -> r;

	[] -> ArcList ;
	2*n -> m;
	if ( poly.go_closed ) then
		0 -> start_index;
		2*n - 2 -> end_index;
		[] -> SegList;
	else
		2 -> start_index;
		2*n - 4 -> end_index;
		[% go_transxyout(PtsList(1), PtsList(2), pane) %]
		-> SegList;
	endif;
	for i from start_index by 2 to end_index do
		 go_get_round_lines( PtsList(((i-2)mod m)+1),PtsList(((i-1)mod m)+1),
							 PtsList(i+1),           PtsList(i+2),
							 PtsList(((i+2)mod m)+1),PtsList(((i+3)mod m)+1),
							 r, pane, poly ) -> (Arc, Segment) ;
		 SegList <> Segment -> SegList;
		 ArcList <> Arc -> ArcList;
	endfor;

	if ( poly.go_closed ) then
		;;; move the first coordinates to the end
		SegList.destpair.destpair -> SegList;
		conspair(conspair([])) -> top_coord;
		SegList nc_<> top_coord -> SegList;
	else
		SegList <> [% go_transxyout(PtsList(2*n-1), PtsList(2*n),
									 pane) %]
		-> SegList;
	endif;

	{% ArcList, SegList %};
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_roundlines = true;

/* --- Revision History --------------------------------------------
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 28/04/93
 *     Changed the comment layout.
 * BR 08/04/93
 *     Removed direct call to XpwDrawSegments() by go_draw_line_segments();
 *     Removed direct call to XpwDrawArcs() by go_draw_arcs();
 */
;;; eof
