/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_composite.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 23rd June 1993
;;; File: go_composite.p
;;; Author: B Rabau

compile_mode :pop11 +strict;

;;; INTRO: The MIXIN go_composite is for objects which consists of multiple
;;; INTRO: elements (such as go_group and go_pane). This information will be
;;; INTRO: used to enable a composite object to be edited as a whole or each
;;; INTRO: of its constituting elements separately. See REF * GO_GROUP and
;;; INTRO: REF * GO_PANE.

uses go_xdefaults;

/* Extra dependencies further in this file:
uses go_screen_object;        ;;; FOR MULTI-METHODS
 */

define :mixin go_composite;
	slot go_is_editable    == false;
;;; REF: Boolean: whether or not elements of composite can be edited.
	slot the_go_components == [];
;;; REF: The handles to the components of the composite group.
enddefine;

;;; ---------------------------------------------------------
;;; DEFAULT FONT

define :method go_font( grp :go_composite ) -> the_font;
lvars grp, the_font;
;;; REF: go_font( COMPOSITE ) -> XFONT;
;;; REF: Gets the default font of the composite object, which is the
;;; REF: default font from its first parent.
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
		if go_visible_in(grp) == [] then
			;;; Make sure it has a font if it was never visualised up to now
			go_default_font -> the_font;
		else
			go_font( go_visible_in(grp)(1) ) -> the_font;
		endif;
enddefine;

;;; ---------------------------------------------------------
;;; MAKE EDITABLE
;;; If the go_composite is "go_is_editable" then its elements can be moved
;;; separately!
;;; NOTE:
;;;       Currently there is no visualisation of this mode!!!

define :method go_make_editable( grp :go_composite );
lvars grp;
;;; REF: go_make_editable( COMPOSITE );
;;; REF: Make the composite object editable (slot "go_is_editable"). This
;;; REF: is currently based on a test in go_find_ui_object in LIB * GO_PANE
;;; REF: If the go_composite is "go_is_editable" then its elements can be
;;; REF: moved separately!
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	true -> go_is_editable( grp );
enddefine;

define :method go_make_uneditable( grp :go_composite );
lvars grp;
;;; REF: go_make_editable( COMPOSITE );
;;; REF: Stop the composite object from being editable (slot "go_is_editable").
;;; REF: If the go_composite is not "go_is_editable" then its elements can't
;;; REF: be moved separately, the composite object reacts as a single unit.
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	false -> go_is_editable( grp );
enddefine;


;;;----------------------------------------------------------
;;; MULTI-METHODS USING GO_SCREEN_OBJECT
uses go_screen_object;
;;;----------------------------------------------------------

;;;----------------------------------------------------------
;;; MEMBERSHIP FUNCTIONS (CHILDREN)

define :method go_is_in( obj :go_screen_object, grp :go_composite );
lvars obj, grp;
;;; REF: go_is_in( SCREEN_OBJECT, COMPOSITE ) -> BOOLEAN;
;;; REF: The boolean true if the screen object is a component of the object,
;;; REF: else it is false.
;;; REF: Note: objects in a go_pane might or might not be visible by the user
;;; REF: at this point since it could be hidden below other objects or be
;;; REF: outside the visible part of the pane on the screen or even be
;;; REF: temporarily cleared.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	if ( lmember(obj, grp.the_go_components) ) then true else false endif;
enddefine;

define :method go_remove_from( obj :go_screen_object, grp :go_composite );
lvars obj, grp;
;;; REF: go_remove_from( SCREEN_OBJECT, COMPOSITE );
;;; REF: Removes the screen object from the list of objects visible in the given
;;; REF: composite object.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	if ( go_is_in(obj, grp) ) then
		go_clear( obj );
		delete(grp, obj.go_visible_in, nonop ==) -> obj.go_visible_in;
		delete(obj, grp.the_go_components, nonop ==) -> grp.the_go_components;
		go_redraw( obj );
	endif;
enddefine;

;;; Add to the beginning of the list:
define :method go_add_to( obj :go_screen_object, grp :go_composite );
lvars obj, grp;
;;; REF: go_add_to( SCREEN_OBJECT, COMPOSITE );
;;; REF: Makes the screen object part of the objects visible in the given
;;; REF: composite group. The screen object will appear on top of every
;;; REF: other object in the composite group when drawn.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	unless ( go_is_in(obj, grp) ) then
		grp :: (obj.go_visible_in) -> obj.go_visible_in;
	endunless;
	if ( lmember(obj, grp.the_go_components) ) then
		pr('Warning: the object is already included!\n');
	else
		obj :: (grp.the_go_components) -> grp.the_go_components;
		go_redraw( obj );
	endif;
enddefine;

;;; Add to the end of the list:
define :method go_annex_to( obj :go_screen_object, grp :go_composite );
lvars obj, grp;
;;; REF: go_annex_to( SCREEN_OBJECT, COMPOSITE );
;;; REF: Makes the screen object part of the objects visible in the given
;;; REF: composite group. The screen object will appear below every
;;; REF: other object in the composite group when drawn.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	unless ( go_is_in(obj, grp) ) then
		grp :: (obj.go_visible_in) -> obj.go_visible_in;
	endunless;
	if ( lmember(obj, grp.the_go_components) ) then
		pr('Warning: the object is already included!\n');
	else
		(grp.the_go_components) <> [% obj %] -> grp.the_go_components;
		go_redraw( obj );
	endif;
enddefine;

;;; ---------------------------------------------------------
;;; REORDERING

define :method go_raise( obj :go_screen_object, grp :go_composite );
lvars obj, grp;
;;; REF: go_raise( SCREEN_OBJECT, COMPOSITE );
;;; REF: Puts the screen object on top of all other objects in the given parent.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	go_remove_from( obj, grp );
	go_add_to( obj, grp );
enddefine;

define :method go_lower( obj :go_screen_object, grp :go_composite );
lvars obj, grp;
;;; REF: go_lower( SCREEN_OBJECT, COMPOSITE );
;;; REF: Moves the screen object under of all other objects in the given parent.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
	go_remove_from( obj, grp );
	go_annex_to( obj, grp );
enddefine;


;;; ---------------------------------------------------------
;;; DECOMPOSITION

;;; Note: uses "go_accepts_events" from go_screen_object class
;;; Note: uses "go_forward_event_to" from go_screen_object class
;;; Note: uses "go_transparent_to_events" from go_screen_object class

define :method go_find_ui_object(x, y, type, raw_event_data, grp :go_composite ) -> accepted_flag;
lvars x, y, type, raw_event_data, grp, accepted_flag = false;
;;; REF: go_find_ui_object( X, Y, TYPE, RAW_EVENT_DATA, COMPOSITE ) -> BOOLEAN;
;;; REF: If a component of the composite contains the location (X,Y) and
;;; REF: it accepts the type of events, then return true, otherwise if the
;;; REF: group accepts the event and the location is in the group then the
;;; REF: group takes the event. If none of the above then it is false.
;;; REF: See also REF * GO_SCREEN_OBJECT/go_accepts_events.
;;; REF: The objects which are found are added to the global variable
;;; REF: go_forward_event_to (see REF * GO_PANE/go_forward_event_to and
;;; REF: REF * GO_PANE/go_transparent_to_events).
;;; REF: X and Y are integer world coordinates of the test position.
;;; REF: TYPE is one of "mouse" or "keyboard"
;;; REF: RAW_EVENT_DATA is a vector (see REF * GO_XACTION/go_expand_event_data)
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
lvars component, contains_flag = false, found_obj = false;
 ;;; start from false until otherwise found
	for component in rev( grp.the_go_components ) do
		if component.isgo_composite then
			if go_find_ui_object(x, y, type, raw_event_data, component) then
				true -> accepted_flag;
				returnif( not(go_transparent_to_events) );
				false -> go_transparent_to_events;
			endif;
		else
			go_contains( x, y, component ) -> found_obj;
			if found_obj then
				true -> contains_flag;
				if go_accepts_events( type, raw_event_data, found_obj ) then
					go_forward_event_to <> [^found_obj] -> go_forward_event_to;
					true -> accepted_flag;
					returnif( not(go_transparent_to_events) );
					false -> go_transparent_to_events;
				endif;
			endif;
		 endif;
	endfor;
	if contains_flag and go_accepts_events( type, raw_event_data, grp ) then
		go_forward_event_to <> [^grp] -> go_forward_event_to;
		true -> accepted_flag;
	endif;
enddefine;


;;; ---------------------------------------------------------
;;; REDRAWING

define :method go_redraw_area(show_scrObj, draw_below, grp :go_composite, scrObj:go_screen_object);
lvars show_scrObj, draw_below, grp, scrObj;
;;; REF: go_redraw_area( DRAW, BELOW, PANE, GO_COMPOSITE );
;;; REF: This recusively calls REF * GO_SCREEN_OBJECT/go_redraw_area
;;; REF: on all containers in which the composite objetc is visible (see
;;; REF: also REF * GO_SCREEN_OBJECT/go_visible_in.
;;; REF: Redraws the area occupied by the screen object if requested by
;;; REF: redrawing all the objects underneath the screen object, then if
;;; REF: requested the screen object itself, then all the objects and editors
;;; REF: on top of it. This is used by the go_redraw and go_clear methods.
;;; REF: DRAW is a boolean indicating if the object itself needs to be drawn.
;;; REF: BELOW is a boolean indicating if objects underneath are requested.
;;; REF: COMPOSITE is a go_composite instance (see REF * GO_COMPOSITE)
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
lvars container;
	for container in grp.go_visible_in do
		go_redraw_area(show_scrObj, draw_below, container, scrObj);
	endfor;
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_composite = true;

/* --- Revision History --------------------------------------------
 * BR 06/07/93
 *     Changed the return parameter of go_find_ui_object() to be a boolean and
 *     the objects which it finds are added to the global variable:
 *     go_forward_event_to (see also global variable go_transparent_to_events).
 * BR 23/06/93
 *     Major redesign of go_find_ui_object() to include event type; now uses
 *     go_accepts_events().
 * BR 05/05/93
 *     Added comments.
 */
;;; eof
