/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_polygon.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 23rd Aug 1993
;;; File: go_polygon.p
;;; Author: B L Rabau

compile_mode :pop11 +strict;

;;; INTRO: The CLASS go_polygon contains both regular and irregular open
;;; INTRO: and closed multi-angled lines. See also REF * GO_POLYLINE for
;;; INTRO: a more specialised version of open polygons.

uses go_screen_object;
uses go_fillable;
uses go_window_pane;
uses go_edit_point;
uses go_rotatable;


/* Extra dependencies further in this file:
uses go_xdraw;
uses go_pane;
 */

include XpwPixmap;

;;;------------------------------------------------------
;;; USEFUL FACILITY

;;; Convenience function
define close_to_a_line( x1, y1, x2, y2, x, y, dist );
lvars x1, y1, x2, y2, x, y, dist;
	;;; In the direct nabourhood of max #go_border pixels of!
	if (y1 = y2) then
		if (abs(y-y1) <= dist) and (x >= min(x1, x2)) and (x <= max(x1, x2))
		then true else false endif;
	elseif ( abs( ((x1-x2)*1.0/(y1-y2))*(y-y1) - (x-x1) ) >= dist ) then false;
	elseif ( x < min(x1, x2) ) or ( y < min(y1, y2) ) then false;
	elseif ( x > max(x1, x2) ) or ( y > max(y1, y2) ) then false;
	else   true;
	endif;
enddefine;

;;;------------------------------------------------------
vars procedure go_initialise_polygon;

define :class go_polygon;
	isa go_fillable go_rotatable go_screen_object;
	slot stored_go_npoints  ==  3;
;;; REF: Number of angles in the polygon.
	slot stored_go_radius   == 40;
;;; REF: Stored value of the radius (outer circle) in world coordinates.
	slot stored_go_rounding ==  0;
;;; REF: Stored value of the rounding radius of corners.
	slot stored_go_closed   == true;
;;; REF: Boolean: whether it is a closed polygon or an open one.
	slot stored_go_regular  == true;
;;; REF: Boolean: whether it is a regular polygon or not.

	on new do procedure() -> poly;
			  lvars poly = apply();
				  go_initialise_polygon( poly );
			  endprocedure;
enddefine;


;;;------------------------------------------------------
;;; OPEN/CLOSED: POLYLINE/POLYGON

define :method go_closed( poly :go_polygon ) -> boolean;
lvars poly , boolean;
;;; REF: go_closed( POLYGON ) -> BOOLEAN;
	poly.stored_go_closed -> boolean;
enddefine;

define :method updaterof go_closed( closing, poly :go_polygon );
lvars closing, poly;
;;; REF: BOOLEAN -> go_closed( POLYGON );
;;; REF: True if it the polygon is open, false if it is closed.
;;; REF: Open polygons have one open side, closed polygons have a closing
;;; REF: line between the last and the first point in go_local_coords.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	unless ( closing == poly.stored_go_closed ) do
		closing -> poly.stored_go_closed;
		if closing then
			go_update_and_draw( poly );          ;;; only go_internal_redraw
		else
			go_update( poly ); go_redraw( poly );
		endif;
	endunless;
enddefine;

;;;------------------------------------------------------
;;; REGULAR/IRREGULAR POLYGON

define :method go_regular( poly :go_polygon ) -> boolean;
lvars poly , boolean;
;;; REF: go_regular( POLYGON ) -> BOOLEAN;
	poly.stored_go_regular -> boolean;
enddefine;

vars procedure go_make_editable;       ;;; defined below
vars procedure go_make_uneditable;     ;;; defined below

define :method updaterof go_regular( regularity, poly :go_polygon );
lvars regularity, poly;
;;; REF: BOOLEAN -> go_regular( POLYGON );
;;; REF: True if it the polygon is regular, false if it is irregularly shaped.
;;; REF: Regular polygons have symmetric convex shapes like equilateral
;;; REF: triangles, squares, pentagons, etc. If a polygon is changed from an
;;; REF: irregular to a regular shape, only the number of points remains (see
;;; REF: method "go_npoints").
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	if not( poly.stored_go_regular ) and ( regularity ) then
		go_clear( poly );
		regularity -> poly.stored_go_regular;
		go_initialise_polygon( poly );
	else
		regularity -> poly.stored_go_regular;
	endif;
	if not( poly.go_editors == [] ) then
		go_make_editable( poly );
	endif;
enddefine;

;;;------------------------------------------------------
;;; GEOMETRY
;;;     since all calculus is done from centre => changes needed

define :method go_radius( poly :go_polygon ) -> radius;
lvars poly , radius;
;;; REF: go_radius( POLYGON ) -> INTEGER;
	poly.stored_go_radius -> radius;
enddefine;

define :method updaterof go_radius( radius, poly :go_polygon );
;;; REF: INTEGER -> go_radius( POLYGON );
;;; REF: The radius of regular polygons. The radius is in world coordinates
;;; REF: and is only used for regular polygons. The radius is calculated
;;; REF: from the centre to the first point in the go_local_coords.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
lvars radius, poly;
	go_clear( poly );
	round(radius) -> poly.stored_go_radius;
	go_initialise_polygon( poly );
enddefine;

define :method go_npoints( poly :go_polygon ) -> n;
lvars poly , n;
;;; REF: go_npoints( POLYGON ) -> INTEGER;
	poly.stored_go_npoints -> n;
enddefine;


define :method updaterof go_npoints( n, poly :go_polygon );
lvars n, poly;
;;; REF: INTEGER -> go_npoints( POLYGON );
;;; REF: The number of angles of the given polygon.
;;; REF: The number of points needs to be between 1 and 64. If the polygon
;;; REF: is regular the new shape is calculated. If it is irregular, then a
;;; REF: number of extra points is added on existing lines.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
lvars flag_eds = false;
	if ( n > 1 ) and ( n < 64 ) then        ;;; arbitrary maximum
		unless ( poly.go_editors == [] ) then
			true -> flag_eds; go_make_uneditable( poly );
		endunless;
		go_clear( poly );
		n -> poly.stored_go_npoints;
		go_initialise_polygon( poly );
		if ( flag_eds ) then go_make_editable( poly ); endif;
	else
		mishap('cannot update number of sides of go_polygon to', [^n]);
	endif;
enddefine;

;;; --------------------------------------------------------------------
;;; INFLUENCING THE ROUNDING OF POLYGONS

vars go_redraw;		;;; DEFINED BELOW

define :method go_rounding( poly :go_polygon ) -> radius;
lvars poly , radius;
;;; REF: go_rounding( POLYGON ) -> INTEGER;
	poly.stored_go_rounding -> radius;
enddefine;

define :method updaterof go_rounding( radius, poly :go_polygon );
lvars radius, poly;
;;; REF: INTEGER -> go_rounding( POLYGON );
;;; REF: The rounding radius of the polygon. If go_rounding is bigger than 3
;;; REF: then the angles between the lines in the polygon will be rounded.
;;; REF: The rounding can never become negative (see also REF * GO_ROUNDLINES).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	if (radius >= 0) then
		go_clear( poly );
		radius -> poly.stored_go_rounding;
		go_update_and_draw( poly );
	else
		mishap('cannot accept negative rounding of a go_polygon', [^radius]);
	endif;
enddefine;

define :method go_smooth( poly :go_polygon );
lvars poly;
;;; REF: go_smooth( POLYGON );
;;; REF: Increase the rounding radius of the polygon by 5 making it smoother.
;;; REF: (see method REF * GO_POLYGON/go_rounding).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	poly.go_rounding + 5 -> poly.go_rounding;
	go_redraw( poly );
enddefine;

define :method go_roughen( poly :go_polygon );
lvars poly;
;;; REF: go_roughen( POLYGON );
;;; REF: Decrease the rounding radius of the polygon by 5 making it less smooth.
;;; REF: (see method "go_rounding"). The rounding can never become negative.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	max( 0, poly.go_rounding - 5 ) -> poly.go_rounding;
	go_redraw( poly );
enddefine;

;;; Load the library that calculates rounded lines.
uses go_roundlines;

;;;------------------------------------------------------------------
;;; POINTS (actual go_local_coords of go_polygon; e.g. in restoring shape)

vars procedure go_set_bounding_box;   ;;; DEFINED BELOW

define :method updaterof go_local_coords( pts_list, poly :go_polygon );
lvars pts_list, poly;
;;; REF: LIST -> go_local_coords( POLYGON );
;;; REF: The world coordinates of the corner points of the polygon. The list
;;; REF: contains a succession of X and Y coordinates of all the points of
;;; REF: the polygon. If the polygon is closed then the first point is not
;;; REF: repeated for the closing line.
;;; REF: LIST is a list world coordinates [X1 Y1 X2 Y2 ...] stored as reals.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	procedure;
	dlocal 1 % go_batch_mode(poly) % = true;
	lvars n, l, flag_eds = false;
		length( pts_list ) -> l;
		l div 2 -> n;
		if ( n * 2 == l ) then
			unless poly.go_editors == [] or n == poly.stored_go_npoints then
				true -> flag_eds;
			endunless;
			n -> stored_go_npoints( poly );
			call_next_method( pts_list, poly );
			go_set_bounding_box( poly );
			if ( flag_eds ) then go_make_editable( poly ); endif;
		else
			mishap('list of go_polygon local coordinates not paired',
				   [^pts_list]);
		endif;
	endprocedure();
enddefine;


;;; --------------------------------------------------------------------
;;; ADAPT COORDINATES OF STORED LISTS (LOCAL COORDS)

uses go_xregions;

;;;------------------------------------------------------
;;; INITIALISING DATA

define :method go_calc_regular( poly :go_polygon ) -> the_points;
lvars poly, the_points;
;;; REF: go_calc_regular( POLYGON ) -> POINTS;
;;; REF: Calculates the absolute world coordinates for a given polygon in the
;;; REF: event of a regular shape. This takes account of the current centre
;;; REF: of the polygon, the number of angles, the rotation and the scaling
;;; REF: of the object (not the scaling of the panes it is in).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
lvars i, n = poly.go_npoints, radius = poly.go_radius;
lvars offset = 360 / n, cosoffset = cos( offset ), sinoffset = sin( offset );
lvars cosangle = 1.0, sinangle = 0.0;
	[%
	   for i from 1 to n do
		  ;;; LOCAL COORDINATES (vs. CENTRE)
		  round( cosangle * radius)  /* -> newx */;
		  round( sinangle * radius)  /* -> newy */;
		  cosangle*cosoffset - sinangle*sinoffset ; ;;; NEW cosangle
		  cosangle*sinoffset + sinangle*cosoffset ; ;;; NEW sinangle
		  -> (cosangle, sinangle) ;                 ;;; OLD no longer needed
	   endfor;
	%] -> the_points;
enddefine;

define :method go_calc_irregular( poly :go_polygon ) -> the_points;
lvars poly, the_points;
;;; REF: go_calc_irregular( POLYGON ) -> POINTS;
;;; REF: Calculates the relative world coordinates for a given polygon in the
;;; REF: event of an irregular shape. This takes account of the current points
;;; REF: of the polygon and the number of angles. If any angles need to be
;;; REF: added then they will be added along the already existing lines. The
;;; REF: first time it is calculated the polygon is considered regular.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
lvars old_length, new_length, diff_length, extra, extra1, m;
lvars x, y, firstx, firsty, nextx, nexty, pointlist = poly.go_local_coords;
	listlength( pointlist ) -> old_length;
	poly.go_npoints * 2 -> new_length;
	if ( new_length < old_length ) then
		[% (old_length - new_length) -> diff_length;
		   (diff_length - 2) div new_length + 1 -> extra; ;;; pts to discard
		   until pointlist == [] do
			   fast_destpair( pointlist ) -> pointlist;   ;;; stack x
			   fast_destpair( pointlist ) -> pointlist;   ;;; stack y
			   for m from 1 to extra do
				   if ( diff_length > 0 ) then            ;;; discard pts
					   fast_destpair( pointlist ) -> pointlist ->;
					   fast_destpair( pointlist ) -> pointlist ->;
					   diff_length - 2 -> diff_length;
				   endif;
			   endfor;
		   enduntil;
		 %] -> the_points;
	elseif ( new_length > old_length ) then
		;;; add a number of go_local_coords on the lines between the points
		[% (new_length - old_length) -> diff_length;
		   (diff_length - 2) div old_length + 1 -> extra; ;;; pts to add
		   extra + 1 -> extra1;
		   fast_destpair( pointlist ) -> pointlist ->> firstx
												   ->> x; ;;; stack x
		   fast_destpair( pointlist ) -> pointlist ->> firsty
												   ->> y; ;;; stack y
		   until pointlist == [] do
			   fast_destpair( pointlist ) -> pointlist -> nextx;
			   fast_destpair( pointlist ) -> pointlist -> nexty;
			   for m from 1 to extra do
				   if ( diff_length > 0 ) then            ;;; add between
					   x + (nextx - x) / extra1 * m;      ;;; stack x
					   y + (nexty - y) / extra1 * m;      ;;; stack y
					   diff_length - 2 -> diff_length;
				   endif;
			   endfor;
			   nextx ->> x;                               ;;; stack x
			   nexty ->> y;                               ;;; stack y
		   enduntil;
		   diff_length div 2 -> extra;                    ;;; still to add
		   extra + 1 -> extra1;
		   for m from 1 to extra do
			   if ( diff_length > 0 ) then
				   x + (firstx - x) / extra1 * m;         ;;; stack x
				   y + (firsty - y) / extra1 * m;         ;;; stack y
				   diff_length - 2 -> diff_length;
			   endif;
		   endfor;
		 %] -> the_points;
	else
		poly.go_local_coords -> the_points;
	endif;
enddefine;


define :method go_set_bounding_box( poly :go_polygon );
lvars poly;
;;; REF: go_set_bounding_box( POLYGON );
;;; REF: Recalculates the bounding box of the polygon containing all the
;;; REF: corner points and recalculates the centre (see also REF * GO_LOCATED).
;;; REF: The centre of a regular polygon does not necessarily correspond to
;;; REF: the centre of the bounding box (Imagine a triangle where the centre
;;; REF: is approximately one third of the height if the base is flat).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
lvars pointlist, the_radius;
	;;; find the go_bounding_box of the object
	if go_regular( poly ) then
		go_radius( poly ) -> the_radius;
		if (the_radius > 0) then max( 1, the_radius * 2);
							else min(-1, the_radius * 2);
		endif ->> poly.stored_go_bounding_width
			  ->  poly.stored_go_bounding_height;
	else
	lvars x, y, xmin, xmax, ymin, ymax, dx, dy;
		go_local_coords( poly ) -> pointlist;
		fast_destpair( pointlist ) -> pointlist ->> xmin ; -> xmax;
		fast_destpair( pointlist ) -> pointlist ->> ymin ; -> ymax;
		until pointlist = [] do
			fast_destpair( pointlist ) -> (x, pointlist);
			fast_destpair( pointlist ) -> (y, pointlist);
			min( x, xmin ) -> xmin;
			max( x, xmax ) -> xmax;
			min( y, ymin ) -> ymin;
			max( y, ymax ) -> ymax;
		enduntil;

		(xmax + xmin) div 2 -> dx;
		(ymax + ymin) div 2 -> dy;
		if ( dx /= 0) or ( dy /= 0) then
		;;; WRONG: local coordinates supposedly are centred around x/ycentre
		;;;        the coordinates can maybe be translated back to that centre
		;;;        or a rotation point can be calculated...
			go_translate_local_coords( -dx, -dy, poly )
			-> stored_go_local_coords( poly );
			dx * poly.go_xscale -> dx;  dy * poly.go_yscale -> dy;
			poly.stored_go_xorigin + dx -> poly.stored_go_xorigin;
			poly.stored_go_yorigin + dy -> poly.stored_go_yorigin;
			;;; Keep rotation point on the same absolute spot
			poly.stored_go_xrotation - dx -> poly.stored_go_xrotation;
			poly.stored_go_yrotation - dy -> poly.stored_go_yrotation;
		endif;
		max(1, round(xmax - xmin)) -> poly.stored_go_bounding_width;
		max(1, round(ymax - ymin)) -> poly.stored_go_bounding_height;
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; THIS IS THE END OF THE SINGLE-METHODS
;;;----------------------------------------------------------------

;;;----------------------------------------------------------------
;;; THIS IS THE START OF THE MULTI-METHODS
;;;----------------------------------------------------------------

;;;----------------------------------------------------------------
;;; MULTI-METHODS USING GO_PANE
uses go_pane;
;;;----------------------------------------------------------------

;;; SCREEN SPACE USAGE:

define :method go_safe_region( pane :go_pane, poly :go_polygon ) -> (x,y,w,h);
lvars pane, poly;
;;; REF: go_safe_region( PANE, POLYGON ) -> (X, Y, W, H);
;;; REF: Returns a tuple with the screen region occupied by the polygon
;;; REF: in the given pane. It includes a safety border which includes line
;;; REF: width and space for editors if any (see REF * GO_EDITOR). If the
;;; REF: polygon is irregular then extra space is foreseen for the effect of
;;; REF: JoinMiter line style whish makes sharp corners where lines join.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
;;; REF: X and Y are screen coordinates of the bottom left corner of the region.
;;; REF: W and H are the width and height in screen coordinates of the region.
	call_next_method( pane, poly ) -> (x, y, w, h);
	if not(poly.go_regular) then
		lvars obj_lw = (poly.go_linewidth or 1), pane_lw = pane.go_linewidth,
			  extra_space = max(obj_lw, pane_lw) * 2;
		;;; Approx. to take into account the JoinStyle == JoinMiter in polygons
		;;; The factor 5.20 was calculated for big linewidths
		;;; round(extra_space * 5.20) div 2 -> extra_space;
		if ( extra_space > 2 ) then
			x - extra_space -> x;
			y - extra_space -> y;
			extra_space + w + extra_space -> w;
			extra_space + h + extra_space -> h;
		endif;
	endif;
enddefine;


define :method go_initialise_polygon( poly :go_polygon );
lvars poly;
;;; REF: go_initialise_polygon( POLYGON );
;;; REF: Recalculates the points of the polygon taking the regularity into
;;; REF: account and then redraws it.
;;; REF: POLYGON is an instance of the go_polygon class (see REF * GO_POLYGON).
	if ( poly.go_regular ) or ( poly.go_local_coords == [] ) then
		go_calc_regular( poly ) -> poly.go_local_coords;
	else
		go_calc_irregular( poly ) -> poly.go_local_coords;
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; METHODS USING DRAWING REGIONS: DEFAULT X-WINDOWS:
uses go_xregions;
;;;----------------------------------------------------------------

;;; DRAWING

define :method go_screen_coords( pane :go_pane, poly :go_polygon );
lvars pane, poly;
;;; REF: go_screen_coords( PANE, POLYGON ) -> VECTOR;
;;; REF: Returns a cached or recalculated vector of screen coordinates for
;;; REF: the polygon in the given pane. The vector contains the list of arcs
;;; REF: of rounded polygons (or else false), and a list of line segments.
;;; REF: VECTOR is a tuple of a list of arcs and a list of line segments.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pointlist, x, y, xmin, xmax, ymin, ymax;
lvars oldx, oldy, newx, newy, firstx, firsty;
	if ( cached_go_coord_list(pane)(poly) ) then
		;;; valid screen coordinates saved
		cached_go_coord_list(pane)(poly);
	else
		if (cached_go_coord_reg(pane)(poly)) then
			XDestroyRegion( cached_go_coord_reg(pane)(poly) );
			false -> cached_go_coord_reg(pane)(poly);
		endif;
		if (poly.go_rounding < 3) then
			;;; calculate screen coordinates and save
			go_world_coords( poly ) -> pointlist;
			{% false,
			   [%
				  go_translistout(pointlist, pane) -> pointlist;
				  explode(pointlist);                 ;;; Put points on stack
				  if poly.go_closed then              ;;; Closed polygon?
					   pointlist(1); pointlist(2);    ;;; Put first X,Y on stack
				  endif;
			   %]
			 %} ->> cached_go_coord_list(pane)(poly);
		else
			go_get_round_polylines( pane, poly )
			->> cached_go_coord_list(pane)(poly);
		endif;
	endif;
enddefine;

define :method go_fgdraw( pane :go_pane, poly :go_polygon );
lvars pane, poly;
;;; REF: fgdraw( PANE, POLYGON );
;;; REF: This draws the part of the polygon in foreground colour on the pane.
;;; REF: If the foreground colour is false the default from the given pane is
;;; REF: used (see also REF * GO_COLOURABLE).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars coordsvector;
	go_screen_coords( pane, poly ) -> coordsvector;

	if ( coordsvector(1) ) then
		go_draw_arcs( pane,  coordsvector(1) );
		go_draw_line_segments( pane, coordsvector(2) );
	else
		go_draw_lines( pane, coordsvector(2), CoordModeOrigin );
	endif;
enddefine;

define :method go_bgdraw( pane :go_pane, poly :go_polygon );
lvars pane, poly;
;;; REF: bgdraw( PANE, POLYGON );
;;; REF: This draws the part of the polygon in background colour on the pane.
;;; REF: If the background colour is false the default from the given pane is
;;; REF: used (see also REF * GO_COLOURABLE).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars coordsvector, the_shape;
	go_screen_coords( pane, poly ) -> coordsvector;

	if poly.go_regular then Convex else Complex endif -> the_shape;
	go_draw_filled_polygon( pane, coordsvector(2), the_shape, CoordModeOrigin);
	if ( coordsvector(1) ) then
		go_draw_filled_arcs(pane, coordsvector(1));
	endif;
enddefine;


;;;------------------------------------------------------
;;; CONTAINS

define :method go_device_coords_xregion( pane: go_pane, poly :go_polygon );
lvars poly;
;;; REF: go_device_coords_xregion( PANE, POLYGON ) -> XREGION;
;;; REF: This replaces the screen coordinates for a given pane.
;;; REF: Returns the cached or calculated Xregion . This is used to find
;;; REF: overlapping with other objects or locations by using the existing
;;; REF: X-facilities (REF * GO_POLYGON/go_overlaps * GO_POLYGON/go_contains).
;;; REF: XREGION is a widget representing a graphics region (REF * XREGION).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pointlist, points_vec, reg, extra_space;
lvars nbr, xabs, yabs, xpoint, ypoint, xfirst, yfirst;
lvars coordsvector, old_rounding = false, old_coordvector = false;
	;;; without absolute position - rest is relative to it
	if  ( cached_go_coord_reg(pane)(poly) )
	and ( cached_go_coord_list(pane)(poly)) then
		cached_go_coord_reg(pane)(poly);
	else
		if (cached_go_coord_reg(pane)(poly)) then
			XDestroyRegion( cached_go_coord_reg(pane)(poly) );
			false -> cached_go_coord_reg(pane)(poly);
		endif;
		if not(cached_go_coord_list(pane)(poly))
		or (cached_go_coord_list(pane)(poly)(1)) then
			;;; TEMPORARILY OVERRIDE THE ROUNDED SHAPE
			stored_go_rounding( poly ) -> old_rounding;
			cached_go_coord_list(pane)(poly)  -> old_coordvector;
			0     -> stored_go_rounding( poly );
			false -> cached_go_coord_list(pane)(poly);
		endif;
		go_screen_coords( pane, poly ) -> coordsvector;
		coordsvector(2) -> pointlist;
		if ( old_rounding ) then
			;;; REINSTATE TEMPORARILY OVERRIDEN ROUNDED SHAPE
			old_rounding    -> stored_go_rounding( poly );
			old_coordvector -> cached_go_coord_list(pane)(poly);
		endif;
		pointlist(1) -> xfirst;
		pointlist(2) -> yfirst;
		applist( pointlist, round );    ;;; Put rounded points on the stack
		poly.go_npoints -> nbr;

		;;; the line-thickness SHOULD BE OF THE OBJECT !!!
		if (poly.go_linewidth) then
			max(poly.go_linewidth, pane.go_linewidth) div -2;
		elseif (pane.go_linewidth > 1) then
			pane.go_linewidth div -2;
		else 1; endif -> extra_space;

		if ( poly.go_closed ) then
			consshortvec( (nbr + 1) * 2 ) -> points_vec;
		else
			conslist( nbr * 2 ) -> pointlist;
			explode( pointlist );
			rev( pointlist ) -> pointlist;
			until pointlist = [] do
				fast_destpair( pointlist ) -> (ypoint, pointlist) ;
				fast_destpair( pointlist ) -> (xpoint, pointlist) ;
				;;; The following goes wrong if new point is on an old line
				xpoint + extra_space; ypoint - extra_space;
			enduntil;
			round(xfirst); round(yfirst);
			consshortvec( (nbr * 4) + 2 ) -> points_vec;
			nbr * 2 -> nbr;
		endif;
		XPolygonRegion( points_vec, nbr, WindingRule )
		->> cached_go_coord_reg(pane)(poly) ->> reg;

		unless (extra_space == 1) then
			XShrinkRegion( reg, extra_space, extra_space );
		endunless;
	endif;
enddefine;

/* CONVENIENCE FUNCTION WHICH COULD BE IN go_screen_object.p (Ian Rogers) */
define :method go_common_pane(obj1 :go_screen_object, obj2 :go_polygon) -> pane;
lvars obj1, obj2, pane;
;;; REF: go_common_pane( SCREEN_OBJECT, POLYGON ) -> GO_PANE;
;;; REF: Returns a pane in which both objects are visualised if any else false.
;;; REF: GO_PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars panes1 = go_visible_in(obj1);
	for pane in go_visible_in(obj2) do
		if pane.isgo_pane then
			returnif(member(pane, panes1))
		endif;
	endfor;
	false or go_default_pane -> pane;
enddefine;


;;;----------------------------------------------------------------
;;; SELECTION CRITERIA

define :method go_close_to_line( x, y, poly :go_polygon );
lvars x, y, poly;
;;; REF: go_close_to_line( X, Y, POLYGON ) -> FALSE_OR_POLYGON;
;;; REF: Returns the polygon if the given position in world coordinates is
;;; REF: close to the a line of the polygon. Otherwise it returns false.
;;; REF: X, Y are integer world coordinates of the test position.
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pointlist = go_local_coords(poly), flag = false, xpoint, ypoint;
	go_transxyin( x, y, poly ) -> (x,y);
	fast_destpair( pointlist ) -> pointlist /* -> x1 */;
	fast_destpair( pointlist ) -> pointlist /* -> y1 */;
	until pointlist = [] do
		fast_destpair( pointlist ) -> (xpoint, pointlist) ;
		fast_destpair( pointlist ) -> (ypoint, pointlist) ;
		xpoint /* -> x2 */;
		ypoint /* -> y2 */;
		quitif( close_to_a_line( /* x1, y1, x2, y2, */ x, y, 4 ) ->> flag );
		xpoint /* -> x1 */;
		ypoint /* -> y1 */;
	enduntil;
	if ( flag ) then poly; else ->; ->; false; endif;
enddefine;


define :method go_contains( x, y, poly :go_polygon );
lvars x, y, poly;
;;; REF: go_contains( X, Y, POLYGON ) -> POLYGON_OR_FALSE;
;;; REF: Checks whether the world coordinates of a test position fall inside
;;; REF: the closed polygon or on the lines for an open polygon. This method
;;; REF: uses the XRegion facilities (see REF * XPointInRegion) when closed.
;;; REF: X, Y are integer world coordinates of the test position.
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars points_vec, reg, pane = go_common_pane(poly, poly);
	if ( poly.go_closed ) then
		if ( pane ) then
			go_device_coords_xregion( pane, poly ) -> reg;
			go_transxyout( x, y, pane ) -> (x, y);
			if ( XPointInRegion( reg, x, y ) ) then poly else false endif;
		else
			false;
		endif;
	else
		go_close_to_line( x, y, poly );
	endif;
enddefine;


;;;------------------------------------------------------
;;; OVERLAPPING

define :method go_overlaps( obj :go_located, poly :go_polygon );
lvars obj, poly;
;;; REF: go_overlaps( LOCATED, POLYGON ) -> BOOLEAN;
;;; REF: This checks whether the bounding box of the located object overlaps
;;; REF: the polygon or not. It returns the boolean true if it overlaps, else
;;; REF: it is false. This method uses the XRegion facilities (see also
;;; REF: REF * XRectInRegion).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars points_vec, reg, x, y, w, h, pane = go_common_pane(obj, poly);
	if ( pane ) /* and go_overlaps_bounding_boxes(obj, poly) */ then
		go_device_coords_xregion( pane, poly ) -> reg;
		go_safe_region( pane, obj ) -> (x, y, w, h);
		XRectInRegion( reg, x, y, w, h );
	else
		false;
	endif;
enddefine;

define :method go_overlaps( poly :go_polygon, obj :go_located );
lvars obj, poly;
;;; REF: go_overlaps( POLYGON, LOCATED ) -> BOOLEAN;
;;; REF: This checks whether the bounding box of the located object overlaps
;;; REF: the polygon or not. It returns the boolean true if it overlaps, else
;;; REF: it is false. This method uses the XRegion facilities (see also
;;; REF: REF * XRectInRegion).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
	go_overlaps( obj, poly );
enddefine;

lvars intersect_region = false;
define lconstant get_intersect_region();
	unless (intersect_region) then
		XCreateRegion() -> intersect_region;
	endunless;
	intersect_region;
enddefine;

define lconstant destroy_intersect_region();
	if (intersect_region) then
		XDestroyRegion( intersect_region );
	endif;
	false -> intersect_region;
enddefine;

define :method go_overlaps( poly1 :go_polygon, poly2 :go_polygon);
lvars poly1, poly2;
;;; REF: go_overlaps( POLYGON1, POLYGON2 ) -> BOOLEAN;
;;; REF: This checks whether the two polygons overlap or not. It returns the
;;; REF: boolean true if it overlaps, else it is false. This method uses the
;;; REF: XRegion facilities (see also REF * XIntersectRegion & REF * XClipBox).
;;; REF: POLYGON1 and POLYGON2 are two go_polygons (REF * GO_POLYGON).
lvars reg1, reg2, intersect, w, h, pane = go_common_pane(poly1, poly2);
	if ( pane ) /* and go_overlaps_bounding_boxes(poly1, poly2) */ then
		go_device_coords_xregion( pane, poly1 ) -> reg1;
		go_device_coords_xregion( pane, poly2 ) -> reg2;
		get_intersect_region() -> intersect;
		XIntersectRegion( reg1, reg2, intersect );
		XClipBox( intersect) -> (, , w, h);
		(w > 0) and (h > 0);
	else
		false;
	endif;
enddefine;


;;;------------------------------------------------------
;;; EDITING FACILITY

;;; IRREGULAR POLYGONS

define :method go_reposition_point( poly :go_polygon, frozen_index ) -> (x, y);
lvars poly, frozen_index;
;;; REF: go_reposition_point( POLYGON, INTEGER ) -> (X, Y);
	go_local_coords(poly)(frozen_index*2 - 1) -> x;
	go_local_coords(poly)(frozen_index*2)     -> y;
	go_transxyout( x, y, poly ) -> (x, y);
enddefine;

define :method updaterof go_reposition_point( x, y, poly :go_polygon, frozen_index );
lvars x, y, poly, frozen_index;
;;; REF: (X, Y) -> go_reposition_point( POLYGON, INTEGER );
;;; REF: The world coordinates of the corner point described by the index
;;; REF: argument. The index starts from 1 until the number of points found
;;; REF: in the go_npoints method (see REF * POLYGON/go_npoints). This is
;;; REF: used when the polygon is editable (REF * POLYGON/go_make_editable).
;;; REF: Updates only make sense for irregular polygons.
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pane;
	go_transxyin( x, y, poly ) -> (x, y);
	x -> stored_go_local_coords(poly)(frozen_index*2 - 1);
	y -> stored_go_local_coords(poly)(frozen_index*2);
	go_update( poly );
	go_set_bounding_box( poly );
enddefine;


;;; REGULAR POLYGONS

define :method go_attach_refpnt( poly :go_polygon ) -> (x, y);
lvars poly, x, y;
;;; REF: go_attach_refpnt( POLYGON ) -> (X, Y);
	go_local_coords(poly)(1) -> x ;
	go_local_coords(poly)(2) -> y ;
	go_transxyout( x, y, poly ) -> (x, y);
enddefine;


define :method updaterof go_attach_refpnt( x, y, poly :go_polygon ) ;
lvars poly, x, y;
;;; REF: (X, Y) -> go_attach_refpnt( POLYGON );
;;; REF: The world coordinates of the reference point of the polygon, which
;;; REF: is the first point in the points list (see also go_angle). This is
;;; REF: used when the polygon is editable (REF * POLYGON/go_make_editable).
;;; REF: Updates only make sense for regular polygons.
;;; REF: If the point is updated then the centre remains and the radius and
;;; REF: angle of the reference point change to go through the new location.
;;; REF: X and Y are the world coordinates of the reference point.
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars xc = poly.go_xcentre, yc = poly.go_ycentre;
	go_batch_mode_on( poly );
		arctan2(x-xc, y-yc) -> poly.go_angle;     ;;; WRONG IF CENTRE \= ORIGIN
		go_transxyin( x, y, poly ) -> (x, y);     ;;; LOCAL COORDS
		round( sqrt( x**2 + y**2 ) ) -> poly.go_radius;
	go_batch_mode_off( poly );
enddefine;

;;; GENERAL EDIT RULES

define :method go_make_uneditable( poly :go_polygon );
lvars poly;
;;; REF: go_make_uneditable( POLYGON );
;;; REF: This removes all editors from the corner points of the polygon (see
;;; REF: also REF * GO_EDIT_POINT).
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pnt;
	if not(poly.go_editors == []) then
		go_batch_mode_on( poly );
		for pnt in poly.go_editors do
			go_remove_from( pnt, go_default_pane );
			false -> go_edited_object( pnt );
		endfor;
		[] -> poly.go_editors;
		go_batch_mode_off( poly );
	endif;
enddefine;


define :method go_make_editable( poly :go_polygon );
lvars poly;
;;; REF: go_make_editable( POLYGON );
;;; REF: This adds editors to the corner points of the polygon (see also
;;; REF: REF * GO_EDIT_POINT). If the polygon is rounded then the original
;;; REF: corners will be used as edit points and the rounding will be
;;; REF: recalculated everytime.
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars centre_pnt, pnt1, i, pnti;
	;;; remove any pending editors:
	go_make_uneditable( poly );
	go_batch_mode_on( poly );

	;;; create new editors at least one in centre and one in a vertex
	[% newgo_edit_point() ->> centre_pnt;         ;;; puts the editor on stack
	   poly -> go_edited_object( centre_pnt );
	   if ( poly.go_regular ) then
		   newgo_edit_point() ->> pnt1;           ;;; puts the editor on stack
		   poly -> go_edited_object( pnt1 );
		   go_attach_refpnt -> go_move_absolute( pnt1 );
	   else
		   for i from 1 to poly.go_npoints do
			   newgo_edit_point() ->> pnti;       ;;; puts the editor on stack
			   poly -> go_edited_object( pnti );
			   go_reposition_point(% i %) -> go_move_absolute( pnti );
		   endfor;
	   endif;
	%] -> poly.go_editors;
	for pnti in poly.go_editors do
		go_add_to( pnti, go_default_pane );
	endfor;
	go_batch_mode_off( poly );
enddefine;

define :method go_add_point( x, y, poly :go_polygon );
lvars x, y, poly;
;;; REF: go_add_point( X, Y, POLYGON );
;;; REF: Used to add an extra point. The X and Y coordinates are used to find
;;; REF: where to add the point if the shape is irregular. If the polygon is
;;; REF: regular, then the number of points is increased.
;;; REF: EVENT_DATA is the vector from REF * GO_XACTON/go_expand_event_data
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pointlist = poly.go_local_coords, add_flag = true /* still to add */;
lvars x y, x1, y1, x2, y2;
	if poly.go_regular then
		poly.go_npoints + 1 -> poly.go_npoints;
	else
		go_transxyin( x, y, poly ) -> (x, y);
		[%
			fast_destpair( pointlist ) -> pointlist ->> x1; ;;; stacked x
			fast_destpair( pointlist ) -> pointlist ->> y1; ;;; stacked y
			until pointlist == [] do
				fast_destpair( pointlist ) -> pointlist -> x2;
				fast_destpair( pointlist ) -> pointlist -> y2;
				if add_flag and close_to_a_line( x1, y1, x2, y2, x, y, 8 ) then
					x; y;                                    ;;; stack new x/y
					false -> add_flag;                       ;;; have added now
				endif;
				x2 ->> x1;                                   ;;; stacked x
				y2 ->> y1;                                   ;;; stacked y
			enduntil;
			if add_flag then                                 ;;; not yet added
				x; y;                                        ;;; stack new x/y
			endif;
		 %] -> go_local_coords( poly );
	endif;
enddefine;


define :method go_remove_point( x, y, poly :go_polygon );
lvars x, y, poly;
;;; REF: go_remove_point( X, Y, POLYGON );
;;; REF: Used to remove a point. The X and Y coordinates are used to find
;;; REF: where to remove the point if the shape is irregular. If the polygon
;;; REF: is regular, then the number of points is reduced.
;;; REF: X and Y are screen coordinates of the bottom left corner of the region.
;;; REF: POLYGON is an instance of the go_polygon class (REF * GO_POLYGON).
lvars pointlist = poly.go_local_coords, del_flag = true /* still to delete */;
lvars x, y, n = poly.go_npoints, x1, y1, newpointslist;
	if poly.go_filled then
		returnif( n <= 3 );        ;;; 3 sides is the min. for a closed shape
	else
		returnif( n <= 2 );        ;;; 2 points is the min. for a line
	endif;
	if poly.go_regular then
		poly.go_npoints - 1 -> poly.go_npoints;
	else
		go_transxyin( x, y, poly ) -> (x, y);
		[%
			until pointlist == [] do
				fast_destpair( pointlist ) -> pointlist -> x1;
				fast_destpair( pointlist ) -> pointlist -> y1;
				if del_flag and ((x1-x)**2 + (y1-y)**2 < 64) then
					false -> del_flag;                       ;;; have deleted
				else
					x1; y1;                                  ;;; stacked x/y
				endif;
			enduntil;
		 %] -> newpointslist;
		if del_flag then                                     ;;; not yet deleted
			n - 1 -> poly.go_npoints;          ;;; = del random
		else
			newpointslist -> go_local_coords( poly );
		endif;
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_polygon = true;

/* --- Revision History --------------------------------------------
 * BR 23/08/93
 *     Removed go_[x/y]radius(): no longer used...
 * BR 09/08/93
 *     Repaired bug in go_local_coords() where go_make_editable() was called to
 *     often resulting in a bug when dragging an editor.
 * BR 29/07/93
 *     Added go_draw_bounding_circle() method which might not be useful after
 *     all.
 * BR 15/07/93
 *     Removed all references to stored_go_local_coords() and replaced by safer
 *     go_local_coords() except for the editors.
 * BR 05/07/93
 *     Added a slot go_editors to avoid the overuse of go_live_object to
 *     also keep the editors within go_make_editable().
 *     Code cleanup (more assignments in the declarations).
 * BR 03/06/93
 *     Removed all references to stored_go_[x/y]loc (go_set_bounding_box).
 * BR 25/05/93
 *     Removed the go_attached method and integrated it in go_attach_refpnt().
 * BR 17/05/93
 *     Removed go_bounding_[width/height]; replaced by more general one in
 *     go_located.p.
 *     Replaced go_attach_centre by go_position_of_centre from go_located.p.
 *     Allowed reals for stored_go_local_coords and corrected bug in go_update_shape
 * BR 07/05/93
 *     Added the method go_update_shape() to allow irregular polygons to
 *     be scaled and rotated.
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 28/04/93
 *     Added comments and made go_rounding() et al. safer to use.
 * BR 22/04/93
 *     Abandoned support for both -CoordModePrevious- & -CoordModeOrigin- and
 *     therefore replaced the GO variable ScreenCoordMode by CoordModeOrigin.
 *     Abandoned method go_update_coordlist()
 * BR 08/04/93
 *     Removed direct call to XpwDrawLine[s]() by go_draw_line[s]();
 *     Removed direct call to XpwDrawSegments() by go_draw_line_segments();
 *     Removed direct call to XpwDrawArcs() by go_draw_arcs();
 *     Removed direct call to XpwFillArcs() by go_draw_filled_arcs();
 *     Removed direct call to XpwDrawRectangle() by go_draw_rectangle();
 *     Removed direct call to XpwClearArea() by go_clear_rectangle()
 *     Removed direct call to XpwFillPolygon() by go_draw_filled_polygon()
 * IR 02/04/93
 *     Avoided go_default_pane with go_common_pane() method.
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * AS 15/02/93
 *     Changed the XpwFillPolygon shape into "Complex" in not go_regular...
 * IR 28/01/93
 *     Updated cached_go_coord_list() in the updaterof go_local_coords().
 * BR 23/12/92
 *     Changed screen_coords_region() for detection of open polylines.
 *     KNOWN BUG: if the newly created go_local_coords which are shifted a little
 *     away from the original go_local_coords, fall on the original line, then the
 *     region still will have thickness zero on that line => go_overlaps then
 *     does not work.
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 24/11/92
 *     Separated out the go_draw() method (see go_screen_object.p and go_xdraw.p)
 *     and added in the corresponding currently X-specific go_fgdraw() & go_bgdraw()
 * BR 11/11/92
 *     Changed for OC 5.01 (used initialise)...
 *     Moved go_centre_to methods from go_polygon.p to go_screen_object.p
 * BR 03/11/92
 *     Changed go_contains to return the object rather than true or false...
 * BR 16/10/92
 *     Added argument to go_draw() to indicate pane (removed rc_window)...
 *     Added argument to go_safe_region() (IDEM)
 * BR 05/10/92
 *     Added go_centre_to() methods.
 *     Removed isa go_located because of new definition of go_screen_object.
 *     Added extra argument to make_(in)visible to indicate pane.
 *     Moved XRegions into separate file go_xregions.p
 * BR 04/09/92
 *     removed the go_drag_and_drop method (see new version of go_located class in
 *     go_located.p). Moved go_safe_region to go_located class.
 * BR 03/09/92
 *     added color caching (see go_get_colour) and optimised the usage of new
 *     regions (only one region "intersect") - might need to check for
 *     reentrance...
 * BR 19/08/92
 *     changed to both absolute and relative coordinates (this might be
 *     restricted to linear rather than curved coordinate systems; see A.S.).
 *     still benefits from go_motionhint == true
 */
;;; eof
