/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_xdrag.p
 > Purpose:         GO file
 > Author:          Jonathan Meyer, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */

compile_mode :pop11 +strict;

;;; section $-go_drag => XpwPixmap;

vars XpwPixmap;

include XpwPixmap;

vars go_debug_event_list = [];

;;; INTRO: -- Purpose ------------------------------------------------------------
;;; INTRO:
;;; INTRO: In many graphical applications it is usefull to be able to grab hold
;;; INTRO: of a graphical "object" and to move it over the screen to a new
;;; INTRO: location.
;;; INTRO:
;;; INTRO: A "go_drag" operation like this can be seen as the following
;;; INTRO: sequence:
;;; INTRO:     - Erase the graphical "object" which is picked up.
;;; INTRO:     - Draw it at new position
;;; INTRO:     - Keep erasing at new position then redrawing at the next ...
;;; INTRO:     - Drop it at the final destination.
;;; INTRO:
;;; INTRO: If this is implemented with callback procedures in X-Windows then it
;;; INTRO: soon is obvious that that this can be rather slow, and result in the
;;; INTRO: "object" lagging behind having to catch up with the mouse.
;;; INTRO:
;;; INTRO: A further complication is that the object which is dragged does
;;; INTRO: not always remain the same during the go_drag operation (see
;;; INTRO: further).
;;; INTRO:
;;; INTRO: To avoid this and to integrate it better with the RC_GRAPHIC library
;;; INTRO: the LIB * RC_DRAG was written.
;;; INTRO:
;;; INTRO:
;;; INTRO: PROCEDURES DEFINED IN THIS FILE:
;;; INTRO:
vars
	 XpwRedrawArea,      ;;; REF: procedure to redraw saved off-screen pixmap
	 XpwQueryPointer,    ;;; REF: procedure to find relative mouse-coordinates
	 XpwUseMotionHints,  ;;; REF: procedure to switch the X-Server's MotionHint
	 XpwFillBackground,  ;;; REF: procedure to fill&colour a widget's background

	 go_safe_size,       ;;; REF: calculates size of object incl. go_linewidth

	 go_make_image,      ;;; REF: makes image of object in separate new widget
	 go_draw_image,      ;;; REF: procedure to go_draw such an image
	 go_erase_image,     ;;; REF: procedure to erase such an image

	 go_startdrag,       ;;; REF: fake callback to initialize the drag operation
	 go_dodrag,          ;;; REF: real callback to do the drag operation
	 go_dropdrag,        ;;; REF: fake callback to finish the drag operation
	 go_dodrop,          ;;; REF: real callback to finish the drag operation

	 go_drag,            ;;; REF: TOP LEVEL DRAG OPERATION
	 go_fixed_drag,      ;;; REF: CONVENIENCE FUNCTION (fixed image go_drag)
	 go_drawn_drag,      ;;; REF: CONVENIENCE FUNCTION (transparent go_drag)

	 go_motionhint;      ;;; REF: TOP LEVEL MOTIONHINT SWITCH

lvars
	 go_dragged_object;  ;;; REF: Local var: false or object which is dragged

exload_batch;
	;;; load X Toolkit libraries
	uses xt_callback;
	uses xt_display;
	uses xpt_coretypes;
	uses xt_resource;
	uses xt_event;
	uses XtN;

	;;; The XpwPixmap widget is used
	XptWidgetSet("Poplog")("PixmapWidget") -> XpwPixmap;

	;;; We will be playing with the following Xlib procedures:
	XptLoadProcedures 'xpw_go_drag'
			   lvars XCopyArea XQueryPointer XCheckMaskEvent;

	;;; Load GO Graphical Objects Library
	;;; uses go_pane;
endexload_batch;

;;; INTRO:
;;; INTRO: This library provides general go_drag routines to RC_GRAPHIC windows,
;;; INTRO: based on the fast CopyArea facilities. The structure used to pass
;;; INTRO: data about a dragged object is defined in "dragContext":
;;; INTRO:
;;; INTRO: Two procedures are passed in by the user to allow the drag operation
;;; INTRO: of a graphical "object" over the window. This allows to:
;;; INTRO:         1. cleanly update the window when the "object" is picked up
;;; INTRO:            (see eraseProc below).
;;; INTRO:         2. go_draw the "object" when it is dragged.
;;; INTRO: The first will allow to "erase" the object currently on the screen
;;; INTRO: before starting to go_drag it. The second procedures draws the
;;; INTRO: "object" which will be dragged. The representation of the object can
;;; INTRO: change during the go_drag if the object is redrawn differently at
;;; INTRO: each procedure call.
;;; INTRO:
;;; INTRO: If the image never changes during the drag operation then the
;;; INTRO: fixedImage flag is set to true. This will make the dragging faster.

;;; INTRO: However the dragged go_area will then always be considered as a
;;; INTRO: non-transparent go_rectangle.
;;; INTRO:
;;; INTRO: If at the end of the dragging operation the drop is not allowed, then
;;; INTRO: the object will be drawn at the startCoords rather than the
;;; INTRO: lastCoords.
;;; INTRO:
;;; INTRO: Most other fields are self explanatory (see also comments below).

;;; REF: -- dragContext data structure -----------------------------------------
;;; REF:
;;; REF: This library provides general drag routines to the Graphical Objects
;;; REF: library (GO), based on the fast XCopyArea facilities (see also
;;; REF: HELP * XCopyArea). The structure used to pass data about a dragged
;;; REF: object is defined in "dragContext":
;;; REF:
;;; REF: defclass dragContext {
;;; REF:   inDrag      :XptBoolean, ;;; true if we are in a go_drag operation
;;; REF:   lastCoords,              ;;; records where the object is drawn
;;; REF:   startCoords,             ;;; X,Y coord where object started from
;;; REF:   startXOffset:int,        ;;; Original mouse X offset with startCoords
;;; REF:   startYOffset:int,        ;;; Original mouse Y offset with startCoords
;;; REF:   dragWidth   :int,        ;;; records width of dragged object
;;; REF:   dragHeight  :int,        ;;; records height of dragged object
;;; REF:   eraseProc   :full,       ;;; procedure that clears object picked up
;;; REF:   drawProc    :full,       ;;; procedure that draws dragged object
;;; REF:   fixedImage  :XptBoolean, ;;; false if object changes during go_drag
;;; REF:   doubleBuffer:XptBoolean, ;;; true if using double buffer imaging
;;; REF:   imageDrag   :full,       ;;; widget with fixed image
;;; REF:   allowedDrop :full,       ;;; Boolean = true if the drop is allowed
;;; REF:   userData    :full,       ;;; whatever the user wants to add
;;; REF: };
;;; REF:
;;; REF: <inDrag>:
;;; REF: This is a Flag which is set to true as long as the go_drag operation
;;; REF: is on. This allows the sophisticated user to switch the go_drag off if
;;; REF: necessary.
;;; REF:
;;; REF: <lastCoords>, <startCoords>:
;;; REF: The coordinates where the object has last been drawn are saved in
;;; REF: lastCoords. This field gets automatically updated by go_drag whenever
;;; REF: the object is redrawn at a new position. The original coordinates are
;;; REF: saved in case the object cannot be dropped (see allowedDrop), and allow
;;; REF: user-access in cases like rubber-banding. Both coordinates represent
;;; REF: the CENTRE of the object (see start*Offset).
;;; REF:
;;; REF: <startXOffset>, <startYOffset>:
;;; REF: When a graphical object is picked up, it is normally not picked up
;;; REF: in its centre. Therefore when we drag it around it is not desirable
;;; REF: desirable to have the mouse jump to that location or the object jump
;;; REF: in the opposite directly.
;;; REF:                                    PICKING UP:\\     //
;;; REF:                                        +-------\\---//-------+
;;; REF:         +---------------------+        |        \\ //        |
;;; REF:         |                     |        |  ?      \ /         |
;;; REF:         |  #                  |        |          #          |
;;; REF:         |   \                 |        |        // \\        |
;;; REF:         |                     |        |       //   \\       |
;;; REF:         |                     |        +------//object\------+
;;; REF:         +--------object-------+              //       \\
;;; REF:
;;; REF: By saving the original offset between the centre's position and the
;;; REF: mouse the object can always be redrawn at the same relative position.
;;; REF: This  provides a smoother movement.
;;; REF:         +---------------------+
;;; REF:         |                     |    AFTER MOVING:
;;; REF:         |   #.......          |        +---------------------+
;;; REF:         |    \     :          |        |                     |
;;; REF:         |                     |        |   #.......          |
;;; REF:         |                     |        |    \     :          |
;;; REF:         +--------object-------+        |                     |
;;; REF:                                        |                     |
;;; REF:                                        +--------object ------+
;;; REF:
;;; REF: <dragWidth>, <dragHeight>:
;;; REF: These fields indicate the go_bounding_width and go_bounding_height of
;;; REF: the bounding rectangle around the object, and are used by go_drag to
;;; REF: refresh the corresponding area during the go_drag operation. The
;;; REF: rectangle is referred to from the centre position.
;;; REF: The user is responsible to update these fields correctly in case of a
;;; REF: changing image (see imageDrag).
;;; REF:
;;; REF: <eraseProc>, <drawProc>:
;;; REF: Two procedures are passed in by the user to allow the drag operation
;;; REF: of a graphical "object" over the window. This allows to:
;;; REF:         1. cleanly update the window when the "object" is picked up
;;; REF:            (see eraseProc below).
;;; REF:         2. go_draw the "object" when it is dragged.
;;; REF: The first will allow to "erase" the object currently on the screen
;;; REF: before starting to go_drag it. The second procedures draws the "object"
;;; REF: which will be dragged. The representation of the object can change
;;; REF: during the go_drag if the object is redrawn differently at each
;;; REF: procedure call.
;;; REF:
;;; REF: <fixedImage>:
;;; REF: If the image never changes during the go_drag operation then the
;;; REF: fixedImage flag is set to true. This will make the dragging faster.
;;; REF: Note that the image will always be drawn on top of all others!
;;; REF: However the dragged area will then always be considered as a non-
;;; REF: transparent rectangular image which is stored in the imageDrag field.
;;; REF:
;;; REF: <doubleBuffer>:
;;; REF: If the pane uses a double buffer then the object is first drawn in
;;; REF: a separate image, and then copied just like a fixed image. This
;;; REF: image is stored in the imageDrag field. The result is in a smoother
;;; REF: dragging (no flashing on the screen). Because the object hierarchy is
;;; REF: is followed, a small time-lag can occur which looks like the object
;;; REF: cannot quite follow the mouse movements (REF * GO_PANE/go_double_buffer).
;;; REF:
;;; REF: <imageDrag>:
;;; REF: This field is used to stire the image from fixedImage and doubleBuffer.
;;; REF:
;;; REF: <allowedDrop>:
;;; REF: If at the end of the dragging operation the drop is not allowed, then
;;; REF: the object will be drawn at the startCoords rather than the lastCoords.
;;; REF: Although this is in principle a boolean go_value, this can be combined
;;; REF: with an active variable which allows the user to check whether the drop
;;; REF: is allowed.
;;; REF:
;;; REF: <userData>:
;;; REF: This field is at the disposal of the user. It can be used to carry data
;;; REF: around from one call to another.
;;; REF:

defclass dragContext {
	 inDrag      :XptBoolean,  ;;; true if we are in a drag operation
	 lastCoords,               ;;; records where the object is drawn
	 startCoords,              ;;; X,Y coord where object started from
	 startXOffset,             ;;; Original mouse X offset with startCoords
	 startYOffset,             ;;; Original mouse Y offset with startCoords
	 dragWidth   :int,         ;;; records width of dragged object
	 dragHeight  :int,         ;;; records height of dragged object
	 eraseProc   :full,        ;;; procedure that clears object picked up
	 drawProc    :full,        ;;; procedure that draws dragged object
	 fixedImage  :XptBoolean,  ;;; false if object changes during go_drag
	 doubleBuffer:XptBoolean,  ;;; true if using double buffer imaging
	 imageDrag   :full,        ;;; widget with fixed image if fixedImage
	 allowedDrop :XptBoolean,  ;;; true if the drop is allowed
	 userData    :full,        ;;; whatever the user wants to add
};


/* LOW LEVEL Xpw ROUTINES */

/*
-- (Utility Procedure 1: XpwRedrawArea) -------------------------------

;;; REF: XpwRedrawArea( WINDOW, X, Y, W, H );
;;; REF: The following procedure is used to 'redraw' a graphics widget from
;;; REF: its pixmap backing store. You don't need to understand the procedure
;;; REF: definition, only its use. (this function will become available in
;;; REF: future versions of Poplog).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: X and Y are screen coordinates where area starts.
;;; REF: W and H are the width and height in screen coordinates of the area.

*/

define global XpwRedrawArea(graphicw, x, y, w, h);
	lvars graphicw, x, y, w, h, win;
	;;; redraws a region of a graphics widget, even if pixmapStatus is
	;;; PixmapOff.
	returnif( not(XtWindow(graphicw)->>win) or
			  fast_XptValue(graphicw, XtN pixmapStatus) == PixmapHasNone );

	exacc [fast] (10) raw_XCopyArea(fast_XtDisplay(graphicw),
						 fast_XptValue(graphicw, XtN pixmap), win,
						 fast_XptValue(graphicw, XtN privateGC),
						 XptCheckInt(x), XptCheckInt(y),
						 XptCheckUnsignedInt(w),
						 XptCheckUnsignedInt(h),
						 x,y);
enddefine;


define global XpwCopyAreaTo(graphicw, widget, x, y, w, h);
	lvars graphicw, widget, x, y, w, h, win;
	;;; redraws a region of a graphics widget, even if pixmapStatus is
	;;; PixmapOff.
	returnif( not(XptValue(widget, XtN pixmap)->>win) or
			  fast_XptValue(graphicw, XtN pixmapStatus) == PixmapHasNone );

	exacc [fast] (10) raw_XCopyArea(fast_XtDisplay(graphicw),
						 fast_XptValue(graphicw, XtN pixmap), win,
						 fast_XptValue(graphicw, XtN privateGC),
						 XptCheckInt(x), XptCheckInt(y),
						 XptCheckUnsignedInt(w),
						 XptCheckUnsignedInt(h),
						 x,y);
enddefine;

/*
-- (Utility Procedure 2: XpwQueryPointer) -----------------------------

;;; REF: XpwQueryPointer(WINDOW, PANE) -> (X, Y);
;;; REF: This procedure simply takes a widget and its pane and returns the
;;; REF: current world coordinates for the mouse relative to the widget and
;;; REF: pane.
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: X and Y are world coordinates the mouse currently points in the pane.

*/

define global XpwQueryPointer(window, pane) /* -> (x,y) */;
	lvars window, pane, xworld, yworld;
	XptCheckWidget(window)->;
	lconstant unused = writeable initintvec(1), x = copy(unused), y = copy(x);
	exacc [fast] (9) raw_XQueryPointer(
							   fast_XtDisplay(window), fast_XtWindow(window),
							   unused, unused, unused, unused, x, y, unused);
	go_transxyin(fast_subscrintvec(1, x), fast_subscrintvec(1, y), pane)
	-> (xworld, yworld);
	round(xworld); round(yworld);
enddefine;

/*
-- (Utility Procedure 3: XpwUseMotionHints) ---------------------------

;;; REF: XpwUseMotionHints( WINDOW ) -> BOOLEAN;
;;; REF: BOOLEAN -> XpwUseMotionHints( WINDOW )
;;; REF: The procedure allows you to access or set the use of motion hints
;;; REF: for a particular window.

;;; REF: The user can ask the X Server to use MotionHints. When a window
;;; REF: requests motion hints, the X server will only send one mouse motion
;;; REF: event if the mouse changes position, the key or button state changes,
;;; REF: the mouse leaves the window. Also, every time the application calls
;;; REF: XQueryPointer or XGetMotionEvents, a further motion event is generated.
;;; REF: You can then write a motion event handler which calls XQueryPointer to
;;; REF: find the current mouse location, and as a side effect generates another
;;; REF: motion event - these motion events will therefore only be generated as
;;; REF: quickly as your application can cope with them.

;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).

*/

include xpt_xevent;

define global XpwUseMotionHints(window);
	lvars window;
	XtBuildEventMask(window) &&/=_0 PointerMotionHintMask
enddefine;

define updaterof XpwUseMotionHints(go_value, widget);
	lvars go_value, widget, mask;
	lconstant dummy_event_p = erasenum(%3%);
	if go_value and not(XpwUseMotionHints(widget)) then
		XtAddEventHandler(widget, PointerMotionHintMask, false,
						  dummy_event_p, false);
	elseif not(go_value) and XpwUseMotionHints(widget) then
		XtRemoveEventHandler(widget, PointerMotionHintMask, false,
							 dummy_event_p, false);
	endif;
enddefine;



/*
-- (Utility Procedure 4: XpwFillBackground) ---------------------------

;;; REF: XpwFillBackground( WINDOW, COLOUR, X, Y, W, H );
;;; REF: Fills a sized background go_area of a widget with a background colour.
;;; REF: This procedure makes the setting of background resources independent
;;; REF: of the type of widget (Graphics widget or pixmap or whatever).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: COLOUR is an index in the colour table of the pane (REF * GO_PANE).
;;; REF: X and Y are screen coordinates where area starts.
;;; REF: W and H are the width and height in screen coordinates of the area.

 */

define global XpwFillBackground(widget, col, x, y, w, h);
	lvars widget, col, x, y, w, h;

	;;; need to use a procedure because, when the save stage of the
	;;; dlocal is run, the widget variable has not been initialised.

	procedure;
		;;; remember the current foreground colour
		dlocal %XptValue(widget, XtN foreground)%;

		;;; set the foreground colour
		XtVaSetValues(widget, XptVaArgList([{foreground ^col}]));

		;;; fill the window go_area with the specified colour
		XpwFillRectangle(widget, x, y, w, h);
	endprocedure();

enddefine;


/*
-- (Utility Procedure 5: go_safe_size) ---------------------------

;;; REF: go_safe_size( DRAG_CONTEXT ) -> (X, Y, W, H);
;;; REF: Calculates the position and size of the region around an object, taken
;;; REF: into acount the possible line width in which it is likely to be drawn.
;;; REF: (see go_linewidth in TEACH * GO).
;;; REF: The coordinate system is transformed to deliver a left top hand
;;; REF: corner x and y position with a width to the go_right and a height down.

;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: X and Y are screen coordinates where area starts.
;;; REF: W and H are the width and height in screen coordinates of the area.

 */
define go_safe_size(aContext) -> (x, y, safe_width, safe_height);
lvars aContext, xc, yc, x, y, safe_width, safe_height, extra_space;
lvars pane = go_default_pane, obj_lw = (go_dragged_object.go_linewidth or 1),
	  pane_lw = pane.go_linewidth;

	;;; WARNING: Transform to user-coordinates
	;;;          Might need adaptation for thicker lines
	;;; START OF AS CENTRE
	explode(aContext.lastCoords) ;
	go_transxyout(pane) -> (xc, yc);

	;;; Make sure CapProjecting lines under an angle of 45degrees are
	;;; covered. Formula: extra_space = lw/2 * sqrt(2)
	max( 2, round(max(obj_lw, pane_lw) * 1.5) ) -> extra_space;

	;;; Approx. to take into account the JoinStyle == JoinMiter in polygons !!!
	;;; The factor 5.20 was calculated for big linewidths
	;;; round(extra_space * 5.20) -> extra_space;
	if (extra_space > 10) then
		round(extra_space * 2) -> extra_space;
	endif;

	;;; WARNING: Might need scaling for width and height
	round(aContext.dragWidth  * pane.go_xscale)-> safe_width;
	round(aContext.dragHeight * pane.go_yscale)-> safe_height;

	;;; WARNING: Normalize go_rectangle: Make sure the corner is top left
	;;;          Might need adaptation for thicker lines
	xc - abs(safe_width div 2) - extra_space -> x ;
	abs(safe_width) + 2*extra_space -> safe_width;

	yc - abs(safe_height div 2) - extra_space -> y;
	abs(safe_height) + 2*extra_space -> safe_height;
enddefine;


/*
-- (Utility Procedures 6: go_make/go_draw/erase_image) ----------------

;;; REF: The following procedures create and go_draw or erase a fixed image from
;;; REF: the screen. The image is created by calling a drawing procedure which
;;; REF: will go_draw in the image's pixmap in the same way as it normally would
;;; REF: on the window screen.

The go_draw and erase procedures will copy or xor the image from the
pixmap on the screen, with the positions indicated in the context.

 */
lvars go_buffer = false; ;;; Internal buffer if needed for fixed image or clear
lvars go_buffer_x = 0;   ;;; Kept in case a fixed image started outside a pane
lvars go_buffer_y = 0;   ;;; then a new start position is calculated!

;;; REF: go_make_image( PROCEDURE, DRAG_CONTEXT, WINDOW ) -> PIXMAP;
;;; REF: Draws an object using the given procedure and drag context in
;;; REF: a new pixmap given the characteristics of the object's window.
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: PIXMAP is a XpwPixmap (REF * XpwPixmap).

define go_make_image(drawingProc, buttonContext, widget) -> buffer;
	lvars drawingProc, buttonContext, widget, buffer = false;
	lvars x, y, safe_width, safe_height, old_widget, old_objects, the_bg;
	lvars pane = go_default_pane,
		  widget_width = XptValue( widget, XtN width, TYPESPEC(:XptShort)),  ;;; pane.go_window_width,
		  widget_height = XptValue( widget, XtN height, TYPESPEC(:XptShort)), ;;; pane.go_window_height,
		  fixed_image_flag = buttonContext.fixedImage,
		  double_buffer_flag = buttonContext.doubleBuffer;

	go_safe_size(buttonContext) -> (x, y, safe_width, safe_height);

	unless (pane.the_go_double_buffer ->> old_widget)
	then unless (go_buffer ->> old_widget)
		 then false -> old_widget;
		 endunless;
	endunless;

	if  (old_widget) and (XptIsLiveType(old_widget, "Widget"))
	then
		if   (widget_width  <= XptValue( old_widget, XtN width  ))
		and  (widget_height <= XptValue( old_widget, XtN height ))
		then old_widget -> buffer;
		else XtDestroyWidget( old_widget );
		endif;
	endif;
	unless (buffer)
	then
		XtCreateWidget('GO buffer image', XpwPixmap, widget, XptArgList([
				   {^XtN width        ^widget_width }
				   {^XtN height       ^widget_height} ])) -> buffer;
		if   (double_buffer_flag)
		then buffer -> pane.the_go_double_buffer;
		else buffer -> go_buffer;
		endif;
	endunless;

	;;; SHOULD NOT BE REALIZED: BUG FIX 26/05/93
	;;;    XtRealizeWidget(buffer);

	;;; -------------------------------------------------------- ;;;
	;;; THIS IS A HARD CODED COPY FOR MULTI-WINDOW DRAWING       ;;;
	;;; Warning: other characteristics may need to be added!     ;;;
	;;; -------------------------------------------------------- ;;;
		XptValue(widget, XtN font)       ->  XptValue( buffer, XtN font );
		XptValue(widget, XtN foreground) ->  XptValue( buffer, XtN foreground );
		XptValue(widget, XtN background) ->> XptValue( buffer, XtN background )
										 ->  the_bg;
		XptValue(widget, XtN lineWidth)  -> XptValue( buffer, XtN lineWidth );
		XptValue(widget, XtN lineStyle)  -> XptValue( buffer, XtN lineStyle );
		XptValue(widget, XtN function)   -> XptValue( buffer, XtN function );
	;;; -------------------------------------------------------- ;;;
	;;; Start the drawing procedures                             ;;;
	;;; -------------------------------------------------------- ;;;

	define lconstant go_make_the_image();
	dlocal 1 % stored_go_window( pane ) %;
		;;; -------------------------------------------------------- ;;;
		;;; Call the procedure that draws the dragged object         ;;;
		;;; -------------------------------------------------------- ;;;
		buffer -> pane.stored_go_window;
		drawingProc(buttonContext);
	enddefine;

	define lconstant go_make_fixed_image();
		dlocal 1 % the_go_components( pane ) % = [% go_dragged_object %];
		dlocal 1 % stored_go_xorigin( pane ) % =
							pane.stored_go_xorigin + go_buffer_x - x;
		dlocal 1 % stored_go_yorigin( pane )  % =
							pane.stored_go_yorigin + go_buffer_y - y;
		XpwFillBackground( buffer, the_bg,
						   go_buffer_x, go_buffer_y, safe_width, safe_height);
		go_make_the_image();
	enddefine;

	if ( fixed_image_flag ) then
		max( min(x, widget_width - safe_width)  , 1) -> go_buffer_x;
		max( min(y, widget_height - safe_height), 1) -> go_buffer_y;
		go_make_fixed_image();
	elseif ( double_buffer_flag ) then
		max( min(x, widget_width - safe_width)  , 1) -> x;
		max( min(y, widget_height - safe_height), 1) -> y;
		XpwCopyAreaTo(widget, buffer, x, y, safe_width, safe_height);
		go_make_the_image();
	else
		go_make_the_image();
	endif;
enddefine;


define go_image_wrapper(gxop, buttonContext, widget, buffer);
lvars gxop, buttonContext, widget, buffer;
;;; REF: go_image_wrapper( OP, DRAG_CONTEXT, WINDOW, PIXMAP );
;;; REF: CONVENIENCE FUNCTION FOR go_draw_image AND go_erase_image
;;; REF: This either draws or undraws with exclusive or the dragged object.
;;; REF: OP is a graphics context operation, eg: GXxor (REF * GXxor,GXcopy).
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: PIXMAP is a XpwPixmap (REF * XpwPixmap).
	lvars safe_width, safe_height, x, y, x0, y0;
	lvars go_image, go_gxop, go_color;

	;;; WARNING: Might need scaling for width and height
	;;;          Might need adaptation for thicker lines
	go_safe_size(buttonContext) -> (x, y, safe_width, safe_height);

	;;; THE AREA MIGHT START FROM X,Y RATHER THAN (0,0) !!!
	if ( buttonContext.fixedImage ) then
		go_buffer_x -> x0;
		go_buffer_y -> y0;
	else
		x -> x0; y -> y0;
	endif;

	XptValue(widget, XtN function) -> go_gxop;    ;;; remember gxop
	gxop -> XptValue(widget, XtN function);

	if (gxop == GXxor) then
		/* CODE USING IMAGES RATHER THAN WIDGETS: --------------------------- */
		XpwGetImage(buffer, x0, y0, safe_width, safe_height, false, false)
		-> go_image ;
		XpwPutImage(widget, go_image, x0, y0, x, y, safe_width, safe_height );
		/* RESTORE XOR'RED BACKGROUND TO COLOR ------------------------------ */
		XptValue(widget, XtN foreground) -> go_color;          ;;; remember
		XptValue(widget, XtN background) -> XptValue(widget, XtN foreground);
		XpwFillRectangle(widget, x, y, safe_width, safe_height);
		go_color -> XptValue(widget, XtN foreground);          ;;; repair pixmap
		XpwRedrawArea(widget, x, y, safe_width, safe_height ); ;;; & screen
		/* ------------------------------------------------------------------ */
	else
		/* CODE USING WIDGETS - faster over net ----------------------------- */
		XpwCopyTo(buffer, widget, x0, y0, safe_width, safe_height, x, y );
		/* ------------------------------------------------------------------ */
	endif;

	go_gxop -> XptValue(widget, XtN function);    ;;; repair gxop

enddefine;


define go_draw_image(buttonContext, widget, buffer);
lvars buttonContext, widget, buffer;
;;; REF: go_draw_image( DRAG_CONTEXT, WINDOW, PIXMAP );
;;; REF: This draws the dragged object with GXcopy (REF * GXcopy) on the window.
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: PIXMAP is a XpwPixmap (REF * XpwPixmap).
		go_image_wrapper(GXcopy, buttonContext, widget, buffer);
enddefine;


define go_erase_image(buttonContext, widget, buffer);
;;; REF: go_draw_image( DRAG_CONTEXT, WINDOW, PIXMAP );
;;; REF: This erases the dragged object with GXxor (REF * GXxor) on the window.
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: PIXMAP is a XpwPixmap (REF * XpwPixmap).

	lvars buttonContext, widget, buffer;
	go_image_wrapper(GXxor, buttonContext, widget, buffer);
enddefine;


;;; endsection;

/*
;;; REF: -- Low level: callbacks -------------------------------------------------
;;; REF:
;;; REF: The following routines are normally not needed by the user, but are
;;; REF: added for completeness of this ref-file.
;;; REF:
;;; REF: There are foure procedures which are implementing the different stages
;;; REF: of the callbacks. The idea behind the go_startdrag() procedure is that
;;; REF: the user will normally already write a handler (see TEACH * GO_DRAG).
;;; REF:
;;; REF: To start the go_drag operation at the click of a button. Therefore the
;;; REF: event of the "Button Press" is normally already consumed by the system.
;;; REF: The go_drag() function thus uses a fake callback rather than a real
;;; REF: one.
;;; REF:
;;; REF: go_startdrag() will make sure that all drawing is done temporarily so
;;; REF: that no time is waisted writing to memory. go_dropdrag() does the
;;; REF: reverse.
;;; REF:
;;; REF:     go_startdrag  ;;; fake callback to initialize the go_drag operation
;;; REF:     go_dodrag,    ;;; real callback to do the go_drag operation
;;; REF:     go_dropdrag,  ;;; fake callback to finish the go_drag operation
;;; REF:     go_dodrop,    ;;; real callback to finish the go_drag operation
;;; REF:
*/


/*
-- Define go_drag routines ------------------------------------------------

;;; REF: mouse_coords( WINDOW, PANE ) -> (X, Y);
;;; REF: INTERNAL procedure to get the last registred mouse positions in
;;; REF: world coordinates. This is a local procedure to LIB * GO_XDRAG.
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).

*/
	define lconstant mouse_coords(window, pane) -> (x, y);
		lvars window, pane, x, y;
		go_transxyin( XptValue(window, XtN mouseX),
					  fast_XptValue(window, XtN mouseY),
					  pane
					) -> (x, y);
		round(x) -> x ; round(y) -> y ;
	enddefine;

   vars go_mouse_coords = mouse_coords;

/*
-- Button Press Handler ---------------------------------------------------

;;; REF: go_startdrag( WINDOW, DRAG_CONTEXT, CALL_DATA );
;;; REF: Callback procedure for when a drag is started. It erases the
;;; REF: dragged object based on the method in the drag context.
;;; REF: It prepares the fixed image if needed. It will turn the pane into
;;; REF: temporary drawing mode (no drawing in the attached memory pixmap).
;;; REF: Then redraws the object for the first time in drag mode.
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: CALL_DATA external pointer indicating the state of the mouse-buttons.

*/

define go_startdrag(window, buttonContext, call);
	lvars window, buttonContext, call;
	lvars mouseX, mouseY, objectX, objectY, buffer, pane;
	if exacc ^int call >= 0 then
		;;; start of drag operation

		explode(buttonContext.lastCoords) -> (objectX, objectY);
		;;; Remove picture
		if (buttonContext.eraseProc) then
			;;; Call the procedure that erases the dragged object
			eraseProc(buttonContext)(buttonContext);
		else
			;;; If no eraseProc is provided the object is drawn
			;;; in a separate pixmap and then xor'ed to allow for
			;;; a rather clean removal (not sensitive to differences
			;;; between drawing in source mode and xor).
			go_make_image(buttonContext.drawProc, buttonContext, window)
			-> buffer;
			go_erase_image(buttonContext, window, buffer);
		endif;

		;;; set variables
		true -> buttonContext.inDrag;

		mouse_coords( window, go_default_pane ) -> (mouseX, mouseY) ;
		( mouseX - objectX ) -> buttonContext.startXOffset;
		( mouseY - objectY ) -> buttonContext.startYOffset;

		;;; disable the backing pixmap for the window so changes
		;;; are temporary. The saved Pixmap represents all graphics
		;;; without the dragged object.
		go_applist( procedure( obj, pane ); lvars obj, pane;
						PixmapOff ->
						XptValue(the_go_window(pane), XtN pixmapStatus);
					endprocedure, go_dragged_object );

		;;; Call the procedure that draws the dragged object
		if (buttonContext.fixedImage) or (buttonContext.doubleBuffer) then
			;;; prepare for optimisation of fixed picture
			if (buttonContext.eraseProc) then    ;;; otherwise already made...
				go_make_image(buttonContext.drawProc, buttonContext, window)
				-> buffer;
			endif;
			buffer -> buttonContext.imageDrag;
			go_draw_image(buttonContext, window, buffer);
		else
			buttonContext.drawProc(buttonContext);
		endif;

	endif;
enddefine;


/*
-- Mouse Motion event handler ---------------------------------------------

;;; REF: go_dodrag( WINDOW, DRAG_CONTEXT, CALL_DATA );
;;; REF: Callback procedure for during a drag. It restores the area of the
;;; REF: previous drag from memory. It finds out where the mouse now is,
;;; REF: then redraws the object on the new location still in drag mode.
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: CALL_DATA external pointer indicating the state of the mouse-buttons.

*/
define /*lconstant*/ go_redraw_old_area(window, buttonContext) -> (x, y, w, h);
lvars window, buttonContext, x, y, w, h;
;;; REF: If fixed don't redraw old area completely (see go_redraw_new_area).
;;; REF: X,  Y,  W  and H  are the original screen dimensions.
	go_safe_size(buttonContext) -> (x, y, w, h);
	unless ( buttonContext.fixedImage or buttonContext.doubleBuffer) then
		XpwRedrawArea(window, x, y, w, h);
	endunless;
enddefine;


define /*lconstant*/ go_redraw_new_area(x, y, w, h, window, buttonContext);
lvars x, y, w, h, window, buttonContext;
lvars xp, yp, wp, hp, x1=x, y1=y, w1=w, h1, x2=x, y2=y, w2, h2;
;;; REF: When we have a fixedImage then only the strips of the rectangle
;;; REF: that stick out under the new area need to be redrawn. This can
;;; REF: only be done after the new area is drawn because we need to know
;;; REF: the new location and size!
;;; REF: X,  Y,  W  and H  are the original screen dimensions.
;;; REF: XP, YP, WP and HP are the new screen dimensions.
;;; REF: Note that for real fixed images W = WP and H = HP.
;;; REF:
;;; REF:           CASE 1                             CASE 2
;;; REF:                                        (xp,yp)
;;; REF:                                           +---------------+
;;; REF:     (x,y)                         (x,p)   |               |
;;; REF:       +-----------------+           +-----|-----------+   |
;;; REF:(xp,yp)|/////////1///////|           |\\2\\|           |   |
;;; REF:   +----------------+====|           |=====+---------------+
;;; REF:   |   |            |\\\\|           |/////////////////|
;;; REF:   |   |            |\\2\|           |////////1////////|
;;; REF:   |   +------------|----+           +-----------------+
;;; REF:   +----------------+
;;; REF:
;;; REF:           CASE 3                             CASE 4
;;; REF:                              (xp,yp)
;;; REF:                                 +----------------+
;;; REF:     (x,p)                       |(xp,yp)         |
;;; REF:       +-----------------+       |   +-----------------+
;;; REF:       |/////////1///////|       |   |            |\\2\|
;;; REF:       |====+----------------+   +----------------+====|
;;; REF:       |\\\\|(xp,yp)     |   |       |/////////////////|
;;; REF:       |\\2\|            |   |       |////////1////////|
;;; REF:       +----|------------+   |       +-----------------+
;;; REF:            +----------------+
;;; REF:
	go_draw_image(buttonContext, window, buttonContext.imageDrag);
	go_safe_size(buttonContext) -> (xp, yp, wp, hp);
	if ( y > yp ) then          ;;; CASE 2 and 4
		yp + hp -> y1;
		abs((y+h)-y1) -> h1;
		hp - (y-yp) -> h2;
	else                        ;;; CASE 1 and 3
		yp -> y2;
		yp - y -> h1;
		h - h1 -> h2;
	endif;
	if ( x > xp ) then          ;;; CASE 1 and 4
		xp + wp -> x2;
		abs((x+w)-x2) -> w2;
	else /* ( x < xp ) */       ;;; CASE 2 and 3
		xp - x -> w2;
	endif;
	if ( w1 > 0 and h1 > 0 ) then
		XpwRedrawArea(window, x1, y1, w1, h1 );
	endif;
	if ( w2 > 0 and h2 > 0 ) then
		XpwRedrawArea(window, x2, y2, w2, h2 );
	endif;
enddefine;


define go_dodrag(window, buttonContext, call);
	dlvars window, buttonContext;
	lvars  call;
	lvars  x, y, x1, y1, w1, h1, buffer;
		;;; don't do anything if we are not in a go_drag
		returnunless(buttonContext.inDrag);

		false -> XptValue(window, XtN autoFlush, TYPESPEC(:XptBoolean));

		;;; erase the previously drawn shape
		;;; WARNING: Might need scaling for width and height
		;;;          Might need adaptation for thicker lines
		go_redraw_old_area(window, buttonContext) -> (x1, y1, w1, h1);

		procedure( obj, pane ); lvars obj, pane;
		lvars win = the_go_window(pane), x, y, w, h;
			unless win == window do
				;;; Not completely safe cleanup of alternative windows
				go_safe_region(pane, obj) -> (x, y, w, h);
				XpwRedrawArea(win, x, y, w, h );
			endunless;
		endprocedure;
	go_applist( /* procedure on stack */, go_dragged_object );

		;;; find out where the mouse is and record where the shape is
		go_mouse_coords( window, go_default_pane ) -> (x, y);
		{^(x-buttonContext.startXOffset) ^(y-buttonContext.startYOffset)}
		-> buttonContext.lastCoords;

		;;; Call the procedure that draws the dragged object
		if (buttonContext.fixedImage) then
			go_redraw_new_area(x1, y1, w1, h1, window, buttonContext);
		elseif (buttonContext.doubleBuffer) then
			go_make_image(buttonContext.drawProc, buttonContext, window)
			-> buttonContext.imageDrag;
			go_redraw_new_area(x1, y1, w1, h1, window, buttonContext);
		else
			buttonContext.drawProc(buttonContext);
		endif;
		true -> XptValue(window, XtN autoFlush, TYPESPEC(:XptBoolean));

enddefine;


/*
-- Button Release Handler -------------------------------------------------

;;; Part 1

;;; REF: go_dropdrag( WINDOW, DRAG_CONTEXT, CALL_DATA );
;;; REF: Redraws the object no longer in drag mode. If the drop is allowed
;;; REF: (REF * GO_XDRAG/dragContext/allowedDrop) then the new location is
;;; REF: used otherwise the original location is used again. The pane is also
;;; REF: changed from temporary to normal drawing mode (REF * pixmapStatus).
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: CALL_DATA external pointer indicating the state of the mouse-buttons.

*/

define go_dropdrag(window, buttonContext, call);
	lvars window, buttonContext, call;
	lvars x, y, pane;
	if exacc ^int call < 0 then

		;;; end of drag operation

		;;; go_clear variables
		false -> buttonContext.inDrag;

		;;; find out where the mouse was and record where the shape is
		if ( buttonContext.allowedDrop ) then
			mouse_coords( window, go_default_pane ) -> (x, y);
			{^(x-buttonContext.startXOffset) ^(y-buttonContext.startYOffset)}
			-> buttonContext.lastCoords;
		else
			copydata(buttonContext.startCoords) -> buttonContext.lastCoords;
		endif;

		;;; turn the backing pixmap back on
		go_applist( procedure( obj, pane ); lvars obj, pane;
						PixmapOn ->
						XptValue(the_go_window(pane), XtN pixmapStatus);
					endprocedure, go_dragged_object );
		XptAppTryEvents( XptDefaultAppContext );

		;;; Call the procedure that draws the dragged object
		buttonContext.drawProc(buttonContext);

	endif;
enddefine;

/*
-- Button Release Handler -------------------------------------------------

;;; Part 2: Convenience function TO CLEANUP THE WIDGET SET

;;; REF: go_dodrop( WINDOW, DRAG_CONTEXT, CALL_DATA );
;;; REF: Cleans up all the structures and callbacks and calls the go_dropdrag
;;; REF: procedure.
;;; REF: WINDOW is a XpwGraphics widget attached to a go_pane (REF * GO_PANE).
;;; REF: DRAG_CONTEXT contains information on the dragged object (see also
;;; REF: REF * GO_XDRAG/dragContext).
;;; REF: CALL_DATA external pointer indicating the state of the mouse-buttons.

*/

lvars last_motion_event = false;

define go_dodrop(window, buttonContext, call);
	lvars window, buttonContext, call;
	lvars cp_dragContext;

 "go_dodrop" :: go_debug_event_list -> go_debug_event_list;

	if exacc ^int call <= 0 then

		;;; COPY to save original for removing the buttonEvent callback
		copydata(buttonContext) -> cp_dragContext;

		;;; END OF DRAG OPERATION
		go_dropdrag(window, cp_dragContext, call);

		;;; REMOVE UNNESSARRY CALLBACKS: needs exact copy of orig data
		false -> buttonContext.userData;
		XtRemoveCallback(window, XtN buttonEvent, ident go_dodrop,
						 buttonContext);
		true  -> buttonContext.inDrag;
		copydata(buttonContext.startCoords) -> buttonContext.lastCoords;
		;;; pr('DESTROY '); pr(buttonContext); pr('\n');
		XtRemoveCallback(window, XtN motionEvent, ident go_dodrag,
						 buttonContext);
		false -> last_motion_event;

	endif;
enddefine;


/* --------------------------------------------------------------------- */
/* TOP LEVEL RC GRAPHIC DRAG ROUTINES                                    */
/*                                                                       */
/* CONVENIENCE WRAP-FUNCTION FOR go_startdrag TO SETUP THE WIDGET SET    */
/*                                                                       */
/* parameters as explained in: defclass dragContext                      */
/*                                                                       */
/* --------------------------------------------------------------------- */

define go_init_drag(erase_proc, draw_proc, fixed_image, double_buffer, obj);
	lvars erase_proc, draw_proc, fixed_image, double_buffer, obj;
	lvars startX, startY, w, h;
	go_xcentre( obj ) -> startX;
	go_ycentre( obj ) -> startY;
	go_bounding_box( obj ) -> (, , w, h);
	consdragContext(false, {^startX ^startY}, {^startX ^startY},
					0, 0, w, h,
					erase_proc, draw_proc, fixed_image, double_buffer,
					identfn, true, false )
enddefine;

vars go_object_dragable = false; ;;; REF: Global Var: used to track objects
;;; REF: Global Var: This is used for time between mouse down and up events.

/*
-- go_drag -----------------------------------------------------------

;;; REF: TOP LEVEL RC GRAPHIC DRAG ROUTINE
;;; REF:
;;; REF: go_drag(CLEAR_PROCEDURE, DRAW_PROCEDURE, FIXED_IMAGE, SCREEN_OBJECT);
;;; REF:
;;; REF: This is the user's interface with the data structure described above.
;;; REF: The user has to provide the data necessary to go_drag. The object
;;; REF: will provide the size and location of the area affected by the drag.
;;; REF:
;;; REF: The function itself does the following:
;;; REF:      1. Erases current graphical object (with CLEAR_PROCEDURE)
;;; REF:      2. Prepares for optimisation if object has a fixed image
;;; REF:      3. Starts the go_drag operation
;;; REF:      4. Fake Button Press Handler (already consumed to call go_drag)
;;; REF:      5. Supposes you are dragging, until further notice...
;;; REF:
;;; REF: Once it has done this, the go_drag operation will be handled by the
;;; REF: event procedures of X. The callbacks are the installed by go_drag()
;;; REF: (see also go_startdrag, go_dodrag, go_dropdrag and go_dodrop).
;;; REF:
;;; REF: CLEAR_PROCEDURE is the procedure used to clear the object.
;;; REF: DRAW_PROCEDURE is the procedure used to draw the object.
;;; REF: FIXED_IMAGE is a boolean value for optimised drawing if image is fixed.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).

 */

define go_drag(eraseProc, drawProc, fixedImage, obj);
	lvars eraseProc, drawProc, fixedImage, obj;
	lvars go_dragContext, pane = go_default_pane, window = the_go_window(pane);

 datakey(obj) :: go_debug_event_list -> go_debug_event_list;

	if last_motion_event then
		npr('Warning: old drag and drop not cleanly finished...');
		npr('         see also go_reinit()');
	else
		true -> last_motion_event;
	endif;

	;;; Remove all other handlers for the mouse motion and up events: go_xaction
	false -> go_object_dragable;

	go_init_drag(eraseProc, drawProc, fixedImage, pane.go_double_buffer, obj)
	-> go_dragContext ;
	obj -> go_dragged_object;

	XtAddCallback(window, XtN buttonEvent, ident go_dodrop,
				  go_dragContext);

	;;; Start of drag operation
	;;; Fake Button Press Handler (it is already consumed to call go_drag)
	go_startdrag(window, go_dragContext, null_external_ptr);

	;;; Supposes you are dragging, until further notice...
	true -> go_dragContext.inDrag;

	XtAddCallback(window, XtN motionEvent, ident go_dodrag,
				  go_dragContext);
	{% window, XtN motionEvent, ident go_dodrag, go_dragContext %}
	-> last_motion_event;
enddefine;

/*
-- go_fixed_drag ----------------------------------------------------------

;;; REF: go_fixed_drag( PROCEDURE, SCREEN_OBJECT );
;;; REF: CONVENIENCE FUNCTION FOR DRAGGING FIXED IMAGES.
;;; REF: The object is only drawn once (in the beginning of the drag) with the
;;; REF: given procedure. The rest of the drag consist of copying that fixed
;;; REF: image on new locations.
;;; REF: This is equivalent to go_drag() with the:
;;; REF:          1. drawProc -> eraseProc;
;;; REF:          2. true -> fixedImage;
;;; REF: Note: The dragged area is always rectangular and is not transparent.
;;; REF: This is the fastest way of dragging.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).

*/

define go_fixed_drag(drawProc, obj);
	lvars drawProc, obj;

	go_drag(false, drawProc, true, obj);

enddefine;


/*

-- go_drawn_drag ----------------------------------------------------------

;;; REF: go_drawn_drag( PROCEDURE, SCREEN_OBJECT );
;;; REF: CONVENIENCE FUNCTION FOR DRAGGING TRANSPARENT IMAGES.
;;; REF: The object is redrawn for each new location while dragging with the
;;; REF: given procedure.
;;; REF: This is equivalent to go_drag() with the:
;;; REF:          1. drawProc -> eraseProc;
;;; REF:          2. false -> fixedImage;
;;; REF: Note: The dragged area will be transparent except for the drawing,
;;; REF: which is redrawn continuously during dragging.
;;; REF: SCREEN_OBJECT is a go_screen_object instance (REF * GO_SCREEN_OBJECT).

*/

define go_drawn_drag(drawProc, obj);
	lvars drawProc, obj;

	go_drag(false, drawProc, false, obj);

enddefine;


/*

-- go_motionhint ----------------------------------------------------------

;;; REF: go_motionhint( PANE ) -> BOOLEAN;
;;; REF: BOOLEAN -> go_motionhint( PANE );
;;; REF: Top level GO motionhint switch. This is a convenience wrap-function
;;; REF: to switch motionhints (REF * GO_XDRAG/XpwQueryPointer
;;; REF: E.g.: To turn MotionHints on:
;;; REF:         true -> go_motionhint( go_default_pane );
;;; REF: E.g.: To turn MotionHints off (default)::
;;; REF:         false -> go_motionhint( go_default_pane );
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).

*/

define go_motionhint( pane );
lvars pane;
	XpwUseMotionHints( the_go_window(pane) );
enddefine;

define updaterof go_motionhint(go_value, pane);
	lvars go_value, pane;
	if ( go_value ) then
		XpwQueryPointer -> go_mouse_coords;
	else
		mouse_coords -> go_mouse_coords;
	endif;
	go_value -> XpwUseMotionHints( the_go_window(pane) );
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_xdrag = true;


/*  --- Revision History ---------------------------------------------------
 * BR 25/06/93
 *     Optimised the fixedImage dragging by not redrawing the area under the
 *     picture itself. Only the old area that sticks out from the new area!
 *     see go_redraw_[old/new]_area
 * BR 26/05/93
 *     THE WHITE WIDGET SHOULD NOT BE REALIZED: because of bg-colour X-problems
 *     deleted XtRealizeWidget(buffer);
 *     Added Comments.
 * BR 18/05/93
 *     Corrected bug in go_dodrag() where go_safe_size was called rather than
 *     go_safe_region().
 * BR 13/05/93
 *     Optimised callbacks away when using go_drag() by setting the global
 *     variable go_object_dragable to false when dragging starts.
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 25/03/1993
 *     Standardized variable names and declarations. Added pane arg to the
 *     go_mouse_coords()/mouse_coords()/XpwQueryPointer() routines
 * BR 13/11/1992
 *     Made standalone and GO specific (almost no references to rc_graphics
 *     left...).
 * BR 13/10/1992
 *     changed mouse_coord() to return integers
 * BR 03/09/1992
 *     changed influence of eraseProc, avoids buffer
 * Jon Meyer 19/08/1992
 *     extra extra_space in go_safe_size take into account the
 *     JoinStyle == JoinMiter in polygons
 * BR 06/08/1992
 *     slight optimisation by reusing the buffer added extra
 *     user-field (userData) to the context (for JLC)
 * BR 22/05/1992
 *     changed to normalize rectangles; this allows negative width or heights
 *     to be entered (HELP * RC_DRAG).
 * BR 21/05/1992
 *     added convenience functions fixed and drawn go_drag.
 * BR 19/05/1992
 *     added eraseProc and allowedDrop.
 * Aaron Sloman 29/03/1992
 *     making rc_motionhint active variable
 * BR 25/03/1992
 *     made object grabbing relative to mouse, changed to use LIB * RC_CONTEXT.
 * BR 24/03/1992
 *     added go_safe_size, reorganised file, comment.
 * BR 23/03/1992
 *     added support for color and lineWidth/Style.
 * BR 18/03/1992
 *     added go_erase_image.
 * BR 17/03/1992
 *     added toplevel RC routines, substantially rewritten callback data and
 *     routines.
*/
;;; eof
