/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_rectangle.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 10th August 1993
;;; File: go_rectangle.p
;;; Author: Ben L.E. Rabau

compile_mode :pop11 +strict;

uses go_polygon;

;;; --------------------------------------------------------------------

;;; INTRO: The CLASS go_rectangle represent all rectangular shapes under any
;;; INTRO: angle with the X and Y axes and is based on the go_polygon class.
;;; INTRO: See also REF * GO_POLYGON. The conditions for being rectangular are
;;; INTRO: four corners and right angles in non-scaled world coordinates. If
;;; INTRO: a rotated rectangle is scaled with different horizontal and vertical
;;; INTRO: scaling factors, it becomes a trapezium.

vars procedure go_keep_rectangular;

define :class go_rectangle;
	isa go_polygon;
	slot stored_go_regular == false;
;;; REF: A rectangle is not a regular go_polygon (see REF * GO_SQUARE).
	slot stored_go_closed  ==  true;
;;; REF: A rectangle is a closed go_polygon.
	slot stored_go_npoints ==     4;
;;; REF: A rectangle has 4 angles.
	slot stored_go_width   ==   100;
;;; REF: Stored value of the width of the rectangle in world coordinates.
	slot stored_go_height  ==    20;
;;; REF: Stored value of the height of the rectangle in world coordinates.

	on cons do procedure() -> rect;
			   lvars rect = apply();
				   go_keep_rectangular( rect );
			   endprocedure;
enddefine;


;;; --------------------------------------------------------------------

define :method updaterof go_npoints( n, rect :go_rectangle );
lvars n, rect;
;;; REF: INTEGER -> go_npoints( RECTANGLE );
;;; REF: Refuses to update the number of points to a non-rectangular shape.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
	unless ( n = 4 ) then
		mishap('cannot update number of sides of a go_rectangle to', [^n]);
	endunless;
enddefine;



define :method go_width( rect :go_rectangle ) -> w;
lvars w, rect;
;;; REF: go_width( RECTANGLE ) -> INTEGER;
	stored_go_width( rect ) -> w;
enddefine;

define :method updaterof go_width( w, rect :go_rectangle );
lvars w, rect;
;;; REF: INTEGER -> go_width( RECTANGLE );
;;; REF: The width of the rectangle in world coordinates. This is
;;; REF: counted between the first and last point in go_local_coords.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
	go_clear( rect );
	w -> rect.stored_go_width ;
	go_keep_rectangular(rect);
enddefine;

define :method go_height( rect :go_rectangle ) -> h;
lvars h, rect;
;;; REF: go_height( RECTANGLE ) -> INTEGER;
	stored_go_height( rect ) -> h;
enddefine;

define :method updaterof go_height( h, rect :go_rectangle );
lvars rect;
;;; REF: INTEGER -> go_height( RECTANGLE );
;;; REF: The height of the rectangle in world coordinates. This is
;;; REF: counted between the first and second point in go_local_coords.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
	go_clear( rect );
	h -> rect.stored_go_height ;
	go_keep_rectangular(rect);
enddefine;


define :method go_keep_rectangular( rect :go_rectangle );
lvars rect;
;;; REF: go_keep_rectangular( RECTANGLE );
;;; REF: Calculates the position of the corners of the rectangle in world
;;; REF: coordinates and stores them in go_local_coords. The bounding box is updated
;;; REF: automatically before the object is redrawn.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
lvars w, h, pane, left, right, up, down;
	;;; LOCAL COORDINATES
	rect.go_width  -> w;
	rect.go_height -> h;
	;;; be carefull not to lose acurracy (for odd w or h)
	[%
		   round(-w / 2)   ->> left;  round(h / 2) ->> up;
		   left;                      round(up - h) ->> down;
		   round(left + w) ->> right; down;
		   right;                     up;
	%] -> rect.stored_go_local_coords;
	go_set_bounding_box( rect );
	go_update_and_draw( rect );
enddefine;


define :method go_initialise_polygon( rect :go_rectangle );
lvars rect;
;;; REF: go_initialise_polygon( RECTANGLE );
;;; REF: Initialise the world and screen coordinates of a (changed?) rectangle.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
	go_keep_rectangular( rect );
enddefine;


;;;----------------------------------------------------------------
;;; EDITING

define :method go_position_corner( rect :go_rectangle, frozen_index ) -> (x, y);
lvars rect, frozen_index;
;;; REF: go_position_corner( RECTANGLE, N ) -> (X, Y);
	stored_go_local_coords(rect)(frozen_index*2 - 1) -> x ;
	stored_go_local_coords(rect)(frozen_index*2)     -> y ;
	go_transxyout(x, y, rect) -> (x, y);
enddefine;

define :method updaterof go_position_corner( x, y, rect :go_rectangle, frozen_index );
lvars x, y, rect, frozen_index;
;;; REF: (X, Y) -> go_position_corner( RECTANGLE, N );
;;; REF: The world coordinates of the N'th point of the rectangle.
;;; REF: X/Y are integer world coordinates.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
;;; REF: N the index [1..4] in the list of points representing the rectangle.
lvars oldx, oldy, dw, dh;
	go_batch_mode_on( rect );
	go_clear( rect );
	stored_go_local_coords(rect)(frozen_index*2 - 1) -> oldx ;
	stored_go_local_coords(rect)(frozen_index*2)     -> oldy ;
	go_transxyout(oldx, oldy, rect) -> (oldx, oldy);
	x - oldx -> dw;
	y - oldy -> dh;
	dw div 2 + rect.go_xcentre -> rect.go_xcentre;
	dh div 2 + rect.go_ycentre -> rect.go_ycentre;
	dw; if (frozen_index == 1) or (frozen_index == 2) then * -1 endif -> dw;
	dh; if (frozen_index == 2) or (frozen_index == 3) then * -1 endif -> dh;
	rect.go_width  + dw -> rect.go_width;
	rect.go_height + dh -> rect.go_height;
	go_batch_mode_off( rect );
enddefine;

define :method go_make_editable( rect :go_rectangle );
lvars rect;
;;; REF: go_make_editable( RECTANGLE );
;;; REF: Add resize editors to the rectangle object.
;;; REF: Editors are instances of go_edit_point (see REF * GO_EDIT_POINT)
;;; REF: which allow resizing by Direct Manipulation with the mouse. For
;;; REF: rectangles these editors are the four corners and the centre point.
;;; REF: RECTANGLE is an instance of go_rectangle (see REF * GO_RECTANGLE).
lvars centre_pnt, pnt1, i, pnti;
	;;; remove any pending editors:
	go_make_uneditable( rect );
	;;; create new editors at least one in centre and one in a vertex
	[% newgo_edit_point() ->> centre_pnt;         ;;; puts the editor on stack
	   rect -> go_edited_object( centre_pnt );
	   go_add_to( centre_pnt, go_default_pane );
	   if ( rect.go_regular ) then
		   newgo_edit_point() ->> pnt1;           ;;; puts the editor on stack
		   rect -> go_edited_object( pnt1 );
		   go_attach_refpnt -> go_move_absolute( pnt1 );
		   go_add_to( pnt1, go_default_pane );
	   else
		   for i from 1 to rect.go_npoints do
			   newgo_edit_point() ->> pnti;       ;;; puts the editor on stack
			   rect -> go_edited_object( pnti );
			   go_position_corner(% i %) -> go_move_absolute( pnti );
			   go_add_to( pnti, go_default_pane );
		   endfor;
	   endif;
	%] -> rect.go_editors;
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_rectangle = true;

/* --- Revision History --------------------------------------------
 * BR 10/08/93
 *     Changed the go_keep_rectangular() method to represent the rectangle
 *     under 0 degree angle. The rest is done in go_rotatable.p and go_polygon.p
 *     Changed go_position_corner() accordingly; see go_transxy[in/out]().
 * BR 05/07/93
 *     Added a slot go_editors to avoid the overuse of go_live_object to
 *     also keep the editors within go_make_editable().
 * BR 08/05/93
 *     Removed code for go_[x/y]centre(); replaced by more general case in
 *     go_located.p
 *     Replaced go_attach_centre() by go_position_of_centre() in go_located.p
 * BR 08/05/93
 *     Added comments
 *     Improved the accuracy of go_keep_rectangular with new default for scale
 * BR 30/04/93
 *     Moved go_square class and go_oblong class into separate file
 *     Changed reposition_corner into go_position_corner and made it a method.
 * BR 22/04/93
 *     Abandoned support for both -CoordModePrevious- & -CoordModeOrigin- and
 *     therefore replaced the GO variable ScreenCoordMode by CoordModeOrigin.
 * BR 15/04/93
 *     Removed global var oblong_radius from go.p and go_oblong class.
 * BR 26/11/92
 *     Global Name changes and code cleanup
 */
;;; eof
