/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_rotatable.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 14th July 1993
;;; File: go_rotatable.p
;;; Author: B Rabau

compile_mode :pop11 +strict;

;;; INTRO: The MIXIN go_rotatable is for objects which can be rotated.
;;; INTRO: A special meaning is attached to the rotation of groups (see
;;; INTRO: REF * GO_GROUP/go_angle).



define :mixin go_rotatable;
	slot stored_go_angle   == 0;
;;; REF: Stored value of angle in degrees to the reference point (to the east).
	slot stored_go_xrotation == 0;
;;; REF: Stored horizontal rotation of the object's local coordinate system.
	slot stored_go_yrotation == 0;
;;; REF: Stored vertical rotation of the object's local coordinate system.
enddefine;


;;; ---------------------------------------------------------
;;; ANGLE:

define :method go_angle( obj :go_rotatable ) -> angle;
lvars obj , angle = obj.stored_go_angle;
;;; REF: go_angle( ROTATABLE ) -> INTEGER;
enddefine;

define :method updaterof go_angle( angle, obj :go_rotatable );
lvars angle, obj;
;;; REF: INTEGER -> go_angle( ROTATABLE );
;;; REF: Current angle of the object.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
	go_clear( obj );
	angle -> obj.stored_go_angle;
	go_update_and_draw( obj );
enddefine;



;;; ---------------------------------------------------------
;;; ROTATION POINT:

define :method go_xrotation( obj :go_rotatable ) -> x;
lvars obj, x = obj.stored_go_xrotation + obj.go_xorigin;
;;; REF: go_xrotation( ROTATABLE ) -> REAL;
enddefine;

define :method updaterof go_xrotation( x, obj :go_rotatable );
lvars x, obj;
;;; REF: REAL -> go_xrotation( ROTATABLE );
;;; REF: The horizontal world coordinate for rotation of the local coords of
;;; REF: the given object.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
	x - obj.go_xorigin -> obj.stored_go_xrotation;
enddefine;

define :method go_yrotation( obj :go_rotatable ) -> y;
lvars obj, y = obj.stored_go_yrotation + obj.go_yorigin;
 ;;; REF: go_yrotation( ROTATABLE ) -> REAL;
enddefine;

define :method updaterof go_yrotation( y, obj :go_rotatable );
lvars y, obj;
;;; REF: REAL -> go_yrotation( ROTATABLE );
;;; REF: The vertical world coordinate for rotation of the local coords of
;;; REF: the given object.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
	y - obj.go_yorigin -> obj.stored_go_yrotation;
enddefine;

;;;----------------------------------------------------------------
;;; CONVENIENCE FUNCTION FOR EDITING FACILITIES

define :method go_position_of_rotation( obj :go_rotatable ) -> (x, y);
lvars x = obj.go_xrotation, y = obj.go_yrotation, locobj;
;;; REF: go_position_of_rotation( ROTATABLE ) -> (X, Y);
enddefine;

define :method updaterof go_position_of_rotation( x, y, obj :go_rotatable );
lvars x, y, locobj;
;;; REF: go_position_of_rotation( X, Y, ROTATABLE );
;;; REF: The world coordinates of the rotation point of a located object. This
;;; REF: is a convenience function using go_xrotation and go_yrotation.
;;; REF: X and Y are the world coordinates for rotation of the local coords
;;; REF: of the object.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
	x -> obj.go_xrotation;
	y -> obj.go_yrotation;
	go_update( obj );
enddefine;
;;; ---------------------------------------------------------
;;; ROTATE

define :method go_rotate( max_angle, by_step, obj :go_rotatable );
lvars max_angle, by_step, obj;
;;; REF: go_rotate( ANGLE, STEP, ROTATABLE );
;;; REF: Rotate the object a step at the time until the given angle.
;;; REF: ANGLE is the integer angle in degrees towards which the object rotates.
;;; REF: STEP is the integer step in degrees taken in the rotation.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
lvars containers = go_visible_in( obj ), start_angle = go_angle(obj), angle;
	for angle from (start_angle mod 360) by by_step to max_angle do;
		applist(containers, go_batch_mode_on);
		angle -> go_angle(obj);
		applist(containers, go_batch_mode_off);
	endfor;
	applist(containers, go_batch_mode_on);
	max_angle -> go_angle(obj);
	applist(containers, go_batch_mode_off);
enddefine;

;;; ---------------------------------------------------------
;;; WORLD VS. LOCAL COORDINATES
;;;
;;; LOCAL HELP FUNCTIONS

define go_calc_rotxyout(x, y, xr, yr, xo, yo, xscale, yscale, cosang, sinang) -> (xt,yt);
lvars x, y, xr, yr, xo, yo, xscale, yscale, cosang, sinang, xt, yt;
	;;; Do the transformation
	(x * xscale) - xr -> x;  ;;; Scale it around the origin and
	(y * yscale) - yr -> y;  ;;; translate to the rotation point
									;;; Rotate it an extra ang degrees and
									;;; translate to the world coord origin
	(x * cosang - y * sinang + xr + xo) -> xt;     ;;; new x coord
	(x * sinang + y * cosang + yr + yo) -> yt;     ;;; new y coord
enddefine;

define go_calc_rotxyin(xt, yt, xr, yr, xo, yo, xscale, yscale, cosang, sinang) -> (x,y);
lvars xt, yt, xr, yr, xo, yo, xscale, yscale, cosang, sinang, x, y;
	;;; Do the reverse transformation
	xt - xo - xr -> xt;             ;;; Make coords local for x/y then
	yt - yo - yr -> yt;             ;;; translate from origin to rotation point
									;;; Rotate it an back ang degrees and
									;;; Then Unscale it around the origin
	((yt * sinang + xt * cosang + xr) / xscale) -> x;
	((yt * cosang - xt * sinang + yr) / yscale) -> y;
enddefine;

;;;------------------------------------------------------
;;; WORLD COORDINATE TRANSFORMATION

;;; go_transxyout is user definable. Make it erase(%1%) to have no effect
;;; Takes two numbers (user coords) and returns two numbers - world coords.
;;; Must leave numbers on stack in the same order

define :method go_transxyout(x, y, obj :go_rotatable ) -> (x,y);
lvars x, y, obj;
;;; REF: go_transxyout( LX, LY, ROTATABLE ) -> ( X, Y );
;;; REF: Calculation of the world coordinates corresponding to
;;; REF: a given position in the objects local coordinates.
;;; REF: LX and LY are the object's local coordinates of a position.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
;;; REF: X and Y are the world coordinates of a position in the pane.
lvars ang = obj.go_angle mod 360;
	if ang = 0 then
		call_next_method(x, y, obj) -> (x,y);
	else
		go_calc_rotxyout(x,                       y,
						 obj.stored_go_xrotation, obj.stored_go_yrotation,
						 obj.go_xorigin,          obj.go_yorigin,
						 obj.go_xscale,           obj.go_yscale,
						 cos(ang),                sin(ang)       ) -> (x,y);
	endif;
enddefine;

;;; go_transxyin is user definable. Make it == erase to have no effect
;;; (Previous line corrected by AS 13/06/93)
;;; Must leave numbers on stack in same order
;;; WARNING: will produce reals in some cases...
;;; Warning. Must be replaced by rc_rotate_xyin if lib rc_rotate_xy used.

define :method go_transxyin(x, y, obj :go_rotatable) -> (x, y);
lvars x, y, obj;
;;; REF: go_transxyin( X, Y, ROTATABLE ) -> ( LX, LY );
;;; REF: Calculation of the object's local coordinates corresponding to a given
;;; REF: position in world coordinates.
;;; REF: X and Y are the world coordinates of a position in the pane.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
;;; REF: LX and LY are the object's local coordinates of a position.
lvars ang = obj.go_angle mod 360;
	if ang = 0 then
		call_next_method(x, y, obj) -> (x,y);
	else
		go_calc_rotxyin(x,                       y,
						obj.stored_go_xrotation, obj.stored_go_yrotation,
						obj.go_xorigin,          obj.go_yorigin,
						obj.go_xscale,           obj.go_yscale,
						cos(ang),                sin(ang)       ) -> (x,y);
	endif;
enddefine;



;;;------------------------------------------------------
;;; LISTS OF WORLD COORDINATES:

define :method go_translistout(coords, obj :go_rotatable ) -> worldcoords;
lvars coords, obj, worldcoords;
;;; REF: go_translistout( LOCALCOORDLIST, ROTATABLE ) -> WORLDCOORDLIST;
;;; REF: Calculates the absolute world coordinates for a given object
;;; REF: This takes account of the current rotation point of the object, the
;;; REF: the extra rotation and extra scaling of the object (not the scaling
;;; REF: of the panes it is in).
;;; REF: LOCALCOORDLIST are the object's local coordinates of a position.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
;;; REF: WORLDCOORDLIST are the world coordinates of a position in the pane.
lvars ang = obj.go_angle mod 360;
	if ang = 0 then
		call_next_method(coords, obj) -> worldcoords;
	else
	lvars x,                            y,
		  xr = obj.stored_go_xrotation, yr = obj.stored_go_yrotation,
		  xo = obj.go_xorigin,          yo = obj.go_yorigin,
		  xs = obj.go_xscale,           ys = obj.go_yscale,
	  cosang = cos(ang),            sinang = sin(ang);
		[%
			until coords == [] do
				fast_destpair( coords ) -> (x, coords);
				fast_destpair( coords ) -> (y, coords);
				go_calc_rotxyout(x, y, xr, yr, xo, yo, xs, ys, cosang, sinang)
				-> (x,y);
				x;     ;;; new x coord
				y;     ;;; new y coord
			enduntil;
		%] -> worldcoords;
	endif;
enddefine;


;;;------------------------------------------------------
;;; LISTS OF LOCAL COORDINATES:

define :method go_translistin(worldcoords, obj :go_rotatable ) -> coords;
lvars worldcoords, obj, coords;
;;; REF: go_translistin( WORLDCOORDLIST, ROTATABLE ) -> LOCALCOORDLIST;
;;; REF: Calculates the absolute world coordinates for a given object
;;; REF: This takes account of the current rotation point of the object, the
;;; REF: the extra rotation and extra scaling of the object (not the scaling
;;; REF: of the panes it is in).
;;; REF: WORLDCOORDLIST are the world coordinates of a position in the pane.
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
;;; REF: LOCALCOORDLIST are the object's local coordinates of a position.
lvars ang = obj.go_angle mod 360;
	if ang = 0 then
		call_next_method(worldcoords, obj) -> coords;
	else
	lvars x,                            y,
		  xr = obj.stored_go_xrotation, yr = obj.stored_go_yrotation,
		  xo = obj.go_xorigin,          yo = obj.go_yorigin,
		  xs = obj.go_xscale,           ys = obj.go_yscale,
	  cosang = cos(ang),            sinang = sin(ang);
		[%
			until worldcoords == [] do
				fast_destpair( worldcoords ) -> (x, worldcoords);
				fast_destpair( worldcoords ) -> (y, worldcoords);
				go_calc_rotxyin(x, y, xr, yr, xo, yo, xs, ys, cosang, sinang)
				-> (x,y);
				x;     ;;; new x coord
				y;     ;;; new y coord
			enduntil;
		%] -> coords;
	endif;
enddefine;


;;;------------------------------------------------------
;;;
	;;; FORMULAE:
	;;; R    === sqrt( (w/2)**2 + (h/2)**2 )
	;;; ang0 === arctan2( w/2, h/2 )
	;;; ang1 === go_angle( obj )
	;;; w0/2 === R * cos( ang0 )
	;;; h0/2 === R * sin( ang0 )
	;;; w1/2 === R * cos( ang0 + ang1 )
	;;;      === R * ( cos(ang0)*cos(ang1) - sin(ang0)*sin(ang1) )
	;;;      === R * ( ((w0/2)/R)*cos(ang1) - ((h0/2)/R)*sin(ang1) )
	;;; w1   === w0*cos(ang1) - h0*sin(ang1)
	;;; h1/2 === R * sin( ang0 + ang1 )
	;;;      === R * ( cos(ang0)*sin(ang1) + sin(ang0)*cos(ang1) )
	;;;      === R * ( ((w0/2)/R)*sin(ang1) + ((h0/2)/R)*cos(ang1) )
	;;; h1   === w0*sin(ang1) + h0*cos(ang1)

define :method go_bounding_width( obj :go_rotatable ) -> w;
lvars obj, w;
;;; REF: go_bounding_width( ROTATABLE ) -> INTEGER;
lvars angle = obj.go_angle mod 360, h;
	call_next_method( obj ) -> w;
	if     angle == 0 then
	elseif angle == 0.0 then
	else
		;;; We need the height but we can't use the method which uses this one.
		;;; This simulates it: Not really safe
		stored_go_bounding_height( obj ) * obj.go_yscale-> h;
		if angle <= 90 then
			round( w*cos(angle) + h*sin(angle) ) -> w;
		elseif angle <= 180 then
			round( w*cos(angle) - h*sin(angle) ) -> w;
		elseif angle <= 270 then
			round( w*cos(angle) + h*sin(angle) ) -> w;
		else
			round( w*cos(angle) - h*sin(angle) ) -> w;
		endif;
	endif;
enddefine;

define :method updaterof go_bounding_width( w, obj :go_rotatable );
lvars obj, h;
;;; REF: INTEGER -> go_bounding_width( PANE, ROTATABLE );
;;; REF: The width of the region occupied by the rotatable screen
;;; REF: object. This is calculated on the basis of the rotation
;;; REF: of the original bounding box over the angle the object holds. This
;;; REF: results in a bounding box which is always too large since it contains
;;; REF: the original box.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
lvars angle = obj.go_angle mod 360, h;
	if     angle == 0 then
	elseif angle == 0.0 then
	else
		;;; We need the height but we can't use the method which uses this one.
		;;; This simulates it: Not really safe
		stored_go_bounding_height( obj ) * obj.go_yscale-> h;
		if angle <= 90 then
		elseif angle <= 180 then
		elseif angle <= 270 then
		else
		endif;
	endif;
	call_next_method( w, obj );
enddefine;


define :method go_bounding_height( obj :go_rotatable ) -> h;
lvars obj, h;
;;; REF: go_bounding_height( PANE, ROTATABLE ) -> INTEGER;
lvars angle = obj.go_angle mod 360, w;
	call_next_method( obj ) -> h;
	if     angle == 0 then
	elseif angle == 0.0 then
	else
		;;; We need the height but we can't use the method which uses this one.
		;;; This simulates it: Not really safe
		stored_go_bounding_width( obj ) * obj.go_xscale-> w;
		if angle <= 90 then
			round( w*sin(angle) + h*cos(angle) ) -> h;
		elseif angle <= 180 then
			round( w*sin(angle) - h*cos(angle) ) -> h;
		elseif angle <= 270 then
			round( w*sin(angle) + h*cos(angle) ) -> h;
		else
			round( w*sin(angle) - h*cos(angle) ) -> h;
		endif;
	endif;
enddefine;

define :method updaterof go_bounding_height( h, obj :go_rotatable );
lvars obj, h;
;;; REF: INTEGER -> go_bounding_height( PANE, ROTATABLE );
;;; REF: The height of the region occupied by the rotatable screen
;;; REF: object. This is calculated on the basis of the rotation
;;; REF: of the original bounding box over the angle the object holds. This
;;; REF: results in a bounding box which is always too large since it contains
;;; REF: the original box.
;;; REF: PANE is an instance of the go_pane class (REF * GO_PANE).
;;; REF: ROTATABLE is an instance of a go_rotatable class (REF * GO_ROTATABLE).
lvars angle = obj.go_angle mod 360;
	if     angle == 0 then
	elseif angle == 0.0 then
	else
		;;; We need the height but we can't use the method which uses this one.
		;;; This simulates it: Not really safe
		if angle <= 90 then
		elseif angle <= 180 then
		elseif angle <= 270 then
		else
		endif;
	endif;
	call_next_method( h, obj );
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_rotatable = true;

/* --- Revision History --------------------------------------------
 * BR 14/07/93
 *     Created this file for convenience for go_arc, go_polygon and go_group
 *     objects.
 */
;;; eof
