/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:			C.all/lib/lr_parser/lib/lr_output.p
 > Purpose:			LALR(1) Parser Generator: saving the parser to a file
 > Author:			Robert John Duncan, Nov 27 1992 (see revisions)
 > Documentation:	REF * LR_PARSER
 > Related Files:
 */

compile_mode:pop11 +strict;

section $-LR =>
	lr_output,
	lr_output_pr,
;

uses lr_core;

;;; =======================================================================

;;; lr_output_pr:
;;;     writes out an item from the parser's symbol table in such a way
;;;     that it can be read back in by the compiler. This default
;;;     version copes with words, strings and integers, but can be
;;;     redefined for anything more complicated.

define global vars lr_output_pr(item);
	lvars item;

	define lconstant pr_char(c);
		lvars c;
		if c == `\n` then
			printf('\\n');
		elseif c fi_< ` ` or c fi_>= `\^?` then
			printf(c, '\\(%p)');
		elseif c == `\\` or c == `\'` then
			printf(c, '\\%c');
		else
			cucharout(c);
		endif;
	enddefine;

	if isstring(item) then
		printf('\''), appdata(item, pr_char), printf('\'');
	elseif isword(item) then
		printf('"\''), appdata(item, pr_char), printf('\'"');
	else
		pr(item);
	endif;
enddefine;

;;; output_symbol:
;;;		writes out a single symbol using lr_output_pr

define lconstant output_symbol(name, symbol);
	lvars name, symbol;
	printf(name, '\tlconstant %p=');
	lr_output_pr(symbol);
	printf(';\n');
enddefine;

;;; output_vector:
;;;		writes out a vector of items whose printing representation is
;;;		generated by the procedure ___gen, packed into 72-column lines

define lconstant output_vector(name, size, gen);
	lvars i, s, n = 999, name, size, procedure gen;
	printf(name, '\tlconstant %p={%%');
	fast_for i to size do
		gen(i) or 'false' -> s;
		n fi_+ datalength(s) -> n;
		if n fi_>= 72 then
			printf('\n\t\t');
			datalength(s) fi_+ 8 -> n;		;;; 8 = 2 tabs
		endif;
		appdata(s, cucharout);
		printf(','), n fi_+ 1 -> n;
	endfor;
	printf('\n\t%%};\n');
enddefine;

;;; output_name:
;;;		writes out the parser name

define lconstant output_name(parser);
	lvars parser;
	output_symbol("name", parser_name(parser));
enddefine;

;;; output_tables:
;;;		writes out the compressed parsing tables

define lconstant output_tables(parser);
	lvars parser, (defaults, actions, gotos) = explode(parser_tables(parser));

		;;; map table entries to integers to maintain sharing
	define lvars rownumber =
		newproperty([], max(16, 2*datalength(actions)), false, "perm");
	enddefine;

		;;; write out each possible reduction as a 3-vector
		;;; and add to the rownumber cache: reductions occur in the
		;;; -defaults- and -actions- tables
	define lconstant output_reductions(tabname);
		lvars i, row, tabname, nrules = 0;

		define lconstant add_reduction(rule);
			lvars rule;
			if isvector(rule) and not(rownumber(rule)) then
				;;; write it out
				printf('\t\t{%p %p %p}\n', [% explode(rule) %]);
				;;; remember it
				nrules fi_+ 1 ->> nrules -> rownumber(rule);
			endif;
		enddefine;

		printf(tabname, '\tlconstant %p={\n');
		appdata(defaults, add_reduction);
		fast_for i to datalength(actions) do
			if fast_subscrv(i, actions) ->> row then
				appdata(row, add_reduction);
			endif;
		endfor;
		;;; special case for "accept"
		printf('\t\taccept\n\t};\n');
		nrules fi_+ 1 -> rownumber("accept");
	enddefine;

		;;; write out all the distinct rows in a table, giving each a
		;;; unique reference in the -rownumber- cache
	define lconstant output_rows(tabname, table);
		lvars i, j, item, row, table, tabname, nrows = 0;
		printf(tabname, '\tlconstant %p={\n');
		fast_for i to datalength(table) do
			if fast_subscrv(i, table) ->> row then
				unless rownumber(row) then
					;;; this row not yet seen: write it out
					printf('\t\t{%%');
					fast_for j to datalength(row) do
						fast_subscrv(j, row) -> item;
						printf(
							if isinteger(item) then
								item, '%p,'
							else
								;;; reduction in an action row
								rownumber(item), 'R(%p),'
							endif
						);
					endfor;
					printf('%%}\n');
					;;; remember it
					nrows fi_+ 1 ->> nrows -> rownumber(row);
				endunless;
			endif;
		endfor;
		printf('\t};\n');
	enddefine;

		;;; write out one of the parser tables, whose entries have been
		;;; previously cached in the table -tabname-.
	define lconstant output_cached_table(name, tabname, table);
		dlvars table, name, tabname;

		define lconstant gen(i) -> s;
			lvars i, s;
			if fast_subscrv(i, table) ->> s then
				sprintf(rownumber(s), tabname, '%p(%p)') -> s;
			endif;
		enddefine;

		output_vector(name, datalength(table), gen);
	enddefine;

	output_reductions("R");
	output_rows("A", actions);
	output_rows("G", gotos);
	output_cached_table("defaults", "R", defaults);
	output_cached_table("actions", "A", actions);
	output_cached_table("gotos", "G", gotos);
	printf('\tlconstant tables =\n');
	printf('\t\t$-LR$-consparser_tables(defaults,actions,gotos);\n');
enddefine;

;;; output_symbols:
;;;		write out the parser's symbol table

define lconstant output_symbols(parser);
	lvars parser, symbols = parser_symbols(parser);

	define lconstant output_symbol_vector(name, vector);
		dlvars name, vector;

		define lconstant gen(i);
			lvars	i;
			dlocal	cucharout = identfn;
			consstring(#| lr_output_pr(fast_subscrv(i, vector)) |#);
		enddefine;

		output_vector(name, datalength(vector), gen);
	enddefine;

	returnunless(symbols)(printf('\tlconstant symbols=false;\n'));
	output_symbol_vector("terminals", parser_terminal_symbols(symbols));
	output_symbol_vector("nonterminals", parser_nonterminal_symbols(symbols));
	output_symbol("start", parser_start_symbol(symbols));
	printf('\tlconstant symbols =\n');
	printf('\t\t$-LR$-consparser_symbols(terminals,nonterminals,start,false);\n');
enddefine;

;;; lr_output:
;;;		writes a program to ______output which will recreate the ______parser and bind
;;;		it to ____word

define global lr_output(decls, word, parser, output);
	lvars	decls, word, parser, output, close_output = false;
	dlocal	cucharout, pop_pr_quotes = false, pr = sys_syspr;

	Checkr_parser(parser) -> parser;
	unless isprocedure(output) then
		not(isdevice(output)) -> close_output;
		discout(output) -> output;
	endunless;

	if decls == [] then
		[vars] -> decls;
	elseunless islist(decls) then
		[^decls] -> decls;
	endif;

	output -> cucharout;

	printf(';;; This declaration was generated by LIB * LR_OUTPUT\n');
	applist(decls, spr);
	printf(word, '%p = $-LR$-consparser(\n');
	printf('lblock;\n');

	output_name(parser);
	output_tables(parser);
	output_symbols(parser);

	printf('\tname,tables,symbols;\n');
	printf('endlblock);\n');

	if close_output then output(termin) endif;
enddefine;

endsection;		/* $-LR */


/* --- Revision History ---------------------------------------------------
--- Robert John Duncan, Jul  1 1993
		No need to force table compression
 */
