/* --- Copyright University of Sussex 1996. All rights reserved. ----------
 > File:            C.vax/src/ass.p
 > Purpose:
 > Author:          John Gibson (see revisions)
 */

;;;----------------- RUN-TIME ASSEMBLER (VAX) --------------------------------

#_INCLUDE 'declare.ph'
#_INCLUDE 'vmdefs.ph'
#_INCLUDE 'external.ph'

global constant
		_popenter, _popuenter, _popuncenter, _checkall,
		_setstklen, _setstklen_diff
	;

global vars
		pop_debugging, _trap
	;

section $-Sys$-Vm;

constant
		procedure (Trans_structure, Get_procedure, Drop_I_code,
		Is_register, Code_pass)
	;

vars
		asm_clist, asm_instr, asm_struct_list,
		_asm_pass, _asm_drop_ptr, _asm_code_offset, _asm_offset_diff,
		_Nlocals, _Nreg, _Npopreg, _Nstkvars, _Npopstkvars
	;

endsection;

;;; ----------------------------------------------------------------------

section $-Sys$-Vm;


lvars		_regmask, _strsize, _big_procedure
	;


lconstant
	_AP		= _16:C,
	_MOVL	= _16:D0,
	_R0		= _16:50,
	_R1		= _16:51,
	_R2		= _16:52,
	;


;;; --- MAPPING REGISTERS TO IDENTIFIERS --------------------------------

	;;; register identifiers
	;;; value is popint reg number shifted left 1, plus bit 0 if pop reg
protected register constant
	pop_reg_A		=  7<<1 || 1,
	pop_reg_B		=  6<<1 || 1,
	nonpop_reg_A	=  8<<1,
	nonpop_reg_B	=  9<<1,
	nonpop_reg_C	= 10<<1,
	nonpop_reg_D	= 11<<1,

	arg_reg_0		=  0<<1,
	arg_reg_1		=  1<<1,
	arg_reg_2		=  2<<1,

	chain_reg		=  4<<1,	;;; CHAIN_REG = r4
	;

	;;; Lists of pop and nonpop registers for local vars, in order of
	;;; allocation. These lists start with a list of registers that should be
	;;; local to every procedure.
constant
	asm_pop_registers	= [%[], ident pop_reg_A, ident pop_reg_B%],
	asm_nonpop_registers= [%[], ident nonpop_reg_A, ident nonpop_reg_B,
								ident nonpop_reg_C, ident nonpop_reg_D%];


define Is_address_reg() with_nargs 1;
	Is_register()			;;; every reg is an address reg
enddefine;


;;; --- CODE-DROPPING PROCEDURES -----------------------------------------

	;;; put a longword in the procedure record
	;;; just increment code offset unless final pass
define Drop_l(_longword);
	lvars _longword;
	unless _asm_pass then
		_longword -> _asm_drop_ptr!(l)++ -> _asm_drop_ptr;
	endunless;
	@@(l){_asm_code_offset}++ -> _asm_code_offset;
enddefine;

	;;; put a word in the procedure record
	;;; just increment code offset unless final pass
define Drop_w(_word);
	lvars _word;
	unless _asm_pass then
		_word -> _asm_drop_ptr!(s)++ -> _asm_drop_ptr;
	endunless;
	@@(s){_asm_code_offset}++ -> _asm_code_offset;
enddefine;

	;;; put a byte in the procedure record
	;;; just increment code offset unless final pass
define Drop_b(_byte);
	lvars _byte;
	unless _asm_pass then
		_byte -> _asm_drop_ptr!(b)++ -> _asm_drop_ptr;
	endunless;
	@@(b){_asm_code_offset}++ -> _asm_code_offset;
enddefine;

	;;; plant an immediate operand
	;;; just increment code offset unless final pass
define Drop_#(_literal);
	lvars _literal;
	unless _asm_pass then
		if _literal _gr _63 then
			;;; immediate literal
			_literal, _16:8F -> _asm_drop_ptr!(b)++ -> !(l)++ -> _asm_drop_ptr;
		else
			;;; short literal
			_literal -> _asm_drop_ptr!(b)++ -> _asm_drop_ptr;
		endif;
	endunless;
	if _literal _gr _63 then
		@@(l){ @@(b){_asm_code_offset}++ }++ -> _asm_code_offset;
	else
		@@(b){_asm_code_offset}++ -> _asm_code_offset;
	endif;
enddefine;

define lconstant Drop_move_#(_num);
	lvars _num;
	if _num _lteq _63 then
		Drop_w(_shift(_num, _8) _add _16:D0)		;;; movl #num<short lit>
	elseif _num _sgr _0 then
		if _num _slteq _16:FF then
			Drop_w(_16:8F9A), Drop_b(_num)			;;; movzbl #num
		elseif _num _slteq _16:FFFF then
			Drop_l(_shift(_num, _16) _add _16:8F3C)	;;; movzwl #num
		else
			Drop_w(_16:8FD0);   					;;; movl #num
			Drop_l(_num)
		endif
	elseif _-16:80 _slteq _num then
		Drop_w(_16:8F98), Drop_b(_num)				;;; cvtbl #num
	elseif _-16:8000 _slteq _num then
		Drop_l(_shift(_num, _16) _add _16:8F32)		;;; cvtwl #num
	else
		Drop_w(_16:8FD0);   						;;; movl #num
		Drop_l(_num)
	endif
enddefine;

	;;; absolute address operand @#<address>
define lconstant Drop_abs() with_nargs 1;
	Drop_b(_16:9F);				;;; mode = @#
	Drop_l(/* address */)
enddefine;

	;;; jsb to absolute address
define lconstant Drop_jsb_abs() with_nargs 1;
	Drop_w(_16:9F16);			;;; jsb @#
	Drop_l(/* address */)
enddefine;

	;;; register offset operand
define lconstant Drop_reg_offs(_offs, _reg, _defer);
	lvars _defer, _offs, _reg;
	if _defer then _16:10 else _0 endif -> _defer;
	if _zero(_offs) and _zero(_defer) then
		Drop_b(_reg _add _16:60)				;;; (rn)
	elseif _-16:80 _slteq _offs and _offs _slteq _16:7F then
		Drop_b(_reg _add _defer _add _16:A0);	;;; {@}byteoffs(rn)
		Drop_b(_offs)
	elseif _-16:8000 _slteq _offs and _offs _slteq _16:7FFF then
		Drop_b(_reg _add _defer _add _16:C0);	;;; {@}wordoffs(rn)
		Drop_w(_offs)
	else
		Drop_b(_reg _add _defer _add _16:E0);	;;; {@}longoffs(rn)
		Drop_l(_offs)
	endif
enddefine;

	;;; plant operand for a structure - relative to structure
	;;; address at start of code if in heap or absolute if not
	;;; Relies on @@ID_VALOF being 0
define lconstant Drop_structure(_opspec, defer);
	lvars argstruct, defer, _opspec;

	;;; replace structure with offset or reg ident on first pass (nonop
	;;; on subsequent passes)
	Trans_structure(asm_instr!INST_ARGS[_opspec])
							->> argstruct -> asm_instr!INST_ARGS[_opspec];

	if issimple(argstruct) then
		_int(argstruct) -> argstruct;		;;; offset as a sysint
		if _neg(argstruct) then
			;;; negated offset for on-stack var, shifted left 1
			;;; if bit 0 is set, access is via a run-time ident, i.e.
			;;; another indirection -- this is disabled by defer being false
			unless argstruct _bitst _1 then false -> defer endunless;
			Drop_reg_offs(_negate(_shift(argstruct, _-1)), _16:E, defer)
		else
			;;; else via argstruct address in table
			;;; calculate offset for word offset to place in table
			argstruct _sub _strsize _sub _asm_code_offset _sub _3 -> _opspec;
			if _big_procedure and _opspec _slt _-16:8000 then
				;;; allow longword offset
				;;; mode = relative deferred / relative
				Drop_b(if defer then _16:FF else _16:EF endif);
				Drop_l(_opspec _sub @@(s)++);
			else
				;;; word offset will be enough
				;;; mode = relative deferred / relative
				Drop_b(if defer then _16:DF else _16:CF endif);
				Drop_w(_opspec);
			endif;
		endif
	else
		;;; absolute or literal
		if defer then
			if (Is_register(argstruct) ->> _opspec) then
				Drop_b(_int(_opspec) _add _R0)
			else
				Drop_abs(argstruct)		;;; mode = absolute
			endif
		else
			Drop_#(argstruct)			;;; literal
		endif
	endif
enddefine;


;;; --- FIELD ACCESS INSTRUCTIONS (used by "conskey") --------------------


lconstant

	field_get_ops	= list_assoc_val(%[
						^t_BIT					16:EF		;;; extzv
						^t_BYTE					16:9A		;;; movzbl
						^t_SHORT				16:3C		;;; movzwl
						^t_WORD					16:D0		;;; movl
						^(t_BIT||t_SIGNED)		16:EE		;;; extv
						^(t_BYTE||t_SIGNED)		16:98		;;; cvtbl
						^(t_SHORT||t_SIGNED)	16:32		;;; cvtwl
						^(t_WORD||t_SIGNED)		16:D0		;;; movl
						]%),

	field_put_ops	= list_assoc_val(%[
						^t_BIT					16:F0		;;; insv
						^t_BYTE					16:F6		;;; cvtlb
						^t_SHORT				16:F7		;;; cvtlw
						^t_WORD					16:D0		;;; movl
						^(t_BIT||t_SIGNED)		16:F0		;;; insv
						^(t_BYTE||t_SIGNED)		16:F6		;;; cvtlb
						^(t_SHORT||t_SIGNED)	16:F7		;;; cvtlw
						^(t_WORD||t_SIGNED)		16:D0		;;; mmovl
						]%),

	field_addr_ops	= list_assoc_val(%[
						^t_BYTE					16:9E		;;; movab
						^t_SHORT				16:3E		;;; movaw
						^t_WORD					16:DE		;;; moval
						^t_DOUBLE				16:7E		;;; movaq
						^(t_BYTE||t_SIGNED)		16:9E		;;; movab
						^(t_SHORT||t_SIGNED)	16:3E		;;; movaw
						^(t_WORD||t_SIGNED)		16:DE		;;; moval
						]%),

	field_shift		= list_assoc_val(% [%
						t_BYTE,			0,
						t_SHORT,		1,
						t_WORD,			2,
						t_DOUBLE,		3,
						%] %),

	;

define lconstant Do_field(type, field_ops, _size, opnd, offset, upd, exptr,
																_other);
	lvars	type, upd, opnd, field_ops, offset, exptr,
			_offs, _base, _size, _opcode, _reg, _subreg, _other, _bsize
		;

	;;; check field args
	unless opnd then
		;;; structure is on top of stack - use r0
		;;; operand on the stack
		if field_ops /== field_addr_ops or exptr or offset /== 0 then
			;;; get operand from stack into r0
			Drop_w(_16:8CD0), Drop_b(_R0);		;;; movl (ap)+, r0
			_0 -> _reg
		else
			;;; pushing address of pop struct at 0 offset -- nothing to do
			return
		endif
	elseif Is_register(Trans_structure(opnd)) ->> _reg then
		_int(_reg) -> _reg
	else
		mishap(0, 'SYSTEM ERROR 1 IN Do_field')
	endunless;

	if exptr then
		;;; deref exptr times
		fast_repeat exptr times
			Drop_b(_MOVL);						;;; movl
			Drop_reg_offs(_0, _reg, false);		;;; (_reg)
			Drop_b(_R0);						;;; r0
			_0 -> _reg
		endrepeat
	endif;

	if isinteger(offset) then
		;;; record field
		_int(offset) -> _offs
	else
		;;; vector field - offset must be register or false
		unless offset then
			;;; pull subscript off stack to r1
			Drop_w(_16:8CD0), Drop_b(_R1);		;;; movl (ap)+, r1
			_1 -> _subreg
		elseif Is_register(Trans_structure(offset)) ->> _subreg then
			_int(_subreg) -> _subreg
		else
			mishap(0, 'SYSTEM ERROR 2 IN Do_field')
		endunless;
		;;; convert subscript to sysint
		Drop_w(_16:8F78), Drop_b(_-2);				;;; ashl #-2
		_subreg _add _R0 -> _offs;					;;; _subreg operand
		Drop_w(_shift(_offs, _8) _add _offs)		;;; _subreg, _subreg
	endif;

	_int(field_ops(type)) -> _opcode;
	_int(_size) -> _size;

	if type fi_&& t_BASE_TYPE == t_BIT then

		;;; bitfield
		unless isinteger(offset) then
			;;; vector field -- _offs is operand for _subreg
			Drop_w(_shift(_offs, _8) _add _16:D7);	;;; decl _subreg
			Drop_w(_shift(_size, _8) _add _16:C4);	;;; mull2 #size
			Drop_b(_offs)							;;; _subreg
		endunless;
		_0 -> _base;

		Drop_b(_opcode);
		if upd then Drop_b(_other) endif; 			;;; _other is source
		if isinteger(offset) then Drop_#(_offs) else Drop_b(_offs) endif;
		Drop_b(_size)

	else

		;;; non-bitfield -- get base offset
		if isinteger(offset) then
			;;; record field -- bit _offs to bytes
			@@(b){_offs|1}
		else
			;;; vector field -- _subreg is reg containing sysint subscript
			_shift(_size, _int(field_shift(type fi_&& t_BASE_TYPE))) -> _bsize;
			if _size /== _1 then
				;;; pushing addr of array of compound external fields
				;;; need to multiply rather than use index scaling
				Drop_b(_16:C4);			;;; mull2
				Drop_#(_bsize);         ;;; #_bsize
				Drop_b(_offs);			;;; _subreg
				_16:9E -> _opcode		;;; movab
			endif;
			;;; base offset is -bytesize for base 1 subscript
			_negate(_bsize)
		endif -> _base;

		Drop_b(_opcode);
		if upd then Drop_b(_other) endif; 				;;; _other is source
		;;; _subreg as index reg for vector field
		unless isinteger(offset) then Drop_b(_subreg _add _16:40) endunless
	endif;

	;;; base reg operand
	Drop_reg_offs(_base, _reg, false);

	unless upd then Drop_b(_other) endunless	;;; _other is dest
enddefine;

define I_PUSH_FIELD();
	lvars type, cvtpint, size, opnd, offset, exptr;
	explode(asm_instr) -> exptr -> cvtpint -> offset -> opnd -> size -> type
																	-> ;

	Do_field(type, field_get_ops, size, opnd, offset, false, exptr,
					if cvtpint then _R0 else _16:7C endif);	;;; r0 or -(ap)
	if cvtpint then
		;;; convert r0 to popint
		Drop_l(_16:50500278);			;;; ashl #2, r0, r0
		Drop_l(_16:7C5003C9)			;;; bisl3 #3, r0, -(ap)
	endif
enddefine;

define I_POP_FIELD();
	lvars type, size, opnd, offset, exptr;
	explode(asm_instr) -> exptr -> offset -> opnd -> size -> type -> ;

	Do_field(type, field_put_ops, size, opnd, offset, true, exptr,
					_16:8C)	;;; (ap)+
enddefine;

define I_PUSH_FIELD_ADDR();
	lvars type, size, opnd, offset, exptr;
	explode(asm_instr) -> exptr -> offset -> opnd -> size -> type -> ;

	;;; stack the field address
	Do_field(type, field_addr_ops, size, opnd, offset, false, exptr,
					_16:7C) ;;; -(ap)
enddefine;


;;; --- ASSEMBLER PROCEDURES FOR INDIVIDUAL INSTRUCTIONS -------------------


	;;; for move instructions
define lconstant Drop_move(defer);
	lvars defer;
	if isinteger(defer) then
		Drop_move_#(asm_instr!INST_ARGS[_0])
	else
		Drop_b(_MOVL);						;;; movl
		Drop_structure(_0, defer)			;;; {@}<arg>
	endif;
	if asm_instr!V_LENGTH /== _2 then
		Drop_structure(_1, true)			;;; @<arg>  (move)
	else
		Drop_b(_16:7C);						;;; -(ap)   (push)
	endif;
enddefine;


;;; --- ROUTINES TO HANDLE EACH INSTRUCTION ------------------------------

define I_MOVE		= Drop_move(%true%) enddefine;
define I_MOVEQ		= Drop_move(%false%) enddefine;
define I_MOVENUM	= Drop_move(%0%) enddefine;
define I_MOVEADDR	= I_MOVENUM enddefine;

define I_MOVES();			;;; move stack, i.e. dup
	Drop_b(_MOVL);			;;; movl
	Drop_w(_16:7C6C)		;;; (ap), -(ap)
enddefine;

define I_MOVE_CALLER_RETURN();
	fast_chain(asm_instr!INST_ARGS[_2])		;;; I_MOVE or I_MOVEADDR
enddefine;

	;;; push an unsigned sysint, represented as a pop (big)integer
define I_PUSH_UINT();
	Drop_move_#(Pint_->_uint(asm_instr!INST_ARGS[_0], _-1));
	Drop_b(_16:7C);						;;; -(ap)
enddefine;

define I_ERASE = Drop_w(%_16:8CD5%) enddefine;		;;; tstl (ap)+

define I_POP();
	Drop_w(_16:8CD0);			;;; movl (ap)+
	Drop_structure(_0, true);	;;; @<arg>
enddefine;

	;;; pop into quoted variable
	;;; only meaningful with an lvar containing a run-time ident
define I_POPQ();
	Drop_w(_16:8CD0);			;;; movl (ap)+
	Drop_structure(_0, false)	;;; <arg>
enddefine;

define I_STORE();
	Drop_w(_16:6CD0);			;;; movl (ap)
	Drop_structure(_0, true)	;;; @<arg>
enddefine;

define I_SWAP();
	lvars _i, _j;
	@@(w)[_int(asm_instr!INST_ARGS[_0])] -> _i;
	@@(w)[_int(asm_instr!INST_ARGS[_1])] -> _j;
	Drop_b(_MOVL);								;;; movl i(ap), r0
	Drop_reg_offs(_i, _AP, false);
	Drop_b(_R0);

	Drop_b(_MOVL);								;;; movl j(ap), i(ap)
	Drop_reg_offs(_j, _AP, false);
	Drop_reg_offs(_i, _AP, false);

	Drop_b(_MOVL);								;;; movl r0, j(ap)
	Drop_b(_R0);
	Drop_reg_offs(_j, _AP, false);
enddefine;


;;; --- FAST FIELD INSTRUCTIONS -----------------------------------------

define lconstant Drop_fsource(_arg);
	lvars instr, _op, _arg;
	dlocal asm_instr;
	if (asm_instr!INST_ARGS[_arg] ->> instr) then
		;;; specified by move instruction in arg field
		instr!INST_OP -> _op;
		if _op == I_MOVENUM then
			Drop_#(instr!INST_ARGS[_0])
		else
			instr -> asm_instr;
			Drop_structure(_0, _op == I_MOVE);	;;; {@}<arg>
		endif;
	else
		;;; from stack
		Drop_b(_16:8C);			;;; (ap)+
	endif;
enddefine;

define lconstant Drop_fdestin();
	lvars instr;
	;;; plant destination operand for fast procedures
	;;; see if can optimise a following pop
	fast_front(fast_back(asm_clist)) -> instr;
	if isvector(instr) and instr!INST_OP == I_POP then
		instr -> asm_instr;
		Drop_structure(_0, true);	;;; @<arg>
		fast_back(asm_clist) -> asm_clist	;;; erase the I_POP
	else
		Drop_b(_16:7C)				;;; -(ap)
	endif
enddefine;

define I_FASTFIELD();
	lvars _offs = asm_instr!INST_ARGS[_0];	;;; false if fast_destpair
	Drop_b(_MOVL);						;;; movl
	unless asm_instr!INST_ARGS[_1] or _offs /== 0 then
		;;; no need to go thru r0
		Drop_b(_16:9C);					;;; @(ap)+
		Drop_fdestin();
		return
	endunless;
	;;; else go thru r0
	Drop_fsource(_1);
	Drop_w(_16:D050);					;;; r0; movl
	if _offs then
		_int(_offs) -> _offs
	else
		;;; fast_destpair
		Drop_reg_offs(@@P_FRONT, _0, false);	;;; P_FRONT(r0)
		Drop_w(_16:D07C);				;;; -(ap); movl
		@@P_BACK -> _offs
	endif;
	Drop_reg_offs(_offs, _0, false);	;;; offs(r0)
	Drop_fdestin()
enddefine;

define I_UFASTFIELD();
	lvars _offs = _int(asm_instr!INST_ARGS[_0]);
	Drop_b(_MOVL);						;;; movl
	Drop_fsource(_1);
	Drop_b(_R0);						;;; r0
	Drop_w(_16:8CD0);					;;; movl (ap)+
	Drop_reg_offs(_offs, _0, false)		;;; offs(r0)
enddefine;

define I_FASTSUBV();
	Drop_b(_16:C1);						;;; addl3
	Drop_fsource(_1);
	Drop_w(_16:508C);					;;; (ap)+, r0
	Drop_b(_MOVL);						;;; movl
	;;; arg 0 is offset to vector elements as popint -- subtract popint 1
	;;; to account for base 1 subscript and popint bits
	Drop_reg_offs(_int(asm_instr!INST_ARGS[_0]) _sub 1, _0, false);
	Drop_fdestin()
enddefine;

define I_UFASTSUBV();
	Drop_b(_16:C1);					;;; addl3
	Drop_fsource(_1);
	Drop_l(_16:8CD0508C);			;;; (ap)+, r0;  movl (ap)+
	;;; arg 0 is offset to vector elements as popint -- subtract popint 1
	;;; to account for base 1 subscript and popint bits
	Drop_reg_offs(_int(asm_instr!INST_ARGS[_0]) _sub 1, _0, false)
enddefine;


;;; --- FAST INTEGER + AND - -------------------------------------------------

define lconstant Drop_fast_+-_arg1(_opcode);
	lvars instr, _opcode;
	;;; addl or subl plus _opcode of _0 or _1 gives 2 or 3 arg version
	if asm_instr!INST_ARGS[_0] then _16:C0 else _16:C2 endif
										_add _opcode -> _opcode;
	if (asm_instr!INST_ARGS[_1] ->> instr) and instr!INST_OP == I_MOVENUM then
		Drop_b(_opcode);				;;; addl?/subl?
		Drop_#(instr!INST_ARGS[_0] _biclear _3)
	else
		Drop_w(_16:03CB);			;;; bicl3 #3
		Drop_fsource(_1);			;;; arg1
		Drop_b(_R0);				;;; r0
		Drop_b(_opcode);			;;; addl?/subl?
		Drop_b(_R0)					;;; r0
	endif;
enddefine;

	;;; {I_FAST_+-_2 <plusflag> <operand 1> <operand 2/destination>}
define I_FAST_+-_2();
	Drop_fast_+-_arg1(_0);			;;; opcode is addl2/subl2
	if asm_instr!INST_ARGS[_2] then
		Drop_fsource(_2)			;;; source/destination
	else
		Drop_b(_16:6C)				;;; (ap)
	endif
enddefine;

	;;; {I_FAST_+-_3 <plusflag> <operand 1> <operand 2> <destination>}
define I_FAST_+-_3();
	Drop_fast_+-_arg1(_1);			;;; opcode is addl3/subl3
	Drop_fsource(_2);				;;; arg2
	if asm_instr!INST_ARGS[_3] then
		Drop_structure(_3, true)	;;; destination
	else
		Drop_b(_16:7C)				;;; -(ap)
	endif
enddefine;


;;; --- CALL INSTRUCTIONS -----------------------------------------------

lconstant macro USER = 0;		;;; indicates userstack is arg

	;;; for call instructions
define lconstant Drop_call(defer, _routine);
	lvars defer, _routine;
	Drop_b(_MOVL);					;;; movl
	if defer == USER then
		Drop_b(_16:8C)				;;; (ap)+
	else
		Drop_structure(_0, defer)	;;; {@}<arg>
	endif;
	Drop_b(_R0);					;;; r0
	if _routine then
		;;; call via check routine
		Drop_jsb_abs(_routine)		;;; jsb @#routine
	else
		;;; call direct
		Drop_b(_16:16);				;;; jsb
		;;;  @PD_EXECUTE(r0) -- relies on PD_EXECUTE field having 0 offset
		Drop_w(_16:00B0)
	endif
enddefine;

define I_CALL	= Drop_call(%true,  _popenter%) enddefine;
define I_CALLQ	= Drop_call(%false, _popenter%) enddefine;
define I_CALLP	= Drop_call(%true,  false%) 	enddefine;
define I_CALLS	= Drop_call(%USER,  _popenter%) enddefine;
define I_CALLPS	= Drop_call(%USER,  false%) 	enddefine;

define I_CALLPQ();
	Drop_b(_MOVL);				;;; movl
	;;; get PD_EXECUTE field - depends on the offset of this field
	;;; being 0 (and the same as ID_VALOF). Can then do jsb (r0)
	;;; instead of jsb @PD_EXECUTE(r0)
	Drop_structure(_0, true);	;;; @<arg> - gets PD_EXECUTE
	Drop_b(_R0);				;;; r0
	Drop_w(_16:6016)			;;; jsb (r0)
enddefine;

define I_CALLABS();				;;; call absolute (i.e. system) procedure
	;;; jsb @#<entry address>
	Drop_jsb_abs(asm_instr!INST_ARGS[_0]!PD_EXECUTE)
enddefine;

	;;; {I_CHAIN_REG <reg ident>}
	;;; chain procedure in reg
define I_CHAIN_REG();
	Drop_b(_16:17);				;;; jmp
	;;;  @PD_EXECUTE(reg) -- relies on PD_EXECUTE field having 0 offset
	Drop_w(_int(Is_register(asm_instr!INST_ARGS[_0])) _add _16:00B0);
enddefine;

	;;; call subroutine
define I_CALLSUB();
	Drop_jsb_abs(asm_instr!INST_ARGS[_0])	;;; jsb @#<entry address>
enddefine;

	;;; chain subroutine
define I_CHAINSUB();
	Drop_w(_16:9F17);				;;; jmp @#
	Drop_l(asm_instr!INST_ARGS[_0])	;;; _routine
enddefine;

	;;; call subroutine or system procedure via reg
define I_CALLSUB_REG();
	lvars _addr, _reg;
	fast_front(asm_instr!INST_ARGS[_0]) -> _addr;
	if Is_register(_addr) ->> _reg then
		Drop_w(_shift(_int(_reg), _8) _add _16:6016)	;;; jsb (Rn)
	else
		Drop_jsb_abs(_addr)							;;; jsb @#<entry address>
	endif
enddefine;


define I_UCALL	= Drop_call(%true,  _popuenter%)	enddefine;
define I_UCALLQ	= Drop_call(%false, _popuenter%)	enddefine;
define I_UCALLP	= Drop_call(%true,  _popuncenter%)	enddefine;
define I_UCALLPQ= Drop_call(%false, _popuncenter%)	enddefine;
define I_UCALLS	= Drop_call(%USER,  _popuenter%)	enddefine;
define I_UCALLPS= Drop_call(%USER,  _popuncenter%)	enddefine;


define I_BRCOND(_ifso, _ifnot, is_so, _broffset, _argnum);
	lvars is_so, _argnum, _br, _pass = _asm_pass, _broffset, _ifso, _ifnot;

	define lconstant Brcond_byte(_ifso, _ifnot, is_so, _broffset, _argnum);
		lvars is_so, _broffset, _ifso, _ifnot, _argnum;
		Drop_b(if is_so then
				  _ifso				;;; opcode for ifso, or etc
			   else
				  _ifnot			;;; opcode for ifnot,and etc
			   endif);
		Drop_b(_broffset _sub _asm_code_offset _sub _1)
	enddefine;

	define lconstant Brcond_long(_ifso, _ifnot, is_so, _broffset, _argnum);
		lvars is_so, _broffset, _ifso, _ifnot, _argnum;
		;;; branch round a jmp, so swop ifso/ifnot opcodes
		Drop_w(if is_so then
				  _16:0600 _add _ifnot			;;; br ifnot +6
			   else
				  _16:0600 _add _ifso			;;; br ifso +6
			   endif);
		Drop_w(_16:EF17);						;;; jmp relative
		Drop_l(_broffset _sub _asm_code_offset _sub _4)
	enddefine;

	_broffset _sub _asm_code_offset -> _br;
	if not(_pass) then
		;;; last pass -- plant default code (branch round a brw)
		Drop_w(if is_so then
				  _16:0300 _add _ifnot			;;; br ifnot +3
			   else
				  _16:0300 _add _ifso			;;; br ifso +3
			   endif);
		Drop_b(_16:31);							;;; brw
		Drop_w(_br _sub _5)
	elseif _pass == 0 then
		;;; first pass -- just increment offset by default code space
		_asm_code_offset _add _5 -> _asm_code_offset
	elseif _pass == 2 then
		;;; extra optimising -- see if branch offset fits in a byte
		_br _sub _2 -> _br;
		if _br _sgr _0 then
			;;; forward -- correct for previous savings
			_br _add _asm_offset_diff -> _br
		endif;
		if _-16:80 _slteq _br and _br _slt _16:80 then
			;;; change branch routine for next pass
			Brcond_byte -> asm_instr!INST_ARGS[_argnum];
			_asm_offset_diff _sub _3 -> _asm_offset_diff;	;;; saved 3 bytes
			_asm_code_offset _add _2 -> _asm_code_offset
		else
			_asm_code_offset _add _5 -> _asm_code_offset
		endif
	else
		;;; extra pass (= 1) for big procedure
		_br _sub _5 -> _br;
		if _br _slt _-16:8000 or _br _sgreq _16:4000 then
			;;; need a br round a relative jmp with longword offset
			;;; change branch routine for next pass
			Brcond_long -> asm_instr!INST_ARGS[_argnum];
			_asm_code_offset _add _8 -> _asm_code_offset
		else
			_asm_code_offset _add _5 -> _asm_code_offset
		endif
	endif
enddefine;

	;;; call the I_BRCOND routine for an I_IF_opt, I_BOOL_opt or I_IF_CMP
	;;; apply the conditional branch routine in arg2
	;;; (initial branch routine is I_BRCOND)
define lconstant Drop_if(instr) with_nargs 3;		;;; ifso/ifnot codes on stack
	lvars instr;
	dlocal asm_instr = instr;
	fast_apply(/* ifso, ifnot */
				instr!INST_ARGS[_1],
				_int(fast_front(instr!INST_ARGS[_0])),
				_2,
				instr!INST_ARGS[_2]);
enddefine;

define I_IF_opt();
	;;; r5 contains the address of false
	Drop_w(_16:55D1);						;;; cmpl r5
	;;; drop input operand or stack pop
	Drop_fsource(_3);
	Drop_if(_16:12, _16:13,	asm_instr)		;;; IFSO/bneq, IFNOT/beql
enddefine;

define I_BOOL_opt();
	lvars instr = asm_instr, opnd;
	;;; if there's an explicit operand, we have to push it
	if (instr!INST_ARGS[_3] ->> opnd) then Drop_I_code(opnd) endif;
	;;; r5 contains the address of false
	Drop_w(_16:55D1);						;;; cmpl r5
	Drop_b(_16:6C);							;;; (ap)
	Drop_if(_16:12, _16:13,	asm_instr);		;;; OR/bneq, AND/beql
	;;; erase the boolean value
	Drop_w(_16:8CD5)						;;; tstl (ap)+
enddefine;


	;;; {I_IF_CMP <_routine> <operand 1> <operand 2> <I_IF_opt instr>}
define I_IF_CMP();
	lvars instr = asm_instr, _type;

	lconstant compare_ops =
		[%	nonop _eq,  16:13, 16:12, nonop _neq,   16:12, 16:13,
			nonop _sgr, 16:19, 16:18, nonop _slteq, 16:18, 16:19,
			nonop _slt, 16:14, 16:15, nonop _sgreq, 16:15, 16:14
		%];

	Drop_b(_16:D1);							;;; cmpl
	Drop_fsource(_1);
	Drop_fsource(_2);
	;;; get ifso/ifnot br opcodes for compare subroutine
	fast_back(lmember(instr!INST_ARGS[_0], compare_ops)) -> _type;
	;;; do the I_IF_opt
	Drop_if(_int(fast_front(_type)),			;;; opcode for SO
			_int(fast_front(fast_back(_type))),	;;; opcode for NOT
			instr!INST_ARGS[_3])
enddefine;

	/*	{I_IF_TAG _________routine _______operand ______________I_IF_opt-instr}
		where _________routine is _issimple or _isinteger
	*/
define I_IF_TAG();
	lvars instr = asm_instr;
	Drop_w(	if instr!INST_ARGS[_0] == _issimple then
				_16:01D3				;;; bitl #1, ...
			else
				_16:02D3				;;; bitl #2, ...
			endif);
	;;; drop input operand or stack pop
	Drop_fsource(_1);
	Drop_if(_16:12, _16:13,	instr!INST_ARGS[_2])	;;; IFSO/bneq, IFNOT/beql
enddefine;


	;;; branch with default (word) offset
define I_BR_std(_broffset, _argnum);
	lvars _broffset, _argnum;
	Drop_b(_16:31);					;;; brw
	Drop_w(_broffset _sub _asm_code_offset _sub _2)
enddefine;

define I_BR(_broffset, _argnum);
	lvars _br, _argnum, _broffset, _pass = _asm_pass;

	define lconstant Br_byte(_broffset, _argnum);
		lvars _broffset, _argnum;
		Drop_b(_16:11);					;;; brb
		Drop_b(_broffset _sub _asm_code_offset _sub _1)
	enddefine;

	define lconstant Br_long(_broffset, _argnum);
		lvars _broffset, _argnum;
		Drop_w(_16:EF17);				;;; jmp relative longword offset
		Drop_l(_broffset _sub _asm_code_offset _sub _4)
	enddefine;

	_broffset _sub _asm_code_offset -> _br;
	if not(_pass) then
		;;; last pass -- drop code for default brw
		Drop_b(_16:31);					;;; brw
		Drop_w(_br _sub _3)
	elseif _pass == 0 then
		;;; first pass -- just increment offset for default brw
		_asm_code_offset _add _3 -> _asm_code_offset
	elseif _pass == 2 then
		;;; extra optimising -- see if br offset fits in a byte
		_br _sub _2 -> _br;
		if _br _sgr _0 then
			;;; forward -- correct for previous savings
			_br _add _asm_offset_diff -> _br
		endif;
		if _zero(_br) then
			;;; don't need it at all
			#_< erasenum(%2%) >_# -> asm_instr!INST_ARGS[_argnum];
			_asm_offset_diff _sub _3 -> _asm_offset_diff	;;; saved 3 bytes
		elseif _-16:80 _slteq _br and _br _slt _16:80 then
			;;; brb will do
			Br_byte -> asm_instr!INST_ARGS[_argnum];
			_asm_offset_diff _sub _1 -> _asm_offset_diff;	;;; saved 1 byte
			_asm_code_offset _add _2 -> _asm_code_offset
		else
			_asm_code_offset _add _3 -> _asm_code_offset
		endif
	else
		;;; extra pass (= 1) for big procedure
		_br _sub _3 -> _br;
		if _br _slt _-16:8000 or _br _sgreq _16:4000 then
			;;; change for next pass
			Br_long -> asm_instr!INST_ARGS[_argnum];
			_asm_code_offset _add _6 -> _asm_code_offset	;;; rel jmp
		else
			_asm_code_offset _add _3 -> _asm_code_offset
		endif
	endif
enddefine;

	;;; {I_SWITCH <label list> <else label> <operand>}
define I_SWITCH();
	lvars lablist, elselab, _laboffset, _base;
	asm_instr!INST_ARGS[_0] -> lablist;
	asm_instr!INST_ARGS[_1] -> elselab;

	if elselab then
		;;; don't need to save arg in r0
		Drop_b(_16:78);					;;; ashl
		Drop_w(_16:FE8F);				;;; #-2
		Drop_fsource(_2);				;;; operand
		Drop_b(_R1)						;;; r1
	else
		Drop_b(_MOVL);					;;; movl
		Drop_fsource(_2);				;;; operand
		Drop_l(_16:FE8F7850);			;;; r0; ashl #-2
		Drop_w(_16:5150)				;;; r0, r1
	endif;

	Drop_w(_16:51CF);					;;; casel r1
	Drop_w(_16:8F01);					;;; #1, #...
	Drop_l(_int(listlength(lablist)-1));	;;; limit
	_asm_code_offset -> _base;			;;; br displacements are from here
	until lablist == [] do
		fast_front(destpair(lablist) -> lablist) -> _laboffset;
		Drop_w(_int(_laboffset) _sub _base)	;;; the br displacement
	enduntil;

	unless elselab then
		;;; no else label, so put arg back on stack
		Drop_b(_MOVL);					;;; movl
		Drop_w(_16:7C50)				;;; r0, -(ap)
	endunless
enddefine;

	;;; {I_PLOG_IFNOT_ATOM <fail_label} I_BRCOND}
define I_PLOG_IFNOT_ATOM();
	lvars instr = asm_instr;
	lconstant _FLAB = _0, _FBR = _1;
	;;; apply the I_BRCOND? routine in arg2
	fast_apply(_16:12, _16:13,				;;; bneq, beql
				true,						;;; select bneq
				_int(fast_front(instr!INST_ARGS[_FLAB])),
				_FBR,
				instr!INST_ARGS[_FBR])
enddefine;

	;;; {I_PLOG_TERM_SWITCH <fail_label} I_BRCOND <var_label> I_BRCOND}
	;;; after call to plog switch routine, switch afterwards to var_label,
	;;; fail_label or drop thru.
define I_PLOG_TERM_SWITCH();
	lvars instr = asm_instr;
	lconstant _FLAB = _0, _FBR = _1, _VLAB = _2, _VBR = _3;
	;;; branch if greater than set to var_label
	fast_apply(_16:14, _16:15,				;;; SO/bgtr, NOT/bleq
				true,
				_int(fast_front(instr!INST_ARGS[_VLAB])),
				_VBR,
				instr!INST_ARGS[_VBR]);
	;;; branch if less than set to fail_label
	fast_apply(_16:19, _16:18,				;;; SO/blss, NOT/bgeq
				true,
				_int(fast_front(instr!INST_ARGS[_FLAB])),
				_FBR,
				instr!INST_ARGS[_FBR]);
	;;; else fall thru for pair/term case (push on arg_reg_0 follows)
enddefine;

define I_CHECK();
	;;; test for interrupt
	Drop_b(_16:E8);					;;; blbs
	Drop_abs(ident _trap);			;;; @#_trap
	Drop_b(_16:09);					;;; +9 (to the jsb @#_checkall)
	;;; test for user stack overflow
	Drop_w(_16:5CD1);				;;; cmpl ap,
	Drop_abs(ident _userlim);		;;; @#_userlim
	Drop_w(_16:061E);				;;; bgequ +6 (round the jsb @#_checkall)
	;;; call to _checkall
	Drop_jsb_abs(_checkall)			;;; jsb @#_checkall
enddefine;

define I_LISP_TRUE();
	;;; r5 contains the address of false
	Drop_l(_16:126C55D1);			;;; cmpl r5, (ap); bnequ
	if nil _gr _63 then
		Drop_b(_16:07);				;;; +7;
		Drop_w(_16:8FD0);			;;; movl #
		Drop_l([])					;;; nil
	else
		;;; nil is near the beginning of memory
		Drop_w(_16:D003);			;;; +3; movl
		Drop_b([])					;;; #nil
	endif;
	Drop_b(_16:6C)					;;; (ap)
enddefine;

define I_STACKLENGTH();
	Drop_w(_16:5CC3);					;;; subl3 ap,
	Drop_abs(ident _userhi);			;;; @#_userhi,
	Drop_l(_16:5003C950);				;;; r0; bisl3 #3, r0
	Drop_fdestin()	 					;;; <destination>
enddefine;


	;;; I_SETSTKLEN    <saved stklen>          <nresults>
	;;; 			   false or MOVE instr     false or MOVENUM instr
	;;; 	see ALISP.S for _setstklen and _setstklen_diff

define I_SETSTACKLENGTH();
	lvars sl, nr;
	asm_instr!INST_ARGS[_0] -> sl;
	asm_instr!INST_ARGS[_1] -> nr;
	if nr then
		Drop_b(_16:C1);                         ;;; addl3
		Drop_#(_int((nr fi_* 4) fi_- 3));       ;;; <adjusted nresults>,
		Drop_fsource(_0);                       ;;; <saved stacklength>,
		Drop_b(_R0);                            ;;; r0;
		Drop_w(_16:50C3);                       ;;; subl3 r0,
		Drop_abs(ident _userhi);                ;;; _userhi,
		Drop_l(_16:5C50D150);					;;; r0;  cmpl r0, ap
		Drop_w(_16:0613);                       ;;; beql +6 (round the jsb)
		Drop_jsb_abs(_setstklen_diff)           ;;; jsb _setstklen_diff
	else
		Drop_jsb_abs(_setstklen)                ;;; jsb _setstklen
	endif
enddefine;


	;;; generate code to unwind stack frame without return
define I_UNWIND_SF();
	lvars _num, _offs;

	;;; remove owner address and on-stack vars
	Drop_b(_16:C0);				;;; addl
	Drop_#(@@(w)[_Nstkvars _add _1]);
	Drop_b(_16:5E);				;;; sp

	;;; restore dynamic locals -- depends on @@ID_VALOF being 0
	for @@(w)[_Nlocals _sub _1] -> _offs step --@@(w){_offs} -> _offs
	till _neg(_offs) then
		Drop_w(_16:8ED0);				;;; movl (sp)+
		if _big_procedure then
			Drop_b(_16:FF);				;;; @<longword offset>
			Drop_l( --@@(w){_offs _sub _strsize _sub _asm_code_offset} );
		else
			Drop_b(_16:DF);				;;; @<word offset>
			Drop_w( --@@(s){_offs _sub _strsize _sub _asm_code_offset} );
		endif;
	endfor;

	;;; restore registers
	unless _zero(_regmask) then
		_6 -> _num;
		until _num == _12 do
			if _regmask _bitst _shift(_1, _num) then
				Drop_w(_16:8ED0);			;;; movl (sp)+
				Drop_b(_num _add _R0);		;;; reg num
			endif;
			_num _add _1 -> _num;
		enduntil;
	endunless;
enddefine;

	;;; return from procedure
define I_RETURN();
	Drop_b(_16:05)					;;; rsb
enddefine;

	;;; generate code to construct stack frame
define I_CREATE_SF();
	lvars _num, _offs, _limit;

	;;; registers
	unless _zero(_regmask) then
		_11 -> _num;				;;; stack registers
		until _num == _5 do
			if _regmask _bitst _shift(_1, _num) then
				Drop_w(_shift(_num, _8) _add _16:50DD);	;;; pushl rn
			endif;
			_num _sub _1 -> _num;
		enduntil;
	endunless;

	;;; dynamic locals -- depends on @@ID_VALOF being 0
	if _Nlocals _gr _2 then
		Drop_w(_16:CFDE);				;;; moval pc rel word offset
		Drop_w( --@@(s)-{_strsize _add _asm_code_offset} );
		Drop_b(_R0);					;;; r0
		for _1 -> _num step _num _add _1 -> _num till _num _gr _Nlocals then
			Drop_w(_16:90DD);			;;; pushl @(r0)+
		endfor;
	else
		@@(w)[_Nlocals] -> _limit;
		for _0 -> _offs step @@(w){_offs}++ -> _offs till _offs == _limit do
			Drop_w(_16:DFDD);			;;; pushl @
			Drop_w( --@@(s){_offs _sub _strsize _sub _asm_code_offset} );
		endfor;
	endif;

	;;; on-stack locals
	for _Npopstkvars -> _num step _num _sub _1 -> _num till _zero(_num) then
		;;; pop ones - must be initialised to pacify the garbage collector
		Drop_w(_16:03DD);				;;; pushl popint 0
	endfor;
	_Nstkvars _sub _Npopstkvars -> _num;
	if _num _sgr _0 then
		;;; nonpop ones - just decrement sp
		Drop_b(_16:C2);					;;; subl2
		Drop_#(@@(w)[_num]);
		Drop_b(_16:5E);					;;; sp
	endif;

	;;; owner address
	Drop_w(_16:CFDF);					;;; pushal
	Drop_w( --@@(s)-{@@PD_TABLE{_strsize _add _asm_code_offset}} );
enddefine;

define I_LABEL();
	;;; set front of label pair to popint offset from code start PD_EXECUTE
	_pint(_asm_code_offset) -> fast_front(asm_clist)
enddefine;


	;;; assembler to produce machine code in procedure record from
	;;; pop assembler input
define Do_consprocedure(codelist, reg_locals) -> pdr;
	lvars pdr, reg, reg_locals, codelist, _code_offset, _size, _reg_spec;

	dlocal _regmask, _asm_drop_ptr, _strsize, _big_procedure, _asm_pass;

	;;; construct reg mask from reg_locals
	_0 -> _regmask;
	for reg in reg_locals do
		_shift(_1, _int(Is_register(reg))) _biset _regmask -> _regmask
	endfor;

	;;; first pass - calculate instruction offsets
	_0 -> _strsize;
	false -> _big_procedure;
	Code_pass(0, codelist) -> _code_offset;
	@@(w)[_int(listlength(asm_struct_list))] -> _strsize;
	_strsize _add _code_offset -> _size;
	if _size _greq _16:8000 then
		;;; big procedure - needs an extra pass to
		;;; cope with longword displacements etc
		true -> _big_procedure;
		Code_pass(1, codelist) -> _code_offset
	endif;
	unless pop_debugging then
		Code_pass(2, codelist) -> _code_offset
	endunless;

	;;; can now calculate total size of procedure and allocate store for it
	@@PD_TABLE{_strsize _add _code_offset | b.r} _sub @@POPBASE -> _size;

	;;; register spec is switch offset to 20-byte code section (see aprocess.s)
	( (_Nreg _sub _Npopreg) _mult #_< _int(length(asm_pop_registers)) >_#
					 _add _Npopreg) _mult _20 -> _reg_spec;

	;;; get the procedure record with header filled in and struct table
	;;; generated (sets up _asm_drop_ptr)
	Get_procedure(_size, _reg_spec) -> pdr;

	;;; final pass - generate code
	Code_pass(false, codelist) -> 			;;; generate code
enddefine;

endsection;		/* $-Sys$-Vm */



/* --- Revision History ---------------------------------------------------
--- John Gibson, Sep 27 1996
		Added I_IF_TAG.
--- John Gibson, Nov 27 1990
		Optimised I_PUSH_FIELD_ADDR to save code with 0 offset.
--- John Gibson, Sep 29 1990
		Changed value given to -Get_procedure- for PD_REGMASK to be
		switch offset for code in aprocess.s
--- John Gibson, Aug 18 1990
		-size- argument to field instructions now in units of type size;
		changed -Do_field- to detect size/==1 case for subscripting
		external array of structures/arrays etc.
--- John Gibson, Mar 21 1990
		Changed field access procedures so that when -exptr- argument
		is true, it's an integer specifying deref first that many times.
--- John Gibson, Dec 12 1989
		Changes for new pop pointers, plus tidying up some code
--- John Gibson, Nov 29 1989
		Reg r5 now contains address of false (no longer assumed 0) --
		made appropriate changes.
--- John Gibson, Jun 30 1989
		Added I_CALLPS and I_UCALLPS (for -fast_apply-)
--- John Gibson, Jun  4 1989
		Replaced pop_optimise with not(pop_debugging)
--- John Gibson, Apr 30 1989
		Put into section $-Sys$-Vm.
--- John Gibson, Dec 11 1988
		Fixed bug in -Drop_move- (was taking instruction to be a 2-operand
		move when asm_instr!V_LENGTH == _3 instead of
		asm_instr!V_LENGTH /== _2, which meant that it didn't work with
		I_MOVE_CALLER_RETURN where V_LENGTH is _4).
--- Roger Evans, Oct 10 1988
		Modified Do_field to accept -offset- <false> to
		mean vector subscript on stack, and have a new arg -ext- meaning
		plant code for external field access (ie indirect on E_PTR field
		and do 3rd longword correction on offset). I_PUSH_FIELD,
		I_PUSH_FIELD_ADDR and I_POP_FIELD take this -ext- arg an pass it on
--- John Gibson, Jul 27 1988
		Added I_MOVE_CALLER_RETURN
--- John Gibson, Jun  7 1988
		Introduced -I_LABEL- (called by -Code_pass- directly)
--- John Gibson, May 22 1988
		Fields in I_PUSH/POP_FIELD specifying operands now contain lexical
		identifier tokens (which get allocated to registers) rather than
		register identifiers directly (thus -Do_field- now calls
		-Trans_structure- on them).
--- John Gibson, Mar 27 1988
		-list_assoc_val- into section Sys
--- John Gibson, Feb 10 1988
		Pint_->_uint now in section Sys
 */
