/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_labelled.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 2nd May 1993
;;; File: go_labelled.p
;;; Author: Ben L.E. Rabau

compile_mode :pop11 +strict;

;;; This file is based on the original go_labelled.p of GO-alpha1
;;; but has additional facilities to deal with go_offsets in the
;;; labels as requested by Ian Rogers (Univ of Sussex).

;;; INTRO: The MIXIN go_labelled can be inherited to attach a label to an
;;; INTRO: object. The label will have the same positions the original
;;; INTRO: object, unless the go_label_x_offset()/go_label_y_offset() are
;;; INTRO: used to displace it.
;;; INTRO: The dimensions of the label is only taken into account if the
;;; INTRO: inheritance precedes all other classes or mixins:
;;; INTRO:     e.g. CORRECT: isa go_labelled go_rectangle;
;;; INTRO:          WRONG  : isa go_rectangle go_labelled;
;;; INTRO:

uses go_xdefaults;

;;;------------------------------------------------------
;;; LABELLED MIXIN:

define :mixin go_labelled;
	slot stored_go_label          =  'Label';
;;; REF: The object's string used in the label.
	slot stored_go_label_font     == false;
;;; REF: The XFontStruct used for this object if not the default from the
;;; REF: pane. The applications default XFont can be retrieved from:
;;; REF: REF * GO_XDEFAULTS/go_default_font
	slot stored_go_label_x_offset == 0;
;;; REF: The horizontal displacement in world coordinates from the left
;;; REF: of the bounding box. The bounding box of the label itself will
;;; REF: also be extended to the right but this can be clipped by the other
;;; REF: inherited classes.
	slot stored_go_label_y_offset == 0;
;;; REF: The vertical displacement in world coordinates from the bottom
;;; REF: of the bounding box. The bounding box of the label itself will
;;; REF: also be extended to the top but this can be clipped by the other
;;; REF: inherited classes.
enddefine;

;;;------------------------------------------------------------------
;;; LABEL

define :method go_label( lab :go_labelled );
lvars lab;
;;; REF: go_label( LABELLED ) -> STRING;
;;; REF: Gets the text of the go_labelled object.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	lab.stored_go_label;
enddefine;

define :method updaterof go_label( string, lab :go_labelled );
lvars string, lab;
;;; REF: STRING -> go_label( LABELLED );
;;; REF: Sets the new text of the go_labelled object.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	go_clear( lab );
	string -> lab.stored_go_label;
	go_internal_redraw( lab );
enddefine;

;;;------------------------------------------------------------------
;;; FONT

define :method go_label_font( lab :go_labelled ) -> the_font;
lvars lab, the_font = lab.stored_go_label_font;
;;; REF: go_label_font( LABELLED ) -> XFONT;
;;; REF: Gets the font of the go_labelled object
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	unless the_font then
		if go_visible_in(lab) == [] then
			;;; Make sure it has a font if it was never visualised up to now
			go_default_font ->> lab.stored_go_label_font -> the_font;
		else
			go_font( go_visible_in(lab)(1) )
			->> lab.stored_go_label_font -> the_font;
		endif;
	endunless;
enddefine;

define :method updaterof go_label_font( font, lab :go_labelled );
lvars font, lab;
;;; REF: XFONT -> go_label_font( LABELLED );
;;; REF: Sets a new font of the go_labelled object
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	go_clear( lab );
	font -> lab.stored_go_label_font;
	go_internal_redraw( lab );
enddefine;

;;;------------------------------------------------------------------
;;; OFFSET

define :method go_label_x_offset( lab :go_labelled );
lvars lab;
;;; REF: go_label_x_offset( LABELLED ) -> INTEGER;
;;; REF: Gets the horizontal offset on either side of the go_labelled object.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	lab.stored_go_label_x_offset;
enddefine;

define :method updaterof go_label_x_offset( x_offs, lab :go_labelled );
lvars x_offs, lab;
;;; REF: INTEGER -> go_label_x_offset( LABELLED );
;;; REF: Sets a new horizontal offset on either side of the go_labelled object.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	go_clear( lab );
	x_offs -> lab.stored_go_label_x_offset;
	go_internal_redraw( lab );
enddefine;

define :method go_label_y_offset( lab :go_labelled );
lvars lab;
;;; REF: go_label_y_offset( LABELLED ) -> INTEGER;
;;; REF: Gets the vertical offset on either side of the go_labelled object.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	lab.stored_go_label_y_offset;
enddefine;

define :method updaterof go_label_y_offset( y_offs, lab :go_labelled );
lvars y_offs, lab;
;;; REF: INTEGER -> go_label_y_offset( LABELLED );
;;; REF: Sets a new vertical offset on either side of the go_labelled object .
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	go_clear( lab );
	y_offs -> lab.stored_go_label_y_offset;
	go_internal_redraw( lab );
enddefine;

;;;------------------------------------------------------------------

define :method go_label_text_width( st :go_labelled ) -> w;
lvars st, w = 10;
;;; REF: go_label_text_width( LABELLED ) -> INTEGER;
;;; REF: Calculates how wide the text itself is in world coordinates.
;;; REF: If a font is given in go_label_font( label ) then the font is used
;;; REF: Otherwise the font is derived from the first pane it is shown in or
;;; REF: from the application's default (see also ~/.Xdefaults file) if not
;;; REF: shown. It returns at least 10 (even if string is empty) to avoid
;;; REF: losing the item on the screen...
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
lvars the_font = go_label_font( st );
	if the_font then
		max( go_font_width( the_font, go_label(st) ), 10 ) -> w;
	endif;
enddefine;

define :method go_label_text_height( st :go_labelled ) -> h;
lvars st, h = 10;
;;; REF: go_label_text_height( LABELLED ) -> INTEGER;
;;; REF: Calculates how heigh the text itself is in world coordinates.
;;; REF: If a font is given in go_label_font( label ) then the font is used
;;; REF: Otherwise the font is derived from the first pane it is shown in or
;;; REF: from the application's default (see also ~/.Xdefaults file) if not
;;; REF: shown.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
lvars the_font = go_label_font( st );
	if the_font then
		go_font_height( the_font ) -> h;
	endif;
enddefine;

define :method go_label_width( st :go_labelled );
lvars st;
;;; REF: go_label_width( LABELLED ) -> INTEGER;
;;; REF: Calculates how wide the text and offset are in world coordinates.
;;; REF: see go_label_text_width() for calculation of the width of the text
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	go_label_text_width(st) + abs(st.go_label_x_offset) * 2
enddefine;

define :method go_label_height( st :go_labelled );
lvars st;
;;; REF: go_label_height( LABELLED ) -> INTEGER;
;;; REF: Calculates how heigh the text and offset are in world coordinates.
;;; REF: see go_label_text_height() for calculation of the height of the text
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	go_label_text_height(st) + abs(st.go_label_y_offset) * 2
enddefine;

;;;------------------------------------------------------------------
;;; DRAW

define :method go_draw_label( pane :go_pane, lab :go_labelled );
lvars pane, lab;
;;; REF: go_draw_label( PANE, LABELLED );
;;; REF: Draws the go_labelled object on the given pane (used by go_fgdraw).
;;; REF: This takes axes orientation and label-offsets into account. If
;;; REF: The label does not fit in the width of the bounding box then the
;;; REF: label will be clipped on whole characters at the end of the string or
;;; REF: if the offset is negative clipped on whole characters at the beginning
;;; REF: of the string (or both if necessary). If the height does't fit in
;;; REF: the bounding box, then NO label is printed (the user is advised to
;;; REF: use a smaller font (see REF * GO_LABELLED/go_label_font).
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
lvars xc = lab.go_xcentre,          yc = lab.go_ycentre,
	  w0 = abs(lab.go_bounding_width  * pane.go_xscale),
	  h0 = abs(lab.go_bounding_height * pane.go_yscale),
	  w1 = lab.go_label_text_width, h1 = lab.go_label_text_height,
	  dx = lab.go_label_x_offset,   dy = lab.go_label_y_offset,
	  xscale = pane.go_xscale * lab.go_xscale,
	  yscale = pane.go_yscale * lab.go_yscale,
	  max_w0 = abs(w1/xscale),      max_h0 = abs(h1/yscale),
	  max_dx = (w0 - max_w0),       max_dy = (h0 - max_h0), /* div 2,  WRONG ? */
	  the_label = lab.go_label,     the_font = lab.go_label_font,
	  length_substr, x1, y1;

	;;; WARNING: THIS NEEDS CLIPPING OR FONT SCALING RATHER THAN NO-OP!!!
	if (max_dy < 0) or (max_dy < dy) or (dy < 0) then
		;;; WARNING: THIS NEEDS CLIPPING OR FONT SCALING RATHER THAN NO-OP!!!
		printf('Font height does not fit for object: %p, with label %p\n',
			   [% datakey(lab), go_label(lab) %]);
	else
		if (max_dx >= 0) and (dx >= 0) then
			min(max_dx, dx) -> dx;
		else
			;;; MANUAL TRUNCATION RATHER THAN CLIPPING
			length( the_label )         -> length_substr;
			;;; CLIP THE RIGHT HAND SIDE: END OF STRING:
			while ( go_font_width( the_font,  the_label ); + dx > w0 ) do
				length_substr - 1 -> length_substr;
				if (length_substr <= 0) then return; endif;
				substring(1, length_substr, the_label) -> the_label;
			endwhile;
			;;; CLIP THE LEFT HAND SIDE: BEGINNING OF STRING:
			if (dx < 0) then
				go_font_width( the_font,  the_label ); + dx -> w0;
				while ( go_font_width( the_font,  the_label ); > w0 ) do
					length_substr - 1 -> length_substr;
					if (length_substr <= 0) then return; endif;
					substring(2, length_substr, the_label) -> the_label;
				endwhile;
				0 -> dx;
			endif;
		endif;
		xc + (dx - w0 div 2)*sign(xscale) -> x1;
		yc - (dy - h0 div 2)*sign(yscale) -> y1;
		go_transxyout( x1, y1, pane ) -> (x1, y1);
		go_draw_text( pane, x1, y1, the_label, the_font );
	endif;
enddefine;


define :method go_fgdraw( pane :go_pane, lab :go_labelled );
lvars pane, lab;
;;; REF: go_fgdraw( PANE, LABELLED );
;;; REF: Draws the go_labelled object and the label on the given pane.
;;; REF: The label is always on top of the go_labelled object.
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
	call_next_method( pane, lab );
	go_draw_label( pane, lab );
enddefine;

;;;-------------------------------------------------------------------
;;; Add characters (or delete ...)
include x_keysyms.ph;

define :method go_label_editing( ch, st :go_labelled );
lvars ch, st;
;;; REF: go_label_editing( char, LABELLED );
;;; REF: Handles the given character input on the label of the given
;;; REF: go_labelled object. All recognisable input is seen as editing the
;;; REF: label of the object. No support for multi-lines is foreseen!
;;; REF: All unrecognised keys (like function keys or modifiers) produce
;;; REF: warning messages on the tty. Only uses keys from the X-interface.
;;; REF: See also REF * XT_LIBS and INCLUDE * X_KEYSYMS
;;; REF: LABELLED is an instance of the go_labelled mixin (REF * GO_LABBELED).
lvars txt = go_label(st);
	if ( IsKeypadKey(ch) ) then
		printf('Keypad key %p cannot be handled in a go_label\n', [% ch %]);
	elseif ( IsCursorKey(ch) ) then
		printf('Cursor key %p cannot be handled in a go_label\n', [% ch %]);
	elseif ( IsPFKey(ch) ) then
		printf('PF key %p cannot be handled in a go_label\n', [% ch %]);
	elseif ( IsFunctionKey(ch) ) then
		printf('Function key %p cannot be handled in a go_label\n', [% ch %]);
	elseif ( IsMiscFunctionKey(ch) ) then
		printf('Miscellaneous Function key %p cannot be handled in a go_label\n',
			   [% ch %]);
	elseif ( IsModifierKey(ch) ) then
		printf('Modifier key %p is not handled separately in a go_label\n',
			   [% ch %]);
	elseif ( ch == XK_BackSpace ) or ( ch == XK_Delete ) then
		unless (txt = '') then
			substring( 1, (length( txt ) - 1 ), txt ) -> go_label(st);
		endunless;
	elseif ( ch == XK_Tab ) then
		txt sys_>< '\t' -> go_label(st);
	elseif ( ch == XK_Linefeed ) or ( ch == XK_Return ) then
		printf('Cannot put a linefeed or a return character in a go_label\n');
	elseif ( ch == XK_Clear ) then
		'' -> go_label(st);
	elseif ( ch == XK_Escape ) then
		txt sys_>< '\^' -> go_label(st);
	else
		;;; printf('Add %p to the go_label %p\n', [% ch, txt %]);
		txt sys_>< consstring(ch,1) -> go_label(st);
	endif;
enddefine;


;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_labelled = true;

/* --- Revision History --------------------------------------------
 * BR 02/06/93
 *     Elaborate cheching and truncating added to avoid too long or too
 *     high labels if the bounding box is too small.
 *     Corrected bug in go_label_editing for substring if already empty...
 * BR 18/05/93
 *     Added support for fonts (see also go_xdraw.p).
 * BR 14/05/93
 *     Moved the go_draw_label() method from go_labelled into go_text (this
 *     used to limit the label to the first half of the bounding_box).
 * BR 07/05/93
 *     Changed go_transxyout() to no longer include the screen object (see
 *     LIB * GO_PANE)
 * BR 30/04/93
 *     Moved go_label class and go_text_input class into separate files.
 * BR 20/04/93
 *     Optimized go_draw_label() and added a go_make_invisible() method to
 *     take changes in default-font hadling possible (removed go_default_pane).
 * BR 16/04/93
 *     Added go_clear()/go_internal_redraw() for updaterof go_bounding_width()
 * BR 16/04/93
 *     Removed direct call to XpwDrawString() by go_draw_text()
 *     Removed direct call to XpwTextWidth() by go_text_width()
 *     Removed direct call to XpwFontHeight() by go_text_height()
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * IR 09/03/93
 *    Added go_label_text_{height,width}()
 * BR 05/02/93
 *     Added go_offsets (active on both sides of the go_label) as requested
 *     by Ian Rogers from the Univ. of Sussex. This file is currently
 *     considered as an alternative version of the one in the GO library.
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 16/11/92
 *     Removed the label_offset: (see groups for go_offsets)
 * BR 05/11/92
 *     Added the CLASS go_label:
 * BR 09/09/92
 *     Added the label_offsets to the go_label_width/go_label_height() methods
 * BR 21/08/92
 *     Reverted the changes made because of call_next_method (4.01)
 *     problems with width/height methods !!! therefore changed to
 *     go_label_width and go_label_height.
 * BR 13/08/92
 *     added XpwTextHeight/Width code as already defined in edit_text
 * BR 23/07/92
 *     made include file from basic objects: go_labelled,
 *     previously defined in sketch_basics.p
 */
;;; eof
