/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_located.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													   ;;; 3rd June 1993
;;; File: go_located.p
;;; Author: J L Cunningham

compile_mode :pop11 +strict;

;;; INTRO: The MIXIN go_located is one of the base classes in the Graphical
;;; INTRO: Objects library, and introduces the concept of the "bounding box".
;;; INTRO: Each object has an enclosing rectangle parrallel with the axes.
;;; INTRO: This bounding box makes a lot of geometrical operations easier by
;;; INTRO: offering a bounding width and height and uniform reference points
;;; INTRO: like a corner and the centre for each object however complex an
;;; INTRO: irregular. Additional scaling around the centre of the bounding box
;;; INTRO: completes this class. See also scaling of panes in which the
;;; INTRO: objects will be visualized (see REF * GO_PANE).

define :mixin go_located;
	slot stored_go_xorigin         == 0;
;;; REF: Stored horizontal origin of the object's local coordinate system.
	slot stored_go_yorigin         == 0;
;;; REF: Stored vertical origin of the object's local coordinate system.
	slot stored_go_bounding_width  == 1;
;;; REF: Stored Width in local coordinates of bounding box.
	slot stored_go_bounding_height == 1;
;;; REF: Stored Height in local coordinates of bounding box.
	slot stored_go_xscale          == 1;
;;; REF: Stored X scale of object (horizontal compression).
	slot stored_go_yscale          == 1;
;;; REF: Stored Y scale of object (vertical compression).
	slot stored_go_local_coords     == [];
;;; REF: X and Y-coordinates from go_xorigin and from go_yorigin
enddefine;

;;;------------------------------------------------------------------
;;; WORLD VS. LOCAL COORDS
;;; ---------------------------------------------------------
;;; ORIGIN:

define :method go_xorigin( obj :go_located ) -> x;
lvars obj , x = stored_go_xorigin( obj );
;;; REF: go_xorigin( SCREEN_OBJECT ) -> INTEGER;
enddefine;

define :method updaterof go_xorigin( x, obj :go_located );
lvars x, obj;
;;; REF: INTEGER -> go_xorigin( SCREEN_OBJECT );
;;; REF: The horizontal origin of the local coordinates of the object.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
	x -> stored_go_xorigin( obj );
enddefine;

define :method go_yorigin( obj :go_located ) -> y;
lvars obj , y = stored_go_yorigin( obj );
;;; REF: go_yorigin( SCREEN_OBJECT ) -> INTEGER;
enddefine;

define :method updaterof go_yorigin( y, obj :go_located );
lvars y, obj;
;;; REF: INTEGER -> go_yorigin( SCREEN_OBJECT );
;;; REF: The vertical origin of the local coordinates of the object.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
	y -> stored_go_yorigin( obj );
enddefine;


define :method go_origin_to( x, y, i :go_located );
lvars x, y, i;
;;; REF: go_origin_to( X, Y, LOCATED );
;;; REF: Sets both horizontal and vertical origin of a located object.
;;; REF: This is a convenience method using go_xorigin and go_yorigin.
;;; REF: X and Y are world coordinates of the new origin position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	x -> go_xorigin( i );
	y -> go_yorigin( i );
enddefine;

;;; ---------------------------------------------------------
;;; CENTRE:
;;; For non-rotatable objects the centre is also the origin for
;;; rotatable objects it is the rotated point of the origin.

vars procedure go_transxyout;     ;;; DEFINED BELOW

define :method go_xcentre( obj :go_located ) -> x;
lvars obj , x;
;;; REF: go_xcentre( SCREEN_OBJECT ) -> INTEGER;
	 go_transxyout( 0, 0, obj ) -> (x,);
enddefine;

define :method updaterof go_xcentre( new_xcentre, obj :go_located );
lvars new_xcentre, obj;
;;; REF: INTEGER -> go_xcentre( SCREEN_OBJECT );
;;; REF: The horizontal centre of the local coordinates of the object.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
	(new_xcentre - obj.go_xcentre) + obj.go_xorigin -> obj.go_xorigin;
enddefine;

define :method go_ycentre( obj :go_located ) -> y;
lvars obj , y;
;;; REF: go_ycentre( SCREEN_OBJECT ) -> INTEGER;
	 go_transxyout( 0, 0, obj ) -> (,y);
enddefine;

define :method updaterof go_ycentre( new_ycentre, obj :go_located );
lvars new_ycentre, obj;
;;; REF: INTEGER -> go_ycentre( SCREEN_OBJECT );
;;; REF: The vertical centre of the local coordinates of the object.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
	(new_ycentre - obj.go_ycentre) + obj.go_yorigin -> obj.go_yorigin;
enddefine;


define :method go_centre_to( x, y, i :go_located );
lvars x, y, i;
;;; REF: go_centre_to( X, Y, LOCATED );
;;; REF: Sets both horizontal and vertical centre of a located object.
;;; REF: This is a convenience method using go_xcentre and go_ycentre.
;;; REF: X and Y are world coordinates of the new centre position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	x -> go_xcentre( i );
	y -> go_ycentre( i );
enddefine;

;;;----------------------------------------------------------------
;;; CONVENIENCE FUNCTION FOR EDITING FACILITIES

define :method go_position_of_origin( locobj :go_located ) -> (x, y);
lvars x = locobj.go_xorigin, y = locobj.go_yorigin, locobj;
;;; REF: go_position_of_origin( LOCATED ) -> (X, Y);
enddefine;

define :method updaterof go_position_of_origin( x, y, locobj :go_located );
lvars x, y, locobj;
;;; REF: go_position_of_origin( X, Y, LOCATED );
;;; REF: The world coordinates of the origin point of the local coordinates
;;; REF: of a located object. This is a convenience function using the
;;; REF: go_xorigin and go_yorigin methods.
;;; REF: X and Y are the world coordinates of the origin point of the object.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	x -> locobj.go_xorigin;
	y -> locobj.go_yorigin;
enddefine;


define :method go_position_of_centre( locobj :go_located ) -> (x, y);
lvars x = locobj.go_xcentre, y = locobj.go_ycentre, locobj;
;;; REF: go_position_of_centre( LOCATED ) -> (X, Y);
enddefine;

define :method updaterof go_position_of_centre( x, y, locobj :go_located );
lvars x, y, locobj;
;;; REF: go_position_of_centre( X, Y, LOCATED );
;;; REF: The world coordinates of the centre point of the bounding box
;;; REF: of a located object. This is a convenience function using the
;;; REF: go_xcentre and go_ycentre methods.
;;; REF: X and Y are the world coordinates of the centre point of the object.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	x -> locobj.go_xcentre;
	y -> locobj.go_ycentre;
enddefine;

;;;------------------------------------------------------------------
;;; SCALING
;;; Scaling should always happen around the centre?!

define :method go_xscale( i :go_located );
lvars i;
;;; REF: go_xscale( LOCATED ) -> REAL;
	i.stored_go_xscale;
enddefine;

define :method updaterof go_xscale( scale, i :go_located );
lvars scale, i;
;;; REF: REAL -> go_xscale( LOCATED );
;;; REF: The horizontal scaling factor of a located object. This will
;;; REF: preserve the position of the centre of the bounding box. Care is
;;; REF: taken that the value does not become ZERO.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	if (scale > 0) then max( 0.001, scale*1.0) -> i.stored_go_xscale;
				   else min(-0.001, scale*1.0) -> i.stored_go_xscale;
	endif;
enddefine;

define :method go_yscale( i :go_located );
lvars i;
;;; REF: go_yscale( LOCATED ) -> REAL;
	i.stored_go_yscale;
enddefine;

define :method updaterof go_yscale( scale, i :go_located );
lvars scale, i;
;;; REF: REAL -> go_yscale( LOCATED );
;;; REF: The vertical scaling factor of a located object. This will
;;; REF: preserve the position of the centre of the bounding box. Care is
;;; REF: taken that the value does not become ZERO.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	if (scale > 0) then max( 0.001, scale) -> i.stored_go_yscale;
				   else min(-0.001, scale) -> i.stored_go_yscale;
	endif;
enddefine;


define :method go_scale( i :go_located ) -> scale;
lvars scale, i;
;;; REF: go_scale( LOCATED ) -> REAL;
lvars xscale = i.go_xscale, yscale = i.go_yscale;
	if xscale = yscale then xscale -> scale;
	else mishap('Scale: X and Y scale differ for the object', [^i]);
	endif;
enddefine;


define :method updaterof go_scale( scale, i :go_located );
lvars scale, i;
;;; REF: REAL -> go_scale( LOCATED );
;;; REF: Sets both horizontal and vertical scaling factor of a located object.
;;; REF: This is a convenience method using go_xscale and go_yscale.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	scale -> go_xscale( i );
	scale -> go_yscale( i );
enddefine;


vars go_bounding_width, go_bounding_height;		;;; Defined later...

;;;------------------------------------------------------------------
;;; LOCATION OF BOUNDING BOX
;;; BR go_xloc, go_yloc changed to methods due to limitation of objectclass
;;; on inheritance of updaters on go_regular slots.

define :method go_xloc( i :go_located );
lvars i;
;;; REF: go_xloc( LOCATED ) -> INTEGER;
	i.go_xcentre - (i.go_bounding_width  div 2);
enddefine;

define :method updaterof go_xloc( x, i :go_located );
lvars x, i;
;;; REF: INTEGER -> go_xloc( LOCATED );
;;; REF: The horizontal world coordinate of the reference corner of
;;; REF: the bounding box of a located object. In a positive orientation of
;;; REF: the axes, the reference point is the bottom-left corner.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	x + (i.go_bounding_width  div 2) -> i.go_xcentre;
enddefine;

define :method go_yloc( i :go_located );
lvars i;
;;; REF: go_yloc( LOCATED ) -> INTEGER;
	i.go_ycentre - (i.go_bounding_height  div 2);
enddefine;

define :method updaterof go_yloc( y, i :go_located );
lvars y, i;
;;; REF: INTEGER -> go_yloc( LOCATED );
;;; REF: The vertical world coordinate of the reference corner of
;;; REF: the bounding box of a located object. In a positive orientation of
;;; REF: the axes, the reference point is the bottom-left corner.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	y + (i.go_bounding_height div 2) -> i.go_ycentre;
enddefine;


define :method go_move_to( x, y, i :go_located );
lvars x, y, i;
;;; REF: go_move_to( X, Y, LOCATED );
;;; REF: Sets both horizontal and vertical reference of a located object.
;;; REF: This is a convenience method using go_xloc and go_yloc.
;;; REF: X and Y are world coordinates of the new reference position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	if x then x -> go_xloc( i ); endif;
	if y then y -> go_yloc( i ); endif;
enddefine;

;;;------------------------------------------------------------------
;;; DIMENSIONS OF BOUNDING BOX
;;; BR 2 go_bounding_width, go_bounding_height added as methods due to limitation of objectclass
;;; on inheritance of updaters on go_regular slots.

define :method go_bounding_width( i :go_located );
lvars i;
;;; REF: go_bounding_width( LOCATED ) -> INTEGER;
	round(i.stored_go_bounding_width * i.go_xscale);
enddefine;

define :method updaterof go_bounding_width( new_w, i :go_located );
lvars new_w, i;
;;; REF: INTEGER -> go_bounding_width( LOCATED );
;;; REF: The width in world coordinates of the bounding box of a
;;; REF: located object. This influences the scaling (see go_xscale).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
lvars old_w;
	i.stored_go_bounding_width -> old_w;
	if (new_w = old_w) then 1.0 -> i.go_xscale; return; endif;
	if (abs(new_w) < 1)  do
		1*sign(new_w) -> new_w;    ;;; Minimum value is +/- ONE
	else
		round(new_w) -> new_w;
	endif;
	( new_w / old_w ) * 1.0 -> i.go_xscale;
enddefine;

define :method go_bounding_height( i :go_located );
lvars i;
;;; REF: go_bounding_height( LOCATED ) -> INTEGER;
	round(i.stored_go_bounding_height * i.go_yscale);
enddefine;

define :method updaterof go_bounding_height( new_h, i :go_located );
lvars new_h, i;
;;; REF: INTEGER -> go_bounding_height( LOCATED );
;;; REF: The height in world coordinates of the bounding box of a
;;; REF: located object. This influences the scaling (see go_xscale).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
lvars old_h;
	i.stored_go_bounding_height -> old_h;
	if (new_h = old_h) then 1.0 -> i.go_yscale; return; endif;
	if (abs(new_h) < 1)  do
		1*sign(new_h) -> new_h;    ;;; Minimum value is +/- ONE
	else
		round(new_h) -> new_h;
	endif;
	( new_h / old_h ) * 1.0 -> i.go_yscale;
enddefine;

;;;------------------------------------------------------------------
;;; BOUNDING BOX

define :method go_bounding_box( locObj :go_located ) -> (x, y, w, h);
lvars locObj, x, y, w = locObj.go_bounding_width, h = locObj.go_bounding_height;
;;; REF: go_bounding_box( LOCATED ) -> (X, Y, W, H);
;;; REF: Gets the dimensions in world coordinates of the bounding box of a
;;; REF: located object. This uses the go_xcentre, go_ycentre,
;;; REF: go_bounding_width and go_bounding_height methods.
;;; REF: X and Y are the world coordinates of the object's reference point.
;;; REF: W and H are sizes in world coordinates of the object's bounding box.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	locObj.go_xcentre - (w div 2) -> x;
	locObj.go_ycentre - (h div 2) -> y;
enddefine;


;;;------------------------------------------------------
;;; LOCAL HELP FUNCTIONS

define go_calc_transxyout(x, y, xc, yc, xscale, yscale) -> (x,y);
lvars x, y, xc, yc, xscale, yscale;
	;;; Do the transformation
	round(if xscale == 1 then x else x * xscale endif + xc) -> x;
	round(if yscale == 1 then y else y * yscale endif + yc) -> y;
enddefine;

define go_calc_transxyin(x, y, xc, yc, xscale, yscale) -> (x,y);
lvars x, y, xc, yc, xscale, yscale;
	;;; Do the reverse transformation
	;;; deal with most common case first
	;;; use separate "elseif" clauses since "or" isn't optimised (AS 93/06/13)
	if     xscale ==  1  then x  - xc
	elseif xscale =  1.0 then x  - xc
	elseif xscale == -1  then xc - x
	elseif xscale = -1.0 then xc - x
	else                               ;;; should check for risk of overflow?
		((x - xc) * 1.0 / xscale)
	endif -> x;

	if     yscale ==  1  then y  - yc
	elseif yscale =  1.0 then y  - yc
	elseif yscale == -1  then yc - y
	elseif yscale = -1.0 then yc - y
	else                               ;;; should check for risk of overflow?
		((y - yc) * 1.0 / yscale)
	endif -> y;
enddefine;

;;;------------------------------------------------------
;;; COORDINATE TRANSFORMATION

;;; go_transxyout is user definable. Make it erase(%1%) to have no effect
;;; Takes two numbers (user coords) and returns two numbers - coords.
;;; Must leave numbers on stack in the same order

define :method go_transxyout(x, y, obj :go_located ) -> (x,y);
lvars x, y, obj;
;;; REF: go_transxyout( LX, LY, SCREEN_OBJECT ) -> ( X, Y );
;;; REF: Calculation of the transformed coordinates corresponding to
;;; REF: a given position in the objects local coordinates.
;;; REF: LX and LY are the object's local coordinates of a position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
;;; REF: X and Y are the transformed coordinates of the position.
	go_calc_transxyout(x, y,
					   obj.stored_go_xorigin, obj.stored_go_yorigin,
					   obj.stored_go_xscale,  obj.stored_go_yscale) -> (x,y);
enddefine;

;;; go_transxyin is user definable. Make it == erase to have no effect
;;; (Previous line corrected by AS 13/06/93)
;;; Must leave numbers on stack in same order
;;; WARNING: will produce reals in some cases...
;;; Warning. Must be replaced by rc_rotate_xyin if lib rc_rotate_xy used.

define :method go_transxyin(x, y, obj :go_located) -> (x, y);
lvars x, y, obj;
;;; REF: go_transxyin( X, Y, SCREEN_OBJECT ) -> ( LX, LY );
;;; REF: Calculation of the object's local coordinates corresponding to a given
;;; REF: position in previously transformed coordinates.
;;; REF: X and Y are the previously transformed coordinates of the position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
;;; REF: LX and LY are the object's local coordinates of a position.
	go_calc_transxyin(x, y,
					  obj.stored_go_xorigin, obj.stored_go_yorigin,
					  obj.stored_go_xscale,  obj.stored_go_yscale) -> (x,y);
enddefine;



;;;------------------------------------------------------
;;; LISTS OF COORDINATES:

define :method go_translistout(coords, obj :go_located ) -> transcoords;
lvars coords, obj, transcoords;
;;; REF: go_translistout( LOCALCOORDLIST, SCREEN_OBJECT ) -> TRANSCOORDLIST;
;;; REF: Calculates the transformed coordinates for a given object
;;; REF: This takes account of the current centre of the object, the
;;; REF: the extra rotation and extra scaling of the object (not the scaling
;;; REF: of the panes it is in).
;;; REF: LOCALCOORDLIST are the object's local coordinates of a position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
;;; REF: TRANSCOORDLIST are the transformed coordinates of a position in the pane.
lvars x,                          y,
	  xc = obj.stored_go_xorigin, yc = obj.stored_go_yorigin,
	  xs = obj.stored_go_xscale,  ys = obj.stored_go_yscale;
	[%
		until coords == [] do
			fast_destpair( coords ) -> coords -> x;
			fast_destpair( coords ) -> coords -> y;
			go_calc_transxyout(x, y, xc, yc, xs, ys) -> (x,y);
			x;     ;;; new x coord
			y;     ;;; new y coord
		enduntil;
	%] -> transcoords;
enddefine;


;;;------------------------------------------------------
;;; LISTS OF LOCAL COORDINATES:

define :method go_translistin(transcoords, obj :go_located ) -> coords;
lvars transcoords, obj, coords;
;;; REF: go_translistin( TRANSCOORDLIST, SCREEN_OBJECT ) -> LOCALCOORDLIST;
;;; REF: Calculates the previously transformed coordinates for a given object
;;; REF: This takes account of the current centre of the object, the
;;; REF: the extra rotation and extra scaling of the object (not the scaling
;;; REF: of the panes it is in).
;;; REF: TRANSCOORDLIST are the transformed coordinates of a position in the pane.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED).
;;; REF: LOCALCOORDLIST are the object's local coordinates of a position.
lvars x,                          y,
	  xc = obj.stored_go_xorigin, yc = obj.stored_go_yorigin,
	  xs = obj.stored_go_xscale,  ys = obj.stored_go_yscale;
	[%
		until transcoords == [] do
			fast_destpair( transcoords ) -> transcoords -> x;
			fast_destpair( transcoords ) -> transcoords -> y;
			go_calc_transxyin(x, y, xc, yc, xs, ys) -> (x,y);
			x;     ;;; new x local coord
			y;     ;;; new y local coord
		enduntil;
	%] -> coords;
enddefine;


;;;------------------------------------------------------------------
;;; LOCAL COORDINATES

define :method go_local_coords( i :go_located ) -> offset_list;
lvars i, offset_list = stored_go_local_coords( i );
;;; REF: go_local_coords( LOCATED ) -> LIST;
enddefine;

define :method updaterof go_local_coords( offset_list, i :go_located );
lvars offset_list, i;
;;; REF: LIST -> go_local_coords( LOCATED );
;;; REF: LIST is the list of horizontal and vertical integer coordinates of
;;; REF: points in the object's local coordinate system.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	if offset_list.islist and (offset_list.listlength mod 2) == 0 then
		offset_list -> stored_go_local_coords( i );
	else
		mishap('Local coordinates need to be entered as a paired list');
	endif;
enddefine;

;;; --------------------------------------------------------------------
;;; HELP FUNCTION:

define :method go_translate_local_coords( dx, dy, i :go_located ) -> newcoords;
lvars dx, dy, i, newcoords;
;;; REF: go_translistin( DX, DY, SCREEN_OBJECT ) -> NEWLOCALCOORDLIST;
;;; REF: Calculates the local coordinates for a given object after translating
;;; REF: over a given distance.
;;; REF: DX and DY are the horizontal and vertical offset in local coordinates.
;;; REF: NEWLOCALCOORDLIST are the object's local coordinates after translation.
lvars local_coords = go_local_coords( i ), x, y;
	[% until local_coords = [] do
			fast_destpair( local_coords ) -> (x, local_coords);
			fast_destpair( local_coords ) -> (y, local_coords);
			x + dx; y +dy;
	   enduntil; %] -> newcoords;
enddefine;

;;; ---------------------------------------------------------
;;; WORLD COORDINATES

define :method go_world_coords( i :go_located ) -> list;
lvars i, list = go_translistout(i.go_local_coords, i);
;;; REF: go_world_coords( LOCATED ) -> WORLDCOORDLIST;
enddefine;

define :method updaterof go_world_coords( list, i :go_located );
lvars i, list;
;;; REF: WORLDCOORDLIST -> go_world_coords( LOCATED );
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
;;; REF: WORLDCOORDLIST is a list of world coordinates of the groups offsets.
	go_translistin(list, i) -> i.go_local_coords;
enddefine;



;;; ----------------------------------------------------------------
;;; CONTAINS (position in rectangular go_area of bounding-box)
define :method go_contains(mx, my, rect :go_located ) /* -> obj_or_false */ ;
lvars mx, my, rect;
;;; REF: go_contains( X, Y, LOCATED ) -> LOCATED_OR_FALSE;
;;; REF: Checks whether the world coordinates of a test position fall inside
;;; REF: the bounding box of a located object. This is a good default method
;;; REF: for testing whether a position falls inside an object. If not
;;; REF: inside then the method returns the boolean false, else it returns
;;; REF: the object itself.
;;; REF: X and Y are the world coordinates of the test position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
lvars x, y, w, h;
	go_bounding_box( rect ) -> (x, y, w, h);
	if w < 0 then -w -> w; x - w -> x; endif;
	if h < 0 then -h -> h; y - h -> y; endif;
	if(      x <= mx
		and mx <= x + w
		and  y <= my
		and my <= y + h
	  ) then rect else false endif;
enddefine;


;;;------------------------------------------------------------------
;;; OVERLAPS (the rectangular areas of bounding-boxes)

define :method go_overlaps_bounding_boxes( rec1:go_located, rec2:go_located );
lvars rec1, rect2;
;;; REF: go_overlaps_bounding_boxes( LOCATED1, LOCATED2 ) -> BOOLEAN;
;;; REF: Checks whether the bounding boxes of two located objects overlap or
;;; REF: not. This is a good default method for testing whether two objects
;;; REF: overlap. It returns the boolean true if it overlaps, else it is false.
;;; REF: LOCATED1 and LOCATED2 are go_located instances (see REF * GO_LOCATED)
lvars x11, x12, y11, y12, x21, x22, y21, y22;
	rec1.go_xloc -> x11;
	x11 + rec1.go_bounding_width -> x12;
	rec1.go_yloc -> y11;
	y11 + rec1.go_bounding_height -> y12;
	rec2.go_xloc -> x21;
	x21 + rec2.go_bounding_width -> x22;
	rec2.go_yloc -> y21;
	y21 + rec2.go_bounding_height -> y22;
		not( min(x21,x22) > max(x11,x12) or min(x11,x12) > max(x21,x22) )
	and not( min(y21,y22) > max(y11,y12) or min(y11,y12) > max(y21,y22) )
enddefine;

define :method go_overlaps( rec1:go_located, rec2:go_located );
lvars rec1, rect2;
;;; REF: go_overlaps( LOCATED1, LOCATED2 ) -> BOOLEAN;
;;; REF: This uses the go_overlaps_bounding_boxes method to check whether two
;;; REF: objects overlap. It checks whether the bounding boxes of two located
;;; REF: objects overlap or not. It returns the boolean true if it overlaps,
;;; REF: else it is false.
;;; REF: LOCATED1 and LOCATED2 are go_located instances (see REF * GO_LOCATED)
	go_overlaps_bounding_boxes( rec1, rec2 );
enddefine;

;;;----------------------------------------------------------------
;;; Variable for "uses"
vars go_located = true;

/* --- Revision History --------------------------------------------
 * BR 03/06/93
 *     Removed all references to stored_go_[x/y]loc and just calculates if
 *     needed. This saves trying to keep them in sync with the centre.
 * BR 17/05/93
 *     Made go_[x/y]centre the main methods and made go_[x/y]loc dependent!
 * BR 06/05/93
 *     Changed go_bounding_[width/height] to influence the scaling so
 *     that this method works for all objects!
 * BR 06/05/93
 *     Adapted the xscale/yscale for new default value = 1;
 *     Added go_scale() method.
 *     Added comments.
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 03/11/92
 *     Changed go_contains to return the object rather than true or false...
 * BR 16/10/92
 *     Added argument to go_draw() to indicate pane (removed rc_window)...
 *     Added the same argument to go_safe_region() and moved this to the
 *     go_screen_object class
 * BR 05/10/92
 *     Resolved conflicts between places where go_draw was used and defined
 *     for the file go_located.p.
 *     Defined new objectclass AREA to optimise usage go_rectangle (smaller).
 *     Added default go_overlaps.
 *     Removed update_position(), go_class_attributes() and be_near() (see
 *     MajorChanges).
 * BR 21/09/92
 *     added go_contains wich was previously defined in go_rectangular.p
 * BR 04/09/92
 *     replaced go_drag_and_drop() method to use the LIB * RC_DRAG library (counts
 *     on the new version of go_internal_redraw() defined in the go_screen_object class
 *     (see go_objects.p).
 *     added go_safe_region (previously defined in go_polygon.p).
 * BR 20/08/92
 *     added general go_bounding_width/go_bounding_height methods (accessing stored_go_bounding_width/go_bounding_height)
 *     added go_bounding_box() and go_draw_bounding_box() methods (from stroke)
 * BR 20/07/92
 *     made include file from basic object: go_located, previously defined
 *     in sketch_basics.p
 */
;;; eof
