 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; Pop11 mode and inferior pop11 mode for gnu-emacs.                ;;
 ;;                                                                  ;;
 ;;         Written by Richard Caley and Brian Logan.                ;;
 ;;                                                                  ;;
 ;;         Flavours information by Angela Marie Gilham.             ;;
 ;;                                                                  ;;
 ;; Problems, gripes, bribes etc should be sent to                   ;;
 ;; rjc@cstr.ed.ac.uk.                                               ;;
 ;;                                                                  ;;
 ;; This code is based on emacs c-mode and prolog-mode and so is     ;;
 ;; distributed under the terms of the gnu general public licence.   ;;
 ;; Basically this means you can do whatever you want with it        ;;
 ;; except ship a compiled version without this source code. You     ;;
 ;; should have received a copy of the gnus general public           ;;
 ;; licence with this code in the file COPYING.                      ;;
 ;;                                                                  ;;
 ;; This code was written to allow pop11 development to be done in   ;;
 ;; emacs with (almost) as much support as it has in Ved.            ;;
 ;; Specifically it allows code to be loaded from an emacs buffer    ;;
 ;; into a running pop11 and it allows poplog documentation to be    ;;
 ;; read from emacs (see pop-sys-file-mode.el).                      ;;
 ;;                                                                  ;;
 ;; There are two major limitations which arise from assumptions     ;;
 ;; which emacs makes about comments:                                ;;
 ;;                                                                  ;;
 ;;         First, since emacs will only recognise 2 characters as   ;;
 ;;         a start of comment token, pop-mode treats `;;' as the    ;;
 ;;         pop11 start of  comment character. I have never seen     ;;
 ;;         that in pop11 code anywhere other than as a comment      ;;
 ;;         start, but you never know...                             ;;
 ;;                                                                  ;;
 ;;         Second, and more significantly, the code will only       ;;
 ;;         understand either PL-I style /*...*/ comments or ';;;'   ;;
 ;;         end of line coments not both. In general this is not a   ;;
 ;;         problem. most people only use one or the other inside    ;;
 ;;         procedures. Set the variable pop-use-pl1-comments as     ;;
 ;;         apropriate. I have it off since I use PL-I style         ;;
 ;;         comments only as block comments at file level and        ;;
 ;;         end-of-line comments inside my code.                     ;;
 ;;                                                                  ;;
 ;; The indentation code and pop-closeit cope with most pop-11       ;;
 ;; constructs I have come across. Basically they cover my coding    ;;
 ;; style and that used in code I have to maintain. Please extend    ;;
 ;; the lists of keywords and indentation data to cover more         ;;
 ;; syntax.  Please send any such extensions to me and I will put    ;;
 ;; them in any new versions.                                        ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; THINGS TO DO:                                                    ;;
 ;;         Add epoch support.                                       ;;
 ;;                                                                  ;;
 ;;         Make it understand all of pop syntax (probably           ;;
 ;;         impossible on a turing machine :-().                     ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; revised 17/07/89 bsl
;;; Bugs: the revisions seem to have broken indent-newline-indent
;;;       something seems to have stepped on pop-cache indent-newline-indent
;;;		  with pop-cache-valid nil works (?)
;;; Note: the new 'shell-prompt-patterns' have not been tested with shell
;;;       mode

;;; rjc 20/7/89
;;; fixed bug with indenting? Caused by pop-syntax table being set up
;;; incorrectly 
;;;
;;; Noticed (another - sigh) bug in handling of declarations.
;;; Since an lvars at the begining of a line is recognised as a start of
;;; definition by beginning-of-define it will not indent a line of this
;;; form even if it appears inside a define ... enddefine. Could remove
;;; vars and lvars from pop-definition-starters, but it is nice to be able
;;; to go to the start of vars definitions at top level.
;;;
;;; made '%' a self-matching bracket in the hope of removing the hack in
;;; calculate-pop-indent, but parse-partial-sexp seems to not note
;;; them as starting expressions, and so does not do the indentation.
;;; this is 18.44 on edai.

;;; 28th Nov 89, RJC
;;; Changed to ^C<whatever> compilation commands.

;;; 5th Dec 89 rjc
;;; added pop-closit. Slight bug when closing nested defines - it 
;;;                      may find the wrong name.          
;;; I think it understands comments now - added pop-skip-over-whitespace 
;;;                      which skips comments. It's horrible.
;;; Made it build regular expressions rather than search lists. Don't know
;;;                      if it is any faster.
;;; added the help file reading stuff.

;;; 14th Feb 1990 rjc
;;; Fixed pop-closit. Seems to work in all cases now, even inside 
;;;			lists.
;;; changed the indentation code to use an association list
;;;			to find out how to indent things.
;;; moved the rest of the code over to regular expressions.
;;;
;;; added showlib.

;;; 29th Jan 1991 rjc
;;; Added Flavours definitions.
;;; Added pop-forward-word and pop-backward-word.
;;; Added pop-complete-word.

;;; 11th Feb 1991 rjc
;;; Changed movement commands so that they are more like the
;;; 			lisp equivalents.
;;; Added assignments for generic movement procedures.

;;; 24th Feb 1991 rjc
;;; Moved the inferior pop mode into a separate file.
;;; Made pop-complete-word list the completions if the
;;;			start is ambiguous.

;;; 27th Feb 1991 bsl
;;; added autoloading for pop-teach, pop-showlib, plog-help, plog-doc and 
;;; plog-ref  (why not simply 'require this stuff?)
;;; added default key bindings for pop-indent-definition and pop-indent-region
;;; renamed indent-newline-indent -> pop-newline-indent
;;;         semicolon-indent -> pop-semicolon-indent
;;;         pop-move|back-over-structure -> pop-forward|backward-expression

;;; 2nd April 1991 rjc
;;; Allowed for `procedure' after `define' not needing and `endprocedure'
;;; Bad use of interactive in pop-indent-region.

;;; 18th April 1991 rjc
;;; Fixed pop-syntax-table. The bug is too embarassing to explain here... 

(provide 'pop-mode)

(autoload 'pop-help "pop-sys-file-mode" "Read poplog help file" t )
(autoload 'pop-teach "pop-sys-file-mode" "Read poplog teach file" t )
(autoload 'pop-doc "pop-sys-file-mode" "Read poplog doc file" t )
(autoload 'pop-ref "pop-sys-file-mode" "Read poplog ref file" t )
(autoload 'pop-showlib "pop-sys-file-mode" "Read poplog lib file" t )
(autoload 'pop-next-help "pop-sys-file-mode" "Go to next help reference" t)
(autoload 'pop-get-help "pop-sys-file-mode" "Read the help file for the current word" t)
(autoload 'pop-ploghelp "pop-sys-file-mode" "Read poplog prolog help file" t )
(autoload 'pop-plogteach "pop-sys-file-mode" "Read poplog prolog teach file" t)
(autoload 'pop-plogshowlib "pop-sys-file-mode" "Read poplog prolog lib file" t)

(autoload 'run-pop "inferior-pop-mode" "Run a pop11 process" t)
(autoload 'pop-compile-region "inferior-pop-mode" 
	  "Send a region to a pop process" t)
(autoload 'pop-compile-procedure "inferior-pop-mode" 
	  "Send current procedure to a pop process" t)
(autoload 'pop-compile-buffer "inferior-pop-mode" 
	  "Send current buffer to a pop process" t)

;;; add pop mode to auto mode list if necessary

(if (assoc "\\.p" auto-mode-alist)
	nil
  (setq auto-mode-alist (nconc auto-mode-alist '(("\\.p" . pop-mode)))))

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; The following variables are meant for YOU gentle reader.         ;;
 ;;                                                                  ;;
 ;; These define pop11 as I see it, pop11 is, however a fuzzy        ;;
 ;; beast, so if you have problems feel free to hack them around.    ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar pop-use-pl1-comments nil
	"*pop11 mode is set up to expect /* */ comments if this is true, end
of line comments otherwise" )

(defvar pop-mode-hook nil
	"*Run when entering pop11 mode" )

(defvar pop-label-regexp
	"\\(\\sw\\|\\s_\\)+\\s-*:\\s-*"
	"*regular expression matching pop11 labels")

(defvar pop-declaration-indent-step 5
	"* Distance to indent lines following a declaration (eg \"vars\" )" )

(defvar pop-label-indent 4
	"* Distance that labels are outdented relative to the
surrounding code" )

(defvar pop-indentation-info
	'(
	  ( "define" 0 4 )
	  ( "enddefine" -4 0 )
	  ( "procedure" 0 4 )
	  ( "endprocedure" -4 0 )

	  ( "flavour" 0 4 )
	  ( "endflavour" -4 0 )
	  ( "defmethod" 0 4 )
	  ( "enddefmethod" -4 0 )

	  ( "if" 0 8 )
	  ( "unless" 0 8 )
	  ( "then" 0 -4 )
	  ( "elseif" -4 8 )
	  ( "elseunless" -4 8 )
	  ( "else" -4 4 )
	  ( "endif" -4 0 )
	  ( "endunless" -4 0 )

	  ( "while" 0 8 )
	  ( "endwhile" -4 0 )

	  ( "for" 0 8 )
	  ( "fast_for" 0 8 )
	  ( "do" 0 -4 )
	  ( "endfor" -4 0 )

	  ( "until" 0 8 )
	  ( "enduntil" -4 0 )

	  ( "repeat" 0 8 )
	  ( "times" 0 -4 )
	  ( "forever" 0 -4 )
	  ( "endrepeat" -4 0 )

	  ( "switchon" 0 8 )
	  ( "case" -4 8 )
	  ( "notcase" -4 8 )
	  ( "andcase" -4 4 )
	  ( "notandcase" -4 4 )
	  ( "orcase" -4 4 )
	  ( "notorcase" -4 4 )
	  ( "endswitchon" -8 0 )
	  ) 
	"* An association list which determines how pop11 structures are
to be indented.  Each entry starts with a string giving the name of a
syntax word. This is followed by two numbers, the first gives the 
change in indentation of the current line while the second gives the
change in indentation of succeeding lines.

Thus the default entry for \"else\" is ( \"else\" -4 4 ) This outdents
the else, to line up with the if, and indents the following code." )

(defvar pop-declaration-starters
	'(
	  "recordclass"
	  "vectorclass"
	  "vars"
	  "lvars"
	  "section"
	  "constant"
	  "lconstant"
	  )
	"* List of words which start declarations in pop11. All
words which start constructs terminated by a semicolon ( eg vars )")

(defvar pop-declaration-modifiers
	  '(
	    "global"
	    "procedure"
	    "constant"
	    "lconstant"
	    "lvars"
	    "syntax"
	    "macro"
	    "updaterof"
	    )
	  "* list of words which modify declarations.")

(defvar pop-interesting-declaration-modifiers
	  '(
	    "updaterof"
	    "macro"
	    "syntax"
	    ) 
	  "* list of words which modify declarations which
are interesting enough to be included in the comment which closes it." )

(defvar pop-definition-starters
	'(
	  "define"
	  "flavour"
	  "defmethod"
	  "recordclass"
	  "vectorclass"
	  "section"
	  "vars"
	  "lvars"
	  "constant"
	  "lconstant"
	  )
	"* List of words which start definitions in pop11. Used to
find the start of a definition ")

(defvar pop-definition-enders
	'(
	  "enddefine"
	  "endsection"
	  "endflavour"
	  "enddefmethod"
	  )
	"* List of words used to end definitions in pop11" )

(defvar pop-open-close-assoc
	'(
	  ( "define" . "enddefine" )
	  ( "flavour" . "endflavour" )
	  ( "defmethod" . "enddefmethod" )
	  ( "procedure" . "endprocedure" )
	  ( "if" . "endif" )
	  ( "unless" . "endunless" )
	  ( "while" . "endwhile" )
	  ( "until" . "enduntil" )
	  ( "fast_for" . "endfor" )
	  ( "for" . "endfor" )
	  ( "repeat" . "endrepeat" )
	  ( "switchon" . "endswitchon" )
	  )
	"* Association list of open-close pairs. Used to scan
over structures." )

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; Now for pop-mode's personal variables. Many of these are         ;;
 ;; compiled form the above lists when pop-mode is run.              ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar pop-syntax-regexp nil )

(defvar pop-syntax-table nil
	"Syntax table for pop11" )

(define-abbrev-table 'pop-mode-abbrev-table ())

(defvar pop-mode-abbrev-table nil
  "Abbrev tabe in use in pop-mode buffers.")

(defvar pop-mode-map (make-sparse-keymap)
	"Key Map for pop11 mode" )

(defvar pop-declaration-starters-regexp nil
	"Regular expression matching all declaration starting words." )

(defvar pop-declaration-modifiers-regexp nil
	"Regexp matching declaration modifiers" )

(defvar pop-interesting-declaration-modifiers-regexp nil
	"Regular expression matching interesting declaration modifiers" )

(defvar pop-definition-start-regexp nil
       "Regular expression matching the start of a ( top level )
definition." )

(defvar pop-definition-starters-regexp nil
	"Regular expression matching the start of a definition" )

(defvar pop-definition-enders-regexp nil
	"Regular expression matching the end of a definition" )

(defvar pop-closer-regexp nil
	"regular expression matching only end of structure
syntax words")

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; Set up symbol table. Note that it needs to know which kind of    ;;
 ;; comment to use                                                   ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun pop-syntax-table ()
  (if (not pop-syntax-table)
	  (progn
	      (setq pop-syntax-table (make-syntax-table))
	      (set-syntax-table pop-syntax-table)

	      (modify-syntax-entry ?\\ "\\" pop-syntax-table)
	      (modify-syntax-entry ?/ "_" pop-syntax-table)
	      (modify-syntax-entry ?* "_" pop-syntax-table)
	      (modify-syntax-entry 59 "_" pop-syntax-table) ; ?;
	      (modify-syntax-entry ?\" "\"" pop-syntax-table)
	      (modify-syntax-entry ?' "\"" pop-syntax-table)
	      (modify-syntax-entry ?` "\"" pop-syntax-table)
	      (modify-syntax-entry ?: "_" pop-syntax-table)
	      (modify-syntax-entry ?_ "_" pop-syntax-table)
	      (modify-syntax-entry ?! "_" pop-syntax-table)
	      (modify-syntax-entry ?# "_" pop-syntax-table)
	      (modify-syntax-entry ?$ "_" pop-syntax-table)
	      (modify-syntax-entry ?& "_" pop-syntax-table)
	      (modify-syntax-entry ?= "_" pop-syntax-table)
	      (modify-syntax-entry ?- "_" pop-syntax-table)
	      (modify-syntax-entry ?~ "_" pop-syntax-table)
	      (modify-syntax-entry ?^ "_" pop-syntax-table)
	      (modify-syntax-entry 124 "_" pop-syntax-table) ; ?|
	      (modify-syntax-entry ?@ "_" pop-syntax-table)
	      (modify-syntax-entry ?+ "_" pop-syntax-table)
	      (modify-syntax-entry ?< "_" pop-syntax-table)
	      (modify-syntax-entry ?> "_" pop-syntax-table)
	      (modify-syntax-entry ?? "_" pop-syntax-table)
	      (modify-syntax-entry ?% "$%" pop-syntax-table)
	      (modify-syntax-entry ?, "_" pop-syntax-table)
	      (modify-syntax-entry ?. "_" pop-syntax-table)
	      (modify-syntax-entry ?\( "()" pop-syntax-table)
	      (modify-syntax-entry ?\) ")(" pop-syntax-table)
	      (modify-syntax-entry ?\[ "(]" pop-syntax-table)
	      (modify-syntax-entry ?\] ")[" pop-syntax-table)
	      (modify-syntax-entry ?\{ "(}" pop-syntax-table)
	      (modify-syntax-entry ?\} "){" pop-syntax-table)
	      )
	  )
  (if pop-use-pl1-comments
      (progn
       (modify-syntax-entry 59 "_" pop-syntax-table) ; ?;
       (modify-syntax-entry ?\n " " pop-syntax-table)
       (modify-syntax-entry ?/ "_ 14" pop-syntax-table)
       (modify-syntax-entry ?* "_ 23" pop-syntax-table))
      (progn
       (modify-syntax-entry ?\n ">" pop-syntax-table)
       (modify-syntax-entry 59 "_ 12" pop-syntax-table) ; ?;
       (modify-syntax-entry ?/ "_" pop-syntax-table)
       (modify-syntax-entry ?* "_" pop-syntax-table)))
  (set-syntax-table pop-syntax-table)
  )

(defun pop-mode-commands (map)
       (define-key map "\n" 'pop-newline-indent)
       (define-key map ";" 'pop-semicolon-indent)
       (define-key map "\e]" 'pop-closeit)
       (define-key map "\e\C-i" 'pop-complete-word)

       (define-key map "\e\C-@" 'pop-mark-structure)
       (define-key map "\e\C-h" 'pop-mark-procedure)

       (define-key map "\e\C-q" 'pop-indent-structure)
       (define-key map "\e\C-p" 'pop-indent-definition)
       (define-key map "\e\C-r" 'pop-indent-region)

       (define-key map "\en" 'pop-forward-expression)
       (define-key map "\ep" 'pop-backward-expression)
       (define-key map "\e\C-n" 'pop-forward-expression)
       (define-key map "\e\C-p" 'pop-backward-expression)
       (define-key map "\en" 'pop-forward-word)
       (define-key map "\ep" 'pop-backward-word)
       (define-key map "\e\C-a" 'pop-beginning-of-define)
       (define-key map "\e\C-e" 'pop-end-of-define)

       (define-key map "\e\C-x" 'pop-compile-procedure)
       (define-key map "\C-c\C-p" 'pop-compile-procedure)
       (define-key map "\C-c\C-r" 'pop-compile-region)
       (define-key map "\C-c\C-b" 'pop-compile-buffer)

;;     (define-key map "\en" 'pop-next-help)
       (define-key map "\eh" 'pop-get-help)
       )

(pop-mode-commands pop-mode-map)

(defun pop-mode-variables ()
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'pop-indent-line)
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'pop-comment-indent)
  ; the following variables depend on pop-use-pl1-comments
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-end)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'parse-sexp-ignore-comments)
  (make-local-variable 'end-of-definition-func)
  (setq end-of-definition-func 'pop-end-of-define)
  (make-local-variable 'beginning-of-definition-func)
  (setq beginning-of-definition-func 'pop-beginning-of-define)
  (if pop-use-pl1-comments
      (progn
       (setq comment-start-skip "/\\* *" )
       (setq comment-start "/* ")
       (setq comment-end " */")
       (setq parse-sexp-ignore-comments t)
       )
      (progn
       (setq comment-start-skip ";;; *" )
       (setq comment-start ";;; ")
       (setq comment-end "")
       (setq parse-sexp-ignore-comments nil)))
  )

(defun list-to-regexp (list &optional fn mod)

       "Make a regular expression which matches the strings in LIST
If FN is given it is used to process elements of LIST before they are used.
MOD, if given is used to process the individual regular expressions before
they are concatenated together"

       (let ((l () ) re)
	    (if (not mod)
		(setq mod 'identity)
		)
	    (if (not fn)
		(setq fn 'identity)
		)
	    (while (consp list)
		   (setq re (apply mod 
				   (regexp-quote (apply fn (car list) nil))
				   nil ))
		   (setq l (cons "\\|" (cons re l)))
		   (setq list (cdr list))
		   )
	    (apply (function concat) (cdr l))
	    )
       )

(defun pop-mode ()
       "Major mode for editing POP-11 code. Be sure that pop-use-pl1-coments
is set how you want it before calling this procedure.

Commands:
\\{pop-mode-map} "

       (interactive)
       (kill-all-local-variables)
       (use-local-map pop-mode-map)
       (setq major-mode 'pop-mode)
       (setq mode-name "Pop11")
       (setq local-abbrev-table pop-mode-abbrev-table)
       (setq pop-syntax-regexp
	     (list-to-regexp pop-indentation-info 
			     'car '(lambda (x) (concat "\\<" x "\\>"))))
       (setq pop-closer-regexp
	     (list-to-regexp pop-open-close-assoc 'cdr))
       (setq pop-declaration-starters-regexp
	     (list-to-regexp pop-declaration-starters))
       (setq pop-declaration-modifiers-regexp
	     (concat (list-to-regexp pop-declaration-modifiers) "\\|[0-9]+" ))
       (setq pop-interesting-declaration-modifiers-regexp
	     (concat (list-to-regexp pop-interesting-declaration-modifiers) "\\|[0-9]+" ))
       (setq pop-definition-start-regexp
	     (concat "^\\(" (list-to-regexp pop-definition-starters) "\\)"))
       (setq pop-definition-starters-regexp
	     (list-to-regexp pop-definition-starters) )
       (setq pop-definition-enders-regexp
	     (list-to-regexp pop-definition-enders nil
			     '(lambda (x) (concat "^[ \t]*" x))))

       (pop-syntax-table)
       (pop-mode-variables)
       (run-hooks 'pop-mode-hook))

(defun next-word ()

       (buffer-substring (point) (save-excursion
				  (forward-word 1)
				  ( while (equal (char-after (point)) ?_ )
					  (forward-word 1)
					  )
				  (point) ))
       )

(defun pop-indent-line ()
       "Indent the current line as a pop-11 statement. Looks at
        the previous line and checks for indentation"

       (interactive)

       (let ((indent (calculate-pop-indent))
	     where)
	    (if (eq indent nil )
		(setq indent 0)
		(if (eq indent t)
		    (setq indent (calculate-pop-indent-within-comment))
		    )
		)
	    (save-excursion
	      (beginning-of-line)
	      (skip-chars-forward " \t")
	      (setq where (point))
	      (if (<= (current-column) indent)
		  (indent-to indent)
		  (progn
		    (move-to-column indent)
		    (if (save-excursion (backward-char 1) (looking-at "\t"))
			(backward-char 1))
		    (delete-region (point) where)
		    (indent-to indent)
		    )
		  )
	      )
	    (if (< (current-column) indent)
		(move-to-column indent)
		)
	    )
       )

(defun looking-at-oneof (list)
       "t if point is at the start of one of the words in
        LIST"
       (let* ((fl nil) (i list) 
	      (start (point))
	      (word (save-excursion (forward-word 1) (buffer-substring start (point))))
	      )
	     (while (and i (not fl))
		    (if (equal word (car i))
			(setq fl t)
			)
		    (setq i (cdr i))
		    )
	     fl
	     )
       )

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; The following are used to cache information relating to the      ;;
 ;; indentation of lines. This allows us to re-use information       ;;
 ;; from the last call if, as is often the case, it is still         ;;
 ;; valid.                                                           ;;
 ;;                                                                  ;;
 ;; Testing for validity is hard, here we compromise. We assume it   ;;
 ;; is valid if a marker at the begining of the last line is still   ;;
 ;; there and has not moved ( and, of course, if this is the         ;;
 ;; following line ).                                                ;;
 ;;                                                                  ;;
 ;; Programatic calls of pop-indent line which can assure validity   ;;
 ;; ( eg pop-indent-region and pop-newline-indent ) locally set   ;;
 ;; pop-cache-valid to t, thus forcing use of the cache.             ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defvar pop-cache-line-info nil)
(defvar pop-cache-parse-start nil)
(defvar pop-cache-state nil)
(defvar pop-cache-decls nil)
(defvar pop-cache-where nil)
(defvar pop-cache-where-marker (make-marker))
(defvar pop-cache-valid nil)

(defun calculate-pop-indent ()
  "Return appropriate indentation for current line as pop code.
In usual case returns an integer: the column to indent to.
Returns nil if line starts inside a string, t if in a comment."
  (save-excursion
    (beginning-of-line)
    (let ( old-state 
	   old-decls
	   (indent-point (point))
	   (case-fold-search nil)
	   (parse-sexp-ignore-comments t) ;locally do this globally may
					;not
	   decls line-info state parse-start 
	   info
	   indent)

	 (if (or pop-cache-valid
		 (and (eql pop-cache-where
			   (marker-position pop-cache-where-marker))
		      (eql (save-excursion (forward-line -1) (point))
			   pop-cache-where)))
	     ;; cached information is valid
	     (progn
;	       (debug nil 'cache)
	       (goto-char (setq parse-start pop-cache-parse-start))
	       (setq line-info pop-cache-line-info)
	       (setq old-state (setq state pop-cache-state))
	       (setq decls pop-cache-decls)
	       )
	     (if (not (looking-at pop-definition-starters-regexp))
		 (pop-beginning-of-define)
		 )
	     (setq parse-start (point))
	     (setq line-info (list (list 0 (point) nil)))
	     (setq old-state (setq state '(0 nil nil nil nil)))
	     (setq decls nil)
	     )

;      (debug nil parse-start line-info state)
      (while (< (point) indent-point)
;	     (debug nil 'one-line (point))
	     (setq next-line (save-excursion 
			       (end-of-line)
			       (parse-partial-sexp (point) indent-point
						   nil t nil)
			       (beginning-of-line)
			       (skip-chars-forward " \t\n");
;			       (debug nil 'next-line)
			      (point)))

	     (setq old-decls decls)
	     ;; look for declaration starts and ends
	     (if (looking-at pop-declaration-starters-regexp)
		 (setq decls (cons (car state) decls))
		 )
	     (if (and decls
		      (eql (car decls) (car state))
		      (looking-at "[^\n;]*;\\s-*$"))
		 (setq decls (cdr decls))
		 )

;	     (debug nil state line-info )
	     (while (and line-info state
			 (<= (car state) (car (car line-info)) ))
;		    (debug nil 'popping )
		    (setq line-info (cdr line-info))
		    )
	     (setq line-info (cons (list (car state) (point) old-decls) line-info))
;	     (if (eql (car state) 0)
;		 (debug nil (point))
		 (setq parse-start (point))
		 (setq old-state state)
;		 )
	     (setq state (parse-partial-sexp (point)
					     (min next-line
						  indent-point) nil nil state))
	     )

;      (debug nil decls old-decls line-info indent-point)

      (setq pop-cache-line-info line-info)
      (setq pop-cache-state old-state)
      (setq pop-cache-parse-start parse-start)
      (setq pop-cache-decls old-decls)
      (setq pop-cache-where indent-point)
      (set-marker pop-cache-where-marker indent-point)

;      (debug nil 'cached pop-cache-where pop-cache-where-marker)

      (cond ((or (nth 3 state) (nth 4 state))
	     (nth 4 state))

	    (t 
	      (if (> (car state) (car (car line-info)))
		  ;; we have pushed a level, indent to start of expression
		      (progn
			  ;; goto containing expression
			(goto-char (car (cdr state)))
			(forward-char 1)
			  ;; special case for "[%"
			  ;; should not be needed since % is defined as a
			  ;; bracket, however emacs is broke.
			(if (looking-at "[ \t]*%") 
			    (progn
			      (skip-chars-forward " \t")
			      (forward-char 1) 
			      )
			    )
			(skip-chars-forward " \t")
			(setq indent (current-column))
			)
		      ;; Otherwise goto last line at same level	
		      (while (< (car state) (car (car line-info)))
			     (setq line-info (cdr line-info))
			     )
		      (goto-char (nth 1 (car line-info)))
		      (skip-chars-forward " \t")
		      (setq indent (current-column))
		      )
	      (skip-chars-forward " \t\n\f")
;	      (debug nil 'decls decls (nth 2 (car line-info)))
	      (setq indent (+ indent
			      (* (- (length decls) (length (nth 2 (car line-info))))
				 pop-declaration-indent-step)))
	      (setq next-line (save-excursion
				(goto-char indent-point)
				(forward-line 1)
				(point)))
	     (while (< (point) indent-point)
		    (if (and (looking-at pop-syntax-regexp)
			     (setq info (assoc (next-word) 
						 pop-indentation-info)))
			(setq indent (+ indent (nth 2 info)))
			
			(if (and (not decls) (looking-at pop-label-regexp))
			    (progn
;			     (debug)
			     (setq indent (+ indent pop-label-indent))
			     )
			    )
			)
		    (forward-sexp 1)
		    (skip-chars-forward "\t\n\f ")
		    (if (and (< (point) next-line)
			     (looking-at pop-syntax-regexp)
			     (setq info (assoc (next-word) 
					       pop-indentation-info ))
			     )
			(setq indent (+ indent 
					  (nth 1 info)))
			(if (and (not decls) (looking-at pop-label-regexp))
			    (setq indent (- indent pop-label-indent))
			    )
			)
		    )
	     indent
	     )
	    )
      )
    )
  )

(defun calculate-pop-indent-within-comment ()
  "Return the indentation amount for line, assuming that
the current line is to be regarded as part of a block comment."

  (let (end star-start)
    (save-excursion
      (beginning-of-line)
      (skip-chars-forward " \t")
      (setq star-start (= (following-char) ?\*))
      (skip-chars-backward " \t\n")
      (setq end (point))
      (beginning-of-line)
      (skip-chars-forward " \t")
      (and (re-search-forward "/\\*[ \t]*" end t)
	   star-start
	   (goto-char (1+ (match-beginning 0))))
      (current-column))))

	    
(defun pop-beginning-of-define (&optional num)
	"Move back to beginning of definition"

	(interactive "p")

	(setq num (or num 1))

	(if (not (looking-at pop-definition-start-regexp))
	    (forward-line 1)
	    )
	(while (> num 0)
	       (re-search-backward pop-definition-start-regexp nil 'bob)
	       (setq num (1- num))
	       )
	)

(defun pop-end-of-define (&optional num)
       "Move to end of definition"

       (interactive "p")

       (setq num (or num 1))

       (if (> num -1 )
	   (let ((pos (point)))
		(forward-char 1)
		(pop-beginning-of-define)
		
		(while (or (> num 0 ) (>= pos (point)))
		       (re-search-forward pop-definition-start-regexp nil 'bob)
		       (beginning-of-line)
		       (pop-forward-expression)
		       (setq num (1- num))
		       (forward-line 1)
		       (beginning-of-line)
		       )
		)
	   (progn
	    (pop-beginning-of-define (- 1 num))
	    (pop-end-of-define 1)
	    )
	   )
       )

(defun pop-procedure-ends ()

       "return a pair giving the start end end of the current procedure."

       (let ( start end )
	    (save-excursion
	     (pop-beginning-of-define)
	     (setq start (point))
	     (pop-forward-expression)
	     (setq end (point))
	     )
	    (cons start end)
	    )
       )

(defun pop-mark-procedure ()
  "Mark the current procedure"

  (interactive)
  (let ((proc (pop-procedure-ends)))
       (push-mark (car proc))
       (goto-char (cdr proc))
       (message "Mark set"))
  )

(defun pop-mark-structure (n)
	"Set mark after ARG structures"

	(interactive "p")

	(let ((i 0) where)
	     (save-excursion
	       (while (< i n)
		      (pop-forward-expression)
		      (setq i (1+ i))
		      )
	       (setq where (point))
	       )
	     (set-mark where)
	     (message "Mark set")
	     )
	)

(defun pop-indent-region (start end)
       "Indent all lines in region"

       (interactive "m\nd")

       (if (not (markerp start))
	   (setq start (set-marker (make-marker) start))
	   )
       (if (not (markerp end))
	   (setq end (set-marker (make-marker) end))
	   )

       (save-excursion
	 (let ((pop-cache-valid) (lines (count-lines start end)))
	      (if ( < lines 20)
		  (setq lines -1)
		  )
	      (goto-char start)
	      (while (< (point) (marker-position end))
		     (if (< 0 lines)
			 (progn
			  (if (eq (mod lines 10) 0 )
			      (message "%d lines" lines))
			  (setq lines (1- lines))
			  )
			 )
		     (pop-indent-line)
		     (setq pop-cache-valid t)
		     (forward-line 1)
		     )
	      (if (>= lines 0) (message "Done"))
	      )
	 )
       )

(defun pop-indent-definition ()
       "Indent the definition containing point"

       (interactive)
       (save-excursion
	 (pop-beginning-of-define)
	 (let ( (start (point))
		end )
	      (pop-forward-expression)
	      (pop-indent-region start (point))
	      )
	 )
       )

(defun pop-indent-structure ()
       "Indent structure starting at point"

       (interactive)
       (let ( (end (save-excursion (pop-forward-expression) (point))) )
	    (pop-indent-region (point) end)
	    )
       )

(defun pop-forward-word (n)

       "move forward over a pop11 word"

       (interactive "p")

       (while (> n 0)
	      (forward-word 1)
	      (skip-chars-forward "_")
	      (while (looking-at "\\sw")			  
		     (forward-word 1)
		     (skip-chars-forward "_")
		     )

	      (setq n (1- n))
	      )
       )

(defun pop-backward-word (n)

       "move backward over a pop11 word"

       (interactive "p")

       (while (> n 0)
	      (forward-word -1)
	      (skip-chars-backward "_")
	      (while (and (> (point) 0)
			  (save-excursion (forward-char -1) 
					  (looking-at "\\sw"))
			  )
		     (forward-word -1)
		     (skip-chars-backward "_")
		     )
	      (setq n (1- n))
	      )
       )

(defun pop-forward-expression (&optional n)

       "Move forward N syntax structures"

       (interactive "p")
       (or n (setq n 1))

       (while ( > n 0 )
	      (let* ((them (catch 'syntax (pop-end-of-this-structure)))
		    (what (car them))
		    (end (cdr them)))

		   (if (< end 0 )
		       (progn
			(goto-char (- end))
			(error "Unmatched %s" what)
			)
		       (goto-char end)
		       )
		   )
	      (setq n (1- n))
	      )
       )

(defun pop-end-of-this-structure (&optional finish)

       "Returns a pair ( what . end ) where what is a pop syntax opener and 
end is the point in the file at which it ends. If end is negative then there
was an unclosed structure of type what *starting* at end"

       (setq finish (or finish (point-max)))

       (save-excursion
	(let (start
	      opener
	      closer
	      (parse-sexp-ignore-comments t) ;locally do this globally may not
	      closer-reg
	      i flag )
	     (skip-chars-forward " \n\t\f")
	     (setq start (point))
	     (forward-sexp 1)
	     (setq opener (buffer-substring start (point)))
	     (setq closer (assoc opener pop-open-close-assoc))
	     (pop-skip-over-whitespace)
	     (if (string-match  pop-definition-starters-regexp opener )
		  (while (looking-at pop-declaration-modifiers-regexp)
			 (pop-forward-word 1)
			 (pop-skip-over-whitespace)
			 )
		  )
		  
	     (if closer
		 (progn
		  (setq closer-reg (concat (regexp-quote (cdr closer)) "\\>"))
		  (while (not (or (looking-at pop-closer-regexp)
				  (looking-at pop-definition-enders-regexp)
				  (>= (point) finish)))
			 (goto-char (cdr (pop-end-of-this-structure finish)))
			 (pop-skip-over-whitespace)
			 )
		  (if (not (looking-at closer-reg))
		      (throw 'syntax (cons opener (- start)))
		      )
		  (forward-sexp 1)
		  )
		 )
	     (cons opener (point))
	     )
	)
       )

(defun pop-backward-expression (n)
       "Move back/wards over a pop11 construct. Hack"

       (interactive "p")
       (or n (setq n 1))

       (while (> n 0)
	      (skip-chars-backward " \n\t\f")
	      (let (
		    ( old (point))
		    )
		   (backward-sexp 1)
		   (while (or
			    (and (> (point) (point-min))
				 (not (<= old (save-excursion
						(pop-forward-expression)
						(point)))))
			    (looking-at pop-closer-regexp)
			    )
			  (backward-sexp 1)
			  )
		   )
	      (setq n (1- n))
	      )
       )

(defun pop-comment-indent ()

       (save-excursion
	 (let ( next)
	      (end-of-line)
	      (save-excursion
		(beginning-of-line)
		(parse-partial-sexp (point) (point-max) nil t)
		(setq next (point))
		)
	      (if (< (point) next)
		  (calculate-pop-indent)
		  (max (1+ (current-column)) comment-column)
		  )
	      )
	 )
       )

(defun pop-semicolon-indent ()

       "Inserts a semicolon and then indents the line"
       (interactive)

       (insert-char ?; 1)
       (indent-for-tab-command)
       )

(defun pop-newline-indent ()
       "indent for new line"
       
       (interactive)
       (let ((pop-cache-valid))
		;;; (indent-for-tab-command) revised 17.07.89 bsl 
	    ;;; (setq pop-cache-valid t)
	    (newline)
	    (indent-for-tab-command)
	    )
       )

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; Pop-closit closes the last open structure. eg, puts in the       ;;
 ;; endfor to an open for. It indents things correctly most of the   ;;
 ;; time.                                                            ;;
 ;;                                                                  ;;
 ;; When closing a `define' it adds a comment naming the thing       ;;
 ;; defined and possibly the fact that it is a macro etc.            ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun pop-closeit(n)

       "Close tha last N syntactic structures"

       (interactive "p")

       (let ( indent
	      what 
	      start
	      (place (point)))
	    (save-excursion
	     (setq start (save-excursion (pop-beginning-of-define) (point)))
	     (condition-case foo (progn
				  (goto-char (scan-lists (point) -1 1))
				  (goto-char (scan-lists (point) 1 -1))
				  (skip-chars-forward " \t%" )
				  )
			     (error (goto-char start)))

	     (if (< (point) start)
		 (goto-char start))

	     (let* ((them (catch 'syntax (pop-end-of-this-structure place))))
		   (setq start  (- (cdr them)))
		   (if (or (< start 0 ) (> start place))
		       (error "Nothing unclosed")
		       )
		   (goto-char start)
		   (setq what (car them))
		   (setq indent (current-column))
		   )
	     )
	    (beginning-of-line)
	    (if (not (looking-at "[ \t]*\n"))
		(progn
		 (end-of-line)
		 (newline 1)
		 ))
	    (insert-string (cdr (assoc what pop-open-close-assoc)))
	    (insert-string ";")

	    (if (equal what "define")
		(progn
		 (insert-string "    ;;; ")
		 (insert-string (pop-define-name start))
		 ))
	    (pop-indent-line)
	    )
       )

(defun pop-define-name (&optional where)

       "Returns the name of the current define, or the one starting
at WHERE, if that is given. The name  will include any `interesting'
keywords in the declaration. See pop-interesting-declaration-modifiers"

       (interactive "d")

       (save-excursion
	(let ( ( name "") )
	     (if where
		 (goto-char where)
		 (pop-beginning-of-define)
		 )
	     
	     (forward-sexp 1)
	     (pop-skip-over-whitespace)
	     (while (looking-at pop-declaration-modifiers-regexp)
		    (if (looking-at pop-interesting-declaration-modifiers-regexp)
			(setq name (concat name " "
					   (buffer-substring (point)
							     (progn
							      (forward-sexp 1)
							      (point)))))

			(forward-sexp 1)
			)
		    (pop-skip-over-whitespace)
		    )
	     (let ((start (point)))
		  (forward-sexp 1)
		  (concat name " " (buffer-substring start (point)))
		  )
	     )
	)
       )



(defun pop-skip-over-whitespace ()
       "Moves forward over whitespace and comments. This is a little
weird."

       (interactive)
	(let* ( (fl t)
		parse-sexp-ignore-comments
		(last (point))
		nocom )
	      (setq parse-sexp-ignore-comments t)
	      (setq nocom (save-excursion (forward-sexp) (point)))
	      (setq parse-sexp-ignore-comments nil)

	      (while (not (eql (save-excursion (forward-sexp) (point)) nocom))
		     (forward-sexp)
		     )
	      (skip-chars-forward " \t\n\f")
	      )
	)

       
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; Code to implement word completion in pop-mode and                ;;
 ;; inferior-pop-mode. This relies on haveing a pop11 running so     ;;
 ;; that it can ask for a list of words.                             ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun pop-get-words-command (init file)
       
       "Returns a string containing a pop11 command to get a list
of all words starting with INIT into the file FILE"

       (format "procedure; vars cucharout=discout('%s'); lvars l=match_wordswith('@a%s'); l.pr; 1.nl; termin.cucharout; endprocedure();1.nl;\n"
	       file
	       init)
       )

(defun pop-get-words-starting (init)

       "Returns a list of words starting with the given string.
If there is no pop runing or it is doing something, signal an error"

       (interactive "sWords starting: ")

       (let* (
	     (pop-process
	      (or (eval inferior-pop-process-variable) 
		  (error (concat "No " inferior-pop-name " process"))))
	     (pop-buffer (process-buffer pop-process))
	     (this-buffer (current-buffer))
	     (tmp-buffer (get-buffer-create "* pwtmp *"))
	     (file (make-temp-name "/tmp/emacsw"))
	     (command (pop-get-words-command init file))
	     oldsize
	     origplace
	     list
	     )

	     (if (not (eq (process-status pop-process) 'run))
		 (error (concat inferior-pop-name " process not runing"))
		 )

	     (save-excursion
	      (set-buffer pop-buffer)
	      (end-of-buffer)
	      (if (not (pop-top-level-p))
		  (error "not a pop top level")
		  )
	      (setq origplace (point))
	      (insert ";;; fetching words\n")
	      (setq oldsize (buffer-size))
	      (send-string pop-process  command)
	      (while (= oldsize (buffer-size))
		     (sit-for 1)
		     )
	      (delete-region origplace (point))
	      )
	     
	     (save-excursion
	      (set-buffer tmp-buffer)
	      (erase-buffer)
	      (insert-file file)
	      (replace-string "[" "(")
	      (replace-string "]" ")")
	      (beginning-of-buffer)
	      (setq list (read tmp-buffer))
	      )
	     (delete-file file)
	     (kill-buffer tmp-buffer)
	     list
	     )
       )
       
(defun pop-try-complete (init words)

       "Return the common substring of all completions of INIT
which are elements of WORDS. Just like try-complete but takes a list 
instead of an alist."

       (if words
	   (try-completion init (mapcar 
				 (function 
				  (lambda (w) (list (symbol-name w)))) 
				 words))
	   nil)
       
       )

(defun pop-complete-word ()

       "Complete pop11 word at or before point"

       (interactive)

       (pop-backward-word 1)

       (let ((start (point))
	     end wd words comp)
	    (pop-forward-word 1)
	    
	    (setq end (point))
	    (setq wd (buffer-substring start end))
	    (setq words (pop-get-words-starting wd))
	    (setq comp (pop-try-complete wd words))

	    (cond
	     ((eq comp t) t)
	     ((equal comp wd)
	      (with-output-to-temp-buffer "*Help*"
					  (display-completion-list words))
	      (message "Making completion list...%s" "done")
	      )
	     (comp 
	      (delete-region start end)
	      (insert comp)
	      )
	     (t 
	      (message "no completions for " wd)
	      )
	     )
	    )
       )
	    
