 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;;                 Pop-sys-file-mode                                ;;
 ;;                                                                  ;;
 ;;         Richard Caley 1990 and beyond.                           ;;
 ;;         Prolog support by Brian Logan.                           ;;
 ;;                                                                  ;;
 ;;         Bugs etc to rjc@cstr.ed.ac.uk                            ;;
 ;;                                                                  ;;
 ;; A major mode for reading poplog help documentation.              ;;
 ;;                                                                  ;;
 ;; This implements my model of help reading, not the standard       ;;
 ;; poplog one -- there is only one help buffer rather than one      ;;
 ;; per help file.  Use the command pop-help-save to create a        ;;
 ;; dedicated buffer for this help file. The idea is that you        ;;
 ;; flick through the help files using M-n and M-h (or `/' and       ;;
 ;; `?', which I prefer) and then when you find one of interest,     ;;
 ;; hit `s' to save it. It then stays in the other buffer while      ;;
 ;; you scan some more. This would be much nicer if it worked        ;;
 ;; nicely in epoch, creating a new X window for the saved           ;;
 ;; buffer...                                                        ;;
 ;;                                                                  ;;
 ;; If pop-short-help-command is set true then the commands          ;;
 ;; M-Xhelp etc can be used to get help rather than M-Xpop-help      ;;
 ;; etc. Nice, but not for non poplog people.                        ;;
 ;;                                                                  ;;
 ;; This code is distributed under the conditions of the Gnu         ;;
 ;; General Public Licence, see COPYING                              ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 ;;                                                                  ;;
 ;; THINGS TO DO:                                                    ;;
 ;;         A ved_?? equivalent would be nice.                       ;;
 ;;                                                                  ;;
 ;;         Some nice way of _creating_ help files. This would be    ;;
 ;;         a mix of pop-sys-file-mode and text-mode. Versions of    ;;
 ;;         ved_indexify and ved_header would be nice                ;;
 ;;                                                                  ;;
 ;;         Epoch support.                                           ;;
 ;;                                                                  ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 29th Jan 1991 rjc
;;; Fixed a problem with pop-get-help near the start of a buffer.
;;; Also made pop-next-help a little more discerning about
;;; what it considers a cross reference. 

;;; 24th Feb 1991 rjc
;;; added more directories.
;;; added the index bulding code.
;;; added pop-apropos

;;; 27th Feb 1991 bsl
;;; added pop-plogteach and pop-ploglib for prolog teach and lib files
;;; added DEL to the default key map for pop-sys-file-mode
;;; changed pop-sys-file-mode-hooks to pop-sys-file-mode-hook for compatibility
;;; with pop-mode
;;; fixed typo in pop-short-help-command(s)

;;; 22nd April 1991
;;; Now understands usepop, poplocal and poplocal emacs
;;;	variables if set to override environment.
;;; Added default type so that ref index etc work.

(provide 'pop-sys-file-mode)


(defvar pop-short-help-commands nil
	  "*If non-nil when pop-sys-file-mode is loaded then the commands
help, ref, doc and teach are defined as synonyms for pop-help etc. "
	  )

(defvar pop-default-type nil
	"Buffer local variable giving the default type of
documentation to look for." )

(defvar pop-sys-file-map
	(make-sparse-keymap)
	"Key map for poplog help file mode" )

(defun pop-set-help-keys (map)

       (define-key map " " 'scroll-up)
       (define-key map "\C-?" 'scroll-down)

       (define-key map "?" 'pop-get-help)
       (define-key map "/" 'pop-next-help)
       (define-key map "\eh" 'pop-get-help)
       (define-key map "\en" 'pop-next-help)
       (define-key map "\eg" 'pop-goto-section)

       (define-key map "s" 'pop-save-help-buffer)
       (define-key map "S" 'pop-save-help-buffer)
       (define-key map "\es" 'pop-save-help-buffer)
       )

(pop-set-help-keys pop-sys-file-map)

(defvar usepop (or (getenv "usepop") "."))
(defvar poplocal (or (getenv "poplocal") "."))
(defvar poplib (or (getenv "poplib") "."))

(defvar pop-help-dirs
	(list
	 (concat poplib "/help")
	 (concat poplocal "/local/help")
	 (concat usepop "/pop/local/help")
	 (concat usepop "/pop/help")
	 )
	"* list of directories where help looks for poplog help files"
	)

(defvar pop-teach-dirs
	(list
	 (concat poplib "/teach")
	 (concat poplocal "/local/teach")
	 (concat usepop "/pop/local/teach")
	 (concat usepop "/pop/teach")
	 )
	"* list of directories where help looks for poplog teach files"
	)

(defvar pop-doc-dirs
	(list
	 (concat poplib "/doc")
	 (concat poplocal "/local/doc")
	 (concat usepop "/pop/local/doc")
	 (concat usepop "/pop/doc")
	 )
	"* list of directories where help looks for poplog doc files"
	)

(defvar pop-ref-dirs
	(list
	 (concat poplib "/ref")
	 (concat poplocal "/local/ref")
	 (concat usepop "/pop/local/ref")
	 (concat usepop "/pop/ref")
	 )
	"* list of directories where help looks for poplog ref files"
	)

(defvar pop-lib-dirs
	(list
	 (concat poplib "/lib")
	 (concat poplocal "/local/lib")
	 (concat poplocal "/local/lib/lib")
	 (concat poplocal "/local/lib/auto")
	 (concat poplocal "/local/lib/ved")
	 (concat poplocal "/local/lib/contrib")
	 (concat usepop "/pop/local/lib")
	 (concat usepop "/pop/local/lib/lib")
	 (concat usepop "/pop/local/lib/auto")
	 (concat usepop "/pop/local/lib/ved")
	 (concat usepop "/pop/local/lib/contrib")
	 (concat usepop "/pop/lib")
	 (concat usepop "/pop/lib/lib")
	 (concat usepop "/pop/lib/auto")
	 (concat usepop "/pop/lib/ved")
	 (concat usepop "/pop/lib/contrib")
	 )
	"* list of directories where help looks for poplog lib files"
	)

(defvar pop-ploghelp-dirs
	(list
	 (concat poplib "/ploghelp")
	 (concat poplocal "/local/plog/help")
	 (concat usepop "/pop/plog/local/ploghelp")
	 (concat usepop "/pop/plog/help")
	 )
	"* list of directories where help looks for poplog prolog help files"
	)

(defvar pop-plogteach-dirs
	(list
	 (concat poplib "/plogteach")
	 (concat poplocal "/local/plog/teach")
	 (concat usepop "/pop/plog/local/teach")
	 (concat usepop "/pop/plog/teach")
	 )
	"* list of directories where help looks for poplog prolog teach files"
	)

(defvar pop-ploglib-dirs
	(list
	 (concat poplib "/lib")
	 (concat poplocal "/local/plog/lib")
	 (concat poplocal "/local/plog/lib/lib")
	 (concat poplocal "/local/plog/lib/auto")
	 (concat poplocal "/local/plog/lib/contrib")
	 (concat usepop "/pop/plog/local/lib")
	 (concat usepop "/pop/plog/local/lib/lib")
	 (concat usepop "/pop/plog/local/lib/auto")
	 (concat usepop "/pop/plog/local/lib/contrib")
	 (concat usepop "/pop/plog/lib")
	 (concat usepop "/pop/plog/lib/lib")
	 (concat usepop "/pop/plog/lib/auto")
	 (concat usepop "/pop/plog/lib/contrib")
	 )
	"* list of directories where help looks for poplog prolog lib files"
	)

(defconst pop-sys-file-types
	'(
	  ( "REF"  "ref" . pop-ref-dirs )
	  ( "HELP"  "help" . pop-help-dirs )
	  ( "TEACH"  "teach" . pop-teach-dirs )
	  ( "DOC"  "doc" . pop-doc-dirs )
	  ( "LIB"  "lib" . pop-lib-dirs )
	  ( "PLOGHELP"  "ploghelp" . pop-ploghelp-dirs )
	  ( "PLOGTEACH" "plogteach" . pop-plogteach-dirs )
	  ( "PLOGLIB" "ploglib" . pop-ploglib-dirs)
	  )
	"*Association list mapping cross references to help types."
	)

(defvar pop-sys-file-mode-hook nil
	"* Hook run when setting pop sys file mode" )

(defvar pop-index-item
	nil
	"pointer to the index item last looked at" )

(defun pop-sys-file-mode ()

       "Major mode for reading poplog documentation.
Commands:
\\{pop-sys-file-map}"

       (interactive)
       (kill-all-local-variables)
       (use-local-map pop-sys-file-map)
       (setq major-mode 'pop-sys-file-mode)
       (setq mode-name "Poplog Help")
       (setq buffer-read-only t)
       (make-local-variable 'pop-help-type)
       (make-local-variable 'pop-help-subject)
       (make-local-variable 'pop-default-type)
       (run-hooks 'pop-sys-file-mode-hook)
       )

(defun pop-get-summary (file)
       "Returns a one line summary of the helpfile FILE"

       (let ((name (file-name-nondirectory file))
	     (buffer (get-buffer-create " sumtmp " ))
	     line
	     )

	    (save-excursion
	     (set-buffer buffer)
	     (erase-buffer)
	     (insert-file file)
	     (beginning-of-buffer)
	     (re-search-forward "^$" nil t)
	     (if (re-search-forward "^\\(\\s-+[^-]\\)" nil t)
		 (beginning-of-line)
		 (forward-line -1)
		 )
	     (skip-chars-forward "\\s-\n")
	     (setq line (buffer-substring (point) (save-excursion 
						   (end-of-line) (point))))
	     (format "%20s  |  %s" (concat "* " name) line)
	     )
	    )
       )
	     

(defun pop-create-index (what where apropos)
       "Create an index for the help files of type WHAT in directories WHERE"
       
       (let (some all last start
		  (fill-prefix "\t\t       ")
	     )

	    (if apropos
		(insert "Apropos " topic "\n\n")
		(insert "Index of " what "files\n\n")
		)
	    (while (not (eq where nil))
		   (if (file-directory-p (car where))
		       (setq all (append all (directory-files 
					      (car where) 
					      t 
					      (or apropos "[a-z].*"))))
		       )
		   (setq where (cdr where))
		   )
	    (setq all (sort all (function equal)))
	    
	    (setq last nil)
	    (while (not (eq all nil))
		   (if (not (equal (car all) last))
		       (if apropos
			   (progn
			    (setq start (point))
			    (insert (pop-get-summary (car all)) ?\n)
			    (fill-region start (point))
			    (sit-for 0)
			    )
			   (progn
			    (setq name (file-name-nondirectory (car all)))
			    (insert name)
			    (if (> (current-column) 60)
				(progn
				 (insert ?\n )
;				 (sit-for 0)
				 )
				(insert-char ? (- 20 (length name)))
				)
			    )
			   )
		       )
		   (setq last (car all))
		   (setq all (cdr all))
		   )
	    )
       )
		   
	
(defun pop-sys-file (helpfile what where &optional topic)

       "Finds a poplog help file for HELPFILE of type WHAT in the
directories WHERE"

       (let ( (filename nil) (filename-z nil) (fl nil) buf )

	    (setq what (or what "help"))

	    (setq buf (get-buffer-create (concat "*pop " what "*")))
	    (if (not (eq buf (current-buffer)))
		(switch-to-buffer-other-window buf))
	    
	    (setq buffer-read-only nil)
	    (delete-region (point-min) (point-max))
	    (beginning-of-buffer)

	    (cond
	     ((eq helpfile t)
	      (pop-create-index what where topic)
	      (setq pop-default-type (upcase what))
	      (setq helpfile (concat "apropos " topic))
	      (setq topic (concat "* " topic " |"))
	      (beginning-of-buffer)
	      )
	     ((equal helpfile "index")
	      (pop-create-index what where nil)
	      (setq pop-default-type (upcase what))
	      (beginning-of-buffer)
	      )	     
	     (t			;look up helpfile
		(while (and (consp where) (not fl) )
		       (setq filename (concat (car where) "/" helpfile ))
		       ( if (equal what "lib")
			    (setq filename (concat filename ".p"))
			    )
		       ( if (equal what "ploglib")
			    (setq filename (concat filename ".pl"))
			    )
		       (setq filename-z (concat filename ".Z"))
		       (if (file-readable-p filename)
			   (setq fl t)
			   )

		       (if (file-readable-p filename-z)
			   (setq fl 'Z)
			   )
		       (setq where (cdr where))
		       )
		(if (not fl)
		    (error "No %s file for %s, try %s index or %s %sfiles" 
			   what helpfile what what what )
		    (insert-file filename)
		    )
		(setq pop-default-type nil)
		)
	       )

	    (if topic
		(if (not (search-forward topic nil t))
		    (message "no information on %s in %s" topic helpfile)
		    (search-backward topic nil t)
		    )
		)
		    
	    (setq buffer-read-only t)       
	    (setq pop-index-item nil)
	    (if (not (eq major-mode 'pop-sys-file-mode))
		(pop-sys-file-mode))

	    (setq pop-help-type what)
	    (setq pop-help-subject helpfile)
	    )
       )

(defun pop-apropos (pattern)
       "get summary help for everything matching PATTEN"

       (interactive "sApropos ")
       (pop-sys-file t "help" pop-help-dirs pattern)
       )

(defun pop-help (subject)

       "Get poplog help for SUBJECT"

       (interactive "sHelp for? " )
       
       (pop-sys-file subject "help" pop-help-dirs)
       )

(defun pop-teach (subject)

       "Get poplog teach file for SUBJECT"

       (interactive "sTeach for? " )
       
       (pop-sys-file subject "teach" pop-teach-dirs)
       )

(defun pop-doc (subject)

       "Get poplog doc file for SUBJECT"

       (interactive "sDoc for? " )
       
       (pop-sys-file subject "doc" pop-doc-dirs)
       )

(defun pop-ref (subject)

       "Get poplog ref file for SUBJECT"

       (interactive "sRef for? " )
       
       (pop-sys-file subject "ref" pop-ref-dirs)
       )

(defun pop-showlib (subject)

       "Get poplog library for SUBJECT"

       (interactive "sShowlib? " )
       
       (pop-sys-file subject "lib" pop-lib-dirs)
       )

(defun pop-ploghelp (subject)

       "Get poplog prolog help file for SUBJECT"

       (interactive "sPlog help for? " )
       
       (pop-sys-file subject "ploghelp" pop-ploghelp-dirs)
       )

(defun pop-plogteach (subject)

       "Get poplog prolog teach file for SUBJECT"

       (interactive "sPlog Teach for? " )
       
       (pop-sys-file subject "plogteach" pop-plogteach-dirs)
       )

(defun pop-plogshowlib (subject)

       "Get poplog prolog library for SUBJECT"

       (interactive "sPlog Showlib? " )
       
       (pop-sys-file subject "ploglib" pop-ploglib-dirs)
       )

(defun pop-next-help (n)

       "Go to next cross reference"

       (interactive "p")

       (while (> n 0)
	      (if (not (re-search-forward "[^*]\\*[^*]" nil t))
		  
		  (error "No more references in this file")
		  )
	      (setq n (1- n))
	      (forward-char -1)
	      )
       (forward-char -1)
       )

(defun pop-get-help ()

       "Get help for the current word"

       (interactive)

       (let ( ( what nil)
	      ( dirs nil)
	      ( topic nil)
	      ( type (or pop-default-type "HELP"))
	      start
	      helpfile )

	    (if (eql (char-after (point)) ?*)
		(forward-word 1))

	    (while (and (not (eq (current-column) 0)) 
			(save-excursion (forward-char -1) (looking-at "\\sw")))
		   (forward-char -1))

	    (save-excursion
	     (skip-chars-backward " \t\n")
	     (if (eql (char-after (1- (point))) ?* )
		 (progn
		  (forward-char -1)
		  (skip-chars-backward " \n\t")
		  (let ((place (point)))
		       (skip-chars-backward "^ \n\t")
		       (setq type (buffer-substring (point) place))
		       )
		  )
		 )
	     )

	    (if (not (setq type (assoc type pop-sys-file-types)))
		(setq type (assoc "HELP" pop-sys-file-types))
		)
	    (setq what (car (cdr type)))
	    (setq dirs (eval (cdr (cdr type))))

	    (save-excursion 
	     (skip-chars-forward " \t\n")
	     (setq start (point))
	     (skip-chars-forward "-a-zA-Z0-9_")
	     (setq helpfile (buffer-substring 
			    start
			    (point)))
	     (if (eql (char-after (point)) ?/)
		 (progn
		  (forward-char 1)
		  (setq start (point))
		  (skip-chars-forward "^ \t\n/")
		  (setq topic  (buffer-substring 
				start
				(point)))
		  )
		 )
	     )

	    (pop-sys-file (downcase helpfile) what dirs topic)
	    )
       )

(defun pop-save-help-buffer (&optional buffer)

       "Make a buffer to keep the buffer BUFFER (current if not given)
in help file in."

       (interactive)

       (if (not buffer)
	   (setq buffer (current-buffer))
	   )

       (let* ((name (concat "*" pop-help-type " " pop-help-subject "*" ))
	      (new (get-buffer name))
	      (p-d-t pop-default-type)
	      (pos (point))
	      )
	     (if new
		 (progn
		  (message "There is already a saved %s file for %s"
			   pop-help-type pop-help-subject)
		  (switch-to-buffer-other-window new)
		  )
		 (progn
		  (setq new (get-buffer-create name))
		  (copy-to-buffer new (point-min) (point-max))
		  (save-excursion
		   (set-buffer new)
		   (pop-sys-file-mode)
		   (setq pop-default-type p-d-t)
		   )
		  (switch-to-buffer-other-window new)
		  )
		 )
	     (goto-char pos)
	     )
       )

(defun pop-goto-section ()

       "Goto the required section of a poplog help file"

       (interactive)

       (beginning-of-line)

       (if (not (looking-at " --" ))
	   (progn
	    (if pop-index-item
		(goto-char pop-index-item)
		
		(beginning-of-buffer)
		(search-forward " --")
		(beginning-of-line)
		(next-line -1)
		)
	    (next-line 1)
	    (if (not (looking-at " --"))
		(progn
		 (beginning-of-buffer)
		 (search-forward " --")
		 )
		)
	    )
	   (progn
	    (setq pop-index-item (point))
	    
	    (forward-char 4)
	    (skip-chars-forward "\t ")

	    (let ((what (buffer-substring (point) (progn (end-of-line) (point)))))

		 (setq what (concat "^-*[ \t]*" (regexp-quote what) "[ \t]*-*[ \t]*$"))
		 (re-search-forward what)
		 (beginning-of-line)
		 )
	    )
	   )
       )

(if pop-short-help-commands
    (progn
     (fset 'help (symbol-function 'pop-help))
     (fset 'ref (symbol-function 'pop-ref))
     (fset 'doc (symbol-function 'pop-doc))
     (fset 'teach (symbol-function 'pop-teach))
     (fset 'showlib (symbol-function 'pop-showlib))
     )
    )

