/*

        Computers and Thought: Chapter 9 - Computer Vision


        Contains POP-11 procedures included in Appendix to Chapter 9

        Also contains lists of grey-levels for 64x64 image of
        Victoria underground sign (victoriavals) and 36x41 template
        of underground sign (signvals).

    Example session:

    /* load up image and template */
    vars victoria_image, sign_template;
    newarray([1 64 1 64]) -> victoria_image;
    fillimage(victoria_image, victoriavals);
    newarray([1 36 1 41]) -> sign_template;
    fillimage(sign_template, signvals);

    /* find matches better than 60% */
    match_image(sign_template, victoria_image, 0.6);

** [There is a good match at position 10 11]
** [There is a good match at position 11 11]
** [There is a good match at position 9 12]
** [There is a good match at position 10 12]
** [There is a good match at position 8 13]
** [There is a good match at position 9 13]

    /* produce region map */
    findregions(victoria_image, [ [3 8] [20 45] [45 80] [85 99] ])
        -> regionmap;

*/


define fillimage(image, list);

  /* to transfer numbers from list into array */
  vars x, y, g, xsize, ysize;

  boundslist(image) --> [1 ?xsize 1 ?ysize];

  /* scan array from top-left to bottom-right
     inserting grey-levels from list */
  for y from 1 to ysize do
    for x from 1 to xsize do
      list --> [?g ??list];
      g -> image(x,y)
    endfor
  endfor
enddefine;


define findedgels(image, thrshld) -> edgemap;
  vars x y xsize ysize;

  /* find the width and height of the given image */
  boundslist(image) --> [1 ?xsize 1 ?ysize];

  /* make an edge map of the same size as the
     given image and initialise each entry to 0 */
  newarray([1 ^xsize 1 ^ysize], 0) -> edgemap;

  /* scan each pixel in turn */
  for y from 2 to ysize-1 do
    for x from 2 to xsize-1 do

      /* does sum of differences between horizontally and
         vertically adjacent pixels exceeds given threshold */
      if abs(image(x,y) - image(x-1,y))
        + abs(image(x,y) - image(x,y-1)) > thrshld
      then
        /* yes so record a 1 in current position */
        1 -> edgemap(x,y)
      endif
    endfor
  endfor

enddefine;


define match_at_xy(edgemap1, edgemap2, x, y) -> goodness;
  vars xx, yy, xsize, ysize, maxtotal, hittotal;

  /* find the size of the edge map from the template */
  boundslist(edgemap1) --> [1 ?xsize 1 ?ysize];

  0 -> maxtotal; 0 -> hittotal;

  /* scan each element of edgemap1 */
  for yy from 1 to ysize do
    for xx from 1 to xsize do

      /* is there an edge element at (xx,yy) in edgemap1 */
      if edgemap1(xx,yy) = 1 then

        /* increment running total of edges */
        1 + maxtotal -> maxtotal;

        /* is there a matching edge at
           corresponding position of edgemap2 */
        if edgemap2(x+xx-1, y+yy-1) = 1 then
          /* yes so increment running total of matching edges */
          1 + hittotal -> hittotal;
        endif
      endif
    endfor
  endfor;

 /* compute goodness of match by dividing number of matching
    edges by number there would be in an ideal match */
  hittotal / maxtotal -> goodness;

enddefine;


define match_image(template, image, thrshld);
  vars xsize, ysize, xxsize, yysize, x, y, edgemap1, edgemap2;
  findedgels(template, 10) -> edgemap1;
  findedgels(image, 10) -> edgemap2;
  boundslist(template) --> [1 ?xxsize 1 ?yysize];
  boundslist(image) --> [1 ?xsize 1 ?ysize];
  for y from 1 to ysize-yysize+1 do
    for x from 1 to xsize-xxsize+1 do
      if match_at_xy(edgemap1, edgemap2, x, y) > thrshld then
        [There is a good match at position ^x ^y]=>
      endif
    endfor
  endfor
enddefine;


define findregions(image, ranges) -> regionmap;
  vars  a, b, table, i, label, xsize, ysize, x, y;

  /* build a table mapping grey-levels to labels */
  /* (2nd arg. to newarray initializes cells to 0) */
  newarray([0 99],0) -> table;
  0 -> label;
  foreach [?a ?b] in ranges do
    label + 1 -> label;
    for i from a to b do
      label -> table(i)
    endfor
  endforeach;

  /* construct a region map from image by assigning label
     from table to each pixel according to grey-level */

  boundslist(image) --> [1 ?xsize 1 ?ysize];
  newarray([1 ^xsize 1 ^ysize]) -> regionmap;
  for y from 1 to ysize do
    for x from 1 to xsize do
      table(image(x,y)) -> regionmap(x,y)
    endfor
  endfor;

enddefine;


/* define lists of grey-levels for Victoria image and Sign template */

vars victoriavals, signvals;

[
4  4  5  10 20 26 29 32 31 24 16 29 47 55 61 68 70 71 72 70 67 64 61 60 59 57 54 55 52 50 48 53 57 60 56 45 28 19 11 6  6  7  15 28 32 21 4  2  4  4  15 62 88 86 80 68 51 33 20 12 10 9  10 14
1  4  15 32 35 30 26 22 22 22 22 25 34 41 45 49 51 53 60 67 70 73 73 73 70 67 63 61 56 56 55 56 57 56 52 43 31 26 17 9  16 29 43 49 36 15 5  2  4  4  15 63 88 87 86 86 83 77 64 50 32 18 11 7
3  13 36 50 50 44 30 16 19 29 29 26 27 30 33 36 39 40 45 51 55 59 61 67 68 70 70 72 70 68 63 64 62 62 58 57 54 50 42 36 40 51 51 44 26 12 6  4  4  5  12 50 83 88 86 86 86 86 85 84 77 64 45 26
10 33 51 51 50 39 22 15 28 45 45 40 35 33 32 30 31 32 34 37 40 44 45 48 44 35 35 54 56 60 66 72 70 72 67 67 64 60 57 57 54 56 52 46 38 26 14 9  7  6  9  31 64 80 86 87 85 87 86 86 86 83 74 60
28 51 55 52 46 29 15 22 43 50 51 50 51 49 46 42 38 32 31 32 32 33 34 35 31 21 14 18 17 17 26 38 52 72 79 81 77 75 69 64 56 54 55 56 55 52 45 37 30 23 25 41 41 60 80 88 86 87 84 82 77 67 52 41
49 54 54 47 31 17 20 37 48 48 48 49 49 50 51 51 49 45 41 39 35 31 25 19 16 10 7  5  6  6  8  17 31 52 62 68 70 73 70 58 35 32 45 52 55 57 57 59 56 54 51 59 51 47 67 81 82 76 66 57 45 35 29 26
54 51 49 35 18 19 36 48 48 48 48 48 48 48 49 50 51 47 46 42 38 29 15 7  6  5  5  6  10 13 20 29 33 39 41 46 48 49 44 30 16 16 32 44 47 51 53 57 58 63 64 65 66 65 66 71 72 62 48 39 32 29 32 35
52 46 36 20 16 32 46 48 48 47 46 48 46 46 46 47 47 42 35 26 17 11 6  4  5  7  14 23 29 32 31 31 29 28 28 28 26 25 22 16 11 13 30 44 47 48 50 51 55 57 60 70 71 72 71 71 71 68 63 59 51 48 48 46
48 36 21 17 29 42 48 48 48 48 46 45 46 45 44 42 36 25 16 8  6  6  6  10 16 26 34 40 41 43 42 36 28 22 16 15 16 16 13 10 9  13 32 46 49 52 55 56 56 54 57 72 77 78 78 76 75 74 71 71 71 68 65 62
38 22 14 26 42 47 48 48 47 47 45 45 43 37 30 22 13 7  5  4  6  10 19 28 35 39 38 38 35 33 29 24 18 14 11 9  10 9  7  5  7  14 31 47 51 54 56 58 58 61 50 42 49 60 68 73 76 77 77 76 75 73 72 70
25 14 20 38 46 47 47 47 46 47 43 37 27 18 11 7  6  5  7  7  11 26 37 40 38 37 32 27 21 17 16 14 11 9  7  5  6  6  5  5  6  13 34 51 53 56 58 60 61 64 51 40 33 36 36 37 42 51 60 70 75 77 78 76
21 15 24 37 41 42 43 43 42 35 25 16 9  6  6  6  10 13 13 13 24 36 37 33 29 23 18 13 12 10 8  7  6  5  4  3  4  4  4  4  6  13 33 52 54 57 60 61 58 61 53 39 26 22 19 18 16 16 17 24 35 43 56 68
23 22 25 29 32 32 32 29 22 13 9  6  6  8  13 22 26 20 15 20 29 29 25 19 17 17 12 8  7  6  5  5  4  4  4  4  5  5  5  5  6  13 32 50 54 58 61 61 59 59 54 35 17 12 11 11 11 10 11 13 15 14 20 46
23 25 25 25 23 18 13 9  7  6  6  8  14 22 31 31 23 15 13 18 18 15 16 12 11 12 7  4  4  4  3  4  4  4  4  3  5  6  5  5  6  12 30 51 54 57 60 61 58 60 53 31 9  7  8  10 15 20 28 37 39 24 21 49
33 28 19 13 10 8  6  6  7  10 15 22 27 29 28 22 15 12 12 12 11 9  10 10 8  8  9  7  6  6  6  5  4  5  5  5  5  5  6  5  6  13 31 53 58 59 61 64 62 61 54 31 11 14 24 36 49 59 65 70 70 56 27 30
26 16 9  6  5  6  8  13 18 21 24 25 25 20 15 12 10 8  9  7  6  5  7  13 19 20 21 20 18 14 10 6  4  4  4  4  5  6  4  4  5  11 27 52 59 59 62 64 61 61 57 42 37 52 61 70 75 78 77 79 79 73 52 20
9  6  6  6  8  15 23 29 30 24 19 18 16 11 10 9  6  5  6  5  6  8  16 29 33 33 30 29 29 29 24 15 6  5  5  4  5  6  5  5  6  12 29 53 60 61 61 65 62 63 61 63 71 78 78 80 77 78 76 71 63 52 38 19
4  6  8  11 16 23 29 29 21 13 12 12 9  6  6  5  4  4  5  5  10 21 28 35 37 35 31 30 29 30 29 28 17 6  4  4  5  6  5  5  7  11 29 53 59 60 61 67 62 62 60 68 80 80 77 73 62 48 44 33 24 18 14 13
8  15 19 19 21 22 24 23 16 10 8  6  6  5  4  5  4  5  5  11 24 32 31 35 38 35 31 31 30 30 30 31 28 16 5  4  6  7  6  5  7  11 26 52 58 58 62 64 61 63 61 65 70 60 49 37 26 17 15 15 16 19 25 35
6  12 17 19 20 22 23 20 14 8  6  4  4  4  4  4  4  4  8  23 33 32 31 35 38 40 39 36 34 31 29 32 30 25 12 3  6  7  6  4  6  10 23 50 57 60 63 65 60 64 59 47 32 22 16 15 14 16 21 30 41 53 64 70
3  5  6  7  9  10 14 16 14 9  5  5  5  4  6  4  5  7  17 32 34 32 34 49 63 73 71 65 52 40 32 32 31 29 21 6  5  6  7  6  6  11 23 49 58 61 62 65 63 64 61 43 17 13 19 27 38 49 58 69 74 78 80 79
3  3  4  4  4  4  4  5  10 9  4  3  4  3  5  5  4  12 27 32 32 35 55 80 86 87 87 86 80 64 42 32 29 28 26 13 4  6  6  5  6  10 22 50 58 60 60 62 60 64 61 44 22 29 52 67 72 76 77 79 77 74 69 60
6  6  6  4  4  3  4  5  8  10 6  3  5  4  5  5  8  20 32 33 36 60 84 91 88 90 90 90 87 85 67 41 29 28 27 20 7  5  7  5  7  11 22 50 59 61 61 64 62 64 62 54 32 35 64 79 80 76 70 63 54 44 35 29
13 12 10 8  7  6  5  5  8  10 5  4  4  4  5  6  12 26 32 34 54 84 89 91 86 89 89 89 86 89 83 61 34 27 28 25 12 5  6  7  7  9  20 47 58 63 61 63 62 64 62 60 54 29 34 56 58 49 36 29 22 20 19 16
19 19 18 17 16 13 10 10 11 10 5  4  5  4  5  8  19 30 31 45 77 93 90 91 87 89 89 90 89 91 87 80 49 29 29 30 20 11 13 12 9  9  21 47 61 61 61 65 62 64 63 58 51 34 15 22 27 22 19 18 16 12 10 8
19 19 19 18 19 17 16 15 13 9  5  4  4  4  6  10 24 31 35 65 90 93 89 91 87 90 90 92 90 91 89 86 63 34 29 30 26 23 22 16 9  9  19 47 60 62 61 62 64 65 61 51 29 17 16 16 15 13 11 9  7  5  5  4
19 19 19 19 19 18 17 17 15 9  5  4  5  5  6  15 28 31 44 81 93 93 90 90 89 91 90 92 89 88 80 71 51 32 27 32 42 51 36 18 9  8  19 46 61 63 62 62 63 65 61 50 23 11 11 10 8  7  5  5  5  5  4  3
15 17 17 16 18 17 17 16 14 9  5  4  4  4  8  19 30 33 55 90 90 91 90 90 88 89 85 79 68 57 46 38 38 45 45 54 76 78 57 25 7  6  17 44 59 62 62 62 61 63 60 48 17 3  5  5  4  4  4  3  4  4  3  3
15 16 16 15 15 16 16 15 14 10 5  4  5  5  9  23 31 38 72 95 91 90 87 83 75 64 54 45 41 44 45 49 51 81 77 73 87 71 62 38 7  7  18 44 58 64 63 62 59 62 62 48 15 1  4  4  5  4  4  4  5  4  5  6
19 20 20 18 17 16 16 13 13 10 5  4  5  4  11 27 31 45 80 90 80 71 60 50 45 44 50 52 62 77 70 71 57 89 89 87 88 79 64 32 6  6  16 43 58 62 60 63 61 62 63 48 15 0  3  3  4  3  3  3  4  3  5  10
21 25 27 25 24 23 21 19 16 10 5  5  5  5  15 29 32 44 63 58 47 44 43 44 63 73 80 86 80 75 84 83 67 91 80 75 82 74 52 21 9  7  17 42 58 64 61 64 62 64 64 51 17 1  3  4  4  4  4  4  6  5  5  13
21 26 28 27 26 26 25 22 19 10 5  5  6  8  16 26 26 29 38 44 57 70 74 73 80 86 90 93 83 71 84 85 79 74 57 43 48 39 27 17 9  6  16 40 58 65 62 62 62 63 63 51 18 1  3  4  4  3  4  4  4  4  4  7
24 28 30 30 29 28 28 26 21 11 6  8  12 15 21 30 37 60 69 68 84 86 88 89 79 86 93 90 83 81 70 55 62 48 36 36 31 21 16 10 7  7  16 40 60 67 64 63 61 64 64 54 20 1  3  5  4  3  5  4  4  3  4  3
26 28 30 29 30 30 29 27 22 11 6  11 20 28 51 67 66 87 80 83 90 86 89 89 80 85 80 64 59 57 52 49 54 47 37 37 29 10 7  8  6  7  15 38 61 65 63 63 59 63 64 55 22 1  3  4  4  3  4  3  4  3  3  3
26 27 29 29 30 31 31 28 25 13 6  15 44 51 74 93 87 90 80 88 91 84 79 70 66 64 53 51 55 66 73 83 80 52 33 35 27 9  6  8  6  9  14 38 62 67 64 64 61 63 64 57 24 1  3  4  5  3  4  4  4  3  4  4
22 22 22 24 25 26 29 27 23 11 7  15 54 64 78 92 85 89 86 75 69 66 51 51 57 64 70 80 86 93 92 94 82 48 29 34 24 6  6  7  6  8  14 34 61 67 61 64 61 62 65 55 24 1  3  4  4  4  5  6  6  3  3  3
27 25 25 24 24 23 23 24 21 11 7  14 51 79 82 73 56 64 61 50 50 61 66 78 85 91 92 96 95 97 95 96 75 38 30 33 22 8  6  7  6  7  15 33 59 65 62 64 63 62 64 55 25 2  3  6  8  10 11 12 10 4  4  4
32 30 32 30 29 29 29 26 22 12 6  11 31 58 56 38 34 40 57 74 81 89 92 96 94 96 95 96 96 98 96 92 64 33 31 30 17 6  5  7  6  6  13 32 55 64 61 63 62 63 63 55 29 4  5  7  10 12 13 13 10 5  4  4
34 32 35 34 35 33 35 33 26 13 7  10 17 22 27 33 37 48 79 97 96 97 95 97 96 97 96 97 97 98 95 83 47 31 32 26 10 5  6  7  6  6  13 31 54 64 63 64 62 64 62 58 33 8  9  11 11 11 11 11 10 6  4  6
35 34 35 36 36 35 36 35 28 13 6  8  11 11 19 35 37 40 70 97 97 98 95 98 95 98 95 96 96 96 89 67 35 32 32 19 6  4  6  6  6  5  12 29 52 64 62 62 63 64 64 57 33 10 10 13 16 14 15 16 13 6  4  6
37 37 37 37 38 37 38 37 31 14 6  6  7  6  12 32 36 37 55 89 98 99 96 97 96 99 95 98 96 96 77 43 33 34 28 13 5  6  6  6  6  6  12 27 52 65 61 61 62 64 63 56 33 9  12 16 19 19 20 22 20 8  4  10
34 34 36 37 37 36 38 38 32 14 6  5  6  5  9  26 37 36 42 70 93 98 96 97 96 98 95 97 94 82 50 34 35 33 21 6  5  6  6  5  6  6  11 26 51 64 61 60 61 61 62 57 31 5  9  17 20 22 21 25 23 10 3  11
28 28 30 31 32 34 36 36 31 14 6  6  5  6  7  18 35 37 37 49 74 94 94 99 96 97 95 95 79 52 35 36 35 28 13 4  6  6  6  6  6  6  11 26 49 64 64 62 61 64 63 58 31 5  6  13 21 24 25 28 26 12 3  11
30 28 28 27 26 27 29 29 26 13 6  5  6  5  6  11 29 36 36 38 48 70 83 91 91 90 83 68 48 35 33 35 32 18 6  3  5  5  6  6  6  6  11 24 48 63 63 62 61 62 61 57 33 5  3  6  14 22 25 29 26 11 3  10
39 37 38 35 33 34 33 31 26 15 6  6  6  4  6  7  19 35 36 36 40 45 52 63 66 64 53 42 37 37 34 35 24 9  5  5  5  6  5  7  7  6  11 24 48 62 64 62 61 62 62 58 38 11 4  4  7  11 16 21 21 11 3  10
40 37 38 39 37 39 40 38 32 16 6  5  5  5  6  5  9  25 37 37 40 41 38 39 44 44 38 37 36 36 34 26 12 5  5  5  5  5  6  6  7  6  10 22 48 64 61 61 60 61 62 58 41 18 9  5  5  5  7  8  10 7  3  6
41 38 38 38 38 39 41 40 35 17 6  6  6  6  6  6  6  13 29 38 39 38 37 38 39 41 37 36 36 34 25 14 6  6  6  6  6  8  8  9  11 9  10 20 48 69 62 60 59 59 61 59 43 21 16 13 6  4  6  5  6  4  5  5
40 38 38 38 37 39 39 38 36 18 6  6  6  5  6  5  5  6  13 28 37 38 36 36 38 41 35 34 34 23 10 7  7  6  7  7  7  9  10 11 11 11 10 19 45 65 60 57 58 59 59 57 42 22 18 17 13 6  4  4  5  4  4  5
40 38 41 40 40 39 39 38 36 19 7  6  6  5  6  6  6  5  7  11 22 34 35 34 37 39 33 28 23 14 6  10 10 10 10 9  10 10 11 13 12 11 11 15 32 46 46 46 50 53 54 53 42 25 18 19 20 16 7  3  5  5  4  6
34 35 38 37 36 39 39 39 36 20 6  5  5  5  5  5  5  4  5  6  8  16 21 20 24 28 20 14 15 13 8  10 10 10 10 10 11 11 11 10 10 10 10 10 13 16 17 20 22 26 30 34 32 24 19 20 23 22 10 4  4  4  4  4
32 32 32 33 32 33 35 35 35 20 7  5  6  5  6  5  6  5  6  6  9  15 19 13 15 20 14 11 13 13 10 11 13 11 10 10 10 10 10 10 10 10 10 11 11 10 10 9  8  12 17 20 23 21 19 20 22 30 30 13 3  4  5  5
39 38 39 37 37 35 34 34 29 18 8  6  5  5  6  5  6  6  7  8  10 15 17 10 12 14 11 10 12 11 8  9  9  7  8  8  9  9  10 9  10 10 11 13 16 16 12 5  6  10 22 25 25 24 23 23 27 53 62 23 1  4  5  5
45 45 45 46 45 45 45 44 38 21 8  6  6  7  7  7  8  9  10 10 10 12 11 11 12 12 10 9  7  7  6  5  6  7  8  9  9  10 12 16 18 21 25 27 27 26 16 3  4  11 24 29 28 29 29 26 22 22 26 13 5  6  9  12
45 45 47 46 47 47 46 45 42 23 8  8  9  8  8  9  9  10 10 10 10 9  10 10 10 9  7  4  4  5  5  5  5  7  10 8  11 18 22 28 29 32 33 32 30 28 16 3  4  10 23 29 29 29 27 19 10 6  6  6  10 22 38 38
46 45 47 46 48 47 47 45 42 24 11 10 11 10 10 11 10 10 10 10 10 10 10 10 9  9  9  7  6  6  6  6  7  16 19 11 12 23 26 32 30 31 32 32 31 28 17 4  4  9  22 29 29 29 19 8  6  5  6  6  14 43 66 53
45 45 47 45 46 47 45 43 41 25 10 11 10 10 10 9  9  9  10 10 9  10 10 11 12 14 14 9  6  8  9  11 16 24 26 14 10 22 25 28 26 26 26 25 26 23 13 3  4  5  13 23 25 19 9  4  5  7  13 9  9  15 27 26
42 44 45 45 46 45 45 45 44 33 20 17 17 15 13 13 12 10 10 10 9  16 26 32 33 27 19 13 10 11 13 23 34 40 35 19 12 26 33 36 30 29 31 30 27 22 13 4  4  5  6  10 11 9  6  6  6  24 55 26 18 51 63 47
35 34 35 35 37 37 40 40 41 41 38 35 35 34 30 27 24 18 8  6  7  13 24 30 32 27 14 12 13 12 15 29 42 47 42 23 11 27 35 38 35 33 34 34 32 28 17 3  3  4  4  5  6  4  5  6  13 40 73 50 38 85 89 64
42 38 38 37 37 35 34 34 34 35 35 35 38 38 38 38 36 27 9  4  7  8  8  9  11 15 16 14 13 14 16 30 46 49 44 26 11 26 34 37 34 34 36 36 34 31 19 5  3  4  5  5  5  5  6  12 29 45 75 79 69 66 49 43
46 45 45 45 46 44 42 38 38 41 39 38 38 38 37 36 33 20 10 10 10 14 14 11 22 30 18 11 12 13 13 26 42 48 43 24 10 25 32 36 33 34 37 37 33 31 19 4  3  3  5  4  5  5  8  23 40 42 52 60 56 43 19 21
48 47 47 49 50 48 46 45 45 47 47 48 47 48 47 45 42 38 42 42 40 60 40 35 77 68 22 8  11 13 14 27 42 45 39 21 10 25 33 36 35 37 39 37 34 31 19 6  3  5  6  5  6  6  16 36 42 42 45 45 42 34 30 28
49 47 46 47 48 48 47 46 46 48 48 49 50 49 48 48 48 55 74 70 71 76 51 75 95 81 31 7  13 13 15 28 44 46 42 19 8  21 27 30 29 31 35 34 31 28 19 5  3  4  5  5  5  8  27 42 42 43 45 44 36 27 37 51
50 48 47 47 48 49 50 48 48 48 51 50 51 51 51 51 50 47 41 36 45 46 39 67 73 67 31 11 14 15 16 28 45 47 42 24 6  19 24 28 28 28 29 28 25 23 15 6  4  5  6  5  6  15 35 43 42 42 34 25 38 37 45 77
47 48 47 46 49 50 49 50 48 48 51 51 51 50 51 51 48 47 31 12 18 29 26 29 29 32 27 16 13 16 16 26 43 46 43 26 7  18 29 36 35 35 35 34 30 25 16 5  4  5  5  4  7  17 30 33 35 36 45 64 87 86 71 72
] -> victoriavals;


[
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 88 86 94 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 86 68 54 50 50 50 55 75 95 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 97 66 50 50 50 50 50 50 50 50 52 81 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 97 61 50 50 50 50 50 50 50 50 50 50 50 69 98 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 61 50 50 50 50 50 50 50 50 50 50 50 50 50 75 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 75 50 50 50 50 50 50 50 50 50 50 50 50 50 50 50 87 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 91 52 50 50 50 50 50 54 68 68 64 51 50 50 50 50 50 57 98 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 70 50 50 50 50 51 79 99 99 99 99 95 68 50 50 50 50 50 82 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 95 51 50 50 50 51 87 99 99 99 99 99 99 99 74 50 50 50 50 58 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 74 50 50 50 50 80 99 99 99 99 99 99 99 99 99 68 50 50 50 50 88 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 59 50 50 50 63 99 99 99 99 99 99 99 99 99 99 94 52 50 50 50 72 77 48 64 99 99 99 99 99
99 99 99 99 99 99 94 50 50 50 50 85 99 99 99 99 99 99 99 99 99 99 99 71 46 33 18 4  0  0  43 99 99 99 99 99
99 99 99 99 99 99 85 50 50 50 55 99 99 99 99 99 99 99 99 99 99 81 52 19 1  0  0  0  0  0  27 99 99 99 99 99
99 99 99 99 99 99 75 50 50 50 69 99 99 99 99 99 99 95 69 40 11 0  0  0  0  0  0  0  0  0  17 99 99 99 99 99
99 99 99 99 99 99 68 50 50 50 79 99 99 99 85 56 28 3  0  0  0  0  0  0  0  0  0  0  0  0  2  99 99 99 99 99
99 99 99 99 99 99 61 50 50 50 83 73 44 15 0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  90 99 99 99 99
99 99 99 99 99 99 59 45 30 16 3  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  1  78 99 99 99 99
99 99 99 98 77 48 13 0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  3  28 63 91 99 99 99 99 99
99 99 99 50 0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  11 23 38 49 77 99 99 99 99 99 99 99
99 99 99 59 0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  4  30 59 88 68 50 50 50 82 99 99 99 99 99 99 99
99 99 99 73 0  0  0  0  0  0  0  0  0  0  0  0  0  14 43 71 96 99 99 99 59 50 50 50 87 99 99 99 99 99 99 99
99 99 99 85 0  0  0  0  0  0  0  0  0  2  26 55 84 99 99 99 99 99 99 96 50 50 50 50 95 99 99 99 99 99 99 99
99 99 99 99 0  0  0  0  0  0  5  34 68 94 99 99 99 99 99 99 99 99 99 76 50 50 50 57 99 99 99 99 99 99 99 99
99 99 99 99 12 0  1  11 26 40 50 75 99 99 99 99 99 99 99 99 99 99 98 55 50 50 50 68 99 99 99 99 99 99 99 99
99 99 99 99 53 63 91 64 50 50 50 52 95 99 99 99 99 99 99 99 99 99 78 50 50 50 50 87 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 82 50 50 50 50 64 98 99 99 99 99 99 99 99 89 51 50 50 50 56 98 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 97 55 50 50 50 50 65 97 99 99 99 99 99 86 53 50 50 50 50 78 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 78 50 50 50 50 50 55 78 93 95 84 70 50 50 50 50 50 54 97 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 60 50 50 50 50 50 50 50 50 50 50 50 50 50 50 50 83 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 89 51 50 50 50 50 50 50 50 50 50 50 50 50 50 70 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 85 51 50 50 50 50 50 50 50 50 50 50 50 63 98 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 84 51 50 50 50 50 50 50 50 50 50 64 97 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 91 62 50 50 50 50 50 50 52 80 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 98 89 74 67 66 71 86 98 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99 99
] -> signvals;
