/* --- Copyright University of Sussex 1995. All rights reserved. ----------
 > File:            $popvision/lib/appellipse.p
 > Purpose:         Apply a procedure to all points in or on an ellipse
 > Author:          David S Young, Mar  3 1995
 > Documentation:   HELP * APPELLIPSE
 > Related Files:   See "uses" line
 */


section;

uses boundslist_utils

define ellipsematrix(x0, y0, a, b, phi) -> (x0, y0, t11, t12, t22);
    ;;; Given ellipse params, returns centre and control matrix.
    lvars x0, y0, a, b, phi, t11, t12, t22;
    lvars
    ;;; Ellipse parameters as inverse major and minor axis vectors
        cosPhi = cos(phi), sinPhi = sin(phi),
        (a1, a2) = ( cosPhi/a, sinPhi/a),
        (b1, b2) = (-sinPhi/b, cosPhi/b);
    ;;; Control matrix for ellipse in original coords.
    a1 * a1 + b1 * b1 -> t11;
    a2 * a2 + b2 * b2 -> t22;
    a1 * a2 + b1 * b2 -> t12;
enddefine;

define ellipse_map(x0, y0, t11, t12, t22, region, bounds)
        -> (x0, y0, t11, t12, t22);
    lvars x0, y0, t11, t12, t22, region, bounds;
    lvars
        (Map, ) = region_map(region, bounds),
        (Scale, ) = region_scale(region, bounds),
        (xscale, yscale) = Scale(1, 1);
    ;;; Convert matrix to array coords (needs to be done on matrix not vectors)
    t11 / (xscale * xscale) -> t11;
    t22 / (yscale * yscale) -> t22;
    t12 / (xscale * yscale) -> t12;
    ;;; Convert origin
    Map(x0, y0) -> (x0, y0)
enddefine;

define ellipsequation(t11, t12, t22) -> (k1, k2, k3);
    ;;; Given control matrix, returns coefficients of quadratic for
    ;;; x as a function of y.
    lvars t11, t12, t22,
        k3 = 1.0 / t11,
        k1 = - t12 * k3,
        k2 = k1 * k1 - t22 * k3;
enddefine;

define appellipse(x0, y0, a, b, phi, bounds, proc);
    ;;; Calls proc(x,y) for each pixel x,y in the ellipse, if it is
    ;;; also within the region given by bounds.

    ;;; Does shorter axis fastest to avoid leaving gaps,
    ;;; and does not go outside the specified bounds.

    ;;; Main loop variables (declare first for register allocation)
    lvars x, y;

    lvars x0, y0, a, b, phi, region = false, bounds, procedure proc;

    ;;; Get optional region argument
    if phi.islist then
        (x0, y0, a, b, phi) -> (x0, y0, a, b, phi, region)
    endif;

    ;;; bounds of region for checking later
    if bounds.isarray then boundslist(bounds) -> bounds endif;
    lvars (X0, X1, Y0, Y1) = explode(bounds);
    (min(X0, X1), max(X0, X1)) -> (X0, X1);
    (min(Y0, Y1), max(Y0, Y1)) -> (Y0, Y1);

    ;;; Convert to matrix equation
    lvars t11, t22, t12;
    ellipsematrix(x0, y0, a, b, phi) -> (x0, y0, t11, t12, t22);

    ;;; Convert to image coords
    if region then
        ellipse_map(x0, y0, t11, t12, t22, region, bounds)
            -> (x0, y0, t11, t12, t22)
    endif;

    ;;; Loop over x or y depending on whether extent is greater in x or
    ;;; y direction. Repeat the code to avoid test in inner loop.

    if t11 > t22 then

        ;;; Constants in equations for x as a function of y
        lvars t1, t2, yoff,
            (k1, k2, k3) = ellipsequation(t11, t12, t22),
            vmax = sqrt(t11 / (t11*t22 - t12*t12)),
            ystart = max(Y0, round(y0 - vmax)),
            yend = min(Y1, round(y0 + vmax));

        for y from ystart to yend do
            y - y0 -> yoff;
            k2 * yoff * yoff + k3 -> t2;
            if t2 >= 0 then     ;;; rounding errors can give t2 < 0
                x0 + k1 * yoff -> t1;
                sqrt(t2) -> t2;
                for x from max(X0, round(t1 - t2)) to min(X1, round(t1 + t2)) do
                    proc(x, y)
                endfor
            endif
        endfor

    else

        ;;; Constants in equations for y as a function of x
        lvars t1, t2, xoff,
            (k1, k2, k3) = ellipsequation(t22, t12, t11),
            vmax = sqrt(t22 / (t11*t22 - t12*t12)),
            xstart = max(X0, round(x0 - vmax)),
            xend = min(X1, round(x0 + vmax));

        for x from xstart to xend do
            x - x0 -> xoff;
            k2 * xoff * xoff + k3 -> t2;
            if t2 >= 0 then
                y0 + k1 * xoff -> t1;
                sqrt(t2) -> t2;
                for y from max(Y0, round(t1 - t2)) to min(Y1, round(t1 + t2)) do
                    proc(x, y)
                endfor
            endif
        endfor

    endif

enddefine;

define appellipse_rim(x0, y0, a, b, phi, bounds, proc);
    ;;; Calls proc(x,y) for each pixel x,y on the boundary of the ellipse.
    ;;; Arguments as for appellispe.

    lvars x, y;
    lvars x0, y0, a, b, phi, region = false, bounds, procedure proc;

    ;;; Get optional region argument
    if phi.islist then
        (x0, y0, a, b, phi) -> (x0, y0, a, b, phi, region)
    endif;

    ;;; bounds of region for checking later
    if bounds.isarray then boundslist(bounds) -> bounds endif;
    lvars (X0, X1, Y0, Y1) = explode(bounds);
    (min(X0, X1), max(X0, X1)) -> (X0, X1);
    (min(Y0, Y1), max(Y0, Y1)) -> (Y0, Y1);

    ;;; Get control matrix
    lvars t11, t22, t12;
    ellipsematrix(x0, y0, a, b, phi) -> (x0, y0, t11, t12, t22);

    ;;; Convert to image coords
    if region then
        ellipse_map(x0, y0, t11, t12, t22, region, bounds)
            -> (x0, y0, t11, t12, t22)
    endif;

    ;;; Find points of unit slope - switch over at these from
    ;;; incrementing x to incrementing y.
    lvars
        D = t11 * t22 - t12 * t12,
        T = t11 + t22,
        Twot12 = 2*t12,

        denom1 = sqrt(D * (T - Twot12)),
        Xt1 = (t22 - t12) / denom1,
        Yt1 = (t11 - t12) / denom1,
        denom2 = sqrt(D * (T + Twot12)),
        Xt2 = - (t22 + t12) / denom2,
        Yt2 = (t11 + t12) / denom2,

    ;;; Constants in equations for x as a function of y and vice versa
        (k1x, k2x, k3x) = ellipsequation(t11, t12, t22),
        (k1y, k2y, k3y) = ellipsequation(t22, t12, t11),
        xoff, yoff, t1, t2;

    ;;; Do each of the four quadrants separately. Could save some
    ;;; work by using symmetry, but this would create rounding errors
    ;;; if x0 and y0 not integers. Ranges reduced to avoid overlap.

    for x from max(X0, round(x0+Xt2)) to min(X1, round(x0+Xt1)-1) do
        x - x0 -> xoff;
        y0 + k1y * xoff -> t1;
        k2y * xoff * xoff + k3y -> t2;
        if t2 > 0 then
            round(t1 + sqrt(t2)) -> y;
            if y >= Y0 and y <= Y1 then
                proc(x, y)
            endif
        endif;
    endfor;

    for y from min(Y1, round(y0+Yt1)) by -1 to max(Y0, round(y0-Yt2)+1) do
        y - y0 -> yoff;
        x0 + k1x * yoff -> t1;
        k2x * yoff * yoff + k3x -> t2;
        if t2 > 0 then
            round(t1 + sqrt(t2)) -> x;
            if x >= X0 and x <= X1 then
                proc(x, y)
            endif
        endif;
    endfor;

    for x from min(X1, round(x0-Xt2)) by -1 to max(X0, round(x0-Xt1)+1) do
        x - x0 -> xoff;
        y0 + k1y * xoff -> t1;
        k2y * xoff * xoff + k3y -> t2;
        if t2 > 0 then
            round(t1 - sqrt(t2)) -> y;
            if y >= Y0 and y <= Y1 then
                proc(x, y)
            endif
        endif;
    endfor;

    for y from max(Y0, round(y0-Yt1)) to min(Y1, round(y0+Yt2)-1) do
        y - y0 -> yoff;
        x0 + k1x * yoff -> t1;
        k2x * yoff * yoff + k3x -> t2;
        if t2 > 0 then
            round(t1 - sqrt(t2)) -> x;
            if x >= X0 and x <= X1 then
                proc(x, y)
            endif
        endif;
    endfor;

enddefine;

endsection;
