/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            $popvision/lib/convolve_2d.p
 > Purpose:         Convolve a 2-D array with a 2-D mask (non-circular)
 > Author:          David S Young, Jun  3 1992 (see revisions)
 > Documentation:   HELP CONVOLVE_2D
 > Related Files:   convolve_2d.c
 */

compile_mode:pop11 +strict;

section;

uses popvision
uses objectfile
uses boundslist_utils
uses newsfloatarray

lconstant macro extname = 'convolve_2d',
    obfile = objectfile(extname);

unless obfile then
    mishap(0, 'Cannot find object file')
endunless;

exload extname [^obfile]
    lconstant convolve_2d_f(15)
endexload;

define lconstant float_2d_array_check(arr) /* -> arr */;
    ;;; Checks an array is 2d.
    ;;; Checks the array has a float array vector, and if not,
    ;;; constructs a new array of that type, with the data copied.
    lvars arr;
    unless length(boundslist(arr)) == 4 then
        mishap(arr, 1, 'Need 2-D array')
    endunless;
    if arr.issfloatarray then
        arr
    else
        newsfloatarray(boundslist(arr), arr)
    endif
enddefine;

define procedure convolve_2d(arrin, mask, arrout, region) -> arrout;
    lvars arrin, mask, arrout, region;

    lvars in_x0, in_x1, in_y0, in_y1,
        ms_x0, ms_x1, ms_y0, ms_y1,
        rg_x0, rg_x1, rg_y0, rg_y1,
        ot_x0, ot_x1, ot_y0, ot_y1;

    ;;; Coerce arrays to packed float, check they are 2-D, and
    ;;; get the array bounds into separate variables.
    float_2d_array_check(arrin) -> arrin;
    explode(boundslist(arrin)) -> (in_x0, in_x1, in_y0, in_y1);
    float_2d_array_check(mask) -> mask;
    explode(boundslist(mask)) -> (ms_x0, ms_x1, ms_y0, ms_y1);
    if arrout.islist then
        newsfloatarray(tl(arrout), hd(arrout)) -> arrout
    endif;
    if arrout then
        float_2d_array_check(arrout) -> arrout;
        explode(boundslist(arrout)) -> (ot_x0, ot_x1, ot_y0, ot_y1);
    endif;

    ;;; Construct the output region
    if  region.islist then  ;;; it is explicit as an argument
        ;;; so check it is legal
        region_inclusion_check(
            [% in_x0+ms_x1, in_x1+ms_x0, in_y0+ms_y1, in_y1+ms_y0 %], region);
        if arrout then region_inclusion_check(arrout, region) endif;

        explode(region) -> (rg_x0, rg_x1, rg_y0, rg_y1)

    else    ;;; get region from input array and mask
        in_x0 + ms_x1 -> rg_x0;
        in_x1 + ms_x0 -> rg_x1;
        in_y0 + ms_y1 -> rg_y0;
        in_y1 + ms_y0 -> rg_y1;
        if arrout then  ;;; take the intersection
            max(ot_x0, rg_x0) -> rg_x0;
            min(ot_x1, rg_x1) -> rg_x1;
            max(ot_y0, rg_y0) -> rg_y0;
            min(ot_y1, rg_y1) -> rg_y1;
        endif;

        [% rg_x0, rg_x1, rg_y0, rg_y1 %] -> region
    endif;

    ;;; Mishap if either the mask was bigger than the input image,
    ;;; or the output region is wholly outside the output array
    if rg_x0 > rg_x1 or rg_y0 > rg_y1 then
        mishap(arrin, mask, region, arrout, 4,
            'No valid output region')
    endif;

    ;;; Construct the output array if need be
    unless arrout then
        newsfloatarray(region) -> arrout;
        (rg_x0, rg_x1, rg_y0, rg_y1) -> (ot_x0, ot_x1, ot_y0, ot_y1);
    endunless;

    ;;; All seems ready do the convolution
    exacc convolve_2d_f(
        arrayvector(arrin),             ;;;  in_2d
        in_x1 - in_x0 + 1,              ;;;  in_xsize
        rg_x0 - in_x0,                  ;;;  in_xstart
        rg_y0 - in_y0,                  ;;;  in_ystart
        arrayvector(mask),              ;;;  mask_2d
        ms_x1 - ms_x0 + 1,              ;;;  mask_xsize
        ms_y1 - ms_y0 + 1,              ;;;  mask_ysize
        -ms_x0,                         ;;;  mask_xorig
        -ms_y0,                         ;;;  mask_yorig
        arrayvector(arrout),            ;;;  out_2d
        ot_x1 - ot_x0 + 1,              ;;;  out_xsize
        rg_x0 - ot_x0,                  ;;;  out_xstart
        rg_x1 - ot_x0,                  ;;;  out_xend
        rg_y0 - ot_y0,                  ;;;  out_ystart
        rg_y1 - ot_y0                   ;;;  out_yend
    )

enddefine;

endsection;

/* --- Revision History ---------------------------------------------------
--- David S Young, Jul 13 1993
        Changed -newfloatarray- to -newsfloatarray- and -isfloatarray- to
        -issfloatarray- to avoid name clash with *VEC_MAT package.
--- David S Young, Jun 19 1992
        Changed to test arrays using -isfloatarray-
 */
