/* --- Copyright University of Sussex 1995. All rights reserved. ----------
 > File:            $popvision/lib/convolve_gauss_2d.p
 > Purpose:         Convolve 2-D arrays with Gaussian masks
 > Author:          David S Young, Jun  3 1992 (see revisions)
 > Documentation:   HELP *CONVOLVE_GAUSS_2D
 > Related Files:   LIB *CONVOLVE_2D, LIB *GAUSSMASK
 */

compile_mode:pop11 +strict;

section;

uses popvision
uses boundslist_utils
uses arraysample
uses convolve_2d
uses gaussmask
uses newsfloatarray
uses float_arrayprocs

define convolve_gauss_2d(image, operations /*, arrout*/) -> arrout;
    lvars image, operations, arrout = false;

    ;;; Sort out args, including optional 3rd arg
    if not(operations) or operations.isarray then
        (image, operations) -> (image, operations, arrout)
    endif;
    lconstant zerolist = [0 0];
    if operations.isreal then
        [smoothx ^operations smoothy ^operations] -> operations
    endif;

    ;;; tags for getting work arrays
    lconstant
        wk1tag = consref("convolve_gauss_2d"),
        wk2tag = consref("convolve_gauss_2d");

    if operations == [] then ;;; no operations but need to copy data
        if arrout then
            lvars region = region_intersect(image, arrout);
            arraysample(image, region, arrout, region, "nearest") ->
        else
            copy(image) -> arrout
        endif

    else
        lvars op, sigma, mask, wk1 = wk1tag, wk2 = wk2tag;
        until operations == [] do
            dest(operations) -> (op, operations);
            dest(operations) -> (sigma, operations);
            switchon op ==
            case "smoothx" then
                gaussmask(sigma) -> mask;
                ;;; Make a 2-D array with no y extent
                newanyarray(boundslist(mask) <> zerolist, mask) -> mask;
            case "smoothy" then
                gaussmask(sigma) -> mask;
                newanyarray(zerolist <> boundslist(mask), mask) -> mask;
            case "diffx" then
                diffgaussmask(sigma) -> mask;
                ;;; Make a 2-D array with no y extent
                newanyarray(boundslist(mask) <> zerolist, mask) -> mask;
            case "diffy" then
                diffgaussmask(sigma) -> mask;
                newanyarray(zerolist <> boundslist(mask), mask) -> mask;
            else
                mishap(op, 1, 'Unknown operation')
            endswitchon;

            ;;; Convolve 2-D handles what are really 1-D masks efficiently
            if operations == [] then
                ;;; last operation
                convolve_2d(image, mask, arrout, false) -> arrout
            else
                convolve_2d(image, mask,
                    oldsfloatarray(wk1, region_conv_output(image, mask)),
                    false)
                    -> image;
                ;;; use other work array next time
                (wk1, wk2) -> (wk2, wk1)
            endif
        enduntil
    endif
enddefine;

global vars convolve_dog_ratio = 1.6;

define convolve_dog_2d(image, sigma /*, arrout*/) -> arrout;
    lvars image, sigma, arrout = false;

    unless sigma.isreal then
        (image, sigma) -> (image, sigma, arrout)
    endunless;

    lconstant
        innertag = consref("inner"),
        outertag = consref("outer");
    lvars
        outersigma = convolve_dog_ratio * sigma,
        maxlen = gaussmask_limit(max(sigma, outersigma)),
        outbounds = region_expand(image, -maxlen),
        innerimage = convolve_gauss_2d(image, sigma,
        oldsfloatarray(innertag, outbounds)),
        outerimage = convolve_gauss_2d(image, outersigma,
        oldsfloatarray(outertag, outbounds));
    unless arrout then
        newsfloatarray(outbounds) -> arrout
    endunless;
    float_arraydiff(innerimage, outerimage, arrout) -> arrout
enddefine;

endsection;

/* --- Revision History ---------------------------------------------------
--- David S Young, Jan 27 1995
        Included copy operation for case where no operations specified
--- David S Young, Nov 16 1994
        Changed to reduce garbage creation - allows output arrays to
        be supplied and uses oldsfloatarray for work arrays.
--- David S Young, Jul 13 1993
        Changed -newfloatarray- to -newsfloatarray- and -isfloatarray- to
        -issfloatarray- to avoid name clash with *VEC_MAT package.
--- David S Young, Nov 26 1992
        Changed to use *GAUSSMASK
 */
