/* --- Copyright University of Sussex 1994. All rights reserved. ----------
 > File:            $popvision/lib/horn_schunck.p
 > Purpose:         Estimate optic flow vectors from grey-level gradients
 > Author:          David S Young, Apr  8 1994 (see revisions)
 > Documentation:   HELP * HORN_SCHUNCK
 > Related Files:   See "uses" below
 */

compile_mode:pop11 +strict;

section;

uses popvision
uses convolve_2d
uses gaussmask
uses array_reflect
uses boundslist_utils
uses float_arrayprocs

lvars
    work_array1 = newsfloatarray([0 0 0 0]),       ;;; to avoid garbage
    work_array2 = newsfloatarray([0 0 0 0]);

define lconstant set_work_arrays(region); lvars region;
    ;;; Ensure that the work arrays are the right size
    unless boundslist(work_array1) = region then
        newsfloatarray(region) -> work_array1;
        newsfloatarray(region) -> work_array2
    endunless
enddefine;

define lconstant gauss_smooth(arrin, sigma, arrout);
    lvars arrin, sigma, arrout;
    ;;; Updating version of convolve_gauss_2d.  Also extends output
    ;;; to limits of input array by reflection

    lconstant zerobounds = [0 0];
    lvars
        mask = gaussmask(sigma),
        hmask = newanyarray(boundslist(mask) <> zerobounds, mask),
        vmask = newanyarray(zerobounds <> boundslist(mask), mask),
        midbounds = region_conv_output(arrin, hmask),
        outbounds = region_conv_output(midbounds, vmask);

    ;;; Do horizontal convolution
    convolve_2d(arrin, hmask, work_array1, midbounds) -> ;

    ;;; Do vertical convolution
    convolve_2d(work_array1, vmask, arrout, outbounds) -> ;

    ;;; Extend by reflection
    array_reflect(arrout, outbounds, arrout, boundslist(arrin)) -> ;
enddefine;

define lconstant gradients(im1, im2, sigma) -> (Ex, Ey, Et);
    lvars im1, im2, sigma, Ex, Ey, Et;
    ;;; Find spatial and temporal gradients

    lconstant
        Dx = newanyarray([-1 1 0 0], conssfloatvec(#| -1, 0, 1 |#)),
        Dy = newanyarray([0 0 -1 1], Dx);

    lvars
        bounds = boundslist(im1),
        imsum = float_arraysum(im1, im2, false);
    newsfloatarray(bounds) -> Ex;
    newsfloatarray(bounds) -> Ey;
    gauss_smooth(imsum, sigma, imsum);
    ;;; Divide by 2 to get mean and again by 2 because gradient operators
    ;;; span 2 pixels
    float_multconst(0.25, imsum, imsum) -> ;
    convolve_2d(imsum, Dx, Ex, region_conv_output(imsum, Dx)) -> ;
    convolve_2d(imsum, Dy, Ey, region_conv_output(imsum, Dy)) -> ;

    float_arraydiff(im1, im2, false) -> Et;
    gauss_smooth(Et, sigma, Et);
enddefine;

define lconstant scale_arrays(Ex, Ey, lambd) -> (Px, Py);
    lvars Ex, Ey, lambd, Px, Py;
    float_arraysqr(Ex, false) -> Px;     ;;; temporary use of Px
    float_arraysqr(Ey, false) -> Py;     ;;; temporary use of Py
    float_arraysum(Px, Py, work_array1) -> ;
    float_addconst(lambd, work_array1, work_array1) -> ;
    float_arraydiv(Ex, work_array1, Px) -> ;
    float_arraydiv(Ey, work_array1, Py) -> ;
enddefine;

define lconstant horn_schunck1(Ex, Ey, Et, Px, Py, sigma2, u, v);
    ;;; One iteration, updating u and v
    lvars Ex, Ey, Et, Px, Py, sigma2, u, v;

    ;;; Smooth u and v
    gauss_smooth(u, sigma2, u);
    gauss_smooth(v, sigma2, v);

    ;;; Form dot product
    float_arraymult(Ex, u, work_array1) -> ;
    float_arraymult(Ey, v, work_array2) -> ;
    float_arraysum(work_array1, work_array2, work_array1) -> ;
    float_arraysum(work_array1, Et, work_array2) -> ;

    ;;; Scale
    float_arraymult(work_array2, Px, work_array1) -> ;
    float_arraymult(work_array2, Py, work_array2) -> ;

    ;;; and subtract
    float_arraydiff(u, work_array1, u) -> ;
    float_arraydiff(v, work_array2, v) -> ;
enddefine;

define lconstant horn_schunck0(im1, im2, sigma1, lambd)
        -> (Ex, Ey, Et, Px, Py, u, v);
    lvars im1, im2, sigma1, lambd, Ex, Ey, Et, Px, Py, u, v;
    lvars bounds = boundslist(im1);
    set_work_arrays(bounds);
    gradients(im1, im2, sigma1) -> (Ex, Ey, Et);
    scale_arrays(Ex, Ey, lambd) -> (Px, Py);
    newsfloatarray(boundslist(im1)) -> u;
    newsfloatarray(boundslist(im1)) -> v;
enddefine;

define horn_schunck(im1, im2, sigma1, sigma2, lambd) -> hs;
    lvars im1, im2, sigma1, sigma2, lambd, procedure hs;
    ;;; Returns a procedure that returns updated u and v arrays
    ;;; each time it is called
    lvars (Ex, Ey, Et, Px, Py, u, v) = horn_schunck0(im1, im2, sigma1, lambd);

    define lvars procedure hs /* -> (u, v) */;
        horn_schunck1(Ex, Ey, Et, Px, Py, sigma2, u, v);
        (u, v) /* results */
    enddefine;
enddefine;

endsection;

/* --- Revision History ---------------------------------------------------
--- David S Young, Nov 15 1994
        conssfloat changed to conssfloatvec to fit with new newsfloatarray
 */
