/* --- Copyright University of Sussex 1994. All rights reserved. ----------
 > File:            $popvision/lib/rc_array.p
 > Purpose:         Image display under X, in user coordinates
 > Author:          David S Young, Feb 20 1994 (see revisions)
 > Documentation:   HELP * RC_ARRAY, TEACH * RC_ARRAY
 > Related Files:   LIB * RC_GRAPHIC
 */

/*

This version assumes that it's dealing with an 8-bit display. Processing
is optimised for packed arrays of single precision floats or bytes, as
created by *NEWSFLOATARRAY and *NEWSARRAY.

         CONTENTS - (Use <ENTER> g to access required sections)

 -- Libraries
 -- Array resizing and value lookup
 -- ... Miscellaneous routines
 -- ... Lookup to colour map entries
 -- ... Main array conversion routine
 -- Generating look-up table
 -- ... Obtaining grey-level range for image
 -- ... Histogram equalisation thresholds
 -- ... Dealing with user options
 -- Positioning in window
 -- Overall array preparation procedure
 -- Colour map handling
 -- ... Compressing colour maps when not all entries are needed
 -- ... Allocating colourmap entries
 -- ... Setting up colour-map values
 -- Copying the image to the window
 -- Top-level procedures

The resizing and value lookup section knows how to convert any array
containing only numbers into a byte array suitable for passing to a
display routine.  It needs to be told which part of the array to use,
the region of the window that will be used, in window coordinates, and
the mapping from data to colourmap indices as a lookup table (see
*ARRAYLOOKUP). This part knows nothing about actual display, so is quite
general.

The lookup table generation section converts the user options for colour
map specifications into a lookup table. It needs to know the pixels in
the colour map that have been allocated.

The positioning section knows about *RC_GRAPHIC coordinate conventions,
and so can work out what the display region should be in window
coordinates.  Getting this exactly right is quite delicate. Although
tied to RC_GRAPHIC in this case, the approach is general.

The overall array preparation procedure calls the main procedure from
each of the preceding sections to produce an array that can be drawn on
a suitable window.

The colour map section interprets the user options for colour map
specifications, and returns a vector of the pixel values allocated.

*/

compile_mode:pop11 +strict;

section;

/*
-- Libraries ----------------------------------------------------------
*/

uses popvision
uses rc_graphic
uses boundslist_utils
uses newbytearray
uses newsfloatarray
uses arraysample
uses array_mxmn
uses arraylookup
uses array_hist
uses Xcolour_to_rgb

/*
-- Array resizing and value lookup ------------------------------------
*/

/*
-- ... Miscellaneous routines -----------------------------------------
*/

define lconstant isintarray(arr) /* -> result */;
    ;;; If an array can only hold integers, return no. bits.
    lvars arr;
    lvars k;
    arr.arrayvector.datakey.class_spec -> k;
    if k.isinteger then k else false endif /* -> result */
enddefine;

define lconstant isbitarray(arr) /* -> result */;
    lvars arr;
    isintarray(arr) == 1 /* -> result */
enddefine;

define lconstant isbytearray(arr) /* -> result */;
    lvars arr;
    isintarray(arr) == 8 /* -> result */
enddefine;

define lconstant npoints(region) /* -> int */;
    lvars region;
    lvars (x0, x1, y0, y1) = explode(region);
    (abs(x1 - x0) + 1) * (abs(y1 - y0) + 1)
enddefine;

define lconstant posbounds(reg1, reg2) -> (reg1, reg2);
    ;;; Makes sure that the bounds of reg1 are right way round - if
    ;;; any bound pair is not, then the corresponding bounds of reg2
    ;;; are also swapped
    lvars reg1, reg2;
    lvars rebuild = false,
        (x01, x11, y01, y11) = explode(reg1),
        (x02, x12, y02, y12) = explode(reg2);
    if x01 > x11 then
        (x01, x11, x02, x12) -> (x11, x01, x12, x02); true -> rebuild
    endif;
    if y01 > y11 then
        (y01, y11, y02, y12) -> (y11, y01, y12, y02); true -> rebuild
    endif;
    if rebuild then
        [% x01, x11, y01, y11 %] -> reg1;
        [% x02, x12, y02, y12 %] -> reg2;
    endif
enddefine;

define lconstant ncshiftarr(arr, reg1, reg2) /* -> newarr */;
    lvars arr, reg1, reg2;
    ;;; Return an array sharing the arrayvector of arr, with
    ;;; reg1 in arr mapped to reg2 in newarr. Returns false if
    ;;; region sizes do not match.  2-D only.
    lvars
        (x0, x1, y0, y1) = explode(boundslist(arr)),
        (x01, x11, y01, y11) = explode(reg1),
        (x02, x12, y02, y12) = explode(reg2),
        xsh = x02 - x01,
        ysh = y02 - y01;
    if xsh == x12 - x11 and ysh = y12 - y11 then
        if xsh == 0 and ysh == 0 then
            arr
        else
            lvars ( , v0) = arrayvector_bounds(arr);
            newanyarray([% x0+xsh, x1+xsh, y0+ysh, y1+ysh %], arr, v0-1)
        endif
    else
        false
    endif
enddefine;

/*
-- ... Lookup to colour map entries -----------------------------------
*/

define lconstant arr_to_byte(arr, region, lut) -> arr;
    lvars arr, region, lut;
    unless region then boundslist(arr) -> region endunless;

    unless arr.isbytearray
    and (lut == identfn or lut == round or lut == intof) then
        arraylookup(arr, region, lut, oldbytearray("arr_to_byte", region))
            -> arr
    endunless
enddefine;

/*
-- ... Main array conversion routine ----------------------------------
*/

vars rc_array_sample = "nearest";

define lconstant rci_drawable_array (arr, arr_region, win_region, lut)
        -> newarr;
    ;;; Does not necessarily copy the array data.
    lvars arr, arr_region, win_region, lut, newarr;

    ;;; sort out arguments
    unless arr_region then
        boundslist(arr) -> arr_region
    else
        region_inclusion_check(arr, arr_region)
    endunless;
    unless win_region then
        arr_region -> win_region
    endunless;

    if ncshiftarr(arr, arr_region, win_region) ->> newarr then
        ;;; No need for spatial resampling
        arr_to_byte(newarr, win_region, lut) -> newarr;
        ;;; But might still need to extract region
        unless boundslist(newarr) = win_region then
            arraysample(newarr, win_region,
                oldbytearray("rci_drawable_array", win_region), false,
                "nearest") -> newarr
        endunless

    elseif rc_array_sample /== "nearest"
    or npoints(arr_region) >= npoints(win_region) then
        ;;; Must do spatial resampling - if averaging or interpolating
        ;;; then must sample first; if shrinking then more efficient to do so
        posbounds(win_region, arr_region) -> (win_region, arr_region);
        if arr.isbytearray then
            ;;; keep as bytes
            arraysample(arr, arr_region,
                oldbytearray("rci_drawable_array", win_region), false,
                rc_array_sample) -> newarr
        else
            ;;; convert to floating point for efficiency
            arraysample(arr, arr_region,
                oldsfloatarray("rci_drawable_array", win_region), false,
                rc_array_sample) -> newarr
        endif;
        arr_to_byte(newarr, win_region, lut) -> newarr

    else
        ;;; Expanding and using "nearest" option - can do lookup first
        posbounds(arr_region, win_region) -> (arr_region, win_region);
        arr_to_byte(arr, arr_region, lut) -> newarr;
        posbounds(win_region, arr_region) -> (win_region, arr_region);
        arraysample(newarr, arr_region,
            oldbytearray("rci_drawable_array", win_region), false,
            rc_array_sample) -> newarr
    endif

enddefine;

/*
-- Generating look-up table -------------------------------------------
*/

/*
-- ... Obtaining grey-level range for image ---------------------------
*/

define lconstant val_lims(arr) -> (mn, mx);
    lvars arr, mn = false, mx = false;
    ;;; Return the min and max values that can be held in an integer
    ;;; array, or false if not an integer array.
    lvars nbits = isintarray(arr);
    if nbits then
        if nbits > 0 then
            0 -> mn;
            2 ** nbits - 1 -> mx
        else
            -(2 ** (abs(nbits)-1)) -> mn;
            -mn - 1 -> mx
        endif
    endif
enddefine;

define lconstant getblims(arr, region, bmin, bmax) -> (bmin, bmax);
    ;;; Gets bounds for brightnesses - either data limits or as specified
    ;;; Input of true means use the max or min possible in the integer
    ;;; array.
    lvars arr, region, bmin, bmax;
    lvars gmin gmax amin amax nbits;

    ;;; handle bit arrays as special case
    if isbitarray(arr) then
        (0,1) -> (bmin, bmax)

    else
        if bmin == true or bmax == true then
            val_lims(arr) -> (amin, amax);
            unless amin then
                if bmin == true then false -> bmin endif;
                if bmax == true then false -> bmax endif
            endunless
        endif;

        unless bmin and bmax then
            array_mxmn(arr, region) -> (gmax, gmin);
            if gmax = gmin then
                gmax + 1 -> gmax        ;;; blank region
            endif
        endunless;

        if bmin == true then
            amin -> bmin
        elseif bmin == false then
            gmin -> bmin
        endif;

        if bmax == true then
            amax -> bmax
        elseif bmax == false then
            gmax -> bmax
        endif
    endif
enddefine;

/*
-- ... Histogram equalisation thresholds ------------------------------
*/

define lconstant histthresh(arr, region, bmin, bmax, ncols)
        -> thresh;
    lvars arr, region, bmin, bmax, ncols, thresh = initv(ncols - 1);
    lconstant nbins = 256;
    unless region then boundslist(arr) -> region endunless;

    ;;; Get histogram
    if bmin.isintegral and bmax.isintegral then
        ;;; Adjust limits to count integral values in general
        1 + bmax -> bmax
    elseif bmax - bmin >= 2 then
        bmin - 0.5 -> bmin;
        bmax + 0.5 -> bmax
    endif;
    lvars
        (ndone, hist, nabv) = array_hist(arr, region, bmin, nbins, bmax);

    ;;; Convert to thresholds, with linear interpolation
    lvars ithresh, thr,
        isbyte = arr.isbytearray,
        ihist = 1,                          ;;; input bin number
        val = bmin,                         ;;; lower threshold for input bin
        valinc = (bmax - bmin) / nbins,     ;;; input bin size
        nlast = ndone,                      ;;; number in current input bin
        ntotal = region_size(region),
        nperbin = ntotal / ncols,           ;;; number per output bin
        ntarget = nperbin;                  ;;; no to have done in output
    for ithresh from 1 to ncols - 1 do
        until ndone >= ntarget do
            if ihist > nbins then
                ;;; only come here if high - low < 2 and rounding errors!
                nabv -> nlast       ;;; must reach total now
            else
                hist(ihist) -> nlast;
                ihist + 1 -> ihist
            endif;
            ndone + nlast -> ndone;
            val + valinc -> val
        enduntil;
        val - valinc * (ndone - ntarget) / nlast -> thr;
        ;;; Round threshold for byte arrays
        if isbyte then round(thr) else thr endif -> thresh(ithresh);
        ntarget + nperbin -> ntarget
    endfor
enddefine;

/*
-- ... Dealing with user options --------------------------------------
*/

define lconstant getbounds(arr, region, src_vals) -> (bmin, bmax);
    lvars arr, region, src_vals, bmin = false, bmax = false;
    if src_vals.islist and tl(src_vals) /== [] then
        src_vals(2) -> bmin;
        src_vals(3) -> bmax
    endif;
    getblims(arr, region, bmin, bmax) -> (bmin, bmax)
enddefine;

define lconstant getlut(arr, region, src_vals, dest_cols) -> lut;
    lvars arr, region, src_vals, dest_cols, lut;
    ;;; Returns a lookup-table for arraylookup, given a spec
    ;;; for the source colour mapping and a colour vector.
    lvars bmin, bmax,
        option = src_vals,
        ncols = length(dest_cols);
    unless option then "linear" -> option
    elseif option.islist then hd(option) -> option
    endunless;

    if option == "linear" then
        getbounds(arr, region, src_vals) -> (bmin, bmax);
        ;;; Make linear quantisation table
        lvars incr = (bmax - bmin) / ncols;
        [% bmin + incr, bmax - incr, dest_cols %] -> lut

    elseif option == "sqrt" then
        getbounds(arr, region, src_vals) -> (bmin, bmax);
        ;;; Make square-root quantisation table
        lvars colthresh, colincr = 1/ncols;
        [%
            {% for colthresh from colincr by colincr to 1 - colincr do
                    bmin + colthresh ** 2 * (bmax - bmin)
                endfor %},
            dest_cols %] -> lut

    elseif option == "equalise" then
        getbounds(arr, region, src_vals) -> (bmin, bmax);
        [% histthresh(arr, region, bmin, bmax, ncols),
            dest_cols %] -> lut

    elseif option == "quantise" then
        lvars nquants, quants = src_vals(2);
        if quants.isvector then
            length(quants) + 1 -> nquants
        else
            quants -> nquants
        endif;
        ;;; Weed out colours to correct number
        lvars icol;
        {% for icol from 1 by (ncols-1) / (nquants-1) to ncols do
                dest_cols(round(icol))
            endfor %} -> dest_cols;
        if quants.isvector then
            [% quants, dest_cols %] -> lut
        else
            getbounds(arr, region, tl(src_vals)) -> (bmin, bmax);
            (bmax - bmin) / nquants -> incr;
            if nquants == 2 then
                [% {% bmin + incr %}, dest_cols %] -> lut
            else
                [% bmin + incr, bmax - incr, dest_cols %] -> lut
            endif
        endif

    elseif option == "map" then
        hd(tl(src_vals)) -> lut

    elseif option == "direct" then
        round -> lut

    else
        mishap(option, 1, 'Unrecognised mapping option')
    endif
enddefine;

/*
-- Positioning in window ----------------------------------------------
*/

define lconstant joint_round(x0, x1) -> (x0, x1);
    ;;; Round two numbers. The difference between the results is equal
    ;;; to the difference between the arguments rounded, whilst the mean
    ;;; of the results is as close as possible to the mean of the
    ;;; arguments.
    lvars x0, x1,   d;
    round(x1-x0) -> d;
    round(0.5 * (x1 + x0 - d)) -> x0;
    x0 + d -> x1
enddefine;

define lconstant bothcorners(px0, px1, py0, py1, xsize, ysize)
        -> (px0, px1, py0, py1);
    lvars px0, px1, py0, py1, xsize, ysize;
    ;;; Given the elements of a boundslist spec of a region, and
    ;;; the size of the region, fills in any missing elements of the
    ;;; region spec to make it the required size.
    unless px1.isnumber then
        px0 + xsize -> px1
    elseunless px0.isnumber then
        px1 - xsize -> px0
    endunless;
    unless py1.isnumber then
        py0 + ysize -> py1
    elseunless py0.isnumber then
        py1 - ysize -> py0
    endunless
enddefine;

define lconstant getwinreg(arr, arr_reg, u_reg) -> win_reg;

    ;;; This looks elaborate - but I think it is as simple as it can
    ;;; be whilst still getting the mapping exactly right.

    ;;; Returns a region in window coords in which to draw the region
    ;;; of the array given.
    ;;; The arr_reg argument, if a list, specifies which array elements
    ;;; are to be included.
    ;;; The u_reg argument, if a list, gives the region in USER
    ;;; coords of the array limits - by which I mean the line round
    ;;; the very outside of the array.
    ;;; The u_reg argument may be nested inside another list -
    ;;; in this case it refers to the region of the window to be
    ;;; filled by the elements in arr_reg.

    lvars arr, arr_reg, u_reg, win_reg;

    ;;; Get array outer limits in ARRAY coords
    lvars
        (ax0, ax1, ay0, ay1) = explode(region_expand(arr, 0.5)),
        xasize = ax1 - ax0, yasize = ay1 - ay0;       ;;; array sizes

    ;;; Get array region outer limits in ARRAY coords
    lvars rx0, rx1, ry0, ry1;
    if arr_reg.islist then
        region_inclusion_check(arr, arr_reg);
        explode(region_expand(arr_reg, 0.5))
    else
        (ax0, ax1, ay0, ay1)
    endif -> (rx0, rx1, ry0, ry1);
    lvars xrsize = rx1 - rx0, yrsize = ry1 - ry0;

    ;;; Get array region limits in USER coords
    lvars ux0, ux1, uy0, uy1;
    if islist(u_reg) then
        if islist(hd(u_reg)) then
            ;;; The destination spec applies to the region
            bothcorners(explode(hd(u_reg)), xrsize, yrsize)
                -> (ux0, ux1, uy0, uy1);

        else
            ;;; The destination spec applies to the whole array
            lvars px0, px1, py0, py1;
            bothcorners(explode(u_reg), xasize, yasize)
                -> (px0, px1, py0, py1);
            ;;; Translate from array to region limits
            lvars
                xrat = (px1 - px0) / xasize,
                yrat = (py1 - py0) / yasize;
            (rx0 - ax0) * xrat + px0 -> ux0;
            (rx1 - ax0) * xrat + px0 -> ux1;
            (ry0 - ay0) * yrat + py0 -> uy0;
            (ry1 - ay0) * yrat + py0 -> uy1;
        endif

    else
        ;;; The destination spec defaults to the region coords
        (rx0, rx1, ry0, ry1) -> (ux0, ux1, uy0, uy1)
    endif;
    ;;; [drawing region, user coords % px0, px1, py0, py1 %] =>

    ;;; Get outer limits of drawing region in WINDOW coords
    ;;; Originally just used rc_transxyout, but need to get at results
    ;;; before rounding.
    lvars sx0, sx1, sy0, sy1;
    ux0 * rc_xscale + rc_xorigin -> sx0;
    ux1 * rc_xscale + rc_xorigin -> sx1;
    uy0 * rc_yscale + rc_yorigin -> sy0;
    uy1 * rc_yscale + rc_yorigin -> sy1;
    ;;; [drawing arr_reg, window coords % sx0, sx1, sy0, sy1 %] =>

    ;;; Now get bounds (in boundslist sense) as opposed to limits of
    ;;; window arr_reg. This means shifting in by half a screen pixel, but
    ;;; want to preserve the size, so do joint rounding.
    lvars signed_half;
    sign(sx1 - sx0) * 0.5 -> signed_half;
    sx0 + signed_half -> sx0;
    sx1 - signed_half -> sx1;
    joint_round(sx0, sx1) -> (sx0, sx1);
    sign(sy1 - sy0) * 0.5 -> signed_half;
    sy0 + signed_half -> sy0;
    sy1 - signed_half -> sy1;
    joint_round(sy0, sy1) -> (sy0, sy1);
    ;;; [drawing arr_reg, rounded % sx0, sx1, sy0, sy1 %] =>

    [% sx0, sx1, sy0, sy1 %] -> win_reg
enddefine;

/*
-- Overall array preparation procedure --------------------------------
*/

define lconstant rci_array_ready
        (arr, arr_reg, u_reg, src_vals, col_vals) /* -> newarr */;
    lvars arr, arr_reg, u_reg, src_vals, col_vals;
    lvars
        lut = getlut(arr, arr_reg, src_vals, col_vals),
        win_reg = getwinreg(arr, arr_reg, u_reg);
    rci_drawable_array(arr, arr_reg, win_reg, lut) /* -> newarr */
enddefine;

/*
-- Colour map handling ------------------------------------------------
*/

;;; These control the amount of the public colour map we try to grab
;;; for the two general-purpose display options.
lconstant
    NGREYS    =  64,
    NCOLS     =  64;

/*
-- ... Compressing colour maps when not all entries are needed --------
*/

define lconstant compressmap(image, region, ncols) /* -> rmap */;
    lvars image, region, ncols;
    ;;; Values in the region of the array must be from 0 to ncols-1.
    ;;; Discovers which of these are actually present, and returns a
    ;;; vector containing the values present.
    lconstant hist = initintvec(256);   ;;; optimise for byte case
    lvars nblo, nabv;
    array_hist(image, region, 0, [% 1, ncols, hist %], ncols)
        -> (nblo, , nabv);
    unless nblo == 0 and nabv == 0 then
        mishap(image, ncols, 2, 'Array has values outside colour map range')
    endunless;
    ;;; Construct vector of values present
    lvars ibin;
    {% fast_for ibin from 1 to ncols do
            if hist(ibin) fi_> 0 then
                ibin
            endif
        endfor %}
enddefine;

define lconstant reducecmap(cmap, rmap) /* -> newcmap */;
    lvars cmap, rmap;
    ;;; Reduce the entries in a colour map using the output
    ;;; of compressmap.
    lvars i, oval,
        nnew = length(rmap);
    if length(cmap) == 3 and isvector(cmap(1)) then
        ;;; Have vector-type spec
        lvars
            (r, g, b) = explode(cmap),
            newr = initv(nnew),
            newg = initv(nnew),
            newb = initv(nnew);
        fast_for i from 1 to nnew do
            rmap(i) -> oval;
            r(oval) -> newr(i); g(oval) -> newg(i); b(oval) -> newb(i)
        endfor;
        {% newr, newg, newb %} /* -> newcmap */
    else
        ;;; Straightforward
        {% fast_for i from 1 to nnew do
                cmap(rmap(i))
            endfor %}
    endif
enddefine;

define lconstant rmap_to_lut(rmap, pixels) -> lut;
    lvars rmap, pixels, lut = initintvec(256);
    ;;; Returns a 256-element lookup table for mapping from array
    ;;; values to pixels values for new table.
    lvars i;
    fast_for i from 1 to length(rmap) do
        pixels(i) -> lut(rmap(i))
    endfor
enddefine;

/*
-- ... Allocating colourmap entries -----------------------------------
*/

define lconstant maketestwin /* -> window */;
    ;;; Returns an unmapped widget for colour map manipulation
    ;;; purposes.
    lconstant XpwGraphic = XptWidgetSet("Poplog")("GraphicWidget");
    XptNewWindow('Testwin', {1 1}, [], XpwGraphic,
        [{mappedWhenManaged ^false}])
enddefine;

define lconstant getcrange(win1, win2, ncolours) -> (cols, winused);
    lvars win1, win2, ncolours, cols, winused;
    ;;; Grab colours.  If win1 will accept them with its current
    ;;; colour map, then use that; otherwise try and put them in
    ;;; win2; otherwise give win2 a private colour map and use that.

    lvars crange = XpwAllocColorRange(win1, ncolours, 0,0,0, 0,0,0);
    win1 -> winused;
    unless crange then
        win2 -> winused;
        XpwAllocColorRange(win2, ncolours, 0,0,0, 0,0,0) -> crange;
        unless crange then
            XpwCreateColormap(win2);
            XpwAllocColorRange(win2, ncolours, 0,0,0, 0,0,0) -> crange;
            unless crange then
                mishap(ncolours, 1, 'Unable to allocate colour map entries')
            endunless
        endunless
    endunless;

    ;;; Get pixel values
    lvars i;
    {% for i from 1 to ncolours do
            XpwStackColorRangeInfo(crange, i); erasenum(3);
        endfor %} -> cols
enddefine;

/*
-- ... Setting up colour-map values -----------------------------------
*/

define lconstant linrep(t1, t2, n) /* -> rep */;
    ;;; Returns a repeater that gives n equally-spaced values from
    ;;; t1 to t2. (Keeps going above t2, as in this prog not required
    ;;; to return termin.)
    lvars t1, t2, n;
    lvars
        t = t1,
        tinc = (t2 - t1) / (n - 1);

    procedure /* -> val */;
        t1; /* -> val */
        t1 + tinc -> t1
    endprocedure
enddefine;

define lconstant setspect(colspec, win, pixels);
    ;;; Sets up a colour spectrum using a colour spec given as
    ;;; a list of colour names.
    lvars colspec, win, pixels;

    lvars r, g, b, col1, r0, g0, b0, r1, p, g1, b1, p0, p1, ninseg,
        nsegs = length(colspec) - 1,
        pos = linrep(1, length(pixels), nsegs+1);

    round(pos()) -> p0;
    Xcolour_to_rgb(hd(colspec)) -> (r0, g0, b0);
    XpwChangeColor(win, pixels(p0), r0, g0, b0);
    for col1 in tl(colspec) do
        round(pos()) -> p1;
        Xcolour_to_rgb(col1) -> (r1, g1, b1);
        p1 - p0 + 1 -> ninseg;
        linrep(r0, r1, ninseg) -> r;
        linrep(g0, g1, ninseg) -> g;
        linrep(b0, b1, ninseg) -> b;
        r() -> ; g() -> ; b() -> ;      ;;; first point already done
        for p from p0+1 to p1 do
            XpwChangeColor(
                win, pixels(p), round(r()), round(g()), round(b()))
        endfor;
        p1 -> p0;
        r1 -> r0;   g1 -> g0;   b1 -> b0;
    endfor;
enddefine;

lconstant spectcols_default
    = [black purple4 blue4 cyan4 green3 yellow2 orange red white];
;;; Window to hold permanently-allocated grey-level maps
lvars _win = false;
;;; Permanently-allocated grey-level and spectrum pixels
lvars
    _greyscale = false,
    _spectrum = false,
    _spectcols = spectcols_default;  ;;; current spectrum in words

define lconstant greypix /* -> greyscale */;
    ;;; Sets up grey-scale in colour map, if possible in public
    ;;; colour map associated with permanent window.
    ;;; Refers to:  _greyscale
    ;;;             _win
    ;;;             _rc_window
    lconstant greycols = [black white];
    if _greyscale then
        _greyscale
    else
        lvars g, p, pixels,
            win = _win or (maketestwin() ->> _win);
        getcrange(win, rc_window, NGREYS) -> (pixels, win);
        setspect(greycols, win, pixels);
        _win == win and pixels -> _greyscale;   ;;; using perm window
        pixels /* -> greyscale */
    endif
enddefine;

define lconstant spectpix /* -> spectrum */;
    ;;; Refers to:  _spectrum
    ;;;             _spectcols
    ;;;             _win
    ;;;             _rc_window
    if _spectrum then
        _spectrum
    else
        lvars g, p, pixels,
            win = _win or (maketestwin() ->> _win);
        getcrange(win, rc_window, NCOLS) -> (pixels, win);
        setspect(_spectcols, win, pixels);
        _win == win and pixels -> _spectrum;   ;;; using perm window
        pixels /* -> spectrum */
    endif
enddefine;

define lconstant privpix(array, region, cmap) -> pixels;
    lvars array, region, cmap, pixels;
    ;;; Colour map can be a vector of r, g, b vectors or a vector
    ;;; of individual values. Each value can then be an r,g,b vector
    ;;; or a colour name.
    lvars ncols,
        sep_rgb = length(cmap) == 3 and isvector(cmap(1));
    if sep_rgb then
        length(cmap(1))
    else
        length(cmap)
    endif -> ncols;

    ;;; If array is supplied, compress the colour map
    if array then
        lvars rmap = compressmap(array, region, ncols);
        length(rmap) -> ncols;
        reducecmap(cmap, rmap) -> cmap;
    endif;

    ;;; Do not use permanent window - always assign to rc_window
    getcrange(rc_window, rc_window, ncols) -> (pixels, );
    lvars i;
    if sep_rgb then
        lvars (r, g, b) = explode(cmap);
        for i from 1 to ncols do
            XpwChangeColor(rc_window, pixels(i), r(i), g(i), b(i))
        endfor
    else
        for i from 1 to ncols do
            XpwChangeColor(rc_window, pixels(i), Xcolour_to_rgb(cmap(i)))
        endfor
    endif;

    if array then
        ;;; Need to get lookup table for reduced colour map
        lvars lut = rmap_to_lut(rmap, pixels);
        conspair(lut, pixels) -> pixels
    endif
enddefine;

define lconstant getcols(array, region, arr_cols, win_cols)
        -> (arr_cols, win_cols);
    lvars array, region, arr_cols, win_cols;
    ;;; Allocates colours and returns a vector of the pixels used.
    ;;; (The special case of direct mapping is handled messily by
    ;;; returning a pair with pixels and lut.)
    unless win_cols then "greyscale" -> win_cols endunless;    ;;; default
    if win_cols == "greyscale" then
        greypix() -> win_cols
    elseif win_cols == "spectrum" then
        spectpix() -> win_cols
    elseif ((arr_cols.islist and hd(arr_cols)) or arr_cols) == "direct" then
        "nearest" -> rc_array_sample;   ;;; other options do not make sense
        lvars p = privpix(array, region, win_cols);
        [map % front(p) %] -> arr_cols;
        back(p) -> win_cols
    else
        privpix(false, false, win_cols) -> win_cols
    endif
enddefine;

/*
-- Copying the image to the window ------------------------------------
*/

define lconstant tlclip(arr, width, height, x, y) ->
        (arr, width, height, x, y);
    ;;; XpwDrawImage clips the right and bottom of an array if it would
    ;;; go over the edge of the window, but fails to draw at all if the
    ;;; array would go over the top or left of the window.
    ;;; It would be more efficient to do this along with the other
    ;;; resampling, but the code above would get appreciably more fiddly
    ;;; as it would be necessary to project the window back onto the
    ;;; array to find the input bounds for the resampling.
    lvars arr, width, height, x, y;
    lvars region;
    if x < 0 or y < 0 then
        width + x -> width;     height + y -> height;
        max(x, 0) -> x;         max(y, 0) -> y;
        width - x -> width;     height - y -> height;
        if width > 0 and height > 0 then
            [% x, x+width-1, y, y+height-1 %] -> region;
            arraysample(arr, region, oldbytearray("tlclip", region), false,
                "nearest") -> arr
        endif
    endif
enddefine;

define lconstant drawimage(arr, win);
    ;;; Uses XpwDrawImage to display arr in the window, assuming that
    ;;; arr is all set up with the right boundslist and values.
    lvars arr, win;
    lvars
        (x0, x1, y0, y1) = explode(boundslist(arr)),
        w = x1 - x0 + 1,
        h = y1 - y0 + 1;
    tlclip(arr, w, h, x0, y0) -> (arr, w, h, x0, y0);
    XpwDrawImage(win, w, h, x0, y0, arrayvector(arr))
enddefine;

/*
-- Top-level procedures -----------------------------------------------
*/

define rc_array(arr, arr_region, win_region, arr_cols, win_cols);
    lvars arr, arr_region, win_region, arr_cols, win_cols;
    dlocal rc_array_sample; ;;; may be changed if arr_cols is "direct"

    ;;; start a new window if necessary - do not call rc_start as this
    ;;; resets the coordinates
    unless rc_window.xt_islivewindow then
        rc_new_window(
            rc_window_xsize, rc_window_ysize, rc_window_x, rc_window_y, false)
    endunless;

    drawimage(
        rci_array_ready(
            arr, arr_region, win_region,
            getcols(arr, arr_region, arr_cols, win_cols)
        ),
        rc_window
    )
enddefine;

/* Procedure for changing colour map */

define active rc_spectrum;
    _spectcols
enddefine;

define updaterof active rc_spectrum(cols);
    lvars cols;
    if cols.islist then
        cols
    else
        spectcols_default
    endif -> _spectcols;
    if _spectrum then
        setspect(_spectcols, _win, _spectrum)
    endif
enddefine;

/* Procedure for setting sensible coordinates */

define rc_win_coords;
    ;;; Set the rc user coordinates suitably for images
    0 ->> rc_xorigin -> rc_yorigin;
    1 ->> rc_xscale -> rc_yscale;
enddefine;

endsection;

/* --- Revision History ---------------------------------------------------
--- David S Young, Nov 16 1994
        Uses newbytearray instead of newsarray, and creates work arrays using
        oldbytearray (see * OLDARRAY) to reduce garbage creation.
--- David S Young, Feb 24 1994
        getblims adds one to gmax if region is uniform so that this case
        can be displayed.
 */
