/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            $popvision/lib/sunrasterfile.p
 > Purpose:         Read and write Sun rasterfiles
 > Author:          David S Young, Dec 13 1993 (see revisions)
 > Documentation:   HELP *SUNRASTERFILE, MAN *RASTERFILE
 */

compile_mode:pop11 +strict;

section;

/* Rasterfile constants from /usr/include/rasterfile.h */

defclass lconstant rasterfile {
    >-> ras_magic       :int,       /* magic number */
        ras_width       :int,       /* width (pixels) of image */
        ras_height      :int,       /* height (pixels) of image */
        ras_depth       :int,       /* depth (1, 8, or 24 bits) of pixel */
        ras_length      :int,       /* length (bytes) of image */
        ras_type        :int,       /* type of file; see RT_* below */
        ras_maptype     :int,       /* type of colormap; see RMT_* below */
        ras_maplength   :int        /* length (bytes) of following map */
    /* color map follows for ras_maplength bytes, followed by image */
};

lconstant
    RAS_MAGIC       =   16:59A66A95,

    /* Sun supported ras_type's */
    RT_OLD          =   0,  /* Raw pixrect image in 68000 byte order */
    RT_STANDARD     =   1,  /* Raw pixrect image in 68000 byte order */
    RT_BYTE_ENCODED =   2,  /* Run-length compression of bytes */
    RT_FORMAT_RGB   =   3,  /* XRGB or RGB instead of XBGR or BGR */
    RT_FORMAT_TIFF  =   4,  /* tiff <-> standard rasterfile */
    RT_FORMAT_IFF   =   5,  /* iff (TAAC format) <-> standard rasterfile */
    RT_EXPERIMENTAL =   16:FFFF,    /* Reserved for testing */

    /* Sun registered ras_maptype's */
    RMT_RAW         =   2,
    /* Sun supported ras_maptype's */
    RMT_NONE        =   0,  /* ras_maplength is expected to be 0 */
    RMT_EQUAL_RGB   =   1,  /* red[ras_maplength/3],green[],blue[] */

    /* Rows get rounded out to a multiple of this */
    RAS_ROUNDROW    =   16;

/* A constant record is sufficient for the header */

lconstant
    rasheader = consrasterfile(0, 0, 0, 0, 0, 0, 0, 0),
    (, bits_per_int) = field_spec_info("int"),
    (, bits_per_byte) = field_spec_info("byte"),
    rasheadbytes = length(rasheader) * bits_per_int div bits_per_byte;

;;; Keys for vectors to hold image data - all sizes must
;;; be multiples or exact divisors of -bits_per_byte- (which is
;;; unlikely to be anything other than 8).
lconstant
    key_of_size = newproperty([
            [1      % conskey("rasarray1", 1) %]
            [8      % conskey("rasarray8", 8) %]
            [24     % conskey("rasarray24", 24) %]],
        3, false, "perm");

;;; Recordclass and buffers to hold colour map and default colour map
;;; - default is just full grey-scale ramp.
defclass lconstant raster_cmap :byte;
lconstant cmap_max = 2 ** bits_per_byte;
lvars i;
lconstant
    NRGB = 3,       ;;; no of colour entries in an RGB colour map
    cmap_buff = initraster_cmap(NRGB * cmap_max),
    cmap_default = consraster_cmap(
                        repeat NRGB times
                            for i from 0 to cmap_max-1 do i endfor
                        endrepeat, NRGB * cmap_max);


define lconstant cmap_buff_to_vec(cmap, ncols) /* -> vecs */;
    ;;; Convert a cmap buffer as read from disc to a vector of
    ;;; vectors in the order r, g, b.
    lvars cmap, ncols, vecs;
    lvars col, p;
    newanyarray([1 ^ncols 1 ^NRGB], cmap, true) -> cmap;
    {% for col from 1 to NRGB do
            {% for p from 1 to ncols do cmap(p, col) endfor %}
        endfor %}
enddefine;

define lconstant cmap_vec_to_buff(vecs) /* -> buff */;
    ;;; Convert a vector of vectors to a cmap buffer to write to
    ;;; disc - actually uses the constant vector set up above.
    lvars vecs;
    lvars i, c, col, cmap,
        ncols = length(vecs(1));
    unless ncols <= cmap_max then
        mishap(0, 'Colour map vector too long')
    endunless;
    newanyarray([1 ^ncols 1 ^NRGB], cmap_buff, true) -> cmap;
    for i from 1 to NRGB do
        vecs(i) -> c;
        unless length(c) == ncols then
            mishap(0, 'Colour map vectors different lengths')
        endunless;
        for col from 1 to ncols do
            c(col) -> cmap(col, i);
        endfor
    endfor;
    cmap_buff /* -> buff */
enddefine;


define sunrasterfile(filename) /* -> (array, [cmap])*/;
    lvars filename, return_cmap = false, array, cmap;

    ;;;  Get optional arg - if true, return cmap as well as array
    if filename.isboolean then
        filename -> (filename, return_cmap)
    endif;

    lvars dev = sysopen(filename, 0, false, `N`);

    ;;; Read the header
    unless sysread(dev, rasheader, rasheadbytes) == rasheadbytes then
        mishap(0, 'Unable to read rasterfile header')
    endunless;

    lvars
        dep = rasheader.ras_depth,
        wid = rasheader.ras_width,
        ht = rasheader.ras_height,
        len = rasheader.ras_length,
    ;;; Round up row length
        bits_per_row = ((dep*wid - 1) div RAS_ROUNDROW + 1) * RAS_ROUNDROW,
        bytes_per_row = bits_per_row div bits_per_byte,
        bits_to_fill = bits_per_row - dep*wid,
        veckey = key_of_size(dep);

    ;;; Checks on legality
    unless rasheader.ras_magic = RAS_MAGIC then ;;; big int so not ==
        mishap(filename, 1, 'Not rasterfile - wrong magic number')
    endunless;
    unless rasheader.ras_type == RT_STANDARD then
        mishap(filename, 1, 'Can only read RT_STANDARD rasterfiles')
    endunless;
    unless veckey then
        mishap(filename, dep, 2, 'Depth must be 1, 8 or 24')
    endunless;
    unless len == bytes_per_row * ht then
        mishap(filename, len, wid, ht, dep, 5, 'Length does not match wid * ht')
    endunless;

    ;;; Deal with colour map
    if rasheader.ras_maptype == RMT_NONE then
        unless rasheader.ras_maplength == 0 then
            mishap(filename, 1, 'Expecting 0-length colour map')
        endunless;
        false -> cmap;
    elseif rasheader.ras_maptype == RMT_EQUAL_RGB
    or rasheader.ras_maptype == RMT_RAW then
        ;;; Not really sure what RMT_RAW is supposed to mean.  But other
        ;;; software seems to treat it same as RMT_EQUAL_RGB
        lvars ncols = rasheader.ras_maplength div NRGB;
        unless rasheader.ras_maplength == NRGB * ncols then
            mishap(filename, 1, 'Map length not divisible by no. colours')
        elseunless ncols <= cmap_max then
            mishap(filename, 1, 'Too many entries in colour map')
        endunless;
        unless sysread(dev, cmap_buff, rasheader.ras_maplength)
            == rasheader.ras_maplength then
            mishap(filename, 1, 'Could not read colour map entries')
        endunless;
        if return_cmap then
            cmap_buff_to_vec(cmap_buff, ncols) -> cmap
        endif
    endif;

    ;;; Create vector big enough to hold all the data, and read it
    lvars
        nitems = (len * bits_per_byte - 1) div dep + 1,     ;;; round up
        data = class_init(veckey)(nitems);
    unless sysread(dev, data, len) == len then
        mishap(filename, 1, 'Error reading data')
    endunless;
    ;;; Might as well close the device for tidiness
    sysclose(dev);

    ;;; If necessary, shuffle up the data to avoid gaps at the end of
    ;;; each line.
    ;;; (An alternative approach used in an earlier version of this
    ;;; library involved hacking the boundslist of the array.  This
    ;;; was unacceptable: for one thing the arrayvector wouldn't make
    ;;; sense to most external procedures; for another it wouldn't have
    ;;; worked with 24-bit images; and it was just horrid anyway.)
    lvars in_sub = 1, out_sub = 1;
    if bits_to_fill /== 0 then
        if dep > bits_per_byte then
            ;;; Assume dep is a multiple of bits_per_byte.  Must not use
            ;;; move_subvector as real data may not lie on vector element
            ;;; boundaries if dep = 24.
            lvars bytes_per_row_out = wid * (dep div bits_per_byte);
            repeat ht - 1 times
                in_sub + bytes_per_row -> in_sub;
                out_sub + bytes_per_row_out -> out_sub;
                move_bytes(in_sub, data, out_sub, data, bytes_per_row_out)
            endrepeat
        else
            ;;; Assume bits_per_byte is a multiple of dep.
            ;;; Must not use move_bytes as there may be the odd few bits
            ;;; to fill in if dep = 1.
            lvars items_per_row = bytes_per_row * (bits_per_byte div dep);
            repeat ht - 1 times
                in_sub + items_per_row -> in_sub;
                out_sub + wid -> out_sub;
                move_subvector(in_sub, data, out_sub, data, wid)
            endrepeat
        endif
    endif;

    ;;; Create the output array and maybe colour map
    newanyarray([1 ^wid 1 ^ht], data, true) /* -> array */;
    if return_cmap then
        cmap
    endif
enddefine;


define updaterof sunrasterfile(array, filename);
    lvars array, cmap = false, filename;
    if array.isvector then
        ;;; There is a colour map argument.
        (array, filename) -> (array, cmap, filename)
    endif;

    lvars dev = syscreate(filename, 1, false);

    lvars
        (x0, x1, y0, y1) = explode(boundslist(array)),
        wid = x1 - x0 + 1,
        ht = y1 - y0 + 1,
        data = arrayvector(array),
        dep = class_spec(datakey(data)),
        veckey = key_of_size(dep),
        (vmax, vmin) = arrayvector_bounds(array),
        (bits_before, bytes_before) = ((vmin - 1) * dep) // bits_per_byte,
    ;;; Round up row length
        bits_per_row = ((dep*wid - 1) div RAS_ROUNDROW + 1) * RAS_ROUNDROW,
        bytes_per_row = bits_per_row div bits_per_byte,
        bits_to_fill = bits_per_row - dep*wid,   ;;; do not use //
        len = bytes_per_row * ht;

    unless veckey then
        mishap(datakey(data), 1, 'Illegal type of array for rasterfile')
    endunless;

    ;;; Set up header
    RAS_MAGIC -> rasheader.ras_magic;
    wid -> rasheader.ras_width;
    ht -> rasheader.ras_height;
    dep -> rasheader.ras_depth;
    len -> rasheader.ras_length;
    RT_STANDARD -> rasheader.ras_type;

    ;;; Deal with colour map
    if cmap then
        RMT_EQUAL_RGB -> rasheader.ras_maptype;
        length(cmap(1)) * NRGB -> rasheader.ras_maplength;
        cmap_vec_to_buff(cmap) -> cmap;
    elseif dep == 1 or dep == 24 then
        RMT_NONE -> rasheader.ras_maptype;
        0 -> rasheader.ras_maplength
    else
        RMT_EQUAL_RGB -> rasheader.ras_maptype;
        NRGB * cmap_max -> rasheader.ras_maplength;
        cmap_default -> cmap;
    endif;

    ;;; Write header and colour map
    syswrite(dev, rasheader, rasheadbytes);
    if cmap then
        syswrite(dev, cmap, rasheader.ras_maplength)
    endif;

    ;;; Write the data itself.
    ;;; If nothing to fill and start on a byte boundary, then no need
    ;;; to copy - just write the array vector
    if bits_to_fill == 0 and bits_before == 0 then
        syswrite(dev, bytes_before + 1, data, len)
    else ;;; need to pad each row - use a one-row buffer
        lvars
            items_per_row = ((bits_per_row - 1) div dep) + 1,
            buff = class_init(veckey)(items_per_row),
            in_sub = vmin;
        repeat ht times
            move_subvector(in_sub, data, 1, buff, wid);
            syswrite(dev, buff, bytes_per_row);
            in_sub + wid -> in_sub
        endrepeat
    endif;

    sysclose(dev)
enddefine;

endsection;

/* --- Revision History ---------------------------------------------------
--- David S Young, Dec 23 1993
        Changed "lconstant macro" to "lconstant"; former has no advantage.
--- David S Young, Dec 21 1993
        Removed a couple of numerical constants.
 */
