/* --- Copyright University of Sussex 1993. All rights reserved. ----------
 > File:            C.all/lib/proto/go/lib/go_group.p
 > Purpose:         GO file
 > Author:          Ben Rabau, 1992-1993
 > Documentation:   HELP GO_CLASSES
 > Related Files:
 */
													  ;;; 10th August 1993
;;; File go_group.p
;;; Author Ben Rabau
;;; Info: see go.p

compile_mode :pop11 +strict;

;;; INTRO: The CLASS go_group allows creation of composite objects consisting
;;; INTRO: of a number of other Graphical Objects.
;;; INTRO: New groups can be made (see test_group for example):
;;; INTRO:    1. fill in the classes in the groups in "go_components"
;;; INTRO:    2. give the corresponding x and y offsets in the list:
;;; INTRO:       go_local_coords
;;; INTRO: Or by:
;;; INTRO:    1. Creating a new go_group, then
;;; INTRO:    2. Use go_add_to() or go_annex_to() to add an element
;;; INTRO:       to the front or to the back of the go_group.
;;; INTRO:
;;; INTRO: If you are in "Edit mode" you can create a "Group" which will hold
;;; INTRO: a button and a toggle. You can only move them as a
;;; INTRO: unit. You can edit the separate components by making the group
;;; INTRO: editable (see go_make_editable and go_make_uneditable).
;;; INTRO: In "Live mode" selection works on the separate items of the go_group.
;;; INTRO:
;;; INTRO: Note: This does not change any of the components; everything
;;; INTRO:       is handled by the go_group objectclass. All methods and
;;; INTRO:       behaviour of the separate components remain intact and
;;; INTRO:       have no knowledge of the group they are in.


;;; THIS OBJECT_CLASS CAN BE USED TO DEMONSTRATE THE GROUP APPROACH
;;; (see slide and slider_thumb for component approach).

uses go_screen_object;
uses go_fillable;
uses go_edit_point;
uses go_composite;
uses go_rotatable;

;;; ---------------------------------------------------------

vars procedure go_initialise_group;  ;;; defined later

;;; ---------------------------------------------------------

define :class go_group;
	isa go_fillable go_rotatable go_composite go_screen_object;
	slot go_components              == [];
;;; REF: The class keys of the components of the group.
	slot the_go_components          == [];
;;; REF: The handles to the components of the group.
	slot stored_go_live_object      == false;
;;; REF: interactive go_live_object

	on new do procedure();
				   go_initialise_group();         ;;; make all components
			  endprocedure;
enddefine;


;;; ---------------------------------------------------------
;;; INITIALISATION

define go_initialise_group( constructor ) -> grp;
lvars constructor, grp = constructor();
;;; REF: go_initialise_group( ARGS ) -> GROUP;
;;; REF: This procedure should never be called directly. It constructs the
;;; REF: group object from its parameters and creates new components if any.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component_key, component, offsets = grp.go_local_coords, worldcoords = [];
	go_batch_mode_on( grp );
	[% for component_key in go_components(grp) do
		   class_new(component_key)() ->> component;     ;;; stacked object
		   grp :: go_visible_in(component) -> go_visible_in(component);
		   if offsets == [] then
			   worldcoords <> [% component.go_position_of_centre %]
			   -> worldcoords;
		   endif;
	   endfor;
	 %] -> grp.the_go_components;
	if offsets == [] and not(worldcoords == []) then
		worldcoords -> go_world_coords( grp );;
	else
		go_update( grp );
	endif;
	go_batch_mode_off( grp );
enddefine;


;;;----------------------------------------------------------
;;; HELP VARIABLES

vars procedure go_set_bounding_box;  ;;; defined later

;;; ---------------------------------------------------------
;;; CHECKING PROCEDURES

define lconstant go_validate_bounding_box( grp );
lvars grp;
lvars cache_flag = false, change_flag = false;
lvars x_offs, y_offs, xc, yc, nbr = 1, component;
	if (grp.go_visible_in == []) then
		true -> cache_flag;
	else
		go_applist( procedure( grp, pane ); lvars grp, pane;
					unless cached_go_coord_list(pane)(grp) then
						true ->> cache_flag -> cached_go_coord_list(pane)(grp);
					endunless;
					endprocedure, grp);
	endif;
		lvars offsets = go_world_coords(grp);
		for component in  the_go_components(grp)
		do
			fast_destpair( offsets ) -> (x_offs, offsets) ;
			fast_destpair( offsets ) -> (y_offs, offsets) ;
			go_position_of_centre( component ) -> (xc, yc);
			unless (round(x_offs - xc) == 0) and (round(y_offs - yc) == 0) then
				true -> change_flag;
;;; [% ' X ', xc, x_offs, ' Y ', yc, y_offs %] =>
				go_transxyin( xc, yc, grp )
				-> ( stored_go_local_coords(grp)(nbr),
					 stored_go_local_coords(grp)(nbr+1) );
			endunless;
			nbr + 2 -> nbr;
		endfor;
	if (cache_flag or change_flag) then
		go_set_bounding_box( grp );
	endif;
enddefine;


;;; ---------------------------------------------------------
;;; CHECKING PROCEDURES

define  :method go_update( grp :go_group );
lvars grp;
;;; REF: go_update( GROUP );
;;; REF: Remove all cached screen coordinates and updates the components.
;;; REF: Resets the centre positions of the components based on the current
;;; REF: offsets stored ing the group object (see slots go_local_coords).
;;; REF: Since this might move the components the group might need to be
;;; REF: redrawn after executing this call.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars complist = the_go_components(grp), offsets = go_world_coords(grp);
lvars component, x_offs, y_offs;
	call_next_method( grp );
	for component in complist do
		go_applist( procedure( grp, pane ); lvars grp, pane;
						false -> cached_go_coord_list(pane)(component);
					endprocedure, component );
		fast_destpair( offsets ) -> (x_offs, offsets) ;
		fast_destpair( offsets ) -> (y_offs, offsets) ;
		(x_offs, y_offs) -> go_position_of_centre( component );
	endfor;
	go_validate_bounding_box(grp);
enddefine;


;;; ---------------------------------------------------------
;;; DRAWING

define :method go_clear( grp :go_group );
lvars pane, grp;
;;; REF: go_clear( GROUP );
;;; REF: Clears the group by clearing its components in all panes of the group.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
	for component in grp.the_go_components do
		"forbidden" -> go_visible_now( component );
		if component.isgo_group then go_clear( component ); endif;
	endfor;
	call_next_method( grp );
enddefine;


define :method go_draw( pane :go_pane, grp :go_group );
lvars pane, grp;
;;; REF: go_draw( PANE, GROUP );
;;; REF: Draws the group by drawing its components in the given pane.
;;; REF: PANE is an instance of the go_pane class (see REF * GO_PANE).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;

	;;; go_draw all the go_group-components
	unless go_visible_now( grp )  == "forbidden" then
	  if cached_go_coord_list(pane)(grp) then
		for component in grp.the_go_components do
			false -> go_visible_now( component );
			go_draw( pane, component );
		endfor;
	  else
		for component in grp.the_go_components do
			false -> cached_go_coord_list(pane)( component );
			false -> go_visible_now( component );
			go_draw( pane, component );
		endfor;
	  endif;
	endunless;
	true -> go_visible_now( grp );
enddefine;

;;; ---------------------------------------------------------
;;; SCALING

define :method updaterof go_xscale( the_scale, grp :go_group );
lvars the_scale, grp;
;;; REF: REAL -> go_xscale( GROUP );
;;; REF: Set the horizontal scale of each of the components of the group. Since
;;; REF: the offsets are based on the centre positions of the components
;;; REF: versus the centre of the bounding box of the group, the total group
;;; REF: will be scaled around the centre of the group.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
	;;; X-scale all the go_group-components
	returnif( the_scale = grp.go_xscale );
	procedure;
		dlocal 1 % go_batch_mode( grp ) % = true;
		for component in grp.the_go_components do
			the_scale -> component.go_xscale;
		endfor;
		call_next_method( the_scale, grp );
	endprocedure();
enddefine;


define :method updaterof go_yscale( the_scale, grp :go_group );
lvars the_scale, grp;
;;; REF: REAL -> go_xscale( GROUP );
;;; REF: Set the vertical scale of each of the components of the group. Since
;;; REF: the offsets are based on the centre positions of the components
;;; REF: versus the centre of the bounding box of the group, the total group
;;; REF: will be scaled around the centre of the group.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
	;;; X-scale all the go_group-components
	returnif( the_scale = grp.go_yscale );
	procedure;
		dlocal 1 % go_batch_mode( grp ) % = true;
		for component in grp.the_go_components do
			the_scale -> component.go_yscale;
		endfor;
		call_next_method( the_scale, grp );
	endprocedure();
enddefine;

;;; ---------------------------------------------------------


define :method updaterof go_angle( angle, grp :go_group );
lvars angle, grp;
;;; REF: INTEGER -> go_angle( GROUP );
;;; REF: Rotates the group around the centre to a given angle. All rotatable
;;; REF: objects will also be rotated (over the same relative angle).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars anglediff = angle - grp.go_angle, component;
	returnif( anglediff == 0 );
	procedure;
		dlocal 1 % go_batch_mode( grp ) % = true;
		for component in  the_go_components(grp)
		do
			if component.isgo_rotatable then
				go_angle( component ) + anglediff -> go_angle( component );
			endif;
		endfor;
		call_next_method( angle, grp );
	endprocedure();
enddefine;



;;;------------------------------------------------------
;;; EDIT/LIVE MODE:

define :method go_live_object( grp :go_group );
lvars grp;
;;; REF: go_live_object( GROUP ) -> BOOLEAN;
;;; REF: Whether the group object is in live mode even if the pane is in edit
;;; REF: mode. If false the mode of the pane prevails. This simulates the
;;; REF: corresponding slot of a go_screen_object (REF * GO_SCREEN_OBJECT).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
	stored_go_live_object( grp );
enddefine;

define :method updaterof go_live_object( edmode, grp :go_group );
lvars edmode, grp;
;;; REF: BOOLEAN -> go_live_object( GROUP );
;;; REF: If true the group object is in live mode even if the pane is in edit
;;; REF: mode. If false the mode of the pane prevails. This simulates the
;;; REF: corresponding slot of a go_screen_object (REF * GO_SCREEN_OBJECT).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
 ;;; make all go_group-components the same go_linewidth
	for component in grp.the_go_components do
		edmode -> go_live_object(component );
	endfor;
	edmode -> stored_go_live_object( grp );
enddefine;

;;;----------------------------------------------------------
;;; LINE SUPPORT

define :method updaterof go_linewidth( lw, grp :go_group );
lvars lw, grp;
;;; REF: REAL -> go_linewidth( GROUP );
;;; REF: Sets the line width of each of the components of the group.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
 ;;; make all go_group-components the same go_linewidth
	go_clear( grp );
	for component in grp.the_go_components do
		lw -> go_linewidth(component );
	endfor;
	call_next_method( lw, grp );
enddefine;

;;;----------------------------------------------------------
;;; COLOR SUPPORT

define :method updaterof go_fgcolour( color, grp :go_group );
lvars color, grp;
;;; REF: WORD_OR_STRING_OR_INTEGER -> go_fgcolour( GROUP );
;;; REF: Sets the foreground colour of each of the components of the group.
;;; REF: This will only succeed if all components are colourable. The colour is
;;; REF: stored as words rather than strings or as the integer representing
;;; REF: the colour index in the colour table of the go_pane it will be
;;; REF: shown in. Such an index is not screen independent and is unsafe.
;;; REF: This is used by go_draw (see REF * GO_SCREEN_OBJECT, GO_COLOURABLE).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
 ;;; make all go_group-components the same color
	go_clear( grp );
	for component in grp.the_go_components do
		if (component.isgo_colourable) then
			color -> go_fgcolour(component );
		endif;
	endfor;
	call_next_method( color, grp );
enddefine;

define :method updaterof go_bgcolour( color, grp :go_group );
lvars color, grp;
;;; REF: WORD_OR_STRING_OR_INTEGER -> go_fgcolour( GROUP );
;;; REF: Sets the background colour of each of the components of the group.
;;; REF: This will only work on components which are colourable. The colour is
;;; REF: stored as words rather than strings or as the integer representing
;;; REF: the colour index in the colour table of the go_pane it will be
;;; REF: shown in. Such an index is not screen independent and is unsafe.
;;; REF: This is used by go_draw (see REF * GO_SCREEN_OBJECT, GO_COLOURABLE).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
 ;;; make all go_group-components the same color
	go_clear( grp );
	for component in grp.the_go_components do
		if (component.isgo_colourable) then
			color -> go_bgcolour(component );
		endif;
	endfor;
	call_next_method( color, grp );
enddefine;


define :method updaterof go_filled( bool, grp :go_group );
lvars bool, grp;
;;; REF: WORD_OR_STRING_OR_INTEGER -> go_fgcolour( GROUP );
;;; REF: Sets the background colour of each of the components of the group.
;;; REF: This will only work on components which are colourable. The colour is
;;; REF: stored as words rather than strings or as the integer representing
;;; REF: the colour index in the colour table of the go_pane it will be
;;; REF: shown in. Such an index is not screen independent and is unsafe.
;;; REF: This is used by go_draw (see REF * GO_SCREEN_OBJECT, GO_COLOURABLE).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
 ;;; make all go_group-components the same color
	go_clear( grp );
	for component in grp.the_go_components do
		if (component.isgo_fillable) then
			bool -> go_filled(component );
		endif;
	endfor;
	call_next_method( bool, grp );
enddefine;

;;;----------------------------------------------------------


;;; BR added go_contains for groups:
define :method go_contains( x, y, grp :go_group ) -> component_flag;
lvars x, y, grp, component_flag = false;
;;; REF: go_contains( X, Y, GROUP ) -> LOCATED_OR_FALSE;
;;; REF: Checks whether the world coordinates of a test position fall inside
;;; REF: the group object. A go_group contains a location if one of the objects
;;; REF: in the go_group contains it. If the position is not inside then the
;;; REF: method returns the boolean false, else it returns the object in which
;;; REF: it falls (must be a go_located object).
;;; REF: X and Y are the world coordinates of the test position.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
	;;; start from false until otherwise found but only if inside bounding box
	if call_next_method( x, y, grp ) then
		for component in rev( grp.the_go_components ) do
			go_contains( x, y, component ) -> component_flag;
			quitif( component_flag );
		endfor;
	endif;
enddefine;



;;;----------------------------------------------------------

;;; BR added overlapping for groups:
define :method go_overlaps( obj:go_located, c :go_group ) -> flag;
lvars obj, c, flag = false;
;;; REF: go_contains( LOCATED, GROUP ) -> BOOLEAN;
;;; REF: Checks whether a located object overlaps the group object. An object
;;; REF: overlaps if one of the objects in the go_group overlaps it. It returns
;;; REF: the boolean true if it overlaps, else it is false.
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars component;
	;;; start from false until otherwise found but only if inside bounding box
	if go_overlaps_bounding_boxes( obj, c ) then
		for component in c.the_go_components do
			go_overlaps( obj, component ) -> flag;
			quitif( flag );
		endfor;
	endif;
enddefine;

define :method go_overlaps( c :go_group, obj:go_located ) -> flag;
lvars obj, c, flag;
;;; REF: go_contains( GROUP, LOCATED ) -> BOOLEAN;
;;; REF: Checks whether a located object overlaps the group object. An object
;;; REF: overlaps if one of the objects in the go_group overlaps it. It returns
;;; REF: the boolean true if it overlaps, else it is false.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	go_overlaps( obj, c ) -> flag;
enddefine;


;;; ---------------------------------------------------------
;;; BOUNDING BOX


define :method go_set_bounding_box( grp :go_group );
lvars grp;
;;; REF: go_set_bounding_box( GROUP );
;;; REF: Recalculates the bounding box of the group containing all the
;;; REF: components and recalculates the offsets of the components versus
;;; REF: the new centre of the bounding box (see also REF * GO_LOCATED).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars xmin, xmax, ymin, ymax, lw, lwmax=2;
lvars xc = grp.go_xcentre, yc = grp.go_ycentre, newxc, newyc, dx, dy, offsets;
lvars complist = grp.the_go_components, comp, xcomp, ycomp, wcomp, hcomp;
	if ( complist = [] ) then
		1 -> grp.stored_go_bounding_width;
		1 -> grp.stored_go_bounding_height;
		return;
	endif;  ;;; nothing there

	define lconstant go_recentre( grp );
	lvars grp;
	lvars component, x, y, x_offs, y_offs;
		[%  for component in  the_go_components(grp) do
				go_position_of_centre( component ) -> (x, y);
				go_transxyin(x, y, grp) /* -> (x_offs, y_offs) */;
			endfor;
		 %] -> grp.stored_go_local_coords
	enddefine;

	define lconstant go_normalize( x, y, w, h ) -> ( x, y, w, h );
	lvars x, y, w, h;
		if w < 0 then x + w -> x; -w -> w; endif;
		if h < 0 then y + h -> y; -h -> h; endif;
	enddefine;

	fast_destpair( complist ) -> (comp, complist) ;
	go_normalize( comp.go_bounding_box ) -> (xmin, ymin, wcomp, hcomp);
	xmin + wcomp -> xmax;
	ymin + hcomp -> ymax;
	for comp in complist do
		go_normalize( comp.go_bounding_box ) -> (xcomp, ycomp, wcomp, hcomp);
		min( xcomp,         xmin ) -> xmin;
		max( xcomp + wcomp, xmax ) -> xmax;
		min( ycomp,         ymin ) -> ymin;
		max( ycomp + hcomp, ymax ) -> ymax;
		if ( comp.go_linewidth ->> lw) then max( lw, lwmax ) -> lwmax; endif;
	endfor;
	((xmax + xmin) / 2) -> newxc;  ((ymax + ymin) / 2) -> newyc;
	unless ( intof(newxc - xc) == 0 ) and ( intof(newyc - yc) == 0 ) then
		;;; Make local coordinates to see shift; then shift in world coords
		go_transxyin( newxc, newyc, grp ) -> (dx, dy);
		dx * grp.go_xscale -> dx;  dy * grp.go_yscale -> dy;
		grp.stored_go_xorigin + dx -> grp.stored_go_xorigin;
		grp.stored_go_yorigin + dy -> grp.stored_go_yorigin;
		;;; Keep rotation point on the same absolute spot
		grp.stored_go_xrotation - dx -> grp.stored_go_xrotation;
		grp.stored_go_yrotation - dy -> grp.stored_go_yrotation;
		go_recentre(grp);
	endunless;
	max(1, round(xmax-xmin) ) -> grp.stored_go_bounding_width;
	max(1, round(ymax-ymin) ) -> grp.stored_go_bounding_height;
	lwmax -> grp.stored_go_linewidth;
enddefine;


define :method go_bounding_width( grp :go_group ) -> w;
lvars grp, w;
;;; REF: go_bounding_width( GROUP ) -> INTEGER;
;;; REF: Returns the integer width of the bounding box of the group (in world
;;; REF: coordinates). See also REF * GO_LOCATED
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars pane, reset_flag = false;
	go_validate_bounding_box( grp );
	grp.stored_go_bounding_width -> w;
enddefine;

define :method go_bounding_height( grp :go_group ) -> h;
lvars grp, h;
;;; REF: go_bounding_height( GROUP ) -> INTEGER;
;;; REF: Returns the integer height of the bounding box of the group (in world
;;; REF: coordinates). See also REF * GO_LOCATED
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars pane, reset_flag = false;
	go_validate_bounding_box( grp );
	grp.stored_go_bounding_height -> h;
enddefine;


;;;----------------------------------------------------------
;;; MEMBERSHIP FUNCTIONS (CHILDREN)

uses go_ncremove;

define :method go_remove_from( component:go_located, grp :go_group );
lvars component, grp;
;;; REF: go_remove_from( LOCATED, GROUP );
;;; REF: The located object is no longer a component of the group, and
;;; REF: becomes a separate object again and will appear on top of every
;;; REF: other object in the panes where the group is visible.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
lvars parent, nbr, comp, tmp_offsets;
	0 -> nbr;	;;; place of element in list
	if ( lmember(component, grp.the_go_components) ) then
		for comp in grp.the_go_components do
			nbr + 1 -> nbr;
			quitif( comp == component );
		endfor;
		go_ncremove( nbr, grp.go_components     ) -> grp.go_components;
		go_ncremove( nbr, grp.the_go_components ) -> grp.the_go_components;
		go_ncremove( nbr*2, grp.go_local_coords ) -> tmp_offsets;  ;;; y then x
		go_ncremove( nbr*2 - 1, tmp_offsets     ) -> grp.stored_go_local_coords;
	else
		npr('Warning: no such object in the group');
	endif;
	;;; Update the objects visibility:
	delete(grp, component.go_visible_in, nonop ==) -> component.go_visible_in;
	for parent in grp.go_visible_in do
		go_add_to(component, parent);     ;;;; CURRENTLY UP FRONT !!
	endfor;
	go_internal_redraw( component );
	go_update_and_draw( grp );
enddefine;


define lconstant add_to_grp(component, grp, f_or_b);
lvars component, grp, f_or_b;
lvars parent, nbr, comp, x_offs, y_offs, dx, dy,
	  gxs = grp.go_xscale,
	  gys = grp.go_yscale;
	for parent in grp.go_visible_in do
		if go_is_in( component, parent ) then
			go_remove_from( component, parent );
		endif;
	endfor;
	0 -> nbr;   ;;; place of element in list
	if not( lmember(component, grp.the_go_components) ) then

		;;; SHOULD BE THE <class>'s KEY
		if f_or_b then    ;;; add to front of the group (tail of the lists)
			grp.go_components <> [% datakey(component) %]
				-> grp.go_components;
			grp.the_go_components <> [% component %] -> grp.the_go_components;
		else              ;;; add to the back  of the group (head of the lists)
			datakey(component) :: grp.go_components
				-> grp.go_components;
			component :: grp.the_go_components -> grp.the_go_components;
		endif;
		grp :: (component.go_visible_in) -> component.go_visible_in;
	else
		pr('Warning: this component is already included!\n');
		for comp in grp.the_go_components do
			nbr + 1 -> nbr;
			quitif( comp == component );
		endfor;
	endif;

	go_transxyin( component.go_xcentre, component.go_ycentre, grp )
	-> (x_offs, y_offs); ;;; angle/scale

	lvars offsets = grp.go_local_coords;
	if (nbr = 0) then
		if f_or_b then offsets  <>  [% x_offs, y_offs %];
				  else x_offs :: (y_offs :: offsets);
		endif;
	else
		x_offs -> offsets(nbr*2 - 1);
		y_offs -> offsets(nbr*2);
		offsets;
	endif -> grp.stored_go_local_coords;

	;;; Erase the original object because the depth will change
	go_set_bounding_box(grp);
	go_update_and_draw(grp);

enddefine;

define :method go_add_to( component:go_located, grp :go_group );
;;; REF: go_add_to( LOCATED, GROUP );
;;; REF: The located object is added as a component of the group, and becomes
;;; REF: part of the group where it will appear on top of every other object
;;; REF: in the group.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	add_to_grp(component, grp, true);
enddefine;

define :method go_annex_to( component:go_located, grp :go_group );
;;; REF: go_annex_to( LOCATED, GROUP );
;;; REF: The located object is added as a component of the group, and becomes
;;; REF: part of the group where it will appear below every other object in
;;; REF: the group.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
;;; REF: LOCATED is a go_located instance (see REF * GO_LOCATED)
	add_to_grp(component, grp, false);
enddefine;


;;;------------------------------------------------------------------
;;; COPY: tentative definition for copying an object

define :method go_copy_object( grp :go_group ) -> new_grp;
lvars grp;
;;; REF: go_copy_object( GROUP ) -> GROUP;
;;; REF: The group and its components are all copied and visualized in the
;;; REF: same panes as the original group (see REF * GO_PANE).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars new_grp, obj, pane;
	;;; copies the object and leaves new one on the stack
	procedure;
	dlocal 1 % grp.the_go_components % = [];
		call_next_method( grp ) -> new_grp;
	endprocedure();

	define lconstant copy_without_parent;
	dlocal 1 % obj.go_visible_in % = [];
		go_copy_object( obj );
	enddefine;

	for obj in grp.the_go_components do
	go_add_to( copy_without_parent(), new_grp );
	endfor;
enddefine;

;;; ---------------------------------------------------------
;;; MAKE EDITABLE
;;;

define :method go_make_uneditable( grp :go_group );
lvars grp;
;;; REF: go_make_uneditable( GROUP );
;;; REF: The components of the group become no longer editable and the group
;;; REF: is manipulated as a whole. This removes all editors from the
;;; REF: components (see also REF * GO_EDIT_POINT).
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars pnt, redraw_flag = false;
	for pnt in grp.go_editors do
		go_remove_from( pnt, go_default_pane );
		false -> go_edited_object( pnt );
		true -> redraw_flag;
	endfor;
	if redraw_flag then
		[] -> grp.go_editors;
		go_update_and_draw( grp );
	endif;
enddefine;

define :method go_make_editable( grp :go_group );
lvars grp;
;;; REF: go_make_editable( GROUP );
;;; REF: The group becomes editable in its components rather than to be
;;; REF: manipulated as a whole. Editors in the centre of each component are
;;; REF: used as feedback on the screen and only those can be used to move
;;; REF: objects around is given (see also REF * GO_EDIT_POINT)
;;; REF: WARNING: Currently the depth of the objects is not guaranteed!
;;; REF:          The object which is dragged normally appears below the others.
;;; REF: GROUP is an instance of the go_group class (see REF * GO_GROUP).
lvars obj, pnt;
	go_make_uneditable( grp );
	;;; create new editors at least one in centre and one in a vertex
	[% for obj in grp.the_go_components do
		   newgo_edit_point() ->> pnt;       ;;; puts the editor on stack
		   obj -> go_edited_object( pnt );
		   go_add_to( pnt, go_default_pane );
	   endfor;
	%] -> grp.go_editors;
enddefine;


;;; ---------------------------------------------------------
;;; Variable for "uses"
vars go_group = true;

/* --- Revision History ---------------------------------------------------
 * BR 24/08/93
 *     Updated file for use with new go_local_coords() and go_world_coords()
 *     methods which replace the go_[x/y]offsets() and which are removed.
 *     Renamed go_reset_group() go_update() in line with other classes.
 * BR 05/07/93
 *     Added a slot go_editors to avoid the overuse of go_live_object to
 *     also keep the editors within go_make_editable().
 * BR 21/06/93
 *     Simplified go_contains() since go_find_ui_object() in go_composite.p
 *     can be used for event handling.
 *     Added check in go_contains() and go_overlaps() to check whether its
 *     in the bounding box (should improve complex groups).
 * Aaron Sloman 13/06/93
 *     Changed "==" to "=" in scale updaters
 * BR 03/06/93
 *     Removed all references to stored_go_[x/y]loc (go_[set_]bounding_box).
 * BR 25/05/93
 *     Added support for cached_go_coord_list flag in a method for
 *     go_bounding_box and adapted go_draw accordingly. Optimised some calls
 *     to go_set_bounding_box away by use of go_invalidate_bounding_box().
 * BR 19/05/93
 *     Removed updaters for go_[x/y]loc() since they are taken care for in
 *     the new go_located class methods.
 *     Deleted the go_prevent_draw slot and added a choice "forbidden" to the
 *     go_visible_now slot of the go_screen_object class which is taken
 *     into account in the methods: go_draw(), go_clear(), go_redraw()
 *     or go_internal_redraw().
 * BR 17/05/93
 *     Removed the go_composite CLASS and changed the go_make_[un]editable()
 *     methods to use editors.
 * BR 07/05/93
 *     Adapted (simplified) because new default of go_[x/y]scale() = 1.
 *     Added comments
 * BR 06/04/93
 *     Introduced the CLASS go_composite with a "go_editable" slot (WARNING
 *     this name was used before as name of a mixin; now called go_editor).
 *     This has new go_make_editable()/go_make_uneditable() methods which
 *     are based on a test in: go_find_ui_object() in go_go_pane.p.
 * BR 30/03/93
 *     Changed go_window_pane class into go_pane class.
 * Ian Rogers 21/01/93
 *     Cleaned up definition of go_add_to_group() to construct new<clasname>
 *     and added an annex_to_group(). Both use internal version add_to_grp().
 * BR 21/12/92
 *     Changed the go_add_to_group() order to be closer to go_remove_from_group().
 * BR 18/12/92
 *     Added a go_reset_group() method which sets the items back to the go_local_coords.
 * BR 26/11/92
 *     Changed the order in which go_contains() searches for elements in the list
 * BR 26/11/92
 *     Global Name changes and code cleanup
 * BR 09/11/92
 *     Adapted for use with OC 5.01 : uses initialise;
 * BR 05/11/92
 *     Cleaned up the go_make_visible() method and the comments in this file.
 * BR 03/11/92
 *     Changed go_contains to return the object rather than true or false...
 * BR 05/10/92
 *     changed to a go library (rather than sketch).
 *     go_make_invisible() now has pane as an extra argument (see MajorChanges)
 * BR 04/09/92
 *     removed the go_drag_and_drop() and go_clear() method (see new version of go_located
 *     class in go_located.p).
 * BR, 21 Aug 1992
 *     changed test to check for go_bounding_width/go_bounding_height methods to
 *     isalocated commented out test code
 * BR, 24 Jul 1992
 *     changed name from container to go_group,
 *     added slot "fixed" for fixed/drawn image dragging (:method go_drag_and_drop)
 * BR, 20 Jul 1992
 *     additional methods to handle go_bounding_width, go_bounding_height and locations
 * BR, 16 Jul 1992
 *     corrections for objectclass, deletion of visible bounding-box
 * JLC, 15 Jun 1992
 *     Converted to use objectclass
 * Aaron Sloman, Mar 17 1992
 *     Changed "load" to "lib" in the example
 */
;;; eof
